/**
 ****************************************************************************************
 *
 * @file bam.c
 *
 * @brief Functions and structures used by the UMAC ME BAM (Block ACK Manager) module.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */

#include "bam.h"
#include "bam_task.h"
#include "mac_frame.h"
#include "me_mgmtframe.h"
#include "mm_task.h"
#include "ke_timer.h"
#include "sta_mgmt.h"
#include "vif_mgmt.h"
#include "txu_cntrl.h"
#include "txl_cfm.h"
#include "ps.h"
#if NX_MFP
#include "mfp.h"
#endif
#include "tpc.h"
/**
 * @addtogroup BAM
 * @{
 */

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
/// BaW array
static struct bam_baw bam_baws[NX_MAX_BA_TX];

/// BAM module environment definition.
struct bam_env_tag bam_env[BAM_IDX_MAX];

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
#if NX_AMPDU_TX
#ifdef CFG_RWTL
/// Variable used for a temporary workaround for the TL4 compiler
uint16_t tl_tudiff;
#endif

/**
 ****************************************************************************************
 * @brief Read and returns the current BAM time (used for lifetime expiry computation).
 *
 * @return The BAM time
 *
 ****************************************************************************************
 */
static uint16_t bam_time_get(void)
{
    // Get the current system time and convert it in TUs
    return((hal_machw_time() >> 10) & 0xFFFF);
}

/**
 ****************************************************************************************
 * @brief Compare BAM timer absolute expiration time.
 *
 * @param[in] time1 First time to compare.
 * @param[in] time2 Second time to compare.
 *
 * @return true if time1 is earlier than time2.
 ****************************************************************************************
 */
static bool bam_time_cmp(uint16_t time1, uint16_t time2)
{
    uint16_t diff = time1 - time2;

    #ifdef CFG_RWTL
    /// Temporary workaround for the TL4 compiler
    tl_tudiff = diff;
    #endif

    return (((int16_t)diff) < 0);
}

/**
 ****************************************************************************************
 * @brief Check if a BAM time is in the past or not.
 *
 * @param[in] time Time to compare.
 *
 * @return true if time is in the past, false otherwise
 ****************************************************************************************
 */
static bool bam_time_past(uint32_t time)
{
    return (bam_time_cmp(time, bam_time_get()));
}


/**
 ****************************************************************************************
 * @brief Compute the index of a BaW element from its offset.
 * This function uses a modulo to do the computation.
 *
 * @param[in] baw      Pointer to the BaW structure.
 * @param[in] offset   Offset of the element in the BaW structure.
 *
 * @return The index of the element in the BaW
 ****************************************************************************************
 */
static unsigned int bam_baw_index_compute(struct bam_baw *baw, unsigned int offset)
{
    return ((baw->fsn_idx + offset) % baw->buf_size);
}

/**
 ****************************************************************************************
 * @brief Compute the index of a BaW element from its offset.
 * This function uses a mask. It is used only when the BaW size is a power of 2.
 *
 * @param[in] baw      Pointer to the BaW structure.
 * @param[in] offset   Offset of the element in the BaW structure.
 *
 * @return The index of the element in the BaW
 ****************************************************************************************
 */
static unsigned int bam_baw_index_compute_opt(struct bam_baw *baw, unsigned int offset)
{
    return ((baw->fsn_idx + offset) & baw->mask);
}

/**
 ****************************************************************************************
 * @brief Move the BaW to the next unconfirmed packet.
 * This function goes through the BaW and checks the status of the elements. For each
 * confirmed element the number of credits (i.e the number of positions becoming free in
 * the BaW) that can be allocated to the associated TX queue is increased.
 *
 * @param[in] baw      Pointer to the BaW structure.
 *
 * @return The number of credits
 ****************************************************************************************
 */
static int8_t bam_move_baw(struct bam_baw *baw)
{
    int8_t credits = 0;

    while (baw->states[baw->fsn_idx] == BAW_CONFIRMED) {
        baw->states[baw->fsn_idx] = BAW_FREE;
        baw->fsn = (baw->fsn + 1) & ((1 << 12) - 1);
        baw->fsn_idx = baw->idx_compute(baw, 1);
        credits++;
    }

    return(credits);
}

/**
 ****************************************************************************************
 * @brief Set the state of a sequence number in the given BaW.
 *
 * @param[in] baw      Pointer to the BaW structure.
 * @param[in] sn       Sequence Number to be modified
 * @param[in] state    New state
 ****************************************************************************************
 */
static void bam_set_baw_state(struct bam_baw *baw, uint16_t sn, uint8_t state)
{
    uint16_t baw_offset;
    int index;

    baw_offset = ((uint16_t)(sn - baw->fsn)) & ((1 << 12) - 1);

    ASSERT_ERR(baw_offset < baw->buf_size);

    index = baw->idx_compute(baw, baw_offset);

    baw->states[index] = state;
}

/**
 ****************************************************************************************
 * @brief Check and update the state of the BaW after transmission of a frame
 * This function verifies if the frame should be retried or not. It then updates the
 * state of the BaW to @ref BAW_CONFIRMED if the frame is successful or not to be retried,
 *  and finally move the BaW.
 *
 * @param[in] txdesc   Pointer to the TX descriptor attached to the frame.
 * @param[in] bam_idx  Index of the BAM task
 * @param[in] success  true if frame was successfully transmitted, false otherwise
 *
 * @return The number of credits allocated to the TX queue after the frame confirmation
 ****************************************************************************************
 */
static int8_t bam_check_tx_baw(struct txdesc *txdesc, uint8_t bam_idx,
                               bool success)
{
    struct bam_baw *baw = bam_env[bam_idx].baw;
    struct tx_cfm_tag *cfm = txl_cfm_tag_get(txdesc);
    struct hostdesc *host = &txdesc->host;

    do
    {
        // First check if the MPDU was part of an A-MPDU
        if ((!(txdesc->umac.flags & AMPDU_BIT))
                #if NX_AMSDU_TX
                && (!is_mpdu_split(txdesc))
                #endif
           )
            break;

        // Then check if the MPDU was successful or not
        if (!success && !bam_time_past(host->timestamp))
        {
            // Update the confirmation descriptor
            cfm->status |= TX_STATUS_RETRY_REQUIRED;
            #if !NX_FULLY_HOSTED
            cfm->pn[0] = host->pn[0];
            cfm->pn[1] = host->pn[1];
            cfm->pn[2] = host->pn[2];
            cfm->pn[3] = host->pn[3];
            cfm->sn = host->sn;
            cfm->timestamp = host->timestamp;
            #endif
            return 0;
        }
    } while (0);

    // Update the state in the BAW
    bam_set_baw_state(baw, host->sn, BAW_CONFIRMED);

    // Move the transmission window
    return (bam_move_baw(baw));
}

/**
 ****************************************************************************************
 * @brief Initiate the creation of a TX BlockAck agreement.
 *
 * @param[in] sta_idx Station for which the BlockAck agreement is created
 * @param[in] tid     TID of the BA agreement
 * @param[in] ssn     Sequence number of the first packet sent after the BA agreement
 *                    establishment
 *
 * @return CO_OK if the BA agreement establishment is started, CO_FULL if no more BAM
 * task is available.
 ****************************************************************************************
 */
static uint8_t bam_create_ba_agg(uint8_t sta_idx, uint8_t tid, uint16_t ssn)
{
    uint8_t bam_idx = bam_alloc_new_task(true);

    if (bam_idx == BAM_INVALID_TASK_IDX)
        return  CO_FULL;

    TRACE_LMAC(BA, "{BAM-%d} Start TX agreement with {STA-%d}: tid=%d ssn=%d",
               bam_idx, sta_idx, tid, ssn);

    // Store the useful information in the environment
    bam_env[bam_idx].sta_idx          = sta_idx;
    bam_env[bam_idx].tid              = tid;
    bam_env[bam_idx].dialog_token     = co_rand_byte();
    bam_env[bam_idx].ba_timeout       = BAM_INACTIVITY_TO_DURATION;
    bam_env[bam_idx].ba_policy        = 1;
    bam_env[bam_idx].dev_type         = BA_ORIGINATOR;
    bam_env[bam_idx].buffer_size      = nx_txdesc_cnt[mac_tid2ac[tid]];
    bam_env[bam_idx].ssn              = ssn;
    bam_env[bam_idx].pkt_cnt          = 0;

    // Set the BAM Task Index and device type in the lmac_sta_info_tag
    sta_info_tab[sta_idx].ba_info[tid].bam_idx_tx = bam_idx;

    // Send MAC_BA_ACTION_DELBA in order to ensure that the two stations are synchronized
    // before sending a new block ack agreement request.
    // In fact, it could happen that a block ack agreement is already set up for this
    // TID on the responder station, which might not reply to the ADDBA_REQ, causing
    // the BA agreement is not established anymore.
    // This loss of synchronisation could happen in case, for example, the originator
    // sent a ADDBA_REQ but it didn't receive the ADDBA_RSP, or in case the originator
    // didn't receive the acknowledgment of a DELBA frame.
    bam_send_air_action_frame(sta_idx, &bam_env[bam_idx], MAC_BA_ACTION_DELBA, 0, 0,
                              MAC_RS_TIMEOUT, NULL);

    // Send ADDBA Request packet
    bam_send_air_action_frame(sta_idx, &bam_env[bam_idx], MAC_BA_ACTION_ADDBA_REQ, 0, 0, 0, NULL);

    // Store current time
    sta_mgmt_set_add_ba_time(sta_idx, tid, ke_time());

    // Start the ADDBA RSP Timeout timer
    ke_timer_set(BAM_ADD_BA_RSP_TIMEOUT_IND, KE_BUILD_ID(TASK_BAM, bam_idx),
                 (BAM_RESPONSE_TO_DURATION * TU_DURATION));

    // Set state as wait_rsp
    ke_state_set(KE_BUILD_ID(TASK_BAM, bam_idx), BAM_WAIT_RSP);

    return CO_OK;
}
#endif

void bam_init(void)
{
    for (int bam_idx = 0; bam_idx < BAM_IDX_MAX; bam_idx++)
    {
        bam_env[bam_idx].sta_idx     = INVALID_STA_IDX;

        // Set pointer to BAW for TX BA structures
        if (bam_idx >= BAM_FIRST_TX_IDX)
        {
            bam_env[bam_idx].baw = &bam_baws[bam_idx - BAM_FIRST_TX_IDX];
        }

        // Go to IDLE state
        ke_state_set(KE_BUILD_ID(TASK_BAM, bam_idx), BAM_IDLE);
    }
}

#if NX_AMPDU_TX || NX_REORD
void bam_param_sta_info_tab_reset(uint16_t bam_idx)
{
    uint16_t sta_idx = bam_env[bam_idx].sta_idx;
    uint8_t tid      = bam_env[bam_idx].tid;

    // Initialize the parameters of the BA in the station info table
    switch (bam_env[bam_idx].dev_type)
    {
        case BA_RESPONDER:
        {
            sta_info_tab[sta_idx].ba_info[tid].bam_idx_rx = BAM_INVALID_TASK_IDX;
        } break;

        case BA_ORIGINATOR:
        {
            sta_info_tab[sta_idx].ba_info[tid].bam_idx_tx = BAM_INVALID_TASK_IDX;
        } break;

        default:
        {
            // Should not happen
        } break;
    }
}

uint16_t bam_alloc_new_task(bool tx)
{
    uint8_t start, cnt;

    // The first indexes are reserved for RX BA, while the last ones are for TX
    if (tx)
    {
        start = BAM_FIRST_TX_IDX;
        cnt = BAM_NB_TX_IDX;
    }
    else
    {
        start = BAM_FIRST_RX_IDX;
        cnt = BAM_NB_RX_IDX;
    }

    // Loop on the BAM IDX
    for (int i = start; i < (start + cnt); i++)
    {
        // Check if the BAM task is Idle or not
        if (bam_state[i] == BAM_IDLE)
        {
            // Return bam_task_idx
            return i;
        }
    }

    TRACE_LMAC(BA, "No more BA (TX=%d) task available", tx);
    return BAM_INVALID_TASK_IDX;
}

void bam_delete_ba_agg(uint8_t bam_idx)
{
    // BAM Task id
    ke_task_id_t task_id = KE_BUILD_ID(TASK_BAM, bam_idx);

    TRACE_LMAC(BA, "{BAM-%d} agreement deleted", bam_idx);

    // Stop Inactivity Timeout timer
    ke_timer_clear(BAM_ADD_BA_RSP_TIMEOUT_IND, task_id);
    // Stop Inactivity Timeout timer
    ke_timer_clear(BAM_INACTIVITY_TIMEOUT_IND, task_id);

    // Clear information stored in the STA_MGMT module
    bam_param_sta_info_tab_reset(bam_idx);

    // Go back to idle state
    ke_state_set(task_id, BAM_IDLE);
}

void bam_delete_all_ba_agg(uint8_t sta_idx)
{
    // Loop on the BAM IDX (Instance 0 is BAM Manager)
    for (int i = 0; i < BAM_IDX_MAX; i++)
    {
        // Check if the BAM task is Idle or not
        if (bam_state[i] == BAM_IDLE)
        {
            continue;
        }

        // Check STAID
        if (bam_env[i].sta_idx != sta_idx)
        {
            continue;
        }

        // Go to delete state
        ke_state_set(KE_BUILD_ID(TASK_BAM, i), BAM_DELETE);

        // Delete
        bam_send_mm_ba_del_req(sta_idx, i);
    }
}

void bam_start_inactivity_timer(uint16_t bam_idx)
{
    ke_timer_set(BAM_INACTIVITY_TIMEOUT_IND, KE_BUILD_ID(TASK_BAM, bam_idx),
                 bam_env[bam_idx].ba_timeout * TU_DURATION);
}

void bam_send_mm_ba_add_req(uint16_t sta_idx, uint16_t bam_idx)
{
    // Allocate the message
    struct mm_ba_add_req *req = KE_MSG_ALLOC(MM_BA_ADD_REQ,
                                             TASK_MM, KE_BUILD_ID(TASK_BAM, bam_idx),
                                             mm_ba_add_req);

    // Fill the message parameters
    req->type      = (bam_env[bam_idx].dev_type == BA_ORIGINATOR) ? 0 : 1;
    req->sta_idx   = sta_idx;
    req->tid       = bam_env[bam_idx].tid;
    req->bufsz     = bam_env[bam_idx].buffer_size;
    req->ssn       = bam_env[bam_idx].ssn;

    // Send the message to the task
    ke_msg_send(req);
}

void bam_send_mm_ba_del_req(uint16_t sta_idx, uint16_t bam_idx)
{
    // Allocate the message
    struct mm_ba_del_req *req = KE_MSG_ALLOC(MM_BA_DEL_REQ,
                                             TASK_MM, KE_BUILD_ID(TASK_BAM, bam_idx),
                                             mm_ba_del_req);

    // Sanity check - The BAM task shall be in BAM_DELETE state when the BA deletion
    // request is sent to the LMAC
    ASSERT_ERR(ke_state_get(KE_BUILD_ID(TASK_BAM, bam_idx)) == BAM_DELETE);

    // Fill the message parameters
    req->type      = (bam_env[bam_idx].dev_type == BA_ORIGINATOR) ? 0 : 1;
    req->sta_idx   = sta_idx;
    req->tid       = bam_env[bam_idx].tid;

    // Send the message
    ke_msg_send(req);
}
#endif

void bam_send_air_action_frame(uint8_t sta_idx,
                               struct bam_env_tag *bam_env,
                               uint8_t action,
                               uint8_t dialog_token,
                               uint16_t param,
                               uint16_t status_code,
                               void (*cfm_func)(void *, uint32_t))
{
    struct txl_frame_desc_tag *frame;
    struct mac_hdr *buf;
    struct tx_hd *thd;
    uint8_t ac = AC_VO;
    int txtype;
    uint8_t vif_idx = sta_mgmt_get_vif_idx(sta_idx);
    struct vif_info_tag *vif = &vif_info_tab[vif_idx];
    uint32_t length = 0;

    // Allocate a frame descriptor from the TX path
    txtype = vif_mgmt_get_txtype(vif);
    frame = txl_frame_get(txtype, NX_TXFRAME_LEN);

    if (!frame)
        return;

    // update Tx power in policy table
    tpc_update_frame_tx_power(vif, frame);

    // Get the buffer pointer
    buf = txl_buffer_payload_get(&frame->txdesc);

    // Fill-in the frame
    buf->fctl  = MAC_FCTRL_ACTION;
    // TODO[LT] - Set duration
    buf->durid = 0;
    // Set RA
    MAC_ADDR_CPY(&buf->addr1, sta_mgmt_get_peer_addr(sta_idx));
    // Set TA
    MAC_ADDR_CPY(&buf->addr2, vif_mgmt_get_addr(vif_idx));
    // Set ADDR3
    if (vif_mgmt_get_type(vif_idx) == VIF_AP)
    {
        MAC_ADDR_CPY(&buf->addr3, vif_mgmt_get_addr(vif_idx));
    }
    else
    {
        // STA or IBSS
        MAC_ADDR_CPY(&buf->addr3, sta_mgmt_get_peer_addr(sta_idx));
    }
    // Update the sequence number in the frame
    buf->seq = txl_get_seq_ctrl();

    // Set VIF and STA indexes
    frame->txdesc.host.vif_idx = vif_idx;
    frame->txdesc.host.staid   = sta_idx;

    length = MAC_SHORT_MAC_HDR_LEN;

    #if NX_MFP
    frame->txdesc.umac.head_len = 0;
    frame->txdesc.umac.tail_len = 0;
    if (MFP_UNICAST_PROT == mfp_protect_mgmt_frame(&frame->txdesc, buf->fctl,
                                                   MAC_BA_ACTION_CATEGORY))
    {
        txu_cntrl_protect_mgmt_frame(&frame->txdesc, buf, MAC_SHORT_MAC_HDR_LEN);
        length += frame->txdesc.umac.head_len;
    }
    #endif

    // Build the payload
    switch (action)
    {
        case MAC_BA_ACTION_ADDBA_REQ:
        {
            ac = mac_tid2ac[bam_env->tid];
            length += me_build_add_ba_req(CPU2HW(buf) + length, bam_env);
        } break;

        case MAC_BA_ACTION_ADDBA_RSP:
        {
            length += me_build_add_ba_rsp(CPU2HW(buf) + length, bam_env,
                                          param, dialog_token, status_code);
        } break;

        case MAC_BA_ACTION_DELBA:
        {
            ac = mac_tid2ac[bam_env->tid];
            length += me_build_del_ba(CPU2HW(buf) + length, bam_env,
                                      status_code);
        } break;

        default:
        {
            ASSERT_WARN(0);
        } break;
    }

    #if NX_MFP
    length += frame->txdesc.umac.tail_len;
    #endif
    thd             = &frame->txdesc.lmac.hw_desc->thd;
    thd->dataendptr = (uint32_t)thd->datastartptr + length - 1;
    thd->frmlen     = length + MAC_FCS_LEN;

    if (cfm_func) {
        frame->cfm.cfm_func = cfm_func;
        frame->cfm.env = bam_env;
    }

    // Push the frame for TX
    txl_frame_push(frame, ac);
}

#if NX_AMPDU_TX
void bam_check_ba_agg(struct txdesc *txdesc)
{
    // Get current time
    uint32_t current_time = hal_machw_time();
    uint8_t sta_idx = txdesc->host.staid;
    uint8_t tid = txdesc->host.tid;
    uint8_t bam_idx;
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];

    // Check if peer device supports HT
    if (!STA_CAPA(sta, HT))
        return;

    // Increase the number of data packets for this STA/TID
    sta_mgmt_ba_tx_cnt_inc(sta_idx, tid);

    // No BlockAck agreement established for the 4-way Handshake
    if (sta->ctrl_port_state != PORT_OPEN)
        return;

    // Get matching BAM index in order to know if a BA Agreement already exists
    bam_idx = sta_mgmt_get_tx_bam_idx(sta_idx, tid);

    if (bam_idx != BAM_INVALID_TASK_IDX)
    {
        // Update the last activity time
        bam_env[bam_idx].last_activity_time = current_time;

        // Check the current BAM task state in order to know if establishment is over
        if (ke_state_get(KE_BUILD_ID(TASK_BAM, bam_idx)) == BAM_ACTIVE)
        {
            // An agreement already exists, the packet can be part of an A-MPDU
            struct bam_baw *baw = bam_env[bam_idx].baw;
            txdesc->umac.sn_win = baw->fsn;
            if (rc_check_aggregation(sta))
            {
                txdesc->umac.flags |= AMPDU_BIT;
            }
            txdesc->host.flags |= TXU_CNTRL_UNDER_BA;
            bam_set_baw_state(baw, txdesc->host.sn, BAW_PENDING);
            if (!(txdesc->host.flags & TXU_CNTRL_RETRY))
                txdesc->host.timestamp = bam_time_get() + me_tx_lft_get();

            // Update the packet counter
            bam_env[bam_idx].pkt_cnt++;
        }
    }
    else
    {
        /*
         * Check if we should establish a BlockAck agreement for this STA/TID. For that,
         * we check these three conditions:
         * -> Check if latest trial to establish a BA agreement for this pair is not too
         *    recent, in order to avoid wasting time and power trying to establish the
         *    agreement again and again with a peer consistently rejecting it
         * -> Check if a minimum packet count is available in the TX path for this
         *    STA/TID, indicating that non-negligible traffic is present (e.g not only a
         *    ping)
         * -> Check if peer is not in PS state (we consider that a peer in PS does not
         *    require a lot of traffic and would exit PS if it was the case)
         */
        uint32_t last_ba_time = sta_mgmt_get_add_ba_time(sta_idx, tid);
        if (!hal_machw_time_cmp(current_time, last_ba_time + BAM_ADDBA_REQ_INTERVAL) &&
            (sta_mgmt_ba_tx_cnt_get(sta_idx, tid) >= BAM_BA_AGG_TX_CNT_THRES) &&
            (sta->ps_state != PS_MODE_ON))
        {
            // Try to get a free instance of BAM task
            bam_create_ba_agg(sta_idx, tid, txdesc->host.sn);
        }
    }
}

int8_t bam_tx_cfm(struct txdesc *txdesc, bool success)
{
    uint8_t sta_idx = txdesc->host.staid;
    uint8_t tid = txdesc->host.tid;
    uint8_t bam_idx;
    int8_t credits = 1;
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];

    do
    {
        // Check if peer device supports HT
        if (!STA_CAPA(sta, HT))
            break;

        // Decrease the number of data packets for this STA/TID
        sta_mgmt_ba_tx_cnt_dec(sta_idx, tid);

        // Check if the frame was transmitted under BA agreement
        if (!(txdesc->host.flags & TXU_CNTRL_UNDER_BA))
            break;

        // Get matching BAM index in order to know if the BA Agreement still exists
        bam_idx = sta_mgmt_get_tx_bam_idx(sta_idx, tid);

        // Check if we have an agreement
        if (bam_idx == BAM_INVALID_TASK_IDX)
            break;

        // Check the current BAM task state in order to know if the agreement is active
        if (ke_state_get(KE_BUILD_ID(TASK_BAM, bam_idx)) != BAM_ACTIVE)
            break;

        // Update the packet counter
        bam_env[bam_idx].pkt_cnt--;

        // We have an active agreement, so we need to check the state of our transmission
        // window
        credits = bam_check_tx_baw(txdesc, bam_idx, success);
    } while (0);

    return credits;
}

void bam_baw_init(struct bam_env_tag *bam_env)
{
    struct bam_baw *baw = bam_env->baw;

    // Reset the BAW states
    memset(&baw->states, BAW_FREE, BAM_MAX_TX_WIN_SIZE * sizeof(uint8_t));

    // Initializes the BAW
    baw->fsn = sta_mgmt_get_tx_ssn(bam_env->sta_idx, bam_env->tid);
    baw->fsn_idx = 0;
    baw->buf_size = bam_env->buffer_size;

    // Check if the buffer size is a power of 2 or not
    if (baw->buf_size & (baw->buf_size - 1))
    {
        baw->idx_compute = bam_baw_index_compute;
    }
    else
    {
        baw->mask = baw->buf_size - 1;
        baw->idx_compute = bam_baw_index_compute_opt;
    }
}

int8_t bam_flush_baw(struct bam_baw *baw)
{
    int8_t credits = 0;
    int i;

    for (i = 0; i < baw->buf_size; i++) {
        if (baw->states[i] == BAW_CONFIRMED)
            credits++;
    }

    return(credits);
}

#endif

#if NX_REORD
void bam_rx_active(uint8_t sta_idx, uint8_t tid)
{
    // Retrieve BAM instance index
    uint8_t bam_idx = sta_mgmt_get_rx_bam_idx(sta_idx, tid);

    ASSERT_ERR(bam_idx < BAM_INVALID_TASK_IDX);

    bam_env[bam_idx].last_activity_time = hal_machw_time();
}

void bam_send_del_ba_agg(uint8_t sta_idx, uint8_t tid)
{
    struct bam_env_tag bam;
    bam.sta_idx          = sta_idx;
    bam.tid              = tid;
    bam.dialog_token     = 0;
    bam.ba_timeout       = BAM_INACTIVITY_TO_DURATION;
    bam.ba_policy        = 1;
    bam.dev_type         = BA_RESPONDER;
    bam.buffer_size      = 64;
    bam.ssn              = 0;
    bam.pkt_cnt          = 0;

    bam_send_air_action_frame(sta_idx, &bam, MAC_BA_ACTION_DELBA,
                              0, 0, MAC_BA_ST_INVALID_PARAMETERS, NULL);
}
#endif

/// @} end of group
