/**
 ******************************************************************************
 * @file hsu.c
 *
 * @brief Hardware Security Unit helper functions.
 *
 * Copyright (C) RivieraWaves 2017-2019
 *
 ******************************************************************************
 */
#include "rwnx_config.h"

#if NX_HSU
#include "reg_hsu.h"
#include "me_mic.h"
#include "trace.h"

/**
 * @addtogroup HSU
 * @{
 */

/**
 * Operation modes supported by HSU
 */
enum hsu_modes {
    HSU_MODE_TKIP_MIC = 0,
    HSU_MODE_AES_128_CMAC = 1,
    HSU_MODE_IP_CHK = 2,
    HSU_MODE_SHA_1 = 3,
    HSU_MODE_SHA_256 = 4,
    HSU_MODE_SHA_224 = 5,
    HSU_MODE_HMAC_SHA1 = 6,
    HSU_MODE_HMAC_SHA256 = 7,
    HSU_MODE_HMAC_SHA224 = 8,
    HSU_MODE_SHA_512 = 9,
    HSU_MODE_SHA_384 = 10,
    HSU_MODE_HMAC_SHA512 = 11,
    HSU_MODE_HMAC_SHA384 = 12,
};

/// List features supported by HSU
static uint32_t hsu_status = 0;

/// True if HSU support feature \p m, false otherwise
#define HSU_SUPPORT(m) (hsu_status & HSU_##m##_BIT)

#if NX_FULLY_HOSTED
/// Flag indicating if the HSU is currently locked by someone
static bool hsu_locked;

/// Lock the HSU
/// If the HSU is already locked by someone else, the macro returns false so that the
/// requested operation is then performed in SW
#define HSU_LOCK()                                                                       \
{                                                                                        \
    bool res = false;                                                                    \
    GLOBAL_INT_DISABLE();                                                                \
    if (!hsu_locked)                                                                     \
    {                                                                                    \
        res = true;                                                                      \
        hsu_locked = true;                                                               \
    }                                                                                    \
    GLOBAL_INT_RESTORE();                                                                \
    if (!res)                                                                            \
        return false;                                                                    \
}

/// Unlock the HSU
#define HSU_UNLOCK()  { hsu_locked = false; }
#else
/// Macro unused
#define HSU_LOCK()
/// Macro unused
#define HSU_UNLOCK()
#endif

/// HSU mode value for control register
#define HSU_MODE(m) (hsu_mode(HSU_MODE_##m))

/**
 ******************************************************************************
 * @param[in] mode HSU mode (@ref hsu_modes)
 * @return control register value for requested mode
 ******************************************************************************
 */
__INLINE uint32_t hsu_mode(int mode)
{
    if ((hsu_status & HSU_VERSION_MASK) >= 7)
        return mode << HSU_MODE_LSB;
    else
        return mode << 1;
}

void hsu_init(void)
{
    uint32_t hsu_version;
    hsu_status = hsu_revision_get();

    if (hsu_status >= 0xdead0000)
    {
        hsu_status = 0;
        #if NX_HSU == 2
        ASSERT_ERR(0);
        #endif
        return;
    }

    hsu_version = hsu_status & HSU_VERSION_MASK;
    // First version doesn't include supported feature in revision register
    if (hsu_version < 5)
    {
        hsu_status |= ( HSU_TKIP_MIC_BIT |
                        HSU_AES_128_CMAC_BIT );
        if (hsu_version > 1)
            hsu_status |= HSU_IP_CHK_BIT;
        if (hsu_version > 2)
            hsu_status |= HSU_SHA_1_BIT;
    }

    HSU_UNLOCK();
}


/********************* AES-CMAC **********************************************/
uint64_t hsu_aes_cmac(uint32_t *key, int nb_elem, uint32_t addr[], int len[])
{
    uint64_t mic;
    int i;
    uint32_t ctrl = (HSU_MODE(AES_128_CMAC) |
                     HSU_FIRST_BUFFER_BIT |
                     HSU_START_BIT);

    if (!HSU_SUPPORT(AES_128_CMAC) || (nb_elem == 0))
        return 0;

    for (i = 0; i < nb_elem; i++)
    {
        if (addr[i] == 0 || TST_SHRAM_PTR(HW2CPU(addr[i])))
        {
            ASSERT_WARN(0);
            return 0;
        }
    }

    HSU_LOCK();
    for (i = 0; i < 4; i++)
    {
        hsu_key_tab_set(i, key[i]);
    }

    for (i = 0; i < nb_elem - 1; i++, addr++, len++)
    {
        hsu_source_addr_set(*addr);
        hsu_length_set(*len);
        hsu_status_clear_set(1);
        hsu_control_set(ctrl);
        while (!(hsu_status_set_get() & HSU_DONE_SET_BIT)) {}
        ctrl &= ~HSU_FIRST_BUFFER_BIT;
    }

    ctrl |= HSU_LAST_BUFFER_BIT;
    hsu_source_addr_set(*addr);
    hsu_length_set(*len);
    hsu_status_clear_set(1);
    hsu_control_set(ctrl);
    while (!(hsu_status_set_get() & HSU_DONE_SET_BIT)) {}

    mic = hsu_mic_tab_get(1);
    mic <<= 32;
    mic |= hsu_mic_tab_get(0);
    HSU_UNLOCK();

    return mic;
}


/********************* TKIP-MIC **********************************************/
/**
 ******************************************************************************
 * @brief Computes Michael MIC on the provided buffer using HSU
 *
 * Use HSU to compute Michael MIC.
 * Temporary MIC and remaining byts are restore from @p mic_calc.
 * After computation MIC and remaining bytes are saved in @p mic_calc.
 *
 * @param[in]     ctrl     Value of HSU control registr to use.
 * @param[in,out] mic_calc Pointer to Mic structure
 * @param[in]     data     HW address of the data buffer. Must be in SHARED RAM
 * @param[in]     data_len Length, in bytes, of the data buffer
 *
 * @return true if computation has been done by HSU, false otherwise
 ******************************************************************************
 */
static bool hsu_michael(uint32_t ctrl, struct mic_calc *mic_calc,
                        uint32_t data, uint32_t data_len)
{
    if (!HSU_SUPPORT(TKIP_MIC))
    {
        return false;
    }

    if (data_len && TST_SHRAM_PTR(HW2CPU(data)))
    {
        ASSERT_WARN(0);
        return false;
    }

    HSU_LOCK();
    hsu_mic_tab_set(0, mic_calc->mic_key_least);
    hsu_mic_tab_set(1, mic_calc->mic_key_most);
    hsu_remaining_set(mic_calc->last_m_i);

    hsu_source_addr_set(data);
    hsu_length_set(data_len);
    hsu_status_clear_set(1);
    hsu_control_set(ctrl);
    while (!(hsu_status_set_get() & HSU_DONE_SET_BIT)) {}

    mic_calc->mic_key_least = hsu_mic_tab_get(0);
    mic_calc->mic_key_most = hsu_mic_tab_get(1);
    mic_calc->last_m_i = hsu_remaining_get();
    HSU_UNLOCK();

    return true;
}


bool hsu_michael_init(struct mic_calc *mic_calc, uint32_t *mic_key,
                      uint32_t *aad)
{
    uint32_t ctrl = (HSU_MODE(TKIP_MIC) |
                     HSU_START_BIT);

    if (!HSU_SUPPORT(TKIP_MIC))
    {
        return false;
    }

    // Initialize MIC value
    mic_calc->mic_key_least = mic_key[0];
    mic_calc->mic_key_most  = mic_key[1];
    mic_calc->last_m_i      = 0;
    mic_calc->last_m_i_len  = 0;

    return hsu_michael(ctrl, mic_calc, CPU2HW(aad), 16);
}

bool hsu_michael_calc(struct mic_calc *mic_calc, uint32_t data,
                      uint32_t data_len)
{
    uint32_t ctrl = (HSU_MODE(TKIP_MIC) |
                     HSU_START_BIT);

    return hsu_michael(ctrl, mic_calc, data, data_len);
}

bool hsu_michael_end(struct mic_calc *mic_calc)
{
    uint32_t ctrl = (HSU_MODE(TKIP_MIC) |
                     HSU_LAST_BUFFER_BIT |
                     HSU_START_BIT);

    return hsu_michael(ctrl, mic_calc, 0, 0);
}

/********************* IP CHECKSUM *******************************************/

bool hsu_ip_checksum(uint32_t addr, uint16_t len, uint16_t *checksum)
{
    uint32_t ctrl = (HSU_MODE(IP_CHK) |
                     HSU_LAST_BUFFER_BIT |
                     HSU_START_BIT);

    if (!HSU_SUPPORT(IP_CHK) || TST_SHRAM_PTR(HW2CPU(addr)))
    {
        return false;
    }

    /* Configure HSU for computation */
    HSU_LOCK();
    hsu_source_addr_set(addr);
    hsu_length_set(len);
    hsu_status_clear_set(HSU_DONE_SET_BIT);
    hsu_control_set(ctrl);
    while (!(hsu_status_set_get() & HSU_DONE_SET_BIT)) {}

    *checksum = hsu_mic_tab_get(0) & 0xFFFF;
    HSU_UNLOCK();

    return true;
}

/********************* SHA-X *************************************************/

/// Maximum length in bytes for all input vectors
#define HSU_SHA_MAX_LEN 512

/// Intermediate buffer to concatenate input vectors
static uint8_t hsu_sha_buf[HSU_SHA_MAX_LEN] __SHAREDRAM;

/**
 ******************************************************************************
 * @brief Generic function for SHA-X computation
 *
 * Concatenate input vector in the temporary buffer @ref hsu_sha_buf and then
 * use HSU to compute the requested hash.
 * Caller must ensure that HSU supports the requested mode and that the ouput
 * buffer is big enough.
 *
 * @param[in]  hsu_sha_mode  Mode to configure for HSU
 * @param[in]  nb_elem       Number of elements in @p addr table
 * @param[in]  addr          Table of input vectors
 * @param[in]  len           Table of length, in bytes, of each input vectors
 * @param[in]  sha_size      Size, in 32bits words, of the computed hash
 * @param[out] sha           Output buffer. (Must be at least @p sha_size long)
 *
 * @return true is @p sha buffer has been updated with the requested hash and
 * false otherwise
 ******************************************************************************
 */
static bool hsu_sha_x(uint32_t hsu_sha_mode, int nb_elem, const uint8_t *addr[],
                      const size_t *len, int sha_size, uint32_t *sha)
{
    int i, length = 0;
    uint32_t ctrl = (hsu_sha_mode |
                     HSU_FIRST_BUFFER_BIT |
                     HSU_LAST_BUFFER_BIT |
                     HSU_START_BIT);

    for (i = 0 ; i < nb_elem; i++)
    {
        length += len[i];
    }

    if (length > HSU_SHA_MAX_LEN)
    {
        TRACE_LMAC(ERR, "Temporary buffer for SHA computation not big enough (need %d)", length);
        return false;
    }

    HSU_LOCK();
    // Concatenate input buffers in a single buffer in SHARED RAM
    length = 0;
    for (i = 0 ; i < nb_elem; i++)
    {
        memcpy(&hsu_sha_buf[length], addr[i], len[i]);
        length += len[i];
    }

    hsu_source_addr_set(CPU2HW(hsu_sha_buf));
    hsu_length_set(length);
    hsu_status_clear_set(HSU_DONE_SET_BIT);
    hsu_control_set(ctrl);
    while (!(hsu_status_set_get() & HSU_DONE_SET_BIT)) {}

    for (i = 0; i < sha_size; i ++)
    {
        *sha++ = hsu_sha_tab_get(i);
    }
    HSU_UNLOCK();

    return true;
}

bool hsu_sha1(int nb_elem, const uint8_t *addr[], const size_t *len, uint32_t *sha)
{
    if (!HSU_SUPPORT(SHA_1))
        return false;
    return hsu_sha_x(HSU_MODE(SHA_1), nb_elem, addr, len, 160/32, sha);
}

bool hsu_sha224(int nb_elem, const uint8_t *addr[], const size_t *len, uint32_t *sha)
{
    if (!HSU_SUPPORT(SHA_256_224))
        return false;
    return hsu_sha_x(HSU_MODE(SHA_224), nb_elem, addr, len, 224/32, sha);
}

bool hsu_sha256(int nb_elem, const uint8_t *addr[], const size_t *len, uint32_t *sha)
{
    if (!HSU_SUPPORT(SHA_256_224))
        return false;
    return hsu_sha_x(HSU_MODE(SHA_256), nb_elem, addr, len, 256/32, sha);
}

bool hsu_sha384(int nb_elem, const uint8_t *addr[], const size_t *len, uint32_t *sha)
{
    if (!HSU_SUPPORT(SHA_512_384))
        return false;
    return hsu_sha_x(HSU_MODE(SHA_384), nb_elem, addr, len, 384/32, sha);
}

bool hsu_sha512(int nb_elem, const uint8_t *addr[], const size_t *len, uint32_t *sha)
{
    if (!HSU_SUPPORT(SHA_512_384))
        return false;
    return hsu_sha_x(HSU_MODE(SHA_512), nb_elem, addr, len, 512/32, sha);
}

/********************* HMAC-SHA-X ********************************************/
/**
 ******************************************************************************
 * @brief Generic function for HMAC-SHA-X computation.
 *
 * Concatenate input vectors in the temporary buffer @ref hsu_sha_buf and then
 * use HSU to compute the requested MAC.
 * Caller must ensure that HSU supports the requested mode and that the ouput
 * buffer is big enough.
 *
 *
 * @param[in]  hsu_hmac_mode  Mode to configure for HSU
 * @param[in]  key            Key to use to compute the MAC.
 * @param[in]  key_len        Length, in bytes, of key buffer.
 * @param[in]  nb_elem        Number of buffer in \p addr.
 * @param[in]  addr           TAble of input vectors.
 * @param[in]  len            Length, in bytes, of each element of \p addr.
 * @param[in]  mac_size       Size, in 32bits words, of the output MAC.
 * @param[out] mac            The computed MAC.
 *
 * @return true if the MAC was correctly computed, false otherwise
 ******************************************************************************
 */
bool hsu_hmac_sha_x(uint32_t hsu_hmac_mode, const uint8_t key[],
                    const size_t key_len, int nb_elem, const uint8_t *addr[],
                    const size_t *len, int mac_size, uint32_t *mac)
{
    int i, length = 0;
    uint32_t ctrl = (hsu_hmac_mode |
                     HSU_FIRST_BUFFER_BIT |
                     HSU_START_BIT);

    for (i = 0 ; i < nb_elem; i++)
    {
        length += len[i];
    }
    if ((length > HSU_SHA_MAX_LEN) || (key_len > HSU_SHA_MAX_LEN))
    {
        TRACE_LMAC(ERR, "Temporary buffer for HMAC computation not big enough (need %d)", length);
        return false;
    }

    HSU_LOCK();

    memcpy(&hsu_sha_buf[0], key, key_len);
    hsu_source_addr_set(CPU2HW(hsu_sha_buf));
    hsu_length_set(key_len);
    hsu_status_clear_set(HSU_DONE_SET_BIT);
    hsu_control_set(ctrl);
    while (!(hsu_status_set_get() & HSU_DONE_SET_BIT)) {}

    // Concatenate input buffers in a single buffer in SHARED RAM
    length = 0;
    for (i = 0 ; i < nb_elem; i++)
    {
        memcpy(&hsu_sha_buf[length], addr[i], len[i]);
        length += len[i];
    }

    ctrl &= ~(HSU_FIRST_BUFFER_BIT);
    ctrl |= HSU_LAST_BUFFER_BIT;
    hsu_source_addr_set(CPU2HW(hsu_sha_buf));
    hsu_length_set(length);
    hsu_status_clear_set(HSU_DONE_SET_BIT);
    hsu_control_set(ctrl);
    while (!(hsu_status_set_get() & HSU_DONE_SET_BIT)) {}

    for (i = 0; i < mac_size; i ++)
    {
        *mac++ = hsu_sha_tab_get(i);
    }
    HSU_UNLOCK();

    return true;
}

bool hsu_hmac_sha1(const uint8_t key[], const size_t key_len, int nb_elem,
                   const uint8_t *addr[], const size_t *len, uint32_t *mac)
{
    if (!HSU_SUPPORT(HMAC_SHA1))
        return false;
    return hsu_hmac_sha_x(HSU_MODE(HMAC_SHA1), key, key_len, nb_elem,
                          addr, len, 160/32, mac);
}

bool hsu_hmac_sha224(const uint8_t key[], const size_t key_len, int nb_elem,
                     const uint8_t *addr[], const size_t *len, uint32_t *mac)
{
    if (!HSU_SUPPORT(HMAC_SHA256_SHA224))
        return false;
    return hsu_hmac_sha_x(HSU_MODE(HMAC_SHA224), key, key_len, nb_elem,
                          addr, len, 224/32, mac);
}

bool hsu_hmac_sha256(const uint8_t key[], const size_t key_len, int nb_elem,
                     const uint8_t *addr[], const size_t *len, uint32_t *mac)
{
    if (!HSU_SUPPORT(HMAC_SHA256_SHA224))
        return false;
    return hsu_hmac_sha_x(HSU_MODE(HMAC_SHA256), key, key_len, nb_elem,
                          addr, len, 256/32, mac);
}

bool hsu_hmac_sha384(const uint8_t key[], const size_t key_len, int nb_elem,
                     const uint8_t *addr[], const size_t *len, uint32_t *mac)
{
    if (!HSU_SUPPORT(HMAC_SHA512_SHA384))
        return false;
    return hsu_hmac_sha_x(HSU_MODE(HMAC_SHA384), key, key_len, nb_elem,
                          addr, len, 384/32, mac);
}

bool hsu_hmac_sha512(const uint8_t key[], const size_t key_len, int nb_elem,
                     const uint8_t *addr[], const size_t *len, uint32_t *mac)
{
    if (!HSU_SUPPORT(HMAC_SHA512_SHA384))
        return false;
    return hsu_hmac_sha_x(HSU_MODE(HMAC_SHA512), key, key_len, nb_elem,
                          addr, len, 512/32, mac);
}

/**
 * @}
 */
#endif /* NX_HSU */
