/**
 ******************************************************************************
 * @file hsu.h
 *
 * @brief Hardware Security Unit module definitions.
 *
 * Copyright (C) RivieraWaves 2017-2019
 *
 ******************************************************************************
 */
#ifndef _HSU_H_
#define _HSU_H_

#include "me_mic.h"

/**
 * @defgroup HSU HSU
 * @ingroup UMAC
 * @brief Structures and Helper functions to use the Hardware Security Unit.
 */

/**
 * @addtogroup HSU
 * @{
 */

/**
 ******************************************************************************
 * @brief Checks if HSU is available
 *
 * It updates internal variable @ref hsu_status.
 * If HSU is not available and @ref NX_HSU is defined to 1 then hsu functions
 * will immediately return and software implementation will be used instead.
 * If HSU is not available and @ref NX_HSU is set to 2, then an error is
 * generated.
 ******************************************************************************
 */
void hsu_init(void);

/**
 ******************************************************************************
 * @brief Computes AES-CMAC on several vectors using HSU
 *
 * All vectors MUST be located in SHARED MEMORY.
 *
 * @param[in] key      Key (16 bytes)
 * @param[in] nb_elem  Number of element in @p addr and @p len tables.
 * @param[in] addr     Table of input vector addresses (HW address)
 * @param[in] len      Table of input vector length (in bytes)
 *
 * @return AES_CMAC value of all vectors concatenated.
 ******************************************************************************
 */
uint64_t hsu_aes_cmac(uint32_t *key, int nb_elem, uint32_t addr[], int len[]);

/**
 ******************************************************************************
 * @brief Initializes TKIP MIC computation using HSU
 *
 * @param[out] mic_calc Pointer to Mic structure that will be initialize
 * @param[in]  mic_key  Key to use for MIC computation
 * @param[in]  aad      Additional Authentication Data vector
 *                      (CPU address, 16 bytes long)
 *                      It must be located in SHARED RAM.
 *
 * @return true if computation has been done by HSU, false otherwise
 ******************************************************************************
 */
bool hsu_michael_init(struct mic_calc *mic_calc, uint32_t *mic_key,
                      uint32_t *aad);

/**
 ******************************************************************************
 * @brief Continues TKIP MIC computation using HSU
 *
 * Continue MIC computation with the provided data.
 *
 * @param[in,out] mic_calc Pointer to Mic structure
 * @param[in]     data     Address of the data buffer (HW address)
 *                         It must be located in SHARED RAM.
 * @param[in]     data_len Length, in bytes, of the data buffer
 *
 * @return true if computation has been done by HSU, false otherwise
 ******************************************************************************
 */
bool hsu_michael_calc(struct mic_calc *mic_calc, uint32_t data,
                      uint32_t data_len);

/**
 ******************************************************************************
 * @brief Ends TKIP MIC computation using HSU
 *
 * Ends TKIP computation by adding padding data
 *
 * @param[in,out] mic_calc Pointer to Mic structure that will be initialize
 *
 * @return true if computation has been done by HSU, false otherwise
 ******************************************************************************
 */
bool hsu_michael_end(struct mic_calc *mic_calc);

/**
 ******************************************************************************
 * @brief Computes the IP checksum on the buffer provided.
 *
 * @param[in]  addr     The address of the data buffer on which the checksum is computed
 * @param[in]  len      Length on which the checksum is computed
 * @param[out] checksum The computed checksum
 *
 * @return true if the checksum was correctly computed, false otherwise
 ******************************************************************************
 */
bool hsu_ip_checksum(uint32_t addr, uint16_t len, uint16_t *checksum);

/**
 ******************************************************************************
 * @brief Computes the SHA1 on the buffers provided.
 *
 * For this function buffers doesn't have to be located in Shared memory but the
 * total size of buffers must be lower than @ref HSU_SHA_MAX_LEN.
 *
 * @param[in]  nb_elem  Number of buffer in \p addr.
 * @param[in]  addr     The address of the data buffer on which the hash is computed
 *                      CPU address is expected.
 * @param[in]  len      Length, in bytes, of each element of \p addr.
 * @param[out] sha      The computed hash. MUST be, at least, 20 bytes long
 *
 * @return true if the hash was correctly computed, false otherwise
 ******************************************************************************
 */
bool hsu_sha1(int nb_elem, const uint8_t *addr[], const size_t *len, uint32_t *sha);

/**
 ******************************************************************************
 * @brief Computes the SHA224 on the buffers provided.
 *
 * For this function buffers doesn't have to be located in Shared memory but the
 * total size of buffers must be lower than @ref HSU_SHA_MAX_LEN.
 *
 * @param[in]  nb_elem  Number of buffer in \p addr.
 * @param[in]  addr     The address of the data buffer on which the hash is computed
 *                      CPU address is expected.
 * @param[in]  len      Length, in bytes, of each element of \p addr.
 * @param[out] sha      The computed hash. MUST be, at least, 28 bytes long
 *
 * @return true if the hash was correctly computed, false otherwise
 ******************************************************************************
 */
bool hsu_sha224(int nb_elem, const uint8_t *addr[], const size_t *len, uint32_t *sha);

/**
 ******************************************************************************
 * @brief Computes the SHA256 on the buffers provided.
 *
 * For this function buffers doesn't have to be located in Shared memory but the
 * total size of buffers must be lower than @ref HSU_SHA_MAX_LEN.
 *
 * @param[in]  nb_elem  Number of buffer in \p addr.
 * @param[in]  addr     The address of the data buffer on which the hash is computed
 *                      CPU address is expected.
 * @param[in]  len      Length, in bytes, of each element of \p addr.
 * @param[out] sha      The computed hash. MUST be, at least, 32 bytes long
 *
 * @return true if the hash was correctly computed, false otherwise
 ******************************************************************************
 */
bool hsu_sha256(int nb_elem, const uint8_t *addr[], const size_t *len, uint32_t *sha);

/**
 ******************************************************************************
 * @brief Computes the SHA384 on the buffers provided.
 *
 * For this function buffers doesn't have to be located in Shared memory but the
 * total size of buffers must be lower than @ref HSU_SHA_MAX_LEN.
 *
 * @param[in]  nb_elem  Number of buffer in \p addr.
 * @param[in]  addr     The address of the data buffer on which the hash is computed
 *                      CPU address is expected.
 * @param[in]  len      Length, in bytes, of each element of \p addr.
 * @param[out] sha      The computed hash. MUST be, at least, 48 bytes long
 *
 * @return true if the hash was correctly computed, false otherwise
 ******************************************************************************
 */
bool hsu_sha384(int nb_elem, const uint8_t *addr[], const size_t *len, uint32_t *sha);

/**
 ******************************************************************************
 * @brief Computes the SHA512 on the buffers provided.
 *
 * For this function buffers doesn't have to be located in Shared memory but the
 * total size of buffers must be lower than @ref HSU_SHA_MAX_LEN.
 *
 * @param[in]  nb_elem  Number of buffer in \p addr.
 * @param[in]  addr     The address of the data buffer on which the hash is computed
 *                      CPU address is expected.
 * @param[in]  len      Length, in bytes, of each element of \p addr.
 * @param[out] sha      The computed hash. MUST be, at least, 64 bytes long
 *
 * @return true if the hash was correctly computed, false otherwise
 ******************************************************************************
 */
bool hsu_sha512(int nb_elem, const uint8_t *addr[], const size_t *len, uint32_t *sha);

/**
 ******************************************************************************
 * @brief Computes the HMAC-SHA1 on the buffers provided.
 *
 * For this function key and data buffers doesn't have to be located in Shared
 * memory but the total size of buffers must be lower than @ref HSU_SHA_MAX_LEN.
 *
 * @param[in]  key      Key to use to compute the MAC.
 *                      CPU address is expected.
 * @param[in]  key_len  Length, in bytes, of key buffer.
 * @param[in]  nb_elem  Number of buffer in \p addr.
 * @param[in]  addr     The address of the data buffer on which the MAC is computed
 *                      CPU address is expected.
 * @param[in]  len      Length, in bytes, of each element of \p addr.
 * @param[out] mac      The computed MAC. MUST be, at least, 20 bytes long
 *
 * @return true if the MAC was correctly computed, false otherwise
 ******************************************************************************
 */
bool hsu_hmac_sha1(const uint8_t key[], const size_t key_len, int nb_elem,
                   const uint8_t *addr[], const size_t *len, uint32_t *mac);

/**
 ******************************************************************************
 * @brief Computes the HMAC-SHA224 on the buffers provided.
 *
 * For this function key and data buffers doesn't have to be located in Shared
 * memory but the total size of buffers must be lower than @ref HSU_SHA_MAX_LEN.
 *
 * @param[in]  key      Key to use to compute the MAC.
 *                      CPU address is expected.
 * @param[in]  key_len  Length, in bytes, of key buffer.
 * @param[in]  nb_elem  Number of buffer in \p addr.
 * @param[in]  addr     The address of the data buffer on which the MAC is computed
 *                      CPU address is expected.
 * @param[in]  len      Length, in bytes, of each element of \p addr.
 * @param[out] mac      The computed MAC. MUST be, at least, 28 bytes long
 *
 * @return true if the MAC was correctly computed, false otherwise
 ******************************************************************************
 */
bool hsu_hmac_sha224(const uint8_t key[], const size_t key_len, int nb_elem,
                     const uint8_t *addr[], const size_t *len, uint32_t *mac);

/**
 ******************************************************************************
 * @brief Computes the HMAC-SHA256 on the buffers provided.
 *
 * For this function key and data buffers doesn't have to be located in Shared
 * memory but the total size of buffers must be lower than @ref HSU_SHA_MAX_LEN.
 *
 * @param[in]  key      Key to use to compute the MAC.
 *                      CPU address is expected.
 * @param[in]  key_len  Length, in bytes, of key buffer.
 * @param[in]  nb_elem  Number of buffer in \p addr.
 * @param[in]  addr     The address of the data buffer on which the MAC is computed
 *                      CPU address is expected.
 * @param[in]  len      Length, in bytes, of each element of \p addr.
 * @param[out] mac      The computed MAC. MUST be, at least, 32 bytes long
 *
 * @return true if the MAC was correctly computed, false otherwise
 ******************************************************************************
 */
bool hsu_hmac_sha256(const uint8_t key[], const size_t key_len, int nb_elem,
                     const uint8_t *addr[], const size_t *len, uint32_t *mac);

/**
 ******************************************************************************
 * @brief Computes the HMAC-SHA384 on the buffers provided.
 *
 * For this function key and data buffers doesn't have to be located in Shared
 * memory but the total size of buffers must be lower than @ref HSU_SHA_MAX_LEN.
 *
 * @param[in]  key      Key to use to compute the MAC.
 *                      CPU address is expected.
 * @param[in]  key_len  Length, in bytes, of key buffer.
 * @param[in]  nb_elem  Number of buffer in \p addr.
 * @param[in]  addr     The address of the data buffer on which the MAC is computed
 *                      CPU address is expected.
 * @param[in]  len      Length, in bytes, of each element of \p addr.
 * @param[out] mac      The computed MAC. MUST be, at least, 48 bytes long
 *
 * @return true if the MAC was correctly computed, false otherwise
 ******************************************************************************
 */
bool hsu_hmac_sha384(const uint8_t key[], const size_t key_len, int nb_elem,
                     const uint8_t *addr[], const size_t *len, uint32_t *mac);

/**
 ******************************************************************************
 * @brief Computes the HMAC-SHA512 on the buffers provided.
 *
 * For this function key and data buffers doesn't have to be located in Shared
 * memory but the total size of buffers must be lower than @ref HSU_SHA_MAX_LEN.
 *
 * @param[in]  key      Key to use to compute the MAC.
 *                      CPU address is expected.
 * @param[in]  key_len  Length, in bytes, of key buffer.
 * @param[in]  nb_elem  Number of buffer in \p addr.
 * @param[in]  addr     The address of the data buffer on which the MAC is computed
 *                      CPU address is expected.
 * @param[in]  len      Length, in bytes, of each element of \p addr.
 * @param[out] mac      The computed MAC. MUST be, at least, 64 bytes long
 *
 * @return true if the MAC was correctly computed, false otherwise
 ******************************************************************************
 */
bool hsu_hmac_sha512(const uint8_t key[], const size_t key_len, int nb_elem,
                     const uint8_t *addr[], const size_t *len, uint32_t *mac);

/**
 * @}
 */
#endif /* _HSU_H_ */
