/**
 ****************************************************************************************
 *
 * @file me.c
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 * @brief Definition of initialization functions for the UMAC ME modules.
 *
 ****************************************************************************************
 */

/** @addtogroup ME
 * @{
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */

#include "co_int.h"
#include "co_bool.h"
#include "me.h"
#include "sm.h"
#include "mm.h"
#if NX_BEACONING
#include "apm.h"
#endif
#include "scanu.h"
#include "bam.h"
#include "vif_mgmt.h"
#include "ps.h"
#include "version.h"
#include "reg_mac_core.h"
#include "txl_frame.h"
#if NX_HSU
#include "hsu.h"
#endif

/*
 * PRIVATE VARIABLES
 ****************************************************************************************
 */

struct me_env_tag me_env;

/*
 * PRIVATE FUNCTIONS DEFINITION
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @brief Initialize content of the ME environment. State of the ME task is set to idle.
 ****************************************************************************************
 */
static void me_env_init(void)
{
    // Reset the ME environment
    memset(&me_env, 0, sizeof(me_env));

    // Set the ME task state
    ke_state_set(TASK_ME, ME_IDLE);
}

/*
 * PUBLIC FUNCTIONS DEFINITION
 ****************************************************************************************
 */

void me_init(void)
{
    // Initialize the environment
    me_env_init();

    // Initialize scan module
    scanu_init();

    // initialize all ME components modules and environment
    #if NX_BEACONING
    // reset AP manager
    apm_init();
    #endif

    // reset STA manager
    sm_init();

    // Reset the Block ACK manager
    bam_init();

    #if RW_MESH_EN
    mesh_init();
    #endif //RW_MESH_EN

    #if NX_HSU
    hsu_init();
    #endif
}

struct mac_chan_def *me_freq_to_chan_ptr(uint8_t band, uint16_t freq)
{
    int i, chan_cnt;
    struct mac_chan_def *chan;

    // Get the correct channel table
    chan = (band == PHY_BAND_2G4) ? me_env.chan.chan2G4 : me_env.chan.chan5G;
    chan_cnt = (band == PHY_BAND_2G4) ? me_env.chan.chan2G4_cnt : me_env.chan.chan5G_cnt;

    for (i = 0; i < chan_cnt; i++)
    {
        if (chan[i].freq == freq)
            return &chan[i];
    }

    return NULL;
}

void me_init_chan(struct mac_chan_op *chan)
{
    struct mac_chan_def *chan_def = me_freq_to_chan_ptr(chan->band, chan->prim20_freq);

    if (chan == NULL)
    {
        ASSERT_WARN(0);
        chan->tx_power = MM_DEFAULT_TX_POWER;
        chan->flags = 0;
    }

    chan->tx_power = chan_def->tx_power;
    chan->flags = chan_def->flags;
}

/// @}

