/**
 ****************************************************************************************
 *
 * @file me_task.c
 *
 * @brief The UMAC's ME (MAC Layer Management Entity) module implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

#include "me_task.h"
#include "me.h"

#include "mm.h"
#include "me_utils.h"
#include "scan_task.h"
//#include "tpm.h"
#include "dbg.h"
#include "rxu_task.h"
#include "sta_mgmt.h"
#include "vif_mgmt.h"
#include "ps.h"
#include "co_endian.h"

/** @addtogroup TASK_ME
 * @{
 */


/**
 ****************************************************************************************
 * @brief ME module message handler.
 * This function handles the ME_CONFIG_REQ message from host.
 * This message is sent once after FW reset, and provides fw configuration.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_config_req_handler(ke_msg_id_t const msgid,
                      struct me_config_req const *param,
                      ke_task_id_t const dest_id,
                      ke_task_id_t const src_id)
{
    #if NX_POWERSAVE
    struct mm_set_ps_mode_req *ps = KE_MSG_ALLOC(MM_SET_PS_MODE_REQ, TASK_MM,
                                                 TASK_ME, mm_set_ps_mode_req);
    #endif

    me_env.capa_flags = 0;

    // Copy the HT, VHT and HE capabilities
    if (param->ht_supp)
    {
        LOCAL_CAPA_SET(HT);
        me_env.ht_cap = param->ht_cap;
    }
    #if NX_VHT
    if (param->vht_supp)
    {
        LOCAL_CAPA_SET(VHT);
        me_env.vht_cap = param->vht_cap;
    }
    #endif
    #if NX_HE
    if (param->he_supp)
    {
        LOCAL_CAPA_SET(HE);
        me_env.he_cap = param->he_cap;
    }
    if (param->he_ul_on)
    {
        LOCAL_CAPA_SET(OFDMA_UL);
    }
    #endif

    // Set maximum supported bandwidth
    me_env.phy_bw_max = param->phy_bw_max;

    ke_msg_send_basic(ME_CONFIG_CFM, src_id, dest_id);

    // Perform additional checks depending on HT/VHT/HE support
    if (LOCAL_CAPA(HT))
    {
        // Set the maximum number of NSS supported when using STBC for TX
        me_env.stbc_nss = (phy_get_nss()+1)/2;

        // Get the maximum BW supported
        #if NX_HE
        if (LOCAL_CAPA(HE))
        {
            bool stbc_tx_under_80 = HE_PHY_CAPA_BIT_IS_SET(&me_env.he_cap, STBC_TX_UNDER_80MHZ);
            bool stbc_tx_above_80 = HE_PHY_CAPA_BIT_IS_SET(&me_env.he_cap, STBC_TX_ABOVE_80MHZ);

            if (me_env.phy_bw_max > PHY_CHNL_BW_80)
                me_env.he_stbc_nss = stbc_tx_under_80 & stbc_tx_above_80 ? 1 : 0;
            else
                me_env.he_stbc_nss = stbc_tx_under_80 ? 1 : 0;
        }
        #endif
    }
    else
    {
        me_env.stbc_nss = 0;
    }

    // Set the lifetime of packets sent under BA agreement
    me_env.tx_lft = param->tx_lft;

    #if NX_POWERSAVE
    // Save PS mode flags
    me_env.ps_on = param->ps_on;
    me_env.ps_mode = param->dpsm ? PS_MODE_ON_DYN : PS_MODE_ON;

    // Enable PS mode if required
    if (me_env.ps_on)
    {
        // No requester ID
        me_env.requester_id = TASK_NONE;

        // Send the message to the LMAC
        ps->new_state = me_env.ps_mode;
        ke_msg_send(ps);

        // Move to BUSY state
        ke_state_set(TASK_ME, ME_BUSY);
    }
    #endif

    #if (NX_ANT_DIV)
    // Send message MM_ANT_DIV_INIT_REQ
    struct mm_ant_div_init_req *req = KE_MSG_ALLOC(MM_ANT_DIV_INIT_REQ,
                                                   TASK_MM, TASK_ME,
                                                   mm_ant_div_init_req);
    req->enable = param->ant_div_on;
    ke_msg_send(req);
    #endif //(NX_ANT_DIV)

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief ME module message handler.
 * This function handles the ME_CHAN_CONFIG_REQ from host.
 * This message provides the list of available channels.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_chan_config_req_handler(ke_msg_id_t const msgid,
                           struct me_chan_config_req const *param,
                           ke_task_id_t const dest_id,
                           ke_task_id_t const src_id)
{
    // Copy the channel list
    me_env.chan = *param;

    ke_msg_send_basic(ME_CHAN_CONFIG_CFM, src_id, dest_id);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief ME module message handler.
 * This function handles the ME_SET_CONTROL_PORT_REQ message from host.
 * This message configures the status of a STA's port :
 *  - OPEN: The sta can transmit any kind of data (e.g. after successful 4-way handshake)
 *  - CONTROLLED: The sta can only transmit a specific kind of data
                  (e.g. during 4-way handshake)
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_set_control_port_req_handler(ke_msg_id_t const msgid,
                                struct me_set_control_port_req const *param,
                                ke_task_id_t const dest_id,
                                ke_task_id_t const src_id)
{
    // Get the STA entry
    struct sta_info_tag *sta = &sta_info_tab[param->sta_idx];
    // Get the associated VIF entry
    struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];

    // Copy the HT and VHT capabilities
    sta->ctrl_port_state = param->control_port_open?PORT_OPEN:PORT_CONTROLED;

    // If port is open for a STA, reenable the PS mode now that the association is complete
    // do not reenable PS mode in case of association with the TDLS station
    if ((vif->type == VIF_STA) && (sta->ctrl_port_state == PORT_OPEN)
        #if NX_TDLS
         && !sta->is_tdls
        #endif
       )
    {
        struct me_set_ps_disable_req *ps = KE_MSG_ALLOC(ME_SET_PS_DISABLE_REQ, TASK_ME, TASK_SM,
                                                        me_set_ps_disable_req);
        ps->ps_disable = false;
        ps->vif_idx = sta->inst_nbr;

        ke_msg_send(ps);
    }

    ke_msg_send_basic(ME_SET_CONTROL_PORT_CFM, src_id, dest_id);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief ME module message handler.
 * This function handles the ME_STA_ADD_REQ message host.
 * This message provides information about a STA we will transmit data to (client of
 * an AP, a TDLS peer or a peer Mesh STA (if Mesh Peering Management protocol is
 * handled in userspace)).
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_sta_add_req_handler(ke_msg_id_t const msgid,
                       struct me_sta_add_req const *param,
                       ke_task_id_t const dest_id,
                       ke_task_id_t const src_id)
{
    // Allocate the response structure
    struct mm_sta_add_req sta_add_req;
    struct me_sta_add_cfm *rsp = KE_MSG_ALLOC(ME_STA_ADD_CFM, src_id, dest_id, me_sta_add_cfm);
    uint16_t ampdu_size_max_ht = 0;
    uint32_t ampdu_size_max_vht = 0;
    uint32_t ampdu_size_max_he = 0;
    uint8_t ampdu_spacing_min = 0;
    uint8_t hw_sta_idx;
    uint8_t pm_state = rxu_cntrl_get_pm();
    struct vif_info_tag *vif = &vif_info_tab[param->vif_idx];
    uint32_t paid_gid = 0;

    #if (NX_TDLS)
    // In case of TDLS station disable PS
    if (param->tdls_sta)
    {
        struct me_set_ps_disable_req *ps = KE_MSG_ALLOC(ME_SET_PS_DISABLE_REQ, TASK_ME, TASK_SM,
                                                        me_set_ps_disable_req);
        ps->ps_disable = true;
        ps->vif_idx = param->vif_idx;

        ke_msg_send(ps);
    }
    #endif

    // Compute HT, VHT and HE A-MPDU parameters
    if (param->flags & STA_HT_CAPA)
    {
        struct mac_htcapability const *ht_cap = &param->ht_cap;
        struct mac_vhtcapability const *vht_cap = NULL;
        struct mac_hecapability const *he_cap = NULL;

        #if NX_VHT
        if (param->flags & STA_VHT_CAPA)
        {
            vht_cap = &param->vht_cap;
            // Compute the 'unicast' VHT partial AID and GID
            switch (vif->type)
            {
                // to TDLS STA
                case VIF_STA:
                    paid_gid = mac_paid_gid_ap_compute(&vif->bss_info.bssid,
                                                       param->aid);
                    break;
                // to MESH STA
                case VIF_MESH_POINT:
                    paid_gid = mac_paid_gid_sta_compute(&param->mac_addr);
                    break;
                // from AP to associated STA
                case VIF_AP:
                    paid_gid = mac_paid_gid_ap_compute(&vif->mac_addr, param->aid);
                    break;
                default:
                    break;
            }
        }
        #endif

        #if NX_HE
        if (param->flags & STA_HE_CAPA)
        {
            he_cap = &param->he_cap;
        }
        #endif

        me_get_ampdu_params(ht_cap, vht_cap, he_cap, &ampdu_size_max_ht,
                            &ampdu_size_max_vht, &ampdu_size_max_he,
                            &ampdu_spacing_min);
    }

    sta_add_req.paid_gid = paid_gid;
    sta_add_req.mac_addr = param->mac_addr;
    sta_add_req.bssid_index = 0;
    sta_add_req.max_bssid_ind = 0;
    sta_add_req.ampdu_size_max_ht = ampdu_size_max_ht;
    sta_add_req.ampdu_size_max_vht = ampdu_size_max_vht;
    sta_add_req.ampdu_spacing_min = ampdu_spacing_min;
    sta_add_req.inst_nbr = param->vif_idx;
    #if (NX_TDLS)
    sta_add_req.tdls_sta = param->tdls_sta;
    sta_add_req.tdls_initiator = param->tdls_initiator;
    sta_add_req.tdls_chsw_allowed = param->tdls_chsw_allowed;
    #endif

    // Register the new station
    rsp->status = mm_sta_add(&sta_add_req, &rsp->sta_idx, &hw_sta_idx);

    // If station was successfully allocated
    if (rsp->status == CO_OK)
    {
        struct sta_info_tag *sta = &sta_info_tab[rsp->sta_idx];
        struct sta_capa_info *info = &sta->info;
        struct me_bss_info *bss = &vif->bss_info;
        bool smps = false;
        // Initialize the capabilities of the new STA
        info->rate_set = param->rate_set;
        if (param->flags & STA_QOS_CAPA)
        {
            STA_CAPA_SET(sta, QOS);

            if ((param->flags & STA_HT_CAPA) && LOCAL_CAPA(HT))
            {
                STA_CAPA_SET(sta, HT);
                info->ht_cap = param->ht_cap;
                #if NX_HE
                if ((param->flags & STA_HE_CAPA) && LOCAL_CAPA(HE))
                {
                    STA_CAPA_SET(sta, HE);
                    info->he_cap = param->he_cap;
                    #if NX_BEACONING
                    if (LOCAL_CAPA(OFDMA_UL))
                        txl_he_start_trigger_scheduler(sta);
                    #endif
                }
                #endif
                #if NX_VHT
                if ((param->flags & STA_VHT_CAPA) &&
                    (LOCAL_CAPA(VHT) || STA_CAPA(sta, HE)))
                {
                    STA_CAPA_SET(sta, VHT);
                    info->vht_cap = param->vht_cap;
                }
                #endif

                // Configure some HT and VHT parameters of the peer STA
                smps = me_set_sta_ht_vht_param(sta, bss);
            }
        }

        #if NX_MFP
        if (param->flags & STA_MFP_CAPA)
        {
            STA_CAPA_SET(sta, MFP);
        }
        #endif

        // Initialize U-APSD parameters
        info->uapsd_queues = param->uapsd_queues;
        info->max_sp_len   = param->max_sp_len;
        sta->aid = param->aid;

        // Initialize the RC for this STA
        me_init_rate(sta);

        // Check if STA has notified operation mode in its AssocReq
        if ((param->flags & STA_OPMOD_NOTIF) && !(param->opmode & MAC_OPMODE_RXNSS_TYPE_BIT))
        {
            uint8_t bw = (param->opmode & MAC_OPMODE_BW_MSK) >> MAC_OPMODE_BW_OFT;
            uint8_t nss = (param->opmode & MAC_OPMODE_RXNSS_MSK) >> MAC_OPMODE_RXNSS_OFT;

            // Update maximum supported bandwidth
            me_sta_bw_nss_max_upd(sta->staid, bw, nss);
        }

        // Check if STA is using SMPS
        if (smps)
        {
            // Allow only 1 SS to be transmitted
            me_sta_bw_nss_max_upd(sta->staid, 0xFF, 0);
        }

        // Initialize tx power on first transmit
        sta->pol_tbl.upd_field |= CO_BIT(STA_MGMT_POL_UPD_TX_POWER);

        // Init port status
        sta->ctrl_port_state = (vif->flags & CONTROL_PORT_HOST)?PORT_CONTROLED:PORT_OPEN;
        sta->ctrl_port_ethertype = co_ntohs(vif->u.ap.ctrl_port_ethertype);

        // Get the PM state of the station in case it was catched by the PM monitor
        rsp->pm_state = pm_state;

        if (rsp->pm_state)
        {
            // Set the PM state of the station
            sta->ps_state = pm_state;

            // Update the number of PS stations
            if (!vif->u.ap.ps_sta_cnt)
            {
                PROF_PS_BCMC_STATE_SET();
                PROF_PS_STATE_VAL_SET(PS_MODE_ON);
                mm_ps_change_ind(VIF_TO_BCMC_IDX(vif->index), PS_MODE_ON);
                PROF_PS_BCMC_STATE_CLR();
            }

            // Update the number of PS stations
            vif->u.ap.ps_sta_cnt++;

            #if (NX_P2P_GO)
            p2p_go_ps_state_update(vif);
            #endif //(NX_P2P_GO)
        }
    }

    // Send the confirmation
    ke_msg_send(rsp);

    return (KE_MSG_CONSUMED);
}


/**
 ****************************************************************************************
 * @brief ME module message handler.
 * This function handles the ME_STA_DEL_REQ message from host.
 * This message deletes a STA added via ME_STA_ADD_REQ.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_sta_del_req_handler(ke_msg_id_t const msgid,
                       struct me_sta_del_req const *param,
                       ke_task_id_t const dest_id,
                       ke_task_id_t const src_id)
{
    struct mm_sta_del_req *sta_del = KE_MSG_ALLOC(MM_STA_DEL_REQ, TASK_MM,
                                                  TASK_ME, mm_sta_del_req);
    #if NX_AMPDU_TX || NX_REORD
    // Release all the Block Ack agreement
    bam_delete_all_ba_agg(param->sta_idx);
    #endif

    // Send the message to the LowerMAC
    sta_del->sta_idx = param->sta_idx;
    ke_msg_send(sta_del);

    // In case of TDLS station enable PS
    if (param->tdls_sta)
    {
        struct me_set_ps_disable_req *ps = KE_MSG_ALLOC(ME_SET_PS_DISABLE_REQ, TASK_ME, TASK_SM,
                                                        me_set_ps_disable_req);
        ps->ps_disable = false;
        ps->vif_idx = sta_mgmt_get_vif_idx(param->sta_idx);
        ke_msg_send(ps);
    }

    // Send the confirmation to the upper MAC
    ke_msg_send_basic(ME_STA_DEL_CFM, src_id, dest_id);

    return (KE_MSG_CONSUMED);
}


/**
 ****************************************************************************************
 * @brief ME module message handler.
 * This function handles the ME_TRAFFIC_IND_REQ from host.
 * This message indicates if traffic is available or not for a sta.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_traffic_ind_req_handler(ke_msg_id_t const msgid,
                           struct me_traffic_ind_req const *param,
                           ke_task_id_t const dest_id,
                           ke_task_id_t const src_id)
{
    struct sta_info_tag *sta = &sta_info_tab[param->sta_idx];
    // Indicate if MM_TIM_UPDATE_REQ message must be sent
    bool send_req = true;

    if (param->uapsd)
    {
        // Update the traffic flag in the STA entry
        if (param->tx_avail)
            sta->traffic_avail |= UAPSD_TRAFFIC_HOST;
        else
            sta->traffic_avail &= ~UAPSD_TRAFFIC_HOST;

        if (sta->info.uapsd_queues != MAC_QOS_INFO_STA_UAPSD_ENABLED_ALL)
        {
            send_req = false;
        }
    }
    else
    {
        if (param->tx_avail)
            sta->traffic_avail |= PS_TRAFFIC_HOST;
        else
            sta->traffic_avail &= ~PS_TRAFFIC_HOST;
    }

    if (send_req)
    {
        struct mm_tim_update_req *tim = KE_MSG_ALLOC(MM_TIM_UPDATE_REQ, TASK_MM,
                                                     TASK_ME, mm_tim_update_req);

        tim->aid = sta->aid;
        tim->inst_nbr = sta->inst_nbr;
        tim->tx_avail = param->tx_avail;

        ke_msg_send(tim);
    }

    // Send the confirmation to the upper MAC
    ke_msg_send_basic(ME_TRAFFIC_IND_CFM, src_id, dest_id);

    return (KE_MSG_CONSUMED);
}


/**
 ****************************************************************************************
 * @brief ME module message handler.
 * This function handles the ME_SET_ACTIVE_REQ message from STA, AP or MESH vif.
 * This message requests the ME module to enable/disable the HW. If HW state is not
 * the one asked, the request is forwarded to MM task.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_set_active_req_handler(ke_msg_id_t const msgid,
                          struct me_set_active_req const *param,
                          ke_task_id_t const dest_id,
                          ke_task_id_t const src_id)
{
    // Check if the request can be handled immediately
    if (ke_state_get(TASK_ME) == ME_BUSY)
        return (KE_MSG_SAVED);

    // Check if the current state is already OK
    if ((me_env.active_vifs && param->active) ||
        (!me_env.active_vifs && !param->active))
    {
        // Update the bit corresponding to the VIF
        if (param->active)
            me_env.active_vifs |= CO_BIT(param->vif_idx);

        // Send the confirmation immediately
        ke_msg_send_basic(ME_SET_ACTIVE_CFM, src_id, dest_id);
    }
    else
    {
        struct mm_set_idle_req *active =  KE_MSG_ALLOC(MM_SET_IDLE_REQ, TASK_MM,
                                                       dest_id, mm_set_idle_req);

        // Update the bit corresponding to the VIF
        if (param->active)
            me_env.active_vifs |= CO_BIT(param->vif_idx);
        else
            me_env.active_vifs &= ~CO_BIT(param->vif_idx);

        // Save the ID of the requester
        me_env.requester_id = src_id;

        // Send the message to the LMAC
        active->hw_idle = (me_env.active_vifs == 0);
        ke_msg_send(active);

        // Move to BUSY state
        ke_state_set(dest_id, ME_BUSY);
    }

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief ME module message handler.
 * This function handles MM_SET_IDLE_CFM confirmation from MM task.
 * Message sent by MM task once the HW reached the requested state (IDLE or ACTIVE).
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_idle_cfm_handler(ke_msg_id_t const msgid,
                        void const *param,
                        ke_task_id_t const dest_id,
                        ke_task_id_t const src_id)
{
    // Sanity check - This message can be received only in the BUSY state
    ASSERT_ERR(ke_state_get(dest_id) == ME_BUSY);

    // Send the confirmation to the requester
    if (me_env.requester_id != TASK_NONE)
        ke_msg_send_basic(ME_SET_ACTIVE_CFM, me_env.requester_id, dest_id);

    // Back to IDLE state
    ke_state_set(dest_id, ME_IDLE);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief ME module message handler.
 * This function handles the ME_SET_PS_DISABLE_REQ from STA, AP or MESH vif.
 * This message requests the UMAC to disable or re-enable Power Save mode.
 * If PS state is not the one asked, the request is forwarded to MM task.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_set_ps_disable_req_handler(ke_msg_id_t const msgid,
                              struct me_set_ps_disable_req const *param,
                              ke_task_id_t const dest_id,
                              ke_task_id_t const src_id)
{
    #if NX_POWERSAVE
    uint8_t ps_disable_vifs = me_env.ps_disable_vifs;

    // Update the bit corresponding to the VIF
    if (param->ps_disable)
        me_env.ps_disable_vifs |= CO_BIT(param->vif_idx);
    else
        me_env.ps_disable_vifs &= ~CO_BIT(param->vif_idx);

    // Check if PS is enabled or not
    if (!me_env.ps_on)
    {
        // Send the confirmation immediately
        ke_msg_send_basic(ME_SET_PS_DISABLE_CFM, src_id, dest_id);

        return (KE_MSG_CONSUMED);
    }

    // Check if the request can be handled immediately
    if (ke_state_get(TASK_ME) == ME_BUSY)
        return (KE_MSG_SAVED);

    // Check if the current state is already OK
    if ((ps_disable_vifs && param->ps_disable) ||
        (!ps_disable_vifs && !param->ps_disable))
    {
        // Send the confirmation immediately
        ke_msg_send_basic(ME_SET_PS_DISABLE_CFM, src_id, dest_id);
    }
    else
    {
        struct mm_set_ps_mode_req *ps = KE_MSG_ALLOC(MM_SET_PS_MODE_REQ, TASK_MM,
                                                     dest_id, mm_set_ps_mode_req);

        // Save the ID of the requester
        me_env.requester_id = src_id;

        // Send the message to the LMAC
        ps->new_state = (me_env.ps_disable_vifs == 0)?me_env.ps_mode:PS_MODE_OFF;
        ke_msg_send(ps);

        // Move to BUSY state
        ke_state_set(dest_id, ME_BUSY);
    }
    #else
    // Send the confirmation immediately
    ke_msg_send_basic(ME_SET_PS_DISABLE_CFM, src_id, dest_id);
    #endif

    return (KE_MSG_CONSUMED);
}

#if NX_POWERSAVE
/**
 ****************************************************************************************
 * @brief ME module message handler.
 * This function handles the ME_SET_PS_MODE_REQ message from host.
 * This message provides the new PS mode to set.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_set_ps_mode_req_handler(ke_msg_id_t const msgid,
                           struct me_set_ps_mode_req const *param,
                           ke_task_id_t const dest_id,
                           ke_task_id_t const src_id)
{
    struct mm_set_ps_mode_req *ps;

    // Check if the request can be handled immediately
    if (ke_state_get(TASK_ME) == ME_BUSY)
        return (KE_MSG_SAVED);


    me_env.ps_on = param->ps_state > 0 ? true : false;

    if (me_env.ps_disable_vifs != 0)
    {
        ke_msg_send_basic(ME_SET_PS_MODE_CFM, src_id, dest_id);
        return (KE_MSG_CONSUMED);
    }

    ps = KE_MSG_ALLOC(MM_SET_PS_MODE_REQ, TASK_MM, TASK_ME, mm_set_ps_mode_req);

    // No requester ID
    me_env.requester_id = TASK_NONE;

    // Send the message to the LMAC
    ps->new_state = (me_env.ps_on) ? me_env.ps_mode : PS_MODE_OFF;
    ke_msg_send(ps);

    // Move to BUSY state
    ke_state_set(TASK_ME, ME_BUSY);

    ke_msg_send_basic(ME_SET_PS_MODE_CFM, src_id, dest_id);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief ME module message handler.
 * This function handles MM_SET_PS_MODE_CFM confirmation from the MM task.
 * This message informs the ME module that the PS mode is the requested one.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_ps_mode_cfm_handler(ke_msg_id_t const msgid,
                           void const *param,
                           ke_task_id_t const dest_id,
                           ke_task_id_t const src_id)
{
    // Sanity check - This message can be received only in the BUSY state
    ASSERT_ERR(ke_state_get(dest_id) == ME_BUSY);

    // Send the confirmation to the requester
    if (me_env.requester_id != TASK_NONE)
        ke_msg_send_basic(ME_SET_PS_DISABLE_CFM, me_env.requester_id, dest_id);

    // Back to IDLE state
    ke_state_set(dest_id, ME_IDLE);

    return (KE_MSG_CONSUMED);
}
#endif

/**
 ****************************************************************************************
 * @brief RC module message handler.
 * This function handles the ME_RC_STATS_REQ message from host.
 * This message asks fw for statistics about Rate Control module.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_rc_stats_req_handler(ke_msg_id_t const msgid,
                       struct me_rc_stats_req const *param,
                       ke_task_id_t const dest_id,
                       ke_task_id_t const src_id)
{
    // Allocate the response structure
    struct me_rc_stats_cfm *rsp = KE_MSG_ALLOC(ME_RC_STATS_CFM, src_id, dest_id, me_rc_stats_cfm);

    // retrieve statistics for the requested station (number sta_idx)
    struct sta_info_tag *sta = &sta_info_tab[param->sta_idx];
    struct sta_pol_tbl_cntl *rc = &sta->pol_tbl;
    struct rc_sta_stats *rc_ss = rc->sta_stats;
    uint16_t i;

    // set station index
    rsp->sta_idx = param->sta_idx;
    if (rc_ss)
    {
        // set number of samples
        rsp->no_samples = rc_ss->no_samples;
        // set number of MPDUs transmitted (per sampling interval)
        rsp->ampdu_len = rc_ss->ampdu_len;
        // set number of AMPDUs transmitted (per sampling interval)
        rsp->ampdu_packets = rc_ss->ampdu_packets;
        // set average number of MPDUs in each AMPDU frame (EWMA)
        rsp->avg_ampdu_len = rc_ss->avg_ampdu_len;
        // set step 0 of the retry chain
        rsp->sw_retry_step = rc_ss->sw_retry_step;
        // set current value of the trial TX countdown
        rsp->sample_wait = rc_ss->sample_wait;
        // copy the retry chain steps
        memcpy(rsp->retry_step_idx, rc_ss->retry_step_idx, sizeof(rsp->retry_step_idx));
        // copy the statistics (only the first RC_MAX_N_SAMPLE)
        memcpy(&rsp->rate_stats[0], &rc_ss->rate_stats[0], sizeof(rsp->rate_stats) -
                                                    NX_HE * sizeof(struct rc_rate_stats));
        // calculate TP
        for (i = 0; i < rc_ss->no_samples; i++)
        {
            rsp->tp[i] = rc_calc_tp(rc_ss, i);
        }
        #if NX_HE
        GLOBAL_INT_DISABLE();
        // Copy the HE statistics - must be done with interrupt disabled as some parts
        // are modified under interrupt
        rsp->rate_stats[RC_HE_STATS_IDX] = rc_ss->rate_stats[RC_HE_STATS_IDX];
        // calculate HE TB throughput
        rsp->tp[RC_HE_STATS_IDX] = rc_calc_tp(rc_ss, RC_HE_STATS_IDX);
        GLOBAL_INT_RESTORE();
        #endif
    }
    else
    {
        rsp->no_samples = 0;
    }

    // Send the confirmation
    ke_msg_send(rsp);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief RC module message handler.
 * This function handles ME_RC_SET_RATE_REQ from host.
 * This function sets fixed_rate_cfg in the RC structure.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_rc_set_rate_req_handler(ke_msg_id_t const msgid,
                           struct me_rc_set_rate_req const *param,
                           ke_task_id_t const dest_id,
                           ke_task_id_t const src_id)
{
    struct sta_info_tag *sta = &sta_info_tab[param->sta_idx];
    struct sta_pol_tbl_cntl *pt = &sta->pol_tbl;
    struct rc_sta_stats *rc_ss = pt->sta_stats;
    ASSERT_ERR(rc_ss != NULL);
    uint16_t fixed_rate_cfg = param->fixed_rate_cfg;

    if (fixed_rate_cfg != RC_FIXED_RATE_NOT_SET)
    {
        if (rc_check_fixed_rate_config(rc_ss, fixed_rate_cfg))
        {
            // Set the fixed rate configuration in the RC structure
            rc_ss->fixed_rate_cfg = fixed_rate_cfg;
            // Update the fixed rate request flag
            rc_ss->info &= ~RC_FIX_RATE_STATUS_MASK;
            rc_ss->info |= RC_FIX_RATE_REQ_MASK;
        }
    }
    else
    {
        // Set the fixed rate configuration in the RC structure
        rc_ss->fixed_rate_cfg = fixed_rate_cfg;
        rc_ss->info &= ~RC_FIX_RATE_STATUS_MASK;
        // Check if the rates should be updated
        if (rc_ss->info & RC_FIX_RATE_UPD_SS_REQ_MASK)
        {
            // Update the rates of rate control algorithm
            rc_update_sample_table(param->sta_idx);
            rc_ss->info &= ~RC_FIX_RATE_UPD_SS_REQ_MASK;
        }
    }

    return (KE_MSG_CONSUMED);
}


/**
 ****************************************************************************************
 * @brief ME module message handler.
 * This function handles the ME_CONFIG_MONITOR_REQ message from the host.
 * This message requests the ME module to configure the monitor interface context.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_config_monitor_req_handler(ke_msg_id_t const msgid,
                              struct me_config_monitor_req const *param,
                              ke_task_id_t const dest_id,
                              ke_task_id_t const src_id)
{
    struct me_config_monitor_cfm *rsp;
    struct vif_info_tag *vif;

    // Check if the request can be handled immediately
    if (ke_state_get(TASK_ME) == ME_BUSY)
        return (KE_MSG_SAVED);

    // Allocate the response structure
    rsp = KE_MSG_ALLOC(ME_CONFIG_MONITOR_CFM, src_id, dest_id, me_config_monitor_cfm);

    // Sanity check - One monitor interface shall exist
    ASSERT_ERR(vif_mgmt_env.monitor_vif != INVALID_VIF_IDX);
    vif = &vif_info_tab[vif_mgmt_env.monitor_vif];

    #if NX_UF_EN
    if (phy_uf_supported())
        phy_uf_enable(param->uf);
    #endif /* NX_UF_EN */

    if (!vif->chan_ctxt)
    {
        rsp->chan_index = CHAN_CTXT_UNUSED;
        ke_msg_send(rsp);
        return (KE_MSG_CONSUMED);
    }

    rsp->chan_index = vif->chan_ctxt->idx;

    // Check if there is no active managed interface
    if ((param->chan_set) && (co_list_cnt(&vif_mgmt_env.used_list) == 1))
    {
        struct mm_chan_ctxt_update_req *req = KE_MSG_ALLOC(MM_CHAN_CTXT_UPDATE_REQ, TASK_MM, TASK_ME,
                                                               mm_chan_ctxt_update_req);

        // Fill-in the channel context addition request
        req->chan_index = vif->chan_ctxt->idx;
        req->chan = param->chan;

        // Send the message
        ke_msg_send(req);

        // The channel has been updated as requested
        rsp->chan = param->chan;
    }
    else
    {
        // The channel has not been updated, inform host which one is in use.
        rsp->chan = vif->chan_ctxt->channel;
    }

    ke_msg_send(rsp);

    return (KE_MSG_CONSUMED);
}

/// DEFAULT handler definition.
const struct ke_msg_handler me_default_state[] =
{
    {ME_CONFIG_REQ, (ke_msg_func_t)me_config_req_handler},
    {ME_CHAN_CONFIG_REQ, (ke_msg_func_t)me_chan_config_req_handler},
    {ME_SET_CONTROL_PORT_REQ, (ke_msg_func_t)me_set_control_port_req_handler},
    {ME_STA_ADD_REQ, (ke_msg_func_t)me_sta_add_req_handler},
    {ME_STA_DEL_REQ, (ke_msg_func_t)me_sta_del_req_handler},
    {ME_SET_ACTIVE_REQ, (ke_msg_func_t)me_set_active_req_handler},
    {ME_SET_PS_DISABLE_REQ, (ke_msg_func_t)me_set_ps_disable_req_handler},
    {ME_TRAFFIC_IND_REQ, (ke_msg_func_t)me_traffic_ind_req_handler},
    {ME_SET_ACTIVE_CFM, ke_msg_discard},
    {MM_SET_IDLE_CFM, (ke_msg_func_t)mm_set_idle_cfm_handler},
    #if NX_POWERSAVE
    {ME_SET_PS_MODE_REQ, (ke_msg_func_t)me_set_ps_mode_req_handler},
    {MM_SET_PS_MODE_CFM, (ke_msg_func_t)mm_set_ps_mode_cfm_handler},
    #endif
    {MM_STA_DEL_CFM, ke_msg_discard},
    {MM_CHAN_CTXT_UPDATE_CFM, ke_msg_discard},
    {MM_TIM_UPDATE_CFM, ke_msg_discard},
    {MM_SET_EDCA_CFM, ke_msg_discard},
    {MM_SET_MU_EDCA_CFM, ke_msg_discard},
    {MM_SET_UORA_CFM, ke_msg_discard},
    {MM_SET_TXOP_RTS_THRES_CFM, ke_msg_discard},
    {MM_SET_BSS_COLOR_CFM, ke_msg_discard},
    {ME_RC_STATS_REQ, (ke_msg_func_t)me_rc_stats_req_handler},
    {ME_RC_SET_RATE_REQ, (ke_msg_func_t)me_rc_set_rate_req_handler},
    {ME_CONFIG_MONITOR_REQ, (ke_msg_func_t)me_config_monitor_req_handler},
};


/// Specifies the message handlers that are common to all states.
const struct ke_state_handler me_default_handler = KE_STATE_HANDLER(me_default_state);


/// Defines the place holder for the states of all the task instances.
ke_state_t me_state[ME_IDX_MAX];


/// @} end of group
