/**
 ****************************************************************************************
 *
 * @file me_utils.c
 *
 * @brief All utility functions manipulating rates, etc.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "me_utils.h"
#include "co_math.h"
#include "sta_mgmt.h"
#include "vif_mgmt.h"
#include "mac_ie.h"
#include "hal_desc.h"
#include "mac_frame.h"
#include "me.h"
#include "mm.h"
#include "rc.h"
#include "me_mgmtframe.h"
#include "tpc.h"
#include "txl_cfm.h"
#include "txl_he.h"
#if (RW_BFMER_EN)
#include "bfr.h"
#endif //(RW_BFMER_EN)

/*
 * DEFINES
 ****************************************************************************************
 */
#if NX_HE
/// Mask of the BSS color information in the BSS HE operation element
#define ME_BSS_COLOR_MSK (MAC_HE_OPER_BSS_COLOR_DISABLED_BIT |                           \
                          MAC_HE_OPER_BSS_COLOR_PARTIAL_BIT | MAC_HE_OPER_BSS_COLOR_MASK)
#endif

/*
 * FUNCTION IMPLEMENTATION
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Updates Channel context using latest bss information,
 *
 * This function is called a bss parameters change implies a modification on the channel
 * configuration. (e.g. bandwidth change)
 *
 * @param[in] vif VIF entry
 ****************************************************************************************
 */
static void me_chan_ctxt_update(struct vif_info_tag *vif)
{
    struct me_bss_info *bss = &vif->bss_info;
    struct mm_chan_ctxt_update_req *req = KE_MSG_ALLOC(MM_CHAN_CTXT_UPDATE_REQ, TASK_MM,
                                                       TASK_ME, mm_chan_ctxt_update_req);

    // Sanity check - We shall have a channel context assigned when this procedure is
    // executed
    ASSERT_ERR(vif->chan_ctxt != NULL);

    // Fill-in the channel context addition request
    req->chan_index = vif->chan_ctxt->idx;
    req->chan = bss->chan;

    // Send the message
    ke_msg_send(req);
}

bool me_set_sta_ht_vht_param(struct sta_info_tag *sta, struct me_bss_info *bss)
{
    struct sta_capa_info *info = &sta->info;

    uint8_t local_supp_bw = me_env.phy_bw_max;
    uint8_t peer_supp_bw = PHY_CHNL_BW_20;
    uint8_t local_stbc_nss = me_env.stbc_nss;
    uint8_t peer_stbc_nss;
    bool smps = false;

    #if NX_HE
    if (STA_CAPA(sta, HE))
    {
        uint8_t chan_width_set = HE_PHY_CAPA_VAL_GET(&info->he_cap, CHAN_WIDTH_SET);
        bool stbc_rx_under_80 = HE_PHY_CAPA_BIT_IS_SET(&info->he_cap, STBC_RX_UNDER_80MHZ);
        bool stbc_rx_above_80 = HE_PHY_CAPA_BIT_IS_SET(&info->he_cap, STBC_RX_ABOVE_80MHZ);

        // Check the peer supported BW
        if (chan_width_set & HE_PHY_CAPA_CHAN_WIDTH_SET_80PLUS80_MHZ_IN_5G)
            peer_supp_bw = PHY_CHNL_BW_80P80;
        else if (chan_width_set & HE_PHY_CAPA_CHAN_WIDTH_SET_160MHZ_IN_5G)
            peer_supp_bw = PHY_CHNL_BW_160;
        else if (chan_width_set & HE_PHY_CAPA_CHAN_WIDTH_SET_40MHZ_80MHZ_IN_5G)
            peer_supp_bw = PHY_CHNL_BW_80;
        else if (chan_width_set & HE_PHY_CAPA_CHAN_WIDTH_SET_40MHZ_IN_2G)
            peer_supp_bw = PHY_CHNL_BW_40;

        if (peer_supp_bw > PHY_CHNL_BW_80)
            peer_stbc_nss = stbc_rx_under_80 & stbc_rx_above_80 ? 1 : 0;
        else
            peer_stbc_nss = stbc_rx_under_80 ? 1 : 0;

        local_stbc_nss = me_env.he_stbc_nss;
    }
    else
    #endif
    #if NX_VHT
    if (STA_CAPA(sta, VHT))
    {
        // Check the peer supported BW
        switch (info->vht_cap.vht_capa_info & MAC_VHTCAPA_SUPP_CHAN_WIDTH_MSK)
        {
            case MAC_VHTCAPA_SUPP_CHAN_WIDTH_160:
                peer_supp_bw = PHY_CHNL_BW_160;
                break;
            case MAC_VHTCAPA_SUPP_CHAN_WIDTH_160_80P80:
                peer_supp_bw = PHY_CHNL_BW_80P80;
                break;
            default:
                peer_supp_bw = PHY_CHNL_BW_80;
                break;
        }
        peer_stbc_nss = (info->vht_cap.vht_capa_info & MAC_VHTCAPA_RXSTBC_MSK) >>
                                                        MAC_VHTCAPA_RXSTBC_OFT;
        peer_stbc_nss = (peer_stbc_nss <= 4) ? peer_stbc_nss : 0;
    }
    else
    #endif
    {
        peer_stbc_nss = (info->ht_cap.ht_capa_info & MAC_HTCAPA_RX_STBC_MSK) >>
                                                        MAC_HTCAPA_RX_STBC_OFT;

        if (info->ht_cap.ht_capa_info & MAC_HTCAPA_40_MHZ)
            peer_supp_bw = PHY_CHNL_BW_40;
    }

    info->phy_bw_max = co_min(peer_supp_bw, local_supp_bw);
    info->bw_cur = me_phy2mac_bw(me_get_sta_bw(info->phy_bw_max, bss->chan.type));
    info->stbc_nss = co_min(peer_stbc_nss, local_stbc_nss);

    if ((info->ht_cap.ht_capa_info & MAC_HTCAPA_SMPS_MSK) !=
                                           MAC_HTCAPA_SMPS_DISABLE)
        smps = true;

    return (smps);
}

uint8_t me_11ac_mcs_max(uint16_t mcs_map)
{
    uint8_t mcs_max;

    switch (mcs_map & MAC_VHT_MCS_MAP_MSK)
    {
        case MAC_VHT_MCS_MAP_0_7:
            mcs_max = 7;
            break;
        case MAC_VHT_MCS_MAP_0_8:
            mcs_max = 8;
            break;
        case MAC_VHT_MCS_MAP_0_9:
            mcs_max = 9;
            break;
        default:
            mcs_max = 7;
            break;
    }

    return(mcs_max);
}

uint8_t me_11ax_mcs_max(uint16_t mcs_map)
{
    uint8_t mcs_max;

    switch (mcs_map & MAC_HE_MCS_MAP_MSK)
    {
        case MAC_HE_MCS_MAP_0_7:
            mcs_max = 7;
            break;
        case MAC_HE_MCS_MAP_0_9:
            mcs_max = 9;
            break;
        case MAC_HE_MCS_MAP_0_11:
            mcs_max = 11;
            break;
        default:
            mcs_max = 7;
            break;
    }

    return(mcs_max);
}

uint8_t me_11ac_nss_max(uint16_t mcs_map)
{
    uint8_t nss_max;

    // Go through the MCS map to check how many SS are supported
    for (nss_max = 7; nss_max > 0; nss_max--)
    {
        if (((mcs_map >> (2 * nss_max)) & MAC_VHT_MCS_MAP_MSK) != MAC_VHT_MCS_MAP_NONE)
            break;
    }

    return(nss_max);
}

uint8_t me_11n_nss_max(uint8_t *mcs_set)
{
    uint8_t nss_max;

    // Go through the MCS map to check how many SS are supported
    for (nss_max = 3; nss_max > 0; nss_max--)
    {
        if (mcs_set[nss_max] != 0)
            break;
    }

    return(nss_max);
}

uint8_t me_legacy_ridx_min(uint16_t rate_map)
{
    uint8_t i;

    for (i = 0; i < MAC_RATESET_LEN; i++)
    {
        if (rate_map & (1 << i))
        {
            break;
        }
    }

    return i;
}

uint8_t me_legacy_ridx_max(uint16_t rate_map)
{
    uint8_t i;
    uint8_t mcs_max;

    if (rate_map != 0 )
    {
        for (i = 0; i < MAC_RATESET_LEN; i++)
        {
            if (rate_map & (1 << (HW_RATE_54MBPS - i)))
            {
                break;
            }
        }
        mcs_max = HW_RATE_54MBPS - i;
    }
    else
    {
        mcs_max = MAC_RATESET_LEN;
    }

    return mcs_max;
}

void me_get_ampdu_params(struct mac_htcapability const *ht_cap,
                         struct mac_vhtcapability const *vht_cap,
                         struct mac_hecapability const *he_cap,
                         uint16_t *ampdu_size_max_ht,
                         uint32_t *ampdu_size_max_vht,
                         uint32_t *ampdu_size_max_he,
                         uint8_t *ampdu_spacing_min)
{
    *ampdu_size_max_ht = 0;
    *ampdu_size_max_vht = 0;
    *ampdu_size_max_he = 0;

    if (ht_cap)
    {
        #if NX_VHT
        int vht_exp = 0;
        #endif
        int ht_exp = (ht_cap->a_mpdu_param & MAC_AMPDU_LEN_EXP_MSK) >>
                                                               MAC_AMPDU_LEN_EXP_OFT;
        int min_spc = (ht_cap->a_mpdu_param & MAC_AMPDU_MIN_SPACING_MSK) >>
                                                            MAC_AMPDU_MIN_SPACING_OFT;
        *ampdu_spacing_min = (min_spc < 3)?1:(0x01 << (min_spc - 3));
        *ampdu_size_max_ht = (1 << (MAC_HT_MAX_AMPDU_FACTOR + ht_exp)) - 1;

        #if NX_VHT
        if (vht_cap)
        {
            vht_exp = (vht_cap->vht_capa_info & MAC_VHTCAPA_MAX_A_MPDU_LENGTH_EXP_MSK) >>
                                                    MAC_VHTCAPA_MAX_A_MPDU_LENGTH_EXP_OFT;
            *ampdu_size_max_vht = (1 << (MAC_HT_MAX_AMPDU_FACTOR + vht_exp)) - 1;
        }
        #endif

        #if NX_HE
        if (he_cap)
        {
            int he_exp_ext = HE_MAC_CAPA_VAL_GET(he_cap, MAX_A_AMPDU_LEN_EXP);
            if (vht_cap)
            {
                if (vht_exp == 7)
                    *ampdu_size_max_he = (1 << (MAC_HT_MAX_AMPDU_FACTOR
                                                         + 7 + he_exp_ext)) - 1;
                else
                    *ampdu_size_max_he = *ampdu_size_max_vht;
            }
            else
            {
                if (ht_exp == 3)
                    *ampdu_size_max_he = (1 << (MAC_HT_MAX_AMPDU_FACTOR
                                                         + 3 + he_exp_ext)) - 1;
                else
                    *ampdu_size_max_he = *ampdu_size_max_ht;
            }
        }
        #endif
    }
}

/**
 ****************************************************************************************
 * @brief Extracts ERP protection information and save it in BSS structure
 *
 * @param[in]  erp_addr    Address of ERP IE.
 * @param[out] prot_status Pointer of prot_status in @ref me_bss_info structure
 ****************************************************************************************
 */
static void me_erp_prot_check(uint32_t erp_addr, uint16_t *prot_status)
{
    /*
     * Extract received ERP parameters field value
     *      Bit 0 - NonERP Present
     *      Bit 1 - Use Protection
     *      Bit 2 - Barker Preamble Mode
     */
    uint8_t erp_val = co_read8p(erp_addr + MAC_ERP_PARAM_OFT);

    // Clear current ERP Protection status
    *prot_status &= ~MAC_PROT_ERP_STATUS_MASK;

    if (erp_val & MAC_ERP_NON_ERP_PRESENT)
    {
        // Non_ERP STA present
        *prot_status |= MAC_PROT_NONERP_PRESENT_BIT;
    }

    if (erp_val & MAC_ERP_USE_PROTECTION)
    {
        // Use protection
        *prot_status |= MAC_PROT_USE_PROTECTION_BIT;
    }

    if (erp_val & MAC_ERP_BARKER_PREAMBLE_MODE)
    {
        // Barker Preamble Mode
        *prot_status |= MAC_PROT_BARKER_PREAMB_BIT;
    }
}

/**
 ****************************************************************************************
 * @brief Updates protection status inside the policy table linked to a STA
 *
 * @param[in] sta    STA entry
 * @param[in] prot_status New protection status to use
 ****************************************************************************************
 */
static void me_pol_tbl_prot_upd(struct sta_info_tag *sta, uint16_t prot_status)
{
    // Policy Table Info
    struct sta_pol_tbl_cntl *pol_tbl = &sta->pol_tbl;
    // VIF Interface
    struct vif_info_tag *vif;

    // Reset Protection Configuration in the policy table info structure
    pol_tbl->prot_cfg = 0;

    // Require update of protection configuration field in policy table
    pol_tbl->upd_field |= CO_BIT(STA_MGMT_POL_UPD_PROT);

    /*
     ***********************************************************************
     * NAV Protection Frame Exchange
     *      If Protection has to be enabled, CTS-to-self mechanism is used
     *      Protection will be enabled in following case:
     *          - ERP element's Use Protection Bit set to 1 (11b protection)
     ***********************************************************************
     */
    if (prot_status & MAC_PROT_USE_PROTECTION_BIT)
    {
        // Use CTS-to-self
        pol_tbl->prot_cfg |= PROT_SELF_CTS;

        // Get VIF interface
        vif = &vif_info_tab[sta->inst_nbr];

        // 802.11b protection required, use highest 11b rate allowed by AP
        pol_tbl->prot_cfg |= ((vif->bss_info.high_11b_rate) << MCS_INDEX_PROT_TX_RCX_OFT);
    }

    // Reset PPDU TX Configuration in the policy table info structure
    pol_tbl->ppdu_tx_cfg = 0;

    /*
     ***********************************************************************
     * Preamble Type of PPDU for Transmission
     ***********************************************************************
     */
    if (!(prot_status & MAC_PROT_BARKER_PREAMB_BIT))
    {
        pol_tbl->ppdu_tx_cfg |= PRE_TYPE_TX_RCX_MASK;
    }
    // Update preamble type in the RC algorithm
    rc_update_preamble_type(sta->staid, (pol_tbl->ppdu_tx_cfg > 0));
}

/**
 ****************************************************************************************
 * @brief Updates bandwidth inside the policy table linked to a STA
 *
 * @param[in] sta         STA entry
 * @param[in] new_phy_bw  New bandwidth to use
 ****************************************************************************************
 */
static void me_pol_tbl_bw_upd(struct sta_info_tag *sta, uint8_t new_phy_bw)
{
    // Policy Table Info
    struct sta_pol_tbl_cntl *pol_tbl = &sta->pol_tbl;
    uint8_t new_bw = me_phy2mac_bw(me_get_sta_bw(sta->info.phy_bw_max, new_phy_bw));

    // Check if the new bandwidth of BSS is compliant with the supported one
    if (new_bw != sta->info.bw_cur)
    {
        struct sta_capa_info *info = &sta->info;
        uint8_t local_supp_nss = 0;
        uint8_t peer_supp_nss = 0;
        uint8_t nss;

        #if NX_HE
        if (STA_CAPA(sta, HE))
        {
            ASSERT_ERR(LOCAL_CAPA(HE));

            // Check the peer and local supported NSS (HE)
            peer_supp_nss = me_11ac_nss_max(info->he_cap.mcs_supp.rx_mcs_80);
            local_supp_nss = me_11ac_nss_max(me_env.he_cap.mcs_supp.tx_mcs_80);

            nss = co_min(local_supp_nss, peer_supp_nss);
        }
        else
        #endif
        #if NX_VHT
        if (STA_CAPA(sta, VHT))
        {
            ASSERT_ERR(LOCAL_CAPA(VHT));

            // Check the peer and local supported NSS (VHT)
            peer_supp_nss = me_11ac_nss_max(info->vht_cap.rx_mcs_map);
            local_supp_nss = me_11ac_nss_max(me_env.vht_cap.tx_mcs_map);

            nss = co_min(local_supp_nss, peer_supp_nss);
        }
        else
        #endif
        {
            ASSERT_ERR(LOCAL_CAPA(HT));
            // Check the peer and local supported NSS (VHT)
            peer_supp_nss = me_11n_nss_max(info->ht_cap.mcs_rate);
            local_supp_nss = me_11n_nss_max(me_env.ht_cap.mcs_rate);

            nss = co_min(local_supp_nss, peer_supp_nss);
        }
        // Compute new BW
        sta->info.bw_cur = new_bw;

        // Update max BW allowed in the RC algorithm
        rc_update_bw_nss_max(sta->staid, sta->info.bw_cur, nss);

        // Require update of protection configuration field in policy table
        pol_tbl->upd_field |= CO_BIT(STA_MGMT_POL_UPD_BW);
    }
}

#if NX_HE
/**
 ****************************************************************************************
 * @brief Request the update of the HE BSS color in the policy table linked to a STA.
 *
 * The function does nothing if the sta is not a HE one.
 *
 * @param[in] sta STA entry
 ****************************************************************************************
 */
static void me_pol_tbl_bss_color_upd(struct sta_info_tag *sta)
{
    if (STA_CAPA(sta, HE))
    {
        struct sta_pol_tbl_cntl *pol_tbl = &sta->pol_tbl;
        pol_tbl->upd_field |= CO_BIT(STA_MGMT_POL_UPD_BSS_COLOR);
    }
}
#endif

/**
 ****************************************************************************************
 * @brief Converts rate value present in (Extended) supported rates IE with enum used by HW
 *
 * @param[in] rate Rate to convert
 * @return enum value corresponding to @p rate
 ****************************************************************************************
 */
uint8_t me_rate_translate(uint8_t rate)
{
    uint8_t hwrate = 0;

    rate &= ~MAC_BASIC_RATE;
    switch(rate)
    {
        case MAC_RATE_1MBPS:
            hwrate = HW_RATE_1MBPS;
            break;
        case MAC_RATE_2MBPS:
            hwrate = HW_RATE_2MBPS;
            break;
        case MAC_RATE_5_5MBPS:
            hwrate = HW_RATE_5_5MBPS;
            break;
        case MAC_RATE_11MBPS:
            hwrate = HW_RATE_11MBPS;
            break;
        case MAC_RATE_48MBPS:
            hwrate = HW_RATE_48MBPS;
            break;
        case MAC_RATE_24MBPS:
            hwrate = HW_RATE_24MBPS;
            break;
        case MAC_RATE_12MBPS:
            hwrate = HW_RATE_12MBPS;
            break;
        case MAC_RATE_6MBPS:
            hwrate = HW_RATE_6MBPS;
            break;
        case MAC_RATE_54MBPS:
            hwrate = HW_RATE_54MBPS;
            break;
        case MAC_RATE_36MBPS:
            hwrate = HW_RATE_36MBPS;
            break;
        case MAC_RATE_18MBPS:
            hwrate = HW_RATE_18MBPS;
            break;
        case MAC_RATE_9MBPS:
            hwrate = HW_RATE_9MBPS;
            break;
        default:
            hwrate = 0xFF;
    }
    return(hwrate);
}

uint16_t me_legacy_rate_bitfield_build(struct mac_rateset const *rateset, bool basic_only)
{
    int i;
    uint16_t rates = 0;

    // Build the legacy rates bitfield
    for (i = 0; i < rateset->length; i++)
    {
        // If the current rate is not basic, then we go to the next one
        if (basic_only && !(rateset->array[i] & MAC_BASIC_RATE))
            continue;

        // Convert the rate into an index
        int bit_pos = me_rate_translate(rateset->array[i]);

        // Check if the rate is consistent
        ASSERT_WARN(bit_pos < MAC_RATESET_LEN);

        // Set the corresponding bit in the bitfield
        if (bit_pos < MAC_RATESET_LEN)
        {
            rates |= CO_BIT(bit_pos);
        }
    }

    return (rates);
}

uint16_t me_rate_bitfield_vht_build(uint16_t mcs_map_1, uint16_t mcs_map_2)
{
    uint8_t i;
    uint16_t mcs_map = 0xFFFF;

    for (i = 0; i < 8; i++)
    {
        uint8_t mcs_cfg_1 = (mcs_map_1 >> (i << 1)) & MAC_VHT_MCS_MAP_MSK;
        uint8_t mcs_cfg_2 = (mcs_map_2 >> (i << 1)) & MAC_VHT_MCS_MAP_MSK;
        if ((mcs_cfg_1 == MAC_VHT_MCS_MAP_NONE) || (mcs_cfg_2 == MAC_VHT_MCS_MAP_NONE))
        {
            break;
        }
        else
        {
            mcs_map &= ~(MAC_VHT_MCS_MAP_MSK << (i << 1));
            mcs_map |= (co_min(mcs_cfg_1, mcs_cfg_2) & MAC_VHT_MCS_MAP_MSK) << (i << 1);
        }
    }

    return mcs_map;
}

uint16_t me_build_capability(uint8_t vif_idx)
{
    uint16_t capa_info = 0;
    struct vif_info_tag *vif = &vif_info_tab[vif_idx];

    capa_info |= MAC_CAPA_ESS;
//    if (vif->type == VIF_AP)
//        // add ESS
//        capa_info |= MAC_CAPA_ESS;
//    else if (vif->type == VIF_IBSS)
//        // add the IBSS mode
//        capa_info |= MAC_CAPA_IBSS;

    // add Qos
    //capa_info |= MAC_CAPA_QOS;

    // add privacy
    capa_info &= ~MAC_CAPA_PRIVA;

    // Preamble
    capa_info |= MAC_CAPA_SHORT_PREAMBLE;
    // add slot policy
    capa_info |= MAC_CAPA_SHORT_SLOT;
    // add PBCC

    // add channel agility

    if (vif->bss_info.chan.band == PHY_BAND_5G)
        capa_info |= MAC_CAPA_SPECTRUM;

    // add BA
    // immediate BA supported
    //capa_info |= MAC_CAPA_IMMEDIATE_BA;

    return capa_info;
}

#if NX_HE
uint32_t me_build_bss_color_reg(struct me_bss_info *bss)
{
    uint32_t bss_color = bss->he_oper & MAC_HE_OPER_BSS_COLOR_MASK;

    // Format as per BSS COLOR register format
    bss_color = bss->he_oper & MAC_HE_OPER_BSS_COLOR_MASK;
    bss_color >>= MAC_HE_OPER_BSS_COLOR_OFT;
    bss_color <<= NXMAC_BSS_COLOR_LSB;
    bss_color |= NXMAC_BSS_COLOR_EN_BIT;
    if (!(bss->he_oper & MAC_HE_OPER_BSS_COLOR_DISABLED_BIT))
        bss_color |= NXMAC_BSS_COLOR_EN_BIT;
    if (bss->he_oper & MAC_HE_OPER_BSS_COLOR_PARTIAL_BIT)
        bss_color |= NXMAC_PARTIAL_BSS_COLOR_EN_BIT;

    return bss_color;
}
#endif

void me_init_rate(struct sta_info_tag *sta)
{
    rc_init(sta);
    me_update_buffer_control(sta);
}


void me_init_bcmc_rate(struct sta_info_tag *sta)
{
    struct sta_pol_tbl_cntl *rc = &sta->pol_tbl;
    struct mac_rateset *rates = &sta->info.rate_set;
    uint8_t max_rate = 0;
    int i;

    ASSERT_ERR(rates->length != 0);
    for (i = 0; i < rates->length; i++)
    {
        if ((rates->array[i] & ~MAC_BASIC_RATE) > max_rate)
            max_rate = rates->array[i] & ~MAC_BASIC_RATE;
    }

    rc_init_bcmc_rate(sta, me_rate_translate(max_rate));

    rc->upd_field = 0;
}


struct txl_buffer_control *me_update_buffer_control(struct sta_info_tag *sta)
{
    // Policy Table Info
    struct sta_pol_tbl_cntl *pol_tbl = &sta->pol_tbl;
    struct txl_buffer_control *buf_ctrl = pol_tbl->buf_ctrl;

    // Check if a field has to be updated
    if (pol_tbl->upd_field)
    {
        int i;
        uint8_t stbc_nss = 0;
        uint8_t sta_stbc_nss = 0;
        int use_stbc = 0;
        // Next TX Policy Table
        struct tx_policy_tbl *pol = &buf_ctrl->policy_tbl;
        // Rate control info
        uint32_t rate_info[RATE_CONTROL_STEPS];
        // Power control info
        uint32_t pwr_info[RATE_CONTROL_STEPS];
        // PHY control information 1 and 2
        uint32_t phycntrlinfo1, phycntrlinfo2;
        // Trial rate info
        uint32_t trial_rate;
        #if NX_HE
        // Indicate if used rates allow this frame to contain a +HTC if sent as singleton
        bool can_use_htc = (buf_ctrl->tx_flags & TX_SWDESC_UMAC_HTC_BIT) != 0;
        #else
        bool can_use_htc = false;
        #endif
        #if RW_BFMER_EN
        // Indicate if used rates allow this frame to be beamformed if sent as singleton
        bool can_use_bfm = (buf_ctrl->tx_flags & TX_SWDESC_UMAC_BEAMFORM_BIT) != 0;
        // Nc received in the latest beamforming report
        uint8_t nc = bfr_get_last_nc(sta->staid);
        // Read number of antennas
        uint8_t ntx = phy_get_ntx();
        can_use_bfm = (nc != BFR_INVALID_NC);
        #else
        bool can_use_bfm = false;
        uint8_t nc = 0, ntx = 0;
        #endif //(RW_BFMER_EN)

        // Get the currently used rate control information
        phycntrlinfo1 = pol->phycntrlinfo1;
        phycntrlinfo2 = pol->phycntrlinfo2;
        for (i = 0; i < RATE_CONTROL_STEPS; i++)
        {
            rate_info[i] = pol->ratecntrlinfo[i];
            pwr_info[i] = pol->powercntrlinfo[i];
        }

        // Update rate configuration (pre_type, short_gi, bw, nss, mcs)
        if (pol_tbl->upd_field & CO_BIT(STA_MGMT_POL_UPD_RATE))
        {
            PROF_RC_UPD_RETRY_CHAIN_SET();

            struct rc_sta_stats *rc_ss = pol_tbl->sta_stats;
            sta_stbc_nss = sta->info.stbc_nss;
            #if NX_HE
            can_use_htc = true;
            #endif
            for (i = 0; i < RATE_CONTROL_STEPS; i++)
            {
                uint8_t nss = 0;
                uint16_t idx = rc_ss->retry_step_idx[i];
                uint32_t new_rate_info = (RC_MAX_NUM_RETRY << N_RETRY_RCX_OFT) |
                                         (rc_ss->rate_stats[idx].rate_config &
                                          (MCS_INDEX_TX_RCX_MASK | BW_TX_RCX_MASK |
                                           SHORT_GI_TX_RCX_MASK | PRE_TYPE_TX_RCX_MASK |
                                           FORMAT_MOD_TX_RCX_MASK));
                uint32_t format = (new_rate_info & FORMAT_MOD_TX_RCX_MASK) >>
                                                               FORMAT_MOD_TX_RCX_OFT;

                // Get the number of SS of this frame
                if (format >= FORMATMOD_HT_MF)
                {
                    uint32_t mcs_idx = (new_rate_info & MCS_INDEX_TX_RCX_MASK)
                                                                 >> MCS_INDEX_TX_RCX_OFT;
                    #if NX_VHT
                    if (format == FORMATMOD_VHT)
                    {
                        // Extract NSS
                        nss = (mcs_idx & VHT_NSS_MASK) >> VHT_NSS_OFT;
                    }
                    else
                    #endif
                    #if NX_HE
                    if (format == FORMATMOD_HE_SU)
                    {
                        // Extract NSS
                        nss = (mcs_idx & VHT_NSS_MASK) >> VHT_NSS_OFT;

                        // Set LTF type
                        txl_he_ltf_type_set(new_rate_info, &pwr_info[i]);
                    }
                    else
                    #endif
                    {
                        // Extract NSS
                        nss = (mcs_idx & HT_NSS_MASK) >> HT_NSS_OFT;
                    }

                    // Check if we have to enable or disable STBC
                    if (i == 0)
                    {
                        if (nss < sta_stbc_nss)
                        {
                            stbc_nss = nss;
                            use_stbc = 1;
                        }
                    }
                    else if (use_stbc && (nss != stbc_nss))
                    {
                        use_stbc = 0;
                    }
                }

                // Clear the current configuration
                rate_info[i] &= ~(N_RETRY_RCX_MASK | MCS_INDEX_TX_RCX_MASK |
                                  BW_TX_RCX_MASK | SHORT_GI_TX_RCX_MASK |
                                  PRE_TYPE_TX_RCX_MASK | FORMAT_MOD_TX_RCX_MASK);
                // Set new configuration
                rate_info[i] |= new_rate_info;

                // Check if beamforming can be used with this policy table
                #if RW_BFMER_EN
                if (can_use_bfm)
                {
                    if ((format != FORMATMOD_VHT) || (nss > nc) || (nss == ntx))
                    {
                        can_use_bfm = false;
                    }
                }
                #endif //(RW_BFMER_EN)
                #if NX_HE
                if (can_use_htc)
                {
                    if (format < FORMATMOD_HT_MF)
                    {
                        can_use_htc = false;
                    }
                }
                #endif
            }

            // Check if STBC can be used with this policy table
            phycntrlinfo1 &= ~STBC_PT_MASK;
            if (use_stbc)
            {
                phycntrlinfo1 |= (stbc_nss + 1) << STBC_PT_OFT;
            }

            #if RW_BFMER_EN
            // Allow/Disallow Beamforming based on the previous check
            if (can_use_bfm)
                buf_ctrl->tx_flags |= TX_SWDESC_UMAC_BEAMFORM_BIT;
            else
                buf_ctrl->tx_flags &= ~TX_SWDESC_UMAC_BEAMFORM_BIT;
            #endif
            #if NX_HE
            if (can_use_htc)
                buf_ctrl->tx_flags |= TX_SWDESC_UMAC_HTC_BIT;
            else
                buf_ctrl->tx_flags &= ~TX_SWDESC_UMAC_HTC_BIT;
            #endif

            // Force an update of the protection configuration because some
            // new rates might have to be protected, some others might have to be
            // unprotected
            pol_tbl->upd_field |= CO_BIT(STA_MGMT_POL_UPD_PROT);
            PROF_RC_UPD_RETRY_CHAIN_CLR();
        }

        // Update trial rate configuration (pre_type, short_gi, bw, nss, mcs)
        if (pol_tbl->upd_field & CO_BIT(STA_MGMT_POL_TRIAL_RATE))
        {
            PROF_RC_UPD_RETRY_CHAIN_TRIAL_SET();

            struct rc_sta_stats *rc_ss = pol_tbl->sta_stats;
            uint8_t idx = rc_ss->trial_idx;
            uint32_t new_rate_info = (RC_MAX_NUM_RETRY << N_RETRY_RCX_OFT) |
                                     (rc_ss->rate_stats[idx].rate_config &
                                      (MCS_INDEX_TX_RCX_MASK | BW_TX_RCX_MASK |
                                       SHORT_GI_TX_RCX_MASK | PRE_TYPE_TX_RCX_MASK |
                                       FORMAT_MOD_TX_RCX_MASK));

            // Clear the current configuration
            trial_rate = pol->ratecntrlinfo[0];
            trial_rate &= ~(N_RETRY_RCX_MASK | MCS_INDEX_TX_RCX_MASK |
                          BW_TX_RCX_MASK | SHORT_GI_TX_RCX_MASK |
                          PRE_TYPE_TX_RCX_MASK | FORMAT_MOD_TX_RCX_MASK);

            // Set trial rate configuration
            rc_ss->trial_rate = trial_rate | new_rate_info;
            // Check if STBC can be used with this policy table
            use_stbc = (phycntrlinfo1 & STBC_PT_MASK) != 0;
            stbc_nss = ((phycntrlinfo1 & STBC_PT_MASK) >> STBC_PT_OFT) - 1;

            // Check if STBC and Beamforming can be used for the trial transmission
            rc_trial_check_bfm_stbc_htc(buf_ctrl, rc_ss->trial_rate,
                                        can_use_bfm, use_stbc, stbc_nss, nc, ntx,
                                        can_use_htc);

            PROF_RC_UPD_RETRY_CHAIN_TRIAL_CLR();
        }

        if (pol_tbl->upd_field & CO_BIT(STA_MGMT_POL_UPD_PROT))
        {
            for (i = 0; i < RATE_CONTROL_STEPS; i++)
            {
                uint8_t format = (rate_info[i] & FORMAT_MOD_TX_RCX_MASK) >>
                                                               FORMAT_MOD_TX_RCX_OFT;
                uint8_t rate = (rate_info[i] & MCS_INDEX_TX_RCX_MASK) >>
                                                               MCS_INDEX_TX_RCX_OFT;
                // Protect only OFDM transmissions
                if ((format >= FORMATMOD_NON_HT_DUP_OFDM) || (rate >= HW_RATE_6MBPS))
                {
                    // Clear the current used Protection Fields
                    rate_info[i] &= ~STA_MGMT_PROT_HW_MASK;

                    // Apply computed values
                    rate_info[i] |= pol_tbl->prot_cfg;
                }
            }
        }

        // update Tx power
        if (pol_tbl->upd_field & CO_BIT(STA_MGMT_POL_UPD_TX_POWER))
        {
            struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];
            uint8_t idx;

            tpc_get_vif_tx_power(vif, NULL, &idx);
            for (i = 0; i < RATE_CONTROL_STEPS; i++)
            {
                pwr_info[i] &= ~TX_PWR_LEVEL_MASK;
                pwr_info[i] |= TX_PWR_LEVEL_SET(idx);
            }
        }

        #if NX_HE
        // update BSS color
        if (pol_tbl->upd_field & CO_BIT(STA_MGMT_POL_UPD_BSS_COLOR))
        {
            struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];
            struct me_bss_info *bss = &vif->bss_info;
            uint8_t color = 0;

            phycntrlinfo2 &= ~BSS_COLOR_MASK;
            if (!(bss->he_oper & MAC_HE_OPER_BSS_COLOR_DISABLED_BIT))
                color = (bss->he_oper & MAC_HE_OPER_BSS_COLOR_MASK) >>
                                                           MAC_HE_OPER_BSS_COLOR_OFT;
            phycntrlinfo2 |= (color << BSS_COLOR_OFT);
        }
        #endif

        // Update the policy table (needs to be done with interrupt disabled
        // as this table can be copied in IRQ context before transmission)
        GLOBAL_INT_DISABLE();
        pol->phycntrlinfo1 = phycntrlinfo1;
        pol->phycntrlinfo2 = phycntrlinfo2;
        for (i = 0; i < RATE_CONTROL_STEPS; i++)
        {
            pol->ratecntrlinfo[i] = rate_info[i];
            pol->powercntrlinfo[i] = pwr_info[i];
        }
        GLOBAL_INT_RESTORE();

        // Clear update bit field
        pol_tbl->upd_field = 0;
    }

    return (buf_ctrl);
}

void me_bw_check(uint32_t ht_op_addr, uint32_t vht_op_addr, struct me_bss_info *bss)
{
    struct mac_chan_op *chan = &bss->chan;

    // By default we consider that the BW will be 20MHz
    chan->type = PHY_CHNL_BW_20;
    chan->center1_freq = bss->chan.prim20_freq;
    chan->center2_freq = 0;

    // Check if there is a HT operation element
    if ((ht_op_addr != 0) && (me_env.phy_bw_max >= PHY_CHNL_BW_40))
    {
        uint8_t sec_ch_oft = co_read8p(ht_op_addr + MAC_HT_OPER_PRIM_CH_OFT + 1) & 3;
        if (sec_ch_oft != 0)
        {
            // Compute the secondary channel frequency offset
            int8_t freq_offset = (sec_ch_oft == 1) ? 10 : -10;
            chan->center1_freq += freq_offset;
            chan->type = PHY_CHNL_BW_40;
        }
    }

    // Check if there is a VHT operation element
    #if NX_VHT
    if ((vht_op_addr != 0) && (me_env.phy_bw_max >= PHY_CHNL_BW_80))
    {
        uint8_t chan_width = co_read8p(vht_op_addr + MAC_VHT_CHAN_WIDTH_OFT) & 3;
        uint8_t center0 = co_read8p(vht_op_addr + MAC_VHT_CENTER_FREQ0_OFT);
        uint8_t center1 = co_read8p(vht_op_addr + MAC_VHT_CENTER_FREQ1_OFT);
        me_vht_bandwidth_parse(chan_width, center0, center1, chan);
    }
    #endif
}

void me_beacon_check(uint8_t vif_idx, uint16_t length, uint32_t bcn_addr)
{
    // Check if ERP field is present in the received beacon
    uint32_t addr, ht_op_addr = 0, vht_op_addr = 0;
    // Get VIF interface
    struct vif_info_tag *vif      = &vif_info_tab[vif_idx];
    struct sta_info_tag *sta = (struct sta_info_tag *)co_list_pick(&vif->sta_list);
    struct me_bss_info *bss = &vif->bss_info;
    // Keep current Protection status and BW
    uint16_t cur_prot_status = bss->prot_status;
    uint8_t cur_phy_bw = bss->chan.type;
    uint8_t cur_pwr_const = bss->power_constraint;
    #if NX_HE
    uint32_t cur_bss_color = bss->he_oper & ME_BSS_COLOR_MSK;
    #endif
    uint8_t csa_count, csa_mode;
    uint32_t var_part_addr = bcn_addr + MAC_BEACON_VARIABLE_PART_OFT;
    uint32_t var_part_len = length - MAC_BEACON_VARIABLE_PART_OFT;

    // Reset protection status before the checks
    bss->prot_status = 0;

    /*
     *================================================================
     * Check if ERP field is present in the received beacon
     *================================================================
     */
    if (bss->chan.band == PHY_BAND_2G4)
    {
        addr = mac_ie_erp_find(var_part_addr, var_part_len);

        if (addr)
        {
            me_erp_prot_check(addr, &bss->prot_status);
        }
    }

    /*
     *================================================================
     * Check if WMM element changed
     *================================================================
     */
    if (BSS_CAPA(bss, QOS) && (vif->type == VIF_STA))
    {
        bool changed;

        // Extract WMM element
        me_extract_edca_params(var_part_addr, var_part_len, &bss->edca_param, &changed);

        if (changed)
        {
            // EDCA parameters
            for (int i = 0; i < AC_MAX; i++)
            {
                struct mm_set_edca_req *edca = KE_MSG_ALLOC(MM_SET_EDCA_REQ, TASK_MM,
                                                            TASK_ME, mm_set_edca_req);
                #if NX_UAPSD
                bool uapsd = (vif->u.sta.uapsd_queues & ~CO_BIT(i)) != 0;
                #else
                bool uapsd = false;
                #endif

                edca->ac_param = bss->edca_param.ac_param[i];
                edca->hw_queue = i;
                edca->inst_nbr = vif->index;
                edca->uapsd = uapsd;

                ke_msg_send(edca);
            }
        }
    }

    /*
     *================================================================
     * Check if HT and VHT Operation Element are present in
     * the received beacon
     *================================================================
     */
    if (LOCAL_CAPA(HT))
    {
        ht_op_addr = mac_ie_ht_oper_find(var_part_addr, var_part_len);

        #if NX_VHT || NX_HE
        /*
         *================================================================
         * Check if VHT Operation and/or Operation Mode Notification Element
         * is present in the received beacon
         *================================================================
         */
        if (LOCAL_CAPA(VHT) || LOCAL_CAPA(HE))
        {
            vht_op_addr = mac_ie_vht_oper_find(var_part_addr, var_part_len);

            if (vif->type == VIF_STA)
            {
                uint32_t op_mode_notif_addr = mac_ie_op_mode_notif_find(var_part_addr,
                                                                        var_part_len);

                if (op_mode_notif_addr)
                {
                    uint8_t opmode = co_read8p(op_mode_notif_addr + MAC_INFOELT_INFO_OFT);
                    struct sta_info_tag *sta = &sta_info_tab[vif->u.sta.ap_id];

                    // This frame is valid if peer device is VHT or HE capable
                    if ((STA_CAPA(sta, VHT) || STA_CAPA(sta, HE)) &&
                        !(opmode & MAC_OPMODE_RXNSS_TYPE_BIT))
                    {
                        uint8_t bw = (opmode & MAC_OPMODE_BW_MSK) >> MAC_OPMODE_BW_OFT;
                        uint8_t nss = (opmode & MAC_OPMODE_RXNSS_MSK) >> MAC_OPMODE_RXNSS_OFT;

                        // Update maximum supported bandwidth
                        me_sta_bw_nss_max_upd(vif->u.sta.ap_id, bw, nss);
                    }
                }

                #if NX_HE
                if (BSS_CAPA(bss, HE))
                {
                    bool changed;
                    struct mm_set_uora_req *req = KE_MSG_ALLOC(MM_SET_UORA_REQ, TASK_MM,
                                                               TASK_ME, mm_set_uora_req);

                    me_extract_uora_params(var_part_addr, var_part_len, req);
                    // Send the UORA parameters to the LMAC
                    ke_msg_send(req);

                    me_extract_mu_edca_params(var_part_addr, var_part_len,
                                              &bss->mu_edca_param, &changed);

                    if (changed)
                    {
                        // Send the MU EDCA parameters to the LMAC
                        struct mm_set_mu_edca_req *req =
                                               KE_MSG_ALLOC(MM_SET_MU_EDCA_REQ, TASK_MM,
                                                            TASK_ME, mm_set_mu_edca_req);

                        for (int i = 0; i < AC_MAX; i++)
                        {
                            req->param[i] = bss->mu_edca_param.ac_param[i];
                        }

                        ke_msg_send(req);
                    }
                }
                #endif
            }
            #if NX_HE
            if (LOCAL_CAPA(HE))
            {
                uint16_t txop_dur_rts_thres = MAC_HE_OPER_TXOP_DUR_RTS_THRES_DISABLED;
                // Retrieve the HE operation field
                if (me_extract_he_oper(var_part_addr, var_part_len, bss))
                {
                    txop_dur_rts_thres = (bss->he_oper & MAC_HE_OPER_TXOP_DUR_RTS_THRES_MSK)
                                                        >> MAC_HE_OPER_TXOP_DUR_RTS_THRES_OFT;
                }
                // As per 802.11ax, update threshold only if it is non-NULL in the element
                if (txop_dur_rts_thres)
                {
                    struct mm_set_txop_rts_thres_req *req =
                                         KE_MSG_ALLOC(MM_SET_TXOP_RTS_THRES_REQ, TASK_MM,
                                                      TASK_ME, mm_set_txop_rts_thres_req);

                    req->txop_dur_rts_thres = txop_dur_rts_thres;
                    req->inst_nbr = vif->index;

                    ke_msg_send(req);
                }
                // Check if BSS color has changed and update it if necessary
                if (cur_bss_color != (bss->he_oper & ME_BSS_COLOR_MSK))
                {
                    struct mm_set_bss_color_req *req =
                                         KE_MSG_ALLOC(MM_SET_BSS_COLOR_REQ, TASK_MM,
                                                      TASK_ME, mm_set_bss_color_req);

                    req->bss_color = me_build_bss_color_reg(bss);

                    ke_msg_send(req);
                }
            }
            #endif
        }
        #endif
    }
    me_bw_check(ht_op_addr, vht_op_addr, bss);

    // Check for channel switch
    csa_count = me_extract_csa(var_part_addr, var_part_len, &csa_mode, &vif->csa_channel);
    if (csa_count)
    {
        if (vif->type == VIF_STA)
        {
            if ((!vif->u.sta.csa_count) &&
                (csa_mode == MAC_SWITCH_MODE_TX_TRAFFIC_STOP))
            {
                mm_send_csa_traffic_ind(vif->index, false);
            }
            vif->u.sta.csa_count = csa_count;
        }
        #if NX_BCN_AUTONOMOUS_TX
        else if (vif->type == VIF_AP) // if not defined this function is not called for AP itf
        {
            vif->u.ap.csa_count = csa_count;
        }
        #endif
    }

    // Check if power constraint changed
    me_extract_power_constraint(var_part_addr, var_part_len, bss);

    if (cur_pwr_const != bss->power_constraint)
        tpc_update_vif_tx_power(vif);

    // Check if we need to change the channel context
    if (cur_phy_bw < bss->chan.type)
    {
        me_chan_ctxt_update(vif);
    }

    // Update the protection and BW of STA linked to this VIF
    while (sta != NULL)
    {
        if (cur_prot_status != bss->prot_status)
        {
            // Update Policy Table for Protection Frames
            me_pol_tbl_prot_upd(sta, bss->prot_status);
        }

        if (cur_phy_bw != bss->chan.type)
        {
            me_pol_tbl_bw_upd(sta, bss->chan.type);
        }

        #if NX_HE
        if (cur_bss_color != (bss->he_oper & ME_BSS_COLOR_MSK))
        {
            me_pol_tbl_bss_color_upd(sta);
        }
        #endif

        sta = (struct sta_info_tag *)co_list_next(&sta->list_hdr);
    }
}

void me_sta_bw_nss_max_upd(uint8_t sta_idx, uint8_t bw, uint8_t nss)
{
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    struct sta_pol_tbl_cntl *pol_tbl = &sta->pol_tbl;
    struct sta_capa_info *info = &sta->info;
    struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];
    uint8_t local_supp_nss = 0;
    uint8_t peer_supp_nss = 0;

    #if NX_HE
    if (STA_CAPA(sta, HE))
    {
        ASSERT_ERR(LOCAL_CAPA(HE));

        // Check the peer and local supported NSS (HE)
        peer_supp_nss = me_11ac_nss_max(info->he_cap.mcs_supp.rx_mcs_80);
        local_supp_nss = me_11ac_nss_max(me_env.he_cap.mcs_supp.tx_mcs_80);

        nss = co_min(nss, co_min(local_supp_nss, peer_supp_nss));
    }
    else
    #endif
    #if NX_VHT
    if (STA_CAPA(sta, VHT))
    {
        ASSERT_ERR(LOCAL_CAPA(VHT));

        // Check the peer and local supported NSS (VHT)
        peer_supp_nss = me_11ac_nss_max(info->vht_cap.rx_mcs_map);
        local_supp_nss = me_11ac_nss_max(me_env.vht_cap.tx_mcs_map);

        nss = co_min(nss, co_min(local_supp_nss, peer_supp_nss));
    }
    else
    #endif
    {
        ASSERT_ERR(LOCAL_CAPA(HT));

        // Check the peer and local supported NSS (VHT)
        peer_supp_nss = me_11n_nss_max(info->ht_cap.mcs_rate);
        local_supp_nss = me_11n_nss_max(me_env.ht_cap.mcs_rate);

        nss = co_min(nss, co_min(local_supp_nss, peer_supp_nss));
    }

    bw = co_min(bw, me_phy2mac_bw(info->phy_bw_max));

    #if NX_TDLS
    if (sta_idx == vif->u.sta.tdls_chsw_sta_idx)
    {
        info->bw_cur = bw;
    }
    else
    #endif
    {
        info->bw_cur = co_min(bw, me_phy2mac_bw(vif->bss_info.chan.type));
    }

    // Update max BW allowed in the RC algorithm
    rc_update_bw_nss_max(sta->staid, info->bw_cur, nss);

    // Require update of protection configuration field in policy table
    pol_tbl->upd_field |= CO_BIT(STA_MGMT_POL_UPD_BW);
}

uint16_t me_tx_cfm_amsdu(struct txdesc *txdesc)
{
    struct hostdesc *host = &txdesc->host;
    struct sta_info_tag *sta = &sta_info_tab[host->staid];

    if (host->tid == 0xff)
        return 0;

    if (sta->inst_nbr == 0xFF)
       return 0;

    #if NX_AMPDU_TX
    // AMSDU allowed only under BA (because of retry)
    if (sta->ba_info[host->tid].bam_idx_tx == BAM_INVALID_TASK_IDX)
        return 0;

    if (!bam_env[sta->ba_info[host->tid].bam_idx_tx].amsdu)
        return 0;
    #endif

    // Get the max aggregation size
    return rc_get_max_amsdu_len(sta);
}

void me_vht_bandwidth_parse(uint8_t width, uint8_t center0, uint8_t center1,
                            struct mac_chan_op *chan)
{
    int delta = 0;

    if ((width == 0) || (width >= 4))
        return;

    chan->center1_freq = phy_channel_to_freq(chan->band, center0);
    chan->center2_freq = phy_channel_to_freq(chan->band, center1);

    switch (width)
    {
        case 1:
            // 80MHz, 160MHz or 80+80MHz channel
            // center0: center of the 80MHz channel that contains the primary
            // center1: 0 (for 80MHz)
            //          center of the 160MHz channel (for 160MHz)
            //          center of the secondary 80MHz channel (for 80+80MHz)
            if (chan->center2_freq)
            {
                if (chan->center1_freq > chan->center2_freq)
                    delta = chan->center1_freq - chan->center2_freq;
                else
                    delta = chan->center2_freq - chan->center1_freq;
            }

            if ((delta == 40) && (me_env.phy_bw_max >= PHY_CHNL_BW_160))
            {
                chan->type = PHY_CHNL_BW_160;
                chan->center1_freq = chan->center2_freq;
                chan->center2_freq = 0;
            }
            else if ((delta > 40) && (me_env.phy_bw_max == PHY_CHNL_BW_80P80))
            {
                chan->type = PHY_CHNL_BW_80P80;
            }
            else
            {
                chan->type = PHY_CHNL_BW_80;
                chan->center2_freq = 0;
            }
            break;
        case 2:
            // 160MHZ channel
            // center0: center of the 160MHz channel
            // center1: 0
            if (me_env.phy_bw_max >= PHY_CHNL_BW_160)
            {
                chan->type = PHY_CHNL_BW_160;
            }
            else
            {
                // 160MHz not supported, only use the 80MHz that contains the primary channel
                chan->type = PHY_CHNL_BW_80;
                if (chan->center1_freq < chan->prim20_freq)
                    chan->center1_freq += 40;
                else
                    chan->center1_freq -= 40;
            }
            break;
        case 3:
            // 80+80Mhz channel
            // center0: center of the primary 80MHz channel
            // center1: center of the secondary 80MHz channel
            if (me_env.phy_bw_max == PHY_CHNL_BW_80P80)
            {
                chan->type = PHY_CHNL_BW_80P80;
            }
            else
            {
                // 80+80MHZ not supported, only used priumary 80MHz channel
                chan->type = PHY_CHNL_BW_80;
                chan->center2_freq = 0;
            }
            break;
        default:
            break;
    }
}

