/**
 ******************************************************************************
 * @file mfp.c
 *
 * @brief Manangement Frames protection module.
 *
 * Copyright (C) RivieraWaves 2015-2019
 *
 ******************************************************************************
 */

#include "vif_mgmt.h"
#include "mac_ie.h"
#include "mfp.h"
#include "mfp_bip.h"

#if NX_MFP

/**
 * @addtogroup UMACMFP
 * @{
 */

/**
 ******************************************************************************
 * @brief Check if it is an action frame indicated as "Group Addressed Privacy".
 *
 * "Group Addressed Privacy" action frame are sent encrypted using the group key.
 * So far it is only used for MESH and MultiHop action frames.
 *
 * @param[in] frame_cntl The Frame Control field of the Frame
 * @param[in] action     Action field if the frame is an Action frame
 *
 * @return true for "group Addressed privacy" action frame and false otherwise.
 ******************************************************************************
 */
static bool mfp_is_group_addressed_privacy(uint16_t frame_cntl, uint8_t action)
{
    return  (((frame_cntl & MAC_FCTRL_TYPE_MASK) == MAC_FCTRL_MGT_T) &&
             ((frame_cntl & MAC_FCTRL_SUBT_MASK) == MAC_FCTRL_ACTION_ST) &&
             ((action == MAC_MESH_ACTION_CATEGORY) ||
              (action == MAC_MULTIHOP_ACTION_CATEGORY)));
}

/**
 ******************************************************************************
 * @brief Check validity of MGMT MIC IE.
 *
 * @param[in] frame       Pointer to the frame buffer
 * @param[in] length      Size, in bytes, of the frames
 * @param[in] machdr_len  Size, in bytes, of the mac header in the frame
 * @param[in] vif         Poiner to VIF structure
 * @param[in] sta         Pointer to STA structure (only if MESH is enabled)
 *
 * @return True if the frame contains a valid MGMT MIC IE, false otherwise.
 ******************************************************************************
 */
static bool mfp_check_mmic_ie(uint32_t frame, uint16_t length,
                              uint16_t machdr_len, struct vif_info_tag *vif,
                              struct sta_info_tag *sta)
{
    struct key_info_tag *key;
    uint16_t key_id, i;
    uint64_t pn;
    uint64_t mic, mic_check;
    uint32_t mmic_ie;

    if (length < machdr_len + 2 + MAC_MGMT_MIC_LEN)
    {
        return false;
    }

    mmic_ie = mac_ie_mgmt_mic_find(frame + machdr_len + 2, length - machdr_len - 2);

    if (mmic_ie == 0)
    {
        return false;
    }

    key_id = co_read16p(mmic_ie + MAC_MGMT_MIC_KEYID_OFT);
    pn = co_read64p(mmic_ie + MAC_MGMT_MIC_IPN_OFT) & 0xffffff;
    mic = co_read64p(mmic_ie + MAC_MGMT_MIC_MIC_OFT);

    if (key_id != 4 && key_id != 5)
    {
        return false;
    }

    #if (RW_MESH_EN)
    if (vif->type == VIF_MESH_POINT)
    {
        key = &sta->sta_sec_info.key_mfp_mesh_info[key_id - MAC_DEFAULT_KEY_COUNT];
    }
    else
    #endif //(RW_MESH_EN)
    {
        key = &vif->key_info[key_id];
    }

    if (!key->valid)
    {
        return false;
    }

    if (pn <= key->rx_pn[0])
    {
        return false;
    }

    key->rx_pn[0] = pn;

    // reset mic before computation
    for (i = 0; i < MAC_MGMT_MIC_MIC_LEN; i++)
    {
        co_write8p(mmic_ie + MAC_MGMT_MIC_MIC_OFT + i, 0);
    }

    mic_check = mfp_compute_bip(key, frame, length - MAC_MGMT_MIC_LEN, machdr_len, 0);
    if (mic_check != mic)
    {
        return false;
    }

    return true;
}

/*
 * PUBLIC FUNCTION DEFINITIONS
 ******************************************************************************
 */
bool mfp_is_robust_frame(uint16_t frame_cntl, uint8_t action)
{
    if ((frame_cntl & MAC_FCTRL_TYPE_MASK) != MAC_FCTRL_MGT_T)
        return false;

    switch (frame_cntl & MAC_FCTRL_SUBT_MASK) {
        case MAC_FCTRL_DEAUTHENT_ST:
        case MAC_FCTRL_DISASSOC_ST:
            return true;
        case MAC_FCTRL_ACTION_ST:
            switch (action) {
                case MAC_PUBLIC_ACTION_CATEGORY:
                case MAC_HT_ACTION_CATEGORY:
                case MAC_UNPROT_WNM_ACTION_CATEGORY:
                case MAC_SELF_PROT_ACTION_CATEGORY:
                case MAC_VENDOR_ACTION_CATEGORY:
                    return false;
                default:
                    return true;
            }
        default :
            return false;
    }
}

bool mfp_ignore_mgmt_frame(struct rx_cntrl_rx_status *rx_status, uint32_t *frame,
                           uint16_t frmlen, bool *upload)
{
    struct vif_info_tag *vif;
    struct sta_info_tag *sta;
    uint8_t action;

    // don't ignore non robust mgmt frames
    action = co_read8p((uint32_t)(CPU2HW(frame) + rx_status->machdr_len));
    if (!mfp_is_robust_frame(rx_status->frame_cntl, action))
    {
        return false;
    }

    // don't expect robust mgt frames from unknown STA (even if MFP is not enabled)
    if ((rx_status->sta_idx == INVALID_STA_IDX) ||
        (rx_status->vif_idx == INVALID_VIF_IDX))
    {
        *upload = false;
        return true;
    }

    vif = &vif_info_tab[rx_status->vif_idx];
    sta = &sta_info_tab[rx_status->sta_idx];

    // If no defaut_key, then encryption is not (yet) enabled.
    // Test defaut_key and not default_mgmt_key because in some cases we want to
    // do the following check even when MFP is not enabled:
    // - "group addressed privacy" action frames
    // - unicast robust mgt frames must always be protected for MESH STA
    //   (capa MFP is always set on MESH STA)
    if (! vif->default_key)
    {
        return false;
    }

    if (rx_status->frame_info & RXU_CNTRL_GROUP_ADDRESSED)
    {
        if (mfp_is_group_addressed_privacy(rx_status->frame_cntl, action))
        {
            // "Group Addressed Privacy" Action frame must be encrypted using group key
            return !(rx_status->frame_cntl & MAC_FCTRL_PROTECTEDFRAME);
        }
        else if (!vif->default_mgmt_key)
        {
            // MFP not enabled
            return false;
        }
        else if (mfp_check_mmic_ie(CPU2HW(frame), frmlen, rx_status->machdr_len, vif, sta))
        {
            // Other broad/muliticast must have a valid MGMT MIC IE
            return false;
        }
        else
        {
            *upload = false;
            return true;
        }
    }
    else if (!STA_CAPA(sta, MFP) ||
             (rx_status->frame_cntl & MAC_FCTRL_PROTECTEDFRAME))
    {
        return false;
    }
    else
    {
        // Unicast frame without encryption
        if ((MAC_FCTRL_IS(rx_status->frame_cntl, DEAUTHENT) ||
             MAC_FCTRL_IS(rx_status->frame_cntl, DISASSOC)) &&
            (action == MAC_RS_CLASS_2_FORBIDDEN ||
             action == MAC_RS_CLASS_3_FORBIDDEN))
        {
            // upload frame to host to start SA query procedure
            *upload = true;
        }
        else
        {
            *upload = false;
        }
    }

    return true;
}

enum mfp_protection mfp_protect_mgmt_frame(struct txdesc *txdesc,
                                           uint16_t fctl, uint8_t action)
{
    struct hostdesc *host = &txdesc->host;
    struct vif_info_tag *vif = &vif_info_tab[host->vif_idx];
    struct sta_info_tag *sta = &sta_info_tab[host->staid];

    // See mfp_ignore_mgmt_frame to known why default_mgmt_key is not tested here
    if ((host->staid == INVALID_STA_IDX) ||
        (!vif->default_key))
    {
        return MFP_NO_PROT;
    }

    if (!mfp_is_robust_frame(fctl, action))
    {
        // don't protect non robust mgmt frames
        return MFP_NO_PROT;
    }

    if (MAC_ADDR_GROUP(&sta->mac_addr))
    {
        if (mfp_is_group_addressed_privacy(fctl, action))
            return MFP_UNICAST_PROT;
        else if (!vif->default_mgmt_key)
            return MFP_NO_PROT;
        else
            return MFP_MULTICAST_PROT;
    }

    if (!STA_CAPA(sta, MFP))
    {
        return MFP_NO_PROT;
    }

    return MFP_UNICAST_PROT;
}


int mfp_add_mgmt_mic(struct txdesc *txdesc, uint32_t frame, int len, uint32_t mmic)
{
    struct hostdesc *host = &txdesc->host;
    struct vif_info_tag *vif = &vif_info_tab[host->vif_idx];
    struct key_info_tag *key;
    uint32_t mmic_ie;
    uint64_t mic;
    int i;

    if (!vif->default_mgmt_key)
    {
        return 0;
    }
    key = vif->default_mgmt_key;

    if (mmic)
        mmic_ie = mmic;
    else
        mmic_ie = frame + len;

    // IE ID - len - key-id
    co_write8p(mmic_ie + MAC_MGMT_MIC_ID_OFT, MAC_ELTID_MGMT_MIC);
    co_write8p(mmic_ie + MAC_MGMT_MIC_LEN_OFT, MAC_MGMT_MIC_LEN - 2);
    co_write16p(mmic_ie + MAC_MGMT_MIC_KEYID_OFT, key->key_idx);

    // IPN
    key->tx_pn++;
    for (i = 0; i < MAC_MGMT_MIC_IPN_LEN; i++)
    {
        co_write8p(mmic_ie + MAC_MGMT_MIC_IPN_OFT + i, (key->tx_pn >> (8*i)) & 0xff);
    }

    // MIC
    for (i = 0; i < MAC_MGMT_MIC_MIC_LEN; i++)
    {
        co_write8p(mmic_ie + MAC_MGMT_MIC_MIC_OFT + i, 0);
    }

    mic = mfp_compute_bip(key, frame, len, MAC_SHORT_MAC_HDR_LEN, mmic);
    for (i = 0; i < MAC_MGMT_MIC_MIC_LEN; i++)
    {
        co_write8p(mmic_ie + MAC_MGMT_MIC_MIC_OFT + i, (mic >> (8*i)) & 0xff);
    }

    return MAC_MGMT_MIC_LEN;
}

/**
 * @}
 */
#endif /* NX_MFP */
