/**
 ****************************************************************************************
 *
 * @file rc.c
 *
 * Copyright (C) RivieraWaves 2015-2019
 *
 * @brief The Rate Control module implementation.
 *
 ****************************************************************************************
 */

/** @addtogroup RC
* @{
*/

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "me.h"
#include "mm.h"
#include "me_utils.h"

#include "txl_cfm.h"
#include "rc.h"
#include "vif_mgmt.h"
#if RW_BFMER_EN
#include "bfr.h"
#endif

extern struct me_env_tag me_env;

/*
 * PRIVATE VARIABLES
 ****************************************************************************************
 */

/// Array of rate control statistics
struct rc_sta_stats sta_stats[NX_REMOTE_STA_MAX];

/// Transmission duration in nsecs of 1200 bytes for a HT/VHT MCS.
/// Index of the table is (MCS_IDX << 3) | (BW << 1) | (GI_400)
///  where BW is 0 for 20 MHz, 1 for 40MHz and 2 for 80MHz
///        GI_400 is 1 if packet is being sent with 400ns GI, 0 if 800ns GI
static const uint32_t rc_duration_ht_ampdu[10*4*2] =
{
    //BW20,GI800          BW20,GI400            BW40,GI800            BW40,GI400           MCS Index
    [  0] =  1476923,     [  1] =  1329231,     [  2] =   711111,     [  3] =   640000, // MCS 0
    [  8] =   738462,     [  9] =   664615,     [ 10] =   355556,     [ 11] =   320000, // MCS 1
    [ 16] =   492308,     [ 17] =   443077,     [ 18] =   237037,     [ 19] =   213333, // MCS 2
    [ 24] =   369231,     [ 25] =   332308,     [ 26] =   177778,     [ 27] =   160000, // MCS 3
    [ 32] =   246154,     [ 33] =   221538,     [ 34] =   118519,     [ 35] =   106667, // MCS 4
    [ 40] =   184615,     [ 41] =   166154,     [ 42] =    88889,     [ 43] =    80000, // MCS 5
    [ 48] =   164103,     [ 49] =   147692,     [ 50] =    79012,     [ 51] =    71111, // MCS 6
    [ 56] =   147692,     [ 57] =   132923,     [ 58] =    71111,     [ 59] =    64000, // MCS 7
    [ 64] =   123077,     [ 65] =   110769,     [ 66] =    59259,     [ 67] =    53333, // MCS 8
    [ 72] =   110769,     [ 73] =    99692,     [ 74] =    53333,     [ 75] =    48000, // MCS 9
    //BW80,GI800          BW80,GI400            BW160,GI800           BW160,GI400          MCS Index
    [  4] =   328205,     [  5] =   295385,     [  6] =   164103,     [  7] =   147692, // MCS 0
    [ 12] =   164103,     [ 13] =   147692,     [ 14] =    82051,     [ 15] =    73846, // MCS 1
    [ 20] =   109402,     [ 21] =    98462,     [ 22] =    54701,     [ 23] =    49231, // MCS 2
    [ 28] =    82051,     [ 29] =    73846,     [ 30] =    41026,     [ 31] =    36923, // MCS 3
    [ 36] =    54701,     [ 37] =    49231,     [ 38] =    27350,     [ 39] =    24615, // MCS 4
    [ 44] =    41026,     [ 45] =    36923,     [ 46] =    20513,     [ 47] =    18462, // MCS 5
    [ 52] =    36467,     [ 53] =    32821,     [ 54] =    18234,     [ 55] =    16410, // MCS 6
    [ 60] =    32821,     [ 61] =    29538,     [ 62] =    16410,     [ 63] =    14769, // MCS 7
    [ 68] =    27350,     [ 69] =    24615,     [ 70] =    13675,     [ 71] =    12308, // MCS 8
    [ 76] =    24615,     [ 77] =    22154,     [ 78] =    12308,     [ 79] =    11077, // MCS 9
};

#if NX_HE
/// Transmission duration in nsecs of 1200 bytes for a HE SU MCS.
/// Index of the table is (MCS_IDX * 6) + ((BW >> 1) * 3) + (GI)
/// where BW is 0 for 20 MHz, 1 for 40MHz and 2 for 80MHz
static const uint32_t rc_duration_he_ampdu[12*2*3] =
{
    //BW20/40,GI0.8us    BW20/40,GI1.6us      BW20/40,GI3.2us    MCS Index
    [  0] = 1115897,     [  1] = 1181538,     [  2] = 1312821,   // MCS 0
    [  6] =  557949,     [  7] =  590769,     [  8] =  656410,   // MCS 1
    [ 12] =  371966,     [ 13] =  393846,     [ 14] =  437607,   // MCS 2
    [ 18] =  278974,     [ 19] =  295385,     [ 20] =  328205,   // MCS 3
    [ 24] =  185983,     [ 25] =  196923,     [ 26] =  218803,   // MCS 4
    [ 30] =  139487,     [ 31] =  147692,     [ 32] =  164103,   // MCS 5
    [ 36] =  123989,     [ 37] =  131282,     [ 38] =  145869,   // MCS 6
    [ 42] =  111590,     [ 43] =  118154,     [ 44] =  131282,   // MCS 7
    [ 48] =   92991,     [ 49] =   98462,     [ 50] =  109402,   // MCS 8
    [ 54] =   83692,     [ 55] =   88615,     [ 56] =   98462,   // MCS 9
    [ 60] =   74393,     [ 61] =   78769,     [ 62] =   87521,   // MCS 10
    [ 66] =   66954,     [ 67] =   70892,     [ 68] =   78769,   // MCS 11
    //BW80/160,GI0.8us   BW80/160,GI1.6us     BW80/160,GI3.2us   MCS Index
    [  3] =  266449,     [  4] =  282122,     [  5] =  313469,   // MCS 0
    [  9] =  133224,     [ 10] =  141061,     [ 11] =  156735,   // MCS 1
    [ 15] =   88816,     [ 16] =   94041,     [ 17] =  104490,   // MCS 2
    [ 21] =   66612,     [ 22] =   70531,     [ 23] =   78367,   // MCS 3
    [ 27] =   44408,     [ 28] =   47020,     [ 29] =   52245,   // MCS 4
    [ 33] =   33306,     [ 34] =   35265,     [ 35] =   39184,   // MCS 5
    [ 39] =   29605,     [ 40] =   31347,     [ 41] =   34830,   // MCS 6
    [ 45] =   26645,     [ 46] =   28212,     [ 47] =   31347,   // MCS 7
    [ 51] =   22204,     [ 52] =   23510,     [ 53] =   26122,   // MCS 8
    [ 57] =   19985,     [ 58] =   21160,     [ 59] =   23511,   // MCS 9
    [ 63] =   17763,     [ 64] =   18808,     [ 65] =   20898,   // MCS 10
    [ 69] =   15988,     [ 70] =   16929,     [ 71] =   18810,   // MCS 11
};

/// Transmission duration in nsecs of 1200 bytes for a HE TB using a 26-tone RU
/// Index of the table is (MCS_IDX * 3) + (GI)
///        GI is 0 for 0.8us, 1 for 1.6us and 2 for 3.2us
static const uint32_t rc_duration_he_ru26[12*3] =
{
    //GI0.8us            GI1.6us              GI3.2us            MCS Index
    [  0] = 10880000,    [  1] = 11520000,    [  2] = 12800000,  // MCS 0
    [  3] =  5440000,    [  4] =  5760000,    [  5] =  6400000,  // MCS 1
    [  6] =  3626666,    [  7] =  3840000,    [  8] =  4266666,  // MCS 2
    [  9] =  2720000,    [ 10] =  2880000,    [ 11] =  3200000,  // MCS 3
    [ 12] =  1813333,    [ 13] =  1920000,    [ 14] =  2133333,  // MCS 4
    [ 15] =  1360000,    [ 16] =  1440000,    [ 17] =  1600000,  // MCS 5
    [ 18] =  1208888,    [ 19] =  1280000,    [ 20] =  1422222,  // MCS 6
    [ 21] =  1088000,    [ 22] =  1152000,    [ 23] =  1280000,  // MCS 7
    [ 24] =   906666,    [ 25] =   960000,    [ 26] =  1066666,  // MCS 8
    [ 27] =   816000,    [ 28] =   864000,    [ 29] =   960000,  // MCS 9
    [ 30] =   725333,    [ 31] =   768000,    [ 32] =   853333,  // MCS 10
    [ 33] =   652800,    [ 34] =   691200,    [ 35] =   768000,  // MCS 11
};

/// Transmission duration in nsecs of 1200 bytes for a HE TB using a 52-tone RU
/// Index of the table is (MCS_IDX * 3) + (GI)
///        GI is 0 for 0.8us, 1 for 1.6us and 2 for 3.2us
static const uint32_t rc_duration_he_ru52[12*3] =
{
    //GI0.8us            GI1.6us              GI3.2us            MCS Index
    [  0] = 5440000,     [  1] = 5760000,     [  2] = 6400000,   // MCS 0
    [  3] = 2720000,     [  4] = 2880000,     [  5] = 3200000,   // MCS 1
    [  6] = 1813333,     [  7] = 1920000,     [  8] = 2133333,   // MCS 2
    [  9] = 1360000,     [ 10] = 1440000,     [ 11] = 1600000,   // MCS 3
    [ 12] =  906666,     [ 13] =  960000,     [ 14] = 1066666,   // MCS 4
    [ 15] =  680000,     [ 16] =  720000,     [ 17] =  800000,   // MCS 5
    [ 18] =  604444,     [ 19] =  640000,     [ 20] =  711111,   // MCS 6
    [ 21] =  544000,     [ 22] =  576000,     [ 23] =  640000,   // MCS 7
    [ 24] =  453333,     [ 25] =  480000,     [ 26] =  533333,   // MCS 8
    [ 27] =  408000,     [ 28] =  432000,     [ 29] =  480000,   // MCS 9
    [ 30] =  362666,     [ 31] =  384000,     [ 32] =  426666,   // MCS 10
    [ 33] =  326400,     [ 34] =  345600,     [ 35] =  384000,   // MCS 11
};

/// Transmission duration in nsecs of 1200 bytes for a HE TB using a 106-tone RU
/// Index of the table is (MCS_IDX * 3) + (GI)
///        GI is 0 for 0.8us, 1 for 1.6us and 2 for 3.2us
static const uint32_t rc_duration_he_ru106[12*3] =
{
    //GI0.8us            GI1.6us              GI3.2us            MCS Index
    [  0] = 2560000,     [  1] = 2710588,     [  2] = 3011764,   // MCS 0
    [  3] = 1280000,     [  4] = 1355294,     [  5] = 1505882,   // MCS 1
    [  6] =  853333,     [  7] =  903529,     [  8] = 1003921,   // MCS 2
    [  9] =  640000,     [ 10] =  677647,     [ 11] =  752941,   // MCS 3
    [ 12] =  426666,     [ 13] =  451764,     [ 14] =  501960,   // MCS 4
    [ 15] =  320000,     [ 16] =  338823,     [ 17] =  376470,   // MCS 5
    [ 18] =  284444,     [ 19] =  301176,     [ 20] =  334640,   // MCS 6
    [ 21] =  256000,     [ 22] =  271058,     [ 23] =  301176,   // MCS 7
    [ 24] =  213333,     [ 25] =  225882,     [ 26] =  250980,   // MCS 8
    [ 27] =  192000,     [ 28] =  203294,     [ 29] =  225882,   // MCS 9
    [ 30] =  170666,     [ 31] =  180705,     [ 32] =  200784,   // MCS 10
    [ 33] =  153600,     [ 34] =  162635,     [ 35] =  180705,   // MCS 11
};

static const uint32_t *rc_ru_duration[3] =
{
    [0] = rc_duration_he_ru26,
    [1] = rc_duration_he_ru52,
    [2] = rc_duration_he_ru106,
};

#endif

/// Transmission duration in nsecs of 1200 bytes.
/// Index of the table is (MCS_IDX<<1) | (pre_type)
///  where pre_type is 0 for long and short preamble
///                    1 for long preamble only
static const uint32_t rc_duration_cck[4*2] =
{
    //short preamble      long preamble       MCS Index
    [  0] = 10452000,     [  1] = 10548000, // MCS 0
    [  2] =  5380000,     [  3] =  5476000, // MCS 1
    [  4] =  2315000,     [  5] =  2411000, // MCS 2
    [  6] =  1439000,     [  7] =  1535000, // MCS 3
};

/// Transmission duration in nsecs of 1200 bytes.
/// Index of the table is (MCS_IDX - 4)
static const uint32_t rc_duration_non_ht[8] =
{
                      // MCS Index
    [  0] =  1600000, // MCS 4
    [  1] =  1068000, // MCS 5
    [  2] =   800000, // MCS 6
    [  3] =   536000, // MCS 7
    [  4] =   400000, // MCS 8
    [  5] =   268000, // MCS 9
    [  6] =   200000, // MCS 10
    [  7] =   180000, // MCS 11
};

/*
 * PRIVATE FUNCTIONS DECLARATION
 ****************************************************************************************
 */
static void rc_calc_prob_ewma(struct rc_rate_stats *rc_rs);
static uint16_t rc_set_previous_mcs_index(struct rc_sta_stats *rc_ss, uint16_t rate_config);
static uint16_t rc_set_next_mcs_index(struct rc_sta_stats *rc_ss, uint16_t rate_config);
static bool rc_check_rate_duplicated(struct rc_sta_stats *rc_ss, uint16_t rate_config);
static uint16_t rc_new_random_rate(struct rc_sta_stats *ss);
static bool rc_set_trial_tx(struct rc_sta_stats *rc_ss);
static bool rc_update_stats(struct rc_sta_stats *rc_ss, bool init);
static void rc_update_retry_chain(struct rc_sta_stats *rc_ss, uint32_t *cur_tp);
static void rc_get_new_samples(struct rc_sta_stats *rc_ss);
static bool rc_update_stats_fixed_rate(struct rc_sta_stats *rc_ss);

/*
 * PRIVATE FUNCTIONS DEFINITION
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @brief Perform EWMA (Exponentially Weighted Moving Average) calculation;
 * @param[in] old_val old value
 * @param[in] new_val new value
 * @param[in] weight weight to be used in the EWMA
 * @return Result of EWMA.
 ****************************************************************************************
 */
static inline uint32_t ewma(uint32_t old_val, uint32_t new_val, uint32_t weight)
{
    return ((new_val * (EWMA_DIV - weight) + old_val * weight) / EWMA_DIV);
}

/**
 ****************************************************************************************
 * @brief Extract the format from the rate configuration.
 * @param[in] rate_config rate configuration to be updated
 * @return Format.
 ****************************************************************************************
 */
static uint8_t rc_get_format_mod(uint16_t rate_config)
{
    return (rate_config & FORMAT_MOD_TX_RCX_MASK) >> FORMAT_MOD_TX_RCX_OFT;
}

/**
 ****************************************************************************************
 * @brief Extract the bandwidth from the rate configuration.
 * @param[in] rate_config rate configuration to be updated
 * @return Bandwidth.
 ****************************************************************************************
 */
static uint8_t rc_get_bw(uint16_t rate_config)
{
    return (rate_config & BW_TX_RCX_MASK) >> BW_TX_RCX_OFT;
}

/**
 ****************************************************************************************
 * @brief Extract the number of spatial streams from the rate configuration.
 * @param[in] rate_config rate configuration to be updated
 * @return Number of spatial streams.
 ****************************************************************************************
 */
static uint8_t rc_get_nss(uint16_t rate_config)
{
    uint8_t nss = 0;
    uint8_t format_mod = rc_get_format_mod(rate_config);

    switch (format_mod)
    {
        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
        {
            nss = (rate_config & HT_NSS_MASK) >> HT_NSS_OFT;
        } break;

        #if NX_VHT || NX_HE
        case FORMATMOD_VHT:
        case FORMATMOD_HE_SU:
        case FORMATMOD_HE_MU:
        {
            nss = (rate_config & VHT_NSS_MASK) >> VHT_NSS_OFT;
        } break;
        #endif

        default:
            break;
    }

    return nss;
}

/**
 ****************************************************************************************
 * @brief Extract the MCS / rate index from the rate configuration.
 * @param[in] rate_config rate configuration to be updated
 * @return MCS / rate index.
 ****************************************************************************************
 */
static uint8_t rc_get_mcs_index(uint16_t rate_config)
{
    uint8_t format_mod = rc_get_format_mod(rate_config);
    uint8_t mcs = 0;

    switch (format_mod)
    {
        case FORMATMOD_NON_HT:
        case FORMATMOD_NON_HT_DUP_OFDM:
        {
            mcs = (rate_config & MCS_INDEX_TX_RCX_MASK) >> MCS_INDEX_TX_RCX_OFT;
        } break;
        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
        {
            mcs = (rate_config & HT_MCS_MASK) >> HT_MCS_OFT;
        } break;

        #if NX_VHT || NX_HE
        case FORMATMOD_VHT:
        case FORMATMOD_HE_SU:
        case FORMATMOD_HE_MU:
        {
            mcs = (rate_config & VHT_MCS_MASK) >> VHT_MCS_OFT;
        } break;
        #endif

        default:
            break;
    }

    return mcs;
}

#if NX_HE
/**
 ****************************************************************************************
 * @brief Extract the HE Guard interval setting from the rate configuration.
 * @param[in] rate_config rate configuration to be updated
 * @return Guard interval.
 ****************************************************************************************
 */
static uint8_t rc_get_he_gi(uint32_t rate_config)
{
    return ((rate_config & HE_GI_TYPE_TX_RCX_MASK) >> HE_GI_TYPE_TX_RCX_OFT);
}
#endif

/**
 ****************************************************************************************
 * @brief Extract the Short Guard interval setting from the rate configuration.
 * @param[in] rate_config rate configuration to be updated
 * @return 0 if LGI, 1 if SGI.
 ****************************************************************************************
 */
static uint8_t rc_get_sgi(uint32_t rate_config)
{
    return ((rate_config & SHORT_GI_TX_RCX_MASK) >> SHORT_GI_TX_RCX_OFT);
}

/**
 ****************************************************************************************
 * @brief Extract the preamble type from the rate configuration.
 * @param[in] rate_config rate configuration to be updated
 * @return Preamble type.
 ****************************************************************************************
 */
static uint8_t rc_get_pre_type(uint16_t rate_config)
{
    return (rate_config & PRE_TYPE_TX_RCX_MASK) >> PRE_TYPE_TX_RCX_OFT;
}

/**
 ****************************************************************************************
 * @brief Calculates probability of success with EWMA.
 * @param[in] rc_rs pointer to rate control station statistics structure
 ****************************************************************************************
 */
static void rc_calc_prob_ewma(struct rc_rate_stats *rc_rs)
{
    uint32_t success = rc_rs->success;
    uint32_t attempts = rc_rs->attempts;

    if (attempts > 0)
    {
        rc_rs->sample_skipped = 0;
        uint32_t cur_prob = RC_FRAC(success, attempts);

        if (0 == rc_rs->old_prob_available)
        {
            if (cur_prob > 0)
            {
                rc_rs->probability = cur_prob - 1;
            }
            else
            {
                rc_rs->probability = 0;
            }
        }
        else
        {
            rc_rs->probability = ewma(rc_rs->probability, cur_prob, EWMA_LEVEL);
        }
        rc_rs->old_prob_available = 1;
    }
    else
    {
        if (rc_rs->sample_skipped < 0xFF)
        {
            rc_rs->sample_skipped++;
        }
    }
}

/**
 ****************************************************************************************
 * @brief Updates the MCS / rate index of the rate configuration, setting the previous
 * if available.
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @param[in] rate_config rate configuration to be updated
 * @return Rate configuration updated.
 ****************************************************************************************
 */
static uint16_t rc_set_previous_mcs_index(struct rc_sta_stats *rc_ss, uint16_t rate_config)
{
    uint16_t new_config = rate_config;
    uint8_t format_mod = rc_get_format_mod(rate_config);
    uint8_t mcs = rc_get_mcs_index(rate_config);

    switch (format_mod)
    {
        case FORMATMOD_NON_HT:
        case FORMATMOD_NON_HT_DUP_OFDM:
        {
            uint8_t mcs_min;
            // Decrease only within the same modulation (DSSS/CCK or OFDM)
            if (mcs < HW_RATE_6MBPS)
                mcs_min = rc_ss->r_idx_min;
            else
                mcs_min = co_max(HW_RATE_6MBPS, rc_ss->r_idx_min);

            while(mcs > mcs_min)
            {
                mcs--;
                if (rc_ss->rate_map_l & CO_BIT(mcs))
                {
                    new_config = (rate_config & ~MCS_INDEX_TX_RCX_MASK) | mcs;
                    break;
                }
            }
        } break;
        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
        {
            uint8_t nss = rc_get_nss(rate_config);
            while (mcs > 0)
            {
                mcs--;
                if (rc_ss->rate_map.ht[nss] & CO_BIT(mcs))
                {
                    new_config = (rate_config & ~HT_MCS_MASK) | mcs;
                    if (rc_ss->short_gi)
                    {
                        new_config |= SHORT_GI_TX_RCX_MASK;
                    }
                    break;
                }
            }
        } break;

        #if NX_VHT
        case FORMATMOD_VHT:
        {
            if (mcs > 0)
            {
                new_config = (rate_config & ~VHT_MCS_MASK) | (mcs-1);
                if (rc_ss->short_gi)
                {
                    new_config |= SHORT_GI_TX_RCX_MASK;
                }
            }
        } break;
        #endif

        #if NX_HE
        case FORMATMOD_HE_SU:
        {
            if (mcs > 0)
            {
                new_config = (rate_config & ~VHT_MCS_MASK) | (mcs-1);
            }
        } break;
        #endif

        default:
            break;
    }

    return new_config;
}

/**
 ****************************************************************************************
 * @brief Updates the MCS / rate index of the rate configuration, setting the next
 * if available.
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @param[in] rate_config rate configuration to be updated
 * @return Rate configuration updated.
 ****************************************************************************************
 */
static uint16_t rc_set_next_mcs_index(struct rc_sta_stats *rc_ss, uint16_t rate_config)
{
    uint16_t new_config = rate_config;
    uint8_t format_mod = rc_get_format_mod(rate_config);
    uint8_t mcs = rc_get_mcs_index(rate_config);

    switch (format_mod)
    {
        case FORMATMOD_NON_HT:
        case FORMATMOD_NON_HT_DUP_OFDM:
        {
            uint8_t mcs_max;
            // Increase only within the same modulation (DSSS/CCK or OFDM)
            if (mcs >= HW_RATE_6MBPS)
                mcs_max = rc_ss->r_idx_max;
            else
                mcs_max = co_min(HW_RATE_11MBPS, rc_ss->r_idx_max);

            while(mcs < mcs_max)
            {
                mcs++;
                if (rc_ss->rate_map_l & CO_BIT(mcs))
                {
                    new_config = (rate_config & ~MCS_INDEX_TX_RCX_MASK) | mcs;
                    break;
                }
            }
        } break;

        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
        {
            uint8_t nss = rc_get_nss(rate_config);
            while (mcs < rc_ss->mcs_max)
            {
                mcs++;
                if (rc_ss->rate_map.ht[nss] & CO_BIT(mcs))
                {
                    new_config = (rate_config & ~HT_MCS_MASK) | mcs;
                    if (rc_ss->short_gi)
                    {
                        new_config |= SHORT_GI_TX_RCX_MASK;
                    }
                    break;
                }
            }
        } break;

        #if NX_VHT
        case FORMATMOD_VHT:
        {
            uint8_t nss = rc_get_nss(rate_config);
            uint8_t mcs_max_ss = 7 + ((rc_ss->rate_map.vht >> (nss << 1)) & MAC_VHT_MCS_MAP_MSK);
            if ((mcs < rc_ss->mcs_max) && ((mcs+1) <= mcs_max_ss))
            {
                new_config = (rate_config & ~VHT_MCS_MASK) | (mcs+1);
                if (rc_ss->short_gi)
                {
                    new_config |= SHORT_GI_TX_RCX_MASK;
                }
            }
        } break;
        #endif

        #if NX_HE
        case FORMATMOD_HE_SU:
        {
            uint8_t nss = rc_get_nss(rate_config);
            uint8_t mcs_max_ss = 7 + 2 * ((rc_ss->rate_map.he >> (nss << 1)) & MAC_HE_MCS_MAP_MSK);
            if ((mcs < rc_ss->mcs_max) && ((mcs+1) <= mcs_max_ss))
            {
                new_config = (rate_config & ~VHT_MCS_MASK) | (mcs+1);
            }
        } break;
        #endif

        default:
            break;
    }

    return new_config;
}

/**
 ****************************************************************************************
 * @brief Checks if the rate configuration is already present in the sample table.
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @param[in] rate_config rate configuration to be checked
 * @return Whether the rate configuration is already present in the sample table.
 ****************************************************************************************
 */
static bool rc_check_rate_duplicated(struct rc_sta_stats *rc_ss, uint16_t rate_config)
{
    uint32_t i = 0;
    bool ret = 0;

    while (i < rc_ss->no_samples)
    {
        if (rate_config == rc_ss->rate_stats[i].rate_config)
        {
            ret = 1;
            break;
        }
        i++;
    }
    return ret;
}

#if NX_DEBUG
/**
 ****************************************************************************************
 * @brief Checks if the rate configuration of the sample i of the sample table is correct.
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @param[in] i index of the sample in the sample table
 ****************************************************************************************
 */
static void rc_check_rate_config(struct rc_sta_stats *rc_ss, uint16_t i)
{
    uint16_t rate_config = rc_ss->rate_stats[i].rate_config;
    uint8_t format_mod = rc_get_format_mod(rate_config);
    uint8_t pre_type = rc_get_pre_type(rate_config);
    uint8_t sgi = rc_get_sgi(rate_config);
    uint8_t bw = rc_get_bw(rate_config);
    uint8_t nss = rc_get_nss(rate_config);
    uint8_t mcs = rc_get_mcs_index(rate_config);

    #if NX_HE
    ASSERT_ERR(format_mod <= FORMATMOD_HE_SU);
    #elif NX_VHT
    ASSERT_ERR(format_mod <= FORMATMOD_VHT);
    #else
    ASSERT_ERR(format_mod <= FORMATMOD_HT_GF);
    #endif

    // check format and modulation
    switch (rc_ss->format_mod)
    {
        case FORMATMOD_NON_HT:
        case FORMATMOD_NON_HT_DUP_OFDM:
        {
            ASSERT_ERR(format_mod < FORMATMOD_HT_MF);
        } break;
        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
        {
            if (rc_ss->r_idx_min <= HW_RATE_11MBPS)
            {
                // DSSS/CCK
                ASSERT_ERR(format_mod < FORMATMOD_VHT);
            }
            else
            {
                ASSERT_ERR((format_mod == FORMATMOD_HT_MF) || (format_mod == FORMATMOD_HT_GF));
            }
        } break;

        #if NX_VHT
        case FORMATMOD_VHT:
        {
            ASSERT_ERR(format_mod == FORMATMOD_VHT);
        } break;
        #endif

        #if NX_HE
        case FORMATMOD_HE_SU:
        {
            if (rc_ss->r_idx_min <= HW_RATE_11MBPS)
            {
                // DSSS/CCK is possible
                ASSERT_ERR((format_mod == FORMATMOD_HE_SU) || (format_mod == FORMATMOD_NON_HT));
            }
            else
            {
                ASSERT_ERR(format_mod == FORMATMOD_HE_SU);
            }
        } break;
        #endif

        default:
            break;
    }

    switch (format_mod)
    {
        case FORMATMOD_NON_HT:
        case FORMATMOD_NON_HT_DUP_OFDM:
        {
            if ((mcs <= HW_RATE_11MBPS) && (rc_ss->type == 1))
            {
                ASSERT_ERR(pre_type == 1);
            }
            else
            {
                ASSERT_ERR(pre_type <= 1);
            }
            ASSERT_ERR(sgi == 0);
            ASSERT_ERR(bw == BW_20MHZ);
            ASSERT_ERR(nss == 0);
            ASSERT_ERR(mcs >= rc_ss->r_idx_min);
            ASSERT_ERR(mcs <= rc_ss->r_idx_max);
            ASSERT_ERR((rc_ss->rate_map_l & CO_BIT(mcs)));
        }
        break;

        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
        {
            ASSERT_ERR(pre_type == 0);
            ASSERT_ERR(sgi <= rc_ss->short_gi);
            ASSERT_ERR(bw <= rc_ss->bw_max);
            ASSERT_ERR(nss <= rc_ss->no_ss);
            ASSERT_ERR(mcs <= rc_ss->mcs_max);
            ASSERT_ERR((rc_ss->rate_map.ht[nss] & CO_BIT(mcs)));
        }
        break;

        #if NX_VHT
        case FORMATMOD_VHT:
        {
            ASSERT_ERR(pre_type == 0);
            ASSERT_ERR(sgi <= rc_ss->short_gi);
            ASSERT_ERR(bw <= rc_ss->bw_max);
            ASSERT_ERR(nss <= rc_ss->no_ss);
            ASSERT_ERR(mcs <= rc_ss->mcs_max);
            ASSERT_ERR(mcs <= (7 + (rc_ss->rate_map.vht >> (nss << 1) & MAC_VHT_MCS_MAP_MSK)));
            ASSERT_ERR(((mcs == 6) && (bw == BW_80MHZ) && ((nss == 3) || (nss == 6))) == 0);
            ASSERT_ERR(((mcs == 9) && (bw == BW_20MHZ) && (nss != 2) && (nss != 5)) == 0);
            ASSERT_ERR(((mcs == 9) && (bw == BW_80MHZ) && (nss == 5)) == 0);
            ASSERT_ERR(((mcs == 9) && (bw == BW_160MHZ) && (nss == 2)) == 0);
        }
        break;
        #endif

        #if NX_HE
        case FORMATMOD_HE_SU:
        {
            uint8_t gi = rc_get_he_gi(rate_config);
            ASSERT_ERR(gi <= 2);
            ASSERT_ERR(bw <= rc_ss->bw_max);
            ASSERT_ERR(nss <= rc_ss->no_ss);
            ASSERT_ERR(mcs <= rc_ss->mcs_max);
            ASSERT_ERR(mcs <= (7 + 2 * (rc_ss->rate_map.he >> (nss << 1) & MAC_HE_MCS_MAP_MSK)));
        }
        break;
        #endif

        default:
            break;
    }
}

/**
 ****************************************************************************************
 * @brief Checks if the rate configurations of the sample table are correct.
 * @param[in] rc_ss pointer to rate control station statistics structure
 ****************************************************************************************
 */
static void rc_check_stats_rate_config(struct rc_sta_stats *rc_ss)
{
    uint32_t i;

    for (i = 0; i < rc_ss->no_samples; i++)
    {
        rc_check_rate_config(rc_ss, i);
    }
}
#endif

/**
 ****************************************************************************************
 * @brief Gets a new random rate configuration.
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @return The random rate configuration
 ****************************************************************************************
 */
static uint16_t rc_new_random_rate(struct rc_sta_stats *rc_ss)
{
    // pick a random rate
    uint16_t rd = co_rand_hword();
    uint8_t format = rc_ss->format_mod;
    uint16_t rate_cfg = format << FORMAT_MOD_TX_RCX_OFT;
    uint8_t bw_min = rc_ss->bw_max > BW_20MHZ ? (rc_ss->bw_max - 1) : rc_ss->bw_max;

    switch (format)
    {
        case FORMATMOD_NON_HT:
        case FORMATMOD_NON_HT_DUP_OFDM:
        {
            uint8_t r_idx_l_max = rc_ss->r_idx_max;
            uint8_t r_idx_l_min = rc_ss->r_idx_min;
            uint8_t r_idx = (((rd & MCS_INDEX_TX_RCX_MASK) >> MCS_INDEX_TX_RCX_OFT) %
                    (r_idx_l_max - r_idx_l_min + 1)) + r_idx_l_min;
            if ((rc_ss->rate_map_l & CO_BIT(r_idx)) == 0) // rate index not allowed
            {
                r_idx = r_idx_l_max;
            }

            rate_cfg |= r_idx << MCS_INDEX_TX_RCX_OFT;

            if (r_idx == HW_RATE_1MBPS) // rate index 0 (CCK 1Mbps) allows only long preamble
            {
                rate_cfg |= PRE_TYPE_TX_RCX_MASK;
            }
            else if ((r_idx > HW_RATE_1MBPS) && (r_idx <= HW_RATE_11MBPS)) // CCK rate: set preamble type
            {
                rate_cfg |= (rd & PRE_TYPE_TX_RCX_MASK) | (rc_ss->type << PRE_TYPE_TX_RCX_OFT);
            }
        } break;
        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
        {
            uint8_t r_idx_max = rc_ss->r_idx_max;
            uint8_t r_idx_min = rc_ss->r_idx_min;

            if ((r_idx_min <= HW_RATE_11MBPS) &&
                (((rd & FORMAT_MOD_TX_RCX_MASK) >> FORMAT_MOD_TX_RCX_OFT) % 2)) // 2.4 GHz
            {
                // CCK
                uint8_t r_idx = (((rd & MCS_INDEX_TX_RCX_MASK) >> MCS_INDEX_TX_RCX_OFT) %
                                 (r_idx_max - r_idx_min + 1)) + r_idx_min;
                rate_cfg = FORMATMOD_NON_HT << FORMAT_MOD_TX_RCX_OFT;
                if ((rc_ss->rate_map_l & CO_BIT(r_idx)) == 0) // rate index not allowed
                {
                    r_idx = r_idx_max;
                }
                rate_cfg |= r_idx << MCS_INDEX_TX_RCX_OFT;
                if (r_idx == HW_RATE_1MBPS) // MCS0 (1Mbps) allows only long preamble
                {
                    rate_cfg |= PRE_TYPE_TX_RCX_MASK;
                }
                else
                {
                    rate_cfg |= (rd & PRE_TYPE_TX_RCX_MASK) | (rc_ss->type << PRE_TYPE_TX_RCX_OFT);
                }
            }
            else
            {
                uint8_t bw_max = rc_ss->bw_max;
                uint8_t short_gi = rc_ss->short_gi;
                uint8_t mcs_max = rc_ss->mcs_max;
                uint8_t no_ss_max = rc_ss->no_ss;
                uint8_t nss = ((rd & HT_NSS_MASK) >> HT_NSS_OFT) % (no_ss_max + 1);
                uint8_t mcs = ((rd & HT_MCS_MASK) >> HT_MCS_OFT) % (mcs_max + 1);
                if ((rc_ss->rate_map.ht[nss] & CO_BIT(mcs)) == 0) // rate index not allowed
                {
                    mcs = mcs_max;
                }
                rate_cfg |= (((rd & SHORT_GI_TX_RCX_MASK) >> SHORT_GI_TX_RCX_OFT) % (short_gi + 1)) << SHORT_GI_TX_RCX_OFT;
                #if RC_USE_MAX_BW
                rate_cfg |= bw_max << BW_TX_RCX_OFT;
                #else
                rate_cfg |= ((((rd & BW_TX_RCX_MASK) >> BW_TX_RCX_OFT) % (bw_max - bw_min + 1)) + bw_min) << BW_TX_RCX_OFT;
                #endif
                rate_cfg |= nss << HT_NSS_OFT;
                rate_cfg |= mcs << HT_MCS_OFT;
            }
        } break;

        #if NX_VHT
        case FORMATMOD_VHT:
        {
            uint8_t bw_max = rc_ss->bw_max;
            uint8_t short_gi = rc_ss->short_gi;
            uint8_t no_ss_max = rc_ss->no_ss;
            uint8_t mcs_max = rc_ss->mcs_max;

            uint8_t mcs = ((rd & VHT_MCS_MASK) >> VHT_MCS_OFT) % (mcs_max + 1);
            uint8_t no_ss = ((rd & VHT_NSS_MASK) >> VHT_NSS_OFT) % (no_ss_max + 1);
            uint8_t bw = 0;

            if (mcs > (7 + (rc_ss->rate_map.vht >> (no_ss << 1) & MAC_VHT_MCS_MAP_MSK)))
            {
                mcs = 7 + (rc_ss->rate_map.vht >> (no_ss << 1) & MAC_VHT_MCS_MAP_MSK);
            }

            #if RC_USE_MAX_BW
            bw = bw_max;
            #else
            bw = (((rd & BW_TX_RCX_MASK) >> BW_TX_RCX_OFT) % (bw_max - bw_min + 1)) + bw_min;
            #endif
            // do not allow some missing VHT rates
            if ((mcs == 6) && (bw == BW_80MHZ) && ((no_ss == 3) || (no_ss == 6)))
            {
                no_ss = no_ss - 1;
            }
            if (mcs == 9)
            {
                if ((bw == BW_20MHZ) && (no_ss != 2) && (no_ss != 5))
                {
                    mcs = 8;
                }
                if ((bw == BW_80MHZ) && (no_ss == 5))
                {
                    no_ss = no_ss - 1;
                }
                if ((bw == BW_160MHZ) && (no_ss == 2))
                {
                    no_ss = no_ss - 1;
                }
            }
            rate_cfg |= ((((rd & SHORT_GI_TX_RCX_MASK) >> SHORT_GI_TX_RCX_OFT) % (short_gi + 1)) << SHORT_GI_TX_RCX_OFT) |
                        (bw << BW_TX_RCX_OFT) |
                        (no_ss << VHT_NSS_OFT) |
                        (mcs << MCS_INDEX_TX_RCX_OFT);
        } break;
        #endif

        #if NX_HE
        case FORMATMOD_HE_SU:
        {
            uint8_t bw_max = rc_ss->bw_max;
            uint8_t no_ss_max = rc_ss->no_ss;
            uint8_t mcs_max = rc_ss->mcs_max;

            uint8_t mcs = ((rd & VHT_MCS_MASK) >> VHT_MCS_OFT) % (mcs_max + 1);
            uint8_t no_ss = ((rd & VHT_NSS_MASK) >> VHT_NSS_OFT) % (no_ss_max + 1);
            uint8_t gi = (rd & HE_GI_TYPE_TX_RCX_MASK) >> HE_GI_TYPE_TX_RCX_OFT;
            uint8_t bw = 0;

            if (mcs > (7 + 2 * (rc_ss->rate_map.he >> (no_ss << 1) & MAC_HE_MCS_MAP_MSK)))
            {
                mcs = 7 + 2* (rc_ss->rate_map.he >> (no_ss << 1) & MAC_HE_MCS_MAP_MSK);
            }

            if (gi > 2)
                gi = 2;

            #if RC_USE_MAX_BW
            bw = bw_max;
            #else
            bw = (((rd & BW_TX_RCX_MASK) >> BW_TX_RCX_OFT) % (bw_max - bw_min + 1)) + bw_min;
            #endif
            rate_cfg |= (gi << HE_GI_TYPE_TX_RCX_OFT) |
                        (bw << BW_TX_RCX_OFT) |
                        (no_ss << VHT_NSS_OFT) |
                        (mcs << MCS_INDEX_TX_RCX_OFT);
        } break;
        #endif

        default:
            break;
    }

    return rate_cfg;
}

/**
 ****************************************************************************************
 * @brief Checks if the rate configuration corresponds to a CCK rate.
 * @param[in] rate_config rate configuration
 * @return Whether the rate configuration corresponds to a CCK rate
 ****************************************************************************************
 */
static bool is_cck_group(uint16_t rate_config)
{
    uint8_t format = rc_get_format_mod(rate_config);
    uint8_t mcs = rc_get_mcs_index(rate_config);
    bool is_cck = 0;

    if ((format < FORMATMOD_HT_MF) && (mcs < HW_RATE_6MBPS))
    {
        is_cck = 1;
    }

    return is_cck;
}

/**
 ****************************************************************************************
 * @brief Selects a new rate to be used as trial rate and stores its index.
 * This function updates the trial period value, depending on the average AMPDU size, and
 * picks a new random rate from the sample table. The index of the selected rate is stored
 * in the station statistics structrure.
 * The new rate is discarded if:
 * - it is equal to one of the steps of the current retry chain
 * - probability of the sample is already greater than 95%
 * - in case of AMPDU transmission, HT station 2.4 GHz: the rate is a DSSS/CCK rate.
 * - TX duration of the sample is greater than TX durations of steps 0, 1 and 2,
 *   respecting the samples frequently skipped.
 * - current TP of the sample (taking into account the SW retries) is greater than
 *   TP of step 0 (HT, VHT), respecting the samples frequently skipped.
 * The trial sample shall be inserted as step 0 of the retry chain if (HT/VHT STA) its
 * expected throughput (calculated taking into account the current number of SW retry
 * request) is greater than the throughput of step 0; (NON-HT STA) if TX duration of the
 * sample is less than TX duration of step 0. Otherwise the trial rate is inserted as
 * step 1 of the retry chain.
 *
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @return Whether a trial sample has been selected and a trial transmission is requested
 ****************************************************************************************
 */
static bool rc_set_trial_tx(struct rc_sta_stats *rc_ss)
{
    struct rc_rate_stats *rc_rand = NULL;
    int i;
    uint16_t random_rate_idx = 0;
    uint32_t cur_max_tp_streams;
    uint32_t sample_streams;
    uint32_t sample_dur;
    uint32_t dur_retry_1;
    uint32_t dur_retry_2;
    uint32_t max_skipped;
    uint8_t format_mod = rc_ss->format_mod;
    #if NX_AMPDU_TX
    bool tx_ampdu = (rc_ss->info & RC_AGG_TX_MASK) != 0;
    #endif

    // Update countdown for trial tx
    rc_ss->sample_wait = RC_TRIAL_PERIOD * RC_TRUNC(rc_ss->avg_ampdu_len);

    // Pick a random sample from the sample table
    random_rate_idx = co_rand_hword() % rc_ss->no_samples;
    for (i = 0; i < rc_ss->no_samples; i++)
    {
        random_rate_idx = (random_rate_idx + i) % rc_ss->no_samples;
        rc_rand = &rc_ss->rate_stats[random_rate_idx];
        // Sampling might add some overhead (RTS, no aggregation)
        // to the frame. Hence, don't use sampling for the currently
        // used rate.
        if (random_rate_idx == rc_ss->retry_step_idx[0])
        {
            continue;
        }
        // Do not sample if the probability is already higher than 95%
        // to avoid wasting airtime
        if (rc_rand->probability > RC_FRAC(95, 100))
        {
            continue;
        }
        #if NX_AMPDU_TX
        // Do not sample if the random sample is a DSSS/CCK rate and
        // next packet is part of a AMPDU
        if (tx_ampdu && (rc_get_format_mod(rc_rand->rate_config) < FORMATMOD_HT_MF))
        {
            continue;
        }
        #endif

        // If all the previous tests are negative, we've found our sample rate, no need
        // to loop further
        break;
    }

    // Ensure we have a sample to try
    if (i == rc_ss->no_samples)
        return false;

    sample_dur = rc_get_duration(rc_rand);
    if (format_mod >= FORMATMOD_HT_MF)
    {
        // Make sure that lower rates get sampled only occasionally,
        // if the link is working perfectly.
        cur_max_tp_streams = rc_get_nss(rc_ss->rate_stats[rc_ss->retry_step_idx[0]].rate_config);
        sample_streams = rc_get_nss(rc_rand->rate_config);
        dur_retry_1 = rc_get_duration(&rc_ss->rate_stats[rc_ss->retry_step_idx[1]]);
        dur_retry_2 = rc_get_duration(&rc_ss->rate_stats[rc_ss->retry_step_idx[2]]);
        max_skipped = rc_ss->rate_stats[random_rate_idx].old_prob_available ? 32 : RC_TRUNC(rc_ss->avg_ampdu_len);
        if ((sample_dur >= dur_retry_1) &&
            (((cur_max_tp_streams - 1) < sample_streams) ||
             (sample_dur >= dur_retry_2)))
        {
            if (rc_rand->sample_skipped < max_skipped)
            {
                return false;
            }
            rc_ss->sample_slow += 1;
            if (rc_ss->sample_slow > 2)
            {
                if (rc_ss->sample_slow > 0xF)
                {
                    rc_ss->sample_slow = 0xF;
                }
                return false;
            }
        }
    }

    // Set trial rate index
    rc_ss->trial_idx = random_rate_idx;
    rc_ss->trial_status = RC_TRIAL_STATUS_WAIT_CFM;

    return true;
}

/**
 ****************************************************************************************
 * @brief Checks if a trial transmission is requested.
 *
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @return Whether a trial transmission is requested.
 ****************************************************************************************
 */
static bool rc_check_trial_tx(struct rc_sta_stats *rc_ss)
{
    // Do not sample if fixed rate is set
    if (rc_ss->info & RC_FIX_RATE_EN_MASK)
        return false;

    // Do not sample if a trial TX is already ongoing
    if (rc_ss->trial_status != RC_TRIAL_STATUS_WAIT)
        return false;

    // Check if it's time for a trial transmission
    if (rc_ss->sample_wait > 0)
    {
        rc_ss->sample_wait --;
        return false;
    }

    // Check if sample table needs to be updated
    if ((rc_ss->info & RC_SS_UPD_REQ_MASK) == RC_SS_UPD_REQ_MASK)
    {
        rc_get_new_samples(rc_ss);
        rc_ss->info &= ~RC_SS_UPD_REQ_MASK;
    }

    // Get a random trial rate from the sample table
    // and check if the rate can be inserted in the retry table
    return (rc_set_trial_tx(rc_ss));
}

/**
 ****************************************************************************************
 * @brief Sort sample table from lower to higher throughput
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @param[in] cur_tp pointer to the array of current throughputs
 ****************************************************************************************
 */
static void rc_sort_samples_tp(struct rc_sta_stats *rc_ss, uint32_t *cur_tp)
{
    struct rc_rate_stats rc_ss_tmp;
    uint32_t temp_tp;
    uint16_t i, j, last;
    uint32_t size = sizeof(rc_ss_tmp);

    last = rc_ss->no_samples;
    j = rc_ss->no_samples - 1;
    while (last > 0)
    {
        last = 0;
        for (i = 1; i < j; i++)
        {
            if (cur_tp[i] > cur_tp[i + 1])
            {
                memmove(&rc_ss_tmp, &rc_ss->rate_stats[i], size);
                memmove(&rc_ss->rate_stats[i], &rc_ss->rate_stats[i+1], size);
                memmove(&rc_ss->rate_stats[i+1], &rc_ss_tmp, size);
                temp_tp = cur_tp[i];
                cur_tp[i] = cur_tp[i+1];
                cur_tp[i+1] = temp_tp;
                last = i;
            }
        }
        j = last;
    }
}

/**
 ****************************************************************************************
 * @brief Calculate statistics, fills the new retry chain, updates the sample table.
 * This function updates throughput and probabilities of the sample table rates;
 * selects the 1st and 2nd maximum throughput rates, maximum probability rate and updates
 * the retry chain table.
 * Then it replaces not useful samples of the sample table with new ones.
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @param[in] init indicate if the statistics are updated during the initialization
 * @return Whether the retry chain has been modified.
 ****************************************************************************************
 */
static bool rc_update_stats(struct rc_sta_stats *rc_ss, bool init)
{
    bool upd = 0;
    uint16_t i = 0;
    uint32_t cur_tp[RC_MAX_N_SAMPLE];
    uint16_t old_rate_cfg[RATE_CONTROL_STEPS];
    struct rc_rate_stats *rc_rs;

    // Set old retry chain steps
    for (i = 0; i < RATE_CONTROL_STEPS; i++)
    {
        old_rate_cfg[i] = rc_ss->rate_stats[rc_ss->retry_step_idx[i]].rate_config;
    }

    if (rc_ss->ampdu_packets > 0)
    {
        // Calculate EWMA average number of MPDUs in each AMPDU frame
        rc_ss->avg_ampdu_len = ewma(rc_ss->avg_ampdu_len,
                                    RC_FRAC(rc_ss->ampdu_len, rc_ss->ampdu_packets),
                                    EWMA_LEVEL);
        rc_ss->ampdu_len = 0;
        rc_ss->ampdu_packets = 0;
    }

    #if NX_HE
    // Compute HE TB statistics
    rc_rs = &rc_ss->rate_stats[RC_HE_STATS_IDX];
    // Recalculate success probabilities and counters for a rate using EWMA
    rc_calc_prob_ewma(rc_rs);
    rc_rs->attempts = 0;
    rc_rs->success = 0;
    #endif

    rc_ss->sample_slow = 0;

    // Reset current TP and number of retries
    for (i = 0; i < rc_ss->no_samples; i++)
    {
        cur_tp[i] = 0;
        // Reset the number of request of SW retry before the TP calculation: this info is
        // already given by the current probability of success
        rc_ss->rate_stats[i].rate_allowed = 1;
    }

    // Calculate RC algorithm expected throughput for each sample of the stats table,
    // then fill the retry table
    if (rc_ss->fixed_rate_cfg == RC_FIXED_RATE_NOT_SET)
    {
        for (i = 0; i < rc_ss->no_samples; i++)
        {
            rc_rs = &rc_ss->rate_stats[i];
            // Recalculate success probabilities and counters for a rate using EWMA
            rc_calc_prob_ewma(rc_rs);
            // Calculate expected throughput based on the average A-MPDU length, taking into
            // account the expected number of retransmissions and their expected length
            cur_tp[i] = rc_calc_tp(rc_ss, i);
        }

        // Sort sample table by TP
        rc_sort_samples_tp(rc_ss, cur_tp);
        // Update the retry chain with: max TP, 2nd max TP, max probability, lowest rate
        rc_update_retry_chain(rc_ss, cur_tp);

        if (init == 0)
        {
            // Keep in mind we have to update the sample table
            rc_ss->info |= RC_SS_UPD_REQ_MASK;
            // Reset statistics
            for (i = 0; i < rc_ss->no_samples; i++)
            {
                // Reset statistics
                rc_ss->rate_stats[i].attempts = 0;
                rc_ss->rate_stats[i].success = 0;
            }
        }
    }
    else
    {
        upd = rc_update_stats_fixed_rate(rc_ss);
    }

    // check if the retry chain is changed
    for (i = 0; i < RATE_CONTROL_STEPS; i++)
    {
        if (old_rate_cfg[i] != rc_ss->rate_stats[rc_ss->retry_step_idx[i]].rate_config)
        {
            upd = 1;
            break;
        }
    }

    return upd;
}

/**
 ****************************************************************************************
 * @brief Updates the retry chain.
 * This function populates steps 0 and 1 of the retry chain table with the two highest
 * throughputs.
 * Then the function searches the rate, different from the previous two, with max
 * throughput if proability is greater than 95% or with highest probability and uses its
 * index for step 2 of the retry chain.
 * Step 3 is set to the lowest rate (sample 0 of the table) and its TP value is updated.
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @param[in] cur_tp pointer to the array of current throughputs
 ****************************************************************************************
 */
static void rc_update_retry_chain(struct rc_sta_stats *rc_ss, uint32_t *cur_tp)
{
    uint32_t max_prob_idx = 0;
    uint32_t max_tp = 0;
    uint32_t max_prob = 0;
    uint16_t i;
    uint16_t j;

    // Set new steps 0 and 1 of the retry table
    if (((rc_ss->info & RC_AGG_TX_MASK) == 0) && (cur_tp[0] > cur_tp[rc_ss->no_samples - 1]))
    {
        rc_ss->retry_step_idx[0] = 0;
        j = 1;
    }
    else
    {
        rc_ss->retry_step_idx[0] = rc_ss->no_samples - 1;
        j = 2;
    }
    // Allow CCK rates only if highest TP rate is a CCK rate
    if (!is_cck_group(rc_ss->rate_stats[rc_ss->retry_step_idx[0]].rate_config))
    {
        // CCK not allowed
        for (i = 0; i < (rc_ss->no_samples - 1); i++)
        {
            if (is_cck_group(rc_ss->rate_stats[i].rate_config))
            {
                rc_ss->rate_stats[i].rate_allowed = 0;
            }
        }
        rc_ss->rate_stats[rc_ss->retry_step_idx[0]].rate_allowed = 1;
    }
    rc_ss->retry_step_idx[1] = rc_ss->retry_step_idx[0];
    for (i = j; i < rc_ss->no_samples; i++)
    {
        if (rc_ss->rate_stats[rc_ss->no_samples - i].rate_allowed)
        {
            rc_ss->retry_step_idx[1] = rc_ss->no_samples - i;
            break;
        }
    }
    // find max probability
    max_prob_idx = rc_ss->retry_step_idx[1];
    for (i = i + 1; i < rc_ss->no_samples; i++)
    {
        if (rc_ss->rate_stats[rc_ss->no_samples - i].rate_allowed)
        {
            max_prob_idx = rc_ss->no_samples - i;
            break;
        }
    }
    max_tp = cur_tp[max_prob_idx];
    max_prob = rc_ss->rate_stats[max_prob_idx].probability;
    for (i = 0; i < rc_ss->retry_step_idx[1]; i++)
    {
        if (rc_ss->rate_stats[i].rate_allowed == 0)
        {
            continue;
        }
        if (i == rc_ss->retry_step_idx[0])
        {
            continue;
        }
        if (rc_ss->rate_stats[i].probability >= RC_FRAC(95, 100))
        {
            if (cur_tp[i] >= max_tp)
            {
                max_prob_idx = i;
                max_tp = cur_tp[i];
                max_prob = rc_ss->rate_stats[i].probability;
            }
        }
        else if (rc_ss->rate_stats[i].probability >= max_prob)
        {
            max_prob_idx = i;
            max_tp = cur_tp[i];
            max_prob = rc_ss->rate_stats[i].probability;
        }
    }
    // Set new step 2 of the retry table
    rc_ss->retry_step_idx[2] = max_prob_idx;
    // Set lowest rate (step 3) in the retry table
    rc_ss->retry_step_idx[3] = 0;
}

/**
 ****************************************************************************************
 * @brief Checks if the rate configuration is allowed.
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @param[in] rate_config rate configuration to be verified
 * @return Whether the rate configuration is allowed.
 ****************************************************************************************
 */
static bool rc_check_valid_rate(struct rc_sta_stats *rc_ss, uint16_t rate_config)
{
    uint8_t format = rc_get_format_mod(rate_config);
    bool rate_allowed = 1;

    switch (format)
    {
        case FORMATMOD_NON_HT:
        case FORMATMOD_NON_HT_DUP_OFDM:
        {
            uint8_t r_idx = rc_get_mcs_index(rate_config);
            if ((rc_ss->rate_map_l & CO_BIT(r_idx)) == 0) // rate index not allowed
            {
                rate_allowed = 0;
            }
        } break;
        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
        {
            uint8_t nss = rc_get_nss(rate_config);
            uint8_t mcs = rc_get_mcs_index(rate_config);
            if ((rc_ss->rate_map.ht[nss] & CO_BIT(mcs)) == 0) // rate index not allowed
            {
                rate_allowed = 0;
            }
        } break;

        #if NX_VHT
        case FORMATMOD_VHT:
        {
            uint8_t bw = rc_get_bw(rate_config);
            uint8_t nss = rc_get_nss(rate_config);
            uint8_t mcs = rc_get_mcs_index(rate_config);
            // check if MCS allowed in the rate bitmap
            if (mcs > (7 + (rc_ss->rate_map.vht >> (nss << 1) & MAC_VHT_MCS_MAP_MSK)))
            {
                rate_allowed = 0;
            }
            // do not allow some missing VHT rates
            if ((mcs == 6) && (bw == BW_80MHZ) && ((nss == 3) || (nss == 6)))
            {
                rate_allowed = 0;
            }
            else if (mcs == 9)
            {
                if ((bw == BW_20MHZ) && (nss != 2) && (nss != 5))
                {
                    rate_allowed = 0;
                }
                else if ((bw == BW_80MHZ) && (nss == 5))
                {
                    rate_allowed = 0;
                }
                else if ((bw == BW_160MHZ) && (nss == 2))
                {
                    rate_allowed = 0;
                }
            }
        } break;
        #endif

        #if NX_HE
        case FORMATMOD_HE_SU:
        {
            uint8_t nss = rc_get_nss(rate_config);
            uint8_t mcs = rc_get_mcs_index(rate_config);
            // check if MCS allowed in the rate bitmap
            if (mcs > (7 + 2 * (rc_ss->rate_map.vht >> (nss << 1) & MAC_VHT_MCS_MAP_MSK)))
            {
                rate_allowed = 0;
            }
        } break;
        #endif

        default:
            break;
    }

    return rate_allowed;
}

/**
 ****************************************************************************************
 * @brief Selects new samples for the sample table.
 * This function selects a new set of samples:
 * - random rate
 * - same rate configuration of best throughput rate, but with opposite guard interval
 *   (only if HT or VHT station)
 * - MCS index / rate index above the best throughput rate, short GI if allowed
 * - MCS index / rate index below the best throughput rate, short GI if allowed
 * - MCS index / rate index above the 2nd best throughput rate, short GI if allowed
 * - MCS index / rate index below the 2nd best throughput rate, short GI if allowed
 * Samples not available are set to 0xFF.
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @param[in] new_rate_cfg_array pointer to the array of new samples rate configuration
 * @param[in] n_new_samples number of new samples to be set
 ****************************************************************************************
 */
static void rc_get_new_sample_rates(struct rc_sta_stats *rc_ss,
                                    uint16_t *new_rate_cfg_array,
                                    uint8_t n_new_samples)
{
    uint8_t i = 0;
    uint16_t tmp_rate_cfg;
    uint16_t max_tp_rate_cfg = rc_ss->rate_stats[rc_ss->retry_step_idx[0]].rate_config;
    uint16_t max_tp_2_rate_cfg = rc_ss->rate_stats[rc_ss->retry_step_idx[1]].rate_config;

    memset(new_rate_cfg_array, -1, n_new_samples*sizeof(*new_rate_cfg_array));

    for (i = 0; i < n_new_samples; i++)
    {
        switch (i)
        {
            // Random rate
            case 0:
            {
                new_rate_cfg_array[i] =  rc_new_random_rate(rc_ss);
            }
            break;

            // If max_tp is long guard, pick same configuration but short guard
            // If max_tp is short guard, pick same configuration but long guard
            case 1:
            {
                #if NX_HE
                if (rc_get_format_mod(max_tp_rate_cfg) == FORMATMOD_HE_SU)
                {
                    uint8_t gi = (rc_get_he_gi(max_tp_rate_cfg) + 1) % 3;
                    max_tp_rate_cfg &= ~HE_GI_TYPE_TX_RCX_MASK;
                    new_rate_cfg_array[i] = max_tp_rate_cfg | (gi << HE_GI_TYPE_TX_RCX_OFT);
                }
                else
                #endif
                if ((rc_get_format_mod(max_tp_rate_cfg) >= (uint8_t)FORMATMOD_HT_MF) &&
                    (rc_ss->short_gi == 1))
                {
                    new_rate_cfg_array[i] = max_tp_rate_cfg ^ SHORT_GI_TX_RCX_MASK;
                }
            }
            break;

            // MCS index above the best throughput rate
            case 2:
            {
                tmp_rate_cfg = rc_set_next_mcs_index(rc_ss, max_tp_rate_cfg);
                if ((tmp_rate_cfg != max_tp_rate_cfg) &&
                    (rc_check_valid_rate(rc_ss, tmp_rate_cfg)))
                {
                    new_rate_cfg_array[i] = tmp_rate_cfg;
                }
            }
            break;

            // MCS index below the best throughput rate
            case 3:
            {
                tmp_rate_cfg = rc_set_previous_mcs_index(rc_ss, max_tp_rate_cfg);
                if ((tmp_rate_cfg != max_tp_rate_cfg) &&
                    (rc_check_valid_rate(rc_ss, tmp_rate_cfg)))
                {
                    new_rate_cfg_array[i] = tmp_rate_cfg;
                }
            }
            break;

            // MCS index above the 2nd best throughput rate
            case 4:
            {
                tmp_rate_cfg = rc_set_next_mcs_index(rc_ss, max_tp_2_rate_cfg);
                if ((tmp_rate_cfg != max_tp_2_rate_cfg) &&
                    (rc_check_valid_rate(rc_ss, tmp_rate_cfg)))
                {
                    new_rate_cfg_array[i] = tmp_rate_cfg;
                }
            }
            break;

            // MCS index below the 2nd best throughput rate
            case 5:
            {
                tmp_rate_cfg = rc_set_previous_mcs_index(rc_ss, max_tp_2_rate_cfg);
                if ((tmp_rate_cfg != max_tp_2_rate_cfg) &&
                    (rc_check_valid_rate(rc_ss, tmp_rate_cfg)))
                {
                    new_rate_cfg_array[i] = tmp_rate_cfg;
                }
            }
            break;

            default:
                break;
        }
    }
}

/**
 ****************************************************************************************
 * @brief Replaces not useful samples of the sample table with new ones.
 * This function discards some of the samples from the sample table and replaces them
 * with new ones. The samples are discarded if:
 * - samples are not used in the retry chain
 * - their probability is less than 50% or
 * - samples have already been skipped more than 10 times.
 * The new samples are selected in the function rc_get_new_sample_rates().
 * @param[in] rc_ss pointer to rate control station statistics structure
 ****************************************************************************************
 */
static void rc_get_new_samples(struct rc_sta_stats *rc_ss)
{
    // select new samples and reset statistics
    uint32_t new_sample = 0;
    uint16_t new_rate_cfg_array[RC_MAX_N_SAMPLE-4];
    uint16_t i = 1;

    if (rc_ss->no_samples < RC_MAX_N_SAMPLE)
    {
        return;
    }

    rc_get_new_sample_rates(rc_ss, new_rate_cfg_array, RC_MAX_N_SAMPLE-4);

    while (i < rc_ss->no_samples)
    {
        struct rc_rate_stats *rc_rs = &rc_ss->rate_stats[i];
        uint16_t tmp_rate_cfg;

        // check if the sample has to be replaced and if a new sample in the list is available
        if (((rc_rs->probability < RC_FRAC(50, 100)) || (rc_rs->sample_skipped > 10)) &&
            (i != rc_ss->retry_step_idx[0]) &&
            (i != rc_ss->retry_step_idx[1]) &&
            (i != rc_ss->retry_step_idx[2]) &&
            (new_sample < (RC_MAX_N_SAMPLE-4)))
        {
            tmp_rate_cfg = new_rate_cfg_array[new_sample];
            if ((tmp_rate_cfg != (uint16_t)-1) && (!rc_check_rate_duplicated(rc_ss, tmp_rate_cfg)))
            {
                rc_rs->rate_config = tmp_rate_cfg;
                // reset new sample probability
                rc_rs->probability = 0;
                rc_rs->old_prob_available = 0;
                #if NX_DEBUG
                // sanity check
                rc_check_rate_config(rc_ss, i);
                #endif
                // next sample in the table
                i += 1;
            }
            new_sample += 1;
        }
        else
        {
            // go to next sample in the table
            i += 1;
        }
    }
}

/**
 ****************************************************************************************
 * @brief Gets the rate configuration of the lowest allowed rate for the station.
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @return Rate configuration of the lowest allowed rate
 ****************************************************************************************
 */
static uint16_t rc_get_lowest_rate_config(struct rc_sta_stats *rc_ss)
{
    uint16_t rate_config = 0;

    switch (rc_ss->format_mod)
    {
        case FORMATMOD_NON_HT:
        case FORMATMOD_NON_HT_DUP_OFDM:
        {
            if (rc_ss->r_idx_min == 0) // CCK
            {
                // CCK group: lowest rate has preamble type long, rate index 0, BW_20MHZ
                rate_config = (FORMATMOD_NON_HT << FORMAT_MOD_TX_RCX_OFT) |
                        (1 << PRE_TYPE_TX_RCX_OFT);
            }
            else
            {
                // NON-HT: lowest rate has lowest allowed rate index, BW_20MHZ
                rate_config = (FORMATMOD_NON_HT << FORMAT_MOD_TX_RCX_OFT) |
                        (rc_ss->r_idx_min << MCS_INDEX_TX_RCX_OFT);
            }
        } break;
        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
        {
            if (rc_ss->r_idx_min == 0) // CCK
            {
                // CCK group: lowest rate has preamble type long, rate index 0, BW_20MHZ
                rate_config = (FORMATMOD_NON_HT << FORMAT_MOD_TX_RCX_OFT) |
                        (1 << PRE_TYPE_TX_RCX_OFT);
            }
            else
            {
                // HT group: lowest rate is HT, MCS0, BW_20MHZ
                rate_config = (rc_ss->format_mod << FORMAT_MOD_TX_RCX_OFT);
            }
        } break;

        #if NX_VHT
        case FORMATMOD_VHT:
        {
            // VHT only: lowest rate is MCS0, BW_20MHZ
            rate_config = (rc_ss->format_mod << FORMAT_MOD_TX_RCX_OFT);
        } break;
        #endif

        #if NX_HE
        case FORMATMOD_HE_SU:
        {
            if (rc_ss->r_idx_min == 0) // CCK
            {
                // CCK group: lowest rate has preamble type long, rate index 0, BW_20MHZ
                rate_config = (FORMATMOD_NON_HT << FORMAT_MOD_TX_RCX_OFT) |
                              (1 << PRE_TYPE_TX_RCX_OFT);
            }
            else
            {
                // HE group: lowest rate is HE, MCS0, BW_20MHZ. GI1.6
                rate_config = (rc_ss->format_mod << FORMAT_MOD_TX_RCX_OFT) |
                              GI_TYPE_1_6;
            }
        } break;
        #endif

        default:
            break;
    }

    return rate_config;
}

/**
 ****************************************************************************************
 * @brief Gets the rate configuration of the initial rate for the station.
 * Highest rate index for NON-HT, MCS7 or highest allowed for HT, MCS7 for VHT
 *
 * @param[in] rc_ss Pointer to rate control station statistics structure
 * @return Initial rate configuration
 ****************************************************************************************
 */
static uint16_t rc_get_initial_rate_config(struct rc_sta_stats *rc_ss)
{
    uint16_t rate_config = 0;

    switch (rc_ss->format_mod)
    {
        case FORMATMOD_NON_HT:
        case FORMATMOD_NON_HT_DUP_OFDM:
        {
            rate_config = (rc_ss->format_mod << FORMAT_MOD_TX_RCX_OFT) |
                    (rc_ss->type << PRE_TYPE_TX_RCX_OFT) |
                    (rc_ss->r_idx_max << MCS_INDEX_TX_RCX_OFT);
        } break;
        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
        {
            uint8_t mcs = 31 - co_clz((uint32_t)rc_ss->rate_map.ht[rc_ss->no_ss]);
            rate_config = (rc_ss->format_mod << FORMAT_MOD_TX_RCX_OFT) |
                    (rc_ss->short_gi << SHORT_GI_TX_RCX_OFT) |
                    (rc_ss->bw_max << BW_TX_RCX_OFT) |
                    (rc_ss->no_ss << HT_NSS_OFT) |
                    (mcs << MCS_INDEX_TX_RCX_OFT);
        } break;

        #if NX_VHT
        case FORMATMOD_VHT:
        {
            rate_config = (rc_ss->format_mod << FORMAT_MOD_TX_RCX_OFT) |
                    (rc_ss->short_gi << SHORT_GI_TX_RCX_OFT) |
                    (rc_ss->bw_max << BW_TX_RCX_OFT) |
                    (rc_ss->format_mod == FORMATMOD_VHT ? rc_ss->no_ss << VHT_NSS_OFT : rc_ss->no_ss << HT_NSS_OFT) |
                    (7 << MCS_INDEX_TX_RCX_OFT);
        } break;
        #endif

        #if NX_HE
        case FORMATMOD_HE_SU:
        {
            rate_config = (rc_ss->format_mod << FORMAT_MOD_TX_RCX_OFT) |
                          (rc_ss->bw_max << BW_TX_RCX_OFT) |
                          (rc_ss->no_ss << VHT_NSS_OFT) |
                          (7 << MCS_INDEX_TX_RCX_OFT) | GI_TYPE_0_8;
        } break;
        #endif

        default:
            break;
    }

    return rate_config;
}

/**
 ****************************************************************************************
 * @brief Gets the number of samples to be used by the RC algorithm.
 * This function calculates the number of samples to be used for the sample table of the
 * RC algorithm, depending on the number of available rates for the station.
 * The maximum number of samples is limited to RC_MAX_N_SAMPLE.
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @return Number of samples to be used
 ****************************************************************************************
 */
static uint16_t rc_get_num_samples(struct rc_sta_stats *rc_ss)
{
    uint32_t i, j;
    uint16_t n_sample = 0;

    switch (rc_ss->format_mod)
    {
        case FORMATMOD_NON_HT:
        case FORMATMOD_NON_HT_DUP_OFDM:
        {
            uint16_t r_map = rc_ss->rate_map_l;
            n_sample += (r_map & 1); // HW_RATE_1MBPS: only long preamble
            for (i = HW_RATE_2MBPS; i <= HW_RATE_11MBPS; i++)
            {
                n_sample += (((r_map >> i) & 1) << (1-rc_ss->type));
            }
            for (i = HW_RATE_6MBPS; i <= HW_RATE_54MBPS; i++)
            {
                n_sample += ((r_map >> i) & 1);
            }
        } break;

        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
        {
            uint32_t mult = ((rc_ss->bw_max + 1) << rc_ss->short_gi) * (rc_ss->no_ss + 1);
            uint8_t mcs_map = rc_ss->rate_map.ht[0];
            for (j = 0; j < 8; j++)  // MCS 0-7
            {
                n_sample += ((mcs_map & 1) * mult);
                mcs_map = mcs_map >> 1;
            }
            // cck rates
            uint16_t r_map = rc_ss->rate_map_l;
            n_sample += (r_map & 1); // HW_RATE_1MBPS: only long preamble
            for (i = HW_RATE_2MBPS; i <= HW_RATE_11MBPS; i++)
            {
                n_sample += (((r_map >> i) & 1) << (1-rc_ss->type));
            }
        } break;

        #if NX_VHT
        case FORMATMOD_VHT:
        {
            // VHT rates
            uint32_t mult = ((rc_ss->bw_max + 1) << rc_ss->short_gi) * (rc_ss->no_ss + 1);
            uint8_t n_mcs;
            switch (rc_ss->rate_map.vht & MAC_VHT_MCS_MAP_MSK)
            {
                case 0:
                    n_mcs = 8;
                    break;
                case 1:
                case 2:
                    n_mcs = 9;
                    break;
                default:
                    n_mcs = 8;
                    break;
            }
            n_sample = n_mcs * mult;
        } break;
        #endif

        #if NX_HE
        case FORMATMOD_HE_SU:
        {
            // HE rates
            uint32_t mult = ((rc_ss->bw_max + 1) * (rc_ss->short_gi + 1)) * (rc_ss->no_ss + 1);
            uint8_t n_mcs;
            switch (rc_ss->rate_map.he & MAC_HE_MCS_MAP_MSK)
            {
                case 0:
                    n_mcs = 8;
                    break;
                case 1:
                    n_mcs = 9;
                    break;
                case 2:
                    n_mcs = 10;
                    break;
                default:
                    n_mcs = 8;
                    break;
            }
            n_sample = n_mcs * mult;
        } break;
        #endif

        default:
            break;
    }

    if (n_sample > RC_MAX_N_SAMPLE)
    {
        n_sample = RC_MAX_N_SAMPLE;
    }

    return n_sample;
}

/**
 ****************************************************************************************
 * @brief Sets the rate configurations of the sample table entries.
 * This function initializes the rates of the sample table:
 *  - entry 0 is the lowest rate
 *  - entry no_samples-1 is the highest rate
 *  - entry from 1 to no_samples-2 are chosen randomly between the remaining rates.
 * @param[in] rc_ss pointer to rate control station statistics structure
 ****************************************************************************************
 */
static void rc_set_rate_configs(struct rc_sta_stats *rc_ss)
{
    uint16_t i;

    // Reset rate configuration
    for (i = 0; i < rc_ss->no_samples; i++)
    {
        struct rc_rate_stats *rc_rs = &rc_ss->rate_stats[i];
        rc_rs->rate_config = 0xFFFF;
        rc_rs->attempts = 0;
        rc_rs->success = 0;
        rc_rs->probability = 0;
        rc_rs->sample_skipped = 0;
        rc_rs->old_prob_available = 0;
        rc_rs->rate_allowed = 1;
    }

    // Set base rate configuration (lowest allowed rate) as sample number 0
    rc_ss->rate_stats[0].rate_config = rc_get_lowest_rate_config(rc_ss);
    // Set last sample: highest rate index for NON-HT, MCS7 or highest allowed for HT,
    // MCS7 for VHT
    rc_ss->rate_stats[rc_ss->no_samples - 1].rate_config = rc_get_initial_rate_config(rc_ss);
    // Set initial sample rates (do not use duplicated samples) - sample 0 is always the lowest rate
    for (i = 1; i < (rc_ss->no_samples - 1);)
    {
        // Get random rates for the other samples
        uint16_t tmp_rate = rc_new_random_rate(rc_ss);
        // do not use duplicated samples
        if (!rc_check_rate_duplicated(rc_ss, tmp_rate))
        {
            rc_ss->rate_stats[i].rate_config = tmp_rate;
            i++;
        }
    }
    #if NX_DEBUG
    rc_check_stats_rate_config(rc_ss);
    #endif
}

/**
 ****************************************************************************************
 * @brief RC algorithm retry chain and sample table initialization.
 * This function initializes the rates of the sample table and then
 * it sets the retry chain:
 *  - step 0: set to the highest rate (entry no_samples-1 of sample table)
 *  - step 3: set to the lowest rate (entry 0 of sample table)
 *  - step 1 and 2: set randomly (random entry between 1 and no_samples-2)
 * @param[in] sta_idx index of the station
 ****************************************************************************************
 */
static void rc_init_rates(uint8_t sta_idx)
{
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    struct sta_pol_tbl_cntl *pt = &sta->pol_tbl;
    struct rc_sta_stats *rc_ss = pt->sta_stats;
    ASSERT_ERR(rc_ss != NULL);
    uint16_t i;

    // Select new random rates for the stats table
    rc_set_rate_configs(rc_ss);

    // set retry chain steps 0, 1 and 2
    for (i = 0; i < (RATE_CONTROL_STEPS-1); i++)
    {
        rc_ss->retry_step_idx[i] = (rc_ss->no_samples - 1) - i;
    }
    // Set lowest rate (entry 0 of the sample table) as last step of the retry chain
    rc_ss->retry_step_idx[RATE_CONTROL_STEPS-1] = 0;
    // Initialize average AMPDU length
    rc_ss->avg_ampdu_len = RC_FRAC(1, 1);
    // Disable fixed rate
    rc_ss->fixed_rate_cfg = RC_FIXED_RATE_NOT_SET;
    // Reset RC flags
    rc_ss->info = 0;
    // Initialize RC algorithm table
    rc_update_stats(rc_ss, 1);
    // Initialize trial period
    rc_ss->sample_wait =  5;
    // Initialize trial index
    rc_ss->trial_idx = rc_ss->no_samples - RATE_CONTROL_STEPS;
    // Initialize trial status
    rc_ss->trial_status = RC_TRIAL_STATUS_WAIT;
}

#if NX_AMSDU_TX
/**
 ****************************************************************************************
 * @brief Sets the maximum AMSDU length, depending on the number of spatial streams.
 * @param[in] rc_ss pointer to rate control station statistics structure
 ****************************************************************************************
 */
static void rc_set_max_amsdu_len(struct rc_sta_stats *rc_ss)
{
    uint8_t step_0_idx = rc_ss->retry_step_idx[rc_ss->sw_retry_step];
    uint16_t rate_config = rc_ss->rate_stats[step_0_idx].rate_config;
    uint16_t max_amsdu_size = 0;

    // Allow A-MSDU only if higher MCS are used (i.e >= MCS5)
    if (rc_get_mcs_index(rate_config) >= 5)
    {
        // set the max size of AMSDU depending on the number of spatial streams:
        // - 1 SS: max size 4
        // - 2 SS: max size 6
        // - 3-8 SS: max size 6
        if (rc_ss->no_ss == 0)
        {
            max_amsdu_size = 4;
        }
        else if (rc_ss->no_ss == 1)
        {
            max_amsdu_size = 6;
        }
        else
        {
            max_amsdu_size = 6;
        }
    }
    else
    {
        max_amsdu_size = 0;
    }
    rc_ss->curr_amsdu_len = co_min(rc_ss->max_amsdu_len, max_amsdu_size*1550);
}
#endif //NX_AMSDU_TX

#if NX_AMPDU_TX
/**
 ****************************************************************************************
 * @brief Sets/resets the aggregation flag
 * This function updates the value of the aggregation flag depending on the rates set in
 * the retry chain. The aggregation is disabled only for legacy rates, since it's not
 * supported. The aggregation is disabled if:
 * - one the steps 0-2 of the retry chain is a legacy rate
 * - first step of the retry chain is HT/VHT, 1SS, with low MCS and low probability (<10%)
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @param[in] tx_ampdu pointer to the aggregation flag
 ****************************************************************************************
 */
static void rc_set_aggregation_flag(struct rc_sta_stats *rc_ss, bool *tx_ampdu)
{
    bool agg_enabled = 1;

    // check if aggregation has to be disabled
    do
    {
        // disable aggregation only for stations allowing legacy rates
        if (rc_ss->r_idx_min > HW_RATE_54MBPS)
        {
            break;
        }
        // disable aggregation if one of the steps 0-2 is a DSSS/CCK rate
        uint16_t rate_0 = rc_ss->rate_stats[rc_ss->retry_step_idx[0]].rate_config;
        uint16_t rate_1 = rc_ss->rate_stats[rc_ss->retry_step_idx[1]].rate_config;
        uint16_t rate_2 = rc_ss->rate_stats[rc_ss->retry_step_idx[2]].rate_config;
        if (rc_get_format_mod(rate_0) < FORMATMOD_HT_MF ||
            rc_get_format_mod(rate_1) < FORMATMOD_HT_MF ||
            rc_get_format_mod(rate_2) < FORMATMOD_HT_MF)
        {
            agg_enabled = 0;
            break;
        }
        // If fixed rate is set, do not disable aggregation because of low probability
        if (rc_ss->info & RC_FIX_RATE_EN_MASK)
        {
            break;
        }
        // disable aggregation if first step:
        // 1 SS, probability of success < 10%. MCS0 or MCS1
        uint8_t step_0 = rc_ss->sw_retry_step;
        uint16_t step_0_idx = rc_ss->retry_step_idx[step_0];
        struct rc_rate_stats *stats_step_0 = &rc_ss->rate_stats[step_0_idx];
        uint8_t step_0_mcs = rc_get_mcs_index(stats_step_0->rate_config);
        uint8_t step_0_nss = rc_get_nss(stats_step_0->rate_config);
        if (((step_0_mcs <= 2) &&
            (step_0_nss == 0) &&
            (stats_step_0->probability < RC_FRAC(10, 100))) ||
            (stats_step_0->probability < RC_FRAC(1, 100)))
        {
            agg_enabled = 0;
        }
    } while (0);

    rc_ss->info &= ~RC_AGG_ALLOWED_MASK;
    rc_ss->info |= (agg_enabled << RC_AGG_ALLOWED_OFT);

    if (tx_ampdu != NULL)
        *tx_ampdu = agg_enabled;
}
#endif //NX_AMPDU_TX

/**
 ****************************************************************************************
 * @brief Update the retry chain and statistics when a fix rate is set.
 * This function inserts the fix rate in the sample table and updates the retry chain, if
 * a fixed rate request is pending.
 * Otherwise, it updates the statistics for the fixed rate.
 * @param[in] rc_ss pointer to rate control station statistics structure
 * @return Whether the retry chain has been modified.
 ****************************************************************************************
 */
static bool rc_update_stats_fixed_rate(struct rc_sta_stats *rc_ss)
{
    bool update_requested = 0;
    uint16_t i = 0;

    // Fixed rate request
    if (rc_ss->info & RC_FIX_RATE_REQ_MASK)
    {
        uint8_t idx = 0;
        // Check if selected rate is in the sample table
        while (idx < rc_ss->no_samples)
        {
            if (rc_ss->fixed_rate_cfg == rc_ss->rate_stats[idx].rate_config)
            {
                break;
            }
            idx++;
        }
        // If not present in the sample table, insert it
        if (idx == rc_ss->no_samples)
        {
            idx = rc_ss->no_samples - 1;
            rc_ss->rate_stats[idx].rate_config = rc_ss->fixed_rate_cfg;
            rc_ss->rate_stats[idx].probability = 0;
        }
        // Set retry chain
        for (i = 0; i < 3; i++)
        {
            rc_ss->retry_step_idx[i] = idx;
        }
        // Reset statistics
        for (i = 0; i < rc_ss->no_samples; i++)
        {
            rc_ss->rate_stats[i].attempts = 0;
            rc_ss->rate_stats[i].success = 0;
        }
        // Reset trial status
        rc_ss->trial_status = RC_TRIAL_STATUS_WAIT;
        // Set fix rate status
        rc_ss->info &= ~RC_FIX_RATE_STATUS_MASK;
        rc_ss->info |= RC_FIX_RATE_EN_MASK;
        update_requested = 1;
    }
    // Fixed rate is set: update statistics and reset counters
    else
    {
        uint8_t step_0_idx = rc_ss->retry_step_idx[0];
        struct rc_rate_stats *rc_rs = &rc_ss->rate_stats[step_0_idx];
        // Calculate success probability
        rc_calc_prob_ewma(rc_rs);
        // Reset statistics
        for (i = 0; i < rc_ss->no_samples; i++)
        {
            rc_ss->rate_stats[i].attempts = 0;
            rc_ss->rate_stats[i].success = 0;
        }
    }

    return update_requested;
}

/**
 ****************************************************************************************
 * @brief Updates Rate Control statistics after sending an A-MPDU
 *
 * @param[in] txdesc   TX descriptor of the MPDU just transmitted (or of the first MPDU
 *                     of the transmitted A-MPDU)
 * @param[in] attempts Number of attempts with the current retry chain
 * @param[in] failures Number of failures with the current retry chain
 * @param[in] tx_ampdu '1' if the transmitted message is an AMPDU, '0' otherwise
 ****************************************************************************************
 */
static void rc_update_counters(struct txdesc *txdesc, uint32_t attempts,
                               uint32_t failures, bool tx_ampdu)
{
    uint8_t sta_idx = txdesc->host.staid;
    bool trial_tx;
    #if NX_HE
    bool he_tb = false;
    #endif
    struct sta_info_tag *sta;
    struct sta_pol_tbl_cntl *rc;
    struct rc_sta_stats *rc_ss;

    if (sta_idx >= NX_REMOTE_STA_MAX)
        return;

    trial_tx = ((txdesc->host.flags & TXU_CNTRL_RC_TRIAL) == TXU_CNTRL_RC_TRIAL);
    #if NX_HE
    he_tb = (txdesc->lmac.hw_desc->thd.statinfo & HE_TB_TX_BIT) != 0;
    #endif
    sta = &sta_info_tab[sta_idx];
    if (sta->inst_nbr == 0xFF)
        return;

    rc = &sta->pol_tbl;
    rc_ss = rc->sta_stats;
    ASSERT_ERR(rc_ss != NULL);

    // Check if we need to take into account the result of this transmission to update
    // the statistics. This is not the case if the packet was transmitted using a older
    // retry chain version than the currently active one or if it was transmitted in a HE
    // TB PPDU
    if ((RC_CONTROL_GET(txdesc->umac.rc_control, RETRY_CHAIN_VER) != rc_ss->retry_chain_ver)
        #if NX_HE
        && !he_tb
        #endif
       )
    {
        if (trial_tx)
            rc_ss->trial_status = RC_TRIAL_STATUS_WAIT;
        return;
    }

    PROF_RC_UPD_COUNTERS_SET();

    // update the number of AMPDUs transmitted
    rc_ss->ampdu_packets += 1;

    #if NX_AMPDU_TX
    if (tx_ampdu)
    {
        struct rc_rate_stats *rc_rs;
        // update the number of MPDUs transmitted (success and failures)
        rc_ss->ampdu_len += attempts;
        #if NX_HE
        if (he_tb)
        {
            rc_rs = &rc_ss->rate_stats[RC_HE_STATS_IDX];
        }
        else
        #endif
        // update for trial tx
        if (trial_tx)
        {
            PROF_RC_UPD_COUNTERS_TRIAL_SET();
            // Set number of attempts and success for the trial rate
            rc_rs = &rc_ss->rate_stats[rc_ss->trial_idx];
            rc_ss->trial_status = RC_TRIAL_STATUS_WAIT;
            PROF_RC_UPD_COUNTERS_TRIAL_CLR();
        }
        else
        {
            // Set number of attempts and success for first rate of the retry chain
            uint8_t retry_step = RC_CONTROL_GET(txdesc->umac.rc_control, SW_RETRY_STEP);
            uint16_t step_0_idx = rc_ss->retry_step_idx[retry_step];
            rc_rs = &rc_ss->rate_stats[step_0_idx];

            // Check if we should derogate from the normal policy table for next A-MPDU
            // transmissions. We consider that having more than 50% PER inside a A-MPDU
            // will create many retries, and in such case we decrease the MCS until we
            // get good PER again
            if ((2 * failures) > attempts)
            {
                PROF_RC_SW_RETRY_SET();
                if (rc_ss->sw_retry_step < 2)
                    rc_ss->sw_retry_step++;
                else if (rc_ss->sw_retry_step == 2)
                    // Force the statistics to be computed earlier
                    rc_ss->last_rc_time -= (RC_PERIOD_TOUT * 3) / 4;
                PROF_RC_SW_RETRY_CLR();
            }
            else if (rc_ss->sw_retry_step)
            {
                // PER is OK, increase again the MCS
                rc_ss->sw_retry_step--;
            }
        }
        rc_rs->attempts += attempts;
        rc_rs->success += (attempts - failures);
        ASSERT_ERR(rc_rs->attempts >= rc_rs->success);
    }
    else
    #endif
    {
        // update the number of MPDUs transmitted (success and failures)
        rc_ss->ampdu_len += 1;
        #if NX_HE
        if (he_tb)
        {
            struct rc_rate_stats *rc_rs = &rc_ss->rate_stats[RC_HE_STATS_IDX];
            rc_rs->attempts += attempts;
            rc_rs->success += (attempts - failures);
        }
        else
        #endif
        // update for trial tx
        if (trial_tx)
        {
            PROF_RC_UPD_COUNTERS_TRIAL_SET();
            // update the number of attempts and success for each rate of the retry chain
            for (uint32_t i = 0; (i < RATE_CONTROL_STEPS) && (attempts > 0); i++)
            {
                uint16_t rate_idx;
                struct rc_rate_stats *rc_rs;
                if (i == 0)
                    rate_idx = rc_ss->trial_idx;
                else if (i == 1)
                    rate_idx = rc_ss->retry_step_idx[0];
                else
                    rate_idx = rc_ss->retry_step_idx[i];

                rc_rs = &rc_ss->rate_stats[rate_idx];
                if (failures >= RC_MAX_NUM_RETRY)
                {
                    rc_rs->attempts += RC_MAX_NUM_RETRY;
                    attempts -= RC_MAX_NUM_RETRY;
                    failures -= RC_MAX_NUM_RETRY;
                }
                else
                {
                    rc_rs->attempts += attempts;
                    rc_rs->success += (attempts - failures);
                    attempts = 0;
                    failures = 0;
                }
                ASSERT_ERR(rc_rs->attempts >= rc_rs->success);
            }
            rc_ss->trial_status = RC_TRIAL_STATUS_WAIT;
            PROF_RC_UPD_COUNTERS_TRIAL_CLR();
        }
        else
        {
            // update the number of attempts and success for each rate of the retry chain
            for (uint32_t i = 0; (i < RATE_CONTROL_STEPS) && (attempts > 0); i++)
            {
                uint16_t rate_idx = rc_ss->retry_step_idx[i];
                struct rc_rate_stats *rc_rs = &rc_ss->rate_stats[rate_idx];
                if (failures >= RC_MAX_NUM_RETRY)
                {
                    rc_rs->attempts += RC_MAX_NUM_RETRY;
                    attempts -= RC_MAX_NUM_RETRY;
                    failures -= RC_MAX_NUM_RETRY;
                }
                else
                {
                    rc_rs->attempts += attempts;
                    rc_rs->success += (attempts - failures);
                    attempts = 0;
                    failures = 0;
                }
                ASSERT_ERR(rc_rs->attempts >= rc_rs->success);
            }
        }
    }
    PROF_RC_UPD_COUNTERS_CLR();
}

/*
 * PUBLIC FUNCTIONS DEFINITION
 ****************************************************************************************
 */
void rc_cfm_singleton(struct txdesc *txdesc)
{
    // Check the status
    struct tx_cfm_tag *cfm = txl_cfm_tag_get(txdesc);
    uint32_t status = cfm->status;
    uint32_t retry_cnt = ((status & NUM_MPDU_RETRIES_MSK) >> NUM_MPDU_RETRIES_OFT);
    uint32_t trials;
    uint32_t failures;

    // Compute the statistics
    trials = retry_cnt + 1;
    failures = retry_cnt + ((status & RETRY_LIMIT_REACHED_BIT) != 0);

    // Update statistics
    rc_update_counters(txdesc, trials, failures, 0);
}

void rc_cfm_ampdu(struct txdesc *txdesc, uint32_t txed, uint32_t txok)
{
    // Update statistics
    rc_update_counters(txdesc, txed, (txed - txok), 1);
}

void rc_check(struct txdesc *txdesc)
{
    struct hostdesc *host = &txdesc->host;
    uint8_t sta_idx = host->staid;
    struct sta_info_tag *sta;
    struct sta_pol_tbl_cntl *rc;
    struct rc_sta_stats *rc_ss;
    bool update = false;
    #if NX_AMPDU_TX
    bool tx_ampdu;
    #endif

    if (sta_idx >= NX_REMOTE_STA_MAX)
    {
        txdesc->umac.rc_control = 0;
        return;
    }

    sta = &sta_info_tab[sta_idx];
    rc = &sta->pol_tbl;
    rc_ss = rc->sta_stats;
    ASSERT_ERR(rc_ss != NULL);

    #if NX_AMPDU_TX
    tx_ampdu = (txdesc->umac.flags & AMPDU_BIT) ? true : false;
    // update ampdu tx
    if (((rc_ss->info & RC_AGG_ALLOWED_MASK) == 0) || !tx_ampdu)
    {
        rc_ss->info &= ~RC_AGG_TX_MASK;
    }
    else
    {
        rc_ss->info |= RC_AGG_TX_MASK;
    }
    #endif

    // Check if it is time to run the RC algorithm
    if (hal_machw_time_past(rc_ss->last_rc_time + RC_PERIOD_TOUT))
    {
        PROF_RC_STATS_CALC_SET();
        // Calculate statistics and update retry chain
        update |= rc_update_stats(rc_ss, 0);
        // Reset trial status
        rc_ss->trial_status = RC_TRIAL_STATUS_WAIT;
        // Reset SW retry step
        rc_ss->sw_retry_step = 0;
        // Increase retry chain version
        rc_ss->retry_chain_ver = (rc_ss->retry_chain_ver + 1) % 4;
        // Update RC timer
        rc_ss->last_rc_time = hal_machw_time();

        PROF_RC_STATS_CALC_CLR();
    }

    txdesc->umac.rc_control = rc_ss->retry_chain_ver << RC_RETRY_CHAIN_VER_OFT;
    #if NX_AMPDU_TX
    txdesc->umac.rc_control |= rc_ss->sw_retry_step << RC_SW_RETRY_STEP_OFT;
    #endif

    if (update)
    {
        #if NX_AMSDU_TX
        // Set maximum AMSDU len
        rc_set_max_amsdu_len(rc_ss);
        #endif
        #if NX_AMPDU_TX
        rc_set_aggregation_flag(rc_ss, &tx_ampdu);
        if (!tx_ampdu)
        {
            txdesc->umac.flags &= ~AMPDU_BIT;
        }
        #endif
        // Keep in mind we have to update the rate
        rc->upd_field |= CO_BIT(STA_MGMT_POL_UPD_RATE);
    }

    // check if the retry chain has to be updated due to a trial tx
    if (rc_check_trial_tx(rc_ss))
    {
        txdesc->host.flags |= TXU_CNTRL_RC_TRIAL;
        #if NX_AMPDU_TX
        txdesc->umac.rc_control &= ~RC_SW_RETRY_STEP_MSK;
        if (tx_ampdu)
            txdesc->umac.flags |= WHICHDESC_AMPDU_FIRST;
        #endif
        // Keep in mind we have to update the trial rate
        rc->upd_field |= CO_BIT(STA_MGMT_POL_TRIAL_RATE);
    }
}

void rc_init(struct sta_info_tag *sta)
{
    struct sta_pol_tbl_cntl *rc = &sta->pol_tbl;
    struct tx_policy_tbl *pol = &rc->buf_ctrl->policy_tbl;
    uint8_t hw_key_idx = MM_STA_TO_KEY(sta->staid);

    // Set pointer to station RC statistics
    ASSERT_ERR(sta->staid < NX_REMOTE_STA_MAX);
    rc->sta_stats = &sta_stats[sta->staid];
    ASSERT_ERR(rc->sta_stats != NULL);

    struct rc_sta_stats *rc_ss = rc->sta_stats;
    uint32_t phy_cntrl_info1 = phy_get_ntx() << NX_TX_PT_OFT;
    uint32_t phy_cntrl_info2 = TX_NTX_2_ANTENNA_SET(phy_get_ntx());
    int i;

    memset(rc_ss, 0, sizeof(*rc_ss));

    // Check if the peer is a HT, VHT or HE device
    // If it is supporting all those modes, we will choose only one of them for the
    // rate control, i.e. the one bringing the highest performance:
    //   1. HE
    //   2. VHT
    //   3. HT
    if (STA_CAPA(sta, HT))
    {
        struct mac_htcapability *htcap = &sta->info.ht_cap;
        uint16_t max_mpdu_head_tail = MAC_LONG_QOS_HTC_MAC_HDR_LEN + MAC_FCS_LEN;
        #if RW_WAPI_EN
        max_mpdu_head_tail += WPI_IV_LEN + WPI_MIC_LEN;
        #else
        max_mpdu_head_tail += IV_LEN + EIV_LEN + MIC_LEN + ICV_LEN;
        #endif

        // set the bitmap for legacy rates
        rc_ss->rate_map_l = me_legacy_rate_bitfield_build(&sta->info.rate_set, false);
        #if NX_HE
        if (STA_CAPA(sta, HE))
        {
            struct mac_hecapability *hecap = &sta->info.he_cap;
            struct mac_hecapability *hecaploc = &me_env.he_cap;
            uint8_t color = 0;
            struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];
            struct me_bss_info *bss = &vif->bss_info;
            // HE configurations
            rc_ss->format_mod = FORMATMOD_HE_SU;
            rc_ss->no_ss = co_min(me_11ac_nss_max(hecap->mcs_supp.rx_mcs_80),
                                  me_11ac_nss_max(hecaploc->mcs_supp.tx_mcs_80));
            ASSERT_ERR(rc_ss->no_ss < 8);
            // Set the bitmap for HE rates
            rc_ss->rate_map.he = me_rate_bitfield_vht_build(hecap->mcs_supp.rx_mcs_80,
                                                            hecaploc->mcs_supp.tx_mcs_80);
            rc_ss->mcs_max = co_min(me_11ax_mcs_max(hecap->mcs_supp.rx_mcs_80),
                                    me_11ax_mcs_max(hecaploc->mcs_supp.tx_mcs_80));
            ASSERT_ERR(rc_ss->mcs_max < 12);
            // remove legacy OFDM rates from the map
            rc_ss->rate_map_l = rc_ss->rate_map_l & ~0xFF0;
            rc_ss->r_idx_min = me_legacy_ridx_min(rc_ss->rate_map_l);
            ASSERT_ERR((rc_ss->r_idx_min <= HW_RATE_11MBPS) || (rc_ss->r_idx_min == MAC_RATESET_LEN));
            rc_ss->r_idx_max = me_legacy_ridx_max(rc_ss->rate_map_l);
            ASSERT_ERR((rc_ss->r_idx_max <= HW_RATE_11MBPS) || (rc_ss->r_idx_max == MAC_RATESET_LEN));
            rc_ss->type = (rc->ppdu_tx_cfg & PRE_TYPE_TX_RCX_MASK) >> PRE_TYPE_TX_RCX_OFT;

            // Check whether LDPC coding shall be used for transmission to the STA.
            // To do that we ensure first that our PHY is able to use LDPC coding for
            // transmission, and that the peer is able to receive it.
            // We also check our LDPC capability in RX. This check is not mandatory, but
            // it allows disabling completely the LDPC support, in case it is required for
            // testing. Indeed for such purpose our LDPC RX capability advertised to the
            // peers can be manually disabled using a parameter of the WiFi driver. Doing
            // that will therefore also disable the LDPC in TX.
            if (phy_ldpc_tx_supported() &&
                HE_PHY_CAPA_BIT_IS_SET(hecaploc, LDPC_CODING_IN_PAYLOAD) &&
                HE_PHY_CAPA_BIT_IS_SET(hecap, LDPC_CODING_IN_PAYLOAD))
            {
                phy_cntrl_info1 |= FEC_CODING_PT_BIT;
            }

            // Check if we need to get the MAX A-MSDU from the VHT or HT capabilities
            if (STA_CAPA(sta, VHT))
            {
                struct mac_vhtcapability *vhtcap = &sta->info.vht_cap;
                switch (vhtcap->vht_capa_info & MAC_VHTCAPA_MAX_MPDU_LENGTH_MSK)
                {
                    case MAC_VHTCAPA_MAX_MPDU_LENGTH_3895:
                        rc_ss->max_amsdu_len = 3895 - max_mpdu_head_tail;
                        break;
                    case MAC_VHTCAPA_MAX_MPDU_LENGTH_7991:
                        rc_ss->max_amsdu_len = 7991 - max_mpdu_head_tail;
                        break;
                    case MAC_VHTCAPA_MAX_MPDU_LENGTH_11454:
                        rc_ss->max_amsdu_len = 11454 - max_mpdu_head_tail;
                        break;
                    default:
                        rc_ss->max_amsdu_len = 3895 - max_mpdu_head_tail;
                        break;
                }
            }
            else
            {
                if (htcap->ht_capa_info & MAC_HTCAPA_AMSDU)
                {
                    rc_ss->max_amsdu_len = 7935;
                }
                else
                {
                    rc_ss->max_amsdu_len = 3839;
                }
            }

            if (!(bss->he_oper & MAC_HE_OPER_BSS_COLOR_DISABLED_BIT))
                color = (bss->he_oper & MAC_HE_OPER_BSS_COLOR_MASK) >>
                                                           MAC_HE_OPER_BSS_COLOR_OFT;
            phy_cntrl_info2 |= BMCHANGE_BIT | UPLINK_FLAG_BIT | (color << BSS_COLOR_OFT) |
                               (4 << PKT_EXTENSION_OFT);
        }
        else
        #endif
        #if NX_VHT
        if (STA_CAPA(sta, VHT))
        {
            struct mac_vhtcapability *vhtcap = &sta->info.vht_cap;
            struct mac_vhtcapability *vhtcaploc = &me_env.vht_cap;
            // VHT configurations
            rc_ss->format_mod = FORMATMOD_VHT;
            rc_ss->no_ss = co_min(me_11ac_nss_max(vhtcap->rx_mcs_map),
                                  me_11ac_nss_max(vhtcaploc->tx_mcs_map));
            ASSERT_ERR(rc_ss->no_ss < 8);
            // Set the bitmap for VHT rates
            rc_ss->rate_map.vht = me_rate_bitfield_vht_build(vhtcap->rx_mcs_map,
                                                             vhtcaploc->tx_mcs_map);
            rc_ss->mcs_max = co_min(me_11ac_mcs_max(vhtcap->rx_mcs_map),
                                    me_11ac_mcs_max(vhtcaploc->tx_mcs_map));
            ASSERT_ERR(rc_ss->mcs_max < 10);
            // legacy configurations
            rc_ss->type = 0;
            // remove legacy OFDM rates from the map
            rc_ss->rate_map_l = 0;
            rc_ss->r_idx_min = me_legacy_ridx_min(rc_ss->rate_map_l);
            rc_ss->r_idx_max = me_legacy_ridx_max(rc_ss->rate_map_l);

            // Check whether LDPC coding shall be used for transmission to the STA.
            // To do that we ensure first that our PHY is able to use LDPC coding for
            // transmission, and that the peer is able to receive it.
            // We also check our LDPC capability in RX. This check is not mandatory, but
            // it allows disabling completely the LDPC support, in case it is required for
            // testing. Indeed for such purpose our LDPC RX capability advertised to the
            // peers can be manually disabled using a parameter of the WiFi driver. Doing
            // that will therefore also disable the LDPC in TX.
            if (phy_ldpc_tx_supported() &&
                (vhtcaploc->vht_capa_info & MAC_VHTCAPA_RXLDPC) &&
                (vhtcap->vht_capa_info & MAC_VHTCAPA_RXLDPC))
            {
                phy_cntrl_info1 |= FEC_CODING_PT_BIT;
            }

            switch (vhtcap->vht_capa_info & MAC_VHTCAPA_MAX_MPDU_LENGTH_MSK)
            {
                case MAC_VHTCAPA_MAX_MPDU_LENGTH_3895:
                    rc_ss->max_amsdu_len = 3895 - max_mpdu_head_tail;
                    break;
                case MAC_VHTCAPA_MAX_MPDU_LENGTH_7991:
                    rc_ss->max_amsdu_len = 7991 - max_mpdu_head_tail;
                    break;
                case MAC_VHTCAPA_MAX_MPDU_LENGTH_11454:
                    rc_ss->max_amsdu_len = 11454 - max_mpdu_head_tail;
                    break;
                default:
                    rc_ss->max_amsdu_len = 3895 - max_mpdu_head_tail;
                    break;
            }
        }
        else
        #endif
        {
            struct mac_htcapability *htcaploc = &me_env.ht_cap;
            // HT configurations
            rc_ss->format_mod = FORMATMOD_HT_MF;
            rc_ss->no_ss = co_min(me_11n_nss_max(htcap->mcs_rate),
                                  me_11n_nss_max(htcaploc->mcs_rate));
            ASSERT_ERR(rc_ss->no_ss <= 3);
            // Set the bitmap for HT rates
            memcpy(rc_ss->rate_map.ht, htcap->mcs_rate, sizeof(rc_ss->rate_map.ht));
            // MCS0-7 must be supported by all HT STA (but not always set in HT capab)
            rc_ss->rate_map.ht[0] = 0xFF;
            rc_ss->mcs_max = 7;

            // legacy configurations
            // remove legacy OFDM rates from the map
            rc_ss->rate_map_l = rc_ss->rate_map_l & ~0xFF0;
            rc_ss->r_idx_min = me_legacy_ridx_min(rc_ss->rate_map_l);
            ASSERT_ERR((rc_ss->r_idx_min <= HW_RATE_11MBPS) || (rc_ss->r_idx_min == MAC_RATESET_LEN));
            rc_ss->r_idx_max = me_legacy_ridx_max(rc_ss->rate_map_l);
            ASSERT_ERR((rc_ss->r_idx_max <= HW_RATE_11MBPS) || (rc_ss->r_idx_max == MAC_RATESET_LEN));
            rc_ss->type = (rc->ppdu_tx_cfg & PRE_TYPE_TX_RCX_MASK) >> PRE_TYPE_TX_RCX_OFT;

            // Check whether LDPC coding shall be used for transmission to the STA.
            // To do that we ensure first that our PHY is able to use LDPC coding for
            // transmission, and that the peer is able to receive it.
            // We also check our LDPC capability in RX. This check is not mandatory, but
            // it allows disabling completely the LDPC support, in case it is required for
            // testing. Indeed for such purpose our LDPC RX capability advertised to the
            // peers can be manually disabled using a parameter of the WiFi driver. Doing
            // that will therefore also disable the LDPC in TX.
            if (phy_ldpc_tx_supported() &&
                (htcaploc->ht_capa_info & MAC_HTCAPA_LDPC) &&
                (htcap->ht_capa_info & MAC_HTCAPA_LDPC))
            {
                phy_cntrl_info1 |= FEC_CODING_PT_BIT;
            }
            if (htcap->ht_capa_info & MAC_HTCAPA_AMSDU)
            {
                #if NX_AMPDU_TX
                rc_ss->max_amsdu_len = 4095 - max_mpdu_head_tail;
                #else
                rc_ss->max_amsdu_len = 7935;
                #endif
            }
            else
            {
                rc_ss->max_amsdu_len = 3839;
            }
        }
        // bandwidth
        rc_ss->bw_max = sta->info.bw_cur;
        ASSERT_ERR(rc_ss->bw_max <= BW_160MHZ);
        // guard interval
        switch(sta->info.bw_cur)
        {
            case BW_20MHZ:
                if (htcap->ht_capa_info & MAC_HTCAPA_SHORTGI_20)
                    rc_ss->short_gi = 1;
                break;
            case BW_40MHZ:
                if (htcap->ht_capa_info & MAC_HTCAPA_SHORTGI_40)
                    rc_ss->short_gi = 1;
                break;
            #if NX_VHT
            case BW_80MHZ:
                if (sta->info.vht_cap.vht_capa_info & MAC_VHTCAPA_SHORT_GI_80)
                    rc_ss->short_gi = 1;
                break;
            case BW_160MHZ:
                if (sta->info.vht_cap.vht_capa_info & MAC_VHTCAPA_SHORT_GI_160)
                    rc_ss->short_gi = 1;
                break;
            #endif
            default:
                break;
        }
    }
    else
    {
        // set the bitmap for legacy rates
        rc_ss->rate_map_l = me_legacy_rate_bitfield_build(&sta->info.rate_set, false);
        rc_ss->r_idx_min = me_legacy_ridx_min(rc_ss->rate_map_l);
        ASSERT_ERR(rc_ss->r_idx_min < MAC_RATESET_LEN);
        rc_ss->r_idx_max = me_legacy_ridx_max(rc_ss->rate_map_l);
        ASSERT_ERR(rc_ss->r_idx_max < MAC_RATESET_LEN);
        rc_ss->mcs_max = 0xFF;
        rc_ss->bw_max = sta->info.bw_cur;
        ASSERT_ERR(rc_ss->bw_max == BW_20MHZ);
        // preamble type: 0=short and long, 1=only long
        rc_ss->type = (rc->ppdu_tx_cfg & PRE_TYPE_TX_RCX_MASK) >> PRE_TYPE_TX_RCX_OFT;
    }
    // Set number of samples for statistics
    rc_ss->no_samples = rc_get_num_samples(rc_ss);
    ASSERT_ERR(rc_ss->no_samples >= 1);
    ASSERT_ERR(rc_ss->no_samples <= RC_MAX_N_SAMPLE);

    dbg(D_CRT "%s: station_id=%i format_mod=%i pre_type=%i short_gi=%i max_bw=%i\n",
            __func__, sta->staid, rc_ss->format_mod, rc_ss->type, rc_ss->short_gi, rc_ss->bw_max);
    dbg(D_CRT "%s: nss_max=%i mcs_max=%i r_idx_min=%i r_idx_max=%i no_samples=%i\n",
            __func__, rc_ss->no_ss, rc_ss->mcs_max, rc_ss->r_idx_min, rc_ss->r_idx_max, rc_ss->no_samples);

    // Init RC algorithm
    rc_init_rates(sta->staid);

    // Set current rate configuration
    for (i = 0; i < RATE_CONTROL_STEPS; i++)
    {
        uint8_t idx = rc_ss->retry_step_idx[i];
        pol->ratecntrlinfo[i] =
            (RC_MAX_NUM_RETRY << N_RETRY_RCX_OFT) | rc_ss->rate_stats[idx].rate_config;
    }

    #if NX_AMSDU_TX
    // Set maximum AMSDU len
    rc_set_max_amsdu_len(rc_ss);
    #endif
    #if NX_AMPDU_TX
    rc_set_aggregation_flag(rc_ss, NULL);
    #endif

    // Init last_rc_time
    rc_ss->last_rc_time = hal_machw_time();

    pol->upatterntx    = POLICY_TABLE_PATTERN;
    pol->maccntrlinfo1 = hw_key_idx << KEYSRAM_INDEX_RA_OFT;
    pol->maccntrlinfo2 = 0xFFFF0704;
    pol->phycntrlinfo1 = phy_cntrl_info1;
    pol->phycntrlinfo2 = phy_cntrl_info2;
    #if RW_BFMER_EN
    if (bfr_is_enabled())
    {
        /*
         * If frame is not beamformed, the SMM Index field indicates the spatial mapping in the
         * modem.
         *      0 = Stream 0 on antenna 0 and stream 1 on antenna 1
         *      2 = Stream mixed on antenna 0 and antenna 1
         */
        pol->phycntrlinfo2 |= 2 << SMM_INDEX_PT_OFT;
    }
    #endif
    rc->buf_ctrl->mac_control_info = EXPECTED_ACK_NORMAL_ACK | LOW_RATE_RETRY;
    rc->buf_ctrl->phy_control_info = sta->paid_gid;

    // Set update rate flag
    rc->upd_field |= CO_BIT(STA_MGMT_POL_UPD_RATE) | CO_BIT(STA_MGMT_POL_UPD_TX_POWER);
}


uint32_t rc_get_duration(struct rc_rate_stats *rate_stats)
{
    uint32_t duration = 0;
    uint32_t idx;
    uint16_t rate_config = rate_stats->rate_config;
    uint8_t format_mod = rc_get_format_mod(rate_config);
    uint8_t mcs = rc_get_mcs_index(rate_config);

    switch (format_mod)
    {
        case FORMATMOD_NON_HT:
        case FORMATMOD_NON_HT_DUP_OFDM:
        {
            if (mcs < HW_RATE_6MBPS)
            {
                uint8_t pre_type = rc_get_pre_type(rate_config);
                idx = (uint32_t)(mcs << 1) | pre_type;
                duration = rc_duration_cck[idx];
            }
            else
            {
                idx = (mcs - HW_RATE_6MBPS);
                duration = rc_duration_non_ht[idx];
            }
        } break;

        case FORMATMOD_HT_MF:
        case FORMATMOD_HT_GF:
        #if NX_VHT
        case FORMATMOD_VHT:
        #endif
        {
            uint8_t bw = rc_get_bw(rate_config);
            uint8_t gi = rc_get_sgi(rate_config);
            idx = (mcs << 3) | (bw << 1) | gi;
            duration = rc_duration_ht_ampdu[idx] / (rc_get_nss(rate_config) + 1);
        } break;

        #if NX_HE
        case FORMATMOD_HE_MU:
        {
            uint8_t ru_size = rate_stats->ru_and_length & 0x07;
            if (ru_size < 3)
            {
                duration = rc_ru_duration[ru_size][mcs * 3 + rc_get_he_gi(rate_config)] /
                                                           (rc_get_nss(rate_config) + 1);
            }
            else
            {
                uint8_t bw = ru_size - 3;
                uint8_t gi = rc_get_he_gi(rate_config);
                int shift = bw & 0x01;
                idx = (mcs * 6) + ((bw >> 1) * 3) + gi;
                duration = (rc_duration_he_ampdu[idx] >> shift)  / (rc_get_nss(rate_config) + 1);
            }
        } break;

        case FORMATMOD_HE_SU:
        {
            uint8_t bw = rc_get_bw(rate_config);
            uint8_t gi = rc_get_he_gi(rate_config);
            int shift = bw & 0x01;
            idx = (mcs * 6) + ((bw >> 1) * 3) + gi;
            duration = (rc_duration_he_ampdu[idx] >> shift)  / (rc_get_nss(rate_config) + 1);
        } break;
        #endif

        default:
            break;
    }

    return duration;
}

void rc_update_sample_table(uint8_t sta_idx)
{
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    struct sta_pol_tbl_cntl *rc = &sta->pol_tbl;
    struct rc_sta_stats *rc_ss = rc->sta_stats;
    uint32_t cur_tp[RC_MAX_N_SAMPLE];
    ASSERT_ERR(rc_ss != NULL);

    // Update number of samples
    rc_ss->no_samples = rc_get_num_samples(rc_ss);
    ASSERT_ERR(rc_ss->no_samples >= 1);
    ASSERT_ERR(rc_ss->no_samples <= RC_MAX_N_SAMPLE);

    // Select new random rates for the stats table
    rc_set_rate_configs(rc_ss);

    // Update the retry chain with: max TP, 2nd max TP, max probability, lowest rate
    rc_update_retry_chain(rc_ss, cur_tp);

    #if NX_AMSDU_TX
    // Set maximum AMSDU len
    rc_set_max_amsdu_len(rc_ss);
    #endif
    #if NX_AMPDU_TX
    rc_set_aggregation_flag(rc_ss, NULL);
    #endif
    // Keep in mind we have to update the rate
    rc->upd_field |= CO_BIT(STA_MGMT_POL_UPD_RATE);
}

void rc_update_bw_nss_max(uint8_t sta_idx, uint8_t bw_max, uint8_t nss_max)
{
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    struct sta_pol_tbl_cntl *rc = &sta->pol_tbl;
    struct rc_sta_stats *rc_ss = rc->sta_stats;
    ASSERT_ERR(rc_ss != NULL);

    if ((bw_max == rc_ss->bw_max) && (nss_max == rc_ss->no_ss))
        return;

    rc_ss->bw_max = bw_max;
    ASSERT_ERR(rc_ss->bw_max <= BW_160MHZ);
    rc_ss->no_ss = nss_max;
    ASSERT_ERR(rc_ss->no_ss < 8);

    // If fixed rate is set, do not update the sample table
    if (rc_ss->info & RC_FIX_RATE_EN_MASK)
    {
        // keep in mind that we need to update the sample table after the fixed rate time
        rc_ss->info |= RC_FIX_RATE_UPD_SS_REQ_MASK;
        return;
    }

    // Update sample table according with new bw and nss
    rc_update_sample_table(sta_idx);

}

void rc_update_preamble_type(uint8_t sta_idx, uint8_t preamble_type)
{
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    struct sta_pol_tbl_cntl *rc = &sta->pol_tbl;
    struct rc_sta_stats *rc_ss = rc->sta_stats;
    uint16_t i = 0;
    uint32_t cur_tp[RC_MAX_N_SAMPLE];
    ASSERT_ERR(rc_ss != NULL);

    if (rc_ss->type == preamble_type)
        return;

    // Save preamble type
    rc_ss->type = preamble_type;

    // Do changes to the rate only if we move to long preamble
    if (!preamble_type)
        return;

    // If fixed rate is set, do not update the sample table
    if (rc_ss->info & RC_FIX_RATE_EN_MASK)
    {
        // keep in mind that we need to update the sample table after the fixed rate time
        rc_ss->info |= RC_FIX_RATE_UPD_SS_REQ_MASK;
        return;
    }

    for (i = 0; i < rc_ss->no_samples; i++)
    {
        struct rc_rate_stats *rc_rs = &rc_ss->rate_stats[i];
        uint16_t rate_cfg = rc_rs->rate_config;

        // Check if the rate is a CCK one
        if (is_cck_group(rc_rs->rate_config))
        {
            // Set long preamble
            rate_cfg |= PRE_TYPE_TX_RCX_MASK;

            while (rc_check_rate_duplicated(rc_ss, rate_cfg))
            {
                rate_cfg = rc_new_random_rate(rc_ss);
            }
        }

        rc_rs->rate_allowed = 1;
        rc_rs->rate_config = rate_cfg;
        rc_rs->probability = 0;
        rc_rs->old_prob_available = 0;
        cur_tp[i] = 0;
    }

    // Sort sample table by TP
    rc_sort_samples_tp(rc_ss, cur_tp);
    // Update the retry chain with: max TP, 2nd max TP, max probability, lowest rate
    rc_update_retry_chain(rc_ss, cur_tp);

    #if NX_AMSDU_TX
    // Set maximum AMSDU len
    rc_set_max_amsdu_len(rc_ss);
    #endif
    // Keep in mind we have to update the rate
    rc->upd_field |= CO_BIT(STA_MGMT_POL_UPD_RATE);
}

void rc_init_bcmc_rate(struct sta_info_tag *sta, uint8_t basic_rate_idx)
{
    struct sta_pol_tbl_cntl *rc = &sta->pol_tbl;
    struct tx_policy_tbl *pol = &rc->buf_ctrl->policy_tbl;
    uint8_t i;
    uint16_t rate_config;
    uint8_t type = (rc->ppdu_tx_cfg & PRE_TYPE_TX_RCX_MASK) >> PRE_TYPE_TX_RCX_OFT;
    // Set rate configuration
    rate_config = ((basic_rate_idx < HW_RATE_6MBPS ? type : 0) << PRE_TYPE_TX_RCX_OFT) |
                  (basic_rate_idx << MCS_INDEX_TX_RCX_OFT);
    // Set the basic rate idx for all the steps of the retry chain
    for (i = 0; i < RATE_CONTROL_STEPS; i++)
    {
        pol->ratecntrlinfo[i] = (1 << N_RETRY_RCX_OFT) | rate_config;
    }
}

bool rc_check_fixed_rate_config(struct rc_sta_stats *rc_ss, uint16_t fixed_rate_config)
{
    uint8_t fixed_rate_ft = rc_get_format_mod(fixed_rate_config);
    bool update = 0;

    do
    {
        // Check format and modulation
        if (fixed_rate_ft > rc_ss->format_mod)
        {
            break;
        }
        if ((rc_ss->format_mod == FORMATMOD_HE_SU) && (fixed_rate_ft != FORMATMOD_HE_SU)
         && (rc_ss->r_idx_min > HW_RATE_11MBPS))
        {
            break;
        }
        if ((rc_ss->format_mod == FORMATMOD_VHT) && (fixed_rate_ft != FORMATMOD_VHT))
        {
            break;
        }
        if (((rc_ss->format_mod == FORMATMOD_HT_MF) || (rc_ss->format_mod == FORMATMOD_HT_GF)) &&
            ((fixed_rate_ft < FORMATMOD_HT_MF) && (rc_ss->r_idx_min > HW_RATE_11MBPS)))
        {
            break;
        }
        if (fixed_rate_ft < FORMATMOD_HT_MF)
        {
            // Check preamble type
            if ((rc_get_pre_type(fixed_rate_config) == 0) && (rc_ss->type == 1))
            {
                break;
            }
        }
        else
        {
            // Check guard interval
            if (fixed_rate_ft < FORMATMOD_HE_SU)
            {
                if ((rc_get_sgi(fixed_rate_config) == 1) && (rc_ss->short_gi == 0))
                {
                    break;
                }
            }
            // Check bandwidth
            if (rc_get_bw(fixed_rate_config) > rc_ss->bw_max)
            {
                break;
            }
            // Check number of spatial streams
            if (rc_get_nss(fixed_rate_config) > rc_ss->no_ss)
            {
                break;
            }
        }
        // Check MCS / rate index
        if (rc_check_valid_rate(rc_ss, fixed_rate_config) == 0)
        {
            break;
        }

        update = 1;

    } while (0);

    return update;
}

uint32_t rc_calc_tp(struct rc_sta_stats *rc_ss, uint8_t sample_idx)
{
    uint64_t cur_tp;
    uint32_t nsecs = 0;

    struct rc_rate_stats *rc_rs = &rc_ss->rate_stats[sample_idx];
    uint16_t rate_config = rc_rs->rate_config;
    uint32_t prob = rc_rs->probability;
    uint16_t avg_ampdu = RC_TRUNC(rc_ss->avg_ampdu_len);

    if (prob < RC_FRAC(1, 10))
    {
        cur_tp = 0;
    }
    else
    {
        if (!is_cck_group(rate_config))
        {
            // By default use fixed overhead time of 60 usec (ack) + 48 usec + backoff
            uint32_t overhead = 218;
            if (rc_get_format_mod(rate_config) == FORMATMOD_HE_MU)
                // In HE TB we add also the overhead of the trigger frame reception
                overhead = 330;

            nsecs = 1000 * overhead / avg_ampdu;
        }
        nsecs += rc_get_duration(rc_rs);

        // Compute the expected user data throughput (i.e. the one that would be reported
        // by iperf when sending a UDP stream) based on the rate on transmission
        // and its probability of success
        cur_tp = 910000 * ((prob * 1000) / nsecs);
        cur_tp = RC_TRUNC(cur_tp);
    }

    return cur_tp;
}

void rc_trial_check_bfm_stbc_htc(struct txl_buffer_control *buf_ctrl, uint32_t trial_rate,
                                 bool can_use_bfm, bool use_stbc, uint8_t stbc_nss,
                                 uint8_t nc, uint8_t ntx, bool can_use_htc)
{
    uint8_t nss = 0;
    uint32_t trial_config = trial_rate &
                          (MCS_INDEX_TX_RCX_MASK | BW_TX_RCX_MASK |
                           SHORT_GI_TX_RCX_MASK | PRE_TYPE_TX_RCX_MASK |
                           FORMAT_MOD_TX_RCX_MASK);
    uint32_t format = (trial_config & FORMAT_MOD_TX_RCX_MASK) >>
                        FORMAT_MOD_TX_RCX_OFT;

    // Get the number of SS of this frame
    if (format >= FORMATMOD_HT_MF)
    {
        uint32_t mcs_idx = (trial_config & MCS_INDEX_TX_RCX_MASK) >>
                            MCS_INDEX_TX_RCX_OFT;
        #if NX_VHT
        if (format == FORMATMOD_VHT)
        {
            // Extract NSS
            nss = (mcs_idx & VHT_NSS_MASK) >> VHT_NSS_OFT;
        }
        else
        #endif
        {
            // Extract NSS
            nss = (mcs_idx & HT_NSS_MASK) >> HT_NSS_OFT;
        }

        // Check if we can use STBC for this rate
        if (use_stbc && (nss == stbc_nss))
        {
            buf_ctrl->tx_flags |= TX_SWDESC_UMAC_TRIAL_STBC_BIT;
        }
        else
        {
            buf_ctrl->tx_flags &= ~TX_SWDESC_UMAC_TRIAL_STBC_BIT;
        }

        // Check if beamforming can be use with this policy table
        #if RW_BFMER_EN
        if (can_use_bfm)
        {
            if ((format != FORMATMOD_VHT) || (nss > nc) || (nss == ntx))
            {
                can_use_bfm = false;
            }
        }
        #endif //(RW_BFMER_EN)
    }
    else
    {
        #if RW_BFMER_EN
        can_use_bfm = false;
        #endif
        #if NX_HE
        can_use_htc = false;
        #endif
    }

    #if RW_BFMER_EN
    // Allow/Disallow Beamforming
    if (can_use_bfm)
    {
        buf_ctrl->tx_flags |= TX_SWDESC_UMAC_TRIAL_BEAMFORM_BIT;
    }
    else
    {
        buf_ctrl->tx_flags &= ~TX_SWDESC_UMAC_TRIAL_BEAMFORM_BIT;
    }
    #endif

    #if NX_HE
    // Allow/Disallow +HTC
    if (can_use_htc)
        buf_ctrl->tx_flags |= TX_SWDESC_UMAC_TRIAL_HTC_BIT;
    else
        buf_ctrl->tx_flags &= ~TX_SWDESC_UMAC_TRIAL_HTC_BIT;
    #endif
}

/// @}

