/**
 ****************************************************************************************
 *
 * @file rxu_cntrl.c
 *
 * @brief The UMAC's Rx module implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/** @addtogroup UMACRX
 * @{
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */

// standard includes
#include "co_int.h"
// for bool
#include "co_bool.h"
// for NULL
#include <string.h>
// for __INLINE
#include "compiler.h"

// main file include
#include "rxu_cntrl.h"

// Communication with upper layers
#include "macif.h"
// for ring element types
#include "co_ring.h"
// for event bits
#include "ke_event.h"
// for api task
#include "ke_task.h"
// for api task
#include "me_task.h"
// for MIC header length
#include "mac_frame.h"
// LMAC RX Controller
#include "rxl_cntrl.h"
// STA Information
#include "sta_mgmt.h"
// VIF Information
#include "vif_mgmt.h"
// STA Info table
#include "mm.h"
// Function prototypes and HW desc macros
#include "rxl_hwdesc.h"
#include "ps.h"

#include "co_utils.h"
#include "co_endian.h"

#include "hal_desc.h"

#include "llc.h"
#include "bam.h"
#include "mac_ie.h"
#include "me_utils.h"
#include "rxu_task.h"
#include "scanu_task.h"
#include "scanu.h"
#include "sm.h"
#if NX_MFP
#include "mfp.h"
#endif
#if (RW_MESH_EN)
#include "mesh_hwmp.h"
#endif //(RW_MESH_EN)

/*
 * MACROS
 ****************************************************************************************
 */

/// Indicate if the received packet is a Data packet by reading the Frame Control Field
#define RXL_CNTRL_IS_DATA(frame_cntrl)                                                   \
           ((frame_cntrl & MAC_FCTRL_TYPE_MASK) == MAC_FCTRL_DATA_T)
/// Indicate if the received packet is a QOS Data packet by reading the Frame Control Field
#define RXL_CNTRL_IS_QOS_DATA(frame_cntrl)                                               \
           ((frame_cntrl & MAC_FCTRL_TYPESUBTYPE_MASK) == MAC_FCTRL_QOS_DATA)
/// Indicate if the received packet is a Management packet by reading the Frame Control Field
#define RXL_CNTRL_IS_MGT(frame_cntrl)                                                    \
           ((frame_cntrl & MAC_FCTRL_TYPE_MASK) == MAC_FCTRL_MGT_T)
/// Indicate if the received packet is a Control packet by reading the Frame Control Field
#define RXL_CNTRL_IS_CTRL(frame_cntrl)                                                   \
           ((frame_cntrl & MAC_FCTRL_TYPE_MASK) == MAC_FCTRL_CTRL_T)

/// Indicate if the received frame was encrypted (Protected Bit set to 1)
#define RXL_CNTRL_IS_PROTECTED(frame_cntrl)                                              \
           ((frame_cntrl & MAC_FCTRL_PROTECTEDFRAME) == MAC_FCTRL_PROTECTEDFRAME)

/*
 * GLOBAL VARIABLE DECLARATIONS
 ****************************************************************************************
 */

/// The RX module's environment
struct rxu_cntrl_env_tag rxu_cntrl_env;

/// Attribute indicating whether the RX status descriptor pool shall be put in shared RAM
/// or not
#if NX_FULLY_HOSTED
#define __RX_STAT_DESC_LOC
#else
#define __RX_STAT_DESC_LOC __SHAREDRAM
#endif
/// Pool of RX status structures
struct rxu_stat_desc rxu_stat_desc_pool[RX_STAT_DESC_CNT] __RX_STAT_DESC_LOC;

#if (NX_REORD)
/// Pool of reordering structures
struct rxu_cntrl_reord rxu_cntrl_reord_pool[RX_CNTRL_REORD_POOL_SIZE];
#endif //(NX_REORD)

/// Pool of reordering structures
struct rxu_cntrl_defrag rxu_cntrl_defrag_pool[RX_CNTRL_DEFRAG_POOL_SIZE];

/*
 * PRIVATE FUNCTION DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Callback function used for the confirmation of status descriptor upload.
 *
 * @param[in] env Pointer to the descriptor to update upon confirmation
 ****************************************************************************************
 */
static void rxu_upload_cfm(void *env)
{
    #if NX_FULLY_HOSTED
    struct rxu_stat_desc *rx_stat_desc = env;
    rx_stat_desc->rxdesc = NULL;
    #else
    struct rxu_stat_val *val = env;
    val->status = 0;
    rxl_host_irq_mitigation_update(env);
    #endif
}

/**
 ****************************************************************************************
 * @brief Prepare a RX status descriptor for upload to upper layers
 *
 * @param[in] rxdesc    Pointer to the RX descriptor of the frame
 * @param[in] rx_status    Status to be put in the uploaded descriptor
 * @param[in] host_id      Host Id of the buffer for which the descriptor is uploaded
 *
 * @return A pointer to the configured RX status descriptor. NULL if no descriptor was found
 ****************************************************************************************
 */
static struct rxu_stat_desc *rxu_cntrl_desc_prepare(struct rxdesc *rxdesc,
                                                    uint16_t rx_status,
                                                    uint32_t host_id)
{
    // Structure containing the RX descriptor information
    struct rxu_stat_desc *rx_stat_desc = &rxu_stat_desc_pool[rxu_cntrl_env.rxdesc_idx];
    struct co_list_hdr *list_hdr;

    // Sanity check - the number of internal RX desc is sized to ensure there is always
    // one available
    ASSERT_REC_VAL(rx_stat_desc->val.status == 0, NULL);

    PROF_IPCDESC_PREPARE_SET();

    #if NX_FULLY_HOSTED
    list_hdr = &rx_stat_desc->list_hdr;
    rx_stat_desc->rxdesc = rxdesc;
    if (rxdesc)
    {
        rxdesc->upload_cntrl.cb = rxu_upload_cfm;
        rxdesc->upload_cntrl.env = rx_stat_desc;
    }
    #else
    list_hdr = &rx_stat_desc->upload_cntrl.list_hdr;
    rx_stat_desc->upload_cntrl.rxdesc = rxdesc;
    #endif

    // Fullfil the descriptor value structure
    rx_stat_desc->val.status = rx_status;
    rx_stat_desc->val.host_id = host_id;

    // Insert the element in the ready list
    co_list_push_back(&rxu_cntrl_env.rxdesc_ready, list_hdr);

    // Increment the RX Desc index
    rxu_cntrl_env.rxdesc_idx = (rxu_cntrl_env.rxdesc_idx + 1) % RX_STAT_DESC_CNT;

    PROF_IPCDESC_PREPARE_CLR();

    return (rx_stat_desc);
}

/**
 ****************************************************************************************
 * @brief Compute the length of the MAC Header based on the frame control
 *
 * @param[in] frame_cntl  Frame control of the packet
 *
 * @return The length of the MAC Header
 ****************************************************************************************
 */
__INLINE uint8_t rxu_cntrl_machdr_len_get(uint16_t frame_cntl)
{
    // MAC Header length
    uint8_t mac_hdr_len = MAC_SHORT_MAC_HDR_LEN;

    // Check if Address 4 field is present (FDS and TDS set to 1)
    if ((frame_cntl & (MAC_FCTRL_TODS | MAC_FCTRL_FROMDS))
                                    == (MAC_FCTRL_TODS | MAC_FCTRL_FROMDS))
    {
        mac_hdr_len += (MAC_LONG_MAC_HDR_LEN - MAC_SHORT_MAC_HDR_LEN);
    }

    // Check if QoS Control Field is present
    if (RXL_CNTRL_IS_QOS_DATA(frame_cntl))
    {
        mac_hdr_len += (MAC_LONG_QOS_MAC_HDR_LEN - MAC_LONG_MAC_HDR_LEN);
    }

    // Check if HT Control Field is present (Order bit set to 1)
    if (frame_cntl & MAC_FCTRL_ORDER)
    {
        mac_hdr_len += (MAC_LONG_QOS_HTC_MAC_HDR_LEN - MAC_LONG_QOS_MAC_HDR_LEN);
    }

    return (mac_hdr_len);
}

/**
 ****************************************************************************************
 * @brief This function processes the received encrypted frames
 *
 * @param[in] frame     Pointer to the frame
 * @param[in] statinfo  Status information field set by the MAC HW
 *
 * @return Whether the frame shall be uploaded or not
 ****************************************************************************************
 */
static bool rxu_cntrl_protected_handle(uint8_t *frame, uint32_t statinfo)
{
    uint32_t frame_addr = CPU2HW(frame);
    uint16_t *iv = HW2CPU(frame_addr + rxu_cntrl_env.rx_status.machdr_len);
    bool ga = (statinfo & RX_HD_GA_FRAME) != 0;
    struct key_info_tag *key;
    bool upload = true;

    switch (statinfo & RX_HD_DECRSTATUS)
    {
        case RX_HD_DECR_CCMP128_SUCCESS:
            if (ga)
            {
                struct vif_info_tag *vif = &vif_info_tab[rxu_cntrl_env.rx_status.vif_idx];
                key = &vif->key_info[iv[1] >> 14];
            }
            else
            {
                struct sta_info_tag *sta = &sta_info_tab[rxu_cntrl_env.rx_status.sta_idx];
                key = &sta->sta_sec_info.key_info;
            }
            rxu_cntrl_env.rx_status.key = key;
            #if RW_WAPI_EN && !NX_RX_RING
            if (key->cipher == MAC_CIPHER_WPI_SMS4) {
                rxu_cntrl_env.rx_status.machdr_len += WPI_IV_LEN;
                rxu_cntrl_env.rx_status.pn = (((uint64_t)iv[4]) << 48) |
                    (((uint64_t)iv[3]) << 32) | (((uint64_t)iv[2]) << 16) | iv[1];
                rxu_cntrl_env.rx_status.frame_info |= RXU_CNTRL_PN_CHECK_NEEDED;
            } else
            #endif
            {
                rxu_cntrl_env.rx_status.machdr_len += IV_LEN + EIV_LEN;
                rxu_cntrl_env.rx_status.pn = (((uint64_t)iv[3]) << 32) |
                    (((uint64_t)iv[2]) << 16) | iv[0];
                rxu_cntrl_env.rx_status.frame_info |= RXU_CNTRL_PN_CHECK_NEEDED;
            }
            break;
        #if RW_WAPI_EN && NX_RX_RING
        case RX_HD_DECR_WAPI_SUCCESS:
            if (ga)
            {
                struct vif_info_tag *vif = &vif_info_tab[rxu_cntrl_env.rx_status.vif_idx];
                key = &vif->key_info[iv[1] >> 14];
            }
            else
            {
                struct sta_info_tag *sta = &sta_info_tab[rxu_cntrl_env.rx_status.sta_idx];
                key = &sta->sta_sec_info.key_info;
            }
            rxu_cntrl_env.rx_status.key = key;
            rxu_cntrl_env.rx_status.machdr_len += WPI_IV_LEN;
            rxu_cntrl_env.rx_status.pn = (((uint64_t)iv[4]) << 48) |
                (((uint64_t)iv[3]) << 32) | (((uint64_t)iv[2]) << 16) | iv[1];
            rxu_cntrl_env.rx_status.frame_info |= RXU_CNTRL_PN_CHECK_NEEDED;
            break;
        #endif // RW_WAPI_EN && NX_RX_RING
        case RX_HD_DECR_WEP_SUCCESS:
            rxu_cntrl_env.rx_status.machdr_len += IV_LEN;
            break;
        case RX_HD_DECR_TKIP_SUCCESS:
            rxu_cntrl_env.rx_status.machdr_len += IV_LEN + EIV_LEN;
            rxu_cntrl_env.rx_status.pn = (((uint64_t)iv[3]) << 32) |
                                         (((uint64_t)iv[2]) << 16) |
                                          ((iv[0] & 0xFF) << 8) | (iv[1] & 0xFF);
            rxu_cntrl_env.rx_status.frame_info |= RXU_CNTRL_PN_CHECK_NEEDED |
                                                  RXU_CNTRL_MIC_CHECK_NEEDED;
            if (ga)
            {
                struct vif_info_tag *vif = &vif_info_tab[rxu_cntrl_env.rx_status.vif_idx];
                key = &vif->key_info[iv[1] >> 14];
            }
            else
            {
                struct sta_info_tag *sta = &sta_info_tab[rxu_cntrl_env.rx_status.sta_idx];
                key = &sta->sta_sec_info.key_info;
            }
            rxu_cntrl_env.rx_status.key = key;
            break;
        default:
            upload = false;
            break;
    }

    return upload;
}

/**
 ****************************************************************************************
 * @brief Check if the received frame is not a replayed one
 *
 * @param[in] pn   Packet number of the frame
 * @param[in] key  Key structure attached to the frame
 * @param[in] tid  TID of the frame
 *
 * @return true if the frame shall be uploaded, false if it shall be discarded
 ****************************************************************************************
 */
static bool rxu_cntrl_check_pn(uint64_t *pn, struct key_info_tag *key, uint8_t tid)
{
    if (*pn > key->rx_pn[tid])
    {
        key->rx_pn[tid] = *pn;
        return true;
    }

    return false;
}

/**
 ****************************************************************************************
 * @brief Remove security header (if any) in management frame
 * - When using MFP
 * - When using SHARED-KEY authentication
 *
 * @param[in]  rxdesc RX descriptor attached to the frame
 * @param[out] rx_status Pointer to the status structure of the frame
 ****************************************************************************************
 */
static void rxu_cntrl_remove_sec_hdr_mgmt_frame(struct rxdesc *rxdesc,
                                                struct rx_cntrl_rx_status *rx_status)
{
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    struct rx_payloaddesc *payl_d = HW2CPU(dma_hdrdesc->hd.first_pbd_ptr);
    struct mac_hdr *machdr = HW2CPU(payl_d->pbd.datastartptr);
    uint8_t machdr_len = rxu_cntrl_machdr_len_get(machdr->fctl);
    uint8_t payl_offset = rx_status->machdr_len - machdr_len;
    uint16_t *src, *dst, *start;

    if (payl_offset == 0)
        return;

    ASSERT_WARN((payl_offset & 0x1) == 0);

    start = src = dst = (uint16_t *)machdr;
    src  += machdr_len/2 - 1;
    dst  += rx_status->machdr_len/2 - 1;

    while (src >= start)
    {
        *dst-- = *src--;
    }

    dma_hdrdesc->hd.frmlen -= payl_offset;
    rx_status->payl_offset = payl_offset;
    rx_status->machdr_len -= payl_offset;
}

/**
 ****************************************************************************************
 * @brief Convert the 802.11 MAC Header into a 802.3 Ethernet Header
 *
 * @param[in] rxdesc RX descriptor attached to the frame
 ****************************************************************************************
 */
static void rxu_cntrl_mac2eth_update(struct rxdesc *rxdesc)
{
    #if (RW_MESH_EN)
    // Get VIF Information
    struct vif_info_tag *vif = &vif_info_tab[rxu_cntrl_env.rx_status.vif_idx];
    #endif //(RW_MESH_EN)
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    // First payload descriptor -> Buffer will contain the MAC Header (and IV, EIV if present)
    struct rx_payloaddesc *first_pbd
            = (struct rx_payloaddesc *)HW2CPU(dma_hdrdesc->hd.first_pbd_ptr);
    // Map a MAC Header structure on the frame
    struct mac_hdr *machdr = (struct mac_hdr *)HW2CPU(first_pbd->pbd.datastartptr);
    // LLC/SNAP part of the PDU
    struct llc_snap *llc_snap;
    // Ethernet Header
    struct mac_eth_hdr *eth_hdr;
    // Compute MAC Header Length (will IV length + EIV length if present)
    uint16_t machdr_len = rxu_cntrl_env.rx_status.machdr_len;
    uint8_t payl_offset = machdr_len - sizeof_b(struct mac_eth_hdr);

//    PROF_MAC2ETH_UPDATE_SET();

    #if NX_MON_DATA
    if (vif_mgmt_env.monitor_vif != INVALID_VIF_IDX)
    {
        struct rxu_machdrdesc *machdr_desc = &dma_hdrdesc->mac_hdr_backup;

        // Copy MAC Header, LLC and SNAP before overriding
        memcpy(&machdr_desc->buffer, machdr, machdr_len + LLC_802_2_HDR_LEN);
    }
    #endif

    do
    {
        // For QOS frame get User priority and check for A-MSDU
        dma_hdrdesc->flags &= ~(RX_FLAGS_USER_PRIO_INDEX_MSK);
        if (RXL_CNTRL_IS_QOS_DATA(machdr->fctl))
        {
            uint16_t qos;
            uint16_t prio;

            if ((machdr->fctl & MAC_FCTRL_TODS_FROMDS) ==
                MAC_FCTRL_TODS_FROMDS)
            {
                struct mac_hdr_long_qos *qos_hdr = (struct mac_hdr_long_qos *)machdr;
                qos = qos_hdr->qos;
            }
            else
            {
                struct mac_hdr_qos *qos_hdr = (struct mac_hdr_qos *)machdr;
                qos = qos_hdr->qos;
            }

            prio = ((qos & MAC_QOSCTRL_UP_MSK) >> MAC_QOSCTRL_UP_OFT);
            dma_hdrdesc->flags |= ((prio) << RX_FLAGS_USER_PRIO_INDEX_OFT) & RX_FLAGS_USER_PRIO_INDEX_MSK;

            if (qos & MAC_QOSCTRL_AMSDU_PRESENT)
            {
                dma_hdrdesc->flags |= RX_FLAGS_IS_AMSDU_BIT;
                payl_offset += sizeof_b(struct mac_eth_hdr);
                break;
            }
        }

        #if (RW_MESH_EN)
        if ((vif->type == VIF_MESH_POINT)
                && (rxu_cntrl_env.rx_status.dst_idx != INVALID_STA_IDX))
        {
            // -> Packet becomes
            /********************************************************************
             *  DA  |  SA  |  ETHERTYPE  |  MESH_CONTROL  |  LLC/SNAP  |  DATA  |
             ********************************************************************/
            // Skip MAC Header (Mesh Control not included)
            eth_hdr = (struct mac_eth_hdr *)((uint16_t *)machdr + (machdr_len >> 1) - 7);
            // Pointer to the payload - Skip MAC Header and Mesh Control
            llc_snap = (struct llc_snap *)((uint16_t *)machdr + (machdr_len >> 1)
                                                              + (rxu_cntrl_env.rx_status.mesh_ctrl_len >> 1));

            // Set Ethertype
            eth_hdr->len = llc_snap->proto_id;
        }
        else
        #endif //(RW_MESH_EN)
        {
            // Pointer to the payload - Skip MAC Header
            llc_snap = (struct llc_snap *)((uint16_t *)machdr + (machdr_len >> 1));

            /*
             * Ethernet encapsulated packet structure
             ******************************************
             *  MAC HEADER  |  LLC  |  SNAP  |  DATA  |
             ******************************************
             * or
             *************************
             *  MAC HEADER  |  DATA  |
             *************************
             * The type of structure depends on the LLC fields.
             */
            if ((!memcmp(llc_snap, &llc_rfc1042_hdr, sizeof(llc_rfc1042_hdr))
                 //&& (llc_snap->ether_type != LLC_ETHERTYPE_AARP) - Appletalk depracated ?
                 && (llc_snap->proto_id != LLC_ETHERTYPE_IPX))
                || (!memcmp(llc_snap, &llc_bridge_tunnel_hdr, sizeof(llc_bridge_tunnel_hdr))))
            {
                // Case 1 -> Packet becomes
                /********************************************
                 *  DA  |  SA  |  SNAP->ETHERTYPE  |  DATA  |
                 ********************************************/
                /*
                 * Ethernet Header will start 6 half-words (MAC Address length is 6 bytes) before Ethertag
                 * type
                 */
                eth_hdr = (struct mac_eth_hdr *)((uint16_t *)&(llc_snap->proto_id) - 6);

                // Remove length of LLC/SNAP
                payl_offset += 8;

                rxu_cntrl_env.rx_status.eth_len_present = false;
            }
            else
            {
                // Case 2 -> Packet becomes
                /********************************
                 *  DA  |  SA  |  LEN  |  DATA  |
                 ********************************/
                /*
                 * Ethernet Header will start 7 half-words (MAC Address length is 6 bytes and Length
                 * field is 2 bytes) before LLC Snap
                 */
                eth_hdr = (struct mac_eth_hdr *)((uint16_t *)llc_snap - 7);

                // Set length (Initial length - MAC Header Length)
                eth_hdr->len = co_htons(dma_hdrdesc->hd.frmlen - machdr_len);

                rxu_cntrl_env.rx_status.eth_len_present = true;
            }
        }

        // Set DA and SA in the Ethernet Header
        MAC_ADDR_CPY(&eth_hdr->da, &rxu_cntrl_env.rx_status.da);
        MAC_ADDR_CPY(&eth_hdr->sa, &rxu_cntrl_env.rx_status.sa);

    } while(0);

    // Update Frame Length
    dma_hdrdesc->hd.frmlen -= payl_offset;
    rxu_cntrl_env.rx_status.payl_offset = payl_offset;

    #if NX_MON_DATA
    if (vif_mgmt_env.monitor_vif != INVALID_VIF_IDX)
    {
        struct rxu_machdrdesc *machdr_desc = &dma_hdrdesc->mac_hdr_backup;

        //Set length
        machdr_desc->buf_len = payl_offset + sizeof_b(struct mac_eth_hdr);
    }
    #endif

//    PROF_MAC2ETH_UPDATE_CLR();
}

/**
 ****************************************************************************************
 * @brief Prepare and program the upload of a MSDU to the host buffer
 * This function first calls the MPDU to MSDU conversion procedure.
 *
 * @param[in] rxdesc   RX descriptor attached to the frame
 * @param[in] rx_status   RX status to be put in the RX IPC descriptor
 ****************************************************************************************
 */
static void rxu_msdu_upload_and_indicate(struct rxdesc *rxdesc,
                                         uint16_t rx_status)
{
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    struct rx_cntrl_rx_status *rx_stat = &rxu_cntrl_env.rx_status;

    // Put the VIF and STA information to the RX flags
    dma_hdrdesc->flags |= (rx_stat->sta_idx << RX_FLAGS_STA_INDEX_OFT) |
                          (rx_stat->vif_idx << RX_FLAGS_VIF_INDEX_OFT) |
                          (rx_stat->dst_idx << RX_FLAGS_DST_INDEX_OFT);

    // Translate the MAC frame to an Ethernet frame
    rxu_cntrl_mac2eth_update(rxdesc);
    // Program the DMA transfer of the MPDU
    rxl_mpdu_transfer(rxdesc);

    #if NX_MON_DATA
    if (vif_mgmt_env.monitor_vif != INVALID_VIF_IDX)
    {
        // set frame for monitor interface
        rx_status |= RX_STAT_MONITOR;
    }
    #endif

    // Prepare the descriptor for the last received packet
    rxu_cntrl_desc_prepare(rxdesc, rx_status, rxu_cntrl_env.hostid_current);
}

/**
 ****************************************************************************************
 * @brief Prepare and program the upload of a MPDU to the host buffer
 *
 * @param[in] rxdesc   RX descriptor attached to the frame
 * @param[in] rx_status   RX status to be put in the RX IPC descriptor
 ****************************************************************************************
 */
static void rxu_mpdu_upload_and_indicate(struct rxdesc *rxdesc, uint16_t rx_status)
{
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    struct rx_cntrl_rx_status *rx_stat = &rxu_cntrl_env.rx_status;
    struct rx_hd *rhd = &dma_hdrdesc->hd;
    uint32_t statinfo = rhd->statinfo;

    //Check whether we have a monitor interface
    if (vif_mgmt_env.monitor_vif != INVALID_VIF_IDX)
    {
        // Check if vif is invalid
        if (rx_stat->vif_idx == INVALID_VIF_IDX)
        {
            // Send frame to driver for monitoring only
            rx_status = RX_STAT_MONITOR;
        }
        else
        {
            rx_status |= RX_STAT_MONITOR;
        }
    }

    // Put the VIF and STA information to the RX flags
    dma_hdrdesc->flags |= (rx_stat->sta_idx << RX_FLAGS_STA_INDEX_OFT) |
                          (rx_stat->vif_idx << RX_FLAGS_VIF_INDEX_OFT) |
                          RX_FLAGS_IS_MPDU_BIT;

    #if (RW_MESH_EN)
    if (rx_stat->frame_info & RXU_CNTRL_NEW_MESH_PEER)
    {
        dma_hdrdesc->flags |= RX_FLAGS_NEW_MESH_PEER_BIT;
    }
    #endif //(RW_MESH_EN)

    // MPDU is transfered as-is, so no offset is applied
    rxu_cntrl_env.rx_status.payl_offset = 0;
    // Check whether frame was successfully received
    if (statinfo & RX_HD_SUCCESS)
        // Remove security header in mgmt frame
        rxu_cntrl_remove_sec_hdr_mgmt_frame(rxdesc, rx_stat);
    // Program the DMA transfer of the MPDU
    rxl_mpdu_transfer(rxdesc);
    // Prepare the descriptor for the last received packet
    rxu_cntrl_desc_prepare(rxdesc, rx_status, rxu_cntrl_env.hostid_current);
}

/**
 ****************************************************************************************
 * @brief Extract the Destination and Source Addresses from the MAC Header
 * IEEE 802.11 Address Field
 * See IEEE 802.11-2012 Table 8.19
 *
 * | ToDS | FromDS | Addr1 | Addr2 | Addr3 | Addr4 |
 * |:----:|:------:|:-----:|:-----:|:-----:|:-----:|
 * |   0  |   0    |  DA   |  SA   | BSSID |  n/a  |
 * |   0  |   1    |  DA   | BSSID |  SA   |  n/a  |
 * |   1  |   0    | BSSID |  SA   |  DA   |  n/a  |
 * |   1  |   1    |  RA   |  TA   |  DA   |  SA   |
 *
 * The function fills the corresponding fields in the RX status structure
 *
 * @param[in] machdr_ptr Pointer to the MAC Header
 ****************************************************************************************
 */
static void rxu_cntrl_get_da_sa(struct mac_hdr_long *machdr_ptr)
{
    // Get DA
    if (machdr_ptr->fctl & MAC_FCTRL_TODS)
    {
        MAC_ADDR_CPY(&rxu_cntrl_env.rx_status.da, &machdr_ptr->addr3);
    }
    else
    {
        MAC_ADDR_CPY(&rxu_cntrl_env.rx_status.da, &machdr_ptr->addr1);
    }

    // Get SA
    if (machdr_ptr->fctl & MAC_FCTRL_FROMDS)
    {
        if (machdr_ptr->fctl & MAC_FCTRL_TODS)
        {
            MAC_ADDR_CPY(&rxu_cntrl_env.rx_status.sa, &machdr_ptr->addr4);
        }
        else
        {
            MAC_ADDR_CPY(&rxu_cntrl_env.rx_status.sa, &machdr_ptr->addr3);
        }
    }
    else
    {
        MAC_ADDR_CPY(&rxu_cntrl_env.rx_status.sa, &machdr_ptr->addr2);
    }
}

/**
 ****************************************************************************************
 * @brief Extract various information from the MAC Header and put them in the RX status
 * structure
 *
 * @param[in] frame   Pointer to the frame
 ****************************************************************************************
 */
static void rxu_cntrl_machdr_read(uint8_t *frame)
{
    // Map a MAC Header structure on the frame
    struct mac_hdr *machdr = (struct mac_hdr *)frame;
    uint16_t *qos;

    rxu_cntrl_env.rx_status.frame_cntl = machdr->fctl;
    rxu_cntrl_env.rx_status.seq_cntl = machdr->seq;
    rxu_cntrl_env.rx_status.frame_info = 0;

    rxu_cntrl_env.rx_status.sn = machdr->seq >> MAC_SEQCTRL_NUM_OFT;
    rxu_cntrl_env.rx_status.fn = machdr->seq & MAC_SEQCTRL_FRAG_MSK;

    if ((machdr->fctl & MAC_FCTRL_QOS_DATA) == MAC_FCTRL_QOS_DATA)
    {
        // Check if Address4 field is present
        if ((machdr->fctl & MAC_FCTRL_TODS_FROMDS)
                                    == MAC_FCTRL_TODS_FROMDS)
        {
            qos = &((struct mac_hdr_long_qos *)machdr)->qos;
        }
        else
        {
            qos = &((struct mac_hdr_qos *)machdr)->qos;
        }

        // Extract TID from QoS Control Field
        rxu_cntrl_env.rx_status.tid = (*qos & MAC_QOSCTRL_UP_MSK);

        #if (RW_MESH_EN)
        // Keep address of QoS Control field for later Mesh Control handling
        rxu_cntrl_env.rx_status.a_qos_ctrl = CPU2HW(qos);
        #endif //(RW_MESH_EN)
    }
    else
    {
        rxu_cntrl_env.rx_status.tid = 0;
    }

    rxu_cntrl_env.rx_status.machdr_len = rxu_cntrl_machdr_len_get(machdr->fctl);

    if (MAC_ADDR_GROUP(&machdr->addr1))
        rxu_cntrl_env.rx_status.frame_info |= RXU_CNTRL_GROUP_ADDRESSED;

    // Extract Destination Address and Source Address
    rxu_cntrl_get_da_sa((struct mac_hdr_long *)machdr);
}

/**
 ****************************************************************************************
 * @brief Concatenate two parts of MIC stored at different addresses and write the result
 *        in a buffer
 *
 * @param[in] mic_buffer
 * @param[in] mic_p1_len
 * @param[in] mic_p1_addr
 * @param[in] mic_p2_addr
 ****************************************************************************************
 */
static void rxu_cntrl_mic_rd_concat(uint32_t mic_buffer, uint8_t mic_p1_len,
                                    uint32_t mic_p1_addr, uint32_t mic_p2_addr)
{
    // Copy first part of MIC byte per byte
    co_copy8p(mic_buffer, mic_p1_addr, mic_p1_len);
    mic_buffer += mic_p1_len;

    // Copy second part of MIC byte per byte
    co_copy8p(mic_buffer, mic_p2_addr, MIC_LEN - mic_p1_len);
}

/**
 ****************************************************************************************
 * @brief Compare two MIC values
 *
 * @param[in] mic_value1
 * @param[in] mic_value2
 *
 * @return true if the two MIC are equal, false otherwise
 ****************************************************************************************
 */
static bool rxu_cntrl_mic_compare(uint32_t *mic_value1, uint32_t *mic_value2)
{
    return ((*mic_value1 == *mic_value2) &&
            (*(mic_value1 + 1) == *(mic_value2 + 1)));
}

/**
 ****************************************************************************************
 * @brief Send a TKIP MIC failure indication to the host based on the parameter of the
 * current packet.
 ****************************************************************************************
 */
static void rxu_cntrl_mic_failure(void)
{
    struct rx_cntrl_rx_status *rx_status = &rxu_cntrl_env.rx_status;
    struct me_tkip_mic_failure_ind *ind = KE_MSG_ALLOC(ME_TKIP_MIC_FAILURE_IND, TASK_API,
                                                       TASK_ME, me_tkip_mic_failure_ind);

    // Fill in parameters
    ind->addr = sta_info_tab[rx_status->sta_idx].mac_addr;
    ind->tsc = rx_status->pn;
    ind->ga = (rx_status->statinfo & RX_HD_GA_FRAME) != 0;
    ind->vif_idx = rx_status->vif_idx;
    ind->keyid = rx_status->key->key_idx;

    // Send the message
    ke_msg_send(ind);
}

/**
 ****************************************************************************************
 * @brief This function implements the MIC verification functionality used internally
 * by the UMAC's RX module to verify the integrity of the packets components.
 *
 * @param[in]  rxdesc RX descriptor attached to the received frame
 * @param[in]  mic           Pointer to the MIC computation structure
 * @param[in]  first_frag    Flag indicating whether the frame is first fragment or not
 * @param[in]  last_frag     Flag indicating whether the frame is the last fragment or not
 *
 * @return Whether the frame shall be uploaded (true) or not
 *
 ****************************************************************************************
 */
static bool rxu_cntrl_mic_check(struct rxdesc *rxdesc, struct rxu_mic_calc *mic,
                                bool first_frag, bool last_frag)
{
    struct rx_cntrl_rx_status *rx_status = &rxu_cntrl_env.rx_status;
    struct mic_calc *mic_calc_ptr = &mic->mic_calc;
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    /// Received MIC
    uint32_t rx_mic[2];
    uint32_t rx_mic_addr = CPU2HW(rx_mic);
    bool upload = true;

    /*
     * If first fragment, retrieve the MIC key and initialize the mic_calc structure
     * associated with the provided STAID tuple.
     */
    if (first_frag)
    {
        struct key_info_tag *key = rx_status->key;

        // Initializing mic_calc structure
        me_mic_init(mic_calc_ptr, key->u.mic.rx_key, &rx_status->da,
                        &rx_status->sa, rx_status->tid);
    }

    do
    {
        // Payload Descriptor
        struct rx_payloaddesc *pd = HW2CPU(dma_hdrdesc->hd.first_pbd_ptr);
        /*
         * Total payload Length to proceed (excluding MAC Header, IV, EIV)
         */
        uint16_t data_tot_len = dma_hdrdesc->hd.frmlen - rx_status->machdr_len;
        /*
         * Length on which the MIC is computed
         */
        uint16_t remaining_len;
        uint16_t comp_len;
        uint16_t payl_offset;

        /*
         * If either first_frag or last_frag is False, the packet is part of a fragmented
         * MPDU.
         */
        if (!first_frag)
        {
            uint32_t last_bytes_addr = CPU2HW(mic->last_bytes);
            /*
             * If not first fragment, we should have stored 8 last bytes from the previous
             * received fragment in case these blocks contain a part of the MIC value
             */
            if (last_frag && (data_tot_len < MIC_LEN))
            {
                uint32_t data_ptr = pd->pbd.datastartptr + rx_status->machdr_len;
                uint8_t data_len = MIC_LEN - data_tot_len;

                // Continue MIC Calculation with stored bytes not part of MIC
                me_mic_calc(mic_calc_ptr, last_bytes_addr, data_tot_len);

                // Extract the MIC from the stored data and the received data
                rxu_cntrl_mic_rd_concat(CPU2HW(&rx_mic[0]), data_len,
                                        last_bytes_addr + data_tot_len,
                                        data_ptr);

                // Go to End MIC Calculation + MIC verification step
                break;
            }

            // Continue the MIC Calculation with the 8 stored bytes
            me_mic_calc(mic_calc_ptr, last_bytes_addr, MIC_LEN);
        }

        // Loop as long as the MPDU still has data on which we need to compute the MIC
        remaining_len = data_tot_len - MIC_LEN;
        payl_offset = rx_status->machdr_len;
        while (1)
        {
            uint32_t comp_addr = pd->pbd.datastartptr + payl_offset;
            uint16_t payl_len = pd->pbd.dataendptr - pd->pbd.datastartptr + 1;

            // Check if we have reached the last payload buffer containing the MPDU
            if ((remaining_len + payl_offset) <= payl_len)
            {
                // DMA only the remaining bytes of the payload
                comp_len = remaining_len;
            }
            else
            {
                // The complete payload buffer has to be DMA'ed
                comp_len = payl_len - payl_offset;
            }

            // Compute remaining length to be computed
            remaining_len -= comp_len;

            // Compute the MIC on this payload buffer
            me_mic_calc(mic_calc_ptr, comp_addr, comp_len);

            // Check if we have finished to compute the MIC
            if (remaining_len == 0)
                break;

            // Reset the offset
            payl_offset = 0;

            // Move to the next RBD
            pd = HW2CPU(pd->pbd.next);

            // Sanity check - There shall be a payload descriptor available
            ASSERT_REC_VAL(pd != NULL, false);
        }

        // Now read the last 8 bytes
        remaining_len = MIC_LEN;
        payl_offset += comp_len;
        while (1)
        {
            uint16_t payl_len = pd->pbd.dataendptr - pd->pbd.datastartptr + 1;
            uint32_t mic_addr = pd->pbd.datastartptr + payl_offset;
            uint16_t mic_len;

            // Check if we have reached the last payload buffer containing the MPDU
            if ((remaining_len + payl_offset) <= payl_len)
            {
                // Copy only the remaining bytes of the MIC
                mic_len = remaining_len;
            }
            else
            {
                // The complete payload buffer has to copied
                mic_len = payl_len - payl_offset;
            }

            // Copy the MIC chunk
            co_copy8p(rx_mic_addr, mic_addr, mic_len);

            // Compute the remaining length to be DMA'ed to host
            remaining_len -= mic_len;
            rx_mic_addr += mic_len;

            // Check if we have finished to compute the MIC
            if (remaining_len == 0)
                break;

            // Reset the offset
            payl_offset = 0;

            // Move to the next RBD
            pd = HW2CPU(pd->pbd.next);

            // Sanity check - There shall be a payload descriptor available
            ASSERT_REC_VAL(pd != NULL, false);
        }
    } while (0);

    if (last_frag)
    {
        // Finalizing the MIC calculation; to obtain the final MIC key
        me_mic_end(mic_calc_ptr);

        // Finally check the computed MIC value
        upload = rxu_cntrl_mic_compare(&rx_mic[0], (uint32_t *)mic_calc_ptr);
        if (!upload)
            rxu_cntrl_mic_failure();
    }
    else
    {
        // Copy the 8 bytes known as MIC in the structure
        mic->last_bytes[0] = rx_mic[0];
        mic->last_bytes[1] = rx_mic[1];
    }

    return upload;
}

/**
 ****************************************************************************************
 * @brief This function programs the upload of a fragment of a fragmented MSDU that is
 * not the first one.
 *
 * @param[in]  rxdesc       RX descriptor attached to the received fragment
 * @param[in]  defrag       Pointer to the defragmentation structure
 * @param[in]  cpy_len      Length to be uploaded
 ****************************************************************************************
 */
static void rxu_cntrl_defrag_mpdu_transfer(struct rxdesc *rxdesc,
                                           struct rxu_cntrl_defrag *defrag,
                                           uint16_t cpy_len)
{
    PROF_DEFRAG_TRANSFER_SET();

    // Program upload of fragment
    rxl_mpdu_partial_transfer(rxdesc, cpy_len, defrag->dma_addr, defrag->mac_hdr_len,
                              NULL, NULL);

    // Update stored DMA offset
    defrag->dma_addr += cpy_len;

    PROF_DEFRAG_TRANSFER_CLR();
}

/**
 ****************************************************************************************
 * @brief Go through the list of used reassembly structures in order to find a structure
 * whose station index and tid are the provided values
 *
 * @param[in] sta_idx     Index of the transmitter station
 * @param[in] sn          Sequence number of the received frame
 * @param[in] tid         TID of the frame
 *
 * @return The pointer to the found reassembly structure, if any, NULL otherwise
 ****************************************************************************************
 */
static struct rxu_cntrl_defrag *rxu_cntrl_defrag_get(uint8_t sta_idx, uint16_t sn, uint8_t tid)
{
    // Get the first element of the list of used Reassembly structures
    struct rxu_cntrl_defrag *defrag = (struct rxu_cntrl_defrag *)co_list_pick(&rxu_cntrl_env.rxu_defrag_used);

    while (defrag)
    {
        // Compare Station Id and TID
        if ((defrag->sta_idx == sta_idx)
                && (defrag->tid == tid)
                && (defrag->sn == sn))
        {
            // We found a matching structure, escape from the loop
            break;
        }

        defrag = (struct rxu_cntrl_defrag *)defrag->list_hdr.next;
    }

    // Return found element
    return (defrag);
}

/**
 ****************************************************************************************
 * @brief Allocate a reassembly structure from the free pool. If no structure is available
 * it gets the oldest one from the used pool.
 * In such case the oldest MSDU pending for reassembly is deleted.
 *
 * @return The pointer to the allocated reassembly structure
 ****************************************************************************************
 */
static struct rxu_cntrl_defrag *rxu_cntrl_defrag_alloc(void)
{
    // Get the first element of the list of used Reassembly structures
    struct rxu_cntrl_defrag *defrag =
           (struct rxu_cntrl_defrag *)co_list_pop_front(&rxu_cntrl_env.rxu_defrag_free);

    if (!defrag)
    {
        // Get the first element of the list of used Reassembly structures
        defrag = (struct rxu_cntrl_defrag *)co_list_pop_front(&rxu_cntrl_env.rxu_defrag_used);

        // Sanity check - There shall be an available structure
        ASSERT_ERR(defrag != NULL);

        // Inform the host that the allocated data buffer can be freed
        rxu_cntrl_desc_prepare(NULL, RX_STAT_DELETE, defrag->host_id);
    }

    // Return the allocated element
    return (defrag);
}

/**
 ****************************************************************************************
 * @brief This function uploads the length of a reassembled frame after all fragments
 * have been uploaded.
 *
 * @param[in]  defrag    Pointer to the reassembly structure
 ****************************************************************************************
 */
static void rxu_cntrl_defrag_len_update(struct rxu_cntrl_defrag *defrag)
{
    uint16_t status;
    struct rxu_stat_desc *rx_stat_desc;

    PROF_DEFRAG_UPD_LENGTH_SET();
    // Fulfill the descriptor value structure
    status = defrag->eth_len_present?RX_STAT_LEN_UPDATE|RX_STAT_ETH_LEN_UPDATE:RX_STAT_LEN_UPDATE;

    // Prepare the RX descriptor
    rx_stat_desc = rxu_cntrl_desc_prepare(NULL, status, defrag->host_id);

    if (rx_stat_desc)
        rx_stat_desc->val.frame_len = defrag->frame_len;
    PROF_DEFRAG_UPD_LENGTH_CLR();
}

/**
 ****************************************************************************************
 * @brief This function is called when a reassembly procedure timeout expires.
 * It deletes the MSDU pointed by the reassembly structure.
 *
 * @param[in]  env  Pointer to the reassembly structure
 ****************************************************************************************
 */
static void rxu_cntrl_defrag_timeout_cb(void *env)
{
    struct rxu_cntrl_defrag *defrag = env;

    // Inform the host that the allocated data buffer can be freed
    rxu_cntrl_desc_prepare(NULL, RX_STAT_DELETE, defrag->host_id);

    // Remove the structure from the list
    co_list_extract(&rxu_cntrl_env.rxu_defrag_used, &defrag->list_hdr);
    co_list_push_back(&rxu_cntrl_env.rxu_defrag_free, &defrag->list_hdr);

    // Start the descriptor transfer
    macif_rx_desc_upload(&rxu_cntrl_env.rxdesc_ready);
}

/**
 ****************************************************************************************
 * @brief Check if the received frame shall be reassembled.
 * If not it is uploaded and indicated to the host immediately.
 *
 * @param[in] rxdesc  RX descriptor attached to the received frame
 * @param[in] sta_idx Index of the transmitter station
 * @param[in] qos     Flag indicating whether the frame is a QoS one or not
 *
 * @return Whether the frame shall be uploaded or not
 ****************************************************************************************
 */
static bool rxu_cntrl_defrag_check(struct rxdesc *rxdesc,
                                      uint8_t sta_idx, bool qos)
{
    // IPC status
    bool upload = false;
    // Traffic Index - Use best effort if non-QoS
    uint8_t tid = (qos) ? rxu_cntrl_env.rx_status.tid : 0;
    // RX status structure
    struct rx_cntrl_rx_status *rx_status = &rxu_cntrl_env.rx_status;
    // More Fragment
    uint16_t mf = rx_status->frame_cntl & MAC_FCTRL_MOREFRAG;
    // Reassembly structure
    struct rxu_cntrl_defrag *defrag;
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    // Pointer to the RHD of the packet
    struct rx_hd *rhd = &dma_hdrdesc->hd;

    PROF_DEFRAG_CHECK_SET();

    do
    {
        if (!mf && !rx_status->fn)
        {
            struct rxu_mic_calc mic;

            // Perform the MIC check if required
            if (rx_status->frame_info & RXU_CNTRL_MIC_CHECK_NEEDED)
            {
                if (!rxu_cntrl_mic_check(rxdesc, &mic, true, true))
                    break;
                else
                    // Adjust the packet length to remove the MIC from the end of the payload
                    rhd->frmlen -= MIC_LEN;
            }

            // Data is not fragmented, can be forwarded
            rxu_msdu_upload_and_indicate(rxdesc, RX_STAT_FORWARD | RX_STAT_ALLOC);
            upload = true;
            break;
        }

        // Check if a reassembly procedure is in progress
        defrag = rxu_cntrl_defrag_get(sta_idx, rxu_cntrl_env.rx_status.sn, tid);

        if (!defrag)
        {
            // If not first fragment, we can reject the packet
            if (rx_status->fn)
                break;

            // Allocate a Reassembly structure
            defrag = rxu_cntrl_defrag_alloc();

            // Fullfil the Reassembly structure
            defrag->sta_idx     = sta_idx;
            defrag->tid         = tid;
            defrag->sn          = rx_status->sn;
            defrag->all_rcv     = false;
            defrag->next_fn     = 1;
            // Reset buffer DMA Address or offset
            defrag->dma_addr    = 0;
            // Get MAC Header Length
            defrag->mac_hdr_len = rx_status->machdr_len;
            defrag->timer.cb = rxu_cntrl_defrag_timeout_cb;
            defrag->timer.env = defrag;
            mm_timer_set(&defrag->timer, hal_machw_time() + RX_CNTRL_DEFRAG_MAX_WAIT);

            // Perform the MIC check if required
            if (rx_status->frame_info & RXU_CNTRL_MIC_CHECK_NEEDED)
                rxu_cntrl_mic_check(rxdesc, &defrag->mic, true, false);

            // Transfer the frame to the upper layers
            rxu_msdu_upload_and_indicate(rxdesc, RX_STAT_ALLOC);
            defrag->dma_addr = rx_status->host_buf_addr;
            defrag->host_id = rxu_cntrl_env.hostid_current;
            defrag->eth_len_present = rx_status->eth_len_present;
            // Reset total received length - this has to be done after the MSDU upload
            // as the frmlen field is modified in the MAC2ETH conversion
            defrag->frame_len = rhd->frmlen;

            // Push the reassembly structure at the end of the list of used structures
            co_list_push_back(&rxu_cntrl_env.rxu_defrag_used, &defrag->list_hdr);

            // Data is uploaded
            upload = true;
        }
        else
        {
            uint16_t status;
            uint16_t cpy_len;

            // Check the fragment is the one we are waiting for
            if (defrag->next_fn != rx_status->fn)
                // Packet has already been received
                break;

            // Get payload length of fragment
            cpy_len = rhd->frmlen - (uint16_t)defrag->mac_hdr_len;
            defrag->frame_len += cpy_len;

            // Update number of received fragment
            defrag->next_fn++;

            // Perform the MIC check if required
            status = RX_STAT_FORWARD;
            if (rx_status->frame_info & RXU_CNTRL_MIC_CHECK_NEEDED)
            {
                if (!rxu_cntrl_mic_check(rxdesc, &defrag->mic, false, (mf == 0)))
                    status = RX_STAT_DELETE;
                else if (mf == 0)
                    // Adjust the total frame length to remove the MIC from the end
                    defrag->frame_len -= MIC_LEN;
            }

            // Data can be copied
            rxu_cntrl_defrag_mpdu_transfer(rxdesc, defrag, cpy_len);

            if (!mf)
            {
                // Request for a frame length update before forwarding the packet
                rxu_cntrl_defrag_len_update(defrag);

                // Indicate that the packet can now be transmitted
                rxu_cntrl_desc_prepare(NULL, status, defrag->host_id);

                // Clear the reassembly timer
                mm_timer_clear(&defrag->timer);

                // Free the Reassembly structure
                co_list_extract(&rxu_cntrl_env.rxu_defrag_used, &defrag->list_hdr);
                co_list_push_back(&rxu_cntrl_env.rxu_defrag_free, &defrag->list_hdr);
            }

            upload = true;
        }

        // Store the current time
        defrag->time = hal_machw_time();
    } while (0);

    PROF_DEFRAG_CHECK_CLR();

    return (upload);
}

/**
 ****************************************************************************************
 * @brief Duplicate filtering check for non-registered STA.
 * Check if the received packet is the same than the previous received one and discard
 * if required.
 *
 * @param[in] frame     Pointer to the received frame
 *
 * @return Whether the frame shall be uploaded or not
 ****************************************************************************************
 */
static bool rxu_cntrl_duplicate_nsta_check(uint8_t *frame)
{
    // Returned status
    bool upload = false;

    PROF_RX_DUPLI_NSTA_CHECK_SET();

    do
    {
        // Map a MAC Header structure on the frame
        struct mac_hdr *machdr = (struct mac_hdr *)frame;

        // Check retry bit, source address and received sequence control value
        if ((machdr->fctl & MAC_FCTRL_RETRY) &&
            (rxu_cntrl_env.rxu_dupli.last_seq_cntl == machdr->seq) &&
            (!memcmp(&machdr->addr2, &rxu_cntrl_env.rxu_dupli.last_src_addr, sizeof(struct mac_addr))))
        {
            // Reject the packet
            break;
        }

        rxu_cntrl_env.rxu_dupli.last_seq_cntl = machdr->seq;
        rxu_cntrl_env.rxu_dupli.last_src_addr = machdr->addr2;

        upload = true;
    } while (0);

    PROF_RX_DUPLI_NSTA_CHECK_CLR();

    return (upload);
}

/**
 ****************************************************************************************
 * @brief Duplicate filtering check for registered STA.
 * Check if the received packet is the same than the previous received one and discard
 * if required.
 *
 * @param[in] frame_cntrl   Frame control field of the received frame
 * @param[in] sta_idx       Index of the transmitter STA
 * @param[in] qos           Flag indicating whether the frame is a QoS one or not
 *
 * @return Whether the frame shall be uploaded or not
 ****************************************************************************************
 */
static bool rxu_cntrl_duplicate_check(uint16_t frame_cntrl, uint8_t sta_idx, uint8_t qos)
{
    // Returned status
    bool upload = false;

    PROF_RX_DUPLI_CHECK_SET();

    do
    {
        uint16_t *last_seq_cntl;

        if (qos)
        {
            last_seq_cntl = &(sta_info_tab[sta_idx].rx_qos_last_seqcntl[rxu_cntrl_env.rx_status.tid]);
        }
        else
        {
            last_seq_cntl = &(sta_info_tab[sta_idx].rx_nqos_last_seqcntl);
        }

        // Check retry bit and received sequence control value
        if ((frame_cntrl & MAC_FCTRL_RETRY) &&
            (*last_seq_cntl == rxu_cntrl_env.rx_status.seq_cntl))
        {
            // Reject the packet
            break;
        }

        *last_seq_cntl = rxu_cntrl_env.rx_status.seq_cntl;

        upload = true;
    } while (0);

    PROF_RX_DUPLI_CHECK_CLR();

    return (upload);
}

#if (NX_REORD)
/**
 ****************************************************************************************
 * @brief Move the RX reordering window by one
 *
 * @param[in,out] rx_reord    Pointer to the reordering structure
 ****************************************************************************************
 */
static void rxu_cntrl_reord_update(struct rxu_cntrl_reord *rx_reord)
{
    // Move the windows
    rx_reord->elt[rx_reord->rx_status_pos].host_id = 0;
    rx_reord->win_start     = (rx_reord->win_start + 1) & MAC_SEQCTRL_NUM_MAX;
    rx_reord->rx_status_pos = (rx_reord->rx_status_pos + 1) % RX_CNTRL_REORD_WIN_SIZE;
}

/**
 ****************************************************************************************
 * @brief Flush a certain amount of positions of the RX reordering window
 * This function indicates to the host the packets that have been flushed.
 *
 * @param[in,out] rx_reord    Pointer to the reordering structure
 * @param[in]     sn_skipped  Number of RX reordering window positions to be flushed
 ****************************************************************************************
 */
static void rxu_cntrl_reord_flush(struct rxu_cntrl_reord *rx_reord, uint16_t sn_skipped)
{
    // Forward all packets that have already been received
    for (uint16_t i = 0; (i < sn_skipped) && rx_reord->ooo_pkt_cnt; i++)
    {
        uint8_t index = (rx_reord->rx_status_pos + i) % RX_CNTRL_REORD_WIN_SIZE;
        struct rxu_cntrl_reord_elt *elt = &rx_reord->elt[index];
        uint16_t status = RX_STAT_FORWARD;

        PROF_REORD_FLUSH_SET();

        if (elt->host_id)
        {
            if (elt->pn_check)
            {
                ASSERT_ERR(rx_reord->key);

                if (!rxu_cntrl_check_pn(&elt->pn, rx_reord->key, rx_reord->tid))
                    status = RX_STAT_DELETE;
            }

            // Data has already been copied in host memory and can now be forwarded
            rxu_cntrl_desc_prepare(NULL, status, elt->host_id);

            elt->host_id = 0;

            rx_reord->ooo_pkt_cnt--;
        }

        PROF_REORD_FLUSH_CLR();
    }

    rx_reord->win_start     = (rx_reord->win_start + sn_skipped) & MAC_SEQCTRL_NUM_MAX;
    rx_reord->rx_status_pos = (rx_reord->rx_status_pos + sn_skipped) % RX_CNTRL_REORD_WIN_SIZE;
}

/**
 ****************************************************************************************
 * @brief Indicates to the host all the packets that have been unblocked by the reception
 * of the next waited sequence number
 *
 * @param[in,out] rx_reord    Pointer to the reordering structure
 ****************************************************************************************
 */
static void rxu_cntrl_reord_fwd(struct rxu_cntrl_reord *rx_reord)
{
    while (rx_reord->elt[rx_reord->rx_status_pos].host_id)
    {
        struct rxu_cntrl_reord_elt *elt = &rx_reord->elt[rx_reord->rx_status_pos];
        uint16_t status = RX_STAT_FORWARD;

        PROF_REORD_FWD_SET();

        ASSERT_ERR(rx_reord->ooo_pkt_cnt);

        // Perform PN replay check if required
        if (elt->pn_check)
        {
            ASSERT_ERR(rx_reord->key);

            if (!rxu_cntrl_check_pn(&elt->pn, rx_reord->key, rx_reord->tid))
                status = RX_STAT_DELETE;
        }

        // Data has already been copied in host memory and can now be forwarded
        if (rxu_cntrl_desc_prepare(NULL, status, elt->host_id) == NULL)
        {
            PROF_REORD_FWD_CLR();
            break;
        }

        // Update the SN status
        rxu_cntrl_reord_update(rx_reord);

        rx_reord->ooo_pkt_cnt--;

        PROF_REORD_FWD_CLR();
    }
}

/**
 ****************************************************************************************
 * @brief Update the reordering information accordingly with the provided BlockAck Request
 * PDU
 *
 * @param[in] sta_idx  Index of the transmitter station
 * @param[in] frame    Pointer to the received frame
 ****************************************************************************************
 */
static void rxu_cntrl_reord_bar_check(uint8_t sta_idx, uint8_t *frame)
{
    // Map the BAR structure on the received frame
    struct bar_frame *bar = (struct bar_frame *)frame;
    // TID for which BA is required
    uint8_t tid = (bar->bar_cntrl & BAR_CNTL_TID_MSK) >> BAR_CNTL_TID_OFT;
    // Get the associated BA Agreement
    struct rxu_cntrl_reord *reord = sta_info_tab[sta_idx].ba_agmts_rx[tid];
    // Extract the Start Sequence Number
    uint16_t ssn = bar->bar_information >> MAC_SEQCTRL_NUM_OFT;

    PROF_REORD_BAR_CHECK_SET();

    do
    {
        if (!reord)
        {
            break;
        }

        // See IEEE 802.11-2012 Section 9.21.7.3 c)
        if ((ssn == reord->win_start) ||
            (((ssn - reord->win_start) & MAC_SEQCTRL_NUM_MAX) > (MAC_SEQCTRL_NUM_MAX >> 1)))
        {
            // Make no change to the record
            break;
        }

        /*
         * Flush all needed packet so that:
         *      - WinStart = SSN
         *      - WinEnd   = WinStart + WinSize - 1
         */
        rxu_cntrl_reord_flush(reord, (ssn - reord->win_start) & MAC_SEQCTRL_NUM_MAX);

        // Transfer the descriptors pushed in the ready list
        macif_rx_desc_upload(&rxu_cntrl_env.rxdesc_ready);
    } while (0);

    PROF_REORD_BAR_CHECK_CLR();
}

/**
 ****************************************************************************************
 * @brief Perform the reordering checks on the received frame
 * This function may decide to:
 * - Upload and forward the frame immediately to the host if it is in order
 * - Upload and not forward if the frame is not in order
 * - Discard the frame if already received or too old
 *
 * The RX window is updated according to the previous actions.
 *
 * @param[in] rxdesc   RX descriptor attached to the received frame
 * @param[in] sta_idx  Index of the transmitter station
 *
 * @return Whether the frame shall be uploaded or not
 ****************************************************************************************
 */
static bool rxu_cntrl_reord_check(struct rxdesc *rxdesc, uint8_t sta_idx)
{
    // Returned status
    bool upload = true;
    // SN position in the sn status bit field
    uint16_t sn_pos;
    struct rx_cntrl_rx_status *rx_status = &rxu_cntrl_env.rx_status;

    PROF_REORD_CHECK_SET();

    do
    {
        struct rxu_cntrl_reord *reord
                    = sta_info_tab[sta_idx].ba_agmts_rx[rx_status->tid];

        // Check if the received packet had the lowest expected SN
        if (rx_status->sn == reord->win_start)
        {
            // Perform the PN check if required
            if (rx_status->frame_info & RXU_CNTRL_PN_CHECK_NEEDED)
                upload = rxu_cntrl_check_pn(&rx_status->pn, rx_status->key, rx_status->tid);

            // Check if we need to upload the frame
            if (upload)
            {
                rxu_msdu_upload_and_indicate(rxdesc, RX_STAT_FORWARD | RX_STAT_ALLOC);
            }

            // Received packet is within the reordering window, the RX agreement can be
            // considered as active
            reord->active = true;

            // Store current time
            reord->sn_rx_time = hal_machw_time();

            // Update the RX Window
            rxu_cntrl_reord_update(reord);

            // And forward any ready frames
            rxu_cntrl_reord_fwd(reord);
        }
        else
        {
            sn_pos = (rx_status->sn - reord->win_start) & MAC_SEQCTRL_NUM_MAX;

            if (sn_pos >= RX_CNTRL_REORD_WIN_SIZE)
            {
                if (sn_pos < (MAC_SEQCTRL_NUM_MAX >> 1))
                {
                    // Move the window
                    rxu_cntrl_reord_flush(reord, sn_pos - RX_CNTRL_REORD_WIN_SIZE + 1);

                    // Recompute the SN position
                    sn_pos = (rx_status->sn - reord->win_start) & MAC_SEQCTRL_NUM_MAX;
                }
                // Received packet is older than expected. Two cases apply:
                //   - Either the agreement is still not active, and in that case we shall
                //     just consider the packet as being sent outside the BA agreement,
                //     i.e not consider it as a old one
                //   - Or the agreement is active, and in that case the packet has to be
                //     discarded
                else if (!reord->active)
                {
                    // Perform the duplicate check
                    upload = rxu_cntrl_duplicate_check(rx_status->frame_cntl, sta_idx, 1);

                    // Perform the PN check if required
                    if (upload && (rx_status->frame_info & RXU_CNTRL_PN_CHECK_NEEDED))
                        upload = rxu_cntrl_check_pn(&rx_status->pn, rx_status->key, rx_status->tid);

                    // Check if we need to upload the frame
                    if (upload)
                        rxu_msdu_upload_and_indicate(rxdesc, RX_STAT_FORWARD | RX_STAT_ALLOC);

                    // Indicate the activity on this RX BlockAck agreement
                    bam_rx_active(sta_idx, rx_status->tid);
                    break;
                }
                else
                {
                    // Discard the MPDU
                    upload = false;
                    break;
                }
            }

            // Received packet is within the reordering window (or outside but newer than
            // expected), the RX agreement can be considered as active
            reord->active = true;

            sn_pos = (sn_pos + reord->rx_status_pos) % RX_CNTRL_REORD_WIN_SIZE;

            // Check if the packet has already been received
            if (reord->elt[sn_pos].host_id)
            {
                // Discard the MPDU
                upload = false;
                break;
            }

            // Store the PN and keys if required
            if (rx_status->frame_info & RXU_CNTRL_PN_CHECK_NEEDED)
            {
                reord->key = rx_status->key;
                reord->elt[sn_pos].pn = rx_status->pn;
                reord->elt[sn_pos].pn_check = true;
            }
            else
            {
                reord->elt[sn_pos].pn_check = false;
            }

            // Data has been received out of order
            rxu_msdu_upload_and_indicate(rxdesc, RX_STAT_ALLOC);

            // Store the Host ID in the reordering element
            reord->elt[sn_pos].host_id = rxu_cntrl_env.hostid_current;

            reord->ooo_pkt_cnt++;
        }

        // Indicate the activity on this RX BlockAck agreement
        bam_rx_active(sta_idx, rx_status->tid);
    } while (0);

    PROF_REORD_CHECK_CLR();

    return (upload);
}

/**
 ****************************************************************************************
 * @brief This function is called when the timer of a reordering structure expires.
 * It checks whether we are waiting for too long for the next expected SN. In such case
 * the packet is considered lost and the reordering window is updated accordingly.
 *
 * @param[in] env   Pointer to the reordering structure
 ****************************************************************************************
 */
static void rxu_cntrl_reord_timeout_cb(void *env)
{
    struct rxu_cntrl_reord *reord = env;

    // Restart reordering timer
    mm_timer_set(&reord->timer, ke_time() + RX_CNTRL_REORD_MAX_WAIT);

    do
    {
        // Check if there is at least a packet waiting
        if (!reord->ooo_pkt_cnt)
            break;

        // Check if we spent too much time waiting for an SN
        if (!hal_machw_time_past(reord->sn_rx_time + RX_CNTRL_REORD_MAX_WAIT))
            break;

        if (!reord->elt[reord->rx_status_pos].host_id)
        {
            // Consider the next waited packet as received
            rxu_cntrl_reord_update(reord);
        }
        // Send the data
        rxu_cntrl_reord_fwd(reord);

        // If list of descriptor ready for upload was empty and now contains some elements, start transfer
        if (co_list_pick(&rxu_cntrl_env.rxdesc_ready))
        {
            macif_rx_desc_upload(&rxu_cntrl_env.rxdesc_ready);
        }
    } while (0);
}
#endif //(RX_REORD)

/**
 ****************************************************************************************
 * @brief Spurious frame check for non-registered STA.
 *
 * If Class2/Class3 frame is received on AP interface from non authenticated/associated
 * STA, report frame to upper layer.
 * In all other cases ignore the frame.
 *
 * @param[in]     frame       Pointer to the received frame
 * @param[in,out] rxdesc   Pointer to the RX descriptor of the frame
 *
 * @return Whether the frame shall be uploaded or not. If true then dma descriptor inside
 * RX descriptor has been updated to upload the complete frame as it (whatever its type).
 ****************************************************************************************
 */
static bool rxu_cntrl_spurious_check(uint8_t *frame, struct rxdesc *rxdesc)
{
    struct mac_hdr *machdr = (struct mac_hdr *)frame;
    int i;

    for (i = 0 ; i < NX_VIRT_DEV_MAX ; i++) {
        struct vif_info_tag *vif = &vif_info_tab[i];

        if ((vif->active) && (vif->type == VIF_AP) &&
            MAC_ADDR_CMP_PACKED(&machdr->addr1, &vif->mac_addr)) {

            rxu_cntrl_env.rx_status.vif_idx = vif->index;
            // upload complete frame as it
            rxu_mpdu_upload_and_indicate(rxdesc, RX_STAT_SPURIOUS);
            return true;
        }
    }

    return false;
}

/**
 ****************************************************************************************
 * @brief Check whether the received frame shall trigger the PM monitoring procedure.
 * If the received frame corresponds to the expected one, then the PM state of the
 * transmitter is stored.
 *
 * @param[in] frame     Pointer to the received frame
 * @param[in] statinfo  Status information about the received frame as given by the MAC HW
 ****************************************************************************************
 */
static void rxu_cntrl_pm_mon_check(uint8_t *frame, uint32_t statinfo)
{
    // Map a MAC Header structure on the frame
    struct mac_hdr *hdr = (struct mac_hdr *)frame;

    do
    {
        // Check if the PM monitoring is enabled
        if (!rxu_cntrl_env.pm_mon.mon)
            break;

        // Check if the packet is for the local device
        if (statinfo & RX_HD_ADDRMIS)
            break;

        // Compare the MAC address with the expected one
        if (!MAC_ADDR_CMP(&rxu_cntrl_env.pm_mon.addr, &hdr->addr2))
            break;

        // Check if the peer device is going to sleep or not
        if ((hdr->fctl & (MAC_FCTRL_PWRMGT | MAC_FCTRL_MOREFRAG)) == MAC_FCTRL_PWRMGT)
            // Device is going to sleep
            rxu_cntrl_env.pm_mon.pm_state = PS_MODE_ON;
        else
            // Device is going to be active
            rxu_cntrl_env.pm_mon.pm_state = PS_MODE_OFF;
    } while(0);
}

/**
 ****************************************************************************************
 * @brief Route the received action frame based on its type
 *
 * @param[in]     payload       Pointer to the payload of the received frame.
 * @param[in]     length        Length of the received frame.
 * @param[in]     sta_idx       Index of the station that transmitted the received frame.
 * @param[in,out] vif_idx       VIF that received the frame.
 * @param[out]    task_id       Kernel task to which the frame shall be forwarded.
 * @param[in]     machdr_length Length of the MAC Header.
 * @param[out]    need_machdr   Flag indicating whether the MAC Header is needed for
 *                              subsequent processes.
 *
 * @return A boolean indicating if the frame is handled internally, or needs to be forwarded
 * to the host.
 ****************************************************************************************
 */
static bool rxu_mgt_route_action(uint32_t *payload, uint16_t length, uint8_t sta_idx, uint8_t *vif_idx,
                                 ke_task_id_t *task_id, uint16_t machdr_length, bool *need_machdr)
{
    // Extract the category and action type from the frame
    uint32_t addr           = CPU2HW(payload) + machdr_length;
    uint8_t action_category = co_read8p(addr + MAC_ACTION_CATEGORY_OFT);
    uint16_t task_idx       = 0xFF;
    bool upload = false;

    // Handle action frames for known stations only
    if (sta_idx == INVALID_STA_IDX)
    {
        #if (RW_MESH_EN)
        // If Action Category is Self-Protected, authorize frames from unknown STA
        if (action_category != MAC_SELF_PROT_ACTION_CATEGORY)
        #endif //(RW_MESH_EN)
        {
            return true;
        }
    }

    switch (action_category)
    {
        case MAC_BA_ACTION_CATEGORY:
        {
            *task_id = TASK_BAM;
            task_idx = 0;
        } break;

        #if NX_VHT || NX_HE
        case MAC_VHT_ACTION_CATEGORY:
        {
            uint8_t vht_action = co_read8p(addr + MAC_ACTION_ACTION_OFT);
            switch (vht_action)
            {
                case MAC_OP_MODE_NOTIF_VHT_ACTION:
                {
                    uint8_t opmode = co_read8p(addr + MAC_OP_MODE_NOTIF_OPMODE_OFT);
                    struct sta_info_tag *sta = &sta_info_tab[sta_idx];

                    // This frame is valid if peer device is VHT or HE capable
                    if ((STA_CAPA(sta, VHT) || STA_CAPA(sta, HE)) &&
                        !(opmode & MAC_OPMODE_RXNSS_TYPE_BIT))
                    {
                        uint8_t bw = (opmode & MAC_OPMODE_BW_MSK) >> MAC_OPMODE_BW_OFT;
                        uint8_t nss = (opmode & MAC_OPMODE_RXNSS_MSK) >> MAC_OPMODE_RXNSS_OFT;

                        // Update maximum supported bandwidth
                        me_sta_bw_nss_max_upd(sta_idx, bw, nss);
                    }
                } break;

                #if RW_MUMIMO_RX_EN
                case MAC_GROUP_ID_MGMT_VHT_ACTION:
                {
                    // Set the Membership Status and User Position arrays to the PHY
                    phy_set_group_id_info(addr + MAC_GROUP_ID_MGT_MEMBERSHIP_OFT,
                                          addr + MAC_GROUP_ID_MGT_USER_POS_OFT);
                } break;
                #endif

                default:
                {
                    // Other VHT action categories are not supported
                    upload = true;
                } break;
            }
        } break;
        #endif

        case MAC_HT_ACTION_CATEGORY:
        {
            uint8_t ht_action = co_read8p(addr + MAC_ACTION_ACTION_OFT);
            switch (ht_action)
            {
                case MAC_CHAN_WIDTH_HT_ACTION:
                {
                    uint8_t chwidth = co_read8p(addr + MAC_CHAN_WIDTH_WIDTH_OFT);
                    struct sta_info_tag *sta = &sta_info_tab[sta_idx];

                    // This frame is valid if peer device is HT capable
                    if (STA_CAPA(sta, HT) && (chwidth <= BW_40MHZ))
                        // Update maximum supported bandwidth
                        me_sta_bw_nss_max_upd(sta_idx, chwidth, 0xFF);
                } break;

                case MAC_SMPS_HT_ACTION:
                {
                    struct sta_info_tag *sta = &sta_info_tab[sta_idx];

                    // This frame is valid if peer device is HT capable
                    if (STA_CAPA(sta, HT))
                    {
                        uint8_t sm_pwr_ctrl = co_read8p(addr + MAC_SM_PRW_CTRL_OFT);
                        if (sm_pwr_ctrl & MAC_SMPS_ENABLE_BIT)
                            // Limit NSS for this station to 1
                            me_sta_bw_nss_max_upd(sta_idx, 0xFF, 0);
                        else
                            // No limit for the NSS to be used
                            me_sta_bw_nss_max_upd(sta_idx, 0xFF, 0xFF);
                    }
                } break;

                default:
                {
                    // Other HT action categories are not supported
                    upload = true;
                } break;
            }
        } break;

        #ifdef NX_MFP
        case MAC_SA_QUERY_ACTION_CATEGORY:
        {
            uint8_t sa_query_action;
            struct vif_info_tag *vif;

            if ((length - machdr_length) < MAC_SA_QUERY_MIN_LEN)
            {
                break;
            }
            else if (*vif_idx == INVALID_VIF_IDX)
            {
                upload = true;
                break;
            }

            vif = &vif_info_tab[*vif_idx];
            sa_query_action = co_read8p(addr + MAC_SA_QUERY_ACTION_OFT);

            if ((vif->type == VIF_STA) &&
                (sa_query_action == MAC_SA_QUERY_REQUEST))
            {
                *task_id = TASK_SM;
                task_idx = 0;
            }
            else
            {
                upload = true;
                break;
            }
        } break;
	#endif

        #if (RW_MESH_EN)
        case (MAC_SELF_PROT_ACTION_CATEGORY):
        {
            // Get Mesh VIF on which action frame has been received
            struct mesh_vif_info_tag *mvif;

            if (*vif_idx == INVALID_VIF_IDX)
            {
               // Get addressed Mesh VIF based on received Mesh ID
               mvif = mesh_check_mesh_id(CPU2HW(payload) + MAC_SELF_PROT_ACTION_CAPA_LEN,
                                         length - MAC_SELF_PROT_ACTION_CAPA_LEN);

               if (!mvif)
               {
                   break;
               }

               // Update the VIF information
               *vif_idx = mvif->vif_idx;
            }
            else
            {
                struct vif_info_tag *vif = &vif_info_tab[*vif_idx];

                mvif = &mesh_vif_info_tab[vif->mvif_idx];
            }

            if (mvif->user_mpm)
            {
                // These frames are used for MPM, upload them
                upload = true;
                break;
            }
        } // no break

        case (MAC_MESH_ACTION_CATEGORY):
        {
            *task_id = TASK_MESH;
            task_idx = 0;
            *need_machdr = true;
        } break;
        #endif //(RW_MESH_EN)

        default:
        {
            // Other action categories are not supported
            upload = true;
        } break;
    }

    // format the TaskId taking in to account the instance index
    if (*task_id != TASK_NONE)
    {
        // Sanity check
        ASSERT_ERR(task_idx != 0xFF);
        *task_id = KE_BUILD_ID(*task_id, task_idx);
    }

    return upload;
}

/**
 ****************************************************************************************
 * @brief Route the received management frame based on its type
 *
 * @param[in]     framectrl     Frame control of the received frame.
 * @param[in]     length        Length of the received frame.
 * @param[in]     machdr_length Length of the MAC Header.
 * @param[in]     sta_idx       Index of the station that transmitted the received frame.
 * @param[in,out] vif_idx       VIF that received the frame.
 * @param[in]     payload       Pointer to the payload of the received frame.
 * @param[out]    task_id       Kernel task to which the frame shall be forwarded.
 * @param[out]    need_machdr   Flag indicating whether the MAC Header is needed for
 *                              subsequent processes.
 *
 * @return Whether the frame shall be uploaded or not
 ****************************************************************************************
 */
static bool rxu_mgt_route(uint16_t framectrl,
                          uint16_t length,
                          uint16_t machdr_length,
                          uint8_t sta_idx,
                          uint8_t *vif_idx,
                          uint32_t *payload,
                          ke_task_id_t *task_id,
                          bool *need_machdr)
{
    struct vif_info_tag *vif = NULL;
    uint8_t mode = VIF_UNKNOWN;
    bool upload = true;

    // Get the VIF entry
    if (*vif_idx != INVALID_VIF_IDX)
    {
        vif = &vif_info_tab[*vif_idx];
        mode = vif->type;
    }

    // Route the message depending on its type and subtype
    switch (framectrl & MAC_FCTRL_TYPESUBTYPE_MASK)
    {
        // Message arrive to STA in ESS only
        case MAC_FCTRL_ASSOCRSP:
        case MAC_FCTRL_REASSOCRSP:
            if (mode == VIF_STA)
            {
                *task_id = TASK_SM;
            }
            break;

        case MAC_FCTRL_PROBERSP:
            // Check if the ProbeRsp was for us
            if (vif)
            {
                *need_machdr = true;
                *task_id = TASK_SCANU;
            }
            break;

        case MAC_FCTRL_BEACON:
        {
            // Only forward beacons to SCANU task if we are in scanning state
            if (ke_state_get(TASK_SCANU) == SCANU_SCANNING)
            {
                *need_machdr = true;
                *task_id = TASK_SCANU;
                *vif_idx = scanu_env.param->vif_idx;
            }

            #if (RW_MESH_EN)
            do
            {
                struct mesh_vif_info_tag *mvif;

                if (*task_id != TASK_NONE)
                {
                    break;
                }

                if (*vif_idx == INVALID_VIF_IDX)
                {
                   // Get addressed Mesh VIF based on received Mesh ID
                   mvif = mesh_check_mesh_id(CPU2HW(payload) + MAC_BEACON_VARIABLE_PART_OFT,
                                             length - MAC_BEACON_VARIABLE_PART_OFT);

                   if (!mvif)
                   {
                       break;
                   }

                   // Update the VIF information
                   *vif_idx = mvif->vif_idx;
                   vif = &vif_info_tab[*vif_idx];
                }
                else
                {
                    mvif = &mesh_vif_info_tab[vif->mvif_idx];
                }

                if (!mvif->user_mpm)
                {
                    // Forward the frame to the mesh task
                    *need_machdr = true;
                    *task_id = TASK_MESH;
                }
                else
                {
                    if (sta_idx == INVALID_STA_IDX)
                    {
                        // Map a MAC Header on the frame
                        struct mac_hdr *mac_hdr = (struct mac_hdr *)payload;

                        if (mesh_accept_new_peer(&mac_hdr->addr2, *vif_idx,
                                                 CPU2HW(payload) + MAC_BEACON_VARIABLE_PART_OFT,
                                                 length - MAC_BEACON_VARIABLE_PART_OFT))
                        {
                            rxu_cntrl_env.rx_status.frame_info |= RXU_CNTRL_NEW_MESH_PEER;
                        }
                        else
                        {
                            upload = false;
                        }
                    }
                }
            } while (0);
            #endif //(RW_MESH_EN)

            // Handle a received beacon if sent by a peer AP
            if (sta_idx != INVALID_STA_IDX)
            {
                if (vif->active)
                {
                    me_beacon_check(*vif_idx, length, CPU2HW(payload));
                }
            }
            // else keep TASK_NONE as destination task
        } break;

        case MAC_FCTRL_AUTHENT:
            if (sm_external_auth_in_progress())
            {
                break;
            }
        case MAC_FCTRL_DEAUTHENT:
        case MAC_FCTRL_DISASSOC:
            // routed to SM task (case STA in ESS network)
            if (mode == VIF_STA)
            {
                *task_id = TASK_SM;
            }
            break;

        case MAC_FCTRL_ACTION:
            // Route the action frame based on its category and action type
            // in addition to the sta_mode (AP, STA)
            upload = rxu_mgt_route_action(payload, length, sta_idx, vif_idx, task_id, machdr_length, need_machdr);
            break;
        default:
            break;
    }

    return upload;
}

/**
 ****************************************************************************************
 * @brief Perform the checks on a received management frame to decide whether to upload
 * it or forwarded to an internal kernel task.
 *
 * @param[in]     framectrl     Frame control of the received frame.
 * @param[in]     length        Length of the received frame.
 * @param[in]     sta_idx       Index of the station that transmitted the received frame.
 * @param[in,out] vif_idx       VIF that received the frame.
 * @param[in]     rssi          RSSI of the received frame.
 * @param[in]     pbd           Pointer to the payload of the received frame.
 * @param[in]     machdr_length Length of the MAC Header.
 *
 * @return Whether the frame shall be uploaded or not
 ****************************************************************************************
 */
static bool rxu_mgt_frame_ind(uint16_t framectrl,
                              uint16_t length,
                              uint8_t sta_idx,
                              uint8_t *vif_idx,
                              int8_t rssi,
                              struct rx_pbd *pbd,
                              uint16_t machdr_length)
{
    ke_task_id_t dest_id = TASK_NONE;
    uint32_t *payload = HW2CPU(pbd->datastartptr);
    bool copy_mac_hdr = false;
    bool upload;

    // Route the message to the correct task and/or handle it immediately
    upload = rxu_mgt_route(framectrl, length, machdr_length, sta_idx, vif_idx, payload,
                           &dest_id, &copy_mac_hdr);

    // Check if the message has to be forwarded or not
    if (dest_id != TASK_NONE)
    {
        // Allocate the message
        struct rxu_mgt_ind *rx = KE_MSG_ALLOC_VAR(RXU_MGT_IND, dest_id, TASK_RXU, rxu_mgt_ind, length);
        struct phy_channel_info info;
        uint16_t src_offset = 0;

        // Get the information on the current channel from the PHY driver
        phy_get_channel(&info, PHY_PRIM);

        // Fill in the parameter structure fields that can be filled now
        if (!copy_mac_hdr)
        {
            ASSERT_WARN((machdr_length & 0x3) == 0);
            length -= machdr_length;
            src_offset = machdr_length;
        }
        rx->length = length;
        rxl_mpdu_copy(pbd, length, src_offset, rx->payload);

        rx->inst_nbr = *vif_idx;
        rx->sta_idx = sta_idx;
        rx->framectrl = framectrl;
        rx->rssi = rssi;
        rx->center_freq = (info.info1 >> 16) & 0xFFFF;
        rx->band = info.info1 & 0xFF;

        // Send the message to the destination task
        ke_msg_send(rx);

        // The message has been handled internally, no need to upload it
        upload = false;
    }

    return (upload);
}

/**
 ****************************************************************************************
 * @brief Search to which VIF a received management frame is for.
 *
 * @param[in]  hdr      Pointer to the MAC Header of the received frame.
 * @param[in]  statinfo Status information field as given by the MAC HW
 *
 * @return The VIF index if found, @ref INVALID_VIF_IDX otherwise.
 ****************************************************************************************
 */
static uint8_t rxu_mgt_search_rx_vif(struct mac_hdr *hdr, uint32_t statinfo)
{
    uint8_t vif_idx = INVALID_VIF_IDX;

    if (!(statinfo & (RX_HD_ADDRMIS | RX_HD_GA_FRAME)))
    {
        struct vif_info_tag *vif = vif_mgmt_first_used();

        // Go through the VIF entries to find if one is matching the ADDR1 we got
        while (vif != NULL)
        {
            if (MAC_ADDR_CMP(&vif->mac_addr, &hdr->addr1))
            {
                vif_idx = vif->index;
                break;
            }
            vif = (struct vif_info_tag *)co_list_next(&vif->list_hdr);
        }
    }
    else if (statinfo & RX_HD_GA_FRAME)
    {
        // If ProbeReq look at the address 3 to check if it is ours
    }

    return (vif_idx);
}

/**
 ****************************************************************************************
 * @brief This function processes the received frames that could carry useful information
 * for some UMAC features (connection monitoring, power-save mode, etc.)
 *
 * @param[in] rxdesc  SW header descriptor of the frame
 * @param[in] sta_idx Index of the transmitting station if known, @ref INVALID_STA_IDX
 *                    otherwise
 *
 * @return Whether the frame shall be uploaded or not
 ****************************************************************************************
 */
static bool rxu_mgt_frame_check(struct rxdesc* rxdesc, uint8_t sta_idx)
{
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    struct rx_hd *rhd = &dma_hdrdesc->hd;
    struct rx_payloaddesc *payl_d = HW2CPU(rhd->first_pbd_ptr);
    struct rx_cntrl_rx_status *rx_status = &rxu_cntrl_env.rx_status;
    uint32_t *frame = HW2CPU(payl_d->pbd.datastartptr);
    struct mac_hdr *hdr = (struct mac_hdr *)frame;
    int8_t rssi;
    int8_t rx_rssi[2];
    bool upload = true;

    do
    {
        // Currently no support of fragmented management frames in UMAC
        if ((hdr->fctl & MAC_FCTRL_MOREFRAG) ||
            (hdr->seq & MAC_SEQCTRL_FRAG_MSK))
        {
            upload = false;
            break;
        }

        // Retrieve the RSSI values from the RX vector
        rssi = hal_desc_get_rssi(&rhd->rx_vec_1, rx_rssi);

        // Check if the sending STA is known. If not we search the VIF
        // index receiving the frame, if there is one
        if (sta_idx == INVALID_STA_IDX)
        {
            rx_status->vif_idx = rxu_mgt_search_rx_vif(hdr, rhd->statinfo);
        }
        #if NX_MON_DATA
        // Check if the frame is unicast and not for us (only in case of monitor+data interface)
        else if ((vif_mgmt_env.monitor_vif != INVALID_VIF_IDX) &&
                (!MAC_ADDR_GROUP(&hdr->addr1)) &&
                (MAC_ADDR_CMP(&vif_info_tab[rx_status->vif_idx].mac_addr, &hdr->addr1)))
        {
            rx_status->vif_idx = INVALID_VIF_IDX;
        }
        #endif

        #if NX_MFP
        if (!mfp_ignore_mgmt_frame(rx_status, frame, rhd->frmlen, &upload))
        #endif
            upload = rxu_mgt_frame_ind(hdr->fctl, rhd->frmlen, sta_idx, &rx_status->vif_idx,
                                       rssi, &payl_d->pbd, rx_status->machdr_len);

        if (!upload)
            break;

        // Upload the received MPDU without any modification
        rxu_mpdu_upload_and_indicate(rxdesc, RX_STAT_FORWARD | RX_STAT_ALLOC);
    } while (0);

    return (upload);
}

/**
 ****************************************************************************************
 * @brief Initializes the list of RX IPC descriptors
 ****************************************************************************************
 */
static void rxu_cntrl_desc_init(void)
{
    uint32_t i;
    // Initialize the DMA descriptor
    for (i = 0; i < RX_STAT_DESC_CNT; i++)
    {
        #if !NX_FULLY_HOSTED
        rxu_stat_desc_pool[i].upload_cntrl.cb = rxu_upload_cfm;
        rxu_stat_desc_pool[i].upload_cntrl.env = &rxu_stat_desc_pool[i].val;
        rxu_stat_desc_pool[i].upload_cntrl.flags = 0;
        // No data buffer directly attached to the RHD
        rxu_stat_desc_pool[i].dma_desc.src    = CPU2HW(&rxu_stat_desc_pool[i].val.host_id);
        rxu_stat_desc_pool[i].dma_desc.length = 9;
        rxu_stat_desc_pool[i].dma_desc.ctrl   = RX_LLICTRL(1);
        #endif

        rxu_stat_desc_pool[i].val.status = 0;
    }
}

/**
 ****************************************************************************************
 * @brief Check if the frame has to be uploaded to the driver.
 * If a monitor interface is available and the frame has not been uploaded yet, it will
 * be forwarded to the driver.
 *
 * @param[in] rxdesc   Pointer to the current RX descriptor
 * @param[in] upload Boolean indicating if the frame has to be uploaded for data interface
 *
 * @return true if the frame needs to be uploaded to the host memory, false otherwise
 *
 ****************************************************************************************
 */
static bool rxu_upload_monitor(struct rxdesc *rxdesc, bool upload)
{
    if ((vif_mgmt_env.monitor_vif == INVALID_VIF_IDX) || upload)
    {
        return upload;
    }

    rxu_cntrl_env.rx_status.vif_idx = vif_mgmt_env.monitor_vif;
    rxu_mpdu_upload_and_indicate(rxdesc, RX_STAT_MONITOR);
    return true;
}

/*
 * PUBLIC FUNCTION DEFINITIONS
 ****************************************************************************************
 */

void rxu_cntrl_init(void)
{
    co_list_init(&rxu_cntrl_env.rxdesc_ready);

    #if (NX_REORD)
    co_list_init(&rxu_cntrl_env.rxu_reord_free);

    for (uint16_t i = 0; i < RX_CNTRL_REORD_POOL_SIZE; i++)
    {
        co_list_push_back(&rxu_cntrl_env.rxu_reord_free, &rxu_cntrl_reord_pool[i].list_hdr);
    }
    #endif //(NX_REORD)

    co_list_init(&rxu_cntrl_env.rxu_defrag_free);
    co_list_init(&rxu_cntrl_env.rxu_defrag_used);

    for (uint16_t i = 0; i < RX_CNTRL_DEFRAG_POOL_SIZE; i++)
    {
        co_list_push_back(&rxu_cntrl_env.rxu_defrag_free, &rxu_cntrl_defrag_pool[i].list_hdr);
    }

    rxu_cntrl_env.rxu_dupli.last_seq_cntl = 0xFFFF;

    rxu_cntrl_desc_init();
}

bool rxu_cntrl_frame_handle(struct rxdesc* rxdesc)
{
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    struct rx_hd *rhd = &dma_hdrdesc->hd;
    struct rx_pbd *pd = HW2CPU(rhd->first_pbd_ptr);
    uint8_t *frame = HW2CPU(pd->datastartptr);
    bool upload = false;
    struct rx_cntrl_rx_status *rx_status = &rxu_cntrl_env.rx_status;
    uint16_t frame_cntl;
    uint8_t sta_idx;
    uint32_t statinfo = rhd->statinfo;
    uint16_t key_idx_hw;
    bool qos;

    do
    {
        if (!(statinfo & RX_HD_SUCCESS))
            break;

        frame_cntl = co_read16(frame);
        dma_hdrdesc->flags = 0;

        rx_status->sta_idx = INVALID_STA_IDX;
        rx_status->vif_idx = INVALID_VIF_IDX;

        // Read needed information in the MAC Header
        rxu_cntrl_machdr_read(frame);

        if (!(statinfo & RX_HD_KEYIDV))
        {
            // Perform the check on the PM bit if required
            rxu_cntrl_pm_mon_check(frame, statinfo);

            // When the sender is unknown, only management frames are handled here
            if ((frame_cntl & MAC_FCTRL_TYPE_MASK) != MAC_FCTRL_MGT_T)
            {
                upload = rxu_cntrl_spurious_check(frame, rxdesc);
                break;
            }

            if (!rxu_cntrl_duplicate_nsta_check(frame))
                break;

            // Authentication can be encrypted (when using SHARED-KEY)
            if (RXL_CNTRL_IS_PROTECTED(rx_status->frame_cntl) &&
                (((statinfo & RX_HD_DECRSTATUS) != RX_HD_DECR_WEP_SUCCESS) ||
                 !rxu_cntrl_protected_handle(frame, statinfo))) {
                upload = rxu_cntrl_spurious_check(frame, rxdesc);
                break;
            }

            upload = rxu_mgt_frame_check(rxdesc, INVALID_STA_IDX);
            break;
        }

        // Get the HW key index
        key_idx_hw = (uint16_t)RX_HD_KEYID_GET(statinfo);

        // Retrieve the station index and instance number
        sta_idx = (uint8_t)(key_idx_hw - MM_SEC_DEFAULT_KEY_COUNT);

        // Check if the STA is registered
        if (!sta_mgmt_is_valid(sta_idx))
            break;

        rx_status->sta_idx = sta_idx;
        rx_status->vif_idx = sta_info_tab[sta_idx].inst_nbr;

        // Save the status information field
        rx_status->statinfo = statinfo;

        #if NX_BEACONING
        // In AP mode, check if the destination address is part of our known addresses
        if ((vif_info_tab[rx_status->vif_idx].type == VIF_AP) &&
            (!MAC_ADDR_GROUP(&rx_status->da)))
        {
            rx_status->dst_idx = hal_machw_search_addr(&rx_status->da);
        }
        #endif

        // Check 4 addresses
        if ((frame_cntl & MAC_FCTRL_TODS_FROMDS) == MAC_FCTRL_TODS_FROMDS)
        {
            dma_hdrdesc->flags |= RX_FLAGS_4_ADDR_BIT;
        }

        // Check if received frame was encrypted
        if (RXL_CNTRL_IS_PROTECTED(rx_status->frame_cntl) &&
            !rxu_cntrl_protected_handle(frame, rx_status->statinfo))
            break;

        switch (frame_cntl & MAC_FCTRL_TYPE_MASK)
        {
            case MAC_FCTRL_CTRL_T:
                #if (NX_REORD)
                if ((frame_cntl & MAC_FCTRL_TYPESUBTYPE_MASK) == MAC_FCTRL_BAR)
                {
                    // Check the Start Sequence Numbers contained by the receive Block Ack Request PDU
                    rxu_cntrl_reord_bar_check(sta_idx, frame);
                }
                #endif //(NX_REORD)
                break;

            case MAC_FCTRL_MGT_T:
                // Perform the duplicate check
                if (!rxu_cntrl_duplicate_check(frame_cntl, sta_idx, false))
                    break;

                // Perform the PN check if required
                if ((rx_status->frame_info & RXU_CNTRL_PN_CHECK_NEEDED) &&
                    !rxu_cntrl_check_pn(&rx_status->pn, rx_status->key, TID_MGT))
                     break;

                // Now check if the management is of interest for us
                upload = rxu_mgt_frame_check(rxdesc, rx_status->sta_idx);
                break;

            case MAC_FCTRL_DATA_T:
            {
                #if (RW_MESH_EN)
                // Get VIF information
                struct vif_info_tag *vif = &vif_info_tab[rx_status->vif_idx];

                if (vif->type == VIF_MESH_POINT)
                {
                    if (!mesh_accept_frame(sta_idx))
                    {
                        break;
                    }

                    // Extract the Mesh PS status from the received header
                    mesh_ps_rx_data_handle(rx_status->vif_idx, rx_status->sta_idx, (uint32_t)frame);
                }
                #endif //(RW_MESH_EN)

                // Check if the received frame is a NULL frame
                if (frame_cntl & MAC_NODATA_ST_BIT)
                    break;

                qos = ((frame_cntl & MAC_QOS_ST_BIT) != 0);

                #if (RW_MESH_EN)
                if (vif->type == VIF_MESH_POINT)
                {
                    // Check if this frame is for us or if it has to be forwarded to another Mesh STA
                    if (!mesh_hwmp_check_data_dest(vif, sta_idx, (uint32_t)frame, rx_status))
                    {
                        break;
                    }

                    rx_status->mesh_ctrl_len = mesh_rx_get_machdr_add_len(vif, rx_status->a_qos_ctrl);

                    // If frame has to be forwarded to another Mesh Point, keep Mesh Control as is
                    if (rx_status->dst_idx == INVALID_STA_IDX)
                    {
                        rx_status->machdr_len += rx_status->mesh_ctrl_len;
                    }
                }
                #endif //(RW_MESH_EN)

                #if (NX_REORD)
                // Check if a BA Agreement exists for the STAID
                if (qos && !(rx_status->frame_info & RXU_CNTRL_GROUP_ADDRESSED) &&
                    mm_ba_agmt_rx_exists(sta_idx, rx_status->tid))
                {
                    // Check if received packet was expected
                    // If yes, the packet can be uploaded in host memory
                    upload = rxu_cntrl_reord_check(rxdesc, sta_idx);
                }
                // Check if MPDU is part of an A-MPDU by checking its position inside
                // of the A-MPDU.
                else if (RX_AMPDU_MPDUCNT(rhd->ampdu_stat_info) == 1)
                {
                    bam_send_del_ba_agg(sta_idx, rx_status->tid);
                }
                else
                #endif //(NX_REORD)
                {
                    // Perform the duplicate check
                    if (!rxu_cntrl_duplicate_check(frame_cntl, sta_idx, qos))
                        break;

                    // Perform the PN check if required
                    if ((rx_status->frame_info & RXU_CNTRL_PN_CHECK_NEEDED) &&
                        !rxu_cntrl_check_pn(&rx_status->pn, rx_status->key, rx_status->tid))
                         break;

                    /*
                     * Packet has not been detected as a duplicate one. Check if reassembly procedure
                     * has to be used
                     */
                    upload = rxu_cntrl_defrag_check(rxdesc, sta_idx, qos);
                }
            } break;

            default:
                break;
        }
    } while (0);

    //Upload for monitor interface (if any)
    upload = rxu_upload_monitor(rxdesc, upload);

    // Transfer the pending descriptors
    macif_rx_desc_upload(&rxu_cntrl_env.rxdesc_ready);

    return (upload);
}

#if (NX_REORD)
bool rxu_cntrl_reord_create(struct sta_info_tag *sta, uint8_t tid, uint16_t ssn)
{
    struct rxu_cntrl_reord *reord;

    // Check if we already have a reordering structure allocated for this STA/TID pair
    if (sta->ba_agmts_rx[tid] != NULL)
        return false;

    // Get a free reordering structure
    reord = (struct rxu_cntrl_reord *) co_list_pop_front(&rxu_cntrl_env.rxu_reord_free);
    if (!reord)
        return false;

    // Reset the content
    memset(reord, 0, sizeof(struct rxu_cntrl_reord));

    // Configure the RX SN Window
    reord->tid = tid;
    reord->win_start = ssn;
    reord->rx_status_pos = ssn % RX_CNTRL_REORD_WIN_SIZE;
    reord->sn_rx_time = hal_machw_time();
    reord->timer.cb = rxu_cntrl_reord_timeout_cb;
    reord->timer.env = reord;

    // Save the pointer to the reordering structure
    sta->ba_agmts_rx[tid] = reord;

    // Start the reordering timer
    mm_timer_set(&reord->timer, reord->sn_rx_time + RX_CNTRL_REORD_MAX_WAIT);

    return true;
}

void rxu_cntrl_reord_delete(struct sta_info_tag *sta, uint8_t tid)
{
    struct rxu_cntrl_reord *reord = sta->ba_agmts_rx[tid];

    // Sanity check - The element shall exist
    ASSERT_ERR(reord != NULL);

    // Stop the reordering timer
    mm_timer_clear(&reord->timer);

    // Push back the reordering structure to the free list
    co_list_push_back(&rxu_cntrl_env.rxu_reord_free,
                      &reord->list_hdr);

    // Delete the pointer to the reordering structure
    sta->ba_agmts_rx[tid] = NULL;
}
#endif //(NX_REORD)

void rxu_cntrl_monitor_pm(struct mac_addr *addr)
{
    // Only one address can be monitored at a time
    if (rxu_cntrl_env.pm_mon.mon)
        return;

    // Copy the MAC address
    MAC_ADDR_CPY(&rxu_cntrl_env.pm_mon.addr, addr);
    // Set the PM state to active
    rxu_cntrl_env.pm_mon.pm_state = PS_MODE_OFF;
    // Enable the PM monitoring
    rxu_cntrl_env.pm_mon.mon = true;
}

uint8_t rxu_cntrl_get_pm(void)
{
    uint8_t pm_state = rxu_cntrl_env.pm_mon.pm_state;

    // Disable the PM monitoring
    rxu_cntrl_env.pm_mon.mon = false;
    // Set the PM state to active
    rxu_cntrl_env.pm_mon.pm_state = PS_MODE_OFF;

    return (pm_state);
}

void rxu_cntrl_evt(int dummy)
{
    // for profiling
    RX_CNTRL_EVT_SET();

    // Clear the kernel event
    ke_evt_clear(KE_EVT_RXUREADY_BIT);

    // Transfer the pending descriptor list
    macif_rx_desc_upload(&rxu_cntrl_env.rxdesc_ready);

    // for profiling
    RX_CNTRL_EVT_CLR();
}

bool rxu_cntrl_desc_check(void)
{
    return (rxu_stat_desc_pool[rxu_cntrl_env.rxdesc_idx].val.status == 0);
}


/// @}

