/**
 ****************************************************************************************
 *
 * @file sm.c
 *
 * @brief The UMAC's STA Manager (SM) module implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

#include "sm.h"
#include "sm_task.h"
#include "scanu.h"
#include "scanu_task.h"
#include "me.h"
#include "co_utils.h"
#include "co_endian.h"
#include "me_utils.h"
#include "mm_task.h"
#include "rxu_task.h"
#include "mac_frame.h"
#include "mac_ie.h"
#include "me_mgmtframe.h"
#include "vif_mgmt.h"
#include "sta_mgmt.h"
#include "ps.h"
#include "txu_cntrl.h"
#if NX_MFP
#include "mfp.h"
#endif
#include "tpc.h"

/** @addtogroup SM
* @{
*/

struct sm_env_tag sm_env;

/// Timeout, in us, for external authentication
#if NX_FULLY_HOSTED
#define SM_EXTERNAL_AUTH_TIMEOUT 10000000
#else
#define SM_EXTERNAL_AUTH_TIMEOUT 600000
#endif

/**
 ****************************************************************************************
 * @brief Frame transmission confirmation handler
 * If the frame is not successfully transmitted it is pushed again until the timeout
 * expires.
 *
 * @param[in] env     Pointer to the frame descriptor
 * @param[in] status  Status of the transmission
 ****************************************************************************************
 */
static void sm_frame_tx_cfm_handler(void *env, uint32_t status)
{
    ke_state_t sm_state = ke_state_get(TASK_SM);
    struct txl_frame_desc_tag *frame = env;

    // Authentication and Association Request frames can be sent until expiration of
    // SM_RSP_TIMEOUT_IND timer.
    if (!(status & (DESC_DONE_SW_TX_BIT | RETRY_LIMIT_REACHED_BIT | LIFETIME_EXPIRED_BIT)))
        return;

    switch (sm_state)
    {
        case (SM_AUTHENTICATING):
        case (SM_ASSOCIATING):
        {
            // Push again the frame for TX
            txl_frame_push(frame, AC_VO);

            // Inform txl_frame module that descriptor cannot be freed
            frame->keep_desc = true;
        } break;

        default:
        {
            // Nothing to do
        } break;
    }
}

/**
 ****************************************************************************************
 * @brief Initialize the BSS configuration list.
 ****************************************************************************************
 */
static void sm_bss_config_init(void)
{
    // Sanity check - No BSS config should be ongoing
    ASSERT_ERR(co_list_is_empty(&sm_env.bss_config));

    // Initialize the BSS configuration list
    co_list_init(&sm_env.bss_config);
}

/**
 ****************************************************************************************
 * @brief Push a BSS configuration message to the list.
 *
 * @param[in] param  Pointer to the message parameters
 ****************************************************************************************
 */
static void sm_bss_config_push(void *param)
{
    struct ke_msg *msg = ke_param2msg(param);

    co_list_push_back(&sm_env.bss_config, &msg->hdr);
}

/**
 ****************************************************************************************
 * @brief DEAUTH frame transmission confirmation handler
 * Once the transmission is confirmed we can continue the disconnection process.
 *
 * @param[in] env     Pointer to the VIF
 * @param[in] status  Status of the transmission
 ****************************************************************************************
 */
static void sm_deauth_cfm(void *env, uint32_t status)
{
    sm_disconnect_process(env, 0);
}

/**
 ****************************************************************************************
 * @brief Delete the different SW resources that were allocated for this connection.
 *
 * @param[in] vif     Pointer to the VIF
 ****************************************************************************************
 */
static void sm_delete_resources(struct vif_info_tag *vif)
{
    struct me_set_ps_disable_req *ps = KE_MSG_ALLOC(ME_SET_PS_DISABLE_REQ, TASK_ME, TASK_SM,
                                                    me_set_ps_disable_req);
    struct me_set_active_req *idle =  KE_MSG_ALLOC(ME_SET_ACTIVE_REQ, TASK_ME,
                                                    TASK_SM, me_set_active_req);

    // Initialize the BSS configuration list
    sm_bss_config_init();

    #if NX_AMPDU_TX || NX_REORD
    // Release all the Block Ack agreement
    bam_delete_all_ba_agg(vif->u.sta.ap_id);
    #endif

    // Re-allow PS mode in case it was disallowed
    ps->ps_disable = false;
    ps->vif_idx = vif->index;
    sm_bss_config_push(ps);

    // Assoc
    if (vif->active)
    {
        struct mm_set_vif_state_req *assoc = KE_MSG_ALLOC(MM_SET_VIF_STATE_REQ, TASK_MM,
                                                           TASK_SM, mm_set_vif_state_req);
        assoc->active = false;
        assoc->inst_nbr = vif->index;
        sm_bss_config_push(assoc);
    }

    // Delete station
    if (vif->u.sta.ap_id != INVALID_STA_IDX)
    {
        struct mm_sta_del_req *sta_del = KE_MSG_ALLOC(MM_STA_DEL_REQ, TASK_MM,
                                                           TASK_SM, mm_sta_del_req);
        sta_del->sta_idx = vif->u.sta.ap_id;
        sm_bss_config_push(sta_del);
    }

    // Unlink the VIF from the channel context
    if (vif->chan_ctxt != NULL)
    {
        struct mm_chan_ctxt_unlink_req *unlk = KE_MSG_ALLOC(MM_CHAN_CTXT_UNLINK_REQ,TASK_MM,
                                                            TASK_SM, mm_chan_ctxt_unlink_req);
        unlk->vif_index = vif->index;
        sm_bss_config_push(unlk);
    }

    idle->active = false;
    idle->vif_idx = vif->index;
    sm_bss_config_push(idle);

    // Send the first BSS disconnection message
    sm_bss_config_send();
}

void sm_init(void)
{
    // Value of pointer of the scan result is NULL
    sm_env.connect_param = NULL;
    //Station index value
     //Set the state as IDLE
    ke_state_set(TASK_SM, SM_IDLE);
}


void sm_get_bss_params(struct mac_addr const **bssid,
                       struct mac_chan_def const **chan)
{
    struct sm_connect_req const *param = sm_env.connect_param;
    struct mac_scan_result *desired_ap_ptr;

    *bssid = NULL;
    *chan = NULL;

    // In order to launch the join procedure we need the BSSID and the channel,
    // otherwise we will need to get this information by doing a scan
    if (MAC_ADDR_GROUP(&param->bssid))
    {
        desired_ap_ptr = scanu_search_by_ssid(&param->ssid);
        if (desired_ap_ptr)
            *bssid = &desired_ap_ptr->bssid;
    }
    else
    {
        *bssid = &param->bssid;
        desired_ap_ptr = scanu_search_by_bssid(&param->bssid);
    }

    // Get channel
    if (desired_ap_ptr)
        *chan = desired_ap_ptr->chan;
    else if (param->chan.freq != ((uint16_t)-1))
        *chan = &param->chan;
}

void sm_scan_bss(struct mac_addr const *bssid,
                 struct mac_chan_def const *chan)
{
    struct sm_connect_req const *param = sm_env.connect_param;
    // Prepare the scan request
    struct scanu_start_req *req = KE_MSG_ALLOC(SCANU_START_REQ, TASK_SCANU, TASK_SM,
                                                                    scanu_start_req);
    // fill in message
    req->vif_idx = param->vif_idx;
    req->add_ies = 0;
    req->add_ie_len = 0;
    req->ssid[0] = param->ssid;
    req->ssid_cnt = 1;
    req->bssid = bssid?*bssid:mac_addr_bcst;
    if (chan)
    {
        req->chan[0] = *chan;
        req->chan_cnt = 1;
    }
    else
    {
        int i, j;
        struct mac_chan_def *chan[PHY_BAND_MAX] = {me_env.chan.chan2G4, me_env.chan.chan5G};
        uint8_t chan_cnt[PHY_BAND_MAX] = {me_env.chan.chan2G4_cnt, me_env.chan.chan5G_cnt};

        req->chan_cnt = 0;
        for (i = 0; i < PHY_BAND_MAX; i++)
            for (j = 0; j < chan_cnt[i]; j++)
                if (!(chan[i][j].flags & CHAN_DISABLED))
                    req->chan[req->chan_cnt++] = chan[i][j];
    }

    // Send the message
    ke_msg_send(req);

    // We are now waiting for the scan results
    ke_state_set(TASK_SM, SM_SCANNING);
}


void sm_join_bss(struct mac_addr const *bssid,
                 struct mac_chan_def const *chan,
                 bool passive)
{
    struct sm_connect_req const *param = sm_env.connect_param;
    struct scanu_start_req *req = KE_MSG_ALLOC(SCANU_JOIN_REQ, TASK_SCANU, TASK_SM,
                                                                    scanu_start_req);

    req->chan[0] = *chan;
    req->chan_cnt = 1;
    req->ssid[0] = param->ssid;
    req->ssid_cnt = 1;
    req->add_ie_len = 0;
    req->add_ies = 0;
    req->vif_idx = param->vif_idx;
    req->bssid = *bssid;

    // Check if passive scan is required
    if (passive)
    {
        req->chan[0].flags |= CHAN_NO_IR;
    }

    // Set next join active / passive
    sm_env.join_passive = !passive;

    // Send the message
    ke_msg_send(req);

    // We are now waiting for the end of the joining procedure
    ke_state_set(TASK_SM, SM_JOINING);
}


uint8_t sm_add_chan_ctx(uint8_t *chan_idx)
{
    // Retrieve connection information
    struct sm_connect_req const *param = sm_env.connect_param;
    struct vif_info_tag *vif = &vif_info_tab[param->vif_idx];
    struct me_bss_info *bss = &vif->bss_info;

    // Add the channel context and return status and channel index
    return (chan_ctxt_add(&bss->chan, chan_idx));
}


void sm_set_bss_param(void)
{
    int i;
    struct sm_connect_req const *param = sm_env.connect_param;
    struct vif_info_tag *vif = &vif_info_tab[param->vif_idx];
    struct me_bss_info *bss = &vif->bss_info;
    struct me_set_ps_disable_req *ps = KE_MSG_ALLOC(ME_SET_PS_DISABLE_REQ, TASK_ME, TASK_SM,
                                                    me_set_ps_disable_req);
    struct mm_set_bssid_req *bssid = KE_MSG_ALLOC(MM_SET_BSSID_REQ, TASK_MM, TASK_SM,
                                                  mm_set_bssid_req);
    struct mm_set_basic_rates_req *brates = KE_MSG_ALLOC(MM_SET_BASIC_RATES_REQ, TASK_MM,
                                                         TASK_SM, mm_set_basic_rates_req);
    struct mm_set_beacon_int_req *bint = KE_MSG_ALLOC(MM_SET_BEACON_INT_REQ, TASK_MM,
                                                      TASK_SM, mm_set_beacon_int_req);
    struct me_set_active_req *active =  KE_MSG_ALLOC(ME_SET_ACTIVE_REQ, TASK_ME,
                                                     TASK_SM, me_set_active_req);

    // Initialize the BSS configuration list
    sm_bss_config_init();

    // Disable PS mode prior to the association procedure
    ps->ps_disable = true;
    ps->vif_idx = param->vif_idx;
    sm_bss_config_push(ps);

    // BSSID
    bssid->bssid = bss->bssid;
    bssid->inst_nbr = param->vif_idx;
    sm_bss_config_push(bssid);

    // Basic rates
    brates->band = bss->chan.band;
    brates->rates = me_legacy_rate_bitfield_build(&bss->rate_set, true);
    brates->inst_nbr = param->vif_idx;
    sm_bss_config_push(brates);

    // Beacon interval
    bint->beacon_int = bss->beacon_period;
    bint->inst_nbr = param->vif_idx;
    sm_bss_config_push(bint);

    #if NX_HE
    if (BSS_CAPA(bss, HE))
    {
        struct mm_set_bss_color_req *color = KE_MSG_ALLOC(MM_SET_BSS_COLOR_REQ, TASK_MM,
                                                          TASK_SM, mm_set_bss_color_req);
        color->bss_color = me_build_bss_color_reg(bss);
        sm_bss_config_push(color);
    }
    #endif

    // EDCA parameters
    for (i = 0; i < AC_MAX; i++)
    {
        struct mm_set_edca_req *edca = KE_MSG_ALLOC(MM_SET_EDCA_REQ, TASK_MM,
                                                    TASK_SM, mm_set_edca_req);

        edca->ac_param = bss->edca_param.ac_param[i];
        edca->hw_queue = i;
        edca->inst_nbr = param->vif_idx;
        #if NX_UAPSD
        if (ps_uapsd_enabled() &&
            (bss->edca_param.qos_info & MAC_QOS_INFO_AP_UAPSD_ENABLED))
        {
            edca->uapsd = (mac_ac2uapsd[i] & param->uapsd_queues) != 0;
        }
        else
        #endif
        {
            edca->uapsd = false;
        }
        sm_bss_config_push(edca);
    }

    // Go back to ACTIVE after setting the BSS parameters
    active->active = true;
    active->vif_idx = param->vif_idx;
    sm_bss_config_push(active);

    // Send the first BSS configuration message
    sm_bss_config_send();

    // We are now waiting for the channel addition confirmation
    ke_state_set(TASK_SM, SM_BSS_PARAM_SETTING);
}

void sm_bss_config_send(void)
{
    struct ke_msg *msg = (struct ke_msg *)co_list_pop_front(&sm_env.bss_config);

    // Sanity check - We shall have a message available
    ASSERT_ERR(msg != NULL);

    // Send the message
    ke_msg_send(ke_msg2param(msg));
}

void sm_disconnect(uint8_t vif_index, uint16_t reason_code)
{
    struct vif_info_tag *vif = &vif_info_tab[vif_index];
    struct txl_frame_desc_tag *frame;
    struct mac_hdr *buf;
    struct tx_hd *thd;
    struct sta_info_tag *sta = &sta_info_tab[vif->u.sta.ap_id];
    uint32_t length;
    int txtype;

    if ((vif->type != VIF_STA) || (!vif->active))
    {
        ke_msg_send_basic(SM_DISCONNECT_CFM, TASK_API, TASK_SM);
        return;
    }

    sm_env.host_disconnect = true;
    ke_state_set(TASK_SM, SM_DISCONNECTING);

    #if (NX_P2P)
    if (vif->p2p)
    {
        txtype = TX_DEFAULT_5G;
    }
    else
    #endif //(NX_P2P)
    {
        struct me_bss_info *bss = &vif->bss_info;

        // Chose the right rate according to the band
        txtype = (bss->chan.band == PHY_BAND_2G4) ? TX_DEFAULT_24G : TX_DEFAULT_5G;
    }

    // Allocate a frame descriptor from the TX path
    frame = txl_frame_get(txtype, NX_TXFRAME_LEN);
    if (frame != NULL)
    {
        #if NX_MFP
        enum mfp_protection mfp;
        #endif

        // update Tx power in policy table
        tpc_update_frame_tx_power(vif, frame);

        // Get the buffer pointer
        buf = txl_buffer_payload_get(&frame->txdesc);

        // Fill-in the frame
        buf->fctl = MAC_FCTRL_DEAUTHENT;
        buf->durid = 0;
        buf->addr1 = sta->mac_addr;
        buf->addr2 = vif->mac_addr;
        buf->addr3 = sta->mac_addr;
        buf->seq = txl_get_seq_ctrl();

        // Fill-in the confirmation structure
        frame->cfm.cfm_func = sm_deauth_cfm;
        frame->cfm.env = vif;

        // Set VIF and STA indexes
        frame->txdesc.host.vif_idx = vif_index;
        frame->txdesc.host.staid   = vif->u.sta.ap_id;

        length = MAC_SHORT_MAC_HDR_LEN;

        #if NX_MFP
        frame->txdesc.umac.head_len = 0;
        frame->txdesc.umac.tail_len = 0;
        mfp = mfp_protect_mgmt_frame(&frame->txdesc, buf->fctl, 0);
        if (mfp == MFP_UNICAST_PROT)
        {
            txu_cntrl_protect_mgmt_frame(&frame->txdesc, buf, MAC_SHORT_MAC_HDR_LEN);
            length += frame->txdesc.umac.head_len;
        }
        #endif

        // Build the payload
        length += me_build_deauthenticate(CPU2HW(buf) + length, reason_code);

        #if NX_MFP
        if (mfp == MFP_MULTICAST_PROT)
        {
            length += mfp_add_mgmt_mic(&frame->txdesc, CPU2HW(buf), length, 0);
        }
        else if (mfp == MFP_UNICAST_PROT)
        {
            length += frame->txdesc.umac.tail_len;
        }
        #endif

        // Update the length
        thd = &frame->txdesc.lmac.hw_desc->thd;
        thd->dataendptr = (uint32_t)thd->datastartptr + length - 1;
        thd->frmlen = length + MAC_FCS_LEN;

        // Push the frame for TX
        if (!txl_frame_push(frame, AC_VO))
        {
            sm_deauth_cfm(frame->cfm.env, 0);
        }
    }
    else
    {
        // No frame available, simply consider us as disconnected immediately
        sm_disconnect_process(vif, 0);
    }
}

void sm_disconnect_process(struct vif_info_tag *vif, uint16_t reason)
{
    struct sm_disconnect_ind *disc = KE_MSG_ALLOC(SM_DISCONNECT_IND, TASK_API, TASK_SM,
                                                  sm_disconnect_ind);

    // Save a pointer to the VIF we are handling
    sm_env.vif_disconnect = vif;

    // Delete the resources that were allocated for this connection
    sm_delete_resources(vif);

    // Fill in the indication parameters
    disc->reason_code = reason;
    disc->vif_idx = vif->index;
    if (sm_env.ft_over_ds == 1)
    {
        disc->ft_over_ds = 1;
    }

    sm_env.disconnect_ind = ke_param2msg(disc);
}

void sm_connect_ind(uint16_t status)
{
    struct sm_connect_ind *ind = sm_env.connect_ind;
    struct sm_connect_req const *con_par = sm_env.connect_param;
    struct vif_info_tag *vif = &vif_info_tab[con_par->vif_idx];
    struct me_bss_info *bss = &vif->bss_info;

    // Fill the message parameters
    ind->vif_idx = con_par->vif_idx;
    ind->status_code = status;

    if (status == MAC_ST_SUCCESSFUL)
    {
        ind->bssid = bss->bssid;
        ind->ap_idx = vif->u.sta.ap_id;
        ind->ch_idx = vif->chan_ctxt->idx;
        ind->chan = vif->bss_info.chan;
        //ind->aid = ??
        ind->qos = BSS_CAPA(bss, QOS);
        ind->acm = ind->qos?bss->edca_param.acm:0;
        // This connection results from a host request
        ind->roamed = false;
        #if (NX_TDLS)
        memcpy(ind->ac_param, bss->edca_param.ac_param, sizeof(ind->ac_param));
        #endif

        // Send the message
        ke_msg_send(ind);

        // Set state to back to IDLE
        ke_state_set(TASK_SM, SM_IDLE);
    }
    else
    {
        ke_state_set(TASK_SM, SM_DISCONNECTING);
        sm_env.vif_disconnect = vif;
        sm_env.disconnect_ind = ke_param2msg(ind);

        // Delete the resources that were allocated for this connection
        sm_delete_resources(vif);
    }

    // Free the connection parameters
    ke_msg_free(ke_param2msg(sm_env.connect_param));
    sm_env.connect_param = NULL;
    sm_env.ft_over_ds = 0;
}

void sm_auth_send(uint16_t auth_seq, uint32_t *challenge)
{
    struct sm_connect_req const *con_par = sm_env.connect_param;
    struct vif_info_tag *vif = &vif_info_tab[con_par->vif_idx];
    struct txl_frame_desc_tag *frame;
    struct mac_hdr *buf;
    struct tx_hd *thd;
    struct sta_info_tag *sta = &sta_info_tab[vif->u.sta.ap_id];
    uint32_t length;
    // Chose the right rate according to the band
    int txtype;

    #if (NX_P2P)
    if (vif->p2p)
    {
        txtype = TX_DEFAULT_5G;
    }
    else
    #endif //(NX_P2P)
    {
        struct me_bss_info *bss = &vif->bss_info;

        txtype = (bss->chan.band == PHY_BAND_2G4) ? TX_DEFAULT_24G : TX_DEFAULT_5G;
    }

    // Allocate a frame descriptor from the TX path
    frame = txl_frame_get(txtype, NX_TXFRAME_LEN);

    if (frame != NULL)
    {
        // update Tx power in policy table
        tpc_update_frame_tx_power(vif, frame);

        // Get the buffer pointer
        buf = txl_buffer_payload_get(&frame->txdesc);

        // Fill-in the frame
        buf->fctl = MAC_FCTRL_AUTHENT;
        buf->durid = 0;
        buf->addr1 = sta->mac_addr;
        buf->addr2 = vif->mac_addr;
        buf->addr3 = sta->mac_addr;
        buf->seq = txl_get_seq_ctrl();

        // Set VIF and STA indexes
        frame->txdesc.host.vif_idx = vif->index;
        frame->txdesc.host.staid   = vif->u.sta.ap_id;

        length = MAC_SHORT_MAC_HDR_LEN;
        frame->txdesc.umac.head_len = 0;
        frame->txdesc.umac.tail_len = 0;

        if ((con_par->auth_type == MAC_AUTH_ALGO_SHARED) &&
            (auth_seq == MAC_AUTH_THIRD_SEQ))
        {
            // Need to encrypt the auth frame
            txu_cntrl_protect_mgmt_frame(&frame->txdesc, buf, MAC_SHORT_MAC_HDR_LEN);
            length += frame->txdesc.umac.head_len;
        }

        // Build the payload
        length += me_build_authenticate(CPU2HW(buf) + length, con_par->auth_type,
                                        auth_seq, MAC_ST_SUCCESSFUL, challenge);

        length += frame->txdesc.umac.tail_len;

        // Set callback for TX confirmation
        frame->cfm.cfm_func = sm_frame_tx_cfm_handler;
        frame->cfm.env = frame;

        // Update the length
        thd = &frame->txdesc.lmac.hw_desc->thd;
        thd->dataendptr = (uint32_t)thd->datastartptr + length - 1;
        thd->frmlen = length + MAC_FCS_LEN;

        // Push the frame for TX
        txl_frame_push(frame, AC_VO);

        // Start authentication timeout timer
        ke_timer_set(SM_RSP_TIMEOUT_IND, TASK_SM, DEFAULT_AUTHRSP_TIMEOUT);

        // Set the state to AUTH
        ke_state_set(TASK_SM, SM_AUTHENTICATING);
    }
    else
    {
        sm_connect_ind(MAC_ST_FAILURE);
    }
}

void sm_assoc_req_send(void)
{
    //The local variables
    struct sm_connect_req const *con_par = sm_env.connect_param;
    struct vif_info_tag *vif = &vif_info_tab[con_par->vif_idx];
    struct me_bss_info *bss = &vif->bss_info;
    struct mac_addr *ap_old_ptr = NULL;
    struct txl_frame_desc_tag *frame;
    struct mac_hdr *buf;
    struct tx_hd *thd;
    struct sta_info_tag *sta = &sta_info_tab[vif->u.sta.ap_id];
    uint32_t length;
    int txtype;

    // Send Association request through TX path
    #if (NX_P2P)
    if (vif->p2p)
    {
        txtype = TX_DEFAULT_5G;
    }
    else
    #endif //(NX_P2P)
    {
        // Chose the right rate according to the band
        txtype = (bss->chan.band == PHY_BAND_2G4) ? TX_DEFAULT_24G : TX_DEFAULT_5G;
    }

    // Allocate a frame descriptor from the TX path
    frame = txl_frame_get(txtype, NX_TXFRAME_LEN);
    if (frame != NULL)
    {
        struct sm_connect_ind *ind = sm_env.connect_ind;
        uint32_t ie_addr;
        uint16_t ie_len;

        // update Tx power in policy table
        tpc_update_frame_tx_power(vif, frame);

        // Get the buffer pointer
        buf = txl_buffer_payload_get(&frame->txdesc);

        // Fill-in the frame
        if (sm_env.ft_over_ds == 1)
        {
            PROF_FT_REASSOC_SET();
            buf->fctl = MAC_FCTRL_REASSOCREQ;
            ap_old_ptr = &sm_env.ft_old_bssid;
        }
        else
        {
            buf->fctl = MAC_FCTRL_ASSOCREQ;
        }
        buf->durid = 0;
        buf->addr1 = sta->mac_addr;
        buf->addr2 = vif->mac_addr;
        buf->addr3 = sta->mac_addr;
        buf->seq = txl_get_seq_ctrl();

        // Build the payload
        length = me_build_associate_req(CPU2HW(buf) + MAC_SHORT_MAC_HDR_LEN, bss,
                                        ap_old_ptr, vif->index, &ie_addr, &ie_len,
                                        con_par);

        // Set VIF and STA indexes
        frame->txdesc.host.vif_idx = vif->index;
        frame->txdesc.host.staid   = vif->u.sta.ap_id;

        // Set callback for TX confirmation
        frame->cfm.cfm_func = sm_frame_tx_cfm_handler;
        frame->cfm.env = frame;

        // Update the length
        thd = &frame->txdesc.lmac.hw_desc->thd;
        thd->dataendptr = thd->datastartptr + length + MAC_SHORT_MAC_HDR_LEN - 1;
        thd->frmlen = length + MAC_SHORT_MAC_HDR_LEN  + MAC_FCS_LEN;

        // Copy the AssocReq IEs into the indication message
        if (ie_len <= sizeof_b(ind->assoc_ie_buf))
        {
            co_copy8p(CPU2HW(ind->assoc_ie_buf), ie_addr, ie_len);
            ind->assoc_req_ie_len = ie_len;
        }
        else
        {
            ASSERT_WARN(0);
            ind->assoc_req_ie_len = 0;
        }

        // Push the frame for TX
        txl_frame_push(frame, AC_VO);

        // Run association timeout timer
        ke_timer_set(SM_RSP_TIMEOUT_IND, TASK_SM, DEFAULT_ASSOCRSP_TIMEOUT);

        // Move to state associating
        ke_state_set(TASK_SM, SM_ASSOCIATING);
    }
    else
    {
        sm_connect_ind(MAC_ST_FAILURE);
    }
}

void sm_assoc_done(uint16_t aid)
{
    //  Send Association done IND to LMAC
    struct mm_set_vif_state_req *req;
    struct sm_connect_req const *con_par = sm_env.connect_param;

    // Get a pointer to the kernel message
    req = KE_MSG_ALLOC(MM_SET_VIF_STATE_REQ, TASK_MM, TASK_SM, mm_set_vif_state_req);

    // Fill the message parameters
    req->aid = aid;
    req->active = true;
    req->inst_nbr = con_par->vif_idx;

    // Send the message to the task
    ke_msg_send(req);

    // Set state to back to IDLE
    ke_state_set(TASK_SM, SM_ACTIVATING);
}

void sm_auth_handler(struct rxu_mgt_ind const *param)
{
    uint16_t status, auth_type, auth_seq;
    uint32_t payload = CPU2HW(param->payload);

    // Stop the authentication timeout timer
    ke_timer_clear(SM_RSP_TIMEOUT_IND, TASK_SM);

    // Check authentication response (S,F)
    status = co_read16p(payload + MAC_AUTH_STATUS_OFT);

    // Check on the value of the status
    switch (status)
    {
        case MAC_ST_SUCCESSFUL:
            auth_type = co_read16p(payload + MAC_AUTH_ALGONBR_OFT);

            if (auth_type == MAC_AUTH_ALGO_OPEN)
            {
                // Send AIR_ASSOC_REQ to the AIR
                sm_assoc_req_send();
            }
            else if(auth_type == MAC_AUTH_ALGO_SHARED)
            {
                auth_seq = co_read16p(payload + MAC_AUTH_SEQNBR_OFT);

                if (auth_seq == MAC_AUTH_FOURTH_SEQ)
                {
                    // Authentication done Send AIR_ASSOC_REQ to the AIR
                    sm_assoc_req_send();
                }
                else if (auth_seq == MAC_AUTH_SECOND_SEQ)
                {
                    // Need to send challenge encrypted
                    sm_auth_send(MAC_AUTH_THIRD_SEQ,
                                 HW2CPU(payload + MAC_AUTH_CHALLENGE_OFT + MAC_CHALLENGE_TEXT_OFT));
                }
                else
                {
                    ASSERT_WARN(0);
                    sm_connect_ind(MAC_ST_FAILURE);
                }
            }
            break;

        default:
        {
            // No station available, terminate the connection procedure
            sm_connect_ind(status);
            break;
        }
    }
}

void sm_assoc_rsp_handler(struct rxu_mgt_ind const *param)
{
    uint16_t status;
    uint32_t payload = CPU2HW(param->payload);
    struct sm_connect_ind *ind = sm_env.connect_ind;

    // stop the assoc timeout timer
    ke_timer_clear(SM_RSP_TIMEOUT_IND, TASK_SM);

    // get association response status code
    status = co_read16p(payload + MAC_ASSO_RSP_STATUS_OFT);

    if (sm_env.ft_over_ds == 1)
    {
        PROF_FT_REASSOC_CLR();
    }

    // Copy the AssocRsp IEs into the indication message
    if (param->length >= MAC_ASSO_RSP_RATES_OFT)
    {
         uint32_t ie_addr = payload + MAC_ASSO_RSP_RATES_OFT;
         uint16_t ie_len = param->length - MAC_ASSO_RSP_RATES_OFT;
         if ((ie_len + ind->assoc_req_ie_len) > sizeof_b(ind->assoc_ie_buf))
         {
             ASSERT_WARN(0);
             ie_len = sizeof_b(ind->assoc_ie_buf) - ind->assoc_req_ie_len;
         }
         co_copy8p(CPU2HW(ind->assoc_ie_buf) + ind->assoc_req_ie_len, ie_addr, ie_len);
         ind->assoc_rsp_ie_len = ie_len;
    }
    else
    {
        ind->assoc_rsp_ie_len = 0;
    }

    switch (status)
    {
        case MAC_ST_SUCCESSFUL:
            // send MM_SET_ASSOCIATED_REQ to the LMAC
            sm_assoc_done(co_wtohs(co_read16p(payload + MAC_ASSO_RSP_AID_OFT)) & MAC_AID_MSK);
            break;

        default:
            // status is not successful
            sm_connect_ind(status);
            break;
    }
}

int sm_deauth_handler(struct rxu_mgt_ind const *param)
{
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];

    // Check if we are in a disconnection procedure
    if (ke_state_get(TASK_SM) == SM_DISCONNECTING)
        // We are already in a disconnection procedure, so save the message
        return KE_MSG_SAVED;

    // Check if we are in a connection procedure
    if (ke_state_get(TASK_SM) != SM_IDLE)
    {
        struct sm_connect_req const *con_par = sm_env.connect_param;

        // Check on which VIF we are currently in a connection procedure
        if (con_par->vif_idx != param->inst_nbr)
            // We are in a connection procedure on another VIF, so save the message
            return KE_MSG_SAVED;

        // status is not successful
        sm_connect_ind(MAC_ST_FAILURE);
    }
    else if (vif->active)
    {
        uint32_t payload = CPU2HW(param->payload);
        uint16_t reason = co_read16p(payload + MAC_DEAUTH_REASON_OFT);

        // Process to the disconnection
        ke_state_set(TASK_SM, SM_DISCONNECTING);
        sm_disconnect_process(vif, reason);
    }

    return KE_MSG_CONSUMED;
}

#if NX_MFP
void sm_sa_query_handler(struct rxu_mgt_ind const *param)
{
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];
    struct sta_info_tag *ap;
    struct txl_frame_desc_tag *frame;
    struct tx_hd *thd;
    struct mac_hdr *buf;
    uint32_t payload = CPU2HW(param->payload);
    uint8_t  action;
    uint16_t transaction_id, length;
    int txtype;

    // only process sa_query on STA interfaces
    if (!vif->active || vif->type != VIF_STA)
        return;

    // Ensure we recieved the frame form the AP
    if ((param->sta_idx == INVALID_STA_IDX) || (vif->u.sta.ap_id != param->sta_idx))
        return;

    // Check that connection with AP is finished
    ap = &sta_info_tab[param->sta_idx];
    if (ap->ctrl_port_state != PORT_OPEN)
        return;

    action = co_read8p(payload + MAC_SA_QUERY_ACTION_OFT);
    transaction_id = co_read16p(payload + MAC_SA_QUERY_TR_ID_OFT);

    if (action != MAC_SA_QUERY_REQUEST)
        return;

    // Chose the right rate according to the band
    txtype = (param->band == PHY_BAND_2G4) ? TX_DEFAULT_24G : TX_DEFAULT_5G;

    // Allocate frame to send SA Query response
    frame = txl_frame_get(txtype, NX_TXFRAME_LEN);
    if (frame == NULL)
        return;

    // update Tx power in policy table
    tpc_update_frame_tx_power(vif, frame);

    // Get the buffer pointer
    buf = txl_buffer_payload_get(&frame->txdesc);

    buf->fctl = MAC_FCTRL_ACTION;
    buf->durid = 0;
    buf->addr1 = ap->mac_addr;
    buf->addr2 = vif->mac_addr;
    buf->addr3 = ap->mac_addr;
    buf->seq = txl_get_seq_ctrl();

    // Set VIF and STA indexes
    frame->txdesc.host.vif_idx = param->inst_nbr;
    frame->txdesc.host.staid   = param->sta_idx;

    length = MAC_SHORT_MAC_HDR_LEN;

    frame->txdesc.umac.head_len = 0;
    frame->txdesc.umac.tail_len = 0;

    mfp_protect_mgmt_frame(&frame->txdesc, buf->fctl,
                           MAC_SA_QUERY_ACTION_CATEGORY);
    txu_cntrl_protect_mgmt_frame(&frame->txdesc, buf, MAC_SHORT_MAC_HDR_LEN);
    length += frame->txdesc.umac.head_len;

    payload = CPU2HW(buf);
    payload += length;

    co_write8p(payload + MAC_ACTION_CATEGORY_OFT, MAC_SA_QUERY_ACTION_CATEGORY);
    co_write8p(payload + MAC_SA_QUERY_ACTION_OFT, MAC_SA_QUERY_RESPONSE);
    co_write16p(payload + MAC_SA_QUERY_TR_ID_OFT, transaction_id);
    length += 4 + frame->txdesc.umac.tail_len;

    thd = &frame->txdesc.lmac.hw_desc->thd;
    thd->dataendptr = (uint32_t)thd->datastartptr + length - 1;
    thd->frmlen = length + MAC_FCS_LEN;

    // Push the frame for TX
    txl_frame_push(frame, AC_VO);
}
#endif // NX_MFP

void sm_external_auth_start(uint32_t akm)
{
    struct sm_connect_req const *con_par = sm_env.connect_param;
    struct vif_info_tag *vif = &vif_info_tab[con_par->vif_idx];
    struct me_bss_info *bss = &vif->bss_info;
    struct sm_external_auth_required_ind *ind;

    ASSERT_ERR(BSS_CAPA(bss, VALID));

    ind = KE_MSG_ALLOC(SM_EXTERNAL_AUTH_REQUIRED_IND, TASK_API, TASK_SM,
                       sm_external_auth_required_ind);

    ind->vif_idx = con_par->vif_idx;
    ind->ssid = bss->ssid;
    ind->bssid = bss->bssid;
    ind->akm = co_htonl(akm);

    // Wait until host complete the external authentication procedure
    ke_state_set(TASK_SM, SM_EXTERNAL_AUTHENTICATING);

    // Start timeout timer
    ke_timer_set(SM_RSP_TIMEOUT_IND, TASK_SM, SM_EXTERNAL_AUTH_TIMEOUT);

    ke_msg_send(ind);
}

void sm_external_auth_end(uint16_t status)
{
    ke_timer_clear(SM_RSP_TIMEOUT_IND, TASK_SM);

    if (status != MAC_ST_SUCCESSFUL)
    {
        sm_connect_ind(status);
        return;
    }

    // Start association if external authentication succeed
    sm_assoc_req_send();
}

bool sm_external_auth_in_progress(void)
{
    return (ke_state_get(TASK_SM) == SM_EXTERNAL_AUTHENTICATING);
}


int sm_get_rsnie_pmkid_count(uint32_t ies, uint16_t ies_len)
{
    uint32_t rsn_ie, rsn_ie_ptr;
    uint16_t cipher_len, akm_len = 0;
    uint8_t rsn_ie_len;
    rsn_ie = mac_ie_rsn_find(ies, ies_len, &rsn_ie_len);

    if (!rsn_ie || (rsn_ie_len < 30))
        return 0;

    rsn_ie_ptr = rsn_ie + MAC_RSNIE_PAIRWISE_CIPHER_SUITE_CNT_OFT;
    rsn_ie_len -= MAC_RSNIE_PAIRWISE_CIPHER_SUITE_CNT_OFT;

    cipher_len = co_wtohs(co_read16p(rsn_ie_ptr)) * MAC_RSNIE_PAIRWISE_CIPHER_SIZE;
    rsn_ie_ptr += 2 + cipher_len;
    rsn_ie_len -= 2 + cipher_len;

    if (rsn_ie_len < 22)
        return 0;

    akm_len = co_wtohs(co_read16p(rsn_ie_ptr)) * MAC_RSNIE_KEY_MANAGEMENT_SIZE;
    rsn_ie_ptr += 2 + akm_len;
    rsn_ie_len -= 2 + akm_len;

    if (rsn_ie_len < 20)
        return 0;

    rsn_ie_ptr += 2;
    return co_wtohs(co_read16p(rsn_ie_ptr));
}


/// @} //end of group

