/**
 ****************************************************************************************
 *
 * @file sm_task.c
 *
 * @brief The UMAC's STA Manager (SM) module implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

#include "sm_task.h"
#include "co_endian.h"
#include "scanu_task.h"
#include "scanu.h"
#include "sm.h"
#include "me_utils.h"
#include "mac_frame.h"

#include "ke_timer.h"
#include "mac.h"
#include "mm_task.h"
#include "me.h"
#include "me_mgmtframe.h"
#include "mm.h"
#include "vif_mgmt.h"
#include "sta_mgmt.h"
#include "rxu_task.h"
#include "ps.h"

/** @addtogroup TASK_SM
* @{
*/


/**
 ****************************************************************************************
 * @brief @ref SM_CONNECT_REQ message handler.
 * This message Requests the UMAC to join and associate to a specific AP.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
sm_connect_req_handler(ke_msg_id_t const msgid,
                       struct sm_connect_req const *param,
                       ke_task_id_t const dest_id,
                       ke_task_id_t const src_id)
{
    struct mac_addr const *bssid = NULL;
    struct mac_chan_def const *chan = NULL;
    uint8_t status = CO_BUSY;
    int msg_status = KE_MSG_CONSUMED;
    struct sm_connect_cfm *cfm;

    do
    {
        struct vif_info_tag *vif = &vif_info_tab[param->vif_idx];

        if (ke_state_get(TASK_SM) == SM_DISCONNECTING)
            return (KE_MSG_SAVED);

        // Allocate the confirmation message
        cfm = KE_MSG_ALLOC(SM_CONNECT_CFM, src_id, dest_id, sm_connect_cfm);

        // Check if we are not busy
        if (ke_state_get(TASK_SM) != SM_IDLE)
        {
            status = CO_BUSY;
            break;
        }

        // Check if we are not already connected
        if (((vif->type != VIF_STA) || (vif->active)) && (param->auth_type != MAC_AUTH_ALGO_FT))
        {
            status = CO_OP_IN_PROGRESS;
            break;
        }

        if (param->auth_type != MAC_AUTH_ALGO_FT)
        {
            // Sanity check - If we are not associated, we should not have a AP STA index
            // registered neither a channel context
            ASSERT_ERR(vif->u.sta.ap_id == INVALID_STA_IDX);
            ASSERT_ERR(vif->chan_ctxt == NULL);
        }

        if ((param->auth_type == MAC_AUTH_ALGO_SAE) &&
            (sm_get_rsnie_pmkid_count(CPU2HW(param->ie_buf), param->ie_len) > 0))
        {
            uint8_t *update_auth_type = (uint8_t *)&param->auth_type;

            // If at least one PMKID is present SAE can be skipped to test PMKID.
            // If PMKID is not accepted by AP, it is assumed that host will try
            // to reconnect without PMKID
            *update_auth_type = MAC_AUTH_ALGO_OPEN;
        }

        // Save the parameters
        sm_env.connect_param = param;

        // Allocate the kernel message for the connection status forwarding
        sm_env.connect_ind = KE_MSG_ALLOC(SM_CONNECT_IND, src_id, dest_id, sm_connect_ind);

        // Reset ft_over_ds flag
        sm_env.ft_over_ds = 0;

        // Check if it's a FT over DS request
        if (param->auth_type == MAC_AUTH_ALGO_FT)
        {
            PROF_FT_OVER_DS_REQ_SET();
            sm_env.ft_over_ds = 1;
            memcpy(&sm_env.ft_old_bssid, &vif_info_tab[param->vif_idx].bssid, sizeof(sm_env.ft_old_bssid));
            ke_state_set(TASK_SM, SM_DISCONNECTING);
            sm_disconnect_process(&vif_info_tab[param->vif_idx], 0);
            status = CO_OK;
            msg_status = KE_MSG_NO_FREE;
            PROF_FT_OVER_DS_REQ_CLR();
            break;
        }

        // Get the BSSID and Channel from the parameter and/or the scan results
        sm_get_bss_params(&bssid, &chan);

        // Check if we have enough information to launch the join or if we have to scan
        if (bssid && chan)
        {
            sm_join_bss(bssid, chan, false);
        }
        else
        {
            sm_scan_bss(bssid, chan);
        }

        // We will now proceed to the connection, so the status is OK
        status = CO_OK;
        msg_status = KE_MSG_NO_FREE;
    } while(0);

    // Send the confirmation
    cfm->status = status;
    ke_msg_send(cfm);

    return (msg_status);
}

/**
 ****************************************************************************************
 * @brief @ref SM_DISCONNECT_REQ message handler.
 * This message Requests the UMAC to disconnect from a specific AP
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
sm_disconnect_req_handler(ke_msg_id_t const msgid,
                          struct sm_disconnect_req const *param,
                          ke_task_id_t const dest_id,
                          ke_task_id_t const src_id)
{
    // Check if we are in an association process
    if ((ke_state_get(TASK_SM) != SM_IDLE))
        return (KE_MSG_SAVED);

    // Check on the value of the BSS type
    sm_disconnect(param->vif_idx, param->reason_code);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief @ref MM_CONNECTION_LOSS_IND message handler.
 * This message indicates that network connection with the associated AP has been lost.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_connection_loss_ind_handler(ke_msg_id_t const msgid,
                               struct mm_connection_loss_ind const *param,
                               ke_task_id_t const dest_id,
                               ke_task_id_t const src_id)
{
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];

    // Check if we are in a disconnection procedure
    if (ke_state_get(TASK_SM) != SM_IDLE)
        // We are already in a procedure, so save the message
        return KE_MSG_SAVED;

    if ((vif->type != VIF_STA) || (!vif->active))
        return KE_MSG_CONSUMED;

    ke_state_set(TASK_SM, SM_DISCONNECTING);

    // Proceed to the disconnection
    sm_disconnect_process(vif, MAC_RS_UNSPECIFIED);

    return (KE_MSG_CONSUMED);
}


/**
 ****************************************************************************************
 * @brief @ref SM_RSP_TIMEOUT_IND message handler.
 * This message indicates that the programmed AUTH or ASSOC_REQ frame were not successfully
 * transmitted.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
sm_rsp_timeout_ind_handler(ke_msg_id_t const msgid,
                           void const *param,
                           ke_task_id_t const dest_id,
                           ke_task_id_t const src_id)

{
    // Check if we are in a state where such timeout could occur
    if ((ke_state_get(TASK_SM) != SM_AUTHENTICATING) &&
        (ke_state_get(TASK_SM) != SM_EXTERNAL_AUTHENTICATING) &&
        (ke_state_get(TASK_SM) != SM_ASSOCIATING) )
        return (KE_MSG_CONSUMED);

    // Status is not successful
    sm_connect_ind(MAC_ST_FAILURE);

    return (KE_MSG_CONSUMED);
}


/**
 ****************************************************************************************
 * @brief @ref SCANU_START_CFM message handler.
 * This function handles scan confirmation from the scan module.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
scanu_start_cfm_handler(ke_msg_id_t const msgid,
                        void const *param,
                        ke_task_id_t const dest_id,
                        ke_task_id_t const src_id)
{
    struct mac_addr const *bssid = NULL;
    struct mac_chan_def const *chan = NULL;

    // Sanity check - This message can be received only in the SCANNING or JOINING states
    ASSERT_ERR(ke_state_get(TASK_SM) == SM_SCANNING);

    // Get the BSSID and Channel from the parameter and/or the scan results
    sm_get_bss_params(&bssid, &chan);

    // Check if we now have enough information to launch the join procedure
    if (bssid && chan)
    {
        sm_join_bss(bssid, chan, false);
    }
    else
    {
        // We did not find our BSS, so terminate the connection procedure
        sm_connect_ind(MAC_ST_FAILURE);
    }

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief @ref SCANU_JOIN_CFM message handler.
 * This function handles joining confirmation from the scan module.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
scanu_join_cfm_handler(ke_msg_id_t const msgid,
                        void const *param,
                        ke_task_id_t const dest_id,
                        ke_task_id_t const src_id)
{
    struct sm_connect_req const *con_par = sm_env.connect_param;
    struct vif_info_tag *vif;
    struct me_bss_info *bss;

    // Sanity check - This message can be received only in the SCANNING or JOINING states
    ASSERT_ERR(ke_state_get(TASK_SM) == SM_JOINING);

    // Get parameters
    vif = &vif_info_tab[con_par->vif_idx];
    bss = &vif->bss_info;

    // Check if the joining procedure was successful
    if (BSS_CAPA(bss, VALID))
    {
        // Channel Context Index
        uint8_t chan_idx;

        // Create a channel context for the connection and check if the channel context was correctly added
        if (sm_add_chan_ctx(&chan_idx) == CO_OK)
        {
            struct mm_sta_add_req *req = KE_MSG_ALLOC(MM_STA_ADD_REQ, TASK_MM,
                                                      TASK_SM, mm_sta_add_req);
            uint32_t paid_gid = 0;

            // Link the VIF to the channel context
            chan_ctxt_link(con_par->vif_idx, chan_idx);

            // Fill in the MM_STA_ADD_REQ parameters
            req->inst_nbr = con_par->vif_idx;
            req->mac_addr = bss->bssid;
            req->tdls_sta = false;
            req->bssid_index = bss->bssid_index;
            req->max_bssid_ind = bss->max_bssid_ind;
            if (BSS_CAPA(bss, HT))
            {
                struct mac_htcapability *ht_cap = &bss->ht_cap;
                struct mac_vhtcapability *vht_cap = NULL;
                struct mac_hecapability *he_cap = NULL;

                #if NX_VHT
                if (BSS_CAPA(bss, VHT))
                {
                    vht_cap = &bss->vht_cap;
                    paid_gid = mac_paid_gid_sta_compute(&bss->bssid);
                }
                #endif
                #if NX_HE
                if (BSS_CAPA(bss, HE))
                {
                    he_cap = &bss->he_cap;
                }
                #endif
                me_get_ampdu_params(ht_cap, vht_cap, he_cap, &req->ampdu_size_max_ht,
                                    &req->ampdu_size_max_vht, &req->ampdu_size_max_he,
                                    &req->ampdu_spacing_min);
            }
            req->paid_gid = paid_gid;

            // Send the message
            ke_msg_send(req);

            // We are now waiting for the station addition confirmation
            ke_state_set(TASK_SM, SM_STA_ADDING);
        }
        else
        {
            // No channel context available, terminate the connection procedure
            sm_connect_ind(MAC_ST_FAILURE);
        }

        // Save the VIF flags
        vif->flags = con_par->flags;

        // Check if we need to disable the HT/VHT/HE feature
        if (con_par->flags & DISABLE_HT)
        {
            BSS_CAPA_CLR(bss, HT);
            BSS_CAPA_CLR(bss, VHT);
            BSS_CAPA_CLR(bss, HE);
        }
    }
    else
    {
        // If confirmed joined procedure used an active scan, try again with a passive scan
        if (sm_env.join_passive)
        {
            struct mac_addr const *bssid = NULL;
            struct mac_chan_def const *chan = NULL;
            sm_get_bss_params(&bssid, &chan);

            #if (NX_ANT_DIV)
            // Switch antenna paths and try to join again in case of join not completed successfully
            if (!scanu_env.join_status)
            {
                // Request an antenna switch
                ke_msg_send_basic(MM_SWITCH_ANTENNA_REQ, TASK_MM, TASK_SM);
            }
            #endif //(NX_ANT_DIV)
            sm_join_bss(bssid, chan, true);
        }
        else
        {
            // The join was not successful, so terminate the connection procedure
            sm_connect_ind(MAC_ST_FAILURE);
        }
    }

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief @ref MM_STA_ADD_CFM message handler.
 * This function handles the STA addition confirmation message.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_sta_add_cfm_handler(ke_msg_id_t const msgid,
                       struct mm_sta_add_cfm const *param,
                       ke_task_id_t const dest_id,
                       ke_task_id_t const src_id)
{
    // Sanity check - This message can be received only in the STA ADDING state
    ASSERT_ERR(ke_state_get(TASK_SM) == SM_STA_ADDING);

    // Check if the STA was correctly added
    if (param->status == CO_OK)
    {
        struct sta_info_tag *sta = &sta_info_tab[param->sta_idx];
        struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];
        struct sta_capa_info *info = &sta->info;
        struct me_bss_info *bss = &vif->bss_info;

        // Initialize the capabilities of the new STA
        info->rate_set = bss->rate_set;
        if (BSS_CAPA(bss, QOS))
        {
            STA_CAPA_SET(sta, QOS);
        }
        if (BSS_CAPA(bss, HT))
        {
            STA_CAPA_SET(sta, HT);
            info->ht_cap = bss->ht_cap;
            #if NX_HE
            if (BSS_CAPA(bss, HE))
            {
                STA_CAPA_SET(sta, HE);
                info->he_cap = bss->he_cap;
            }
            #endif
            #if NX_VHT
            if (BSS_CAPA(bss, VHT))
            {
                STA_CAPA_SET(sta, VHT);
                info->vht_cap = bss->vht_cap;
            }
            #endif

            // Configure some HT and VHT parameters of the peer STA
            me_set_sta_ht_vht_param(sta, bss);
        }

        // Set the BSS parameters to LMAC
        sm_set_bss_param();

        // Initialize the RC for this STA
        me_init_rate(sta);
    }
    else
    {
        // No station available, terminate the connection procedure
        sm_connect_ind(MAC_ST_FAILURE);
    }
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief @ref ME_SET_PS_DISABLE_CFM message handler.
 * This function starts the procedure of BSS parameter setting.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_set_ps_disable_cfm_handler(ke_msg_id_t const msgid,
                              void const *param,
                              ke_task_id_t const dest_id,
                              ke_task_id_t const src_id)
{
    // Sanity check - This message can be received only in the STA ADDING state
    ASSERT_ERR((ke_state_get(TASK_SM) == SM_BSS_PARAM_SETTING) ||
               (ke_state_get(TASK_SM) == SM_IDLE) ||
               (ke_state_get(TASK_SM) == SM_DISCONNECTING));

    // Check if we were disabling the PS prior to the parameter setting
    if ((ke_state_get(TASK_SM) == SM_BSS_PARAM_SETTING) ||
        (ke_state_get(TASK_SM) == SM_DISCONNECTING))
    {
        // Send the next BSS parameter configuration message
        sm_bss_config_send();
    }

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief BSS parameter setting message handler.
 * This function handles all the confirmation of the BSS parameter setting requests sent
 * during the connection process. It pushes the next parameter setting request.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_bss_param_setting_handler(ke_msg_id_t const msgid,
                             void const *param,
                             ke_task_id_t const dest_id,
                             ke_task_id_t const src_id)
{
    // Sanity check - This message can be received only in the CHAN CTXT ADDING state
    ASSERT_ERR(ke_state_get(TASK_SM) == SM_BSS_PARAM_SETTING);

    // Send the next BSS parameter configuration message
    sm_bss_config_send();

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief @ref ME_SET_ACTIVE_CFM message handler.
 * This function handles the confirmation of move to active state that allows starting
 * the AUTH/ASSOC_REQ exchanges.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_set_active_cfm_handler(ke_msg_id_t const msgid,
                          void const *param,
                          ke_task_id_t const dest_id,
                          ke_task_id_t const src_id)
{
    // Sanity check - This message can be received only in the STA ADDING state
    ASSERT_ERR((ke_state_get(TASK_SM) == SM_BSS_PARAM_SETTING) ||
               (ke_state_get(TASK_SM) == SM_DISCONNECTING));

    // Check if we went to IDLE due to the disconnection procedure
    if (ke_state_get(TASK_SM) == SM_DISCONNECTING)
    {
        struct vif_info_tag *vif = sm_env.vif_disconnect;

        // Invalidate the BSS information structure
        BSS_CAPA_RESET(&vif->bss_info);

        if (sm_env.host_disconnect)
        {
            ke_msg_send_basic(SM_DISCONNECT_CFM, TASK_API, TASK_SM);
            sm_env.host_disconnect = false;
        }
        // Send the end of disconnection procedure indication to the host
        ke_msg_send(ke_msg2param(sm_env.disconnect_ind));
        sm_env.disconnect_ind = NULL;

        if (sm_env.ft_over_ds == 1)
        {
            struct mac_addr const *bssid = NULL;
            struct mac_chan_def const *chan = NULL;
            sm_get_bss_params(&bssid, &chan);
            sm_join_bss(bssid, chan, false);
        }
        else
        {
            ke_state_set(TASK_SM, SM_IDLE);
        }
    }
    else
    {

        if (sm_env.ft_over_ds == 1)
        {
            // Send REASSOC frame
            sm_assoc_req_send();
        }
        else
        {
            struct sm_connect_req const *con_par = sm_env.connect_param;
            switch(con_par->auth_type)
            {
                case MAC_AUTH_ALGO_OPEN:
                case MAC_AUTH_ALGO_SHARED:
                case MAC_AUTH_ALGO_FT:
                    sm_auth_send(MAC_AUTH_FIRST_SEQ, NULL);
                    break;
                case MAC_AUTH_ALGO_SAE:
                    sm_external_auth_start(MAC_RSNIE_AKM_SAE);
                    break;
                default:
                    sm_connect_ind(MAC_ST_FAILURE);
            }

        }
    }

    return (KE_MSG_CONSUMED);
}


/**
 ****************************************************************************************
 * @brief @ref MM_SET_VIF_STATE_CFM message handler.
 * This function is called once the VIF has been put to active state at the end of the
 * association procedure, or to inactive state upon disconnection.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_vif_state_cfm_handler(ke_msg_id_t const msgid,
                             void const *param,
                             ke_task_id_t const dest_id,
                             ke_task_id_t const src_id)
{
    // Check if we are in the connection process
    if (ke_state_get(TASK_SM) == SM_ACTIVATING)
    {
        struct sm_connect_req const *con_par = sm_env.connect_param;
        struct vif_info_tag *vif = &vif_info_tab[con_par->vif_idx];
        struct sta_info_tag *sta = &sta_info_tab[vif->u.sta.ap_id];

        #if NX_POWERSAVE
        struct mm_set_ps_options_req *req;

        // Get a pointer to the kernel message
        req = KE_MSG_ALLOC(MM_SET_PS_OPTIONS_REQ, TASK_MM, TASK_SM, mm_set_ps_options_req);

        // Fill the message parameters
        req->dont_listen_bc_mc = con_par->dont_wait_bcmc;
        req->listen_interval = con_par->listen_interval;
        req->vif_index = con_par->vif_idx;

        // Set the PS options for this VIF
        ke_msg_send(req);
        #endif

        // Open the control port state of the STA
        sta->ctrl_port_state = (vif->flags & CONTROL_PORT_HOST)?PORT_CONTROLED:PORT_OPEN;
        sta->ctrl_port_ethertype = co_ntohs(con_par->ctrl_port_ethertype);

        // If no EAP frame has to be sent, reenable the PS mode now that the association is complete
        if (sta->ctrl_port_state == PORT_OPEN)
        {
            struct me_set_ps_disable_req *ps = KE_MSG_ALLOC(ME_SET_PS_DISABLE_REQ, TASK_ME, TASK_SM,
                                                            me_set_ps_disable_req);

            ps->ps_disable = false;
            ps->vif_idx = con_par->vif_idx;

            ke_msg_send(ps);
        }

        // Association can now be considered as complete
        sm_connect_ind(MAC_ST_SUCCESSFUL);
    }
    // Check if we were setting the VIF state due to a disconnection
    else if (ke_state_get(TASK_SM) == SM_DISCONNECTING)
    {
        // Send the next BSS disconnection message
        sm_bss_config_send();
    }

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief @ref MM_STA_DEL_CFM message handler.
 * This function is called once the STA element associated to a VIF has been deleted.
 *
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_sta_del_cfm_handler(ke_msg_id_t const msgid,
                       struct mm_sta_del_cfm const *param,
                       ke_task_id_t const dest_id,
                       ke_task_id_t const src_id)
{
    // Sanity check - This message can be received during the disconnection procedure only
    ASSERT_ERR(ke_state_get(TASK_SM) == SM_DISCONNECTING);

    // Send the next BSS disconnection message
    sm_bss_config_send();

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief @ref MM_CHAN_CTXT_UNLINK_CFM message handler.
 * This function is called once the channel context associated to a VIF has been unlinked.
 *
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_chan_ctxt_unlink_cfm_handler(ke_msg_id_t const msgid,
                                void const *param,
                                ke_task_id_t const dest_id,
                                ke_task_id_t const src_id)
{
    // Sanity check - This message can be received during the disconnection procedure only
    ASSERT_ERR(ke_state_get(TASK_SM) == SM_DISCONNECTING);

    // Send the next BSS disconnection message
    sm_bss_config_send();

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief @ref RXU_MGT_IND message handler.
 * This function handles the reception of the AUTH/(RE)ASSOCRSP/DEAUTH/DISASSOC frames and
 * dispatch them to the correct function.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
rxu_mgt_ind_handler(ke_msg_id_t const msgid,
                    struct rxu_mgt_ind const *param,
                    ke_task_id_t const dest_id,
                    ke_task_id_t const src_id)
{
    uint16_t fctl = param->framectrl & MAC_FCTRL_TYPESUBTYPE_MASK;
    int msg_status = KE_MSG_CONSUMED;

    // Call correct frame handler
    if ((fctl == MAC_FCTRL_AUTHENT) && (ke_state_get(TASK_SM) == SM_AUTHENTICATING))
    {
        sm_auth_handler(param);
    }
    else if ((fctl == MAC_FCTRL_ASSOCRSP) && (ke_state_get(TASK_SM) == SM_ASSOCIATING))
    {
        sm_assoc_rsp_handler(param);
    }
    else if ((fctl == MAC_FCTRL_REASSOCRSP) && (ke_state_get(TASK_SM) == SM_ASSOCIATING))
    {
        sm_assoc_rsp_handler(param);
    }
    else if ((fctl == MAC_FCTRL_DEAUTHENT) || (fctl == MAC_FCTRL_DISASSOC))
    {
        msg_status = sm_deauth_handler(param);
    }
    else if (fctl == MAC_FCTRL_ACTION)
    {
        #if NX_MFP
        uint32_t payload = CPU2HW(param->payload);
        uint8_t action = co_read8p(payload + MAC_ACTION_CATEGORY_OFT);

        if (action == MAC_SA_QUERY_ACTION_CATEGORY)
            sm_sa_query_handler(param);
        #endif // NX_MFP
    }

    return (msg_status);
}

/**
 ****************************************************************************************
 * @brief @ref SM_EXTERNAL_AUTH_REQUIRED_RSP message handler.
 * This function handles the reception of the SM_EXTERNAL_AUTH_REQUIRED_RSP message sent
 * by the host once the external authentication is completed
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
sm_external_auth_required_rsp_handler(ke_msg_id_t const msgid,
                                      struct sm_external_auth_required_rsp const *param,
                                      ke_task_id_t const dest_id,
                                      ke_task_id_t const src_id)
{
    // Sanity check - This message can be received during external authentication only
    ASSERT_ERR(ke_state_get(TASK_SM) == SM_EXTERNAL_AUTHENTICATING);

    sm_external_auth_end(param->status);

    return (KE_MSG_CONSUMED);
}

/// DEFAULT handler definition.
const struct ke_msg_handler sm_default_state[] =
{
    {SM_CONNECT_REQ, (ke_msg_func_t)sm_connect_req_handler},
    {SM_DISCONNECT_REQ, (ke_msg_func_t)sm_disconnect_req_handler},
    {SCANU_START_CFM, (ke_msg_func_t)scanu_start_cfm_handler},
    {SCANU_JOIN_CFM, (ke_msg_func_t)scanu_join_cfm_handler},
    {SM_RSP_TIMEOUT_IND, (ke_msg_func_t)sm_rsp_timeout_ind_handler},
    {MM_CONNECTION_LOSS_IND, (ke_msg_func_t)mm_connection_loss_ind_handler},
    {MM_STA_ADD_CFM, (ke_msg_func_t)mm_sta_add_cfm_handler},
    {ME_SET_ACTIVE_CFM, (ke_msg_func_t)me_set_active_cfm_handler},
    {MM_SET_BSSID_CFM, mm_bss_param_setting_handler},
    {MM_SET_BASIC_RATES_CFM, mm_bss_param_setting_handler},
    {MM_SET_BEACON_INT_CFM, mm_bss_param_setting_handler},
    {MM_SET_BSS_COLOR_CFM, mm_bss_param_setting_handler},
    {MM_SET_EDCA_CFM, mm_bss_param_setting_handler},
    {MM_SET_VIF_STATE_CFM, mm_set_vif_state_cfm_handler},
    {ME_SET_PS_DISABLE_CFM, (ke_msg_func_t)me_set_ps_disable_cfm_handler},
    #if NX_POWERSAVE
    {MM_SET_PS_OPTIONS_CFM, ke_msg_discard},
    #endif
    {MM_STA_DEL_CFM, (ke_msg_func_t)mm_sta_del_cfm_handler},
    {MM_CHAN_CTXT_UNLINK_CFM, (ke_msg_func_t)mm_chan_ctxt_unlink_cfm_handler},
    {RXU_MGT_IND, (ke_msg_func_t)rxu_mgt_ind_handler},
    {SM_EXTERNAL_AUTH_REQUIRED_RSP, (ke_msg_func_t)sm_external_auth_required_rsp_handler}
};

/// Specifies the message handlers that are common to all states.
const struct ke_state_handler sm_default_handler =
KE_STATE_HANDLER(sm_default_state);

/// Defines the placeholder for the states of all the task instances.
ke_state_t sm_state[SM_IDX_MAX];

/// @} //end of group
