/**
 ****************************************************************************************
 *
 * @file txu_cntrl.c
 *
 * @brief UMAC TX Path implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/** @addtogroup UMACTX
 * @{
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
// for mode
#include "txl_cfm.h"
#include "txu_cntrl.h"
#include "ke_task.h"
#include "mac.h"
#include "mac_frame.h"
#include "sta_mgmt.h"
#include "llc.h"
#include "co_utils.h"
#include "co_endian.h"
#include "vif_mgmt.h"
#include "me_utils.h"
#include "bam.h"
#include "mm.h"
#include "tpc.h"
#if NX_MFP
#include "mfp.h"
#endif
#if (RW_MESH_EN)
#include "mesh.h"
#include "mesh_hwmp.h"
#include "mesh_ps.h"
#endif //(RW_MESH_EN)

/*
 * PRIVATE FUNCTION DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Perform the logical port filtering based on the STA index and the EtherType of
 * the frame.
 * The function may change the connection flags depending on the EtherType.
 *
 * @param[in]       sta_idx     Index of the station the frame has to be transmitted to
 * @param[in]       eth_type    EtherType of the frame
 * @param[in,out]   flags       Transmission flags
 *
 * @return true if the packet can be transmitted, false if it has to be discarded
 ****************************************************************************************
 */
static bool txu_cntrl_logic_port_filter(uint8_t sta_idx, uint16_t eth_type,
                                        uint16_t *flags)
{
    uint8_t port_state = sta_mgmt_get_port_state(sta_idx);
    uint16_t port_proto = sta_mgmt_get_port_ethertype(sta_idx);
    bool is_port_proto = false;

    if (eth_type == port_proto)
    {
        is_port_proto = true;
        /* Don't use 4ADDR mode for connection frames */
        *flags &= ~(TXU_CNTRL_USE_4ADDR);
    }

    /* Accept all packets if the port is OPEN */
    if (port_state == PORT_OPEN)
        return true;

    /* Accept only port control packets if port is CONTROLED */
    if ((port_state == PORT_CONTROLED) && (is_port_proto))
    {
        return true;
    }

    /* Discard all packets in all other cases */
    return false;
}

/**
 ****************************************************************************************
 * @brief Compute the security header and trailer lengths depending on the type of
 * security to be used.
 * The function also assigns the PN to the frame whenever required.
 *
 * @param[in]       txdesc    TX descriptor attached to the frame to be transmitted
 * @param[out]      tail_len  Length of the security trailer
 *
 * @return The length of the security header
 ****************************************************************************************
 */
static int txu_cntrl_sechdr_len_compute(struct txdesc *txdesc, int *tail_len)
{
    struct hostdesc *host = &txdesc->host;
    struct sta_info_tag *sta = &sta_info_tab[host->staid];
    struct key_info_tag *key = *sta->sta_sec_info.cur_key;
    struct vif_info_tag *vif = &vif_info_tab[host->vif_idx];
    int head_len = 0;

    // By default we consider that there won't be any tail
    *tail_len = 0;

    // Check if we have a valid key for this STA
    if (!key || (vif->flags & CONTROL_PORT_NO_ENC &&
                 co_ntohs(host->ethertype) == sta_mgmt_get_port_ethertype(host->staid)))
        return 0;

    // Check which type of security is used with this STA
    switch(key->cipher)
    {
        case MAC_CIPHER_WEP40:
        case MAC_CIPHER_WEP104:
            head_len = IV_LEN;
            *tail_len = ICV_LEN;
            // Check if we need to get a fresh PN
            if (!(host->flags & TXU_CNTRL_RETRY))
            {
                key->tx_pn++;
                memcpy(host->pn, &key->tx_pn, 2 * sizeof(host->pn[0]));
            }
            break;
        case MAC_CIPHER_TKIP:
            head_len = IV_LEN + EIV_LEN;
            *tail_len = MIC_LEN + ICV_LEN;
            // Check if we need to get a fresh PN
            if (!(host->flags & TXU_CNTRL_RETRY))
            {
                key->tx_pn++;
                memcpy(host->pn, &key->tx_pn, 3 * sizeof(host->pn[0]));
            }
            break;
        case MAC_CIPHER_CCMP:
            head_len = IV_LEN + EIV_LEN;
            *tail_len = MIC_LEN;
            // Check if we need to get a fresh PN
            if (!(host->flags & TXU_CNTRL_RETRY))
            {
                key->tx_pn++;
                memcpy(host->pn, &key->tx_pn, 3 * sizeof(host->pn[0]));
            }
            break;
        #if RW_WAPI_EN
        case MAC_CIPHER_WPI_SMS4:
            head_len = WPI_IV_LEN;
            *tail_len = WPI_MIC_LEN;
            if (!(host->flags & TXU_CNTRL_RETRY))
            {
                if (key->hw_key_idx < MM_SEC_DEFAULT_KEY_COUNT) {
                    // broadcast key
                    key->tx_pn++;
                } else {
                    // unicast key
                    key->tx_pn+=2;
                }
                memcpy(host->pn, &key->tx_pn, 4 * sizeof(host->pn[0]));
            }
            break;
        #endif
        default:
            ASSERT_ERR(0);
            break;
    }

    return (head_len);
}

/**
 ****************************************************************************************
 * @brief Compute the size of the different headers and trailers of the frame.
 * The function also assigns the SN to the frame.
 *
 * @param[in]       txdesc    TX descriptor attached to the frame to be transmitted
 ****************************************************************************************
 */
static void txu_cntrl_umacdesc_prep(struct txdesc *txdesc)
{
    struct hostdesc *host = &txdesc->host;
    struct umacdesc *umac = &txdesc->umac;
    #if NX_HE
    struct sta_info_tag *sta = &sta_info_tab[host->staid];
    #endif
    #if RW_MESH_EN || NX_HE
    // Get VIF Information
    struct vif_info_tag *vif = &vif_info_tab[host->vif_idx];
    #endif //(RW_MESH_EN)
    int head_len, tail_len, hdr_len_802_2 = 0;
    uint16_t payl_len;

    // Check if the STA has QoS enabled
    if (host->tid != 0xFF)
    {
        // Packet shall be of type QoS
        head_len = MAC_SHORT_QOS_MAC_HDR_LEN;
        // Check if we need to get a fresh SN
        if (!(host->flags & TXU_CNTRL_RETRY))
            host->sn = sta_mgmt_get_tx_ssn_and_inc(host->staid, host->tid);

        #if (RW_MESH_EN)
        if (vif->type == VIF_MESH_POINT)
        {
            // Check if Mesh Control field is present
            head_len += mesh_tx_data_prepare(vif, host, umac);
        }
        #endif //(RW_MESH_EN)
    }
    else
    {
        head_len = MAC_SHORT_MAC_HDR_LEN;
    }

    if (host->flags & TXU_CNTRL_USE_4ADDR)
    {
        head_len += (MAC_LONG_MAC_HDR_LEN - MAC_SHORT_MAC_HDR_LEN);
    }

    #if NX_HE
    // Frames sent to a HE AP will include a +HTC HE Variant. If sent as HE SU, it will be
    // set as the buffer status (BSR). If sent as HE TB, it will be set as the Uplink
    // Power Headroom (UPH)
    if ((vif->type == VIF_STA) && (STA_CAPA(sta, HE)))
        head_len += MAC_HTC_LEN;
    #endif

    umac->machead_len = head_len;

    // Update the header length
    head_len += txu_cntrl_sechdr_len_compute(txdesc, &tail_len);

    #if (RW_MESH_EN)
    if (!(host->flags & TXU_CNTRL_MESH_FWD))
    #endif //(RW_MESH_EN)
    {
        // Check if the Ethernet frame is of Type II or SNAP
        if (co_ntohs(host->ethertype) > LLC_FRAMELENGTH_MAXVALUE)
        {
            // Update header length - LLC/SNAP will be added after buffer allocation
            head_len += LLC_802_2_HDR_LEN;
            hdr_len_802_2 = LLC_802_2_HDR_LEN;
        }
    }

    // Compute payload length
    #if NX_AMSDU_TX
    if (host->flags & TXU_CNTRL_AMSDU)
    {
        head_len += sizeof_b(struct amsdu_hdr);
    }

    payl_len = 0;
    for (int i = 0; i < host->packet_cnt; i++)
    {
        payl_len += host->packet_len[i];
    }
    #else
    payl_len = host->packet_len;
    #endif

    // Update the different lengths
    umac->payl_len = payl_len;
    umac->head_len = head_len;
    umac->tail_len = tail_len;
    umac->hdr_len_802_2 = hdr_len_802_2;

    #if NX_AMPDU_TX
    umac->phy_flags = 0;
    umac->flags = 0;
    #endif
}

/**
 ****************************************************************************************
 * @brief Compute the header and trailer lengths for a management frame to be transmitted.
 * The function also assigns the PN to the frame whenever required.
 *
 * @param[in] txdesc       TX descriptor attached to the frame to be transmitted
 * @param[in] vif          Pointer to the VIF
 ****************************************************************************************
 */
static void txu_cntrl_umacdesc_mgmt_prep(struct txdesc *txdesc, struct vif_info_tag *vif)
{
    struct hostdesc *host = &txdesc->host;
    struct umacdesc *umac = &txdesc->umac;
    int head_len = 0, tail_len = 0;
    uint8_t pwr_idx, i;

    // Select the policy table to be used
    if ((host->flags & TXU_CNTRL_MGMT_NO_CCK)
            #if (NX_P2P)
            || vif->p2p
            #endif //(NX_P2P)
        )
    {
        umac->buf_control = &txl_buffer_control_5G;
    }
    else
    {
        uint8_t band = vif->chan_ctxt->channel.band;

        umac->buf_control = (band == PHY_BAND_2G4) ? &txl_buffer_control_24G : &txl_buffer_control_5G;
    }

    #if NX_MFP
    if (host->flags & TXU_CNTRL_MGMT_ROBUST)
    {
        enum mfp_protection mfp;

        /* still need to check if MFP is enable on this vif, and since
           we don't know which mgmt frame is it, pretend it is DEAUTH */
        mfp = mfp_protect_mgmt_frame(txdesc, MAC_FCTRL_DEAUTHENT, 0);

        if (mfp == MFP_UNICAST_PROT)
        {
            head_len = txu_cntrl_sechdr_len_compute(txdesc, &tail_len);
            //TODO: Use another policy table, as we will overwrite key index ?
        }
        else if (mfp == MFP_MULTICAST_PROT)
        {
            tail_len = MAC_MGMT_MIC_LEN;
        }
        else
        {
            host->flags &= ~TXU_CNTRL_MGMT_ROBUST;
        }
    }
    #endif

    // update TX Power in policy table
    tpc_get_vif_tx_power(vif, NULL, &pwr_idx);
    for (i = 0; i < RATE_CONTROL_STEPS; i++)
    {
        umac->buf_control->policy_tbl.powercntrlinfo[i] = TX_PWR_LEVEL_SET(pwr_idx);
    }

    // head_len and tail_len are used by LMAC.
    umac->machead_len = 0;
    umac->head_len = head_len;
    umac->tail_len = tail_len;

    #if (NX_AMPDU_TX)
    umac->phy_flags = 0;
    umac->flags = 0;
    #endif //(NX_AMPDU_TX)
}

/**
 ****************************************************************************************
 * @brief Append the MAC Header to the frame.
 *
 * @param[in] txdesc  TX descriptor attached to the frame to be transmitted
 * @param[in] buf     Address of the next byte following the header to be appended
 *
 * @return The address of the MAC Header in the buffer
 ****************************************************************************************
 */
static uint32_t txu_cntrl_mac_hdr_append(struct txdesc *txdesc, uint32_t buf)
{
    struct hostdesc *host = &txdesc->host;
    #if (RW_MESH_EN)
    struct umacdesc *umac_desc = &txdesc->umac;
    #endif //(RW_MESH_EN)
    struct vif_info_tag *vif = &vif_info_tab[host->vif_idx];
    struct sta_info_tag *sta = &sta_info_tab[host->staid];
    struct key_info_tag *key = *sta->sta_sec_info.cur_key;
    struct mac_hdr_qos *machdr;
    struct mac_hdr_long_qos *machdr_4a;
    uint16_t *qos;
    uint16_t order = 0;

    #if NX_HE
    // Write the +HTC HE Variant
    if ((vif->type == VIF_STA) && (STA_CAPA(sta, HE)))
    {
        buf -= MAC_HTC_LEN;
        co_write32p(buf, txl_he_htc_get(txdesc, sta));
        order = MAC_FCTRL_ORDER;
    }
    #endif

    buf -= MAC_SHORT_MAC_HDR_LEN;
    if (host->tid != 0xFF)
    {
        buf -= MAC_HDR_QOS_CTRL_LEN;

        #if (RW_MESH_EN)
        if ((vif->type == VIF_MESH_POINT) && umac_desc->has_mesh_ctrl)
        {
            buf -= (MESH_CTRL_MIN_LEN + (umac_desc->nb_ext_addr * MAC_ADDR_LEN));
        }
        #endif //(RW_MESH_EN)
    }

    if (host->flags & TXU_CNTRL_USE_4ADDR)
    {
        buf -= (MAC_LONG_MAC_HDR_LEN - MAC_SHORT_MAC_HDR_LEN);
        machdr = HW2CPU(buf);
        machdr_4a = HW2CPU(buf);
        qos = &(machdr_4a->qos);
    }
    else
    {
        machdr = HW2CPU(buf);
        machdr_4a = NULL;
        qos = &(machdr->qos);
    }

    // Check if the STA has QoS enabled
    if (host->tid != 0xFF)
    {
        // Packet shall be of type QoS
        machdr->fctl = MAC_QOS_ST_BIT;
        *qos = host->tid << MAC_QOSCTRL_UP_OFT;
        machdr->seq = host->sn << MAC_SEQCTRL_NUM_OFT;
        #if NX_AMSDU_TX
        if (host->flags & TXU_CNTRL_AMSDU)
            *qos |= MAC_QOSCTRL_AMSDU_PRESENT;
        #endif
        if (host->flags & TXU_CNTRL_EOSP)
        {
            *qos |= MAC_QOSCTRL_EOSP;
        }

        #if (RW_MESH_EN)
        if (vif->type == VIF_MESH_POINT)
        {
            if (umac_desc->has_mesh_ctrl)
            {
                *qos |= MAC_QOSCTRL_MESH_CTRL_PRESENT;

                // Fill the Mesh Control Field
                mesh_tx_data_fill_mesh_ctrl(vif, CPU2HW(qos) + MAC_HDR_QOS_CTRL_LEN, host, umac_desc);
            }
            else if (host->flags & TXU_CNTRL_MESH_FWD)
            {
                *qos |= MAC_QOSCTRL_MESH_CTRL_PRESENT;
            }

            if (txdesc->host.staid < NX_REMOTE_STA_MAX)
            {
                // Set mesh PS information in the data frame
                mesh_ps_fill_qos(sta, qos);
            }
        }
        #endif //(RW_MESH_EN)
    }
    else
    {
        machdr->fctl = 0;
        machdr->seq = 0;
    }

    // Frame Control
    machdr->fctl |= MAC_FCTRL_DATA_T | order;
    // Check the type of VIF
    if (txdesc->host.flags & TXU_CNTRL_TDLS)
        machdr->fctl &= ~MAC_FCTRL_TODS_FROMDS;
    else if (host->flags & TXU_CNTRL_USE_4ADDR)
        machdr->fctl |= MAC_FCTRL_TODS_FROMDS;
    else if (vif->type == VIF_STA)
        machdr->fctl |= MAC_FCTRL_TODS;
    else if (vif->type == VIF_AP)
        machdr->fctl |= MAC_FCTRL_FROMDS;
    #if (RW_MESH_EN)
    else if (vif->type == VIF_MESH_POINT)
    {
        machdr->fctl |= MAC_FCTRL_FROMDS;
    }
    #endif //(RW_MESH_EN)

    // Check if the more data bit shall be set
    if (host->flags & TXU_CNTRL_MORE_DATA)
        machdr->fctl |= MAC_FCTRL_MOREDATA;

    /*
     * Addresses
     ************************************************
     * IEEE 802.11 Address Field                    *
     * See IEEE 802.11-2012 Table 8.19              *
     *                                              *
     *  ToDS FromDS Addr1 Addr2 Addr3 Addr4         *
     *    0    0    DA    SA    BSSID n/a           *
     *    0    1    DA    BSSID SA    n/a           *
     *    1    0    BSSID SA    DA    n/a           *
     *    1    1    RA    TA    DA    SA            *
     ************************************************
     */
    MAC_ADDR_CPY(&machdr->addr2, &vif->mac_addr);
    #if (RW_MESH_EN)
    if (vif->type == VIF_MESH_POINT)
    {
        if (host->flags & TXU_CNTRL_USE_4ADDR)
        {
            // Get Mesh Path to be used
            struct mesh_hwmp_path_tag *mpath = &mesh_hwmp_path_pool[umac_desc->path_idx];

            MAC_ADDR_CPY(&machdr_4a->addr3, &mpath->tgt_mac_addr);
            MAC_ADDR_CPY(&machdr_4a->addr1, &sta->mac_addr);

            if (host->flags & TXU_CNTRL_MESH_FWD)
            {
                MAC_ADDR_CPY(&machdr_4a->addr4, &host->eth_src_addr);
            }
            else
            {
                MAC_ADDR_CPY(&machdr_4a->addr4, &vif->mac_addr);
            }
        }
        else if ((machdr->fctl & MAC_FCTRL_TODS_FROMDS) == MAC_FCTRL_FROMDS)
        {
            MAC_ADDR_CPY(&machdr->addr1, &host->eth_dest_addr);
            MAC_ADDR_CPY(&machdr->addr3, &vif->mac_addr);
        }
    }
    else
    #endif //(RW_MESH_EN)
    {
        if ((machdr->fctl & MAC_FCTRL_TODS_FROMDS) == MAC_FCTRL_TODS)
        {
            MAC_ADDR_CPY(&machdr->addr1, &sta->mac_addr);
            MAC_ADDR_CPY(&machdr->addr3, &host->eth_dest_addr);
        }
        else if ((machdr->fctl & MAC_FCTRL_TODS_FROMDS) == MAC_FCTRL_FROMDS)
        {
            MAC_ADDR_CPY(&machdr->addr1, &host->eth_dest_addr);
            MAC_ADDR_CPY(&machdr->addr3, &host->eth_src_addr);
        }
        else if ((machdr->fctl & MAC_FCTRL_TODS_FROMDS) == 0)
        {
            MAC_ADDR_CPY(&machdr->addr1, &host->eth_dest_addr);
            MAC_ADDR_CPY(&machdr->addr3, &vif->bss_info.bssid);
        }
        else
        {
            MAC_ADDR_CPY(&machdr_4a->addr1, &sta->mac_addr);
            MAC_ADDR_CPY(&machdr_4a->addr3, &host->eth_dest_addr);
            MAC_ADDR_CPY(&machdr_4a->addr4, &host->eth_src_addr);
        }
    }

    // Check if the frame will be protected
    if (key && !(vif->flags & CONTROL_PORT_NO_ENC &&
                 co_ntohs(host->ethertype) == sta_mgmt_get_port_ethertype(host->staid)))
    {
        machdr->fctl |= MAC_FCTRL_PROTECTEDFRAME;
    }

    return buf;
}

/**
 ****************************************************************************************
 * @brief Append the Security Header and Trailer to the frame.
 *
 * @param[in] txdesc   TX descriptor attached to the frame to be transmitted
 * @param[in] buf      Address of the next byte following the header to be appended
 * @param[in] umac_pol Select which policy table to update:
 *                     true:  The one in txdesc->umac
 *                     false: The one in txdesc->lmac->buffer
 *
 * For DATA frames, this function is called before the txl_buffer is allocated so it can
 * only update the policy table in txdesc->umac that will be recopied to the txl_buffer
 * once allocated.
 * For MGMT frames, this functions is called after the txl_buffer is allocated so it is
 * necessary to update the policy table in the txl_buffer.
 *
 * @return The address of the Security Header in the buffer
 ****************************************************************************************
 */
static uint32_t txu_cntrl_sec_hdr_append(struct txdesc *txdesc, uint32_t buf, bool umac_pol)
{
    struct hostdesc *host = &txdesc->host;
    struct umacdesc *umac = &txdesc->umac;
    struct sta_info_tag *sta = &sta_info_tab[host->staid];
    struct key_info_tag *key = *sta->sta_sec_info.cur_key;
    struct vif_info_tag *vif = &vif_info_tab[host->vif_idx];
    struct tx_policy_tbl *pol;
    uint32_t mac_control_info1;
    uint16_t *iv;

    // Check if we have a valid key for this STA
    if (!key || (vif->flags & CONTROL_PORT_NO_ENC &&
                 co_ntohs(host->ethertype) == sta_mgmt_get_port_ethertype(host->staid)))
        return buf;

    // Check which type of security is used with this STA
    // Check which encryption type has to be used
    switch(key->cipher)
    {
        case MAC_CIPHER_WEP40:
        case MAC_CIPHER_WEP104:
            // Build IV
            buf -= IV_LEN;
            iv = HW2CPU(buf);
            iv[0] = host->pn[0];
            iv[1] = host->pn[1] | (key->key_idx << 14);
            break;
        case MAC_CIPHER_TKIP:
            // Build IV/EIV
            buf -= IV_LEN + EIV_LEN;
            iv = HW2CPU(buf);
            iv[0] = (host->pn[0] >> 8) | ((host->pn[0] | 0x2000) & 0x7F00);
            iv[1] = (host->pn[0] & 0xFF) | (key->key_idx << 14) | EIV_PRESENT;
            iv[2] = host->pn[1];
            iv[3] = host->pn[2];
            break;
        case MAC_CIPHER_CCMP:
        {
            #if (RW_MESH_EN)
            uint32_t offset = 0;

            if ((vif->type == VIF_MESH_POINT) && umac->has_mesh_ctrl)
            {
                // Get Mesh VIF Information
                struct mesh_vif_info_tag *mvif = &mesh_vif_info_tab[vif->mvif_idx];

                if (mvif->is_auth)
                {
                    offset += (MESH_CTRL_MIN_LEN + (umac->nb_ext_addr * MAC_ADDR_LEN));
                }
            }
            #endif //(RW_MESH_EN)

            // Build IV/EIV
            buf -= IV_LEN + EIV_LEN;
            #if (RW_MESH_EN)
            iv = HW2CPU(buf - offset);
            #else
            iv = HW2CPU(buf);
            #endif //(RW_MESH_EN)
            iv[0] = host->pn[0];
            iv[1] = EIV_PRESENT | (key->key_idx << 14);
            iv[2] = host->pn[1];
            iv[3] = host->pn[2];
        } break;
        #if RW_WAPI_EN
        case MAC_CIPHER_WPI_SMS4:
            buf -= WPI_IV_LEN;
            iv = HW2CPU(buf);
            iv[0] = key->key_idx;
            iv[1] = host->pn[0];
            iv[2] = host->pn[1];
            iv[3] = host->pn[2];
            iv[4] = host->pn[3];
            iv[5] = 0x5c36;
            iv[6] = 0x5c36;
            iv[7] = 0x5c36;
            iv[8] = 0x5c36;
            break;
        #endif
        default:
            ASSERT_ERR(0);
            break;
    }

    // Update the policy table with the right key index
    if (umac_pol)
    {
        pol = &umac->buf_control->policy_tbl;
    }
    else
    {
        struct txl_buffer_control *bufctrl = txl_buffer_control_get(txdesc);
        pol = &bufctrl->policy_tbl;
    }
    mac_control_info1 = pol->maccntrlinfo1 & KEYSRAM_INDEX_RA_MASK;
    pol->maccntrlinfo1 = mac_control_info1 | key->hw_key_idx;

    return (buf);
}

/**
 ****************************************************************************************
 * @brief Append the LLC/SNAP to the frame if required.
 *
 * @param[in] txdesc  TX descriptor attached to the frame to be transmitted
 * @param[in] buf     Address of first byte of the payload
 *
 * @return The address of the LLC Header in the buffer
 ****************************************************************************************
 */
static uint32_t txu_cntrl_llc_hdr_append(struct txdesc *txdesc, uint32_t buf)
{
    struct hostdesc *host = &txdesc->host;
    struct llc_snap *llc_snap_ptr;

    // Check if the Ethernet frame is of Type II or SNAP
    if (co_ntohs(host->ethertype) < LLC_FRAMELENGTH_MAXVALUE)
        return buf;

    // Compute address of LLC snap
    buf -= sizeof_b(struct llc_snap);
    llc_snap_ptr = HW2CPU(buf);

    /*
     ***********************************************************************
     *              TYPE II: Build up the 802_2 Header (LLC)               *
     ***********************************************************************
     * Received from the adapter (Before the LLC translation):
     *    xxxxxxxxxxx   |    Eth Hdr    | Data
     *   [data_offset]        (14)
     *
     * After the LLC translation:
     *    xxxxxxxxxxxxxxx   | 802.2 Hdr | Data
     *     [data_offset]         (8)
     ***********************************************************************
     */
    // Fill the LLC SNAP header the LLC+OUI to the packet
    llc_snap_ptr->dsap_ssap = (LLC_DSAP | LLC_SSAP << 8);
    llc_snap_ptr->control_oui0 = LLC_CTRL;
    llc_snap_ptr->oui1_2 = 0;
    llc_snap_ptr->proto_id = host->ethertype;

    return buf;
}

#if NX_AMSDU_TX
/**
 ****************************************************************************************
 * @brief Append the A-MSDU header to the frame if required.
 *
 * @param[in] txdesc  TX descriptor attached to the frame to be transmitted
 * @param[in] buf     Address of the first byte of the LLC/SNAP
 *
 * @return The address of the A-MSDU Header in the buffer
 ****************************************************************************************
 */
static uint32_t  txu_cntrl_amsdu_hdr_append(struct txdesc *txdesc, uint32_t buf)
{
    struct hostdesc *host = &txdesc->host;
    struct umacdesc *umac = &txdesc->umac;
    struct amsdu_hdr *amsdu_hdr;
    uint16_t length;

    if (!(host->flags & TXU_CNTRL_AMSDU))
        return buf;

    buf -= sizeof_b(struct amsdu_hdr);
    amsdu_hdr = HW2CPU(buf);

    MAC_ADDR_CPY(&amsdu_hdr->da, &host->eth_dest_addr);
    MAC_ADDR_CPY(&amsdu_hdr->sa, &host->eth_src_addr);
    length = host->packet_len[0] + umac->hdr_len_802_2;
    amsdu_hdr->len= co_htons(length);

    return buf;
}
#endif

/**
 ****************************************************************************************
 * @brief Call the RC to check if is time to update the policy table.
 * Attach the buffer control to the packet.
 *
 * @param[in] txdesc  TX descriptor attached to the frame to be transmitted
 *
 ****************************************************************************************
 */
static void txu_cntrl_check_rate(struct txdesc *txdesc)
{
    // Check if rate must be changed
    struct hostdesc *host = &txdesc->host;

    // Check RC status
    rc_check(txdesc);

    // Update Buffer control
    txdesc->umac.buf_control = me_update_buffer_control(&sta_info_tab[host->staid]);

    #if NX_AMPDU_TX
    // Update the PHY flags accordingly
    txdesc->umac.phy_flags = txdesc->umac.buf_control->policy_tbl.ratecntrlinfo[RC_CONTROL_GET(txdesc->umac.rc_control, SW_RETRY_STEP)];
    #endif
}

/**
 ****************************************************************************************
 * @brief Discard a packet that cannot be transmitted.
 *
 * @param[in] txdesc           TX descriptor attached to the frame to be discarded
 * @param[in] access_category  Access category of the frame
 * @param[in] status           Status to be passed to the TXL confirmation module
 *
 ****************************************************************************************
 */
static void txu_cntrl_discard(struct txdesc *txdesc, uint8_t access_category, uint32_t status)
{
    // Increment the packet counter to keep it in track with the confirmation
    txl_cntrl_inc_pck_cnt();

    // Send the confirmation to the host
    GLOBAL_INT_DISABLE();
    txl_cfm_push(txdesc, status, access_category);
    GLOBAL_INT_RESTORE();
}


#if NX_FULLY_HOSTED
/**
 ****************************************************************************************
 * @brief Append a PBD to the txdesc descriptor for the security tail.
 *
 * If TKIP cipher is used, the content of the PBD is also initialized with the actual
 * MIC of the frame. For other ciphers it is left uninitialized as MIC will be generated
 * by MAC HW.
 * This function does nothing if no security is used.
 *
 * @param[in] txdesc  TX descriptor attached to the frame
 *
 ****************************************************************************************
 */
static void txu_cntrl_sec_tail_append(struct txdesc *txdesc)
{
    struct hostdesc *host = &txdesc->host;
    struct umacdesc *umac = &txdesc->umac;
    struct sta_info_tag *sta = &sta_info_tab[host->staid];
    struct key_info_tag *key = *sta->sta_sec_info.cur_key;
    struct mic_calc mic;
    struct tx_hd *thd = &txdesc->lmac.hw_desc->thd;
    struct tx_pbd *pbd;
    struct tx_pbd *pbd_tail;
    uint32_t offset;
    uint32_t data_addr;
    uint32_t data_len;

    // Check if we have a valid key for this STA
    if (!key)
        return;

    pbd_tail = &txdesc->lmac.buffer->pbd_tail;

    // Check which type of security is used with this STA
    if (key->cipher == MAC_CIPHER_TKIP)
    {
        // Initialize MIC computation
        me_mic_init(&mic, key->u.mic.tx_key, &host->eth_dest_addr, &host->eth_src_addr, host->tid);

        // The MIC computation shall include the 802.2 Header, therefore set it as the
        // default negative offset for the data pointer
        thd = &txdesc->lmac.hw_desc->thd;
        offset = umac->hdr_len_802_2;

        // Compute the MIC on the TX buffer descriptor(s)
        pbd = HW2CPU(thd->first_pbd_ptr);
        // Go through the chain of payload descriptors
        while (1)
        {
            data_addr = pbd->datastartptr - offset;
            data_len = pbd->dataendptr + 1 - data_addr;
            me_mic_calc(&mic, data_addr, data_len);
            if (pbd->next)
            {
                pbd = HW2CPU(pbd->next);
            }
            else
            {
                pbd->next = CPU2HW(pbd_tail);
                break;
            }
            offset = 0;
        }

        // End the MIC computation
        me_mic_end(&mic);

        // Copy the MIC into the buffer
        memcpy(&txdesc->lmac.buffer->sec_tail[0], &mic.mic_key_least, MIC_LEN);
    }
    else
    {
        // Chain the security tail at the end of the payload descriptor list
        pbd = HW2CPU(thd->first_pbd_ptr);
        while (pbd->next)
            pbd = HW2CPU(pbd->next);
        pbd->next = CPU2HW(pbd_tail);
    }

    // Fill-in the tail buffer descriptor
    pbd_tail->upatterntx = TX_PAYLOAD_DESC_PATTERN;
    pbd_tail->datastartptr = CPU2HW(&txdesc->lmac.buffer->sec_tail[0]);
    pbd_tail->dataendptr = pbd_tail->datastartptr + umac->tail_len - 1;
    pbd_tail->next = 0;
    pbd_tail->bufctrlinfo = 0;
}

/**
 ****************************************************************************************
 * @brief Links PBDs for security header/tail in MGMT frame pushed by upper layer.
 *
 * In FHOST, for MGMT frames pushed by upper layer there is no space for security header
 * /tail (as we reuse the buffer allocate in fhost). Since in this case the frame is always
 * made of one single PBD, this function will split it into two (one part for MAC header
 * and one part for payload) and a PBD for security header is added between them.
 * This PBD will point to the 'headers' buffer in the txl_buffer_tag which is not used
 * for this kind of frame (as they are pushed with a 802.11 MAC header already formatted).
 * A PDB for security MIC is also added using dedicated PBD in txdesc.
 *
 * @param[in,out] txdesc      TX descriptor for the frame
 * @param[in]     machdr_len  Size in bytes of the MAC header
 *
 * @return HW address of next byte after the security header buffer (as expected
 * by @ref txu_cntrl_sec_hdr_append)
 ****************************************************************************************
 */
static uint32_t txu_cntrl_mgmt_sec_pbd_add(struct txdesc *txdesc, int machdr_len)
{
    struct tx_hd *thd = &txdesc->lmac.hw_desc->thd;
    struct tx_pbd *pbd = (struct tx_pbd *)HW2CPU(thd->first_pbd_ptr);
    uint32_t payload_start, payload_end, sec_hdr_end;

    // Sanity check, frame should be made of one single PBD
    ASSERT_ERR(pbd && (pbd->next == 0) &&
               (pbd == &txdesc->lmac.buffer->pbd[0]));

    payload_start = pbd->datastartptr + machdr_len;
    payload_end = pbd->dataendptr;

    // Only keep MAC header in first pbd
    pbd->dataendptr = payload_start - 1;
    pbd->next = CPU2HW(&txdesc->lmac.buffer->pbd[1]);

    // pbd[1] for security header
    pbd = &txdesc->lmac.buffer->pbd[1];
    pbd->upatterntx = TX_PAYLOAD_DESC_PATTERN;
    pbd->datastartptr = CPU2HW(&txdesc->lmac.buffer->headers[0]);
    pbd->dataendptr = pbd->datastartptr + txdesc->umac.head_len - 1;
    pbd->bufctrlinfo = 0;
    pbd->next = CPU2HW(&txdesc->lmac.buffer->pbd[2]);

    sec_hdr_end = pbd->dataendptr + 1;

    // pbd[2] for mgmt frame paylaod (i.e. after MAC header)
    pbd = &txdesc->lmac.buffer->pbd[2];
    pbd->upatterntx = TX_PAYLOAD_DESC_PATTERN;
    pbd->datastartptr = payload_start;
    pbd->dataendptr = payload_end;
    pbd->bufctrlinfo = 0;
    pbd->next = CPU2HW(&txdesc->lmac.buffer->pbd_tail);

    /// pbd_tail for security tail
    pbd = &txdesc->lmac.buffer->pbd_tail;
    pbd->upatterntx = TX_PAYLOAD_DESC_PATTERN;
    pbd->datastartptr = CPU2HW(&txdesc->lmac.buffer->sec_tail[0]);
    pbd->dataendptr = pbd->datastartptr + txdesc->umac.tail_len - 1;
    pbd->next = 0;
    pbd->bufctrlinfo = 0;

    return sec_hdr_end;
}

#endif //NX_FULLY_HOSTED

/*
 * PUBLIC FUNCTION DEFINITIONS
 ****************************************************************************************
 */

void txu_cntrl_frame_build(struct txdesc *txdesc, uint32_t buf)
{
    #if (RW_MESH_EN)
    if (!(txdesc->host.flags & TXU_CNTRL_MESH_FWD))
    #endif //(RW_MESH_EN)
    {
        // Check if we need to build the LLC/SNAP
        buf = txu_cntrl_llc_hdr_append(txdesc, buf);
    }

    #if NX_AMSDU_TX
    buf = txu_cntrl_amsdu_hdr_append(txdesc, buf);
    #endif

    // Build the SEC header and trailers
    buf = txu_cntrl_sec_hdr_append(txdesc, buf, true);

    // Build the MAC header
    buf = txu_cntrl_mac_hdr_append(txdesc, buf);

    #if NX_FULLY_HOSTED
    // Append the security tail to the packet
    txu_cntrl_sec_tail_append(txdesc);

    // Adjust buffer THD - Shall contain the MAC Header
    txdesc->lmac.hw_desc->thd.datastartptr = buf;
    txdesc->lmac.hw_desc->thd.dataendptr = buf + txdesc->umac.machead_len - 1;
    // Adjust buffer TBD, that should point just after the MAC Header
    txl_buffer_get(txdesc)->pbd[0].datastartptr = txdesc->lmac.hw_desc->thd.dataendptr + 1;
    #endif
}

bool txu_cntrl_push(struct txdesc *txdesc, uint8_t access_category)
{
    // Get the VIF Entry on which packet has to be sent
    struct vif_info_tag *vif;
    bool queue_stop = false;
    uint32_t discard_status = DESC_DONE_TX_BIT;

    do
    {
        vif = &vif_info_tab[txdesc->host.vif_idx];

        if (!vif)
        {
            break;
        }

        // Check if packet can be sent now (current channel could not be the good one)
        if (!txl_cntrl_tx_check(vif))
        {
            // Add the DESC_DONE_SW_TX_BIT in order to force the driver to retransmit the
            // packet
            discard_status |= DESC_DONE_SW_TX_BIT;
            break;
        }

        // Check if the frame is a management frame
        if ((txdesc->host.flags & TXU_CNTRL_MGMT))
        {
            // Prepare the UMAC part of the TX descriptor
            txu_cntrl_umacdesc_mgmt_prep(txdesc, vif);
        }
        else
        {
            // Perform logical port filtering
            if (!txu_cntrl_logic_port_filter(txdesc->host.staid, co_ntohs(txdesc->host.ethertype),
                                             &txdesc->host.flags))
            {
                break;
            }

            // For profiling
            PROF_TX_AC_BG_SET(access_category);

            #if (RW_MESH_EN)
            // If Mesh Point VIF, check if destination can be reached
            if (vif->type == VIF_MESH_POINT)
            {
                if (txdesc->host.staid < NX_REMOTE_STA_MAX)
                {
                    // Check if frame can be sent to the peer mesh STA
                    if (!mesh_ps_tx_data_handle(vif, txdesc->host.staid))
                    {
                        break;
                    }
                }

                if (!mesh_hwmp_path_check(vif, &txdesc->host.eth_dest_addr, &txdesc->umac))
                {
                    break;
                }
            }
            #endif //(RW_MESH_EN)

            // Prepare the UMAC part of the TX descriptor
            txu_cntrl_umacdesc_prep(txdesc);

            #if (NX_AMPDU_TX)
            // Check if the packet can be aggregated
            bam_check_ba_agg(txdesc);
            #endif

            // Check RC stats
            txu_cntrl_check_rate(txdesc);

            #if NX_FULLY_HOSTED
            if (!(txdesc->host.flags & TXU_CNTRL_RETRY))
                // Build headers
                txu_cntrl_frame_build(txdesc, txl_buffer_get(txdesc)->pbd[0].datastartptr);
            #endif

        }

        // Push the descriptor to LMAC
        queue_stop = txl_cntrl_push(txdesc, access_category);

        // Handling is done for this frame
        return (queue_stop);
    } while(0);

    // Packet discarded
    txu_cntrl_discard(txdesc, access_category, discard_status);

    return false;
}

#if !NX_FULLY_HOSTED
void txu_cntrl_tkip_mic_append(struct txdesc *txdesc, uint8_t ac)
{
    struct hostdesc *host = &txdesc->host;
    struct umacdesc *umac = &txdesc->umac;
    struct sta_info_tag *sta = &sta_info_tab[host->staid];
    struct key_info_tag *key = *sta->sta_sec_info.cur_key;
    struct txl_buffer_tag *buffer = txl_buffer_get(txdesc);
    uint32_t buf = CPU2HW(buffer->payload) + umac->head_len - umac->hdr_len_802_2;

    // Check if we have a valid key for this STA
    if (!key)
        return;

    // Check which type of security is used with this STA
    // Check which encryption type has to be used
    switch(key->cipher)
    {
        case MAC_CIPHER_TKIP:
            // Compute MIC
            txl_buffer_mic_compute(txdesc, key->u.mic.tx_key, buf,
                                   umac->hdr_len_802_2 + umac->payl_len, ac);
            break;
        default:
            return;
            break;
    }
}
#endif //!NX_FULLY_HOSTED

void txu_cntrl_cfm(struct txdesc *txdesc)
{
    int8_t credits = 1;
    struct hostdesc *host = &txdesc->host;
    struct tx_cfm_tag *cfm = txl_cfm_tag_get(txdesc);
    // Indicate if transmission is successful
    bool success = ((cfm->status & FRAME_SUCCESSFUL_TX_BIT) != 0);
    // Indicate if retransmission is required by FW
    bool sw_retry = ((cfm->status & (DESC_DONE_SW_TX_BIT | DESC_DONE_TX_BIT)) ==
                                    (DESC_DONE_SW_TX_BIT | DESC_DONE_TX_BIT));

    // Reset the status field
    cfm->status = 0;

    #if !NX_FULLY_HOSTED
    // For debug add sn in all confirmation
    cfm->sn = host->sn;
    #endif

    // Specific handling for management frames transmitted by the host
    if (host->flags & TXU_CNTRL_MGMT)
    {
        // Check if we need to stop the monitoring or not
        if ((host->flags & TXU_CNTRL_MGMT_PM_MON) && !success)
        {
            // Check if we need to stop the monitoring or not
            rxu_cntrl_get_pm();
        }

        // Check if we need to send a EOSP QoS NULL frame
        if (host->flags & TXU_CNTRL_EOSP)
        {
            uint16_t qos = MAC_QOSCTRL_EOSP | (0x07 << MAC_QOSCTRL_UP_OFT);

            // Sanity check - buffered UAPSD frame shall belong to a known STA
            ASSERT_ERR(host->staid != INVALID_STA_IDX);

            // Send the NULL frame
            TRACE_AP(PS, "{VIF-%d} Send Qos NULL frame as EOSP to STA-%d as last"
                     " frame was MGMT (sn=%d)", host->vif_idx, host->staid, host->sn);
            txl_frame_send_qosnull_frame(host->staid, qos, NULL, NULL);
            sta_info_tab[host->staid].ps_service_period = NO_SERVICE_PERIOD;
        }
    }
    else if (txdesc->host.flags & TXU_CNTRL_EOSP)
    {
        sta_info_tab[host->staid].ps_service_period = NO_SERVICE_PERIOD;
        TRACE_AP(PS, "{VIF-%d} EOSP to STA-%d was data frame sn=%d",
                 host->vif_idx, host->staid, host->sn);
    }

    if (sw_retry)
    {
        cfm->status |= TX_STATUS_SW_RETRY_REQUIRED;
    }
    #if (NX_AMPDU_TX)
    // Check if packet is a QoS one
    else if (host->tid != 0xFF)
    {
        // Check if there is a BA agreement for this packet
        credits = bam_tx_cfm(txdesc, success);
        if (is_mpdu_agg(txdesc) && !is_mpdu_last(txdesc)) {
            cfm->ampdu_size = 0;
        }
    }

    if (success)
        cfm->status |= TX_STATUS_ACKNOWLEDGED;

    if (!is_mpdu_agg(txdesc)) {
        cfm->ampdu_size = 1;
    }
    #else
    cfm->ampdu_size = 1;
    #endif //(NX_AMPDU_TX)

    // Update the confirmation descriptor
    cfm->status |= TX_STATUS_DONE;
    cfm->credits = credits;
    #if NX_AMSDU_TX
    cfm->amsdu_size = me_tx_cfm_amsdu(txdesc);
    #endif
}

void txu_cntrl_protect_mgmt_frame(struct txdesc *txdesc, struct mac_hdr *mac_hdr,
                                  uint16_t hdr_len)
{
    uint32_t sec_end;

    if (txdesc->umac.head_len == 0)
    {
        int tail_len, head_len;
        head_len = txu_cntrl_sechdr_len_compute(txdesc, &tail_len);
        txdesc->umac.head_len = head_len;
        txdesc->umac.tail_len = tail_len;

        sec_end = CPU2HW(mac_hdr) + hdr_len + head_len;
    }
    else
    {
        #if NX_FULLY_HOSTED
        sec_end = txu_cntrl_mgmt_sec_pbd_add(txdesc, hdr_len);
        #else
        sec_end = CPU2HW(mac_hdr) + hdr_len + txdesc->umac.head_len;
        #endif
    }

    mac_hdr->fctl |= MAC_FCTRL_PROTECTEDFRAME;
    txu_cntrl_sec_hdr_append(txdesc, sec_end, false);
}

#if NX_FULLY_HOSTED
uint32_t txu_cntrl_mgmt_mic_pbd_append(struct txdesc *txdesc)
{
    struct tx_pbd *pbd_mmic = &txdesc->lmac.buffer->pbd_tail;
    pbd_mmic = &txdesc->lmac.buffer->pbd_tail;
    pbd_mmic->upatterntx = TX_PAYLOAD_DESC_PATTERN;
    pbd_mmic->datastartptr = CPU2HW(&txdesc->lmac.buffer->headers[0]);
    pbd_mmic->dataendptr = pbd_mmic->datastartptr + MAC_MGMT_MIC_LEN;
    pbd_mmic->next = 0;
    pbd_mmic->bufctrlinfo = 0;

    // Sanity check, there should be only one PBD
    ASSERT_ERR(txdesc->lmac.buffer->pbd[0].next == 0);
    txdesc->lmac.buffer->pbd[0].next = CPU2HW(pbd_mmic);

    return pbd_mmic->datastartptr;
}
#endif
/// @}
