/**
 ****************************************************************************************
 *
 * @file dbg.c
 *
 * @brief Definition of the Debug module environment.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup DEBUG
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "dbg.h"
#include "dbg_profiling.h"
#include "dbg_task.h"
#include "la.h"
#include "mm.h"
#include "ipc_emb.h"
#include "rxl_hwdesc.h"
#include "txl_cntrl.h"
#include "reg_mac_core.h"
#include "ke_timer.h"
#include "co_utils.h"
#if NX_FULLY_HOSTED
#include "fhost_ipc.h"
#endif



/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
/// Debug module environment definition. (moved here for host)
struct debug_env_tag dbg_env;


/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
#if NX_DEBUG_DUMP
static void dbg_dump_rx_desc(struct dbg_debug_dump_tag *dbg_dump)
{
    uint32_t dst;
    uint32_t available;
    struct dma_desc *dma_desc = (struct dma_desc *) &(debug_info.dma_desc);
    struct dbg_debug_info_tag *dbg_info = &(debug_info.dbg_info);
    struct rx_hd *rhd;
    struct rx_pbd *rbd;

    // Fill-in the debug information structure
    rxl_current_desc_get(&rhd, &rbd);

    // Dump RX header descriptors
    dbg_info->rhd_len = 0;
    dbg_info->rhd = CPU2HW(rhd);
    dst = CPU2HW(&dbg_dump->rhd_mem);
    available = DBG_RHD_MEM_LEN;
    dma_desc->length = sizeof_b(*rhd);
    dma_desc->ctrl = IPC_DMA_LLI_IRQ_EN |
                                       (IPC_DMA_LLI_DBG_DUMP << IPC_DMA_LLI_IRQ_POS);
    while ((rhd != NULL) && (available >= sizeof_b(*rhd)))
    {
        CHK_SHRAM_PTR(rhd);
        dma_desc->src = CPU2HW(rhd);
        dma_desc->dest = dst;

        // Push the DMA descriptor in the DMA engine
        dma_push(dma_desc, dma_desc, IPC_DMA_CHANNEL_CTRL_RX);

        // Poll for the DMA to be finished
        dma_lli_poll(IPC_DMA_LLI_DBG_DUMP + DMA_LLI_IRQ_LSB);

        dst += dma_desc->length;
        available -= dma_desc->length;
        dbg_info->rhd_len += dma_desc->length;

        rhd = HW2CPU(rhd->next);
    }

    // Dump RX buffer descriptors
    dbg_info->rbd_len = 0;
    dbg_info->rbd = CPU2HW(rbd);
    dst = CPU2HW(&dbg_dump->rbd_mem);
    available = DBG_RHD_MEM_LEN;
    dma_desc->length = sizeof_b(*rbd);
    dma_desc->ctrl = IPC_DMA_LLI_IRQ_EN |
                                       (IPC_DMA_LLI_DBG_DUMP << IPC_DMA_LLI_IRQ_POS);
    while ((rbd != NULL) && (available >= sizeof_b(*rbd)))
    {
        CHK_SHRAM_PTR(rbd);
        dma_desc->src = CPU2HW(rbd);
        dma_desc->dest = dst;

        // Push the DMA descriptor in the DMA engine
        dma_push(dma_desc, dma_desc, IPC_DMA_CHANNEL_CTRL_RX);

        // Poll for the DMA to be finished
        dma_lli_poll(IPC_DMA_LLI_DBG_DUMP + DMA_LLI_IRQ_LSB);

        dst += dma_desc->length;
        available -= dma_desc->length;
        dbg_info->rbd_len += dma_desc->length;

        rbd = HW2CPU(rbd->next);
    }
}


static void dbg_dump_tx_desc(struct dbg_debug_dump_tag *dbg_dump)
{
    uint32_t dst;
    uint32_t available;
    struct dma_desc *dma_desc = (struct dma_desc *) &(debug_info.dma_desc);
    struct dbg_debug_info_tag *dbg_info = &(debug_info.dbg_info);
    struct tx_hd *thd;
    struct tx_pbd *tbd;
    int i;

    for (i = 0; i < NX_TXQ_CNT; i++)
    {
        txl_current_desc_get(i, &thd);
        dbg_info->thd[i] = CPU2HW(thd);

        // Dump TX header descriptors
        dbg_info->thd_len[i] = 0;
        dst = CPU2HW(&dbg_dump->thd_mem[i]);
        available = DBG_THD_MEM_LEN;
        dma_desc->ctrl = IPC_DMA_LLI_IRQ_EN |
                                           (IPC_DMA_LLI_DBG_DUMP << IPC_DMA_LLI_IRQ_POS);
        while ((thd != NULL) && (available >= (sizeof_b(*thd) + sizeof_b(struct tx_policy_tbl))))
        {
            CHK_SHRAM_PTR(thd);
            dma_desc->src = CPU2HW(thd);
            dma_desc->dest = dst;
            dma_desc->length = sizeof_b(*thd);

            // Push the DMA descriptor in the DMA engine
            dma_push(dma_desc, dma_desc, IPC_DMA_CHANNEL_CTRL_RX);

            // Poll for the DMA to be finished
            dma_lli_poll(IPC_DMA_LLI_DBG_DUMP + DMA_LLI_IRQ_LSB);

            dst += dma_desc->length;
            available -= dma_desc->length;
            dbg_info->thd_len[i] += dma_desc->length;

            // Check if we have to upload the policy table too
            if (!((thd->macctrlinfo2 & 0x00200000) &&
                (thd->macctrlinfo2 & 0x00180000)))
            {
                CHK_SHRAM_PTR(HW2CPU(thd->policyentryaddr));
                dma_desc->src = thd->policyentryaddr;
                dma_desc->dest = dst;
                dma_desc->length = sizeof_b(struct tx_policy_tbl);

                // Push the DMA descriptor in the DMA engine
                dma_push(dma_desc, dma_desc, IPC_DMA_CHANNEL_CTRL_RX);

                // Poll for the DMA to be finished
                dma_lli_poll(IPC_DMA_LLI_DBG_DUMP + DMA_LLI_IRQ_LSB);

                dst += dma_desc->length;
                available -= dma_desc->length;
                dbg_info->thd_len[i] += dma_desc->length;
            }

            // Upload the TX payload descriptors
            tbd = HW2CPU(thd->first_pbd_ptr);
            while ((tbd != NULL) && (available >= sizeof_b(*tbd)))
            {
                if (TST_SHRAM_PTR(tbd))
                    break;
                dma_desc->src = CPU2HW(tbd);
                dma_desc->dest = dst;
                dma_desc->length = sizeof_b(*tbd);

                // Push the DMA descriptor in the DMA engine
                dma_push(dma_desc, dma_desc, IPC_DMA_CHANNEL_CTRL_RX);

                // Poll for the DMA to be finished
                dma_lli_poll(IPC_DMA_LLI_DBG_DUMP + DMA_LLI_IRQ_LSB);

                dst += dma_desc->length;
                available -= dma_desc->length;
                dbg_info->thd_len[i] += dma_desc->length;

                // Go to next TBD
                tbd = HW2CPU(tbd->next);
            }

            if (thd->nextmpdudesc_ptr)
                thd = HW2CPU(thd->nextmpdudesc_ptr);
            else
                thd = HW2CPU(thd->nextfrmexseq_ptr);
        }
    }
}

static void dbg_dump_key_ram(void)
{
    #if NX_DEBUG_DUMP_KEY
    TRACE("Key RAM content:")
    for (int i = 0; i < MM_SEC_MAX_KEY_NBR; i++)
    {
        nxmac_encr_cntrl_pack(1, 0, 0, 0, i, 0, 0, 0, 0, 0);
        while(nxmac_new_read_getf());

        TRACE("IDX: %d, CNTRL: %08lx, MAC=%pM", i, TR_32(nxmac_encr_cntrl_get()),
              TR_MAC(NXMAC_ENCR_MAC_ADDR_LOW_ADDR));
        TRACE_BUF("KEY: %pB16", 16, NXMAC_ENCR_KEY_0_ADDR);
    }
    #endif
}

static void dbg_dump_info(struct dbg_debug_dump_tag *dbg_dump, char *msg, uint32_t type)
{
    uint32_t dst = CPU2HW(&dbg_dump->dbg_info);
    struct dma_desc *dma_desc = (struct dma_desc *) &(debug_info.dma_desc);
    struct dbg_debug_info_tag *dbg_info = &(debug_info.dbg_info);
    uint32_t sw_diag_len = 0;
    #if NX_PROFILING_ON
    uint32_t sw_diag_addr = CPU2HW(dbg_info->sw_diag);
    #endif
    #if NX_PROFILING_ON
    int i;
    #endif

    // Fill-in the debug information structure
    dbg_info->error_type = type;
    dbg_info->hw_diag = nxmac_debug_port_sel_get();

    // Fill-in LA configuration
    la_get_conf(&dbg_info->la_conf);

    // Fill-in error message
    co_pack8p(CPU2HW(dbg_info->error),(uint8_t *) msg, DBG_ERROR_TRACE_SIZE);

    // Fill-in PHY channel information
    phy_get_channel(&dbg_info->chan_info, PHY_PRIM);

    // Profiling configuration information
    #if NX_PROFILING_ON
    for (i = 0; i < DBG_PROF_MAX; i++)
    {
        char tmp[64];

        uint32_t len = dbg_snprintf(tmp,
                                    co_min(64, DBG_SW_DIAG_MAX_LEN - sw_diag_len),
                                    "%s\n", dbg_prof_conf[i]);
        if (len >= co_min(64, DBG_SW_DIAG_MAX_LEN - sw_diag_len))
            len = co_min(64, DBG_SW_DIAG_MAX_LEN - sw_diag_len) - 1;
        co_pack8p(sw_diag_addr, (uint8_t *)tmp, len);
        sw_diag_len += len;
        sw_diag_addr += len;
    }
    #endif

    dbg_info->sw_diag_len = sw_diag_len;

    // Disable interrupts as we will poll
    GLOBAL_INT_DISABLE();

    // Only one DMA transfer will be needed to send the complete MSG.
    dma_desc->dest = dst;
    dma_desc->src = CPU2HW(dbg_info);
    dma_desc->length = sizeof_b(*dbg_info);
    dma_desc->ctrl = IPC_DMA_LLI_IRQ_EN |
                                       (IPC_DMA_LLI_DBG_DUMP << IPC_DMA_LLI_IRQ_POS);

    // Push the DMA descriptor in the DMA engine
    dma_push(dma_desc, dma_desc, IPC_DMA_CHANNEL_CTRL_RX);

    // Poll for the DMA to be finished
    dma_lli_poll(IPC_DMA_LLI_DBG_DUMP + DMA_LLI_IRQ_LSB);

    // Reenable interrupts
    GLOBAL_INT_RESTORE();
}

void dbg_dump_error(struct dbg_debug_dump_tag *dbg_dump, char *msg, uint32_t type)
{
    // Dump the embedded logic analyzer trace
    la_dump_trace(dbg_dump);

    // Dump the RX descriptors
    dbg_dump_rx_desc(dbg_dump);

    // Dump the TX descriptors
    dbg_dump_tx_desc(dbg_dump);

    // Dump the Key RAM content
    dbg_dump_key_ram();

    // Dump other debug information (HW and SW diags,
    dbg_dump_info(dbg_dump, msg, type);
}
#endif

#if NX_SYS_STAT
void dbg_sys_stat_reset(void)
{
    // Reset the statistics
    dbg_env.sys_stats.doze_time = 0;
    dbg_env.sys_stats.cpu_sleep_time = 0;
    dbg_env.sys_stats.stats_time = 0;
    dbg_env.sys_stats.start_time = dbg_sys_stats_time();

    // Restart the statistic measurement timer
    ke_timer_set(DBG_SYS_STAT_TIMER, TASK_DBG, DBG_SYS_STAT_TIMEOUT);
}
#endif

void dbg_init(void)
{
    // Reset the environment
    memset(&dbg_env, 0, sizeof(dbg_env));

    #if NX_PRINT == NX_PRINT_IPC
    // Init print
    dbg_env.print_mutex = 0;
    #endif

    // Enable only warnings and more critical per default
    dbg_env.filter_module = DBG_MOD_ALL;
    dbg_env.filter_severity = DBG_SEV_IDX_ERR;

    // Link error in case too many SW profiling diags are defined
    if (DBG_PROF_MAX > 32)
    {
        extern void too_many_sw_profiling_diags_are_defined(void);
        too_many_sw_profiling_diags_are_defined();
    }
}

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
#if NX_DEBUG_DUMP
void dbg_error_save(const char *error)
{
    // Fill-in error message
    memcpy(dbg_env.error, error, DBG_ERROR_TRACE_SIZE);
}

void dbg_error_ind(char *msg, uint32_t type)
{
    struct dbg_debug_dump_tag *dbg_dump =
               (struct dbg_debug_dump_tag *)ipc_emb_hostdbgdumpbuf_get();

    // Dump the LA traces
    if (dbg_dump != NULL)
    {
        // Dump all the debug information about the error
        dbg_dump_error(HW2CPU(dbg_dump), msg, type);

        // Disable the interrupts
        GLOBAL_INT_DISABLE();

        // Indicate the debug dump to the host
        #if NX_FULLY_HOSTED
        fhost_ipc_error_ind();
        #else
        ke_msg_send_basic(DBG_ERROR_IND, TASK_API, TASK_DBG);
        #endif

        // Poll for the DMA to be finished
        dma_lli_poll(IPC_DMA_LLI_MSG + DMA_LLI_IRQ_LSB);

        // Call the IPC message DMA handler
        ipc_emb_msg_dma_int_handler();

        // Restore the interrupts
        GLOBAL_INT_RESTORE();
    }
    else
    {
        dbg(D_CRT "%s", msg);
    }

    // Restart the LA
    la_start();
}
#endif


/// @} end of group

