/**
 ****************************************************************************************
 *
 * @file dbg_task.c
 *
 * @brief Defines the different debug states and messages handlers used by the debug task
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup ASSERT
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "dbg_assert.h"
#include "dbg.h"
#include "ke_event.h"
#include "hal_machw.h"
#include "reg_sysctrl.h"

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
/// Whether to enter endless loop after assert.
int dbg_assert_block = 1;

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
void dbg_assert_rec(const char *condition, const char * file, int line)
{
    // Disable the interrupts
    GLOBAL_INT_DISABLE();

    // Display a trace message showing the error
    TRACE("ASSERT recovery: %F:%d", (line >> 20) , line);
    line = (line & 0xfffff);
    dbg(D_ERR "ASSERT (%s) at %s:%d\n", condition, file, line);

    // Check if a recovery is already pending
    if (!(ke_evt_get() & KE_EVT_RESET_BIT))
    {
        #if NX_DEBUG_DUMP
        char msg[DBG_ERROR_TRACE_SIZE];

        // Write to the trigger bit to stop the internal or external logic analyzer
        sysctrl_diag_trigger_set(1);

        // Prepare the error trace to be forwarded to host
        dbg_snprintf(msg, DBG_ERROR_TRACE_SIZE, "ASSERT (%s) at %s:%d\n", condition, file, line);

        // Save the error trace
        dbg_error_save(msg);

        // Clear the trigger bit
        sysctrl_diag_trigger_set(0);
        #endif

        // Disable MAC HW interrupts
        hal_machw_disable_int();

        // Trigger the reset procedure
        ke_evt_set(KE_EVT_RESET_BIT);
    }

    // Restore the interrupts
    GLOBAL_INT_RESTORE();
}

void dbg_assert_err(const char *condition, const char * file, int line)
{
    #if NX_DEBUG_DUMP
    char msg[DBG_ERROR_TRACE_SIZE];
    #endif

    TRACE("ASSERT error: %F:%d", (line >> 20) , line);
    line = (line & 0xfffff);

    // Stop the interrupts
    GLOBAL_INT_STOP();

    // Write to the trigger bit to stop the internal logic analyzer
    sysctrl_diag_trigger_set(1);

    // Stop the HW to keep it as close as possible to the error state
    nxmac_next_state_setf(HW_IDLE);

    // Display a trace message showing the error
    dbg(D_ERR "ASSERT (%s) at %s:%d\n", condition, file, line);

    #if NX_DEBUG_DUMP
    // Prepare the error trace to be forwarded to host
    dbg_snprintf(msg, DBG_ERROR_TRACE_SIZE, "ASSERT (%s) at %s:%d\n", condition, file, line);

    // Forward the error indication to the host
    dbg_error_ind(msg, DBG_ERROR_FATAL);
    #endif

    while(dbg_assert_block);
}

void dbg_assert_warn(const char *condition, const char * file, int line)
{
    TRACE("ASSERT warning: %F:%d", (line >> 20) , line);
    line = (line & 0xfffff);
    dbg(D_ERR "WARNING (%s) at %s:%d\n", condition, file, line);
}

void dbg_force_trigger(const char *msg)
{
    #if NX_DEBUG_DUMP && defined(CFG_RWTL)
    char msgbuf[256];
    uint8_t *buf = (uint8_t *)msgbuf;
    uint32_t addr = CPU2HW(msg);
    uint8_t val;
    #endif

    // Stop the interrupts
    GLOBAL_INT_DISABLE();

    #if NX_DEBUG_DUMP && defined(CFG_RWTL)
    // Unpack the string
    do
    {
        val = co_read8p(addr++);
        *buf++ = val;
    } while(val != 0);
    msg = (const char *)msgbuf;
    #endif

    // Write to the trigger bit to stop the internal logic analyzer
    sysctrl_diag_trigger_set(1);

    // Clear the trigger bit immediately as the LA will be restarted in
    // dbg_error_ind()
    sysctrl_diag_trigger_set(0);

    #if NX_DEBUG_DUMP
    // Get HW diagnostic port state
    hal_machw_get_diag_state();

    // Forward the error indication to the host
    dbg_error_ind((char *) msg, DBG_ERROR_RECOVERABLE);
    #endif

    // Restore the interrupts
    GLOBAL_INT_RESTORE();
}


/// @}  // end of group ASSERT
