/**
 ****************************************************************************************
 *
 * @file dbg_task.c
 *
 * @brief Defines the different debug states and messages handlers used by the debug task
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup DBG_TASK
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "co_endian.h"

#include "sta_mgmt.h"

#include "ke_timer.h"
#include "reg_mac_core.h"

#include "dbg.h"
#include "dbg_task.h"
#include "co_utils.h"

#include "reg_sysctrl.h"

/*
 * MESSAGE HANDLERS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief This function handles the @ref DBG_MEM_READ_REQ message
 *
 * @param[in]    msgid      Id of the message received
 * @param[in]    param      Pointer to the parameters of the message
 * @param[in]    dest_id    TaskId of the receiving task
 * @param[in]    src_id     TaskId of the sending task
 * @return       Whether the message was consumed or not
 ****************************************************************************************
 */
static int
dbg_mem_read_req_handler(ke_msg_id_t const msgid,
                         struct dbg_mem_read_req const *param,
                         ke_task_id_t const dest_id,
                         ke_task_id_t const src_id)
{
    // Allocate the message response structure
    struct dbg_mem_read_cfm *mem_rd_cfm =
        KE_MSG_ALLOC(DBG_MEM_READ_CFM, src_id, dest_id, dbg_mem_read_cfm);

    /// Read from the corresponding memory location
    mem_rd_cfm->memdata = *(volatile uint32_t *)param->memaddr;
    mem_rd_cfm->memaddr = param->memaddr;

    // Trace the action
    dbg(D_INF D_DBG "DBG: Reading memory: [0x%08x] = 0x%08x / %d\n",
        param->memaddr, mem_rd_cfm->memdata, mem_rd_cfm->memdata);

    // Post the confirm message to the kernel
    ke_msg_send(mem_rd_cfm);

    return KE_MSG_CONSUMED;
}

/**
 ****************************************************************************************
 * @brief This function handles the @ref DBG_MEM_WRITE_REQ message
 *
 * @param[in]    msgid      Id of the message received
 * @param[in]    param      Pointer to the parameters of the message
 * @param[in]    dest_id    TaskId of the receiving task
 * @param[in]    src_id     TaskId of the sending task
 * @return       Whether the message was consumed or not
 ****************************************************************************************
 */
static int
dbg_mem_write_req_handler(ke_msg_id_t const msgid,
                          struct dbg_mem_write_req const *param,
                          ke_task_id_t const dest_id,
                          ke_task_id_t const src_id)
{
    // Allocate the message response structure
    struct dbg_mem_write_cfm *mem_wr_cfm =
        KE_MSG_ALLOC(DBG_MEM_WRITE_CFM, src_id, dest_id, dbg_mem_write_cfm);

    // Write to the corresponding memory location
    *(volatile uint32_t *)param->memaddr = param->memdata;
    mem_wr_cfm->memaddr = param->memaddr;
    mem_wr_cfm->memdata = *(volatile uint32_t *)param->memaddr;

    // Trace the action
    dbg(D_INF D_DBG "DBG: Writing memory with 0x%08x / %d: [0x%08x] = 0x%08x / %d\n",
        param->memdata, param->memdata, param->memaddr,
        mem_wr_cfm->memdata, mem_wr_cfm->memdata);

    // Post the confirm message to the kernel
    ke_msg_send(mem_wr_cfm);

    return KE_MSG_CONSUMED;
}

/**
 ****************************************************************************************
 * @brief This function handles the @ref DBG_SET_MOD_FILTER_REQ message
 *
 * @param[in]    msgid      Id of the message received
 * @param[in]    param      Pointer to the parameters of the message
 * @param[in]    dest_id    TaskId of the receiving task
 * @param[in]    src_id     TaskId of the sending task
 * @return       Whether the message was consumed or not
 ****************************************************************************************
 */
static int
dbg_set_mod_filter_req_handler(ke_msg_id_t const msgid,
                               struct dbg_set_mod_filter_req const *param,
                               ke_task_id_t const dest_id,
                               ke_task_id_t const src_id)
{
    // Trace the action
    dbg(D_INF D_DBG "DBG: Setting module filter: 0x%08x\n", param->mod_filter);

    /// Set the module filter according to the parameters
    dbg_env.filter_module = param->mod_filter;

    // Post the confirm message to the kernel
    ke_msg_send_basic(DBG_SET_MOD_FILTER_CFM, src_id, dest_id);

    return KE_MSG_CONSUMED;
}


/**
 ****************************************************************************************
 * @brief This function handles the @ref DBG_SET_SEV_FILTER_REQ message
 *
 * @param[in]    msgid      Id of the message received
 * @param[in]    param      Pointer to the parameters of the message
 * @param[in]    dest_id    TaskId of the receiving task
 * @param[in]    src_id     TaskId of the sending task
 * @return       Whether the message was consumed or not
 ****************************************************************************************
 */
static int
dbg_set_sev_filter_req_handler(ke_msg_id_t const msgid,
                               struct dbg_set_sev_filter_req const *param,
                               ke_task_id_t const dest_id,
                               ke_task_id_t const src_id)
{
    // Trace the action
    dbg(D_INF D_DBG "DBG: Setting severity filter: 0x%08x\n", param->sev_filter);

    /// Set the severity filter according to the parameters
    dbg_env.filter_severity = param->sev_filter;

    // Post the confirm message to the kernel
    ke_msg_send_basic(DBG_SET_SEV_FILTER_CFM, src_id, dest_id);

    return KE_MSG_CONSUMED;
}


/**
 ****************************************************************************************
 * @brief This function handles the @ref DBG_GET_SYS_STAT_REQ message
 *
 * @param[in]    msgid      Id of the message received
 * @param[in]    param      Pointer to the parameters of the message
 * @param[in]    dest_id    TaskId of the receiving task
 * @param[in]    src_id     TaskId of the sending task
 * @return       Whether the message was consumed or not
 ****************************************************************************************
 */
static int
dbg_get_sys_stat_req_handler(ke_msg_id_t const msgid,
                             void const *param,
                             ke_task_id_t const dest_id,
                             ke_task_id_t const src_id)
{
    struct dbg_get_sys_stat_cfm *cfm = KE_MSG_ALLOC(DBG_GET_SYS_STAT_CFM, src_id,
                                                    dest_id, dbg_get_sys_stat_cfm);

    // Fill in the message parameters
    #if NX_SYS_STAT
    cfm->cpu_sleep_time = dbg_env.sys_stats.cpu_sleep_time;
    cfm->doze_time = dbg_env.sys_stats.doze_time;
    cfm->stats_time = dbg_env.sys_stats.stats_time
                   + ((dbg_sys_stats_time() - dbg_env.sys_stats.start_time) >> 10);

    // Reset and restart the statistic measurements
    dbg_sys_stat_reset();
    #else
    cfm->cpu_sleep_time = 0;
    cfm->doze_time = 0;
    cfm->stats_time = 0;
    #endif

    // Send the confirm message to the kernel
    ke_msg_send(cfm);

    return KE_MSG_CONSUMED;
}


#if NX_SYS_STAT
/**
 ****************************************************************************************
 * @brief This function handles the @ref DBG_SET_SEV_FILTER_REQ message
 *
 * @param[in]    msgid      Id of the message received
 * @param[in]    param      Pointer to the parameters of the message
 * @param[in]    dest_id    TaskId of the receiving task
 * @param[in]    src_id     TaskId of the sending task
 * @return       Whether the message was consumed or not
 ****************************************************************************************
 */
static int
dbg_sys_stat_timer_handler(ke_msg_id_t const msgid,
                           void const *param,
                           ke_task_id_t const dest_id,
                           ke_task_id_t const src_id)
{
    uint32_t time = dbg_sys_stats_time();

    dbg_env.sys_stats.stats_time += (time - dbg_env.sys_stats.start_time) >> 10;
    dbg_env.sys_stats.start_time = time;
    // Restart the statistic measurement timer
    ke_timer_set(DBG_SYS_STAT_TIMER, TASK_DBG, DBG_SYS_STAT_TIMEOUT);

    return KE_MSG_CONSUMED;
}
#endif


/*
 * TASK DESCRIPTOR DEFINITIONS
 ****************************************************************************************
 */
/// DEFAULT handler
const struct ke_msg_handler dbg_default_state[] =
{
    // Memory Read request
    {DBG_MEM_READ_REQ, (ke_msg_func_t)dbg_mem_read_req_handler},
    // Memory write request
    {DBG_MEM_WRITE_REQ, (ke_msg_func_t)dbg_mem_write_req_handler},
    // Module filter request
    {DBG_SET_MOD_FILTER_REQ, (ke_msg_func_t)dbg_set_mod_filter_req_handler},
    // Severity filter request
    {DBG_SET_SEV_FILTER_REQ, (ke_msg_func_t)dbg_set_sev_filter_req_handler},
    // Get system statistics request
    {DBG_GET_SYS_STAT_REQ, (ke_msg_func_t)dbg_get_sys_stat_req_handler},
    #if NX_SYS_STAT
    // System statistics timer handler
    {DBG_SYS_STAT_TIMER, (ke_msg_func_t)dbg_sys_stat_timer_handler},
    #endif
};

/// Specifies the message handlers that are common to all states.
const struct ke_state_handler dbg_default_handler =
    KE_STATE_HANDLER(dbg_default_state);

/// @}  // end of group DBG
