/**
 ******************************************************************************
 *
 * @file trace.c
 *
 * @brief Definition for the trace buffer.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ******************************************************************************
 */

/**
 ******************************************************************************
 * @defgroup TRACE TRACE
 * @ingroup DEBUG
 * @brief Trace buffer debug module.
 *
 * Trace module is enabled when fw is compiled with TRACE=on option.\n
 *
 * Trace module is using a buffer located in shared ram, to store trace event
 * form fw. A trace entry is optimized to only contains IDs and parameters,
 * instead of complete string. This IDs and parameters can then be decoded
 * using dictionary generated at compilation time.
 *
 * A trace entry in memory looks like :
 * @verbatim
 *  15       8 7        0
 * +---------------------+ --+
 * | NB PARAM |  ID MSB  |   |
 * +---------------------+   |
 * |       ID LSB        |   |
 * +---------------------+   +--> header (fixed size)
 * | TIMESTAMP  MSB      |   |
 * +---------------------+   |
 * | TIMESTAMP  LSB      |   |
 * +---------------------+ --+
 * | PARAM(1)            |   |
 * +---------------------+   |
 *    ....                   +--> parameters (variable size)
 * +---------------------+   |
 * | PARAM(x)            |   |
 * +---------------------+ --+
 * @endverbatim
 *
 * Trace buffer is defined by @ref _trace_start and @ref _trace_end whose value
 * is defined in linker script in order to use all the remaining shared ram.
 *
 * @{
 ******************************************************************************
 */

#include "trace.h"

#if NX_TRACE

#include "ipc_shared.h"
#include "hal_machw.h"

/*
 * GLOBAL VARIABLES
 ******************************************************************************
 */
/// Global variable indicating if trace buffer has been initialized.
static bool trace_initialized = false;
/// Global variable indicating if trace buffer must be used as a circular buffer
static bool trace_loop = false;

/**
 * @var _trace_start
 * start address of the trace buffer
 * @var _trace_end
 * end address of the trace buffer
 */
extern uint16_t _trace_start, _trace_end;

/**
 * Trace filter for each component
 */
uint32_t trace_compo_level[TRACE_COMPO_MAX] __SHAREDRAM;

/*
 * DEFINES
 ******************************************************************************
 */
/// Size of trace header (in 16bits words)
#define TRACE_HEADER_LEN 4
/// Maximum number of parameter in one trace entry
#define TRACE_MAX_PARAM  0xff
/// Maximum ID of a trace entry
#define TRACE_MAX_ID     0xffffff
/// Return size (in 16bits words) of the trace entry whose first word is @p i
#define TRACE_LEN(i) (((i) >> 8) + TRACE_HEADER_LEN)

/// Pattern to mark last trace entry
#define TRACE_LAST_ENTRY 0xffff
/// Pattern to mark than trace buffer is ready
#define TRACE_READY  0x1234
/// Pattern to mark than trace buffer is being read or write
#define TRACE_LOCKED 0xdead

/*
 * FUNCTION DEFINITIONS
 ******************************************************************************
 */
void trace_init(bool force, bool loop)
{
    if (trace_initialized && !force)
        return;

    ipc_shared_env.trace_size = (&_trace_end - &_trace_start);
    ipc_shared_env.trace_offset = ((uint32_t)&_trace_start -
                                   (uint32_t)&ipc_shared_env.trace_offset) * CHAR_LEN;
    ipc_shared_env.trace_start = 0;
    ipc_shared_env.trace_end = ipc_shared_env.trace_size + 1;
    ipc_shared_env.trace_pattern = TRACE_READY;
    ipc_shared_env.trace_nb_compo = (TRACE_READY << 16) + TRACE_COMPO_MAX;
    ipc_shared_env.trace_offset_compo = ((uint32_t)trace_compo_level -
                                         (uint32_t)&ipc_shared_env.trace_offset_compo) * CHAR_LEN;

    // set default level here as SHARED RAM is not initialized.
    trace_compo_level[TRACE_COMPO_KERNEL] = TRACE_KERNEL_DEFAULT;
    trace_compo_level[TRACE_COMPO_LMAC] = TRACE_LMAC_DEFAULT;
    trace_compo_level[TRACE_COMPO_CHAN] = TRACE_CHAN_DEFAULT;
    trace_compo_level[TRACE_COMPO_STA] = TRACE_STA_DEFAULT;
    trace_compo_level[TRACE_COMPO_AP] = TRACE_AP_DEFAULT;
    trace_compo_level[TRACE_COMPO_P2P] = TRACE_P2P_DEFAULT;
    trace_compo_level[TRACE_COMPO_MESH] = TRACE_MESH_DEFAULT;
    trace_compo_level[TRACE_COMPO_RTOS] = TRACE_RTOS_DEFAULT;
    trace_compo_level[TRACE_COMPO_APP] = TRACE_APP_DEFAULT;
    trace_compo_level[TRACE_COMPO_FHOST] = TRACE_FHOST_DEFAULT;
    trace_compo_level[TRACE_COMPO_TDLS] = TRACE_TDLS_DEFAULT;

    trace_loop = loop;
    trace_initialized = true;
}


void trace(uint32_t id, uint16_t nb_param, uint16_t *param, bool trace_buf)
{
    uint32_t end;
    uint32_t start;
    uint16_t *ptr;
    uint32_t ts;
    bool loop=0;

    /* Needed to be able to trace within IT handler */
    GLOBAL_INT_DISABLE();

    ipc_shared_env.trace_pattern = TRACE_LOCKED;

    /* Ensure parameters provided are within expected range */
    if (nb_param > TRACE_MAX_PARAM)
        nb_param = TRACE_MAX_PARAM;
    id &= TRACE_MAX_ID;


    end = ipc_shared_env.trace_end;
    start = ipc_shared_env.trace_start;
    ts = hal_machw_time();
    ptr = &_trace_start;

    if (end > ipc_shared_env.trace_size)
    {
        /* init case */
        start = 0;
        end = 0;
    }
    else
    {
        if (end < start)
            loop = 1;

        /* end is the index of the last trace point, so the next one will be
           at end + size of the trace */
        end += TRACE_LEN(ptr[end]);

        if (end + TRACE_HEADER_LEN + nb_param >= ipc_shared_env.trace_size)
        {
            if (!trace_loop)
                goto end;

            if (end < ipc_shared_env.trace_size)
                ptr[end] = TRACE_LAST_ENTRY;

            end = 0;
            start = TRACE_LEN(ptr[end]);
            loop = 1;
        }

        while(loop && (int16_t)(start - end) < (nb_param + TRACE_HEADER_LEN))
        {
            start += TRACE_LEN(ptr[start]);

            if ((start >= ipc_shared_env.trace_size) ||
                (ptr[start] == TRACE_LAST_ENTRY)) {
                start = 0;
                loop = 0;
            }
        }
    }

    /* write trace header */
    ptr += end;
    *ptr++ = (nb_param << 8) | (id >> 16);
    #ifdef CFG_RWX1
    _dsp_asm("nop");
    #endif

    *ptr++ = (id & 0xffff);
    #ifdef CFG_RWX1
    _dsp_asm("nop");
    #endif

    *ptr++ = (ts >> 16);
    #ifdef CFG_RWX1
    _dsp_asm("nop");
    #endif

    *ptr++ = (ts & 0xffff);
    #ifdef CFG_RWX1
    _dsp_asm("nop");
    #endif

    if (trace_buf)
    {
        uint32_t buf;
        buf = (param[1]) | (param[2] << 16);
        if (param[0] > ((TRACE_MAX_PARAM - 1) * 2))
            //set bit 10 to indicate incomplete buffer
            *ptr = ((TRACE_MAX_PARAM - 1) * 2) + (1 << 10);
        else
            *ptr = param[0];
        param = _PTR_ALIGN(buf);
        #ifndef CFG_RWTL
        // set bit 9 to indicate unaligned buffer
        *ptr |= ((buf & 0x1) << 9);
        #endif
        ptr++;
        nb_param--;
    }

    for (int i = 0; i < nb_param ; i++)
    {
        *ptr++ = param[i];
        #ifdef CFG_RWX1
        _dsp_asm("nop");
        #endif
    }

    ipc_shared_env.trace_start = start;
    ipc_shared_env.trace_end = end;

  end:
    GLOBAL_INT_RESTORE();
    ipc_shared_env.trace_pattern = TRACE_READY;
}


void trace_set_filter(unsigned int compo_id, unsigned int level)
{
    if (compo_id == TRACE_COMPO_MAX)
    {
        unsigned int i;
        for (i = 0; i < TRACE_COMPO_MAX; i++)
        {
            trace_compo_level[i] = level;
        }
    }
    else if (compo_id < TRACE_COMPO_MAX)
    {
        trace_compo_level[compo_id] = level;
    }
}

#endif // NX_TRACE



/// @} end of group
