/**
 ****************************************************************************************
 *
 * @file ke_event.c
 *
 * @brief This file contains the event handling primitives.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup KE_EVT
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include <stddef.h>
#include "co_int.h"
#include "co_bool.h"

#include "rwnx_config.h"
#include "dbg_assert.h"

#include "ke_event.h"
#include "ke_env.h"
#include "ke_mem.h"
#include "mm_timer.h"


/*
 * TABLE OF EVENT HANDLERS
 ****************************************************************************************
 */
/// Format of an event handler function
typedef void (*evt_ptr_t)(int);

/// Structure defining an event callback
struct ke_evt_tag
{
    /// Pointer to the event function to call
    evt_ptr_t func;
    /// Parameter to pass to the event function
    int       param;
};

// Include files used to retrieve the event handler function pointers
#include "ke_task.h"
#include "ke_timer.h"
#include "ipc_emb.h"
#include "macif.h"
#include "hal_dma.h"

#include "txl_cntrl.h"
#include "rxl_cntrl.h"
#if NX_UMAC_PRESENT
#include "rxu_cntrl.h"
#endif
#if NX_TX_FRAME
#include "txl_frame.h"
#endif
#include "txl_cfm.h"
#include "mm.h"
#include "rwnx.h"

/// Table of event handlers
static const struct ke_evt_tag ke_evt_hdlr[32] =
{
    [KE_EVT_RESET         ]     = {&rwnxl_reset_evt, 0},
    #if NX_GP_DMA
    [KE_EVT_GP_DMA_DL     ]     = {&hal_dma_evt, DMA_DL},
    [KE_EVT_GP_DMA_UL     ]     = {&hal_dma_evt, DMA_UL},
    #endif
    #if NX_MM_TIMER
    [KE_EVT_MM_TIMER      ]     = {&mm_timer_schedule, 0},
    #endif
    [KE_EVT_KE_TIMER      ]     = {&ke_timer_schedule, 0},
    [KE_EVT_KE_MESSAGE    ]     = {&ke_task_schedule, 0},
    [KE_EVT_HW_IDLE       ]     = {&mm_hw_idle_evt, 0},
    #if NX_BEACONING || ((NX_POWERSAVE || NX_CONNECTION_MONITOR || NX_UMAC_PRESENT) && !NX_MULTI_ROLE)
    [KE_EVT_PRIMARY_TBTT  ]     = {&mm_tbtt_evt, 0},
    #endif
    #if NX_BEACONING
    [KE_EVT_SECONDARY_TBTT]     = {&mm_tbtt_evt, 0},
    #endif
    #if NX_UMAC_PRESENT
    [KE_EVT_RXUREADY      ]     = {&rxu_cntrl_evt, 0},
    #endif
    [KE_EVT_RXREADY       ]     = {&rxl_cntrl_evt, 0},
    [KE_EVT_RXUPLOADED    ]     = {&rxl_dma_evt, 0},
    #if NX_TX_FRAME
    [KE_EVT_TXFRAME_CFM   ]     = {&txl_frame_evt, 0},
    #endif
    #if NX_BEACONING
    [KE_EVT_MACIF_TXDESC_BCN]     = {&macif_tx_evt, AC_BCN},
    #endif
    [KE_EVT_MACIF_TXDESC_AC3]     = {&macif_tx_evt, AC_VO},
    [KE_EVT_MACIF_TXDESC_AC2]     = {&macif_tx_evt, AC_VI},
    [KE_EVT_MACIF_TXDESC_AC1]     = {&macif_tx_evt, AC_BE},
    [KE_EVT_MACIF_TXDESC_AC0]     = {&macif_tx_evt, AC_BK},
    [KE_EVT_MACIF_MSG       ]     = {&macif_msg_evt, 0},
    #if NX_BEACONING
    [KE_EVT_TXCFM_BCN     ]     = {&txl_cfm_evt, AC_BCN},
    #endif
    [KE_EVT_TXCFM_AC3     ]     = {&txl_cfm_evt, AC_VO},
    [KE_EVT_TXCFM_AC2     ]     = {&txl_cfm_evt, AC_VI},
    [KE_EVT_TXCFM_AC1     ]     = {&txl_cfm_evt, AC_BE},
    [KE_EVT_TXCFM_AC0     ]     = {&txl_cfm_evt, AC_BK},
};


/**
 ****************************************************************************************
 * @brief Set events
 *
 * This primitive sets one or more events in the event field variable. It will trigger
 * the call to the corresponding event handlers in the next scheduling call.
 *
 * @param[in]  event       Events that have to be set (bit field).
 *
 ****************************************************************************************
 */
void ke_evt_set(evt_field_t const event)
{
    GLOBAL_INT_DISABLE();
    ke_env.evt_field |= event;
    GLOBAL_INT_RESTORE();
}

/**
 ****************************************************************************************
 * @brief Clear events
 *
 * This primitive clears one or more events in the event field variable.
 *
 * @param[in]  event       Events that have to be cleared (bit field).
 *
 ****************************************************************************************
 */
void ke_evt_clear(evt_field_t const event)
{
    GLOBAL_INT_DISABLE();
    ke_env.evt_field &= ~event;
    GLOBAL_INT_RESTORE();
}


/**
 ****************************************************************************************
 * @brief Event scheduler entry point.
 *
 * This primitive has to be called in the background loop in order to execute the event
 * handlers for the event that are set.
 *
 ****************************************************************************************
 */
void ke_evt_schedule(void)
{
    // Get the volatile value
    uint32_t field = ke_env.evt_field;

    while (field) // Compiler is assumed to optimize with loop inversion
    {
        // Find highest priority event set
        uint32_t event = co_clz(field);

        // Sanity check
        ASSERT_ERR((event < KE_EVT_MAX) && ke_evt_hdlr[event].func);

        // Execute corresponding handler
        (ke_evt_hdlr[event].func)(ke_evt_hdlr[event].param);

        // Update the volatile value
        field = ke_env.evt_field;
    }
}

/**
 ****************************************************************************************
 * @brief This function performs all the initializations of the kernel.
 *
 * It initializes first the heap, then the message queues and the events. Then if required
 * it initializes the trace.
 *
 ****************************************************************************************
 */
void ke_init(void)
{
    // ke_mem_init MUST be called first to be able to allocate memory right from start
    #if KE_MEM_NX
    ke_env.mblock_first = ke_mem_init();

    #if KE_PROFILING
    ke_env.max_heap_used = 0;
    #endif //KE_PROFILING
    #endif //KE_MEM_RW

    // initialize the kernel message queue, mandatory before any message can be transmitted
    ke_env.queue_saved.first = NULL;
    ke_env.queue_saved.last = NULL;
    ke_env.queue_sent.first = NULL;
    ke_env.queue_sent.last = NULL;
    ke_env.queue_timer.first = NULL;
    ke_env.queue_timer.last = NULL;

    // clears all possible pending events
    ke_evt_clear(0xFFFFFFFF);
}

/**
 ****************************************************************************************
 * @brief This function flushes all messages, timers and events currently pending in the
 * kernel.
 *
 ****************************************************************************************
 */
void ke_flush(void)
{
    // free all pending message(s)
    while(1)
    {
        struct ke_msg *msg = (struct ke_msg*) ke_queue_pop(&ke_env.queue_sent);
        if(msg == NULL)
            break;
        ke_msg_free(msg);
    }
    // free all saved message(s)
    while(1)
    {
        struct ke_msg *msg = (struct ke_msg*) ke_queue_pop(&ke_env.queue_saved);
        if(msg == NULL)
            break;
        ke_msg_free(msg);
    }
    // free all timers
    while(1)
    {
        struct ke_timer *timer = (struct ke_timer*) ke_queue_pop(&ke_env.queue_timer);
        if(timer == NULL)
            break;
        ke_free(timer);
    }

    // clears all possible pending events
    ke_evt_clear(0xFFFFFFFF);
}

///@} EVT
