/**
 ****************************************************************************************
 *
 * @file ke_mem.c
 *
 * @brief Implementation of the heap management module.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup KE_MEM
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "co_int.h"
#include "co_bool.h"
#include "arch.h"
#include "dbg_assert.h"

#include "co_math.h"
#include "ke_config.h"
#include "ke_env.h"
#include "ke_mem.h"


/*
 * DEFINES
 ****************************************************************************************
 */
/// Heap size for nX
#ifdef CFG_RWTL
#define KE_HEAP_SIZE     NX_HEAP_SIZE/2
#else
#define KE_HEAP_SIZE     NX_HEAP_SIZE
#endif

/// Free memory block delimiter structure (size must be word multiple)
struct mblock_free
{
    struct mblock_free *next;   ///< Pointer to the next block
    #if CPU_WORD_SIZE == 4
    uint32_t size;              ///< Size of the current free block (including delimiter)
    #elif CPU_WORD_SIZE == 2
    uint16_t size;              ///< Size of the current free block (including delimiter)
    #endif
};

/// Used memory block delimiter structure (size must be word multiple)
struct mblock_used
{
    uint32_t size;              ///< Size of the current used block (including delimiter)
};

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
/// Data area reserved for the kernel heap
#ifdef CFG_RWTL
uint16_t ke_mem_heap[KE_HEAP_SIZE];
#else
uint8_t ke_mem_heap[KE_HEAP_SIZE];
#endif

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
struct mblock_free *ke_mem_init(void)
{
    struct mblock_free *first;

    // align first free descriptor to word boundary
    #if CPU_WORD_SIZE == 4
    first = (struct mblock_free*)CO_ALIGN4_HI((uint32_t)ke_mem_heap);
    #elif CPU_WORD_SIZE == 2
    first = (struct mblock_free*)CO_ALIGN2_HI((uint16_t)ke_mem_heap);
    #else
    #error No word size defined
    #endif

    // protect accesses to descriptors
    GLOBAL_INT_DISABLE();

    // initialize the first block
    //  + compute the size from the last aligned word before heap_end
    first->size = ((uint32_t)&ke_mem_heap[KE_HEAP_SIZE] & (~3)) - (uint32_t)first;
    first->next = NULL;

    // end of protection
    GLOBAL_INT_RESTORE();

    // save the pointer to the first free block
    return first;
}

void *ke_malloc(uint32_t size)
{
    struct mblock_free *node, *found;
    struct mblock_used *alloc;
    uint32_t totalsize;

    #if KE_PROFILING
    uint32_t totalfreesize = 0;
    #endif //KE_PROFILING

    // initialize the pointers
    found = NULL;

    // compute overall block size (including requested size PLUS descriptor size)
    totalsize = CO_ALIGN4_HI(size) + sizeof(struct mblock_used);

    // sanity check: the totalsize should be large enough to hold free block descriptor
    ASSERT_ERR(totalsize >= sizeof(struct mblock_free));

    node = ke_env.mblock_first;

    // protect accesses to descriptors
    GLOBAL_INT_DISABLE();

    // go through free memory blocks list
    while (node != NULL)
    {
        #if KE_PROFILING
        totalfreesize += node->size;
        #endif //KE_PROFILING

        // check if there is enough room in this free block
        if (node->size >= (totalsize + sizeof(struct mblock_free)))
        {
            // if a match was already found, check if this one is smaller
            if ((found == NULL) || (found->size > node->size))
            {
                found = node;
            }
        }
        // move to next block
        node = node->next;
    }

    // sanity check: allocation should always succeed
    ASSERT_ERR(found != NULL);

    #if KE_PROFILING
    if(ke_env.max_heap_used <= KE_HEAP_SIZE - totalfreesize)
        ke_env.max_heap_used = KE_HEAP_SIZE - totalfreesize;
    #endif //KE_PROFILING

    // found a free block that matches, subtract the allocation size from the
    // free block size. If equal, the free block will be kept with 0 size... but
    // moving it out of the linked list is too much work.
    found->size -= totalsize;

    // compute the pointer to the beginning of the free space
    #if CPU_WORD_SIZE == 4
    alloc = (struct mblock_used*) ((uint32_t)found + found->size);
    #elif CPU_WORD_SIZE == 2
    alloc = (struct mblock_used*) ((uint16_t)found + found->size);
    #endif

    // save the size of the allocated block (use low bit to indicate mem type)
    alloc->size = totalsize;

    // move to the user memory space
    alloc++;

    // end of protection (as early as possible)
    GLOBAL_INT_RESTORE();

    return (void*)alloc;
}


void ke_free(void* mem_ptr)
{
    struct mblock_used *freed;
    struct mblock_free *node, *prev_node, *next_node;
    uint32_t size;

    // point to the block descriptor (before user memory so decrement)
    freed = ((struct mblock_used *)mem_ptr) - 1;

    // point to the first node of the free elements linked list
    size = freed->size;
    node = ke_env.mblock_first;
    prev_node = NULL;

    // sanity checks
    ASSERT_ERR(mem_ptr != NULL);
    ASSERT_ERR((uint32_t)mem_ptr > (uint32_t)node);

    // protect accesses to descriptors
    GLOBAL_INT_DISABLE();

    while (node != NULL)
    {
        // check if the freed block is right after the current block
        if ((uint32_t)freed == ((uint32_t)node + node->size))
        {
            // append the freed block to the current one
            node->size += size;

            // check if this merge made the link between free blocks
            if ((uint32_t)node->next == ((uint32_t)node + node->size))
            {
                next_node = node->next;
                // add the size of the next node to the current node
                node->size += next_node->size;
                // update the next of the current node
                node->next = next_node->next;
            }
            goto free_end;
        }
        else if ((uint32_t)freed < (uint32_t)node)
        {
            // sanity check: can not happen before first node
            ASSERT_ERR(prev_node != NULL);

            // update the next pointer of the previous node
            prev_node->next = (struct mblock_free*)freed;

            // check if the released node is right before the free block
            if (((uint32_t)freed + size) == (uint32_t)node)
            {
                // merge the two nodes
                ((struct mblock_free*)freed)->next = node->next;
                ((struct mblock_free*)freed)->size = node->size + (uint32_t)node - (uint32_t)freed;
            }
            else
            {
                // insert the new node
                ((struct mblock_free*)freed)->next = node;
                ((struct mblock_free*)freed)->size = size;
            }
            goto free_end;
        }

        // move to the next free block node
        prev_node = node;
        node = node->next;
    }
    // if reached here, freed block is after last free block and not contiguous
    prev_node->next = (struct mblock_free*)freed;
    ((struct mblock_free*)freed)->next = NULL;
    ((struct mblock_free*)freed)->size = size;

free_end:
    // end of protection
    GLOBAL_INT_RESTORE();
}

/// @}
