/**
 ****************************************************************************************
 *
 * @file macif_fhost.c
 *
 * @brief MACIF implementation when control layer is running locally.
 *
 * Copyright (C) RivieraWaves 2017-2019
 *
 ****************************************************************************************
 */
#include "macif.h"
#include "rtos.h"
#include "fhost.h"
#include "fhost_cntrl.h"
#include "fhost_rx.h"
#include "fhost_tx.h"
#include "ke_mem.h"
#include "ke_task.h"
#include "ke_event.h"
#include "rxu_cntrl.h"
#include "txu_cntrl.h"
#include "txl_cntrl.h"
#if NX_AMPDU_TX
#include "txl_agg.h"
#endif

/*
 * TYPE DEFINITIONS
 ****************************************************************************************
 */
/// MACIF FHOST environment structure
struct macif_fhost_env_tag
{
    /// Flag indicating that the WiFi RX module is currently waiting for RX buffers
    /// and shall be triggered when new buffers are pushed
    bool rx_flow_off;
};


/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
/// MACIF FHOST context
static struct macif_fhost_env_tag macif_fhost_env;

/// Array of bits event to be triggered per TX queue index
const uint32_t macif_evt_bit[NX_TXQ_CNT] =
{
    #if NX_BEACONING
    [AC_BCN] = KE_EVT_MACIF_TXDESC_BCN_BIT,
    #endif
    [AC_VO] = KE_EVT_MACIF_TXDESC_AC3_BIT,
    [AC_VI] = KE_EVT_MACIF_TXDESC_AC2_BIT,
    [AC_BE] = KE_EVT_MACIF_TXDESC_AC1_BIT,
    [AC_BK] = KE_EVT_MACIF_TXDESC_AC0_BIT,
};

/// High priority kernel that should interrupt the IPC TX event
#if NX_BEACONING
#define HIGH_PRIO_EVT (KE_EVT_RXREADY_BIT | KE_EVT_PRIMARY_TBTT_BIT | KE_EVT_MACIF_TXDESC_BCN_BIT)
#else
#define HIGH_PRIO_EVT (KE_EVT_RXREADY_BIT | KE_EVT_PRIMARY_TBTT_BIT)
#endif

/// TBD pattern value stored in shared RAM, in order to be able to DMA it
uint32_t macif_tbd_pattern __SHAREDRAM;

/*
 * FUNCTIONS
 ****************************************************************************************
 */
int macif_kmsg_push(int msg_id, int task_id, void *param, int param_len)
{
    struct ke_msg *kmsg = ke_malloc(sizeof(struct ke_msg) + param_len);
    if (kmsg == NULL)
        return -1;

    kmsg->hdr.next = NULL;
    kmsg->id = msg_id;
    kmsg->dest_id = task_id;
    kmsg->src_id = TASK_API;
    kmsg->param_len = param_len;

    // copy parameters
    memcpy(kmsg->param, param, param_len);

    // dispatch the message to the Wifi task
    ke_msg_send(ke_msg2param(kmsg));
    rtos_wifi_task_resume(false);

    return 0;
}

void macif_msg_evt(int dummy)
{
    /* not used with FHOST */
    ASSERT_ERR(0);
}

void macif_kmsg_fwd(const struct ke_msg *ke_msg)
{
    if (fhost_cntrl_write_msg(FHOST_MSG_ID(FHOST_MSG_KE_WIFI, 0),
                              (void *)ke_msg, sizeof(struct ke_msg) + ke_msg->param_len,
                              10, false))
    {
        ASSERT_WARN(0);
    }
}

void macif_prim_tbtt_ind(void)
{
    /* not used if NX_BCN_AUTONOMOUS_TX is defined, which will always be the case */
    ASSERT_ERR(0);
}

void macif_sec_tbtt_ind(void)
{
    /* not used if NX_BCN_AUTONOMOUS_TX is defined, which will always be the case */
    ASSERT_ERR(0);
}

int macif_init(void)
{
    // Reset the complete environment
    memset(&macif_fhost_env, 0, sizeof(macif_fhost_env));

    macif_tbd_pattern = TX_PAYLOAD_DESC_PATTERN;

    return 0;
}

bool macif_rx_buf_check(void)
{
    GLOBAL_INT_DISABLE();
    macif_fhost_env.rx_flow_off = rtos_queue_is_empty(fhost_rx_env.queue_buf);
    GLOBAL_INT_RESTORE();

    return !macif_fhost_env.rx_flow_off;
}

uint32_t macif_rx_buf_get(uint32_t *host_id)
{
    struct fhost_rx_buf_desc_tag buf;

    // Pop a buffer from the RX buffer queue (without waiting)
    rtos_queue_read(fhost_rx_env.queue_buf, &buf, 0, false);

    *host_id = buf.host_id;

    return buf.addr;
}

void macif_rx_buf_ind(void)
{
    if (macif_fhost_env.rx_flow_off)
    {
        GLOBAL_INT_DISABLE();
        ke_evt_set(KE_EVT_RXREADY_BIT);
        GLOBAL_INT_RESTORE();
        rtos_wifi_task_resume(false);
    }
}

void macif_rx_desc_upload(struct co_list *desc_list)
{
    struct rxu_stat_desc *rx_stat_desc = (struct rxu_stat_desc *)co_list_pick(desc_list);

    // Go through the list of descriptors until one has a non-NULL SW descriptor. Indeed
    // this means that the payload transfer attached to this descriptor is not complete
    // and the descriptor transfer will then be triggered upon the end of the payload
    // upload
    while ((rx_stat_desc != NULL) && (rx_stat_desc->rxdesc == NULL))
    {
        int res __MAYBE_UNUSED;

        // Push a message in the queue
        res = rtos_queue_write(fhost_rx_env.queue_desc, &rx_stat_desc->val, 0, false);
        ASSERT_ERR(res == 0);

        // Extract the descriptor from the list
        co_list_pop_front(desc_list);

        // Free it by clearing its status
        rx_stat_desc->val.status = 0;

        // Get next descriptor
        rx_stat_desc = (struct rxu_stat_desc *)co_list_pick(desc_list);
    }
}

#if NX_UF_EN
void macif_uf_ind(struct rx_vector_desc *rx_vector)
{
    int res;
    struct fhost_rx_uf_buf_desc_tag buftag;
    struct fhost_rx_uf_buf_tag *buf;
    struct rxu_stat_val val;

    // Pop a buffer from the RX buffer queue
    rtos_queue_read(fhost_rx_env.queue_uf_buf, &buftag, -1, true);

    buf = (struct fhost_rx_uf_buf_tag *)buftag.host_id;

    // Get RX vector1 informations
    memcpy(&buf->info.vect.recvec1a, rx_vector->rx_vect1, sizeof(rx_vector->rx_vect1));
    // PHY channel used for RX
    buf->info.phy_info = rx_vector->phy_info;

    val.status = RX_STAT_UF;
    val.host_id = buftag.host_id;
    val.frame_len = 0;
    // Push the message in the queue
    res = rtos_queue_write(fhost_rx_env.queue_desc, &val, 0, true);
    ASSERT_ERR(res == 0);
}
#endif // NX_UF_EN

void macif_tx_data_ind(int queue_idx)
{
    // Set APP TX event
    ke_evt_set(macif_evt_bit[queue_idx]);
    rtos_wifi_task_resume(false);

    // Check if we have more than half of the queue filled. In such case we increase
    // the WiFi task priority in order to flush the queue
    if (rtos_queue_cnt(fhost_tx_env.queue_tx[queue_idx]) >= (nx_txdesc_cnt[queue_idx]/2))
    {
        rtos_wifi_task_prio_high();
    }
}

void macif_tx_evt(int queue_idx)
{
    uint32_t evt_bit = macif_evt_bit[queue_idx];
    #if NX_AMPDU_TX
    bool pushed = false;
    #endif // NX_AMPDU_TX

    // For profiling
    PROF_TX_MACIF_EVT_SET();

    // Clear the event that was just triggered
    ke_evt_clear(evt_bit);

    // Loop on all the descriptors ready for TX
    while (!rtos_queue_is_empty(fhost_tx_env.queue_tx[queue_idx]))
    {
        struct txdesc *txdesc;

        // Check if we have higher priority event pending
        #if NX_BEACONING
        if (!(evt_bit & KE_EVT_MACIF_TXDESC_BCN_BIT))
        #endif
        {
            if (ke_evt_get() & HIGH_PRIO_EVT)
            {
                // Set again the event that was just triggered
                ke_evt_set(evt_bit);
                // For profiling
                PROF_TX_MACIF_EVT_CLR();
                return;
            }
        }

        // Get the descriptor from the queue
        rtos_queue_read(fhost_tx_env.queue_tx[queue_idx], &txdesc, -1, false);

        // Initialize some fields of the descriptor
        txdesc->lmac.agg_desc = NULL;
        txdesc->host.cfm.status = 0;
        #if (RW_BFMER_EN)
        txdesc->lmac.bfr_node = NULL;
        #endif //(RW_BFMER_EN)
        txdesc->umac.buf_control = NULL;

        // Push the descriptor in the UMAC
        txu_cntrl_push(txdesc, queue_idx);

        #if NX_AMPDU_TX
        // Now we have pushed some payloads so we may need to trigger the prepare event
        pushed = true;
        #endif // NX_AMPDU_TX
    }

    #if NX_AMPDU_TX
    // Check if some payloads have been pushed and have to be downloaded
    if (pushed)
    {
        // Check if the current A-MPDU under construction has to be closed
        txl_agg_check(queue_idx);
    }
    #endif // NX_AMPDU_TX

    // For profiling
    PROF_TX_MACIF_EVT_CLR();

    // Now that we have flushed the queue we can reduce the WiFi task priority
    rtos_wifi_task_prio_default();
}

bool macif_tx_q_has_data(int queue_idx)
{
    return (!rtos_queue_is_empty(fhost_tx_env.queue_tx[queue_idx]));
}

uint32_t macif_tx_pattern_addr_get(void)
{
    return CPU2HW(&macif_tbd_pattern);
}

uint32_t macif_buffered_get(uint8_t sta, uint8_t tid)
{
    return 0xFFFFFFFF;
}

void macif_tx_cfm_start(uint8_t access_category)
{
}

void macif_tx_cfm_push(uint8_t access_category, struct txdesc *txdesc)
{
    #if NX_POWERSAVE
    // Decrease the number of packets in the TX path
    txl_cntrl_env.pck_cnt--;
    #endif

    // Push the confirmation to the FHOST
    fhost_tx_cfm_push(access_category, txdesc);
}

void macif_tx_cfm_done(uint8_t access_category, bool poll)
{
}


