/**
 ****************************************************************************************
 *
 * @file macif_ipc.c
 *
 * @brief MACIF implementation when control layer is running on a remote host.
 *
 * Copyright (C) RivieraWaves 2017-2019
 *
 ****************************************************************************************
 */

#include "ipc_emb.h"
#include "dma.h"
#include "macif.h"
#include "ps.h"
#include "rxl_cntrl.h"
#if NX_UMAC_PRESENT
#include "rxu_cntrl.h"
#endif
#include "txl_buffer.h"


/// Maximum number of confirmation uploads that can be pending simultaneously
/// This value shall be a power of 2
#define CFM_PENDING_MAX   16
/// Mask of the confirmation indexes
#define CFM_IDX_MSK (CFM_PENDING_MAX - 1)

/// Offset of the status element in the TX buffer
#define CFM_STATUS_OFFSET   offsetof_b(struct txl_buffer_control, status)

/// Number of prepared confirmations after which we program the DMA to transmit them
#define CFM_CNT_THRESHOLD   8

/// valid pattern for buffers pushed by the driver
#define UF_RX_VEC_VALID_PATTERN 0x0000C0DE

/// Conversion from Access Category to corresponding bridge DMA control field
#define CFM_LLICTRL (IPC_DMA_LLI_COUNTER_EN | (IPC_DMA_LLI_CFM_TX << IPC_DMA_LLI_COUNTER_POS) | \
                     IPC_DMA_LLI_IRQ_EN | (IPC_DMA_LLI_CFM_TX << IPC_DMA_LLI_IRQ_POS))


/*
 * TYPE DEFINITIONS
 ****************************************************************************************
 */
/// MACIF TX confirmation control structure
struct macif_tx_cfm_tag
{
    /// Pointer to the first DMA descriptor to upload
    struct dma_desc *first;
    /// Pointer to the current DMA descriptor to upload
    struct dma_desc *curr;
    /// Bitfield used for the confirmation
    uint32_t cfm_ind;
    /// Number of confirmations pending for upload
    int cfm_cnt;
    /// Array of pending confirmation indications
    uint32_t user_cfm[CFM_PENDING_MAX];
    /// Target LLI count to be reached to indicate the confirmation to host
    uint16_t lli_cnt;
    /// In index for the pending confirmation array
    uint8_t in_idx;
};

/// MACIF IPC environment structure
struct macif_ipc_env_tag
{
    /// Structure grouping the information required for the TX confirmation upload
    struct macif_tx_cfm_tag tx_cfm;
    #if NX_UF_EN
    /// List of free unsupported rx vectors
    struct co_list unsup_rx_vec_free_list;
    #endif //NX_UF_EN
};

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
/// MACIF IPC context
struct macif_ipc_env_tag macif_ipc_env;

/*
 * FUNCTIONS
 ****************************************************************************************
 */
#if NX_UF_EN
/**
 ****************************************************************************************
 * @brief UF platform DMA handler.
 ****************************************************************************************
 */
static void macif_ipc_uf_dma_handler(void *env, int dma_type)
{
    // Retrieve rx vector from env
    struct unsup_rx_vector_desc *desc = (struct unsup_rx_vector_desc*)env;

    // Push it back to the free list - Must be done in critical section as the list is
    // also manipulated under interrupt
    GLOBAL_INT_DISABLE();
    co_list_push_back(&macif_ipc_env.unsup_rx_vec_free_list, &desc->list_hdr);
    GLOBAL_INT_RESTORE();

    // call the LMAC-UMAC interface function to handle the frame
    ipc_emb_unsup_rx_vec_event_ind();
}
#endif //NX_UF_EN

int macif_init(void)
{
    #if NX_UF_EN
    int i;
    #endif //NX_UF_EN

    // Reset the complete environment
    memset(&macif_ipc_env, 0, sizeof(macif_ipc_env));

    macif_ipc_env.tx_cfm.lli_cnt = dma_lli_counter_get(IPC_DMA_LLI_CFM_TX) + 1;
    macif_ipc_env.tx_cfm.in_idx = macif_ipc_env.tx_cfm.lli_cnt & CFM_IDX_MSK;

    #if NX_UF_EN
    /// Initialize rx vector list
    co_list_init(&macif_ipc_env.unsup_rx_vec_free_list);

    for (i = 0; i < UNSUP_RX_VECT_MAX; i++)
    {
        struct unsup_rx_vector_desc *desc = &rx_vector_desc_array[i];

        // Initialize descriptors
        desc->dma_desc.src = CPU2HW(&desc->rx_vector);
        desc->dma_desc.length = sizeof_b(desc->rx_vector);

        desc->gp_dma_desc.dma_desc = &desc->dma_desc;
        desc->gp_dma_desc.cb = macif_ipc_uf_dma_handler;
        desc->gp_dma_desc.env = desc;

        co_list_push_back(&macif_ipc_env.unsup_rx_vec_free_list, &desc->list_hdr);
    }
    #endif //NX_UF_EN

    return 0;
}

void macif_msg_evt(int dummy)
{
    return ipc_emb_msg_evt(dummy);
}

void macif_kmsg_fwd(const struct ke_msg *ke_msg)
{
    return ipc_emb_kmsg_fwd(ke_msg);
}

void macif_prim_tbtt_ind(void)
{
    return ipc_emb_prim_tbtt_ind();
}

void macif_sec_tbtt_ind(void)
{
    return ipc_emb_sec_tbtt_ind();
}

uint8_t macif_rx_get_packet_threshold(void)
{
    return (IPC_RXBUF_CNT / 4);
}

void macif_rx_data_ind(void)
{
    ipc_emb_rxdata_ind();
}

bool macif_rx_buf_check(void)
{
     return ipc_emb_hostrxbuf_check();
}

#if NX_UMAC_PRESENT
uint32_t macif_rx_buf_get(uint32_t *host_id)
{
    return ipc_emb_hostrxbuf_get(host_id);
}
#else
uint32_t macif_rx_buf_get(void)
{
    return ipc_emb_hostrxbuf_get();
}
#endif //(NX_UMAC_PRESENT)

#if NX_UMAC_PRESENT
void macif_rx_desc_upload(struct co_list *desc_list)
{
    struct rxu_stat_desc *rx_stat_desc = (struct rxu_stat_desc *)co_list_pick(desc_list);
    // First DMA descriptor
    struct dma_desc *first_dma_desc= NULL;
    // Last DMA descriptor
    struct dma_desc *last_dma_desc = NULL;

    while (rx_stat_desc)
    {
        uint32_t dma_addr;

        PROF_IPCDESC_TRANSFER_SET();

        // Check if we have descriptors available in the host memory
        if (!ipc_emb_hostrxdesc_check())
        {
            break;
        }

        // Get DMA address
        dma_addr = ipc_emb_hostrxdesc_get();

        if (!first_dma_desc)
        {
            first_dma_desc = &rx_stat_desc->dma_desc;
        }

        rx_stat_desc->dma_desc.dest = dma_addr;

        if (last_dma_desc)
        {
            last_dma_desc->next = CPU2HW(&rx_stat_desc->dma_desc);
        }

        last_dma_desc = &rx_stat_desc->dma_desc;

        // Remove the descriptor from the list
        co_list_pop_front(desc_list);

        // Insert the element in the pending list
        rxl_upload_cntrl_push_pending(&rx_stat_desc->upload_cntrl);

        PROF_IPCDESC_TRANSFER_CLR();

        // Get next element
        rx_stat_desc = (struct rxu_stat_desc *)co_list_pick(desc_list);
    }

    if (first_dma_desc)
    {
        last_dma_desc->next = 0;

        // Push the DMA descriptor in the DMA engine - (Last next pointer set to NULL in dma_push)
        dma_push(first_dma_desc, last_dma_desc, RX_DATA_UPLOAD_CHAN);
    }
}
#endif

#if NX_UF_EN
void macif_uf_ind(struct rx_vector_desc *rx_vector)
{
    struct unsup_rx_vector_desc *desc;
    uint32_t hostbuf;

    // Get a free rx vector descriptor
    desc = (struct unsup_rx_vector_desc *)co_list_pop_front(&macif_ipc_env.unsup_rx_vec_free_list);
    if (desc == NULL)
        return;

    // Get a host buffer for this event
    hostbuf = ipc_emb_hostunsuprxvectbuf_get();
    if (hostbuf == 0)
    {
        co_list_push_back(&macif_ipc_env.unsup_rx_vec_free_list, &desc->list_hdr);
        return;
    }

    // Initialize the descriptor
    desc->dma_desc.dest = hostbuf;
    desc->rx_vector = *rx_vector;
    desc->rx_vector.pattern = UF_RX_VEC_VALID_PATTERN;

    // Program the DMA transfer to the host
    hal_dma_push(&desc->gp_dma_desc, DMA_UL);
}
#endif //NX_UF_EN

void macif_tx_evt(int queue_idx)
{
    return ipc_emb_tx_evt(queue_idx);
}

bool macif_tx_q_has_data(int queue_idx)
{
    return ipc_emb_tx_q_has_data(queue_idx);
}

uint8_t macif_tx_q_len(int queue_idx, int vif_idx)
{
    return ipc_emb_tx_q_len(queue_idx, vif_idx);
}

uint32_t macif_tx_pattern_addr_get(void)
{
    return ipc_emb_tx_pattern_addr_get();
}

uint32_t macif_buffered_get(uint8_t sta, uint8_t tid)
{
    return ipc_emb_buffered_get(sta, tid);
}

#if RW_MUMIMO_TX_EN
void macif_tx_enable_users(int queue_idx, uint8_t active_users)
{
    return ipc_emb_enable_users(queue_idx, active_users);
}
#endif

/**
 ****************************************************************************************
 * @brief   This function checks if the IPC DMA already processed the transfers up to the
 * target count.
 *
 * @param[in] next_lli_cnt  The target LLI count
 *
 * @return true if the target is reached or exceeded, false otherwise
 *
 ****************************************************************************************
 */
static bool macif_tx_cfm_lli_done(uint16_t next_lli_cnt)
{
    return (((uint16_t)(dma_lli_counter_get(IPC_DMA_LLI_CFM_TX) - (next_lli_cnt)))
                                                                 < (((uint16_t)-1) / 2));
}

/**
 ****************************************************************************************
 * @brief Initialize a new TX confirmation upload process.
 * This process shall be closed using @ref macif_tx_cfm_done before starting a new session.
 *
 * @param[in] access_category  Access category for the confirmations
 *
 ****************************************************************************************
 */
void macif_tx_cfm_start(uint8_t access_category)
{
    macif_ipc_env.tx_cfm.first = NULL;
    macif_ipc_env.tx_cfm.curr = NULL;
    #if RW_MUMIMO_TX_EN
    macif_ipc_env.tx_cfm.cfm_ind = 0;
    #else
    macif_ipc_env.tx_cfm.cfm_ind = CO_BIT(access_category);
    #endif
    macif_ipc_env.tx_cfm.cfm_cnt = 0;
}

void macif_tx_cfm_push(uint8_t access_category, struct txdesc *txdesc)
{
    struct dma_desc *desc = &txdesc->lmac.hw_desc->dma_desc;
    struct macif_tx_cfm_tag *tx_cfm = &macif_ipc_env.tx_cfm;

    #if NX_UMAC_PRESENT
    // Fill in the confirmation DMA descriptor
    desc->dest = txdesc->host.status_desc_addr;
    #else
    // Fill in the confirmation DMA descriptor
    #if NX_AMSDU_TX
    desc->dest = txdesc->host.packet_addr[0] + CFM_STATUS_OFFSET;
    #else
    desc->dest = txdesc->host.packet_addr + CFM_STATUS_OFFSET;
    #endif
    #endif
    desc->ctrl = 0;

    #if (RW_MUMIMO_TX_EN)
    // Set user position bit that needs to be confirmed
    tx_cfm->cfm_ind |= CO_BIT(get_user_pos(txdesc) + access_category * RW_USER_MAX);
    #endif

    // Check if the TX descriptor is the first to be confirmed
    if (tx_cfm->first == NULL)
    {
        tx_cfm->first = desc;
    }
    else
    {
        tx_cfm->curr->next = CPU2HW(desc);
    }

    // Update current descriptor pointer
    tx_cfm->curr = desc;

    // Increment the number of confirmed descriptors
    tx_cfm->cfm_cnt++;

    // Check if we have reached the threshold
    if (tx_cfm->cfm_cnt >= CFM_CNT_THRESHOLD)
    {
        // Send the prepared confirmations
        macif_tx_cfm_done(access_category, false);

        // Reset the counter and the first and current pointers
        tx_cfm->cfm_cnt = 0;
        tx_cfm->first = NULL;
        tx_cfm->curr = NULL;
    }

    #if NX_POWERSAVE
    // Decrease the number of packets in the TX path
    txl_cntrl_env.pck_cnt--;
    #endif
}

void macif_tx_cfm_done(uint8_t access_category, bool poll)
{
    struct macif_tx_cfm_tag *tx_cfm = &macif_ipc_env.tx_cfm;
    // Check if some confirmations have to be programmed
    if (tx_cfm->first != NULL)
    {
        // Enable an interrupt on the AC
        tx_cfm->curr->ctrl = CFM_LLICTRL;

        #if NX_POWERSAVE
        GLOBAL_INT_DISABLE();
        ps_env.prevent_sleep |= PS_TX_CFM_UPLOADING;
        #endif

        // Set the confirmation bits into the array
        ASSERT_ERR(tx_cfm->user_cfm[tx_cfm->in_idx] == 0);
        tx_cfm->user_cfm[tx_cfm->in_idx] = tx_cfm->cfm_ind;
        tx_cfm->in_idx = (tx_cfm->in_idx + 1) & CFM_IDX_MSK;

        // Program the DMA
        dma_push(tx_cfm->first, tx_cfm->curr, RX_DATA_UPLOAD_CHAN);

        if (poll)
        {
            // Poll for the completion of the transfer
            dma_lli_poll(IPC_DMA_LLI_CFM_TX);

            // Execute the interrupt handler to forward the interrupt to the upper MAC
            macif_tx_cfm_dma_int_handler();
        }

        #if NX_POWERSAVE
        GLOBAL_INT_RESTORE();
        #endif
    }
}

void macif_tx_cfm_dma_int_handler(void)
{
    struct macif_tx_cfm_tag *tx_cfm = &macif_ipc_env.tx_cfm;

    // read the currently pending DMA interrupt and mask only the LLI IRQs for CFM
    uint32_t irqstatus = dma_int_status_get() & IPC_DMA_LLI_CFM_MASK;

    // For profiling
    PROF_TX_CFM_DMA_IRQ_SET();

    // acknowledge the DMA interrupt
    dma_int_ack_clear(irqstatus);

    // Loop until we reach the DMA LLI counter
    while (macif_tx_cfm_lli_done(tx_cfm->lli_cnt))
    {
        // Compute the out index of the array from the LLI
        uint8_t out_idx = tx_cfm->lli_cnt & CFM_IDX_MSK;

        // acknowledge the DMA interrupt
        dma_int_ack_clear(irqstatus);

        // indicate the confirmations to the host
        ipc_emb_txcfm_ind(tx_cfm->user_cfm[out_idx]);
        tx_cfm->user_cfm[out_idx] = 0;

        // Increase the target LLI counter
        tx_cfm->lli_cnt++;
    }

    #if NX_POWERSAVE
    ps_env.prevent_sleep &= ~PS_TX_CFM_UPLOADING;
    #endif

    // For profiling
    PROF_TX_CFM_DMA_IRQ_CLR();
}


