/**
 ****************************************************************************************
 *
 * @file boot.c
 *
 * @brief This file contains the startup code for the Cortex-Mx processors
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup BOOT
 * @ingroup ARCH
 * @{
 * ****************************************************************************************
 */
/*
 * INCLUDES
 ****************************************************************************************
 */
#include <stdint.h>
#include "intc.h"
#include "dbg_assert.h"

/* Declare linker-defined symbols. The only thing of interest
   regarding these symbols is their *address*. uint32_t hints
   of alignment. */
extern uint32_t _start_bss;
extern uint32_t _end_bss;
extern uint32_t _end_stack;

/*
 * FUNCTION DECLARATIONS
 ****************************************************************************************
 */
extern int wifi_main(void);
void Dummy_Handler(void);
/* Cortex-M core interrupt handlers */
void Reset_Handler(void);
#define ALIAS(sym) __attribute__((weak, alias (sym)))
void NMI_Handler(void) ALIAS("Dummy_Handler");
void HardFault_Handler(void) ALIAS("Dummy_Handler");
void MemManage_Handler(void) ALIAS("Dummy_Handler");
void BusFault_Handler(void) ALIAS("Dummy_Handler");
void UsageFault_Handler(void) ALIAS("Dummy_Handler");
void SVC_Handler(void) ALIAS("Dummy_Handler");
void DebugMon_Handler(void) ALIAS("Dummy_Handler");
void PendSV_Handler(void) ALIAS("Dummy_Handler");
void SysTick_Handler(void) ALIAS("Dummy_Handler");

/*
 * CORE VECTOR TABLE DEFINITION
 ****************************************************************************************
 */
/* 16 standard Cortex-M vectors - these are present in every MCU */
void *core_vector_table[16] __attribute__ ((section(".cortex_vectors"))) = {
    // See http://sourceware.org/binutils/docs/ld/Source-Code-Reference.html
    // why the address is used here (if not intuitive)
    &_end_stack,
    Reset_Handler,
    NMI_Handler,
    HardFault_Handler,
    MemManage_Handler,
    BusFault_Handler,
    UsageFault_Handler,
    0,
    0,
    0,
    0,
    SVC_Handler,
    DebugMon_Handler,
    0,
    PendSV_Handler,
    SysTick_Handler,
};

/*
 * VENDOR VECTOR TABLE DEFINITION (RW interrupt handler)
 ****************************************************************************************
 */
void *vendor_vector_table[] __attribute__ ((section(".vendor_vectors"))) = {
    intc_irq,
};


/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Start-up code.
 *
 * Based on http://sourceware.org/binutils/docs/ld/Output-Section-LMA.html
 *
 * This code is the C entry point of SW.
 *   1. Reset BSS
 *   2. Call wifi_main() function
 *****************************************************************************************
 */
void Reset_Handler(void) {
    register uint32_t *dst;

    /* Clear the bss section */
    dst = &_start_bss;
    while (dst < &_end_bss)
        *dst++ = 0;

    wifi_main();
    while (1);
}

/**
 ****************************************************************************************
 * @brief Dummy handler in case of unexpected exception.
 * Receiving an unexpected exception is considered a bug and triggers an assertion.
 *****************************************************************************************
 */
void Dummy_Handler(void) {
    ASSERT_ERR(0);
    while(1);
}

/// @}

