/**
 ****************************************************************************************
 *
 * @file crm.c
 *
 * @brief Clock and Reset Module definitions.
 *
 * Copyright (C) RivieraWaves 2018-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup CRM
 * @{
 ****************************************************************************************
 */
/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "crm.h"
#include "phy.h"
#include "dbg.h"
#include "reg_mac_core.h"
#include "reg_mdm_stat.h"
#include "reg_mdm_cfg.h"
#include "reg_karst_if.h"
#include <string.h>

#ifdef CFG_VIRTEX6
#include "reg_fpgaa.h"
#endif /* CFG_VIRTEX6 */

#if NX_MDM_VER < 20 || NX_MDM_VER >= 30
#error "This version of CRM is valid for modem version 2x only"
#endif

/// This macro shall be set to one if the MDM is supporting confs 8 and 9 and
/// to 0 otherwise
#define MDM_CONF_0_9    1
// For older MDM version only 8 configurations are available
#if NX_MDM_VER == 20
#undef MDM_CONF_0_9
#define MDM_CONF_0_9    0
#endif

/// Undefined value for the clock configuration
#define CRM_CLKCONF_UNDEFINED   ((uint8_t)-1)

/// CRM driver context structure.
struct crm_env_tag
{
    /// Current clock configuration index
    uint8_t clk_conf;
    /// Frequency of the MAC HW
    uint8_t mac_freq;
    /// Frequency (in HZ) of the CPU
    uint32_t cpu_freq;
};

/// CRM driver context.
static struct crm_env_tag crm_env;

/**
 ******************************************************************************
 * @brief Poll register until bits are set
 *
 * @param[in] reg  Register to poll
 * @param[in] val  Bits to wait
 *
 * @return 0 if bits are set in register, and !=0 otherwise
 ******************************************************************************
 */
static int reg_poll(uint32_t reg, uint32_t val)
{
    int loops = 1 << 20;

    while ((REG_PL_RD(reg) & val) != val && --loops)
        ;
    if (!loops)
    {
        dbg(D_ERR D_PHY "%s @0x%08x 0x%08x failed\n", __func__, reg, val);
        return -1;
    }
    return !loops;
}

/**
 ******************************************************************************
 * @brief Active wait
 *
 * @param[in] us Waiting duration (in us)
 ******************************************************************************
 */
static inline void udelay(uint32_t us)
{
    uint32_t e = nxmac_monotonic_counter_2_lo_get() + us;

    do {
        volatile int n = 1 << 5; // relax
        while (n--)
            ;
    } while ((int32_t)(nxmac_monotonic_counter_2_lo_get() - e) < 0);
}

#ifdef CFG_VIRTEX6
static bool p40m_ok(void)
{
    uint32_t pmref40, i;
    bool ok;

    pmref40 = karst_pmetre_ref40_get();
    ok = true;
    for (i = 0; i < 2 * sizeof(uint32_t); i++) {
        int dig = (pmref40 & ((uint32_t)0xF << (4 * i))) >> (4 * i);
        if (dig != 3 && dig != 4) {
            ok = false;
            break;
        }
    }
    return ok;
}
#endif /* CFG_VIRTEX6 */

/* Mixed Mode Clock Manager defines */
/* output addresses */
/// Address of output0
#define CLKOUT0  0x08
/// Address of output1
#define CLKOUT1  0x0A
/// Address of output2
#define CLKOUT2  0x0C
/// Address of output3
#define CLKOUT3  0x0E
/// Address of output4
#define CLKOUT4  0x10
/// Address of output5
#define CLKOUT5  0x06
/// Address of output6
#define CLKOUT6  0x12
/// Address of feedback output
#define CLKOUTFB 0x14


///  Address to configure clkreg1 reg for output @p out of mmcm @p mmcm
///  note: use KARST_DYNCFGMMC_xxx defined assuming that FPGAA_DYNCFGMMC_xxx
///       have the same values
#define MMCM_REG1(mmcm, out)                    \
    (KARST_DYNCFGMMC_WE_BIT |                   \
     (mmcm << KARST_DYNCFGMMC_SEL_LSB) |        \
     CLKOUT##out)

/// Multiplier to apply to @p clk_ref_freq to get @p vco_freq
#define MMCM_VCO_MULT(clk_ref_freq, vco_freq)     \
    ((((vco_freq/clk_ref_freq) + 1) / 2) << 6 |   \
     (((vco_freq/clk_ref_freq)) / 2) |            \
     0x1000)

/// Divider to apply to @p vco_freq to get @p target_freq
#define MMCM_VCO_DIV(vco_freq, target_freq)    \
    ((((vco_freq/target_freq) + 1) / 2) << 6 | \
     (((vco_freq/target_freq)) / 2) |          \
     0x1000)

/// Convert a VCO divider @p vco_div to its corresponding numeric value
#define MMCM_VCO_TO_NUM_DIV(vco_div) (((vco_div & 0x0FC0) >> 6) + (vco_div & 0x3F))

#if defined(CFG_VIRTEX6)
/* FPGA-A MMCM0 */
#define FPGAA_MMCM0_CLKIN 200
#define FPGAA_MMCM0_VCO   1200
#define MAC_CORE_CLK_ADDR   MMCM_REG1(0, 1)
#define MAC_WT_CLK_ADDR     MMCM_REG1(0, 2)
#define LP_120M_CLK_ADDR    MMCM_REG1(0, 3)
#define MAC_CORE_DIV(x)     MMCM_VCO_DIV(FPGAA_MMCM0_VCO, x)
#define MAC_WT_DIV(x)       MMCM_VCO_DIV(FPGAA_MMCM0_VCO, x)
#define LP_120M_DIV(x)      MMCM_VCO_DIV(FPGAA_MMCM0_VCO, x)
#define MAC_FREQ(x)         (FPGAA_MMCM0_VCO / MMCM_VCO_TO_NUM_DIV(x))

/* FPGA-A MMCM1 */
#define FPGAA_MMCM1_CLKIN 30
#define FPGAA_MMCM1_VCO   960
#define MPIF_FPGAA_CLK_ADDR MMCM_REG1(1, 1)
#define LA_CLK_ADDR         MMCM_REG1(1, 2)
#define LA_DIV(x)           MMCM_VCO_DIV(FPGAA_MMCM1_VCO, x)

/* FPGA-B MMCM0 */
#define KARST_MMCM0_CLKIN 30
#define KARST_MMCM0_VCO   960
#define MPIF_CLK_ADDR MMCM_REG1(0, 6)
#define MPIF_DIV(x)   MMCM_VCO_DIV(KARST_MMCM0_VCO, x)
#define FE_CLK_ADDR   MMCM_REG1(0, 4)
#define FE_DIV(x)     MMCM_VCO_DIV(KARST_MMCM0_VCO, x)
#define PHY_CLK_ADDR  MMCM_REG1(0, 5)
#define PHY_DIV(x)    MMCM_VCO_DIV(KARST_MMCM0_VCO, x)

/* FPGA-B MMCM1 */
#define KARST_MMCM1_CLKIN 30
#define KARST_MMCM1_VCO   840
#define BD_CLK_ADDR       MMCM_REG1(1, 6)
#define BD_DIV(x)         MMCM_VCO_DIV(840, x)
#define KARST_FB_CLK_ADDR MMCM_REG1(1, FB)

#elif defined(CFG_VIRTEX7)

/* MMCM0 */
#define KARST_MMCM0_CLKIN 40
#define KARST_MMCM0_VCO   960
#define MPIF_CLK_ADDR MMCM_REG1(0, 6)
#define MPIF_DIV(x)   MMCM_VCO_DIV(KARST_MMCM0_VCO, x)
#define FE_CLK_ADDR   MMCM_REG1(0, 4)
#define FE_DIV(x)     MMCM_VCO_DIV(KARST_MMCM0_VCO, x)
#define PHY_CLK_ADDR  MMCM_REG1(0, 5)
#define PHY_DIV(x)    MMCM_VCO_DIV(KARST_MMCM0_VCO, x)
#define LA_CLK_ADDR   MMCM_REG1(0, 0)
#define LA_DIV(x)     MMCM_VCO_DIV(KARST_MMCM0_VCO, x)

/* MMCM1 */
#define KARST_MMCM1_CLKIN 40
#define KARST_MMCM1_VCO   840
#define BD_CLK_ADDR       MMCM_REG1(1, 6)
#define BD_DIV(x)         MMCM_VCO_DIV(840, x)
#define KARST_FB_CLK_ADDR MMCM_REG1(1, FB)

/* MMCM 2 */
#define KARST_MMCM2_CLKIN 25
#define KARST_MMCM2_VCO   1200
#define MAC_CORE_CLK_ADDR MMCM_REG1(2, 1)
#define MAC_WT_CLK_ADDR   MMCM_REG1(2, 2)
#define MAC_CORE_DIV(x)   MMCM_VCO_DIV(KARST_MMCM2_VCO, x)
#define MAC_WT_DIV(x)     MMCM_VCO_DIV(KARST_MMCM2_VCO, x)
#define MAC_FREQ(x)       (KARST_MMCM2_VCO / MMCM_VCO_TO_NUM_DIV(x))

#endif /* CFG_VIRTEX6 */

/// Clock configuration structure
struct clk_config {
    /// MAC-PHY interface configuration
    struct mpif_config {
        /// MAC-PHY divider
        uint16_t mpif_div;
        /// LA divider
        uint16_t la_div;
        /// LA clock
        uint16_t la_clock;
    } mpif[2]; ///< Two options for MAC-PHY interface (0: nominal, 1: reduced)
    /// MAC configuration
    struct mac_config {
        /// MAC divider
        uint16_t mac_div;
        /// WT divider
        uint16_t wt_div;
    } mac[2]; ///< Two options for MAC (0: nominal, 1: reduced)
    /// FE divider
    uint16_t fe_div;
    /// PHY divider
    uint16_t phy_div;
    /// BitDomain divider
    uint16_t bd_div;
};

/// Table of all possible clock configurations
static const struct clk_config clk_config[10] = {
    // MDMCONF0: CBW=20MHz, NSS=1, 11nac, BCC
    {
        .mpif[0] = { .mpif_div = MPIF_DIV(30),
                     .la_div   = LA_DIV(60),
                     .la_clock = 60 },
        .mpif[1] = { .mpif_div = MPIF_DIV(30),
                     .la_div   = LA_DIV(60),
                     .la_clock = 60 },
        .mac[0] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .mac[1] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .fe_div  = FE_DIV(40),
        .phy_div = PHY_DIV(60),
        .bd_div  = BD_DIV(60),
    },

    // MDMCONF1: CBW=20MHz, NSS=2, 11n, BCC
    {
        .mpif[0] = { .mpif_div = MPIF_DIV(30),
                     .la_div   = LA_DIV(60),
                     .la_clock = 60 },
        .mpif[1] = { .mpif_div = MPIF_DIV(30),
                     .la_div   = LA_DIV(60),
                     .la_clock = 60 },
        .mac[0] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .mac[1] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .fe_div  = FE_DIV(40),
        .phy_div = PHY_DIV(60),
        .bd_div  = BD_DIV(84),
    },

    // MDMCONF2: CBW=40MHz, NSS=1, 11n, BCC/LDPC
    {
        .mpif[0] = { .mpif_div = MPIF_DIV(60),
                     .la_div   = LA_DIV(120),
                     .la_clock = 120 },
        .mpif[1] = { .mpif_div = MPIF_DIV(60),
                     .la_div   = LA_DIV(120),
                     .la_clock = 120 },
        .mac[0] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .mac[1] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .fe_div  = FE_DIV(80),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(120),
    },

    // MDMCONF3: CBW=40MHz, NSS=2, 11nac, BCC/LDPC
    {
        .mpif[0] = { .mpif_div = MPIF_DIV(60),
                     .la_div   = LA_DIV(120),
                     .la_clock = 120 },
        .mpif[1] = { .mpif_div = MPIF_DIV(60),
                     .la_div   = LA_DIV(120),
                     .la_clock = 120 },
        .mac[0] = { .mac_div = MAC_CORE_DIV(80),
                    .wt_div = MAC_WT_DIV(80) },
        .mac[1] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .fe_div  = FE_DIV(80),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(168),
    },

    // MDMCONF4: CBW=80MHz, NSS=1, BCC
    {
        .mpif[0] = { .mpif_div = MPIF_DIV(60),
                     .la_div   = LA_DIV(120),
                     .la_clock = 120 },
        .mpif[1] = { .mpif_div = MPIF_DIV(60),
                     .la_div   = LA_DIV(120),
                     .la_clock = 120 },
        .mac[0] = { .mac_div = MAC_CORE_DIV(80),
                    .wt_div = MAC_WT_DIV(80) },
        .mac[1] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .fe_div  = FE_DIV(160),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(168),
    },

    // MDMCONF5: CBW=80MHz, NSS=2, BCC
    {
        .mpif[0] = { .mpif_div = MPIF_DIV(120),
                     .la_div   = LA_DIV(120),
                     .la_clock = 120 },
        .mpif[1] = { .mpif_div = MPIF_DIV(80),
                     .la_div   = LA_DIV(160),
                     .la_clock = 160 },
        .mac[0] = { .mac_div = MAC_CORE_DIV(100),
                    .wt_div = MAC_WT_DIV(100) },
        .mac[1] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .fe_div  = FE_DIV(160),
        .phy_div = PHY_DIV(120),
        #if defined(CFG_VIRTEX6)
        .bd_div  = BD_DIV(210),
        #elif defined(CFG_VIRTEX7)
        .bd_div  = BD_DIV(168),
        #endif
    },

    // MDMCONF6: CBW=80MHz, NSS=2, BCC/LDPC
    {
        .mpif[0] = { .mpif_div = MPIF_DIV(120),
                     .la_div   = LA_DIV(120),
                     .la_clock = 120 },
        .mpif[1] = { .mpif_div = MPIF_DIV(80),
                     .la_div   = LA_DIV(160),
                     .la_clock = 160 },
        .mac[0] = { .mac_div = MAC_CORE_DIV(100),
                    .wt_div = MAC_WT_DIV(100) },
        .mac[1] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .fe_div  = FE_DIV(160),
        .phy_div = PHY_DIV(120),
        #if defined(CFG_VIRTEX6)
        .bd_div  = BD_DIV(210),
        #elif defined(CFG_VIRTEX7)
        .bd_div  = BD_DIV(168),
        #endif
    },

    // MDMCONF7: CBW=80MHz, NSS=1, BCC/LDPC
    {
        .mpif[0] = { .mpif_div = MPIF_DIV(120),
                     .la_div   = LA_DIV(120),
                     .la_clock = 120 },
        .mpif[1] = { .mpif_div = MPIF_DIV(80),
                     .la_div   = LA_DIV(160),
                     .la_clock = 160 },
        .mac[0] = { .mac_div = MAC_CORE_DIV(80),
                    .wt_div = MAC_WT_DIV(80) },
        .mac[1] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .fe_div  = FE_DIV(160),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(168),
    },

    // MDMCONF8: CBW=20MHz, NSS=1/2, 11nac, BCC/LDPC
    {
        .mpif[0] = { .mpif_div = MPIF_DIV(60),
                     .la_div   = LA_DIV(120),
                     .la_clock = 120 },
        .mpif[1] = { .mpif_div = MPIF_DIV(60),
                     .la_div   = LA_DIV(120),
                     .la_clock = 120 },
        .mac[0] = { .mac_div = MAC_CORE_DIV(80),
                    .wt_div = MAC_WT_DIV(80) },
        .mac[1] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .fe_div  = FE_DIV(40),
        .phy_div = PHY_DIV(60),
        .bd_div  = BD_DIV(120),
    },

    // MDMCONF9: CBW=40MHz, NSS=2, 11nac, BCC/LDPC
    {
        .mpif[0] = { .mpif_div = MPIF_DIV(120),
                     .la_div   = LA_DIV(120),
                     .la_clock = 120 },
        .mpif[1] = { .mpif_div = MPIF_DIV(80),
                     .la_div   = LA_DIV(160),
                     .la_clock = 160 },
        .mac[0] = { .mac_div = MAC_CORE_DIV(80),
                    .wt_div = MAC_WT_DIV(80) },
        .mac[1] = { .mac_div = MAC_CORE_DIV(40),
                    .wt_div = MAC_WT_DIV(80) },
        .fe_div  = FE_DIV(80),
        .phy_div = PHY_DIV(120),
        #if defined(CFG_VIRTEX6)
        .bd_div  = BD_DIV(210),
        #elif defined(CFG_VIRTEX7)
        .bd_div  = BD_DIV(168),
        #endif
    }
};


/**
 ******************************************************************************
 * @brief Configure one clock source
 *
 * @param[in] addr Address of the register
 * @param[in] data Value to apply
 ******************************************************************************
 */
static void crm_mmc_clk_set(uint32_t addr, uint16_t data)
{
    karst_dyncfgmmc_addr_set(addr);
    karst_dyncfgmmc_data_set(data);
    karst_dyncfgmmc_cntl_set(KARST_DYNCFGMMC_START_BIT);
    reg_poll(KARST_DYNCFGMMC_CNTL_ADDR,  KARST_DYNCFGMMC_START_BIT | KARST_DYNCFGMMC_DONE_BIT);
    karst_dyncfgmmc_cntl_set(0);
}

#ifdef CFG_VIRTEX6
static void fpgaa_mmc_clk_set(uint32_t addr, uint16_t data)
{
    fpgaa_dyncfgmmc_addr_set(addr);
    fpgaa_dyncfgmmc_data_set(data);
    fpgaa_dyncfgmmc_cntl_set(FPGAA_DYNCFGMMC_START_BIT);
    reg_poll(FPGAA_DYNCFGMMC_CNTL_ADDR, FPGAA_DYNCFGMMC_START_BIT | FPGAA_DYNCFGMMC_DONE_BIT);
    fpgaa_dyncfgmmc_cntl_set(0);
}
#endif /* CFG_VIRTEX6 */

/**
 ******************************************************************************
 * @brief Select clock configuration index
 *
 * @param[in] chan_type  Channel bandwidth
 * @return the clock configuration to apply for the selected bandwidth
 ******************************************************************************
 */

static int crm_clk_conf(uint8_t chan_type)
{
    int nss = mdm_nss_getf();
    int ldpc = mdm_ldpcdec_getf();
    #if MDM_CONF_0_9
    bool vht_supp = phy_vht_supported();
    #endif
    int clkconf = 0;

    switch (chan_type)
    {
        case PHY_CHNL_BW_20:
            #if MDM_CONF_0_9
            if (ldpc)
                clkconf = 8;
            else if (nss == 1)
                clkconf = 0;
            else
                clkconf = 1;
            #else
            if (nss == 1)
                clkconf = 0;
            else
                clkconf = 1;
            #endif
            break;
        case PHY_CHNL_BW_40:
            #if MDM_CONF_0_9
            if (nss == 1)
                clkconf = 2;
            else if (!vht_supp)
                clkconf = 3;
            else
                clkconf = 9;
            #else
            if (nss == 1)
                clkconf = 2;
            else
                clkconf = 3;
            #endif
            break;
        case PHY_CHNL_BW_80:
            if (nss == 1)
            {
                if (ldpc)
                    clkconf = 7;
                else
                    clkconf = 4;
            }
            else
            {
                if (ldpc)
                    clkconf = 6;
                else
                    clkconf = 5;
            }
            break;
        default:
            ASSERT_ERR(0);
            break;
    }

    return clkconf;
}

void crm_clk_set(uint8_t chan_type)
{
    int clkconf = crm_clk_conf(chan_type);

    // Check if the clock configuration needs to be modified
    if (clkconf == crm_env.clk_conf)
        return;

    crm_env.clk_conf = clkconf;

    // Set the multiplier for the VCO source of the BD clock
    crm_mmc_clk_set(KARST_FB_CLK_ADDR, MMCM_VCO_MULT(KARST_MMCM1_CLKIN, KARST_MMCM1_VCO));

    crm_mmc_clk_set(FE_CLK_ADDR, clk_config[clkconf].fe_div);
    crm_mmc_clk_set(PHY_CLK_ADDR, clk_config[clkconf].phy_div);
    crm_mmc_clk_set(BD_CLK_ADDR, clk_config[clkconf].bd_div);

    #ifdef CFG_VIRTEX6
    // Switch to mmc0 in case last DYNCFGMMC_ADDR was for mmc1
    karst_dyncfgmmc_addr_set(0);
    do
    {
        karst_dyncfgmmc_cntl_set(KARST_DYNCFGMMC_START_BIT);
        reg_poll(KARST_DYNCFGMMC_CNTL_ADDR,  KARST_DYNCFGMMC_START_BIT | KARST_DYNCFGMMC_DONE_BIT);
        // mmc{0,1,2,3}_lock
        reg_poll(KARST_CLKRST_STAT_ADDR, KARST_MMC_LOCK_MASK);
        karst_dyncfgmmc_cntl_set(0);
    } while (!p40m_ok() && ({udelay(0); 1;}));
    #endif /* CFG_VIRTEX6 */

    mdm_mdmconf_setf(clkconf);
}

void crm_init(void)
{
    int clkconf = crm_clk_conf(mdm_chbw_getf());
    int mac_clck_id;
    uint16_t mac_div;

    memset(&crm_env, 0, sizeof(crm_env));

    // Check if the MAC clock is enough to use nominal MAC PHY clock values or
    // reduced ones
    mac_clck_id = (nxmac_mac_core_clk_freq_getf() < 60) ? 1 : 0;

    crm_mmc_clk_set(MPIF_CLK_ADDR, clk_config[clkconf].mpif[mac_clck_id].mpif_div);

    mac_div = clk_config[clkconf].mac[mac_clck_id].mac_div;

    #if defined(CFG_VIRTEX6)
    fpgaa_mmc_clk_set(MPIF_FPGAA_CLK_ADDR, clk_config[clkconf].mpif[mac_clck_id].mpif_div);
    fpgaa_mmc_clk_set(LA_CLK_ADDR, clk_config[clkconf].mpif[mac_clck_id].la_div);
    fpgaa_la_sampling_freq_set(clk_config[clkconf].mpif[mac_clck_id].la_clock);

    fpgaa_mmc_clk_set(MAC_CORE_CLK_ADDR, mac_div);
    fpgaa_mmc_clk_set(MAC_WT_CLK_ADDR, clk_config[clkconf].mac[mac_clck_id].wt_div);

    #elif defined(CFG_VIRTEX7)
    crm_mmc_clk_set(LA_CLK_ADDR, clk_config[clkconf].mpif[mac_clck_id].la_div);
    karst_la_sampling_freq_set(clk_config[clkconf].mpif[mac_clck_id].la_clock);

    crm_mmc_clk_set(MAC_CORE_CLK_ADDR, mac_div);
    crm_mmc_clk_set(MAC_WT_CLK_ADDR, clk_config[clkconf].mac[mac_clck_id].wt_div);
    #endif /* CFG_VIRTEX7 */

    crm_env.clk_conf = CRM_CLKCONF_UNDEFINED;
    crm_env.mac_freq = MAC_FREQ(mac_div);

    #if defined(CFG_VIRTEX6)
    crm_env.cpu_freq = 105000000;
    #elif defined(CFG_VIRTEX7)
    karst_fmetre_start_setf(1);
    while(!karst_fmetre_done_getf()) {};
    crm_env.cpu_freq = karst_fmetre_plf_getf() * 1000;
    karst_dyncfgmmc_cntl_set(0);
    #endif
}

bool crm_fe_160m(void)
{
    if (crm_env.clk_conf != CRM_CLKCONF_UNDEFINED)
    {
        return (clk_config[crm_env.clk_conf].fe_div == FE_DIV(160));
    }

    return false;
}

uint8_t crm_get_mac_freq(void)
{
    return (crm_env.mac_freq);
}

uint32_t crm_get_cpu_freq(void)
{
    return (crm_env.cpu_freq);
}

void crm_mdm_reset(void)
{
    mdm_swreset_set(MDM_AGCSWRESET_BIT | MDM_DSPSWRESET_BIT | MDM_MDMSWRESET_BIT);
    udelay(10);
    mdm_swreset_set(0x0);
    udelay(10);
}

/// @}

