/**
 ****************************************************************************************
 *
 * @file crm.c
 *
 * @brief Clock and Reset Module definitions.
 *
 * Copyright (C) RivieraWaves 2018-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup CRM
 * @{
 ****************************************************************************************
 */
/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "crm.h"
#include "phy.h"
#include "reg_mdm_cfg.h"
#include "reg_mac_core.h"
#include "dbg.h"
#include <string.h>

#ifdef CFG_VIRTEX6
#error "No support of v20 platform on Virtex6 board"
#endif

#if NX_MDM_VER < 30
#error "This version of CRM is not valid for old versions of the modem"
#endif

/// Undefined value for the clock configuration
#define CRM_CLKCONF_UNDEFINED   ((uint8_t)-1)

/// CRM driver context.
struct crm_env_tag
{
    /// Current clock configuration index
    uint8_t clk_conf;
    /// Frequency of the MAC HW
    uint8_t mac_freq;
    /// Frequency (in HZ) of the CPU
    uint32_t cpu_freq;
};

static struct crm_env_tag crm_env;

static int reg_poll(uint32_t reg, uint32_t val)
{
    int loops = 1 << 20;

    while ((REG_PL_RD(reg) & val) != val && --loops)
        ;
    if (!loops)
    {
        dbg(D_ERR D_PHY "%s @0x%08x 0x%08x failed\n", __func__, reg, val);
        return -1;
    }
    return !loops;
}

static inline void udelay(uint32_t us)
{
    uint32_t e = nxmac_monotonic_counter_2_lo_get() + us;

    do {
        volatile int n = 1 << 5; // relax
        while (n--)
            ;
    } while ((int32_t)(nxmac_monotonic_counter_2_lo_get() - e) < 0);
}

#ifdef CFG_VIRTEX7
/* Mixed Mode Clock Manager defines */
/* output addresses */
#define CLKOUT0  0x08
#define CLKOUT1  0x0A
#define CLKOUT2  0x0C
#define CLKOUT3  0x0E
#define CLKOUT4  0x10
#define CLKOUT5  0x06
#define CLKOUT6  0x12
#define CLKOUTFB 0x14

/*
  address to configure clkreg1 reg for output <out> of mmcm <mmcm>
  note: use CRM_DYNCFGMMC_xxx defined assuming that FPGAA_DYNCFGMMC_xxx
        have the same values
 */
#define MMCM_REG1(mmcm, out)                    \
    (CRM_DYNCFGMMC_WE_BIT |                   \
     (mmcm << CRM_DYNCFGMMC_SEL_LSB) |        \
     CLKOUT##out)

/* Multiplier to apply to <clk_ref_freq> to get <vco_freq> */
#define MMCM_VCO_MULT(clk_ref_freq, vco_freq)     \
    ((((vco_freq/clk_ref_freq) + 1) / 2) << 6 |   \
     (((vco_freq/clk_ref_freq)) / 2) |            \
     0x1000)

/* Divider to apply to <vco_freq> to get <target_freq> */
#define MMCM_VCO_DIV(vco_freq, target_freq)    \
    ((((vco_freq/target_freq) + 1) / 2) << 6 | \
     (((vco_freq/target_freq)) / 2) |          \
     0x1000)

/* Convert a VCO divider <vco_div> to its corresponding numeric value */
#define MMCM_VCO_TO_NUM_DIV(vco_div) (((vco_div & 0x0FC0) >> 6) + (vco_div & 0x3F))

/* MMCM0 */
#define KARST_MMCM0_CLKIN 40
#define KARST_MMCM0_VCO   960
#define MPIF_CLK_ADDR MMCM_REG1(0, 6)
#define MPIF_DIV(x)   MMCM_VCO_DIV(KARST_MMCM0_VCO, x)
#define FE_CLK_ADDR   MMCM_REG1(0, 4)
#define FE_DIV(x)     MMCM_VCO_DIV(KARST_MMCM0_VCO, x)
#define PHY_CLK_ADDR  MMCM_REG1(0, 5)
#define PHY_DIV(x)    MMCM_VCO_DIV(KARST_MMCM0_VCO, x)
#define LA_CLK_ADDR   MMCM_REG1(0, 0)
#define LA_DIV(x)     MMCM_VCO_DIV(KARST_MMCM0_VCO, x)

/* MMCM1 */
#define KARST_MMCM1_CLKIN 40
#define KARST_MMCM1_VCO   840
#define BD_CLK_ADDR       MMCM_REG1(1, 6)
#define BD_DIV(x)         MMCM_VCO_DIV(840, x)
#define KARST_FB_CLK_ADDR MMCM_REG1(1, FB)

/* MMCM 2 */
#define KARST_MMCM2_CLKIN 25
#define KARST_MMCM2_VCO   1200
#define MAC_CORE_CLK_ADDR MMCM_REG1(2, 1)
#define MAC_WT_CLK_ADDR   MMCM_REG1(2, 2)
#define MAC_CORE_DIV(x)   MMCM_VCO_DIV(KARST_MMCM2_VCO, x)
#define MAC_WT_DIV(x)     MMCM_VCO_DIV(KARST_MMCM2_VCO, x)
#define MAC_FREQ(x)       (KARST_MMCM2_VCO / MMCM_VCO_TO_NUM_DIV(x))
#else
#define MPIF_CLK_ADDR     0  // TBD
#define MPIF_DIV(x)      (x) // TBD
#define FE_CLK_ADDR       0  // TBD
#define FE_DIV(x)        (x) // TBD
#define PHY_CLK_ADDR      0  // TBD
#define PHY_DIV(x)       (x) // TBD
#define LA_CLK_ADDR       0  // TBD
#define LA_DIV(x)        (x) // TBD
#define BD_CLK_ADDR       0  // TBD
#define BD_DIV(x)        (x)
#define KARST_FB_CLK_ADDR 0

/* MMCM 2 */
#define MAC_CORE_CLK_ADDR 0
#define MAC_WT_CLK_ADDR   0
#define MAC_CORE_DIV(x)  (x)
#define MAC_WT_DIV(x)    (x)
#define MAC_FREQ(x)      (x)
#endif

static const struct clk_config {
    uint16_t mpif_div;
    uint16_t la_div;
    uint16_t la_clock;
    uint16_t mac_div;
    uint16_t wt_div;
    uint16_t fe_div;
    uint16_t phy_div;
    uint16_t bd_div;
} clk_config[12] = {
    // CONF0: CBW=20MHz, NSS=1, 11nac, BCC
    {
        .mpif_div = MPIF_DIV(30),
        .la_div   = LA_DIV(60),
        .la_clock = 60,
        .mac_div = MAC_CORE_DIV(40),
        .wt_div = MAC_WT_DIV(80),
        .fe_div  = FE_DIV(40),
        .phy_div = PHY_DIV(60),
        .bd_div  = BD_DIV(60),
    },

    // CONF1: CBW=20MHz, NSS=2, 11n, BCC
    {
        .mpif_div = MPIF_DIV(30),
        .la_div   = LA_DIV(60),
        .la_clock = 60,
        .mac_div = MAC_CORE_DIV(40),
        .wt_div = MAC_WT_DIV(80),
        .fe_div  = FE_DIV(40),
        .phy_div = PHY_DIV(60),
        .bd_div  = BD_DIV(84),
    },

    // CONF2: CBW=40MHz, NSS=1, 11n, BCC/LDPC
    {
        .mpif_div = MPIF_DIV(60),
        .la_div   = LA_DIV(120),
        .la_clock = 120,
        .mac_div = MAC_CORE_DIV(40),
        .wt_div = MAC_WT_DIV(80),
        .fe_div  = FE_DIV(80),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(120),
    },

    // CONF3: CBW=40MHz, NSS=2, 11nac, BCC/LDPC
    {
        .mpif_div = MPIF_DIV(60),
        .la_div   = LA_DIV(120),
        .la_clock = 120,
        .mac_div = MAC_CORE_DIV(80),
        .wt_div = MAC_WT_DIV(80),
        .fe_div  = FE_DIV(80),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(168),
    },

    // CONF4: CBW=80MHz, NSS=1, BCC
    {
        .mpif_div = MPIF_DIV(60),
        .la_div   = LA_DIV(120),
        .la_clock = 120,
        .mac_div = MAC_CORE_DIV(80),
        .wt_div = MAC_WT_DIV(80),
        .fe_div  = FE_DIV(160),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(168),
    },

    // CONF5: CBW=80MHz, NSS=2, BCC
    {
        .mpif_div = MPIF_DIV(120),
        .la_div   = LA_DIV(120),
        .la_clock = 120,
        .mac_div = MAC_CORE_DIV(100),
        .wt_div = MAC_WT_DIV(100),
        .fe_div  = FE_DIV(160),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(168),
    },

    // CONF6: CBW=80MHz, NSS=2, BCC/LDPC
    {
        .mpif_div = MPIF_DIV(120),
        .la_div   = LA_DIV(120),
        .la_clock = 120,
        .mac_div = MAC_CORE_DIV(100),
        .wt_div = MAC_WT_DIV(100),
        .fe_div  = FE_DIV(160),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(168),
    },

    // CONF7: CBW=80MHz, NSS=1, BCC/LDPC
    {
        .mpif_div = MPIF_DIV(120),
        .la_div   = LA_DIV(120),
        .la_clock = 120,
        .mac_div = MAC_CORE_DIV(80),
        .wt_div = MAC_WT_DIV(80),
        .fe_div  = FE_DIV(160),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(168),
    },

    // CONF8: CBW=20MHz, NSS=1/2, 11nac, BCC/LDPC
    {
        .mpif_div = MPIF_DIV(60),
        .la_div   = LA_DIV(120),
        .la_clock = 120,
        .mac_div = MAC_CORE_DIV(80),
        .wt_div = MAC_WT_DIV(80),
        .fe_div  = FE_DIV(40),
        .phy_div = PHY_DIV(60),
        .bd_div  = BD_DIV(120),
    },

    // CONF9: CBW=40MHz, NSS=2, 11nac, BCC/LDPC
    {
        .mpif_div = MPIF_DIV(120),
        .la_div   = LA_DIV(120),
        .la_clock = 120,
        .mac_div = MAC_CORE_DIV(80),
        .wt_div = MAC_WT_DIV(80),
        .fe_div  = FE_DIV(80),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(168),
    },

    // CONF10: CBW=20MHz, NSS=1, 11ax, BCC/LDPC
    {
        .mpif_div = MPIF_DIV(120),
        .la_div   = LA_DIV(120),
        .la_clock = 120,
        .mac_div = MAC_CORE_DIV(60),
        .wt_div = MAC_WT_DIV(60),
        .fe_div  = FE_DIV(40),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(120),
    },

    // CONF11: CBW=40MHz, NSS=1, 11ax, BCC/LDPC
    {
        .mpif_div = MPIF_DIV(120),
        .la_div   = LA_DIV(120),
        .la_clock = 120,
        .mac_div = MAC_CORE_DIV(60),
        .wt_div = MAC_WT_DIV(60),
        .fe_div  = FE_DIV(80),
        .phy_div = PHY_DIV(120),
        .bd_div  = BD_DIV(120),
    }
};

static void crm_mmc_clk_set(uint32_t addr, uint16_t data)
{
    crm_dyncfgmmc_addr_set(addr);
    crm_dyncfgmmc_data_set(data);
    crm_dyncfgmmc_cntl_set(CRM_DYNCFGMMC_START_BIT);
    reg_poll(CRM_DYNCFGMMC_CNTL_ADDR,  CRM_DYNCFGMMC_START_BIT | CRM_DYNCFGMMC_DONE_BIT);
    crm_dyncfgmmc_cntl_set(0);
}

static int crm_clk_conf(uint8_t chan_type)
{
    int nss = mdm_nss_getf();
    int ldpc = mdm_ldpcdec_getf();
    bool vht_supp = phy_vht_supported();
    bool he_supp = phy_he_supported();
    int clkconf = 0;

    switch (chan_type)
    {
        case PHY_CHNL_BW_20:
            if (he_supp)
                clkconf = 10;
            else if (ldpc)
                clkconf = 8;
            else if (nss == 1)
                clkconf = 0;
            else
                clkconf = 1;
            break;
        case PHY_CHNL_BW_40:
            if (he_supp)
                clkconf = 11;
            else if (nss == 1)
                clkconf = 2;
            else if (!vht_supp)
                clkconf = 3;
            else
                clkconf = 9;
            break;
        case PHY_CHNL_BW_80:
            if (nss == 1)
            {
                if (ldpc)
                    clkconf = 7;
                else
                    clkconf = 4;
            }
            else
            {
                if (ldpc)
                    clkconf = 6;
                else
                    clkconf = 5;
            }
            break;
        default:
            ASSERT_ERR(0);
            break;
    }

    return clkconf;
}

void crm_clk_set(uint8_t chan_type)
{
    int clkconf = crm_clk_conf(chan_type);

    // Check if the clock configuration needs to be modified
    if (clkconf == crm_env.clk_conf)
        return;

    crm_env.clk_conf = clkconf;

    // Set the multiplier for the VCO source of the BD clock
//    crm_mmc_clk_set(KARST_FB_CLK_ADDR, MMCM_VCO_MULT(KARST_MMCM1_CLKIN, KARST_MMCM1_VCO));

    crm_mmc_clk_set(FE_CLK_ADDR, clk_config[clkconf].fe_div);
    crm_mmc_clk_set(PHY_CLK_ADDR, clk_config[clkconf].phy_div);
//    crm_mmc_clk_set(BD_CLK_ADDR, clk_config[clkconf].bd_div);
}

void crm_init(void)
{
    int clkconf = crm_clk_conf(mdm_chbw_getf());
    uint16_t mac_div;

    memset(&crm_env, 0, sizeof(crm_env));

    crm_mmc_clk_set(MPIF_CLK_ADDR, clk_config[clkconf].mpif_div);

    mac_div = clk_config[clkconf].mac_div;

    crm_mmc_clk_set(LA_CLK_ADDR, clk_config[clkconf].la_div);
    crm_la_sampling_freq_set(clk_config[clkconf].la_clock);

    crm_mmc_clk_set(MAC_CORE_CLK_ADDR, mac_div);
    crm_mmc_clk_set(MAC_WT_CLK_ADDR, clk_config[clkconf].wt_div);

    crm_env.clk_conf = CRM_CLKCONF_UNDEFINED;
    crm_env.mac_freq = MAC_FREQ(mac_div);

    crm_fmetre_start_setf(1);
    while(!crm_fmetre_done_getf()) {};
    crm_env.cpu_freq = crm_fmetre_plf_getf() * 1000;
}

bool crm_fe_160m(void)
{
    if (crm_env.clk_conf != CRM_CLKCONF_UNDEFINED)
    {
        return (clk_config[crm_env.clk_conf].fe_div == FE_DIV(160));
    }

    return false;
}

uint8_t crm_get_mac_freq(void)
{
    return (crm_env.mac_freq);
}

uint32_t crm_get_cpu_freq(void)
{
    return (crm_env.cpu_freq);
}

void crm_mdm_reset(void)
{
    crm_rstctrl_set(CRM_AGCSWRESET_BIT | CRM_PHYSWRESET_BIT);
    udelay(10);
    crm_rstctrl_set(0x0);
    udelay(10);
}

/// @}

