/**
 ****************************************************************************************
 *
 * @file dma.c
 *
 * @brief IPC DMA utility functions
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
// for NULL
#include <stddef.h>
#include <string.h>

#include "dma.h"
#include "reg_dma.h"
#include "co_utils.h"

struct dma_env_tag dma_env;

void dma_init(void)
{
    memset(dma_env.last_dma, 0, sizeof(dma_env.last_dma));

    // configure the channel priorities
    dma_arbitration_set(0x0C);

    // configure the interrupts (enable all interrupts)
    dma_int_unmask_set(DMA_ERROR_BIT | DMA_LLI_IRQ_MASK);
}

void dma_push(struct dma_desc *first, struct dma_desc *last, int channel_idx)
{
    uint32_t value;

    // current DMA descriptor is pushed at the end of the list
    last->next = 0;

    // prevent accesses to the same DMA channel from both BK and IRQ simultaneously
    GLOBAL_INT_DISABLE();

    // set the MUTEX
    dma_channel_mutex_set(1 << channel_idx);

    // read the root in the DMA HW
    if (channel_idx == IPC_DMA_CHANNEL_INTERNAL)
        value = dma_ch4_lli_root_get();
    else
        value = dma_ch_lli_root_get(channel_idx);

    // check if the root is empty
    if (value)
    {
        // sanity check: the last MUST be valid because the root is not empty
        ASSERT_ERR(dma_env.last_dma[channel_idx] != 0);

        // append the descriptor to the last LLI of the list
        dma_env.last_dma[channel_idx]->next = CPU2HW(first);

        // clear the MUTEX
        dma_channel_mutex_clear(1 << channel_idx);
    }
    else
    {
        // write directly the DMA to the root
        if (channel_idx == IPC_DMA_CHANNEL_INTERNAL)
            dma_ch4_lli_root_set(CPU2HW(first));
        else
            dma_ch_lli_root_set(channel_idx, CPU2HW(first));
    }

    // save the new last LLI descriptor on this channel
    dma_env.last_dma[channel_idx] = last;

    // restore interrupts
    GLOBAL_INT_RESTORE();
}

void dma_buserr_isr(void)
{
    // For now consider this error as a fatal one
    ASSERT_ERR(0);
}



