/**
 ****************************************************************************************
 *
 * @file intc.c
 *
 * @brief Definition of the Interrupt Controller (INTCTRL) API.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup INTC
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
// for target dependent compilation directives
#include "rwnx_config.h"
// for __IRQ
#include "compiler.h"
// for ASSERT
#include "dbg_assert.h"

#include "intc.h"
#include "sysctrl.h"

#include "reg_intc.h"
#include "ipc_emb.h"
#include "hal_machw.h"
#include "txl_cntrl.h"
#include "macif.h"
#include "rxl_cntrl.h"
#include "rxl_hwdesc.h"
#include "hal_dma.h"


/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
void intc_spurious(void)
{
    // force error
    ASSERT_ERR(0);
}

/// locally define this type to be able to qualify the array.
typedef void (*void_fn)(void);

/// Array of IRQ handlers.
static const void_fn intc_irq_handlers[32] =
{
    [31-INTC_LLI15] = intc_spurious,
    [31-INTC_LLI14] = intc_spurious,
    [31-INTC_LLI13] = intc_spurious,
    [31-INTC_LLI12] = intc_spurious,
    [31-INTC_LLI11] = intc_spurious,
    [31-INTC_LLI10] = intc_spurious,
    [31-INTC_LLI9] = intc_spurious,
    [31-INTC_LLI8] = intc_spurious,
    [31-INTC_LLI7] = intc_spurious,
    [31-INTC_LLI6] = intc_spurious,
    [31-INTC_LLI5] = intc_spurious,
    [31-INTC_LLI4] = intc_spurious,
    [31-INTC_LLI3] = intc_spurious,
    [31-INTC_LLI2] = intc_spurious,
    [31-INTC_LLI1] = intc_spurious,
    [31-INTC_LLI0] = intc_spurious,
    [31-INTC_DMAEOT] = intc_spurious,
    [31-INTC_IPC3] = intc_spurious,
    [31-INTC_IPC2] = intc_spurious,
    [31-INTC_IPC1] = intc_spurious,
    [31-INTC_IPC0] = intc_spurious,
    [31-INTC_MACINTGEN] = intc_spurious,
    [31-INTC_MACPROT] = intc_spurious,
    [31-INTC_MACTX] = intc_spurious,
    [31-INTC_MACRX] = intc_spurious,
    [31-INTC_MACOTHER] = intc_spurious,
    [31-INTC_MACTIMER] = intc_spurious,
    [31-INTC_PHY] = intc_spurious,
    [31-INTC_UNUSED3] = intc_spurious,
    [31-INTC_UNUSED2] = intc_spurious,
    [31-INTC_UNUSED1] = intc_spurious,
    [31-INTC_UNUSED0] = intc_spurious
};

/** Array of FIQ handlers (they must be defined in the opposite order of the bit because
 * of the use of clz operation).
 */
static const void_fn intc_fiq_handlers[32] =
{
    [31-INTC_LLI15] = intc_spurious,
    [31-INTC_LLI14] = intc_spurious,
    [31-INTC_LLI13] = intc_spurious,
    [31-INTC_LLI12] = intc_spurious,
    #if (NX_GP_DMA)
    [31-INTC_LLI11] = hal_dma_ul_irq,
    #else //(NX_GP_DMA)
    [31-INTC_LLI11] = intc_spurious,
    #endif //(NX_GP_DMA)
    [31-INTC_LLI10] = intc_spurious,
    #if (NX_GP_DMA)
    [31-INTC_LLI9] = hal_dma_dl_irq,
    #else //(NX_GP_DMA)
    [31-INTC_LLI9] = intc_spurious,
    #endif //(NX_GP_DMA)
    #if NX_FULLY_HOSTED
    [INTC_LLI8] = NULL,
    #else
    [INTC_LLI8] = macif_tx_cfm_dma_int_handler,
    #endif
    [31-INTC_LLI7] = ipc_emb_dbg_dma_int_handler,
    [31-INTC_LLI6] = ipc_emb_msg_dma_int_handler,
    [31-INTC_LLI5] = rxl_dma_int_handler,
    [31-INTC_DMAEOT] = intc_spurious,
    #if !NX_FULLY_HOSTED
    [31-INTC_LLI4] = txl_cntrl_dma_isr,
    [31-INTC_LLI3] = txl_cntrl_dma_isr,
    [31-INTC_LLI2] = txl_cntrl_dma_isr,
    [31-INTC_LLI1] = txl_cntrl_dma_isr,
    [31-INTC_LLI0] = txl_cntrl_dma_isr,
    [31-INTC_IPC3] = ipc_emb_tx_irq,
    [31-INTC_IPC2] = ipc_emb_cfmback_irq,
    [31-INTC_IPC1] = ipc_emb_msg_irq,
    #endif
    [31-INTC_IPC0] = intc_spurious,
    [31-INTC_MACINTGEN] = hal_machw_gen_handler,
    #if (NX_BW_LEN_ADAPT)
    [31-INTC_MACPROT] = txl_prot_trigger,
    #else
    [31-INTC_MACPROT] = intc_spurious,
    #endif
    [31-INTC_MACTX] = txl_transmit_trigger,
    [31-INTC_MACRX] = rxl_mpdu_isr,
    [31-INTC_MACOTHER] = intc_spurious,
    [31-INTC_MACTIMER] = rxl_mpdu_isr,
    [31-INTC_PHY] = intc_spurious,
    [31-INTC_UNUSED3] = intc_spurious,
    [31-INTC_UNUSED2] = intc_spurious,
    [31-INTC_UNUSED1] = intc_spurious,
    [31-INTC_UNUSED0] = intc_spurious
};


void intc_init(void)
{
    uint32_t irq_set;
    // by default, all the IRQs are active low
    intc_irq_polarity_set(0);
    // by default, all the FIQs are active low
    intc_fiq_polarity_set(0);

    // enable the supported interrupts
    irq_set = (  CO_BIT(INTC_LLI0) | CO_BIT(INTC_LLI1) | CO_BIT(INTC_LLI2)
               | CO_BIT(INTC_LLI3) | CO_BIT(INTC_LLI4) | CO_BIT(INTC_LLI5)
               | CO_BIT(INTC_LLI6) | CO_BIT(INTC_LLI7)
               | CO_BIT(INTC_LLI9) | CO_BIT(INTC_LLI11)| CO_BIT(INTC_MACPROT)
               | CO_BIT(INTC_MACTX)| CO_BIT(INTC_MACTIMER)
               | CO_BIT(INTC_MACRX) | CO_BIT(INTC_MACINTGEN) );

    #if !NX_FULLY_HOSTED
    irq_set |= (CO_BIT(INTC_IPC3) | CO_BIT(INTC_IPC2) | CO_BIT(INTC_IPC1) | CO_BIT(INTC_LLI8));
    #endif

    intc_fiq_unmask_set(irq_set);
}


__IRQ void intc_irq(void)
{
    uint32_t status;
    uint32_t highbit;

    // retrieve the status in the interrupt controller
    status = intc_irq_status_get();

    // sanity check
    ASSERT_ERR(status != 0);

    // retrieve the highest pending interrupt
    highbit = co_clz(status);

    // call the function handler
    intc_irq_handlers[highbit]();
}

__FIQ void intc_fiq(void)
{
    uint32_t status;
    uint32_t highbit;

    LED_OFF(LED_SLEEP);

    // retrieve the status in the interrupt controller
    status = intc_fiq_rawstatus_get();

    // sanity check
    ASSERT_ERR(status != 0);

    // retrieve the highest pending interrupt
    highbit = co_clz(status);

    // call the function handler
    intc_fiq_handlers[highbit]();
}

/// @}
