/**
 ****************************************************************************************
 *
 * @file intc.c
 *
 * @brief Definition of the Interrupt Controller (INTCTRL) API.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup INTC
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
// for target dependent compilation directives
#include "rwnx_config.h"
// for __IRQ
#include "compiler.h"
#include "intc.h"

#include "reg_intc.h"
#include "sysctrl.h"
#include "ipc_emb.h"
#include "hal_machw.h"
#include "txl_cntrl.h"
#include "macif.h"
#include "rxl_cntrl.h"
#include "rxl_hwdesc.h"
#include "hal_dma.h"
#include "dma.h"
#include "rd.h"


/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Unexpected interrupt handler
 ****************************************************************************************
 */
void intc_spurious(void)
{
    // force error
    ASSERT_ERR(0);
}

/// locally define this type to be able to qualify the array.
typedef void (*void_fn)(void);

/// Array of IRQ handlers
static const void_fn intc_irq_handlers[64] =
{
    [INTC_LLI15] = intc_spurious,
    [INTC_LLI14] = intc_spurious,
    [INTC_LLI13] = intc_spurious,
    [INTC_LLI12] = intc_spurious,
    #if (NX_GP_DMA)
    [INTC_LLI11] = hal_dma_ul_irq,
    #else //(NX_GP_DMA)
    [INTC_LLI11] = intc_spurious,
    #endif //(NX_GP_DMA)
    [INTC_LLI10] = intc_spurious,
    #if (NX_GP_DMA)
    [INTC_LLI9] = hal_dma_dl_irq,
    #else //(NX_GP_DMA)
    [INTC_LLI9] = intc_spurious,
    #endif //(NX_GP_DMA)
    #if NX_FULLY_HOSTED
    [INTC_LLI8] = NULL,
    #else
    [INTC_LLI8] = macif_tx_cfm_dma_int_handler,
    #endif
    [INTC_LLI7] = ipc_emb_dbg_dma_int_handler,
    [INTC_LLI6] = ipc_emb_msg_dma_int_handler,
    [INTC_LLI5] = rxl_dma_int_handler,
    #if !NX_FULLY_HOSTED
    [INTC_LLI4] = txl_cntrl_dma_isr,
    [INTC_LLI3] = txl_cntrl_dma_isr,
    [INTC_LLI2] = txl_cntrl_dma_isr,
    [INTC_LLI1] = txl_cntrl_dma_isr,
    [INTC_LLI0] = txl_cntrl_dma_isr,
    [INTC_IPC3] = ipc_emb_tx_irq,
    [INTC_IPC2] = ipc_emb_cfmback_irq,
    [INTC_IPC1] = ipc_emb_msg_irq,
    #endif
    [INTC_IPC0] = intc_spurious,
    [INTC_MACINTGEN] = hal_machw_gen_handler,
    #if (NX_BW_LEN_ADAPT)
    [INTC_MACPROT] = txl_prot_trigger,
    #else
    [INTC_MACPROT] = intc_spurious,
    #endif
    [INTC_MACTX] = txl_transmit_trigger,
    [INTC_MACRX] = rxl_mpdu_isr,
    [INTC_MACOTHER] = intc_spurious,
    [INTC_MACTIMER] = rxl_mpdu_isr,
    [INTC_DMA_BUSERR] = dma_buserr_isr,
    [INTC_MODEM] = phy_mdm_isr,
    [INTC_RC] = phy_rc_isr
};


/**
 ****************************************************************************************
 * @brief Enable an interrupt in the interrupt controller
 *
 * @param[in] index Index of the interrupt to enable
 ****************************************************************************************
 */
static void intc_enable_irq(int index)
{
    int reg_idx = index / 32;
    int bit_idx = index % 32;

    intc_irq_unmask_set(reg_idx, CO_BIT(bit_idx));
}

/**
 ****************************************************************************************
 * @brief Disable an interrupt in the interrupt controller
 *
 * @param[in] index Index of the interrupt to disable
 ****************************************************************************************
 */
static void intc_disable_irq(int index)
{
    int reg_idx = index / 32;
    int bit_idx = index % 32;

    intc_irq_unmask_clear(reg_idx, CO_BIT(bit_idx));
}

void intc_init(void)
{
    // enable the supported interrupts
    #if !NX_FULLY_HOSTED
    intc_enable_irq(INTC_IPC3);
    intc_enable_irq(INTC_IPC2);
    intc_enable_irq(INTC_IPC1);
    intc_enable_irq(INTC_LLI8);
    #endif
    intc_enable_irq(INTC_LLI0);
    intc_enable_irq(INTC_LLI1);
    intc_enable_irq(INTC_LLI2);
    intc_enable_irq(INTC_LLI3);
    intc_enable_irq(INTC_LLI4);
    intc_enable_irq(INTC_LLI5);
    intc_enable_irq(INTC_LLI6);
    intc_enable_irq(INTC_LLI7);
    intc_enable_irq(INTC_DMA_BUSERR);
    #if (NX_GP_DMA)
    intc_enable_irq(INTC_LLI9);
    intc_enable_irq(INTC_LLI11);
    #endif
    intc_enable_irq(INTC_MACPROT);
    intc_enable_irq(INTC_MACTX);
    intc_enable_irq(INTC_MACTIMER);
    intc_enable_irq(INTC_MACRX);
    intc_enable_irq(INTC_MACINTGEN);
    intc_enable_irq(INTC_MODEM);
    intc_enable_irq(INTC_RC);
}


void intc_irq(void)
{
    LED_OFF(LED_SLEEP);

    // Check if we have some interrupts pending
    if (intc_irq_status_get(0) || intc_irq_status_get(1))
    {
        int irq_idx = intc_irq_index_get();

        ASSERT_ERR(intc_irq_handlers[irq_idx] != NULL);

        // call the function handler
        intc_irq_handlers[irq_idx]();
    }
}

/// @}
