/**
 ****************************************************************************************
 *
 * @file ipc_emb.c
 *
 * @brief IPC module.
 *
 * Copyright C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup IPC
 * @{
 ****************************************************************************************
 */
/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include <string.h>

#include "rwnx_config.h"
#include "dbg_assert.h"

#include "co_math.h"
#include "ke_mem.h"
#include "ke_msg.h"
#include "ke_task.h"
#include "ke_event.h"

#include "tx_swdesc.h"
#include "rxl_cntrl.h"
#include "txl_buffer.h"

#include "dma.h"
#include "ipc_emb.h"
#include "ipc_shared.h"

#include "sysctrl.h"

#include "dbg.h"

#include "txl_cntrl.h"
#if NX_AMPDU_TX
#include "txl_agg.h"
#endif

#if (NX_UMAC_PRESENT)
#include "txu_cntrl.h"
#endif //(NX_UMAC_PRESENT)

/*
 * DEFINES AND MACROS
 ****************************************************************************************
 */
/// Configure the DMA control field for a kernel message upload
#define MSG_LLICTRL(irqenable)                                                           \
    ( (irqenable)?(IPC_DMA_LLI_IRQ_EN|(IPC_DMA_LLI_MSG << IPC_DMA_LLI_IRQ_POS)) : 0)

/// Configure the DMA control field for a debug message upload
#define DBG_LLICTRL(irqenable)                                                           \
    ( (irqenable)?(IPC_DMA_LLI_IRQ_EN|(IPC_DMA_LLI_DBG << IPC_DMA_LLI_IRQ_POS)) : 0)

/// Bit field containing all IPC TX event bits
#if NX_BEACONING
#define ALL_EVENTS_TX  (KE_EVT_MACIF_TXDESC_AC0_BIT | KE_EVT_MACIF_TXDESC_AC1_BIT |  \
                        KE_EVT_MACIF_TXDESC_AC2_BIT | KE_EVT_MACIF_TXDESC_AC3_BIT |  \
                        KE_EVT_MACIF_TXDESC_BCN_BIT)
#else
#define ALL_EVENTS_TX  (KE_EVT_MACIF_TXDESC_AC0_BIT | KE_EVT_MACIF_TXDESC_AC1_BIT |  \
                        KE_EVT_MACIF_TXDESC_AC2_BIT | KE_EVT_MACIF_TXDESC_AC3_BIT)
#endif

/// High priority kernel that should interrupt the IPC TX event
#if NX_UMAC_PRESENT && NX_BEACONING
#define HIGH_PRIO_EVT (KE_EVT_RXREADY_BIT | KE_EVT_PRIMARY_TBTT_BIT | KE_EVT_MACIF_TXDESC_BCN_BIT)
#else
#define HIGH_PRIO_EVT (KE_EVT_RXREADY_BIT | KE_EVT_PRIMARY_TBTT_BIT)
#endif

/*
 * VARIABLES
 ****************************************************************************************
 */
/// Array of bits event to be triggered per TX queue index
const uint32_t ipc_emb_evt_bit[NX_TXQ_CNT] =
{
    #if NX_BEACONING
    [AC_BCN] = KE_EVT_MACIF_TXDESC_BCN_BIT,
    #endif
    [AC_VO] = KE_EVT_MACIF_TXDESC_AC3_BIT,
    [AC_VI] = KE_EVT_MACIF_TXDESC_AC2_BIT,
    [AC_BE] = KE_EVT_MACIF_TXDESC_AC1_BIT,
    [AC_BK] = KE_EVT_MACIF_TXDESC_AC0_BIT,
};

/// Global variable containing the IPC_EMB environment
struct ipc_emb_env_tag ipc_emb_env;

/// Mask for the indexes of the different queues
const int nx_txdesc_cnt_msk[] =
{
    NX_TXDESC_CNT0 - 1,
    NX_TXDESC_CNT1 - 1,
    NX_TXDESC_CNT2 - 1,
    NX_TXDESC_CNT3 - 1,
    #if (NX_BEACONING)
    NX_TXDESC_CNT4 - 1,
    #endif
};

/*
 * FUNCTIONS DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Copy a TX descriptor from its shared RAM location to the CPU data memory
 *
 * @param[in] dst_local Pointer to the destination in data memory
 * @param[in] src_shared Pointer to the source in shared RAM
 *
 ****************************************************************************************
 */
__INLINE void ipc_emb_txdesc_copy(struct txdesc *dst_local, volatile struct txdesc_host *src_shared)
{
    // Copy the host descriptor at the right place in the EMB descriptor
    volatile uint32_t *src = (uint32_t*) &src_shared->api;
    uint32_t *dst = (uint32_t*) &dst_local->host;

    // we are copying bytes 4 by 4. Hope that the additional bytes
    // in the dest will not suffer the overwrite.
    for (unsigned int i = 0; i < (sizeof_b(struct txdesc_api)+3)/4; i++)
    {
        dst[i] = src[i];
    }
}

uint8_t ipc_emb_tx_q_len(int queue_idx, int vif_idx)
{
    volatile struct txdesc_host *ptr, *start, *end;
    uint8_t i = 0, index, nb = 0;

    #if 1 //RW_MUMIMO_TX_EN
    for (i = 0; i < nx_txuser_cnt[queue_idx]; i++)
    {
    #endif
        index = ipc_emb_env.txdesc_idx[queue_idx][i] & nx_txdesc_cnt_msk[queue_idx];
        ptr = &ipc_emb_env.txdesc[queue_idx][i][index];
        end = &ipc_emb_env.txdesc[queue_idx][i][nx_txdesc_cnt_msk[queue_idx]];
        if (index == 0)
            start = NULL;
        else
            start = &ipc_emb_env.txdesc[queue_idx][i][0];

        while (ptr->ready) {
            if (vif_idx == ptr->api.host.vif_idx)
                nb++;
            if (ptr == end) {
                if (start)
                {
                    ptr = start;
                    end = &ipc_emb_env.txdesc[queue_idx][i][index - 1];
                    start = NULL;
                }
                else
                    break;
            }
            else
                ptr++;
        }
    #if 1 // RW_MUMIMO_TX_EN
    }
    #endif

    return nb;
}

/**
 ****************************************************************************************
 */
void ipc_emb_init(void)
{
    int i;

    // clear the entire IPC EMB environment
    memset(&ipc_emb_env, 0, sizeof(ipc_emb_env));

    // Set the TX descriptor array pointers
    for (i = 0; i < RW_USER_MAX; i++)
    {
        ipc_emb_env.txdesc[0][i] = ipc_shared_env.txdesc0[i];
        ipc_emb_env.txdesc[1][i] = ipc_shared_env.txdesc1[i];
        ipc_emb_env.txdesc[2][i] = ipc_shared_env.txdesc2[i];
        ipc_emb_env.txdesc[3][i] = ipc_shared_env.txdesc3[i];
        #if (NX_BEACONING)
        ipc_emb_env.txdesc[4][i] = NULL;
        #endif
    }
    #if (NX_BEACONING)
    ipc_emb_env.txdesc[4][0] = ipc_shared_env.txdesc4[0];
    #endif

    #if RW_MUMIMO_TX_EN
    for (i=0; i < IPC_TXQUEUE_CNT; i++)
    {
        ipc_emb_env.user_active[i] = (1 << nx_txuser_cnt[i]) - 1;
    }
    #endif

    //Set compatibility info
    #ifdef CFG_IPC_VER_10
    ipc_shared_env.comp_info.ipc_shared_version = 10;
    #else
    ipc_shared_env.comp_info.ipc_shared_version = 11;
    #endif /* CFG_IPC_VER_V10 */
    ipc_shared_env.comp_info.msg_api= MSG_API_VER;
    ipc_shared_env.comp_info.msge2a_buf_cnt = IPC_MSGE2A_BUF_CNT;
    ipc_shared_env.comp_info.dbgbuf_cnt = IPC_DBGBUF_CNT;
    ipc_shared_env.comp_info.radarbuf_cnt = IPC_RADARBUF_CNT;
    ipc_shared_env.comp_info.unsuprxvecbuf_cnt = IPC_UNSUPRXVECBUF_CNT;
    ipc_shared_env.comp_info.rxbuf_cnt = IPC_RXBUF_CNT;
    #if NX_UMAC_PRESENT
    ipc_shared_env.comp_info.rxdesc_cnt= IPC_RXDESC_CNT;
    #endif //(NX_UMAC_PRESENT)
    ipc_shared_env.comp_info.bk_txq = NX_TXDESC_CNT0;
    ipc_shared_env.comp_info.be_txq = NX_TXDESC_CNT1;
    ipc_shared_env.comp_info.vi_txq = NX_TXDESC_CNT2;
    ipc_shared_env.comp_info.vo_txq = NX_TXDESC_CNT3;
    #if NX_BEACONING
        ipc_shared_env.comp_info.bcn_txq = NX_TXDESC_CNT4;
    #else
        ipc_shared_env.comp_info.bcn_txq = 0;
    #endif /* NX_BEACONING */
    ipc_shared_env.comp_info.ipc_shared_size = sizeof_b(ipc_shared_env);

    // Set unchanged fields of the MSG DMA descriptor (to send MSGs to App)
    //   - DMA contents always starts at
    ipc_shared_env.msg_dma_desc.src = CPU2HW(&(ipc_shared_env.msg_e2a_buf));
    //   - length
    ipc_shared_env.msg_dma_desc.length = sizeof_b(struct ipc_e2a_msg);
    //   - no other descriptor will be linked to it
    ipc_shared_env.msg_dma_desc.next = 0;
    //   - enable interrupt generation on this descriptor
    ipc_shared_env.msg_dma_desc.ctrl = MSG_LLICTRL(1);
    // Put the pattern in the msg buffer
    ipc_shared_env.msg_e2a_buf.pattern = IPC_MSGE2A_VALID_PATTERN;

    /// Set unchanged fields of the Debug DMA descriptor (to send strings to App)
    //   - DMA contents always starts at
    ipc_shared_env.dbg_dma_desc.src = CPU2HW(&(ipc_shared_env.dbg_buf));
    //   - length
    ipc_shared_env.dbg_dma_desc.length = sizeof_b(struct ipc_dbg_msg);
    //   - no other descriptor will be linked to it
    ipc_shared_env.dbg_dma_desc.next = 0;
    //   - enable interrupt generation on this descriptor
    ipc_shared_env.dbg_dma_desc.ctrl = DBG_LLICTRL(1);
    // Put the pattern in the debug buffer
    ipc_shared_env.dbg_buf.pattern = IPC_DBG_VALID_PATTERN;

    // Sanity check - verify that the FW is compiled with the right IPC version
    ASSERT_ERR(ipc_emb_signature_get() == IPC_EMB_SIGNATURE_RESET);

    // configure the IPC line selection
    #ifdef CFG_IPC_VER_V10
    ipc_app2emb_line_sel_set(0);
    #else
    ipc_app2emb_line_sel_low_set(0);
    ipc_app2emb_line_sel_high_set(0);
    #endif /* CFG_IPC_VER_V10 */

    ipc_app2emb0_sel_setf(0); // DBG
    ipc_app2emb1_sel_setf(1); // MSG
    ipc_app2emb4_sel_setf(2); // RXDESC_BACK
    ipc_app2emb5_sel_setf(2); // RXBUF_BACK
    ipc_app2emb6_sel_setf(2); // TXCFM_BACK

    // TX desc interrupts
    ipc_app2emb8_sel_setf(3);
    ipc_app2emb9_sel_setf(3);
    ipc_app2emb10_sel_setf(3);
    ipc_app2emb11_sel_setf(3);
    ipc_app2emb12_sel_setf(3);

    #if (RW_USER_MAX > 1)
    ipc_app2emb13_sel_setf(3);
    ipc_app2emb14_sel_setf(3);
    ipc_app2emb15_sel_setf(3);
    ipc_app2emb16_sel_setf(3);
    #endif /* RW_USER_MAX > 1 */

    #if (RW_USER_MAX > 2)
    ipc_app2emb17_sel_setf(3);
    ipc_app2emb18_sel_setf(3);
    ipc_app2emb19_sel_setf(3);
    ipc_app2emb20_sel_setf(3);
    #endif /* RW_USER_MAX > 2 */

    #if (RW_USER_MAX > 3)
    ipc_app2emb21_sel_setf(3);
    ipc_app2emb22_sel_setf(3);
    ipc_app2emb23_sel_setf(3);
    ipc_app2emb24_sel_setf(3);
    #endif /* RW_USER_MAX > 3 */

    // enable the all interrupts from APP
    ipc_app2emb_unmask_set(IPC_IRQ_A2E_ALL);
}

/**
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 */
void ipc_emb_tx_flow_off(void)
{
    // mask Tx IPC irqs
    ipc_app2emb_unmask_clear(IPC_IRQ_A2E_TXDESC);

    // Clear all IPC Tx events
    ke_evt_clear(ALL_EVENTS_TX);
}

void ipc_emb_tx_flow_on(void)
{
    // unmask Tx IPC irqs
    ipc_app2emb_unmask_set(IPC_IRQ_A2E_TXDESC);

    // Set all IPC Tx events
    //ke_evt_set(ALL_EVENTS_TX);
}


void ipc_emb_tx_irq(void)
{
    // read HW IPC irq
    uint32_t stat = ipc_app2emb_status_get() & IPC_IRQ_A2E_TXDESC;

    if (stat)
    {
        // For profiling
        PROF_TX_IPC_IRQ_SET();

        // set IPC Tx event
        ke_evt_set(ipc_emb_tx_evt_field(stat));

        // mask Tx IPC irq
        ipc_app2emb_unmask_clear(stat);

        // clear the pending TX IRQ
        ipc_app2emb_ack_clear(stat);

        // For profiling
        PROF_TX_IPC_IRQ_CLR();
    }
}


/**
 ****************************************************************************************
 */
void ipc_emb_tx_evt(int queue_idx)
{
    volatile struct txdesc_host *txdesc_src;
    uint32_t evt_bit = ipc_emb_evt_bit[queue_idx];
    bool pushed = false;
    int i = 0;

    // For profiling
    PROF_TX_MACIF_EVT_SET();

    // Clear the event that was just triggered
    ke_evt_clear(evt_bit);

    #if RW_MUMIMO_TX_EN
    // Loop on all users
    for (i = 0; i < nx_txuser_cnt[queue_idx]; i++)
    #endif
    {
        struct txdesc *txdesc_base = txdesc_array[queue_idx][i];
        volatile struct txdesc_host *txdesc_src_base = ipc_emb_env.txdesc[queue_idx][i];
        uint32_t *txdesc_idx = &ipc_emb_env.txdesc_idx[queue_idx][i];

        #if RW_MUMIMO_TX_EN
        bool stop_user = false;

        // Check if this user position can be scheduled currently
        if (!(ipc_emb_env.user_active[queue_idx] & CO_BIT(i)))
            continue;
        #endif

        txdesc_src = &txdesc_src_base[*txdesc_idx & nx_txdesc_cnt_msk[queue_idx]];

        // Loop on all the ready descriptors
        while (txdesc_src->ready)
        {
            struct txdesc *txdesc_new;

            // Check if we have higher priority event pending
            #if NX_UMAC_PRESENT && NX_BEACONING
            if (!(evt_bit & KE_EVT_MACIF_TXDESC_BCN_BIT))
            #endif
            {
                if (ke_evt_get() & HIGH_PRIO_EVT)
                {
                    // For profiling
                    PROF_TX_MACIF_EVT_CLR();

                    // Set again the event that was just triggered
                    ke_evt_set(evt_bit);
                    return;
                }
            }

            // clear any possible pending TX IRQ (in case the host has pushed new packets)
            ipc_app2emb_ack_clear(CO_BIT(i + RW_USER_MAX * queue_idx + IPC_IRQ_A2E_TXDESC_FIRSTBIT));

            // allocation of the tx descriptor
            txdesc_new = &txdesc_base[*txdesc_idx & nx_txdesc_cnt_msk[queue_idx]];

            // copy the descriptor
            ipc_emb_txdesc_copy(txdesc_new, txdesc_src);

            // Initialize some fields of the descriptor
            txdesc_new->lmac.agg_desc = NULL;
            txdesc_new->lmac.hw_desc->cfm.status = 0;
            #if (RW_BFMER_EN)
            txdesc_new->lmac.bfr_node = NULL;
            #endif //(RW_BFMER_EN)
            #if (NX_UMAC_PRESENT)
            txdesc_new->umac.buf_control = NULL;
            #endif

            #if RW_MUMIMO_TX_EN
            ASSERT_ERR(i == get_user_pos(txdesc_new));
            stop_user =
            #endif
            #if (NX_UMAC_PRESENT)
            // push the descriptor in the UMAC
            txu_cntrl_push(txdesc_new, queue_idx);
            #else
            // push the descriptor in the LMAC
            txl_cntrl_push(txdesc_new, queue_idx);
            #endif //(NX_UMAC_PRESENT)

            // Now we have pushed some payloads so we may need to trigger the prepare event
            pushed = true;

            // the descriptor has been consumed
            txdesc_src->ready = 0;

            // increment the embedded index by one
            (*txdesc_idx)++;

            #if RW_MUMIMO_TX_EN
            // Check if we are done with this user
            if (stop_user)
                break;
            #endif

            // Go to next descriptor
            txdesc_src = &txdesc_src_base[*txdesc_idx & nx_txdesc_cnt_msk[queue_idx]];
        }

        #if RW_MUMIMO_TX_EN
        // Proceed to the user stopping and switch if required
        if (stop_user)
        {
            // Make user inactive
            ipc_emb_env.user_active[queue_idx] &= ~CO_BIT(i);
            // Schedule next user
            continue;
        }
        #endif
    }

    // Check if some payloads have been pushed and have to be downloaded
    if (pushed)
    {
        #if (NX_AMPDU_TX)
        // Check if the current A-MPDU under construction has to be closed
        txl_agg_check(queue_idx);
        #endif //(NX_AMPDU_TX)
    }

    // re-enable the IPC interrupts for the TX queues
    #if RW_MUMIMO_TX_EN
    ipc_app2emb_unmask_set((ipc_emb_env.user_active[queue_idx] << (RW_USER_MAX * queue_idx + IPC_IRQ_A2E_TXDESC_FIRSTBIT))
                           & IPC_IRQ_A2E_TXDESC);
    #else
    ipc_app2emb_unmask_set(CO_BIT(queue_idx + IPC_IRQ_A2E_TXDESC_FIRSTBIT)
                                                        & IPC_IRQ_A2E_TXDESC);
    #endif

    // For profiling
    PROF_TX_MACIF_EVT_CLR();
}

/**
 ****************************************************************************************
 * @brief IRQ handler
 *
 ****************************************************************************************
 */
void ipc_emb_cfmback_irq(void)
{
    // read HW IPC irq
    uint32_t stat = ipc_app2emb_status_get();

    if (stat & IPC_IRQ_A2E_RXBUF_BACK)
    {
        // Mask RX buffer back IPC irq
        ipc_app2emb_unmask_clear(IPC_IRQ_A2E_RXBUF_BACK);

        // Ack it
        ipc_app2emb_ack_clear(IPC_IRQ_A2E_RXBUF_BACK);

        // Trigger the RX event handler because we now have some RX buffers
        // available
        ke_evt_set(KE_EVT_RXREADY_BIT);
    }

    #if NX_UMAC_PRESENT
    if (stat & IPC_IRQ_A2E_RXDESC_BACK)
    {
        // Mask RX buffer back IPC irq
        ipc_app2emb_unmask_clear(IPC_IRQ_A2E_RXDESC_BACK);

        // Ack it
        ipc_app2emb_ack_clear(IPC_IRQ_A2E_RXDESC_BACK);

        // Trigger the RX event handler because we now have some RX buffers
        // available
        ke_evt_set(KE_EVT_RXUREADY_BIT);
    }
    #endif
}

/**
 ****************************************************************************************
 * @brief Retrieve a hostbuf address for future DMA transfer of a MSG
 *
 * This function gives back the hostbuf address according to the current value of
 * the msg buffer index. It just returns the pointer values set by the upper
 * layer at init time and after each reception.
 *
 * @return Hostmsgbuf address.
 *
 ****************************************************************************************
 */
static uint32_t ipc_emb_hostmsgbuf_get(void)
{
    uint32_t hostmsgbuf;

    do
    {
        // Retrieve the wished hostbuf address
        hostmsgbuf = ipc_shared_env.msg_e2a_hostbuf_addr[ipc_emb_env.ipc_msge2a_buf_idx];

        // Check if the host buffer is available
        if (hostmsgbuf == 0)
            break;

        // And now clears it in the hostbuf array
        ipc_shared_env.msg_e2a_hostbuf_addr[ipc_emb_env.ipc_msge2a_buf_idx] = 0;

        // Increase the buffer index
        ipc_emb_env.ipc_msge2a_buf_idx = (ipc_emb_env.ipc_msge2a_buf_idx + 1)%IPC_MSGE2A_BUF_CNT;
    } while(0);

    return hostmsgbuf;
}

/**
 ****************************************************************************************
 * @brief Retrieve a hostbuf address for future DMA transfer of a debug buffer
 *
 * This function gives back the hostbuf address according to the current value of
 * the msg buffer index. It just returns the pointer values set by the upper
 * layer at init time and after each reception.
 *
 * @return Hostmsgbuf address.
 *
 ****************************************************************************************
 */
static uint32_t ipc_emb_hostdbgbuf_get(void)
{
    uint32_t hostdbgbuf;

    do
    {
        // Retrieve the wished hostbuf address
        hostdbgbuf = ipc_shared_env.dbg_hostbuf_addr[ipc_emb_env.ipc_dbg_buf_idx];

        // Check if the host buffer is available
        if (hostdbgbuf == 0)
            break;

        // And now clears it in the hostbuf array
        ipc_shared_env.dbg_hostbuf_addr[ipc_emb_env.ipc_dbg_buf_idx] = 0;

        // Increase the buffer index
        ipc_emb_env.ipc_dbg_buf_idx = (ipc_emb_env.ipc_dbg_buf_idx + 1)%IPC_DBGBUF_CNT;
    } while(0);

    return hostdbgbuf;
}

/*
****************************************************************************************
*/
uint32_t ipc_emb_hostdbgdumpbuf_get(void)
{
    uint32_t hostdbgdumpbuf;

    do
    {
        // Retrieve the wished hostbuf address
        hostdbgdumpbuf = ipc_shared_env.la_dbginfo_addr;

        // Check if the host buffer is available
        if (hostdbgdumpbuf == 0)
            break;

        // And now clears it in the hostbuf array
        ipc_shared_env.la_dbginfo_addr = 0;
    } while(0);

    return (hostdbgdumpbuf);
}

/*
****************************************************************************************
*/
bool ipc_emb_hostrxbuf_check(void)
{
    // Acknowledge the RX buffer back IRQ, in case we would need it later because no
    // buffers are available
    ipc_app2emb_ack_clear(IPC_IRQ_A2E_RXBUF_BACK);

    // Check if the buffer was correctly allocated
    #if (NX_UMAC_PRESENT)
    if (ipc_shared_env.host_rxbuf[ipc_emb_env.ipc_rxbuf_idx].dma_addr == 0)
    #else
    if (ipc_shared_env.host_rxbuf[ipc_emb_env.ipc_rxbuf_idx] == 0)
    #endif //(NX_UMAC_PRESENT)
    {
        // No buffer available, so enable the RX buffer back interrupt to be warned
        // as soon as the host has pushed a new buffer
        ipc_app2emb_unmask_set(IPC_IRQ_A2E_RXBUF_BACK);

        return (false);
    }

    return (true);
}

#if (NX_UMAC_PRESENT)
uint32_t ipc_emb_hostrxbuf_get(uint32_t *host_id)
#else
uint32_t ipc_emb_hostrxbuf_get(void)
#endif //(NX_UMAC_PRESENT)
{
    uint32_t host_buf_dma_addr;

    RX_HOSTBUF_IDX_CLR();
    RX_HOSTBUF_IDX_SET(ipc_emb_env.ipc_rxbuf_idx);

    // Retrieve the wished hostbuf address
    #if (NX_UMAC_PRESENT)
    host_buf_dma_addr = ipc_shared_env.host_rxbuf[ipc_emb_env.ipc_rxbuf_idx].dma_addr;
    #else
    host_buf_dma_addr = ipc_shared_env.host_rxbuf[ipc_emb_env.ipc_rxbuf_idx];
    #endif //(NX_UMAC_PRESENT)

    // Sanity check - dma_addr shall not be 0 as a ipc_emb_hostrxbuf_check should
    // have been called before to ensure that there is a buffer available
    ASSERT_ERR(host_buf_dma_addr != 0);

    #if (NX_UMAC_PRESENT)
    // Now clears it in the hostbuf array
    ipc_shared_env.host_rxbuf[ipc_emb_env.ipc_rxbuf_idx].dma_addr = 0;

    *host_id = ipc_shared_env.host_rxbuf[ipc_emb_env.ipc_rxbuf_idx].hostid;
    #else
    // Now clears it in the hostbuf array
    ipc_shared_env.host_rxbuf[ipc_emb_env.ipc_rxbuf_idx] = 0;
    #endif //(NX_UMAC_PRESENT)

    // Increase the host buffer index
    ipc_emb_env.ipc_rxbuf_idx = (ipc_emb_env.ipc_rxbuf_idx + 1) % IPC_RXBUF_CNT;

    return host_buf_dma_addr;
}

#if (NX_UMAC_PRESENT)
/*
****************************************************************************************
*/
bool ipc_emb_hostrxdesc_check(void)
{
    // Acknowledge the RX buffer back IRQ, in case we would need it later because no
    // buffers are available
    ipc_app2emb_ack_clear(IPC_IRQ_A2E_RXDESC_BACK);

    // Check if descriptors are available
    if (ipc_shared_env.host_rxdesc[ipc_emb_env.ipc_rxdesc_idx].dma_addr == 0)
    {
        // No buffer available, so enable the RX buffer back interrupt to be warned
        // as soon as the host has pushed a new buffer
        ipc_app2emb_unmask_set(IPC_IRQ_A2E_RXDESC_BACK);

        return false;
    }

    return true;
}

uint32_t ipc_emb_hostrxdesc_get(void)
{
    uint32_t host_desc_dma_addr;

    // Retrieve the wished hostbuf address
    host_desc_dma_addr = ipc_shared_env.host_rxdesc[ipc_emb_env.ipc_rxdesc_idx].dma_addr;

    if (host_desc_dma_addr)
    {
        // Now clears it in the hostbuf array
        ipc_shared_env.host_rxdesc[ipc_emb_env.ipc_rxdesc_idx].dma_addr = 0;

        // Increase the host descriptor index
        ipc_emb_env.ipc_rxdesc_idx = (ipc_emb_env.ipc_rxdesc_idx + 1) % IPC_RXDESC_CNT;
    }

    return (host_desc_dma_addr);
}
#endif //(NX_UMAC_PRESENT)

/*
****************************************************************************************
*/
uint32_t ipc_emb_hostradarbuf_get(void)
{
    uint32_t hostbuf;

    do
    {
        // Retrieve the wished hostbuf address
        hostbuf = ipc_shared_env.radarbuf_hostbuf[ipc_emb_env.ipc_radar_buf_idx];

        // Check if the buffer was correctly allocated
        if (hostbuf == 0)
            break;

        // Now clears it in the hostbuf array
        ipc_shared_env.radarbuf_hostbuf[ipc_emb_env.ipc_radar_buf_idx] = 0;

        // Increase the host buffer index
        ipc_emb_env.ipc_radar_buf_idx = (ipc_emb_env.ipc_radar_buf_idx + 1)%IPC_RADARBUF_CNT;

    } while(0);

    return hostbuf;
}

#if NX_UF_EN
uint32_t ipc_emb_hostunsuprxvectbuf_get(void)
{
    uint32_t hostbuf;

    // Retrieve the wished hostbuf address
    hostbuf = ipc_shared_env.unsuprxvecbuf_hostbuf[ipc_emb_env.ipc_unsup_rx_vec_buf_idx];

    if (hostbuf == 0)
        return 0;

    // Now clears it in the hostbuf array
    ipc_shared_env.unsuprxvecbuf_hostbuf[ipc_emb_env.ipc_unsup_rx_vec_buf_idx] = 0;

    // Increase the host buffer index
    ipc_emb_env.ipc_unsup_rx_vec_buf_idx = (ipc_emb_env.ipc_unsup_rx_vec_buf_idx + 1)%IPC_UNSUPRXVECBUF_CNT;

    return hostbuf;
}
#endif

/*
****************************************************************************************
*/
void ipc_emb_rxdata_ind(void)
{
    // Signal the forwarding of the packet to host by setting a LED
    LED_ON(LED_RX);

    // For profiling
    PROF_RX_IPC_IND_SET();

    // Trigger an interrupt the host, indicating a new descriptor is available
    ipc_emb2app_trigger_set(IPC_IRQ_E2A_RXDESC);

    // For profiling
    PROF_RX_IPC_IND_CLR();

    // Shut down the RX LED
    LED_OFF(LED_RX);
}

void ipc_emb_radar_event_ind(void)
{
    // Trigger an interrupt the host, indicating a new radar event is available
    ipc_emb2app_trigger_set(IPC_IRQ_E2A_RADAR);
}

#if NX_UF_EN
void ipc_emb_unsup_rx_vec_event_ind(void)
{
    // Trigger an interrupt the host, indicating a new unsupported rx vector is available
    ipc_emb2app_trigger_set(IPC_IRQ_E2A_UNSUP_RX_VEC);
}
#endif

void ipc_emb_txcfm_ind(uint32_t queue_bits)
{
    // Trigger an interrupt to the host, indicating a transmission confirmation
    ipc_emb2app_trigger_set(queue_bits << IPC_IRQ_E2A_TXCFM_POS);
}

void ipc_emb_prim_tbtt_ind(void)
{
    // Trigger an interrupt to the host, indicating the primary TBTT
    ipc_emb2app_trigger_set(IPC_IRQ_E2A_TBTT_PRIM);
}

void ipc_emb_sec_tbtt_ind(void)
{
    // Trigger an interrupt to the host, indicating the secondary TBTT
    ipc_emb2app_trigger_set(IPC_IRQ_E2A_TBTT_SEC);
}

/**
 ****************************************************************************************
 * @brief Copy and forward kernel messages received from the IPC to the destination task
 *
 * @param[in] kmsg_ipc Address of the message in the shared memory
 *
 ****************************************************************************************
 */
static void ipc_emb_kmsg_hdlr(struct ke_msg *kmsg_ipc)
{
    int i;

    // allocate buffer for incoming msg
    struct ke_msg *kmsg_dst = (struct ke_msg*)
        ke_malloc(sizeof(struct ke_msg) + kmsg_ipc->param_len);
    ASSERT_ERR(kmsg_dst != NULL);

    kmsg_dst->hdr.next = NULL;
    kmsg_dst->id = kmsg_ipc->id;
    kmsg_dst->dest_id = kmsg_ipc->dest_id;
    kmsg_dst->src_id = TASK_API;
    kmsg_dst->param_len = kmsg_ipc->param_len;

    // copy parameters
    uint32_t *src = kmsg_ipc->param;
    uint32_t *dst = kmsg_dst->param;
    for (i = 0; i < kmsg_dst->param_len; i+=4)
    {
        *dst++ = *src++;
    }

    // Read by host for sanity check when ack is received
    kmsg_ipc->src_id = ipc_emb_env.ipc_msgacke2a_cnt++;

    // message received has no external parameter
    ASSERT_ERR(ke_task_local(kmsg_dst->dest_id));

    ipc_emb2app_trigger_set(IPC_IRQ_E2A_MSG_ACK);

    // dispatch the message to the local task
    ke_msg_send(ke_msg2param(kmsg_dst));
}


/**
 ****************************************************************************************
 * @brief IRQ handler
 *
 ****************************************************************************************
 */
void ipc_emb_msg_irq(void)
{
    PROF_MSG_IRQ_SET();

    // check HW IPC irq
    if (ipc_app2emb_status_get() & IPC_IRQ_A2E_MSG)
    {
        // set IPC msg event
        ke_evt_set(KE_EVT_MACIF_MSG_BIT);

        // mask msg IPC irq
        ipc_app2emb_unmask_clear(IPC_IRQ_A2E_MSG);
    }

    PROF_MSG_IRQ_CLR();
}


/**
 ****************************************************************************************
 * @brief EVT handler
 *
 ****************************************************************************************
 */
void ipc_emb_msg_evt(int dummy)
{
    struct ipc_a2e_msg * msg_buf;

    // read irq raw status (because interrupt could be disabled)
    uint32_t status_irq = ipc_app2emb_rawstatus_get();

    while (status_irq & IPC_IRQ_A2E_MSG)
    {
        // only ack bits already set!
        ipc_app2emb_ack_clear(IPC_IRQ_A2E_MSG);

        // Point to the MSG buffer
        msg_buf = (struct ipc_a2e_msg *)&(ipc_shared_env.msg_a2e_buf);

        // Call the msg handler
        ipc_emb_kmsg_hdlr((struct ke_msg*)msg_buf);

        status_irq = ipc_app2emb_rawstatus_get();
    }

    // clear event BEFORE enabling the interrupt
    ke_evt_clear(KE_EVT_MACIF_MSG_BIT);

    // re-enable the interrupt
    ipc_app2emb_unmask_set(IPC_IRQ_A2E_MSG);

    return;
}

void ipc_emb_kmsg_fwd(const struct ke_msg *kmsg_src)
{
    uint32_t host_msg_buf;
    struct ipc_e2a_msg *msg = (struct ipc_e2a_msg *) &(ipc_shared_env.msg_e2a_buf);
    struct dma_desc *msg_dma_desc = (struct dma_desc *) &(ipc_shared_env.msg_dma_desc);

    PROF_MSG_FWD_SET();

    // Get the destination address for DMA transfer (host buffer)
    host_msg_buf = ipc_emb_hostmsgbuf_get();

    // For the time being, put an assertion here
    ASSERT_ERR(host_msg_buf != 0);

    // Build the real message to be transferred
    msg->id = kmsg_src->id;
    msg->param_len = kmsg_src->param_len * CHAR_LEN;
    // no need to fill the dummy tasks ids in
    if(kmsg_src->param_len != 0)
    {
        ASSERT_ERR(kmsg_src->param_len <= sizeof(msg->param));
        memcpy(msg->param, kmsg_src->param, kmsg_src->param_len);
    }

    // Free the incoming message now
    ke_msg_free(kmsg_src);

    // Only one DMA transfer will be needed to send the complete MSG.
    // Configure the unique DMA descriptor (actually the only field that is not constant)
    // - dest: beginning of the host MSG buffer
    msg_dma_desc->dest = host_msg_buf;

    // Push the DMA descriptor in the DMA engine
    dma_push(msg_dma_desc, msg_dma_desc, IPC_DMA_CHANNEL_CTRL_RX);

    PROF_MSG_FWD_CLR();

}

/*
****************************************************************************************
*/
void ipc_emb_msg_dma_int_handler(void)
{
    PROF_MSG_IPC_IND_SET();

    // acknowledge the interrupt
    dma_int_ack_clear(CO_BIT(IPC_DMA_LLI_MSG + DMA_LLI_IRQ_LSB));

    // Trigger an interrupt the host, indicating a new message is available
    ipc_emb2app_trigger_set(IPC_IRQ_E2A_MSG);

    PROF_MSG_IPC_IND_CLR();
}

/*
****************************************************************************************
*/
void ipc_emb_print_fwd(bool poll, const uint32_t len, char *string)
{
    uint32_t host_dbg_buf;
    struct ipc_dbg_msg *dbg_msg = (struct ipc_dbg_msg *) &(ipc_shared_env.dbg_buf);
    struct dma_desc *dbg_dma_desc = (struct dma_desc *) &(ipc_shared_env.dbg_dma_desc);
    uint32_t copy_len;

    // Get the length of the string for the copy (plus 1 because of the NULL terminating
    // string character, to be copied as well)
    copy_len = len + 1;

    // Get the destination address for DMA transfer (host buffer)
    host_dbg_buf = ipc_emb_hostdbgbuf_get();

    // No ASSERT in case the address is NULL, the SW will not hang because of Debug
    // (moreover ASSERT also uses Debug strings and would be stuck...)
    if (host_dbg_buf == 0) {
        return;
    }

    // Check if the string length is not too big, otherwise truncate it to IPC_DBG_PARAM_SIZE
    if (len >= IPC_DBG_PARAM_SIZE)
    {
        string[IPC_DBG_PARAM_SIZE - 3] = '*';
        string[IPC_DBG_PARAM_SIZE - 2] = '\n';
        string[IPC_DBG_PARAM_SIZE - 1] = 0;
        copy_len = IPC_DBG_PARAM_SIZE;
    }

    // Copy the string
    co_pack8p(CPU2HW(dbg_msg->string), (uint8_t *)string, copy_len);

    // Only one DMA transfer will be needed to send the complete MSG.
    // Configure the unique DMA descriptor (actually the only field that is not constant)
    // - dest: beginning of the host buffer
    dbg_dma_desc->dest = host_dbg_buf;

    // Disable interrupts in case we would have to poll
    GLOBAL_INT_DISABLE();

    // Push the DMA descriptor in the DMA engine
    dma_push(dbg_dma_desc, dbg_dma_desc, IPC_DMA_CHANNEL_CTRL_RX);

    // Check if we have to poll for the DMA to be finished
    if (poll)
    {
        dma_lli_poll(IPC_DMA_LLI_DBG + DMA_LLI_IRQ_LSB);

        ipc_emb_dbg_dma_int_handler();
    }

    // Reenable interrupts
    GLOBAL_INT_RESTORE();

}

/*
****************************************************************************************
*/
void ipc_emb_dbg_dma_int_handler(void)
{
    // acknowledge the interrupt
    dma_int_ack_clear(CO_BIT(IPC_DMA_LLI_DBG + DMA_LLI_IRQ_LSB));

    // Trigger an interrupt the host, indicating a new Debug string is available
    ipc_emb2app_trigger_set(IPC_IRQ_E2A_DBG);

}

/// @}
