/**
 ****************************************************************************************
 *
 * @file ipc_emb.h
 *
 * @brief IPC module context on emb side.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

#ifndef _IPC_EMB_H_
#define _IPC_EMB_H_

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "co_int.h"
#include "co_bool.h"

#include "compiler.h"

#include "co_status.h"
#include "ipc_shared.h"
#include "ke_queue.h"
#include "ke_event.h"
#include "reg_ipc_emb.h"

// forward declarations
struct ke_msg;

/**
 ****************************************************************************************
 * @addtogroup IPC
 * @{
 ****************************************************************************************
 */
/// Structure describing the IPC environment
struct ipc_emb_env_tag
{
    /// Queue of RX descriptors pending for indication to host
    struct co_list rx_queue;

    /// Queue of TX confirmations pending for indication to host
    struct co_list cfm_queue;

    #if (NX_UMAC_PRESENT)
    /// Index of the host RX descriptor array
    uint8_t ipc_rxdesc_idx;
    #endif //(NX_UMAC_PRESENT)
    /// Index of the host RX buffer array
    uint8_t ipc_rxbuf_idx;
    /// Index of the host radar buffer array
    uint8_t ipc_radar_buf_idx;
    /// index of the host MSG E2A buffer array
    uint8_t ipc_msge2a_buf_idx;
    /// index of the host Debug buffer array
    uint8_t ipc_dbg_buf_idx;
    /// count of E2A MSG ACKs of A2E MSGs
    uint8_t ipc_msgacke2a_cnt;
    /// Index of the host unsupported rx vector buffer array
    uint8_t ipc_unsup_rx_vec_buf_idx;

    /// Index used that points to the current TX desc
    uint32_t txdesc_idx[IPC_TXQUEUE_CNT][RW_USER_MAX];
    #if RW_MUMIMO_TX_EN
    /// Bitfield indicating which user positions can be scheduled
    uint8_t user_active[IPC_TXQUEUE_CNT];
    #endif
    /// Pointers to the TX descriptor arrays in shared RAM
    volatile struct txdesc_host *txdesc[IPC_TXQUEUE_CNT][RW_USER_MAX];
};

/// Embedded IPC environment variable
extern struct ipc_emb_env_tag ipc_emb_env;

/// Mask of the TX descriptor length per queue. These values allow wrapping the TX descriptor
/// indexes used to address the TX descriptor arrays in shared RAM.
extern const int nx_txdesc_cnt_msk[];

/**
 ****************************************************************************************
 * @brief Get the host address of the TX payload descriptor pattern
 *
 * @return The address of the TX descriptor pattern
 *
 ****************************************************************************************
 */
__INLINE uint32_t ipc_emb_tx_pattern_addr_get(void)
{
    return(ipc_shared_env.pattern_addr);
}

/**
 ****************************************************************************************
 * @brief Get the buffered data from the IPC shared memory for a given sta/tid
 *
 * @param[in] sta  STA index
 * @param[in] tid  TID
 *
 * @return number of bytes buffered
 *
 ****************************************************************************************
 */
__INLINE uint32_t ipc_emb_buffered_get(uint8_t sta, uint8_t tid)
{
    return(ipc_shared_env.buffered[sta][tid]);
}

/**
 ****************************************************************************************
 * @brief Check if some TX descriptors are available on the requested queue
 *
 * @param[in] queue_idx Index of the TX queue
 *
 * @return true if some descriptors are available, false otherwise
 *
 ****************************************************************************************
 */
__INLINE bool ipc_emb_tx_q_has_data(int queue_idx)
{
    volatile struct txdesc_host *txdesc_src;

    #if RW_MUMIMO_TX_EN
    for (int i = 0; i < nx_txuser_cnt[queue_idx]; i++)
    {
        txdesc_src = ipc_emb_env.txdesc[queue_idx][i]
                                  + (ipc_emb_env.txdesc_idx[queue_idx][i] & nx_txdesc_cnt_msk[queue_idx]);

        if (txdesc_src->ready)
            return true;
    }

    return false;
    #else
    txdesc_src = ipc_emb_env.txdesc[queue_idx][0]
                              + (ipc_emb_env.txdesc_idx[queue_idx][0] & nx_txdesc_cnt_msk[queue_idx]);

    // Return the status of the first descriptor
    return (txdesc_src->ready);
    #endif
}

/**
 ****************************************************************************************
 * @brief Check how many tx descriptors are available on the requested queue for a VIF
 *
 * @param[in] queue_idx Index of the TX queue
 * @param[in] vif_idx   Index of the Vif
 *
 * @return Number of txdesc ready on the TX queue for the specified vif
 *
 ****************************************************************************************
 */
uint8_t ipc_emb_tx_q_len(int queue_idx, int vif_idx);

/**
 ****************************************************************************************
 * @brief Convert a TX descriptor interrupt status in to the corresponding kernel event
 * bit field
 *
 * @param[in] stat TX descriptor interrupt status
 *
 * @return The bit field of kernel events
 *
 ****************************************************************************************
 */
__INLINE uint32_t ipc_emb_tx_evt_field(uint32_t stat)
{
    #if RW_MUMIMO_TX_EN
    return (((((stat & IPC_IRQ_A2E_AC0_MSK) != 0) << 0)
         | (((stat & IPC_IRQ_A2E_AC1_MSK) != 0) << 1)
         | (((stat & IPC_IRQ_A2E_AC2_MSK) != 0) << 2)
         | (((stat & IPC_IRQ_A2E_AC3_MSK) != 0) << 3)
         #if (NX_BEACONING)
         | (((stat & IPC_IRQ_A2E_BCN_MSK) != 0) << 4)
         #endif
           )  << (31 - KE_EVT_MACIF_TXDESC_AC0));
    #else
    return ((stat >> IPC_IRQ_A2E_TXDESC_FIRSTBIT) << (31 - KE_EVT_MACIF_TXDESC_AC0));
    #endif
}

#if RW_MUMIMO_TX_EN
/**
 ****************************************************************************************
 * @brief Re-enable the user queues that were disabled
 *
 * @param[in] queue_idx Index of the TX queue
 * @param[in] active_users Bit field showing the users to re-enable
 *
 ****************************************************************************************
 */
__INLINE void ipc_emb_enable_users(int queue_idx, uint8_t active_users)
{
    uint32_t enable = active_users << (RW_USER_MAX * queue_idx + IPC_IRQ_A2E_TXDESC_FIRSTBIT);

    // Update active users
    ipc_emb_env.user_active[queue_idx] = active_users;

    // Re-enable the IPC interrupts for these users
    ipc_app2emb_unmask_set(enable & IPC_IRQ_A2E_TXDESC);

    // Launch the event in case buffers are available
    ke_evt_set(ipc_emb_tx_evt_field(enable));
}
#endif

/**
 ****************************************************************************************
 * @brief Indicate to the upper side that at least one radar event is pending.
 *
 ****************************************************************************************
 */
void ipc_emb_radar_event_ind(void);

#if NX_UF_EN
/**
 ****************************************************************************************
 * @brief Indicate to the upper side that one unsupported rx vector is pending.
 *
 ****************************************************************************************
 */
void ipc_emb_unsup_rx_vec_event_ind(void);
#endif

/**
 ****************************************************************************************
 * @brief Indicate to the upper side that at least on confirmation is pending.
 *
 * @param[in]   queue_bits   Bitfields indicating for which queues confirmations are pending
 *
 ****************************************************************************************
 */
void ipc_emb_txcfm_ind(uint32_t queue_bits);


/**
 ****************************************************************************************
 * @brief Indicate to the upper side that the primary TBTT is pending
 *
 * Implementation of @ref macif_prim_tbtt_ind when control layer is running on a remote
 * host
 ****************************************************************************************
 */
void ipc_emb_prim_tbtt_ind(void);

/**
 ****************************************************************************************
 * @brief Indicate to the upper side that the secondary TBTT is pending
 *
 * Implementation of @ref macif_sec_tbtt_ind when control layer is running on a remote
 * host
 ****************************************************************************************
 */
void ipc_emb_sec_tbtt_ind(void);

/**
 ****************************************************************************************
 * @brief Checks if a host buffer is currently available.
 *
 * If no buffer is available, then the RX buffer pushed interrupt is enabled in order
 * to be warned about a new buffer push when RX flow control is enabled.
 *
 * @return true if a buffer is available, false otherwise.
 *
 ****************************************************************************************
 */
bool ipc_emb_hostrxbuf_check(void);

/**
 ****************************************************************************************
 * @brief Retrieve a hostbuf address for future DMA transfer
 *
 * This function gives back the hostbuf address according to the current value of
 * the rx buffer index. It just returns the pointer values set by the upper
 * layer at init time and after each reception.
 * The buffer address in memory is then cleared and the FIFO index is incremented. The
 * buffer is therefore not available anymore after the call.
 * Prior to the call to this function, a call to @ref ipc_emb_hostrxbuf_check has to be
 * performed to ensure that there is a buffer available.
 *
 * @if full_doc
 * @param[out] host_id    HostId linked to the buffer
 * @endif
 *
 * @return Hostbuf address.
 *
 ****************************************************************************************
 */
#if (NX_UMAC_PRESENT)
uint32_t ipc_emb_hostrxbuf_get(uint32_t *host_id);
#else
uint32_t ipc_emb_hostrxbuf_get(void);
#endif //(NX_UMAC_PRESENT)

#if (NX_UMAC_PRESENT)
/**
 ****************************************************************************************
 * @brief Check if a RX descriptor is available
 *
 * @return true if a descriptor is available, false otherwise.
 *
 ****************************************************************************************
 */
bool ipc_emb_hostrxdesc_check(void);

/**
 ****************************************************************************************
 * @brief Retrieve a host RX descriptor address for future DMA transfer
 *
 * @return RX descriptor address.
 *
 ****************************************************************************************
 */
uint32_t ipc_emb_hostrxdesc_get(void);
#endif //(NX_UMAC_PRESENT)

/**
 ****************************************************************************************
 * @brief Retrieve a hostbuf address for debug dump transfer
 *
 * @return Hostbuf address.
 *
 ****************************************************************************************
 */
uint32_t ipc_emb_hostdbgdumpbuf_get(void);

/**
 ****************************************************************************************
 * @brief Retrieve a hostbuf address for future radar event DMA transfer
 *
 * @return Hostbuf address.
 *
 ****************************************************************************************
 */
uint32_t ipc_emb_hostradarbuf_get(void);

#if NX_UF_EN
/**
 ****************************************************************************************
 * @brief Retrieve a hostbuf address for future rx vector DMA transfer
 *
 * @return Hostbuf address.
 *
 ****************************************************************************************
 */
uint32_t ipc_emb_hostunsuprxvectbuf_get(void);
#endif

/**
 ****************************************************************************************
 * @brief Sends an Data Reception Indication to the upper layer
 *
 * This function warns the upper layer about Data Reception through the setting of
 * IPC_IRQ_E2A_RXDESC trigger.
 *
 ****************************************************************************************
 */
void ipc_emb_rxdata_ind(void);


/**
 ****************************************************************************************
 * @brief Forward a kernel message to the host side (emb side)
 *
 * Implementation of @ref macif_kmsg_fwd when control layer is running on a remote host
 *
 * @param[in] ke_msg  Pointer on the message
 *
 ****************************************************************************************
 */
void ipc_emb_kmsg_fwd(const struct ke_msg *ke_msg);

/**
 ****************************************************************************************
 * @brief Initialize the IPC on the embedded side.
 *
 * @warning Since this function sets some Init values in IPC Shared memory it must not be
 * called before ipc_host_init() which resets all IPC Shared memory at boot.
 *
 ****************************************************************************************
 */
void ipc_emb_init(void);


/**
 ****************************************************************************************
 * @brief Stop the TX flow between the upper layers and the LMAC software
 *
 ****************************************************************************************
 */
void ipc_emb_tx_flow_off(void);

/**
 ****************************************************************************************
 * @brief Restart the TX flow between the upper layers and the LMAC software
 *
 ****************************************************************************************
 */
void ipc_emb_tx_flow_on(void);


/**
 ****************************************************************************************
 * @brief Handle Tx packet interrupt on the emb side
 *
 ****************************************************************************************
 */
void ipc_emb_tx_irq(void);


/**
 ****************************************************************************************
 * @brief Handle kernel message interrupt on the emb side
 *
 ****************************************************************************************
 */
void ipc_emb_msg_irq(void);


/**
 ****************************************************************************************
 * @brief Handle confirmation events interrupt on the emb side
 *
 ****************************************************************************************
 */
void ipc_emb_cfmback_irq(void);

/**
 ****************************************************************************************
 * @brief Kernel event handler for the handling of the TX descriptor pushed by the host
 *
 * @param[in] queue_idx Index of the TX queues for which the event is called
 *
 ****************************************************************************************
 */
void ipc_emb_tx_evt(int queue_idx);

/**
 ****************************************************************************************
 * @brief Kernel event handler for the handling of messages pushed by the host
 *
 * Implementation of @ref macif_msg_evt when control layer is running on a remote host
 *
 * @param[in] dummy Parameter not used but required to follow the kernel event callback
 * format
 *
 ****************************************************************************************
 */
void ipc_emb_msg_evt(int dummy);

/**
 ****************************************************************************************
 * @brief Handles the DMA interrupt generated when the MSG from Embedded side to
 * Application has been transferred.
 *
 ****************************************************************************************
 */
void ipc_emb_msg_dma_int_handler(void);

/**
 ****************************************************************************************
 * @brief Send an EMB string to the APP - IPC API function
 *
 * @param[in]   poll    Poll to ensure that print has gone when we exit the function
 * @param[in]   len     Length of the message
 * @param[in]   string  Pointer to the message
 *                      Start must be word aligned (assumed to be the sprintf buffer)
 *                      The string may be truncated if it is too long.
 *
 ****************************************************************************************
 */
void ipc_emb_print_fwd(bool poll, const uint32_t len, char *string);

/**
 ****************************************************************************************
 * @brief Handles the DMA interrupt generated when the Debug message from Embedded side to
 * Application has been transferred.
 *
 ****************************************************************************************
 */
void ipc_emb_dbg_dma_int_handler(void);

/// @} IPC_MISC

#endif // _IPC_EMB_H_
