/**
 ****************************************************************************************
 *
 * @file ipc_host.c
 *
 * @brief IPC module.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#ifndef __KERNEL__
#include <stdio.h>
#define IPC_ALLOC(a,b)   malloc(a)
#define IPC_FREE(a)      free(a)
#define IPC_HOST_DBG printf
#define REG_SW_SET_PROFILING(env, value)   do{  }while(0)
#define REG_SW_CLEAR_PROFILING(env, value)   do{  }while(0)
#define REG_SW_CLEAR_HOSTBUF_IDX_PROFILING(env)   do{  }while(0)
#define REG_SW_SET_HOSTBUF_IDX_PROFILING(env, val)   do{  }while(0)
#else
#define IPC_ALLOC(a,b)   kzalloc(a,b)
#define IPC_FREE(a)      kfree(a)
#define IPC_HOST_DBG printk
#include <linux/string.h>
#include <linux/kernel.h> // for IPC_HOST_DBG
#endif

#include "reg_ipc_app.h"
#include "ipc_host.h"

/*
 * TYPES DEFINITION
 ****************************************************************************************
 */

const int nx_txdesc_cnt[] =
{
    NX_TXDESC_CNT0,
    NX_TXDESC_CNT1,
    NX_TXDESC_CNT2,
    NX_TXDESC_CNT3,
    #if (NX_BEACONING)
    NX_TXDESC_CNT4,
    #endif
};

const int nx_txdesc_cnt_msk[] =
{
    NX_TXDESC_CNT0 - 1,
    NX_TXDESC_CNT1 - 1,
    NX_TXDESC_CNT2 - 1,
    NX_TXDESC_CNT3 - 1,
    #if (NX_BEACONING)
    NX_TXDESC_CNT4 - 1,
    #endif
};


/*
 * FUNCTIONS DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Handle the reception of a Rx Descriptor
 *
 ****************************************************************************************
 */
static void ipc_host_rxdesc_handler(struct ipc_host_env_tag *env)
{
    // For profiling
    REG_SW_SET_PROFILING(env->pthis, SW_PROF_IRQ_E2A_RXDESC);

    // LMAC has triggered an IT saying that a reception has occurred.
    // Then we first need to check the validity of the current hostbuf, and the validity
    // of the next hostbufs too, because it is likely that several hostbufs have been
    // filled within the time needed for this irq handling
    do {
        #if (NX_UMAC_PRESENT)
        // call the external function to indicate that a RX descriptor is received
        if (env->cb.recv_data_ind(env->pthis,
                                  env->ipc_host_rxdesc_array[env->ipc_host_rxdesc_idx].dma_addr) != 0)
        #else
        // call the external function to indicate that a RX packet is received
        if (env->cb.recv_data_ind(env->pthis,
                        env->ipc_host_rxbuf_array[env->ipc_host_rxbuf_idx].hostid) != 0)
        #endif //(NX_UMAC_PRESENT)
            break;
    }while(1);

    // For profiling
    REG_SW_CLEAR_PROFILING(env->pthis, SW_PROF_IRQ_E2A_RXDESC);
}

/**
 ****************************************************************************************
 */
void ipc_host_msg_handler(struct ipc_host_env_tag *env)
{
    // For profiling
    REG_SW_SET_PROFILING(env->pthis, SW_PROF_IRQ_E2A_MSG);

    // LMAC has triggered an IT saying that a message has been sent to upper layer.
    // Then we first need to check the validity of the current msg buf, and the validity
    // of the next buffers too, because it is likely that several buffers have been
    // filled within the time needed for this irq handling
    // call the external function to indicate that a RX packet is received
    while (env->cb.recv_msg_ind(env->pthis,
                    env->ipc_host_msgbuf_array[env->ipc_host_msge2a_idx].hostid) == 0)


    // For profiling
    REG_SW_CLEAR_PROFILING(env->pthis, SW_PROF_IRQ_E2A_MSG);

}

static void ipc_host_msgack_handler(struct ipc_host_env_tag *env)
{
    uint32_t hostid = env->msga2e_hostid;

    assert(hostid);

    env->msga2e_hostid = 0;
    env->msga2e_cnt++;
    env->cb.recv_msgack_ind(env->pthis, hostid);
}

/**
 ****************************************************************************************
 */
static void ipc_host_dbg_handler(struct ipc_host_env_tag *env)
{
    // For profiling
    REG_SW_SET_PROFILING(env->pthis, SW_PROF_IRQ_E2A_DBG);

    // LMAC has triggered an IT saying that a DBG message has been sent to upper layer.
    // Then we first need to check the validity of the current buffer, and the validity
    // of the next buffers too, because it is likely that several buffers have been
    // filled within the time needed for this irq handling
    // call the external function to indicate that a RX packet is received
    while(env->cb.recv_dbg_ind(env->pthis,
            env->ipc_host_dbgbuf_array[env->ipc_host_dbg_idx].hostid) == 0);

    // For profiling
    REG_SW_CLEAR_PROFILING(env->pthis, SW_PROF_IRQ_E2A_DBG);
}

/**
 ****************************************************************************************
 */
uint32_t ipc_host_tx_flush(struct ipc_host_env_tag *env, const int queue_idx)
{
    uint32_t used_idx = env->txdesc_used_idx[queue_idx];
    uint32_t host_id = env->tx_host_id[queue_idx][used_idx & nx_txdesc_cnt_msk[queue_idx]];

    // call the external function to indicate that a TX packet is freed
    if (host_id != 0)
    {
        // Reset the host id in the array
        env->tx_host_id[queue_idx][used_idx & nx_txdesc_cnt_msk[queue_idx]] = 0;

        // Increment the used index
        env->txdesc_used_idx[queue_idx]++;
    }

    return (host_id);
}

/**
 ****************************************************************************************
 */
static void ipc_host_tx_cfm_handler(struct ipc_host_env_tag *env, const int queue_idx)
{
    // TX confirmation descriptors have been received
    REG_SW_SET_PROFILING(env->pthis, SW_PROF_IRQ_E2A_TXCFM);
    while (1)
    {
        // Get the used index and increase it. We do the increase before knowing if the
        // current buffer is confirmed because the callback function may call the
        // ipc_host_txdesc_get() in case flow control was enabled and the index has to be
        // already at the good value to ensure that the test of FIFO full is correct
        uint32_t used_idx = env->txdesc_used_idx[queue_idx]++;
        uint32_t host_id = env->tx_host_id[queue_idx][used_idx & nx_txdesc_cnt_msk[queue_idx]];

        // Reset the host id in the array
        env->tx_host_id[queue_idx][used_idx & nx_txdesc_cnt_msk[queue_idx]] = 0;

        // call the external function to indicate that a TX packet is freed
        if (host_id == 0)
        {
            // No more confirmations, so put back the used index at its initial value
            env->txdesc_used_idx[queue_idx] = used_idx;
            break;
        }

        if (env->cb.send_data_cfm(env->pthis, host_id) != 0)
        {
            // No more confirmations, so put back the used index at its initial value
            env->txdesc_used_idx[queue_idx] = used_idx;
            env->tx_host_id[queue_idx][used_idx & nx_txdesc_cnt_msk[queue_idx]] = host_id;
            // and exit the loop
            break;
        }

    }
    REG_SW_CLEAR_PROFILING(env->pthis, SW_PROF_IRQ_E2A_TXCFM);
}

/**
 ****************************************************************************************
 */
void ipc_host_init(struct ipc_host_env_tag *env,
                  struct ipc_host_cb_tag *cb,
                  struct ipc_shared_env_tag *shared_env_ptr,
                  void *pthis)
{
    unsigned int i;
    unsigned int size;
    unsigned int * dst;

    // Reset the environments
    // Perform a reset word per word because memset() does not correctly reset all
    // (due to misaligned accesses)
    // Reset the IPC Shared memory
    dst = (unsigned int *)(shared_env_ptr);
    size = (unsigned int)sizeof(struct ipc_shared_env_tag);
    for (i=0; i < size; i+=4)
    {
        *dst++ = 0;
    }
    // Reset the IPC Host environment
    dst = (unsigned int *)(env);
    size = (unsigned int)sizeof(struct ipc_host_env_tag);
    for (i=0; i < size; i+=4)
    {
        *dst++ = 0;
    }
    // Initialize the shared environment pointer
    env->shared = shared_env_ptr;

    // Save the callbacks in our own environment
    env->cb = *cb;

    // Save the pointer to the register base
    env->pthis = pthis;

    // Initialize buffers numbers and buffers sizes needed for DMA Receptions
    env->rx_bufnb = 0;
    env->rx_bufsz = IPC_RXBUF_SIZE;
    env->ipc_e2amsg_bufnb = IPC_MSGE2A_BUF_CNT;
    env->ipc_e2amsg_bufsz = sizeof(struct ipc_e2a_msg);
    env->ipc_dbg_bufnb = IPC_DBGBUF_CNT;
    env->ipc_dbg_bufsz = sizeof(struct ipc_dbg_msg);

    // Initialize the pointers to the hostid arrays
    env->tx_host_id[0] = env->tx_host_id0;
    env->tx_host_id[1] = env->tx_host_id1;
    env->tx_host_id[2] = env->tx_host_id2;
    env->tx_host_id[3] = env->tx_host_id3;
    #if (NX_BEACONING)
    env->tx_host_id[4] = env->tx_host_id4;
    #endif

    // Initialize the pointers to the TX descriptor arrays
    env->txdesc[0] = shared_env_ptr->txdesc0;
    env->txdesc[1] = shared_env_ptr->txdesc1;
    env->txdesc[2] = shared_env_ptr->txdesc2;
    env->txdesc[3] = shared_env_ptr->txdesc3;
    #if (NX_BEACONING)
    env->txdesc[4] = shared_env_ptr->txdesc4;
    #endif
}

/**
 ****************************************************************************************
 */
void ipc_host_patt_addr_push(struct ipc_host_env_tag *env, uint32_t addr)
{
    struct ipc_shared_env_tag *shared_env_ptr = env->shared;

    // Copy the address
    shared_env_ptr->pattern_addr = addr;
}

int ipc_host_rxbuf_push(struct ipc_host_env_tag *env, uint32_t hostid, uint32_t hostbuf)
{
    struct ipc_shared_env_tag *shared_env_ptr = env->shared;

    REG_SW_CLEAR_HOSTBUF_IDX_PROFILING(env->pthis);
    REG_SW_SET_HOSTBUF_IDX_PROFILING(env->pthis, buf_id);

    // Save the hostid and the hostbuf in global array
    env->ipc_host_rxbuf_array[env->ipc_host_rxbuf_idx].hostid   = hostid;
    env->ipc_host_rxbuf_array[env->ipc_host_rxbuf_idx].dma_addr = hostbuf;

    // Copy the hostbuf (DMA address) in the ipc shared memory
    #if (NX_UMAC_PRESENT)
    shared_env_ptr->host_rxbuf[env->ipc_host_rxbuf_idx].hostid   = hostid;
    shared_env_ptr->host_rxbuf[env->ipc_host_rxbuf_idx].dma_addr = hostbuf;
    #else
    shared_env_ptr->host_rxbuf[env->ipc_host_rxbuf_idx] = hostbuf;
    #endif //(NX_UMAC_PRESENT)

    // Signal to the embedded CPU that at least one buffer is available
    ipc_app2emb_trigger_set(env->pthis, IPC_IRQ_A2E_RXBUF_BACK);

    // Increment the array index
    env->ipc_host_rxbuf_idx = (env->ipc_host_rxbuf_idx + 1) % IPC_RXBUF_CNT;

    return (0);
}

#if (NX_UMAC_PRESENT)
/**
 ****************************************************************************************
 */
void ipc_host_rx_bufnb_inc(struct ipc_host_env_tag *env)
{
    env->rx_bufnb++;
}

/**
 ****************************************************************************************
 */
void ipc_host_rx_bufnb_dec(struct ipc_host_env_tag *env)
{
    env->rx_bufnb--;

    assert(env->rx_bufnb >= IPC_RXBUF_CNT);
}

/**
 ****************************************************************************************
 */
int ipc_host_rxdesc_push(struct ipc_host_env_tag *env, uint32_t dma_addr)
{
    struct ipc_shared_env_tag *shared_env = env->shared;

    // Reset the RX Descriptor DMA Address and increment the counter
    env->ipc_host_rxdesc_array[env->ipc_host_rxdesc_idx].dma_addr = dma_addr;
    shared_env->host_rxdesc[env->ipc_host_rxdesc_idx].dma_addr  = dma_addr;

    // Signal to the embedded CPU that at least one descriptor is available
    ipc_app2emb_trigger_set(env->pthis, IPC_IRQ_A2E_RXDESC_BACK);

    env->ipc_host_rxdesc_idx = (env->ipc_host_rxdesc_idx + 1) % IPC_RXDESC_CNT;

    return (0);
}
#endif //(NX_UMAC_PRESENT)

/**
 ****************************************************************************************
 */
int ipc_host_msgbuf_push(struct ipc_host_env_tag *env, uint32_t hostid,
                                                       uint32_t hostbuf)
{
    struct ipc_shared_env_tag *shared_env_ptr = env->shared;

    // Save the hostid and the hostbuf in global array
    env->ipc_host_msgbuf_array[env->ipc_host_msge2a_idx].hostid = hostid;
    env->ipc_host_msgbuf_array[env->ipc_host_msge2a_idx].dma_addr = hostbuf;

    // Copy the hostbuf (DMA address) in the ipc shared memory
    shared_env_ptr->msg_e2a_hostbuf_addr[env->ipc_host_msge2a_idx] = hostbuf;

    // Increment the array index
    env->ipc_host_msge2a_idx = (env->ipc_host_msge2a_idx +1)%IPC_MSGE2A_BUF_CNT;

    return (0);
}

/**
 ****************************************************************************************
 */
int ipc_host_dbgbuf_push(struct ipc_host_env_tag *env, uint32_t hostid,
                                                       uint32_t hostbuf)
{
    struct ipc_shared_env_tag *shared_env_ptr = env->shared;

    // Save the hostid and the hostbuf in global array
    env->ipc_host_dbgbuf_array[env->ipc_host_dbg_idx].hostid = hostid;
    env->ipc_host_dbgbuf_array[env->ipc_host_dbg_idx].dma_addr = hostbuf;

    // Copy the hostbuf (DMA address) in the ipc shared memory
    shared_env_ptr->dbg_hostbuf_addr[env->ipc_host_dbg_idx] = hostbuf;

    // Increment the array index
    env->ipc_host_dbg_idx = (env->ipc_host_dbg_idx +1)%IPC_DBGBUF_CNT;

    return (0);
}

/**
 ****************************************************************************************
 */
void ipc_host_dbginfobuf_push(struct ipc_host_env_tag *env, uint32_t infobuf)
{
    struct ipc_shared_env_tag *shared_env_ptr = env->shared;

    // Copy the hostbuf (DMA address) in the ipc shared memory
    shared_env_ptr->la_dbginfo_addr = infobuf;
}

/**
 ****************************************************************************************
 */
volatile struct txdesc_host *ipc_host_txdesc_get(struct ipc_host_env_tag *env, const int queue_idx)
{
    volatile struct txdesc_host *txdesc_free;
    uint32_t used_idx = env->txdesc_used_idx[queue_idx];
    uint32_t free_idx = env->txdesc_free_idx[queue_idx];

    assert(queue_idx < IPC_TXQUEUE_CNT);
    assert((free_idx - used_idx) <= nx_txdesc_cnt[queue_idx]);

    // Check if a free descriptor is available
    if (free_idx != (used_idx + nx_txdesc_cnt[queue_idx]))
    {
        // Get the pointer to the first free descriptor
        txdesc_free = env->txdesc[queue_idx] + (free_idx & nx_txdesc_cnt_msk[queue_idx]);
    }
    else
    {
        txdesc_free = NULL;
    }

    return txdesc_free;
}

/**
 ****************************************************************************************
 */
void ipc_host_txdesc_push(struct ipc_host_env_tag *env, const int queue_idx, const uint32_t host_id)
{
    volatile struct txdesc_host *txdesc_pushed = env->txdesc[queue_idx]
                                          + (env->txdesc_free_idx[queue_idx] & nx_txdesc_cnt_msk[queue_idx]);

    // Descriptor is now ready
    txdesc_pushed->ready = 0xFFFFFFFF;

    // Save the host id in the environment
    env->tx_host_id[queue_idx][env->txdesc_free_idx[queue_idx] & nx_txdesc_cnt_msk[queue_idx]] = host_id;

    // Increment the index
    env->txdesc_free_idx[queue_idx]++;

    // trigger interrupt!!!
    //REG_SW_SET_PROFILING(env->pthis, CO_BIT(queue_idx+SW_PROF_IRQ_A2E_TXDESC_FIRSTBIT));
    ipc_app2emb_trigger_setf(env->pthis, CO_BIT(queue_idx+IPC_IRQ_A2E_TXDESC_FIRSTBIT));
}

/**
 ****************************************************************************************
 */
void ipc_host_irq(struct ipc_host_env_tag *env, uint32_t status)
{
        // Optimized for only one IRQ at a time
        if (status & IPC_IRQ_E2A_RXDESC)
        {
            // acknowledge the interrupt BEFORE handling the request
            ipc_emb2app_ack_clear(env->pthis, IPC_IRQ_E2A_RXDESC);

            // handle the RX descriptor reception
            ipc_host_rxdesc_handler(env);
        }
        if (status & IPC_IRQ_E2A_TXCFM)
        {
            int i;

            // handle the TX confirmation reception
            for (i = 0; i < IPC_TXQUEUE_CNT; i++)
            {
                uint32_t q_bit = CO_BIT(i + IPC_IRQ_E2A_TXCFM_POS);
                if (status & q_bit)
                {
                    // it is mandatory to acknowledge before handling the confirmation
                    ipc_emb2app_ack_clear(env->pthis, q_bit);

                    // handle the confirmation
                    ipc_host_tx_cfm_handler(env, i);
                }
            }
        }
        if (status & IPC_IRQ_E2A_MSG)
        {
            // acknowledge the interrupt BEFORE handling the request
            ipc_emb2app_ack_clear(env->pthis, IPC_IRQ_E2A_MSG);

            ipc_host_msg_handler(env);
        }
        
        if (status & IPC_IRQ_E2A_MSG_ACK)
        {
            ipc_emb2app_ack_clear(env->pthis, IPC_IRQ_E2A_MSG_ACK);

            ipc_host_msgack_handler(env);
        }

        if (status & IPC_IRQ_E2A_DBG)
        {
            // acknowledge the interrupt BEFORE handling the request
            ipc_emb2app_ack_clear(env->pthis, IPC_IRQ_E2A_DBG);

            ipc_host_dbg_handler(env);
        }

        if (status & IPC_IRQ_E2A_TBTT_PRIM)
        {
            // acknowledge the interrupt BEFORE handling the request
            ipc_emb2app_ack_clear(env->pthis, IPC_IRQ_E2A_TBTT_PRIM);

            env->cb.prim_tbtt_ind(env->pthis);
        }

        if (status & IPC_IRQ_E2A_TBTT_SEC)
        {
            // acknowledge the interrupt BEFORE handling the request
            ipc_emb2app_ack_clear(env->pthis, IPC_IRQ_E2A_TBTT_SEC);

            env->cb.sec_tbtt_ind(env->pthis);
        }
}

/**
 ****************************************************************************************
 */
int ipc_host_msg_push(struct ipc_host_env_tag *env, void *msg_buf, uint16_t len)
{
    int i;
    uint32_t *src, *dst;

    REG_SW_SET_PROFILING(env->pthis, SW_PROF_IPC_MSGPUSH);

    assert(!env->msga2e_hostid);

    // Copy the message into the IPC MSG buffer
    src = (uint32_t*) msg_buf;
    dst = (uint32_t*)&(env->shared->msg_a2e_buf.msg);

    // Copy the message in the IPC queue
    for (i=0; i<len; i+=4)
    {
        *dst++ = *src++;
    }

    env->msga2e_hostid = (uint32_t)msg_buf;

    // Trigger the irq to send the message to EMB
    ipc_app2emb_trigger_set(env->pthis, IPC_IRQ_A2E_MSG);

    REG_SW_CLEAR_PROFILING(env->pthis, SW_PROF_IPC_MSGPUSH);

    return (0);
}

/**
 ****************************************************************************************
 */
void ipc_host_enable_irq(struct ipc_host_env_tag *env, uint32_t value)
{
    // Enable the handled interrupts
    ipc_emb2app_unmask_set(env->pthis, value);
}


/**
 ****************************************************************************************
 */
uint32_t ipc_host_get_status(struct ipc_host_env_tag *env)
{
    volatile uint32_t status;

    status = ipc_emb2app_status_get(env->pthis);

    return status;
}

/**
 ****************************************************************************************
 */
uint32_t ipc_host_get_rawstatus(struct ipc_host_env_tag *env)
{
    volatile uint32_t rawstatus;

    rawstatus = ipc_emb2app_rawstatus_get(env->pthis);

    return rawstatus;
}

