/**
 ****************************************************************************************
 *
 * @file ipc_emb.c
 *
 * @brief IPC module.
 *
 * Copyright C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup IPC
 * @{
 ****************************************************************************************
 */
#include "dbg_assert.h"

#include "co_math.h"
#include "ke_mem.h"
#include "ke_msg.h"
#include "ke_task.h"
#include "ke_event.h"

#include "dma.h"
#include "ipc_emb.h"
#include "ipc_shared.h"
#include "dbg.h"
#include "ipc_shared.h"
#include "reg_ipc_emb.h"

#include "fhost_ipc.h"
#include "rtos_al.h"

/// Structure describing the IPC environment
struct ipc_emb_env_tag
{
    /// index of the host MSG E2A buffer array
    uint8_t ipc_msge2a_buf_idx;
    /// index of the host Debug buffer array
    uint8_t ipc_dbg_buf_idx;
    /// count of E2A MSG ACKs of A2E MSGs
    uint8_t ipc_msgacke2a_cnt;
};

/// Configure the DMA control field for a kernel message upload
#define MSG_LLICTRL(irqenable)                                                           \
    ( (irqenable)?(IPC_DMA_LLI_IRQ_EN|(IPC_DMA_LLI_MSG << IPC_DMA_LLI_IRQ_POS)) : 0)

/// Configure the DMA control field for a debug message upload
#define DBG_LLICTRL(irqenable)                                                           \
    ( (irqenable)?(IPC_DMA_LLI_IRQ_EN|(IPC_DMA_LLI_DBG << IPC_DMA_LLI_IRQ_POS)) : 0)

/// Global variable containing the IPC_EMB environment
static struct ipc_emb_env_tag ipc_emb_env;

/// Global semaphore indicating whether there are messages under transmission from emb to host via DMA
static rtos_semaphore ipc_emb_msg_sem;

void ipc_emb_init(void)
{
    // clear the entire IPC EMB environment
    memset(&ipc_emb_env, 0, sizeof(ipc_emb_env));

    // Set const fields of the MSG DMA descriptor (to send MSGs to Host)
    //   - DMA contents always starts at
    ipc_shared_env.msg_dma_desc.src = CPU2HW(&(ipc_shared_env.msg_e2a_buf));
    //   - length
    ipc_shared_env.msg_dma_desc.length = sizeof_b(struct ipc_fhost_msg);
    //   - no other descriptor will be linked to it
    ipc_shared_env.msg_dma_desc.next = 0;
    //   - enable interrupt generation on this descriptor
    ipc_shared_env.msg_dma_desc.ctrl = MSG_LLICTRL(1);
    // Put the pattern in the msg buffer
    ipc_shared_env.msg_e2a_buf.pattern = IPC_MSGE2A_VALID_PATTERN;

    /// Set const fields of the Debug DMA descriptor (to send strings to Host)
    //   - DMA contents always starts at
    ipc_shared_env.dbg_dma_desc.src = CPU2HW(&(ipc_shared_env.dbg_buf));
    //   - length
    ipc_shared_env.dbg_dma_desc.length = sizeof_b(struct ipc_dbg_msg);
    //   - no other descriptor will be linked to it
    ipc_shared_env.dbg_dma_desc.next = 0;
    //   - enable interrupt generation on this descriptor
    ipc_shared_env.dbg_dma_desc.ctrl = DBG_LLICTRL(1);
    // Put the pattern in the debug buffer
    ipc_shared_env.dbg_buf.pattern = IPC_DBG_VALID_PATTERN;

    // Sanity check - verify that the FW is compiled with the right IPC version
    ASSERT_ERR(ipc_emb_signature_get() == IPC_EMB_SIGNATURE_RESET);

    // configure the IPC line selection
    ipc_app2emb_line_sel_low_set(0);
    ipc_app2emb_line_sel_high_set(0);

    ipc_app2emb0_sel_setf(0); // DBG
    ipc_app2emb1_sel_setf(1); // MSG

    // enable the all interrupts from APP
    ipc_app2emb_unmask_set(IPC_IRQ_A2E_ALL);

    // Will be created when needed
    ipc_emb_msg_sem = NULL;
}


/***************************************************************************************
 * A2E MSG
 **************************************************************************************/
void ipc_emb_msg_irq(void)
{
    PROF_MSG_IRQ_SET();

    // check HW IPC irq
    if (ipc_app2emb_status_get() & IPC_IRQ_A2E_MSG)
    {
        uint16_t id = ipc_shared_env.msg_a2e_buf.id;
        ipc_app2emb_ack_clear(IPC_IRQ_A2E_MSG);
        ipc_shared_env.msg_a2e_buf.id = ipc_emb_env.ipc_msgacke2a_cnt++;
        if (fhost_ipc_write_msg(id, (void *)ipc_shared_env.msg_a2e_buf.data,
                                ipc_shared_env.msg_a2e_buf.len))
        {
            /* queue was full */
            ipc_emb2app_trigger_set(IPC_IRQ_E2A_MSG_ACK);
            ipc_shared_env.msg_a2e_buf.id = 0xdeaddead;
            /* TODO: report 'message ignored' status */
        }
    }

    PROF_MSG_IRQ_CLR();
}

void ipc_emb_ack_msg(void)
{
    ipc_emb2app_trigger_set(IPC_IRQ_E2A_MSG_ACK);
}


/***************************************************************************************
 * E2A MSG
 **************************************************************************************/
/**
 ****************************************************************************************
 * @brief Retrieve a hostbuf address for future DMA transfer of a MSG
 *
 * This function gives back the hostbuf address according to the current value of
 * the msg buffer index. It just returns the pointer values set by the upper
 * layer at init time and after each reception.
 *
 * @return Hostmsgbuf address.
 *
 ****************************************************************************************
 */
static uint32_t ipc_emb_hostmsgbuf_get(void)
{
    uint32_t hostmsgbuf;

    do
    {
        // Retrieve the wished hostbuf address
        hostmsgbuf = ipc_shared_env.msg_e2a_hostbuf_addr[ipc_emb_env.ipc_msge2a_buf_idx];

        // Check if the host buffer is available
        if (hostmsgbuf == 0)
            break;

        // And now clears it in the hostbuf array
        ipc_shared_env.msg_e2a_hostbuf_addr[ipc_emb_env.ipc_msge2a_buf_idx] = 0;

        // Increase the buffer index
        ipc_emb_env.ipc_msge2a_buf_idx = (ipc_emb_env.ipc_msge2a_buf_idx + 1)%IPC_MSGE2A_BUF_CNT;
    } while(0);

    return hostmsgbuf;
}

void ipc_emb_msg_fwd(int id, int len, uint32_t *data)
{
    uint32_t host_msg_buf;
    struct ipc_fhost_msg *msg = (struct ipc_fhost_msg *) &(ipc_shared_env.msg_e2a_buf);
    struct dma_desc *msg_dma_desc = (struct dma_desc *) &(ipc_shared_env.msg_dma_desc);

    PROF_MSG_FWD_SET();

    if ((ipc_emb_msg_sem == NULL) &&
        rtos_semaphore_create(&ipc_emb_msg_sem, 1, 1))
        return;

    rtos_semaphore_wait(ipc_emb_msg_sem, -1);

    // Get the destination address for DMA transfer (host buffer)
    host_msg_buf = ipc_emb_hostmsgbuf_get();

    // For the time being, put an assertion here
    if (!host_msg_buf) {
        TRACE_FHOST(ERR, "Failed to get IPC buffer, message discarded");
        rtos_semaphore_signal(ipc_emb_msg_sem, false);
        return;
    }

    // Build the real message to be transferred
    msg->id = id;
    msg->len = len;
    if(len != 0)
        memcpy(msg->data, data, len);

    // Only one DMA transfer will be needed to send the complete MSG.
    // Configure the unique DMA descriptor (actually the only field that is not constant)
    // - dest: beginning of the host MSG buffer
    msg_dma_desc->dest = host_msg_buf;

    // Push the DMA descriptor in the DMA engine
    dma_push(msg_dma_desc, msg_dma_desc, IPC_DMA_CHANNEL_CTRL_RX);

    PROF_MSG_FWD_CLR();

}

void ipc_emb_msg_dma_int_handler(void)
{
    PROF_MSG_IPC_IND_SET();

    // acknowledge the interrupt
    dma_int_ack_clear(CO_BIT(IPC_DMA_LLI_MSG + DMA_LLI_IRQ_LSB));

    // Trigger an interrupt the host, indicating a new message is available
    ipc_emb2app_trigger_set(IPC_IRQ_E2A_MSG);

    rtos_semaphore_signal(ipc_emb_msg_sem, true);
    PROF_MSG_IPC_IND_CLR();
}

/***************************************************************************************
 * E2A DBG MSG
 **************************************************************************************/
/**
 ****************************************************************************************
 * @brief Retrieve a hostbuf address for future DMA transfer of a debug buffer
 *
 * This function gives back the hostbuf address according to the current value of
 * the msg buffer index. It just returns the pointer values set by the upper
 * layer at init time and after each reception.
 *
 * @return Hostmsgbuf address.
 *
 ****************************************************************************************
 */
static uint32_t ipc_emb_hostdbgbuf_get(void)
{
    uint32_t hostdbgbuf;

    do
    {
        // Retrieve the wished hostbuf address
        hostdbgbuf = ipc_shared_env.dbg_hostbuf_addr[ipc_emb_env.ipc_dbg_buf_idx];

        // Check if the host buffer is available
        if (hostdbgbuf == 0)
            break;

        // And now clears it in the hostbuf array
        ipc_shared_env.dbg_hostbuf_addr[ipc_emb_env.ipc_dbg_buf_idx] = 0;

        // Increase the buffer index
        ipc_emb_env.ipc_dbg_buf_idx = (ipc_emb_env.ipc_dbg_buf_idx + 1)%IPC_DBGBUF_CNT;
    } while(0);

    return hostdbgbuf;
}

void ipc_emb_print_fwd(bool poll, const uint32_t len, char *string)
{
    uint32_t host_dbg_buf;
    struct ipc_dbg_msg *dbg_msg = (struct ipc_dbg_msg *) &(ipc_shared_env.dbg_buf);
    struct dma_desc *dbg_dma_desc = (struct dma_desc *) &(ipc_shared_env.dbg_dma_desc);
    uint32_t copy_len;

    // Get the length of the string for the copy (plus 1 because of the NULL terminating
    // string character, to be copied as well)
    copy_len = len + 1;

    // Get the destination address for DMA transfer (host buffer)
    host_dbg_buf = ipc_emb_hostdbgbuf_get();

    // No ASSERT in case the address is NULL, the SW will not hang because of Debug
    // (moreover ASSERT also uses Debug strings and would be stuck...)
    if (host_dbg_buf == 0) {
        return;
    }

    // Check if the string length is not too big, otherwise truncate it to IPC_DBG_PARAM_SIZE
    if (len >= IPC_DBG_PARAM_SIZE)
    {
        string[IPC_DBG_PARAM_SIZE - 3] = '*';
        string[IPC_DBG_PARAM_SIZE - 2] = '\n';
        string[IPC_DBG_PARAM_SIZE - 1] = 0;
        copy_len = IPC_DBG_PARAM_SIZE;
    }

    // Copy the string
    co_pack8p(CPU2HW(dbg_msg->string), (uint8_t *)string, copy_len);

    // Only one DMA transfer will be needed to send the complete MSG.
    // Configure the unique DMA descriptor (actually the only field that is not constant)
    // - dest: beginning of the host buffer
    dbg_dma_desc->dest = host_dbg_buf;

    // Disable interrupts in case we would have to poll
    GLOBAL_INT_DISABLE();

    // Push the DMA descriptor in the DMA engine
    dma_push(dbg_dma_desc, dbg_dma_desc, IPC_DMA_CHANNEL_CTRL_RX);

    // Check if we have to poll for the DMA to be finished
    if (poll)
    {
        dma_lli_poll(IPC_DMA_LLI_DBG + DMA_LLI_IRQ_LSB);

        ipc_emb_dbg_dma_int_handler();
    }

    // Reenable interrupts
    GLOBAL_INT_RESTORE();
}

void ipc_emb_dbg_dma_int_handler(void)
{
    // acknowledge the interrupt
    dma_int_ack_clear(CO_BIT(IPC_DMA_LLI_DBG + DMA_LLI_IRQ_LSB));

    // Trigger an interrupt the host, indicating a new Debug string is available
    ipc_emb2app_trigger_set(IPC_IRQ_E2A_DBG);

}

/***************************************************************************************
 * E2A DBG DUMP
 **************************************************************************************/
uint32_t ipc_emb_hostdbgdumpbuf_get(void)
{
    uint32_t hostdbgdumpbuf;

    do
    {
        // Retrieve the wished hostbuf address
        hostdbgdumpbuf = ipc_shared_env.la_dbginfo_addr;

        // Check if the host buffer is available
        if (hostdbgdumpbuf == 0)
            break;

        // And now clears it in the hostbuf array
        ipc_shared_env.la_dbginfo_addr = 0;
    } while(0);

    return (hostdbgdumpbuf);
}


/***************************************************************************************
 * CONFIG buffer
 **************************************************************************************/
void *ipc_emb_config_get_next(void *start, uint16_t *id, uint16_t *len)
{
    uint32_t *config_ptr;

    if (!start)
    {
        config_ptr = (uint32_t *)ipc_shared_env.config;
    }
    else
    {
        uint16_t len_prev;
        config_ptr = start;
        len_prev = config_ptr[-1] >> 16;
        config_ptr += (len_prev / 4);
    }

    if (! config_ptr[0])
    {
        return NULL;
    }

    *id = config_ptr[0] & 0xffff;
    *len = config_ptr[0] >> 16;
    return &config_ptr[1];
}


/**
 * @}
 */
