/**
 ****************************************************************************************
 *
 * @file la.c
 *
 * @brief Embedded logic analyzer module.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup LA
 * @{
 ****************************************************************************************
 */
/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include <string.h>

#include "rwnx_config.h"
#include "dbg_assert.h"
#include "boot.h"

#include "rxl_cntrl.h"
#include "txl_cntrl.h"

#include "reg_la.h"
#include "reg_mac_core.h"
#include "reg_sysctrl.h"
#include "dma.h"
#include "la.h"
#include "la_shared.h"
#include "la_mem.h"

#include "dbg.h"
#include "co_utils.h"


/*
 * FUNCTIONS DEFINITIONS
 ****************************************************************************************
 */
void la_get_conf(struct la_conf_tag *conf)
{
    int i = 0;

    conf->conf[i++] = la_sampling_mask0_get();
    conf->conf[i++] = la_sampling_mask1_get();
    conf->conf[i++] = la_sampling_mask2_get();
    conf->conf[i++] = la_trigger_mask0_get();
    conf->conf[i++] = la_trigger_mask1_get();
    conf->conf[i++] = la_trigger_mask2_get();
    conf->conf[i++] = la_trigger_value0_get();
    conf->conf[i++] = la_trigger_value1_get();
    conf->conf[i++] = la_trigger_value2_get();
    conf->conf[i++] = la_version_get();

    conf->trace_len = (la_writeaddr_getf() + 1) * sizeof_b(struct la_mem_format);
    conf->diag_conf = sysctrl_diag_conf1_get();
}

/**
 ****************************************************************************************
 * @brief trace fw trace timestamp and stop LA trace
 *
 * This function simply stops the capture if LA version is lower than 5.0.
 * Otherwise this function will first trace the value of MAC HW counter 2 (the one used
 * to timestamp fw trace) and then stop the LA.
 * To do so:
 * - All LA signals are masked (so that we're sure no hardware signals will interfere)
 * - Then diags configuration is updated to force software diags on both diag0/1 and
 *   diag4/5. We output software diags twice so that we can be sure that LA fully capture
 *   the counter value when both value are equal.
 * - The counter value is then output on software diags
 * - LA trace is stopped
 * - LA configuration is reset to its original state.
 *
 ****************************************************************************************
 */
static void la_dump_trace_timestamp_and_stop(void)
{
    uint32_t conf[2];
    uint32_t mask[4];

    if ((la_version_get() & LA_LA_VERSION_MASK) < 0x050000)
    {
        la_stop();
        return;
    }

    // Mask all LA signals
    mask[0] = la_sampling_mask0_get();
    la_sampling_mask0_set(0);
    mask[1] = la_sampling_mask1_get();
    la_sampling_mask1_set(0);
    mask[2] = la_sampling_mask2_get();
    la_sampling_mask2_set(0);
    mask[3] = la_sampling_mask3_get();
    la_sampling_mask3_set(0);

    // Force Software diags on diag0/1 and diag4/5 banks
    conf[0] = sysctrl_diag_conf0_get();
    sysctrl_diag_conf0_pack(0x1f, 1, 0x1e, 1);
    conf[1] = sysctrl_diag_conf2_get();
    sysctrl_diag_conf2_pack(0x1f, 1, 0x1e, 1);

    // Trace FW trace counter via SW diags
    nxmac_sw_clear_profiling_clear(0xFFFFFFFF);
    la_sampling_mask0_set(0xFFFFFFFF);
    la_sampling_mask2_set(0xFFFFFFFF);
    nxmac_sw_set_profiling_set(hal_machw_time());
    nxmac_sw_clear_profiling_clear(0xFFFFFFFF);

    // Stop LA
    la_stop();

    // restore LA config
    sysctrl_diag_conf0_set(conf[0]);
    sysctrl_diag_conf2_set(conf[1]);
    la_sampling_mask0_set(mask[0]);
    la_sampling_mask1_set(mask[1]);
    la_sampling_mask2_set(mask[2]);
    la_sampling_mask3_set(mask[3]);
}

void la_dump_trace(struct dbg_debug_dump_tag *dbg_dump)
{
    uint32_t dst = CPU2HW(&dbg_dump->la_mem[0]);
    struct dma_desc *dma_desc = (struct dma_desc *) &(la_shared_env.dma_desc);
    int remaining, curr_line;

    la_dump_trace_timestamp_and_stop();
    remaining = la_writeaddr_getf() + 1;
    curr_line = (la_firstsample_get() + 1) % LA_MEM_LINE_COUNT;

    while (remaining > 0)
    {
        int i;
        int line_count = LA_SHARED_LINE_COUNT;

        if (remaining <= line_count)
            line_count = remaining;

        for (i = 0; i < line_count; i++)
        {
            la_shared_env.la_buf[i] = la_mem_mac[curr_line];

            curr_line = (curr_line + 1) % LA_MEM_LINE_COUNT;
        }
        dma_desc->src = CPU2HW(la_shared_env.la_buf);

        // Only one DMA transfer will be needed to send the complete MSG.
        dma_desc->dest = dst;
        dma_desc->length = line_count * sizeof_b(struct la_mem_format);
        dma_desc->ctrl = (IPC_DMA_LLI_IRQ_EN |
                          (IPC_DMA_LLI_DBG_DUMP << IPC_DMA_LLI_IRQ_POS));

        // Push the DMA descriptor in the DMA engine
        dma_push(dma_desc, dma_desc, IPC_DMA_CHANNEL_CTRL_RX);

        // Poll for the DMA to be finished
        dma_lli_poll(IPC_DMA_LLI_DBG_DUMP + DMA_LLI_IRQ_LSB);

        // acknowledge the interrupt
        dma_int_ack_clear(CO_BIT(IPC_DMA_LLI_DBG_DUMP + DMA_LLI_IRQ_LSB));

        // Update the DMA transfer parameters
        remaining -= line_count;
        dst += dma_desc->length;
    }
}

void la_init(void)
{
    if ((la_version_get() & 0xFFFFFF) < 0x50000)
    {
        // Set the sampling mask
        // SW Traces
        la_sampling_mask0_set(0xFFFFFFFF);
        // HW Traces
        la_sampling_mask1_set(0xFFFFFFFF);
        // MACPHY Traces
        la_sampling_mask2_set(0x300300FF);
    }
    else
    {
        // Set the sampling mask
        // SW Traces
        la_sampling_mask0_set(0x0000CFFF);
        // HW Traces
        la_sampling_mask1_set(0xFFFFFFFF);
        // MACPHY Traces
        la_sampling_mask2_set(0xFFFFFFFF);
        // MACPHY Traces
        la_sampling_mask3_set(0xFFFFFFFF);
    }
    // Set trigger level
    la_trigger_point_set(0x100);

    // Start the logic analyzer
    la_start();
}

void la_start(void)
{
    // Start the logic analyzer
    la_cntrl_set(LA_START_BIT);
}

void la_stop(void)
{
    // Stop the logic analyzer
    la_cntrl_set(LA_STOP_BIT);
}

/// @}

