/**
 ****************************************************************************************
 *
 * @file led.c
 *
 * @brief Demo LEDs driver
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup LED
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "led.h"
#include "rwnx_config.h"

#if (NX_PLF_VER >= 30)
#include "reg_karst_if.h"
#include "mm_timer.h"
#include "hal_machw.h"

/// Led status
struct led_tag
{
    /// Led index
    uint32_t idx;
    /// Blink half period in us (set to 0 if not blinking)
    uint32_t period_us;
    /// Blink timer
    struct mm_timer_tag timer;
};

/// Leds context variable
struct led_tag led_env[LED_MAX];
/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Get led structure
 *
 * @param[in] led_idx  LED index
 * @return pointer to LED structure or NULL if invalid paramter or led not enabled
 ****************************************************************************************
 */
__INLINE struct led_tag* led_get(enum led_name led_idx)
{
    if (!karst_v7_gpio_en_getf() || (led_idx >= LED_MAX))
        return NULL;

    return &led_env[led_idx];
}

/**
 ****************************************************************************************
 * @brief Led blink timer callback
 *
 * Call for each half period.
 *
 * @param[in] env  Pointer to led structure
 ****************************************************************************************
 */
static void led_blink_cb(void *env)
{
    struct led_tag *led = env;
    uint32_t period_us = led->period_us;

    // reset period_us to 0 to avoid trying the clear timer
    led->period_us = 0;
    led_toggle(led->idx);
    led->period_us = period_us;
    mm_timer_set(&led->timer,  hal_machw_time() + period_us);
}

void led_enable()
{
    uint32_t i;

    memset(led_env, 0, sizeof(led_env));

    for (i = 0 ; i < LED_MAX; i++)
    {
        led_env[i].idx = i;
        led_env[i].timer.cb = led_blink_cb;
        led_env[i].timer.env = &led_env[i];
    }

    karst_v7_gpio_en_setf(1);
}

void led_disable()
{
    uint32_t i;

    for (i = 0 ; i < LED_MAX; i++)
    {
        if (led_env[i].period_us)
        {
            mm_timer_clear(&led_env[i].timer);
            led_env[i].period_us = 0;
        }
    }
    karst_v7_gpio_en_setf(0);
}

void led_set(enum led_name led_idx, int state)
{
    struct led_tag *led = led_get(led_idx);

    if (!led)
        return;

    // Stop blinking if any
    if (led->period_us)
    {
        mm_timer_clear(&led->timer);
        led->period_us = 0;
    }

    switch (led_idx)
    {
        case LED_RED:
            karst_v7_gpio5_setf(state);
            break;
        case LED_GREEN:
            karst_v7_gpio6_setf(state);
            break;
        default:
            break;
    }
}

void led_toggle(enum led_name led_idx)
{
    switch (led_idx)
    {
        case LED_RED:
            return led_set(led_idx, !karst_v7_gpio5_getf());
        case LED_GREEN:
            return led_set(led_idx, !karst_v7_gpio6_getf());
        default:
            return;
    }
}

void led_blink(enum led_name led_idx, int period_ms)
{
    struct led_tag *led = led_get(led_idx);

    if (!led)
        return;

    if (period_ms < 100)
        period_ms = 100;

    led_toggle(led_idx);
    led->period_us = (period_ms * 1000) >> 1;
    mm_timer_set(&led->timer,  hal_machw_time() + led->period_us);
}
#endif //(NX_PLF_VER >= 30)

/// @} BUTTON
