/**
 ****************************************************************************************
 *
 * @file led.h
 *
 * @brief Demo LEDs driver APIs
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

#ifndef _LED_H_
#define _LED_H_

/**
 ****************************************************************************************
 * @defgroup LED LED
 * @ingroup EXTINT
 * @brief Driver for LEDS connected to the reference platform used for demo
 * purposes. Two LEDS are available (red and green).
 * @{
 ****************************************************************************************
 */

/// List of available leds
enum led_name
{
    /// Red led
    LED_RED,
    /// Green led
    LED_GREEN,
    /// Maximum number of leds
    LED_MAX,
};

/*
 * FUNCTION DECLARATIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Enable the control of the LEDs
 *
 * This function activates the GPIO outputs controlling the LEDs. This function shall be
 * called prior to play with the red and green LED state
 ****************************************************************************************
 */
void led_enable();

/**
 ****************************************************************************************
 * @brief Disable the control of the LEDs
 ****************************************************************************************
 */
void led_disable();

/**
 ****************************************************************************************
 * @brief Set the state of a LED
 *
 * @param[in] led_idx  Led id
 * @param[in] state    Set to 1 to switch on LED, and to 0 to switch it off
 ****************************************************************************************
 */
void led_set(enum led_name led, int state);

/**
 ****************************************************************************************
 * @brief Toggle the state of a LED
 *
 * @param[in] led_idx  Led id
 ****************************************************************************************
 */
void led_toggle(enum led_name led);

/**
 ****************************************************************************************
 * @brief Set the state of the green LED
 *
 * Blink a LED indefinitely until @ref led_set or @ref led_toggle is called.
 * The led will be switched on for half the period and switched off for the rest.
 * This used software timer, so period lower limit is 100ms.
 *
 * @param[in] led_idx      Led id
 * @param[in] period_ms    Blink period in ms (minimum 100)
 ****************************************************************************************
 */
void led_blink(enum led_name led_idx, int period_ms);

/// @} LED

#endif // _LED_H_
