/**
 ****************************************************************************************
 *
 * @file phy_aetnensis.c
 *
 * Copyright (C) RivieraWaves 2016-2019
 *
 * When setting a channel, several procedures can be advantageously left-out depending on
 * whether the band, bw, frequency remain unchanged (calibrations, old modem clk mmc
 * toggles ..), e.g. for multi-channel. Although the changes are trivial they might get
 * in the way of properly measuring the RF behavior.
 ****************************************************************************************
 */
#include "rwnx_config.h"
#include "phy.h"
#include "dbg.h"

#include "hal_machw.h"
#include "reg_mac_core.h"

#include "reg_mdm_stat.h"
#include "reg_mdm_cfg.h"
#include "reg_mdmdsss_cfg.h"

#include "reg_riu.h"
#include "reg_rc.h"
#include "reg_rc_gt.h"
#include "reg_rc_nft.h"
#include "reg_macbypass.h"
#include "rd.h"

#if NX_MDM_VER != 20
#error "Needs to be compiled with modem version 20"
#endif

#define WIFI_24G
//#define WIFI_5G
#ifdef WIFI_24G
#define PHY_AETNENSIS_TXRXID     0
#else
#define PHY_AETNENSIS_TXRXID     1
#endif


/// Structure containing the parameters of the AETNENSIS PHY configuration
struct phy_aetnensis_cfg_tag
{
};

/// PHY driver context.
struct phy_env_tag
{
    /// Aetnensis configuration parameters
    struct phy_aetnensis_cfg_tag cfg;
    /// Currently configured channel
    struct mac_chan_op chan;
};

struct phy_env_tag phy_env;

#define REG_DFLT_SW_CTRL    ((1 << RC_PRESCALER_LSB) & (~RC_START_DONE_BIT))

#define __RIU_RADARFIFO_ADDR 0x60C04000


// Command header field definitions
/// Response Request offset
#define RRQ_OFT          15
/// Response Request bit
#define RRQ_BIT         (0x01 << RRQ_OFT)
/// Command ID offset
#define CMDID_OFT        4
/// Command ID mask
#define CMDID_MSK       (0x3F << CMDID_OFT)
/// Command argument count offset
#define CMD_ARGC_OFT     0
/// Command argument count mask
#define CMD_ARGC_MSK    (0x0F << ARGC_OFT)

// Response header field definitions
/// Error Flag offset
#define EF_OFT           15
/// Error Flag bit
#define EF_BIT          (0x01 << EF_OFT)
/// Task started status offset
#define TS_OFT           14
/// Task started status bit
#define TS_BIT          (0x01 << TS_OFT)
/// Response ID offset
#define RSPID_OFT        4
/// Response ID mask
#define RSPID_MSK       (0x3F << RSPID_OFT)
/// Response argument count offset
#define RSP_ARGC_OFT     0
/// Response argument count mask
#define RSP_ARGC_MSK    (0x0F << RSP_ARGC_OFT)

/// RF command IDs
enum
{
    /// Set the channel to tune the RF to
    TRX_SET_RF_CHANNEL_ID = 0x05,
    /// Get the channel the RF is tuned to
    TRX_GET_RF_CHANNEL_ID = 0x06,
    /// Set the channel bandwidth
    TRX_SET_CHANNEL_BW_ID = 0x07,
    /// Get the channel bandwidth
    TRX_GET_CHANNEL_BW_ID = 0x08,
};

/// Gain table - @todo Update the values
static const uint32_t gaintable[REG_RC_GT_COUNT] =
{
    0x00020100,   // RWNXAGCGT0
    0x01000201,   // RWNXAGCGT1
    0x02010002,   // RWNXAGCGT2
    0x00020100,   // RWNXAGCGT3
    0x00000201,   // RWNXAGCGT4
    0x00000000,   // RWNXAGCGT5
    0x00000000,   // RWNXAGCGT6
    0x00000000,   // RWNXAGCGT7
    0x00000000,   // RWNXAGCGT8
    0x00000000,   // RWNXAGCGT9
    0x00000000,   // RWNXAGCGT10
    0x00000000,   // RWNXAGCGT11
    0x00000000,   // RWNXAGCGT12
    0x00000000,   // RWNXAGCGT13
    0x00000000,   // RWNXAGCGT14
    0x00000000,   // RWNXAGCGT15
};

/// Noise factor table - @todo Update the values
static const uint32_t nftable[REG_RC_NFT_COUNT] =
{
    0x1F1F1F1F, // NFT_RWNXAGCNFT0_RESET
    0x1F1F1F1F, // NFT_RWNXAGCNFT1_RESET
    0x1C1D1E1F, // NFT_RWNXAGCNFT2_RESET
    0x1A1B1B1C, // NFT_RWNXAGCNFT3_RESET
    0x1718191A, // NFT_RWNXAGCNFT4_RESET
    0x14151616, // NFT_RWNXAGCNFT5_RESET
    0x0F111213, // NFT_RWNXAGCNFT6_RESET
    0x0B0C0C0D, // NFT_RWNXAGCNFT7_RESET
    0x09090A0B, // NFT_RWNXAGCNFT8_RESET
    0x07070808, // NFT_RWNXAGCNFT9_RESET
    0x04050506, // NFT_RWNXAGCNFT10_RESET
    0x04040404, // NFT_RWNXAGCNFT11_RESET
    0x04040404, // NFT_RWNXAGCNFT12_RESET
    0x04040404, // NFT_RWNXAGCNFT13_RESET
    0x04040404, // NFT_RWNXAGCNFT14_RESET
    0x04040404, // NFT_RWNXAGCNFT15_RESET
};

static void phy_set_table(const uint32_t val[], uint32_t reg_addr, int reg_cnt)
{
    int i;

    // Copy the table into the registers
    for (i = 0; i < reg_cnt; i++)
    {
        REG_PL_WR(reg_addr, val[i]);
        reg_addr += 4;
    }
}

static inline void mdelay(uint32_t ms)
{
    uint32_t e = nxmac_monotonic_counter_2_lo_get() + ms * 1000;

    do {
        volatile int n = 1 << 10; // relax
        while (n--)
            ;
    } while ((int32_t)(nxmac_monotonic_counter_2_lo_get() - e) < 0);
}

static inline void udelay(uint32_t us)
{
    uint32_t e = nxmac_monotonic_counter_2_lo_get() + us;

    do {
        volatile int n = 1 << 10; // relax
        while (n--)
            ;
    } while ((int32_t)(nxmac_monotonic_counter_2_lo_get() - e) < 0);
}

static void mdm_reset(void)
{
    mdm_swreset_set(0x1);
    mdelay(2);
    mdm_swreset_set(0x0);
    mdelay(2);
}

/*
 * Definitions of MODEM Configurations
 * MDMCONF0: CBW=20MHz, NSS=1, 11nac, BCC
 * MDMCONF1: CBW=20MHz, NSS=2, 11n, BCC
 * MDMCONF2: CBW=40MHz, NSS=1, 11n, BCC/LDPC
 * MDMCONF3: CBW=40MHz, NSS=1, 11nac, BCC/LDPC
 * MDMCONF4: CBW=80MHz, NSS=1, BCC
 * MDMCONF5: CBW=80MHz, NSS=2, BCC
 * MDMCONF6: CBW=80MHz, NSS=2, BCC/LDPC
 * MDMCONF7: CBW=80MHz, NSS=1, BCC/LDPC
 */
static int phy_mdm_conf(uint8_t chan_type)
{
    int ldpc = mdm_ldpcdec_getf();
    int mdmconf = 0;

    switch (chan_type)
    {
        case PHY_CHNL_BW_20:
            mdmconf = 0;
            break;
        case PHY_CHNL_BW_40:
            mdmconf = 2;
            break;
        case PHY_CHNL_BW_80:
            if (ldpc)
                mdmconf = 7;
            else
                mdmconf = 4;
            break;
        default:
            ASSERT_ERR(0);
            break;
    }

    return mdmconf;
}

static uint16_t rf_rsp_get(int l)
{
    int rsp_to = 20;
    uint16_t rsphdr = 0;

    do
    {
        int loops = 1 << 20;

        // Start the transaction
        rc_sw_rfcmd_ctrl_set(RC_START_DONE_BIT | RC_READNOTWRITE_BIT | RC_APICMD_BIT | 1);

        // Poll for the start_done bit going back to 0
        while (rc_start_done_getf() && --loops);
        if (!loops)
        {
            dbg(D_ERR D_PHY "%s:%d failed\n", __func__, l);
            ASSERT_ERR(0);
            return 0;
        }

        // Read the response header
        rsphdr = rc_header_or_addr_getf();
        if (rsphdr)
            break;

        // Wait a bit between each attempt
        udelay(50);
    } while (--rsp_to);

    // Ensure we got a response
    if (!rsp_to)
    {
        dbg(D_ERR D_PHY "%s:%d failed\n", __func__, l);
        ASSERT_ERR(0);
        return 0;
    }

    return (rsphdr);
}

static uint8_t rf_cmd_send(uint16_t rrq, uint16_t cmdid, uint16_t argc, int l)
{
    int loops = 1 << 20;
    uint16_t cmdhdr = rrq | (cmdid << CMDID_OFT) | (argc << CMD_ARGC_OFT);
    uint8_t rsp_argc = 0;

    // Start the transaction
    rc_sw_rfcmd_ctrl_set(RC_START_DONE_BIT | RC_APICMD_BIT |
                         (cmdhdr << RC_HEADER_OR_ADDR_LSB));

    // Poll for the start_done bit going back to 0
    while (rc_start_done_getf() && --loops);
    if (!loops)
    {
        dbg(D_ERR D_PHY "%s:%d - %04X failed\n", __func__, l, cmdid);
        ASSERT_ERR(0);
        return 0;
    }

    // If a response is needed, wait for it
    if (rrq)
    {
        uint16_t rsphdr = rf_rsp_get(l);
        uint16_t rspid = (rsphdr & RSPID_MSK) >> RSPID_OFT;

        // Check if the response is valid
        if ((rsphdr & EF_BIT) || (rspid != cmdid))
        {
            dbg(D_ERR D_PHY "%s:%d - %04X Rsp Error - EF %d, RSPID %04X\n", __func__, l,
                cmdid, (rsphdr & EF_BIT) != 0, rspid);
            ASSERT_ERR(0);
            return 0;
        }

        // Get the number of response arguments
        rsp_argc = (rsphdr & RSP_ARGC_MSK) >> RSP_ARGC_OFT;
    }

    return (rsp_argc);
}

#define rf_send_1_param_cmd(rrq,cmdid,arg) _rf_send_1_param_cmd(rrq,cmdid,arg,__LINE__)
static uint8_t _rf_send_1_param_cmd(uint16_t rrq, uint16_t cmdid, uint16_t arg, int l)
{
    // Set the parameter
    rc_sw_rfdata1_ctrl_set(arg);

    // Send the command and wait for the response if needed
    return(rf_cmd_send(rrq, cmdid, 1, l));
}

#define rf_send_2_param_cmd(rrq,cmdid,arg0,arg1) _rf_send_2_param_cmd(rrq,cmdid,arg0,arg1,__LINE__)
static uint8_t _rf_send_2_param_cmd(uint16_t rrq, uint16_t cmdid, uint16_t arg0, uint16_t arg1, int l)
{
    // Set the parameters
    rc_sw_rfdata1_ctrl_set((arg1 << RC_DATA2_LSB) | arg0);

    // Send the command and wait for the response if needed
    return(rf_cmd_send(rrq, cmdid, 2, l));
}

static void rf_set_channel(uint8_t band, uint16_t freq1, uint8_t chantype, uint8_t txrxid)
{
    uint16_t bw_mode;
    uint16_t channel = phy_freq_to_channel(band, freq1);

    if (chantype == PHY_CHNL_BW_20)
        bw_mode = 20;
    else if (chantype == PHY_CHNL_BW_40)
        bw_mode = 40;
    else
        bw_mode = 80;

    // Set the BW
    rf_send_1_param_cmd(RRQ_BIT, TRX_SET_CHANNEL_BW_ID, (txrxid << 12) | bw_mode);
    // Set the Channel
    rf_send_1_param_cmd(RRQ_BIT, TRX_SET_RF_CHANNEL_ID, (txrxid << 12) | channel);
}

static void phy_hw_set_channel(uint8_t band, uint16_t freq, uint16_t freq1, uint8_t chantype, uint8_t index)
{
    unsigned int pssel;
    int mdmconf;

    dbg(D_INF "%s: band=%d freq=%d freq1=%d chantype=%d sx=%d\n",__func__,band,freq,freq1,chantype,index);

    rc_rf_sel_2g4_setf(band == PHY_BAND_2G4);
    mdm_rxdsssen_setf(band == PHY_BAND_2G4);
    riu_ofdmonly_setf(band == PHY_BAND_5G);
    nxmac_abgn_mode_setf(band == PHY_BAND_2G4?MODE_802_11N_2_4:MODE_802_11AC_5);

    mdmconf = phy_mdm_conf(chantype);
    mdm_mdmconf_setf(mdmconf);
    mdm_reset();

    // Set LDPC table selection
    if (mdmconf == 7)
        mdm_ldpcdectablesel_setf(0);
    else
        mdm_ldpcdectablesel_setf(1);

    /**
     * PSSEL - use pssel of RW-WSDM-RIU-HW-REG.xls
     */
    if (chantype == PHY_CHNL_BW_40)
    {
        // pssel = 1 or 2
        pssel = freq < freq1 ? 1 : 2;
    }
    else if (chantype == PHY_CHNL_BW_80)
    {
        // pssel = 0, 1, 2 or 3
        int _offs = freq1 - freq;
        if (_offs > 0)
            pssel = _offs > 10 ? 0 : 1;
        else
            pssel = -_offs > 10 ? 3 : 2;
    }
    else
    {
        pssel = 0;
    }

    riu_psselect_setf(pssel);

    /* workaround periodic (6-12us) peaks caused by the DINI
     * switching power supply
     * only observed in 2.4G
     */
    if(band == PHY_BAND_5G)
    {
      riu_ofdmonly_setf(1);   // AGC detection OFDM only
      #if (NX_RADAR_DETECT)
      riu_radardeten_setf(1); // Enable radar detection
      mdm_radartimclkforce_setf(1); // Enable radar timer clock
      riu_irqmacradardeten_setf(1); // Enable radar detection interrupt
      #endif
      riu_rwnxagcevt3_set(0);     /* write 0 to disable DSSS detection correlator */
     }
    else
    {
      riu_ofdmonly_setf(0);   // AGC detection OFDM and DSSS
      #if (NX_RADAR_DETECT)
      riu_radardeten_setf(0); // Disable radar detection
      mdm_radartimclkforce_setf(0); // Disable radar timer clock
      riu_irqmacradardeten_setf(0); // disable radar detection interrupt
      #endif
      riu_rwnxagcevt3_set(RIU_RWNXAGCEVT3_RESET);/* write default to enable DSSS detection correlator */
    }

    //  3us for TX RAMPUP <=> TXRFRAMPUP = 360
    if (chantype == PHY_CHNL_BW_20)
    {
        mdm_txstartdelay_setf(180);
        mdm_txctrl1_pack(0, 0, 28, 19);
        // TBE for 60MHz
        mdm_tbe_count_adjust_20_setf(0);
        mdm_txctrl3_pack(720, 1080);
        mdm_dcestimctrl_pack(0, 0, 0, 15, 15);
        // For FPGA, divide value by 2 due to timing constraints
        mdm_waithtstf_setf(7);

        mdm_tddchtstfmargin_setf(6);

        // No ACI margin in BW=20MHz due to latency on HTSIG decoding
        riu_rwnxagcaci20marg0_set(0);
        riu_rwnxagcaci20marg1_set(0);
        riu_rwnxagcaci20marg2_set(0);
    }
    else
    {
        mdm_txstartdelay_setf(360);
        // TBE for 120MHz
        mdm_tbe_count_adjust_20_setf(2);
        mdm_txctrl3_pack(1440, 2160);

        if (chantype == PHY_CHNL_BW_40)
        {
            mdm_txctrl1_pack(0, 39, 82, 39);
            mdm_rxtdctrl0_pack(18, 64, 252, 13);
            mdm_dcestimctrl_pack(0, 0, 8, 30, 31);
            // For FPGA, divide value by 2 due to timing constraints
            mdm_waithtstf_setf(15);

            mdm_tddchtstfmargin_setf(6);
        }
        else
        {
            mdm_txctrl1_pack(22, 60, 105, 48);
            mdm_rxtdctrl0_pack(18, 64, 247, 23);
            mdm_dcestimctrl_pack(0, 0, 38, 43, 63);
            // For FPGA, divide value by 2 due to timing constraints
            mdm_waithtstf_setf(31);

            mdm_tddchtstfmargin_setf(14);
        }

        // Set back default ACI margin
        riu_rwnxagcaci20marg0_set(RIU_RWNXAGCACI20MARG0_RESET);
        riu_rwnxagcaci20marg1_set(RIU_RWNXAGCACI20MARG1_RESET);
        riu_rwnxagcaci20marg2_set(RIU_RWNXAGCACI20MARG2_RESET);
    }

    // Configure maximum BW
    mdm_txcbwmax_setf(chantype);
    nxmac_max_supported_bw_setf(chantype);

    rf_set_channel(band, freq1, chantype, PHY_AETNENSIS_TXRXID);
}

static void phy_hw_init(const struct phy_aetnensis_cfg_tag *cfg)
{
    /***************************************************************************
     * RF INIT
     **************************************************************************/
    #ifdef WIFI_24G
        //ret = rf_aetnensis_init_lb();
    #elif defined(WIFI_5G)
        //ret = rf_aetnensis_init_hb();
    #else
    #error("must specify the carrier band, LB or HB!");
    #endif

    /***************************************************************************
     * MODEM INIT
     **************************************************************************/
    // Check if the MDM version is what we expect
    ASSERT_ERR(mdm_version_getf() == 0);

    mdm_mdmconf_set(phy_mdm_conf(BW_20MHZ));
    mdm_reset();

    mdm_rxmode_set(MDM_RXSTBCEN_BIT | MDM_RXGFEN_BIT  |
                   MDM_RXMMEN_BIT   | MDM_RXDSSSEN_BIT);
    mdm_rxnssmax_setf(mdm_nss_getf() - 1);
    mdm_rxndpnstsmax_setf(mdm_nsts_getf() - 1);
    mdm_rxldpcen_setf(mdm_ldpcdec_getf());
    mdm_rxvhten_setf(phy_vht_supported());
    mdm_rxmumimoen_setf(mdm_mumimorx_getf());

    // Set DSSS precomp
    mdm_precomp_setf(45);

    // Allow GF/SGI/STBC (bit14 reset) - TEMPORARY!!!!
    mdm_rxframeviolationmask_setf(0xFFFFBFFF);

    mdm_txmode_set(MDM_TXSTBCEN_BIT | MDM_TXGFEN_BIT  |
                   MDM_TXMMEN_BIT   | MDM_TXDSSSEN_BIT);
    mdm_txnssmax_setf(mdm_nss_getf());
    mdm_ntxmax_setf(mdm_ntx_getf());
    mdm_txcbwmax_setf(mdm_chbw_getf());
    mdm_txldpcen_setf(mdm_ldpcenc_getf());
    mdm_txvhten_setf(phy_vht_supported());
    mdm_txmumimoen_setf(mdm_mumimotx_getf());

    // AGC reset mode - don't turn off RF if rxreq de-asserted for few cycles after a RXERR
    mdm_rxtdctrl1_set(mdm_rxtdctrl1_get() | 0x01);

    // Enable automatic smoothing filter selection from SNR, then disable force
    mdm_cfgsmoothforce_setf(0);

    // Limit NDBPSMAX to 1x1 80 MCS9 LGI(390.0Mb/s) / SGI (433.3Mb/s)
    mdm_rxctrl1_set(0x06180618);

    // Adjust FFT short GI 80MHz window
    mdm_td_adjust_80_short_gi_setf(3);

    //PHYDSSSCONFIG_set_mdmbcntl2(32'h002d1205);
    mdm_mdmbcntl2_set(0x002d1205);

    // Enable RC clock
    mdm_rcclkforce_setf(1);

    /***************************************************************************
     * RIU INIT
     **************************************************************************/
    // limit RIU to 1 active antenna
    riu_activeant_setf(1);
    /* limit AGC with a single antenna (path0) */
    riu_combpathsel_setf(1);

    // Tx Digital gain
    riu_rwnxfectrl0_set(0x001a1a1a);

    /// @todo Check if we need to do that
    /*riu_rwnxagcevt2_set(0);*/ /* write 0 to disable disappearence detection */
    riu_crossupthrqdbm_setf(0x200); /* write 0x200 to disable AGC cross-up */
    riu_crossdnthrqdbm_setf(0x200); /* write 0x200 to disable AGC cross-down */

    riu_rwnxagcccatimeout_set(8000000); // 100ms
    riu_irqmacccatimeouten_setf(1);

    /***************************************************************************
     * RC INIT
     **************************************************************************/
    /// RC delays - @todo update values
    rc_hw_rx2tx_delay_set(0x00C800A8);
    rc_hw_rxidle2rx_delay_set(RC_HW_RXIDLE2RX_DELAY_RESET);
    rc_extpa_seq_delay_set(RC_EXTPA_SEQ_DELAY_RESET);
    rc_hw_rx_agcgainref_set(0x00450007);


    #ifdef WIFI_24G
    rc_static_cfg_set(1);
    //`RCAETNENSIS_setf_APICMDADDR LB (16'h1440);
    rc_sw_rfapicmd_addr_set(0x1440);
    #else
    rc_static_cfg_set(0);
    //`RCAETNENSIS_setf_APICMDADDR HB (16'h1400);
    rc_sw_rfapicmd_addr_set(0x1400);
   #endif

    // Set gain table
    phy_set_table(gaintable, GT_RWNXAGCGT0_ADDR, REG_RC_GT_COUNT);
    // Set noise factor table
    phy_set_table(nftable, NFT_RWNXAGCNFT0_ADDR, REG_RC_NFT_COUNT);

    /***************************************************************************
     * DEFAULT CHANNEL
     **************************************************************************/
    phy_hw_set_channel(PHY_BAND_2G4, 2162, 2152, PHY_CHNL_BW_40, 0);

    /***************************************************************************
     * MAC BYPASS INIT
     **************************************************************************/
    macbyp_trigger_set(0x00000012);
    macbyp_ctrl_set(0x00000100);
}

void phy_init(const struct phy_cfg_tag *config)
{
    const struct phy_aetnensis_cfg_tag *cfg = (const struct phy_aetnensis_cfg_tag *)&config->parameters;

    phy_hw_init(cfg);

    phy_env.cfg               = *cfg;
    phy_env.chan.band         = PHY_BAND_5G;
    phy_env.chan.type         = PHY_CHNL_BW_OTHER;
    phy_env.chan.prim20_freq  =
    phy_env.chan.center1_freq =
    phy_env.chan.center2_freq = PHY_UNUSED;
}

void phy_mdm_isr(void)
{
    dbg(D_ERR D_PHY "%s: TODO\n", __func__);
}

void phy_rc_isr(void)
{
    uint32_t irq_status = riu_rwnxmacintstatmasked_get();

    riu_rwnxmacintack_clear(irq_status);

    #if NX_RADAR_DETECT
    if (irq_status & RIU_IRQMACRADARDETMASKED_BIT)
    {
        PROF_RADAR_IRQ_SET();
        rd_event_ind(PHY_PRIM);
        PROF_RADAR_IRQ_CLR();
    }
    #endif

    if (irq_status & RIU_IRQMACCCATIMEOUTMASKED_BIT)
        mdm_reset();

    ASSERT_REC(!(irq_status & RIU_IRQMACCCATIMEOUTMASKED_BIT));
}

void phy_get_version(uint32_t *version_1, uint32_t *version_2)
{
    *version_1 = mdm_hdmconfig_get();
    // TODO Add version reading for other PHY elements than modem.
    *version_2 = 0;
}

void phy_set_channel(const struct mac_chan_op *chan, uint8_t index)
{
    if (index > 0)
    {
        dbg(D_ERR D_PHY "%s: radio %d does not exist\n", __func__, index);
        return;
    }

    if (phy_env.chan.band == chan->band &&
        phy_env.chan.type == chan->type &&
        phy_env.chan.flags == chan->flags &&
        phy_env.chan.prim20_freq == chan->prim20_freq &&
        phy_env.chan.center1_freq == chan->center1_freq &&
        phy_env.chan.center2_freq == chan->center2_freq)
    {
        dbg(D_INF D_PHY "%s: Setting same channel, do nothing\n", __func__);
        return;
    }

    dbg(D_INF D_PHY "%s: c:%d c1:%d bw:%d\n", __func__,
        chan->prim20_freq, chan->center1_freq, (1 << chan->type) * 20);

    phy_hw_set_channel(chan->band, chan->prim20_freq, chan->center1_freq, chan->type, index);
    phy_env.chan = *chan;
}

void phy_get_channel(struct phy_channel_info *info, uint8_t index)
{
    if (index > 0)
    {
        dbg(D_ERR D_PHY "%s: radio %d does not exist\n", __func__, index);
    }
    // Map the band, channel type, primary channel index on info1
    info->info1 = phy_env.chan.band | (phy_env.chan.type << 8) | (phy_env.chan.prim20_freq << 16);
    // Map center freq on info2
    info->info2 = phy_env.chan.center1_freq | (phy_env.chan.center2_freq << 16);
}

void phy_stop(void)
{
    if (nxmac_current_state_getf() != HW_IDLE)
        dbg(D_ERR "%s MAC state != IDLE\n", __func__);
}

uint32_t phy_get_channel_switch_dur(void)
{
    return 4500;
}

#if NX_RADAR_DETECT
bool phy_has_radar_pulse(int rd_idx)
{

    ASSERT_ERR(rd_idx == PHY_PRIM);

    return (riu_radfifoempty_getf() == 0);
}

bool phy_get_radar_pulse(int rd_idx, struct phy_radar_pulse *pulse)
{
    ASSERT_ERR(rd_idx == PHY_PRIM);

    // Check if FIFO is empty
    if (riu_radfifoempty_getf())
        return (false);

    pulse->pulse = REG_PL_RD(__RIU_RADARFIFO_ADDR);

    return (true);
}
#endif

bool phy_vht_supported(void)
{
    #if NX_MDM_VER > 20
    return ((mdm_vht_getf() != 0) || (mdm_chbw_getf() > PHY_CHNL_BW_40));
    #else
    return true;
    #endif
}

bool phy_he_supported(void)
{
    return false;
}

bool phy_uf_supported(void)
{
    return false;
}

void phy_uf_enable(bool enable)
{
}

bool phy_ldpc_tx_supported(void)
{
    return true;
}

bool phy_ldpc_rx_supported(void)
{
    return true;
}

bool phy_bfmee_supported(void)
{
    return (mdm_bfmee_getf() != 0);
}

bool phy_bfmer_supported(void)
{
    return (mdm_bfmer_getf() != 0);
}

bool phy_mu_mimo_rx_supported(void)
{
    return (mdm_mumimorx_getf() != 0);
}

bool phy_mu_mimo_tx_supported(void)
{
    return (mdm_mumimotx_getf() != 0);
}

#if RW_MUMIMO_RX_EN
void phy_set_group_id_info(uint32_t membership_addr, uint32_t userpos_addr)
{
    int i;

    // Set membership status
    for(i=0; i<MDM_MUMIMO_GROUPID_TAB_COUNT; i++)
    {
        mdm_mumimo_groupid_tab_set(i, co_read32p(membership_addr + 4 * i));
    }

    // Set user position
    for(i=0; i<MDM_MUMIMO_USERPOSITION_TAB_COUNT; i++)
    {
        mdm_mumimo_userposition_tab_set(i, co_read32p(userpos_addr + 4 * i));
    }
}
#endif

void phy_set_aid(uint16_t aid)
{
}

uint8_t phy_get_bw(void)
{
    return (mdm_chbw_getf());
}

uint8_t phy_get_nss(void)
{
    return (mdm_nss_getf() - 1);
}

uint8_t phy_get_ntx(void)
{
    return (mdm_ntx_getf() - 1);
}

uint8_t phy_get_nrx(void)
{
    return (mdm_nrx_getf() - 1);
}

#if RW_BFMER_EN
uint8_t phy_get_bfr_mem_size(void)
{
    return (mdm_bfmer_mem_size_getf());
}
#endif

void phy_get_rf_gain_idx(int8_t *power, uint8_t *idx)
{
    /* TO BE COMPLETED. For now set idx to power for debug */
    *idx = (uint8_t)*power;
}

void phy_get_rf_gain_capab(int8_t *max, int8_t *min)
{
    /* TO BE UPDATED with correct value */
    *max = 25; // dBm
    *min = 0;  // dBm
}

#if NX_DEBUG_DUMP
void phy_get_diag_state(struct dbg_debug_info_tag *dbg_info)
{
}
#endif

uint8_t phy_switch_antenna_paths(void)
{
    return 0;
}
