/**
 ****************************************************************************************
 *
 * @file phy_custom_rf.c
 *
 * Copyright (C) RivieraWaves 2014-2019
 *
 * When setting a channel, several procedures can be advantageously left-out depending on
 * whether the band, bw, frequency remain unchanged (calibrations, old modem clk mmc
 * toggles ..), e.g. for multi-channel. Although the changes are trivial they might get
 * in the way of properly measuring the RF behavior.
 ****************************************************************************************
 */
#include "rwnx_config.h"
#include "phy.h"
#include "dbg.h"

#include "hal_machw.h"
#include "reg_mac_core.h"

#if NX_MDM_VER < 30
#include "reg_mdm_stat.h"
#include "reg_mdmdsss_cfg.h"
#endif
#include "reg_mdm_cfg.h"

#include "reg_riu.h"
#include "reg_rc.h"
#include "reg_macbypass.h"

#include "crm.h"

#include "rd.h"
#include "uf.h"

#if NX_UF_EN
#define RXV_REG_MIN_VERS    2
#endif

/// Structure containing the parameters of the Custom RF configuration, coming from host
/// or any other mean
struct phy_custom_cfg_tag
{
};

/// PHY driver context structure.
struct phy_env_tag
{
    /// Configuration parameters
    struct phy_custom_cfg_tag cfg;
    /// Currently configured channel
    struct mac_chan_op chan;
};

/// PHY driver context
struct phy_env_tag phy_env;

/// Radar fifo address
#define __RIU_RADARFIFO_ADDR 0x60C04000

/**
 ****************************************************************************************
 * @brief Configure new channel at RF level
 *
 * @param[in] band      Channel band
 * @param[in] freq1     Channel center frequency
 * @param[in] chantype  Channel bandwidth
 ****************************************************************************************
 */
static void custom_rf_set_channel(uint8_t band, uint16_t freq1, uint8_t chantype)
{
    // Add the RF/ADC/DAC/etc. configuration code for channel switch
}

/**
 ****************************************************************************************
 * @brief Configure Primary/Secondary at RIU level
 *
 * @param[in] freq      Channel frequency of primary
 * @param[in] freq1     Channel center frequency
 * @param[in] chantype  Channel bandwidth
 ****************************************************************************************
 */
static void riu_pssel_set(uint16_t freq, uint16_t freq1, uint8_t chantype)
{
    unsigned int pssel;

    /**
     * PSSEL - use pssel of RW-WSDM-RIU-HW-REG.xls
     */
    if (chantype == PHY_CHNL_BW_40)
    {
        // pssel = 1 or 2
        pssel = freq < freq1 ? 1 : 2;
    }
    else if (chantype == PHY_CHNL_BW_80)
    {
        // pssel = 0, 1, 2 or 3
        int _offs = freq1 - freq;
        if (_offs > 0)
            pssel = _offs > 10 ? 0 : 1;
        else
            pssel = -_offs > 10 ? 3 : 2;
    }
    else
    {
        pssel = 0;
    }

    riu_psselect_setf(pssel);
}

/**
 ****************************************************************************************
 * @brief Configure Primary at Modem level
 *
 * @param[in] freq      Channel frequency of primary
 * @param[in] freq1     Channel center frequency
 * @param[in] chantype  Channel bandwidth
 ****************************************************************************************
 */
static void mdm_primary_set(uint16_t freq, uint16_t freq1, uint8_t chantype)
{
    #if NX_MDM_VER >= 30
    unsigned int primary;

    if (chantype == PHY_CHNL_BW_40)
    {
        primary = freq < freq1 ? 0 : 1;
    }
    else if (chantype == PHY_CHNL_BW_80)
    {
        primary = 0; // TBD
    }
    else
    {
        primary = 0;
    }

    mdm_primaryind_set(primary);
    #endif
}

/**
 ****************************************************************************************
 * @brief Configure new channel at modem level
 *
 * @param[in] band      Channel band
 * @param[in] freq      Channel frequency of primary
 * @param[in] freq1     Channel center frequency
 * @param[in] chantype  Channel bandwidth
 * @param[in] flags     Channel flags
 ****************************************************************************************
 */
static void mdm_set_channel(uint8_t band, uint16_t freq, uint16_t freq1, uint8_t chantype,
                            uint8_t flags)
{
    // Reset the MDM/RIU before configuration
    crm_mdm_reset();

    /*
     *************************************************************************************
     * Band-dependent (2.4G, 5G) settings
     *************************************************************************************
     */
    // By default we consider we won't be on a DFS channel
    #if NX_RADAR_DETECT
    riu_radardeten_setf(0); // Disable radar detection
    crm_radartimclkforce_setf(0); // Disable radar timer clock
    riu_irqmacradardeten_setf(0); // disable radar detection interrupt
    #endif
    #if NX_MAC_HE
    nxmac_disable_tbru_26_resp_setf(0); // Allow RU26
    #endif
    if(band == PHY_BAND_5G)
    {
        rc_rf_sel_2g4_setf(0);
        nxmac_abgn_mode_setf(MODE_802_11AC_5);
        mdm_rxdsssen_setf(0);
        riu_ofdmonly_setf(1);   // AGC detection OFDM only
        #if NX_RADAR_DETECT || NX_MAC_HE
        if (flags & CHAN_RADAR)
        {
            #if NX_RADAR_DETECT
            riu_radardeten_setf(1); // Enable radar detection
            crm_radartimclkforce_setf(1); // Enable radar timer clock
            riu_irqmacradardeten_setf(1); // Enable radar detection interrupt
            #endif
            #if NX_MAC_HE
            nxmac_disable_tbru_26_resp_setf(1); // No RU26 when tuned to a DFS channel
            #endif
        }
        #endif
        riu_rwnxagcevt3_set(0);     /* write 0 to disable DSSS detection correlator */
    }
    else
    {
        rc_rf_sel_2g4_setf(1);
        nxmac_abgn_mode_setf(MODE_802_11N_2_4);
        mdm_rxdsssen_setf(1);
        riu_ofdmonly_setf(0);   // AGC detection OFDM and DSSS
        riu_rwnxagcevt3_set(RIU_RWNXAGCEVT3_RESET);/* write default to enable DSSS detection correlator */
    }

    /*
     *************************************************************************************
     * Frequency-dependent settings
     *************************************************************************************
     */
    #if NX_MDM_VER >= 30
    // For symbol clock error compensation in HE TB
    mdm_invcarrierfreq_setf((1 << 26) / freq1);
    #endif

    /*
     *************************************************************************************
     * Bandwidth-dependent settings
     *************************************************************************************
     */
    // Set PSSEL
    riu_pssel_set(freq, freq1, chantype);
    // Set Primary Channel in MDM
    mdm_primary_set(freq, freq1, chantype);
    // Configure maximum BW
    mdm_txcbwmax_setf(chantype);
    nxmac_max_supported_bw_setf(chantype);
    // Put default values
    mdm_fdoctrl0_set(MDM_FDOCTRL0_RESET);
    #if NX_MDM_VER < 30
    mdm_tbectrl2_set(0x00007F05);
    #else
    mdm_conf_bw_setf(chantype);
    mdm_rxcbwmax_setf(chantype);
    // To be uncommented when NX_MDM_VER=30 supports 80 MHz
    //mdm_tbectrl2_set(MDM_TBECTRL2_RESET);
    #endif
    #if NX_MDM_VER >= 22
    mdm_smoothforcectrl_set(0);
    #else
    mdm_smoothctrl_set(0x01880C06);
    #endif

    //  3us for TX RAMPUP <=> TXRFRAMPUP = 360
    if (chantype == PHY_CHNL_BW_20)
    {
        #if NX_MDM_VER < 30
        mdm_txstartdelay_setf(180);
        mdm_txctrl1_pack(0, 0, 28, 48);
        // TBE for 60MHz
        mdm_tbe_count_adjust_20_setf(0);
        mdm_txctrl3_pack(720, 1080);
        mdm_dcestimctrl_pack(0, 0, 0, 15, 15);
        // For FPGA, divide value by 2 due to timing constraints
        mdm_waithtstf_setf(7);
        #else
        mdm_txstartdelay_setf(384);
        mdm_txctrl1_pack(64, 96);
        // TBE for 120MHz
        mdm_tbe_count_adjust_20_setf(4);
        mdm_dcestimctrl_pack(0, 2, 10, 15);
        // For FPGA, divide value by 2 due to timing constraints
        mdm_waithtstf_setf(7);
        mdm_tdsyncoff20_setf(25);
        #endif

        #if NX_MDM_VER < 30
        mdm_tddchtstfmargin_setf(6);
        #else
        mdm_tddchtstfmargin_setf(1);
        #endif

        // No ACI margin in BW=20MHz due to latency on HTSIG decoding
        riu_rwnxagcaci20marg0_set(0);
        riu_rwnxagcaci20marg1_set(0);
        riu_rwnxagcaci20marg2_set(0);

        #if NX_MDM_VER >= 30
        // Increase DC convergence due to Karst RF performance
        riu_dccenteredholdtime50ns_setf(15);
        #endif
    }
    else
    {

        #if NX_MDM_VER < 30
        // TBE for 120MHz
        mdm_tbe_count_adjust_20_setf(2);
        mdm_txstartdelay_setf(360);
        mdm_txctrl3_pack(1440, 2160);
        #endif

        if (chantype == PHY_CHNL_BW_40)
        {
            #if NX_MDM_VER < 30
            mdm_txctrl1_pack(0, 39, 82, 96);
            mdm_rxtdctrl0_pack(18, 64, 252, 13);
            mdm_dcestimctrl_pack(0, 0, 8, 30, 31);
            // For FPGA, divide value by 2 due to timing constraints
            mdm_waithtstf_setf(15);
            #else
            mdm_txstartdelay_setf(384);
            mdm_txctrl1_pack(64, 96);
            mdm_dcestimctrl_pack(0, 3, 16, 31);
            // For FPGA, divide value by 2 due to timing constraints
            mdm_waithtstf_setf(15);
            mdm_tbe_count_adjust_20_setf(4);
            mdm_tdsyncoff20_setf(24);
            #endif

            #if NX_MDM_VER < 30
            mdm_tddchtstfmargin_setf(6);
            #else
            mdm_tddchtstfmargin_setf(1);
            #endif
        }
        else // chantype == PHY_CHNL_BW_80
        {
            #if NX_MDM_VER < 30
            mdm_txctrl1_pack(22, 60, 105, 120);
            mdm_rxtdctrl0_pack(18, 64, 247, 23);
            mdm_dcestimctrl_pack(0, 0, 38, 43, 63);
            // For FPGA, divide value by 2 due to timing constraints
            mdm_waithtstf_setf(31);
            #endif

            mdm_tddchtstfmargin_setf(14);

            #if NX_MDM_VER >= 22 && NX_MDM_VER < 30
            mdm_cfgvhtsts2smoothforce_setf(1);
            mdm_cfgvhtsts2smooth_setf(2);
            #else
            mdm_smoothctrl_set(0x018E0C06);
            #endif
            mdm_tbectrl2_set(0x00007F03);
        }

        // Set back default ACI margin
        riu_rwnxagcaci20marg0_set(RIU_RWNXAGCACI20MARG0_RESET);
        riu_rwnxagcaci20marg1_set(RIU_RWNXAGCACI20MARG1_RESET);
        riu_rwnxagcaci20marg2_set(RIU_RWNXAGCACI20MARG2_RESET);

        #if NX_MDM_VER >= 30
        // Set back default DC parameters for BW > 20MHz
        riu_dccenteredholdtime50ns_setf(RIU_DCCENTEREDHOLDTIME50NS_RST);
        #endif

    }

    #if NX_MDM_VER >= 21
    /* Reset RX IQ compensation if available */
    if (riu_iqcomp_getf())
    {
        riu_iqestiterclr_setf(1);
    }
    #endif /* NX_MDM_VER >= 21 */
}

/**
 ****************************************************************************************
 * @brief Configure a new channel
 *
 * @param[in] band      Channel band
 * @param[in] freq      Channel primary frequency
 * @param[in] freq1     Channel center frequency
 * @param[in] chantype  Channel bandwidth
 * @param[in] flags     Channel flags
 * @param[in] index     RIU index to configure (ignored if only one RIU)
 ****************************************************************************************
 */
static void phy_hw_set_channel(uint8_t band, uint16_t freq, uint16_t freq1,
                               uint8_t chantype, uint8_t flags, uint8_t index)
{
    dbg(D_INF D_PHY "%s: band=%d freq=%d freq1=%d chantype=%d sx=%d\n",__func__,band,freq,freq1,chantype,index);

    /*
     *************************************************************************************
     * Clock configuration
     *************************************************************************************
     */
    crm_clk_set(chantype);

    /*
     *************************************************************************************
     * MODEM/RIU configuration
     *************************************************************************************
     */
    mdm_set_channel(band, freq, freq1, chantype, flags);

    /*
     *************************************************************************************
     * RF/RF board configuration
     *************************************************************************************
     */
    custom_rf_set_channel(band, freq1, chantype);
}

/**
 ****************************************************************************************
 * @brief Initialize RF layer
 *
 * @param[in] cfg Custom RF configuration
 ****************************************************************************************
 */
static void custom_rf_init(const struct phy_custom_cfg_tag *cfg)
{
    // Add the initial configuration of the RF/ADC/DAC/etc.
}

/**
 ****************************************************************************************
 * @brief Initialize Modem
 *
 * @param[in] cfg Custom RF configuration
 ****************************************************************************************
 */
static void mdm_init(const struct phy_custom_cfg_tag *cfg)
{
    // Check if we are compiled for this version of the PHY
    ASSERT_ERR((mdm_major_version_getf() + 2) * 10 + mdm_minor_version_getf()
                                                                 == NX_MDM_VER);

    // Reset the MDM/RIU before configuration
    crm_mdm_reset();

    /*
     *************************************************************************************
     * MODEM configuration
     *************************************************************************************
     */
    // Supported features
    mdm_rxmode_set(MDM_RXMMEN_BIT | MDM_RXDSSSEN_BIT);
    mdm_rxnssmax_setf(mdm_nss_getf() - 1);
    mdm_rxndpnstsmax_setf(mdm_nsts_getf() - 1);
    mdm_rxldpcen_setf(mdm_ldpcdec_getf());
    mdm_rxvhten_setf(phy_vht_supported());
    mdm_rxstbcen_setf(1);
    #if NX_MDM_VER < 30
    mdm_rxgfen_setf(1);
    mdm_rxmumimoen_setf(mdm_mumimorx_getf());
    mdm_rxmumimoapeplenen_setf(mdm_mumimorx_getf());
    #else
    mdm_rxdcmen_setf(mdm_he_getf());
    mdm_rxheen_setf(mdm_he_getf());
    mdm_rxvhtmumimoen_setf(mdm_vht_getf() & mdm_mumimorx_getf());
    mdm_rxhemumimoen_setf(mdm_he_getf() & mdm_mumimorx_getf());
    #endif

    // Set DSSS precomp
    mdm_precomp_setf(45);

    #if NX_MDM_VER == 20
    #if RW_NX_LDPC_DEC
    // Set LDPC table selection for FPGA limitation
    mdm_ldpcdectablesel_setf(2);
    #endif
    #endif

    // Allow GF/SGI/STBC (bit14 reset) - TEMPORARY!!!!
    mdm_rxframeviolationmask_setf(0xFFFFBFFF);

    mdm_txmode_set(MDM_TXSTBCEN_BIT | MDM_TXGFEN_BIT  |
                   MDM_TXMMEN_BIT   | MDM_TXDSSSEN_BIT);
    mdm_txnssmax_setf(mdm_nss_getf() - 1);
    mdm_ntxmax_setf(mdm_ntx_getf() - 1);
    mdm_txcbwmax_setf(mdm_chbw_getf());
    mdm_txldpcen_setf(mdm_ldpcenc_getf());
    mdm_txvhten_setf(phy_vht_supported());
    #if NX_MDM_VER >= 30
    mdm_txheen_setf(phy_he_supported());
    mdm_tdfocpeslopeen_setf(1);
    #endif
    mdm_txmumimoen_setf(mdm_mumimotx_getf());

    #if NX_MDM_VER < 30
    /* AGC reset mode
     don't turn off RF if rxreq de-asserted for few cycles after a RXERR */
    mdm_rxtdctrl1_set(mdm_rxtdctrl1_get()|1);
    #endif

    /* Enable automatic smoothing filter selection from SNR, then disable force */
    #if NX_MDM_VER < 22
    mdm_cfgsmoothforce_setf(0);
    #endif

    #if NX_MDM_VER < 30
    if (mdm_nss_getf() == 1)
    {
        /* limit NDBPSMAX to 1x1 80 MCS7 LGI(292.5Mb/s) / SGI (325.0Mb/s) */
        mdm_rxctrl1_set(0x04920492);
    }
    else
    {
        #if defined(CFG_VIRTEX6)
        /* limit NDBPSMAX to 2x2 80 MCS4 LGI(351Mb/s) / SGI (390.0Mb/s) */
        mdm_rxctrl1_set(0x057C057C);
        #elif defined(CFG_VIRTEX7)
        /* No limitation on VIRTEX7 platform */
        mdm_rxctrl1_set(0x0C300C30);
        #endif
    }
    #else
    mdm_txtdsfoctrl_set(0x10000000);
    mdm_txtdcfoctrl_set(0x10000000);
    mdm_rxctrl5_set(0x00670780);
    mdm_tdfoctrl0_set(0x00340100);
    #endif

    /*
     *************************************************************************************
     * RIU configuration
     *************************************************************************************
     */
    /* Enable RC clock */
    crm_rcclkforce_setf(1);

    /* RC delays */
    rc_hw_tx_delay_set(0xa0);

    #if NX_MDM_VER >= 21
    /* Enable RX IQ compensation if available */
    if (riu_iqcomp_getf())
    {
        riu_rxiqphaseesten_setf(1);
        riu_rxiqgainesten_setf(1);
        riu_rxiqphasecompen_setf(1);
        riu_rxiqgaincompen_setf(1);
        riu_iqestiterclr_setf(1);
    }
    #endif /* NX_MDM_VER >= 21 */

    /* limit RIU to 1 or 2 antenna active depending on modem capabilities */
    if (mdm_nss_getf() == 2)
    {
        riu_activeant_setf(3);
        riu_combpathsel_setf(3);
    }
    else
    {
        riu_activeant_setf(1);
        /* limit AGC with a single antenna (path0) */
        riu_combpathsel_setf(1);
    }

    // Tx Digital gain
    #if NX_MDM_VER < 30
    riu_rwnxfectrl0_set(0x001A1A1A);
    if (mdm_ntx_getf() == 2)
        riu_rwnxfectrl1_set(0x001A1A1A);
    #else
    riu_rifsdeten_setf(0);
    riu_rwnxfectrl0_set(0x00343434);
    if (mdm_ntx_getf() == 2)
        riu_rwnxfectrl1_set(0x00343434);

    /* disable the riu dccentered compensation */
    riu_dccenteredtype_setf(0);
    #endif

    riu_crossupthrqdbm_setf(0x200); /* write 0x200 to disable AGC cross-up */
    riu_crossdnthrqdbm_setf(0x200); /* write 0x200 to disable AGC cross-down */

    riu_rwnxagcccatimeout_set(8000000); // 100ms
    riu_irqmacccatimeouten_setf(1);

    /* Enable HW antenna selection */
    riu_rxpathselfromreg_setf(0);

    /*
     *************************************************************************************
     * MACBYPASS configuration
     *************************************************************************************
     */
    #if NX_MDM_VER >= 30
    macbyp_clken_set(1); // enable clock
    #endif
    macbyp_trigger_set(0x00000012);
    macbyp_ctrl_set(0x00000100);

    #if NX_UF_EN
    if (macbyp_version_get() >= RXV_REG_MIN_VERS)
    {
        macbyp_clken_set(1); // enable clock
        macbyp_int3_gen_setf(0x07);// configure interrupt generation
        macbyp_mode_setf(1); //set mode RX
    }
    #endif
}

/**
 ****************************************************************************************
 * @brief Initialize PHY interface
 *
 * @param[in] cfg Custom RF configuration
 ****************************************************************************************
 */
static void phy_hw_init(const struct phy_custom_cfg_tag *cfg)
{
    /*
     *************************************************************************************
     * MODEM/RIU configuration
     *************************************************************************************
     */
    mdm_init(cfg);

    /*
     *************************************************************************************
     * RF/RF board configuration
     *************************************************************************************
     */
    custom_rf_init(cfg);

    /*
     *************************************************************************************
     * Set a default channel
     *************************************************************************************
     */
    phy_hw_set_channel(PHY_BAND_5G, 5180, 5180, PHY_CHNL_BW_20, 0, 0);
}

void phy_init(const struct phy_cfg_tag *config)
{
    const struct phy_custom_cfg_tag *cfg = (const struct phy_custom_cfg_tag *)&config->parameters;

    phy_hw_init(cfg);

    phy_env.cfg               = *cfg;
    phy_env.chan.band         = PHY_BAND_5G;
    phy_env.chan.type         = PHY_CHNL_BW_OTHER;
    phy_env.chan.prim20_freq  =
    phy_env.chan.center1_freq =
    phy_env.chan.center2_freq = PHY_UNUSED;
}

void phy_mdm_isr(void)
{
    #if NX_UF_EN
    if (macbyp_int3_state_getf())
    {
        //Indicate that an unsupported HT frame has been capture
        uf_event_ind();

        //interrupt 3 ack
        macbyp_int3_ack_clearf(1);
    }
    #endif
}

/**
 * FIXME: This is for debug purpose only and does not call rd_event_ind
 */
void phy_rc_isr(void)
{
    uint32_t irq_status = riu_rwnxmacintstatmasked_get();

    riu_rwnxmacintack_clear(irq_status);

    #if NX_RADAR_DETECT
    if (irq_status & RIU_IRQMACRADARDETMASKED_BIT)
    {
        PROF_RADAR_IRQ_SET();
        rd_event_ind(PHY_PRIM);
        PROF_RADAR_IRQ_CLR();
    }
    #endif

    if (irq_status & RIU_IRQMACCCATIMEOUTMASKED_BIT)
        crm_mdm_reset();

    ASSERT_REC(!(irq_status & RIU_IRQMACCCATIMEOUTMASKED_BIT));
}

void phy_get_version(uint32_t *version_1, uint32_t *version_2)
{
    *version_1 = mdm_hdmconfig_get();
    // TODO Add version reading for other PHY elements than modem.
    *version_2 = mdm_hdmversion_get();
}

void phy_set_channel(const struct mac_chan_op *chan, uint8_t index)
{
    if (index > 0)
    {
        dbg(D_ERR D_PHY "%s: radio %d does not exist\n", __func__, index);
        return;
    }

    if (phy_env.chan.band == chan->band &&
        phy_env.chan.type == chan->type &&
        phy_env.chan.flags == chan->flags &&
        phy_env.chan.prim20_freq == chan->prim20_freq &&
        phy_env.chan.center1_freq == chan->center1_freq &&
        phy_env.chan.center2_freq == chan->center2_freq)
    {
        dbg(D_INF D_PHY "%s: Setting same channel, do nothing\n", __func__);
        return;
    }

    dbg(D_INF D_PHY "%s: c:%d c1:%d bw:%d\n", __func__,
        chan->prim20_freq, chan->center1_freq, (1 << chan->type) * 20);

    phy_hw_set_channel(chan->band, chan->prim20_freq, chan->center1_freq, chan->type,
                       chan->flags, index);
    phy_env.chan = *chan;
}

void phy_get_channel(struct phy_channel_info *info, uint8_t index)
{
    if (index > 0)
    {
        dbg(D_ERR D_PHY "%s: radio %d does not exist\n", __func__, index);
    }
    // Map the band, channel type, primary channel index on info1
    info->info1 = phy_env.chan.band | (phy_env.chan.type << 8) | (phy_env.chan.prim20_freq << 16);
    // Map center freq on info2
    info->info2 = phy_env.chan.center1_freq | (phy_env.chan.center2_freq << 16);
}

void phy_stop(void)
{
    if (nxmac_current_state_getf() != HW_IDLE)
        dbg(D_ERR "%s MAC state != IDLE\n", __func__);
}

uint32_t phy_get_channel_switch_dur(void)
{
    return 1000;
}

#if NX_RADAR_DETECT
bool phy_has_radar_pulse(int rd_idx)
{

    ASSERT_ERR(rd_idx == PHY_PRIM);

    return (riu_radfifoempty_getf() == 0);
}

bool phy_get_radar_pulse(int rd_idx, struct phy_radar_pulse *pulse)
{
    ASSERT_ERR(rd_idx == PHY_PRIM);

    // Check if FIFO is empty
    if (riu_radfifoempty_getf())
        return (false);

    pulse->pulse = REG_PL_RD(__RIU_RADARFIFO_ADDR);

    return (true);
}
#endif

bool phy_vht_supported(void)
{
    #if NX_MDM_VER > 20
    return ((mdm_vht_getf() != 0) || (mdm_chbw_getf() > PHY_CHNL_BW_40));
    #else
    return true;
    #endif
}

bool phy_he_supported(void)
{
    #if NX_MDM_VER < 30
    return false;
    #else
    return (mdm_he_getf() != 0);
    #endif
}

bool phy_uf_supported(void)
{
    #if NX_UF_EN
    return true ? macbyp_version_get() >= RXV_REG_MIN_VERS : false;
    #else
    return false;
    #endif
}

void phy_uf_enable(bool enable)
{
    #if NX_UF_EN
    // enable/disable macbypass interrupt line 3
    macbyp_interrupt3_en_setf(enable);
    #endif
}

bool phy_ldpc_tx_supported(void)
{
    return (mdm_ldpcenc_getf() != 0);
}

bool phy_ldpc_rx_supported(void)
{
    return (mdm_ldpcdec_getf() != 0);
}

bool phy_bfmee_supported(void)
{
    return (mdm_bfmee_getf() != 0);
}

bool phy_bfmer_supported(void)
{
    return (mdm_bfmer_getf() != 0);
}

bool phy_mu_mimo_rx_supported(void)
{
    return (mdm_mumimorx_getf() != 0);
}

bool phy_mu_mimo_tx_supported(void)
{
    return (mdm_mumimotx_getf() != 0);
}

#if RW_MUMIMO_RX_EN
void phy_set_group_id_info(uint32_t membership_addr, uint32_t userpos_addr)
{
    int i;

    // Set membership status
    for(i=0; i<MDM_MUMIMO_GROUPID_TAB_COUNT; i++)
    {
        mdm_mumimo_groupid_tab_set(i, co_read32p(membership_addr + 4 * i));
    }

    // Set user position
    for(i=0; i<MDM_MUMIMO_USERPOSITION_TAB_COUNT; i++)
    {
        mdm_mumimo_userposition_tab_set(i, co_read32p(userpos_addr + 4 * i));
    }
}
#endif

void phy_set_aid(uint16_t aid)
{
    #if NX_MDM_VER >= 30
    mdm_hestaid_setf(0, aid);
    mdm_hestaid_setf(1, 0);
    mdm_hestaid_setf(2, 2047);
    #endif
}

uint8_t phy_get_bw(void)
{
    return (mdm_chbw_getf());
}

uint8_t phy_get_nss(void)
{
    return (mdm_nss_getf() - 1);
}

uint8_t phy_get_ntx(void)
{
    return (mdm_ntx_getf() - 1);
}

uint8_t phy_get_nrx(void)
{
    return (mdm_nrx_getf() - 1);
}

#if RW_BFMER_EN
uint8_t phy_get_bfr_mem_size(void)
{
    return (mdm_bfmer_mem_size_getf());
}
#endif

void phy_get_rf_gain_idx(int8_t *power, uint8_t *idx)
{
    /* TO BE COMPLETED. For now set idx to power for debug */
    *idx = (uint8_t)*power;
}

void phy_get_rf_gain_capab(int8_t *max, int8_t *min)
{
    /* TO BE UPDATED with correct value */
    *max = 25; // dBm
    *min = 0;  // dBm
}

#if NX_DEBUG_DUMP
void phy_get_diag_state(struct dbg_debug_info_tag *dbg_info)
{
}
#endif

uint8_t phy_switch_antenna_paths(void)
{
    uint8_t value = 0;

    PROF_ANT_DIV_SWITCH_SET();
    PROF_ANT_DIV_SWITCH_CLR();

    return value;
}

