/**
 ****************************************************************************************
 *
 * @file phy_elma.c
 *
 * @brief c.f. KIWLANT65R2H44A_User_Guide.pdf v0.14
 *
 * Copyright (C) RivieraWaves 2014-2019
 *
 * When setting a channel, several procedures can be advantageously left-out depending on
 * whether the band, bw, frequency remain unchanged (calibrations, old modem clk mmc
 * toggles ..), e.g. for multi-channel. Although the changes are trivial they might get
 * in the way of properly measuring the RF behavior.
 ****************************************************************************************
 */
#include "rwnx_config.h"
#include "phy.h"
#include "dbg.h"

// for nxmac_current_state_getf and HW_IDLE
#include "hal_machw.h"
#include "reg_mac_core.h"

#include "reg_mdm_stat.h"
#include "reg_mdm_cfg.h"

#include "reg_riu.h"
#include "reg_rc.h"
#include "reg_fcu.h"
#include "reg_elma_if.h"

#include "rd.h"


#define MASK8(h,l) ((((uint8_t)1<<((h) - (l) + 1)) - 1) << (l))
#define SET8(v1,v2,h,l) (((uint8_t)(v1) & ~MASK8(h,l)) | (((uint8_t)(v2) << (l)) & MASK8(h,l)))

#define __FCU_RADARFIFO_ADDR    0x60C09800
#define __RIU_RADARFIFO_ADDR    0x60C04000

#define FCU_PRESENT_MASK        ((uint32_t)0xFF000000)
#define FCU_PRESENT_LSB         24
#define FCU_PRESENT() (((riu_rwnxversion_get()) & FCU_PRESENT_MASK) >> FCU_PRESENT_LSB == 5)

#define LOGENT_RIDX             0
#define RXRFMIXT_RIDX           1
#define RXLNAT_RIDX             2
#define TXRFMIXT_RIDX           3
#define TXPAT_RIDX              4

// One of the B0 boards we got had rev 0x82
#define REV_A0                  1

static int  rev;
static bool use_pll_doubler;

/// PHY driver context.
struct phy_env_tag
{
    /// Currently configured channel
    struct mac_chan_op chan;
};

static const struct {
    uint8_t r, h, l;
} lo_rf_params[] = {
    [LOGENT_RIDX]   = {0x40, 3, 0},
    [RXRFMIXT_RIDX] = {0x41, 7, 3},
    [RXLNAT_RIDX]   = {0x47, 7, 4},
    [TXRFMIXT_RIDX] = {0x45, 4, 0},
    [TXPAT_RIDX]    = {0x41, 2, 0}
};

struct lo_rf_setting {
    uint16_t RF;
    uint8_t vals[5]; // LOGenT, RXRFMixT, RXLNAT, TXRFMixT, TXPAT
};

static const struct lo_rf_setting freqdep_table[] = {
        {4810, {13, 27, 11, 15, 2}},
        {5003, {15, 27, 12, 14, 1}},
        {5076, {15, 27, 10, 13, 1}},
        {5173, {15, 27,  8, 12, 1}},
        {5245, {15, 24,  6, 12, 1}},
        {5342, {15, 21,  4, 12, 1}},
        {5511, {12, 14,  3, 11, 0}},
        {5681, { 9,  7,  1, 10, 0}},
        {5802, { 7,  2,  0,  9, 0}},
        {5947, { 5,  0,  0,  6, 0}},
};

static int spi_read(uint8_t reg, uint8_t *val)
{
    int loops = 1 << 20;

    rc_sw_ctrl_pack(1, 0, 1, 1, 1, reg, 0xff);
    while (rc_start_done_getf() && --loops)
        ;
    if (!loops)
    {
        dbg(D_ERR D_PHY "%s @0x%02x failed\n", __func__, reg);
        return -1;
    }
    *val = rc_data_getf();

    return 0;
}

static uint8_t __spi_read(uint8_t reg)
{
    uint8_t val = 0;
    return spi_read(reg, &val) ?: val;
}

static int spi_write(uint8_t reg, uint8_t val)
{
    int loops = 1 << 20;

    rc_sw_ctrl_pack(1, 0, 1, 0, 0, reg, val);
    while (rc_start_done_getf() && --loops)
        ;
    if (!loops)
    {
        dbg(D_ERR D_PHY "%s @0x%02x failed\n", __func__, reg);
        return -1;
    }
    return 0;
}

static void set_lo_rf_reg(const struct lo_rf_setting *lo_rf_e,
                          unsigned int ridx)
{
    uint8_t reg, val;
    reg = lo_rf_params[ridx].r;
    val = SET8(__spi_read(reg), lo_rf_e->vals[ridx], lo_rf_params[ridx].h,
               lo_rf_params[ridx].l);
    spi_write(reg, val);
}

static uint32_t spi_poll(uint8_t reg, uint8_t val)
{
    int loops = 1 << 20;
    while ((__spi_read(reg) & val) != val && --loops)
        ;
    return !loops;
}

static int reg_poll(uint32_t reg, uint32_t val)
{
    int loops = 1 << 20;
    while ((REG_PL_RD(reg) & val) != val && --loops)
        ;
    return !loops;
}

static int rc_calibrate(uint8_t chantype, uint8_t sx)
{
    unsigned int bw_sel, bw_param, rc_cal;
    int ret;

    /**
     * RC CAL - 6.4.1 RC calibration
     */
    spi_write(0x7c, 0x01 + 10 * sx); // select page SX0 (1) or SX1 (11)
    spi_write(0x4c, 0x1f); // datasheet says 0x1d

    spi_write(0x7c, 0x00); // select page 0
    spi_write(0x00 + 0x50 * sx, 0xc0);

    if ((ret = spi_poll(0x02 + 0x50 * sx, CO_BIT(5))))
    {
        dbg(D_ERR D_PHY "%s:RC cal not ready (0x%02x)\n", __func__,
            __spi_read(0x02 + 0x50 * sx));
        return ret;
    }

    rc_cal = __spi_read(0x02 + 0x50 * sx) & 0x1f;
    switch (chantype) {
    case PHY_CHNL_BW_20:
        bw_sel = 4;
        break;
    case PHY_CHNL_BW_40:
        bw_sel = 2;
        break;
    case PHY_CHNL_BW_80:
        bw_sel = 1;
        break;
    default:
        bw_sel = 1;
        dbg(D_ERR D_PHY "%s:Unsupported bandwidth (%d)\n", __func__, chantype);
        break;
    }
    bw_param = (42 * bw_sel * (36 + rc_cal)) / 47 - 12;

    spi_write(0x00 + 0x50 * sx, 0x00);

    spi_write(0x7c, 0x01 + 10 * sx); // select page SX0 (1) or SX1 (11)
    spi_write(0x4c, 0x05);

    if (!sx)
    {
        spi_write(0x78, 0x55); // RX0-3
        spi_write(0x7c, 0x02); // select page RX0 (2)
        spi_write(0x77, bw_param & 0xff);
        spi_write(0x78, 0xaa); // TX0-3
        spi_write(0x7c, 0x03); // select page TX0 (3)
        spi_write(0x40, bw_param & 0xff);
        spi_write(0x78, 0x00);
    }
    else
    {
        spi_write(0x7c, 0x02 + 8 * sx); // select page RX4 (10)
        spi_write(0x77, bw_param & 0xff);
    }

    /**
     * VCO CAL
     */
    spi_write(0x7c, 0x00); // select page 0
    spi_write(0x01 + 0x50 * sx, 0x1f); // Enable VCO
    spi_write(0x00 + 0x50 * sx, 0x1e); // Enable synthesizer, PLL &
                                       // VCO calibration
    if ((ret = spi_poll(0x03 + 0x50 * sx, CO_BIT(5))))
    {
        dbg(D_ERR D_PHY "%s:VCO cal not ready (0x%02x)\n", __func__,
            __spi_read(0x03 + 0x50 * sx));
        return ret;
    }

    return 0;
}

static int force2040_toggle(uint8_t chantype)
{
    uint32_t rxmodes, f2040_val, f2040_msk;

    f2040_msk = MDM_FORCE40_BIT | MDM_FORCE20_BIT;
    if (chantype == PHY_CHNL_BW_20)
        f2040_val = MDM_FORCE20_BIT;
    else if (chantype == PHY_CHNL_BW_40)
        f2040_val = MDM_FORCE40_BIT;
    else
        f2040_val = 0;

    rxmodes = (mdm_rxmodes_get() & ~f2040_msk) | f2040_val;

    mdm_rxmodes_set(rxmodes);
    if (reg_poll(ELMA_CLKRST_STAT_ADDR, ELMA_MMC_2_LOCK_BIT |
                                        ELMA_MMC_1_LOCK_BIT |
                                        ELMA_MMC_0_LOCK_BIT))
        return -1;
    mdm_rxmodes_set(rxmodes ^ f2040_msk);
    if (reg_poll(ELMA_CLKRST_STAT_ADDR, ELMA_MMC_2_LOCK_BIT |
                                        ELMA_MMC_1_LOCK_BIT |
                                        ELMA_MMC_0_LOCK_BIT))
        return -1;
    mdm_rxmodes_set(rxmodes);
    if (reg_poll(ELMA_CLKRST_STAT_ADDR, ELMA_MMC_2_LOCK_BIT |
                                        ELMA_MMC_1_LOCK_BIT |
                                        ELMA_MMC_0_LOCK_BIT))
        return -1;

    return 0;
}

static void __phy_hw_set_channel(uint16_t freq, uint16_t freq1, uint8_t chantype, uint8_t sx)
{
    unsigned int ipart, fpart, pssel, i;

    /**
     * BW dependent modem settings
     */
    if (force2040_toggle(chantype))
        dbg(D_ERR D_PHY "%s: MMCs not locked while toggling 2040\n", __func__);

    /**
     * PSSEL - use pssel of RW-WSDM-RIU-HW-REG.xls
     */
    if (chantype == PHY_CHNL_BW_40)
    {
        // pssel = 1 or 2
        pssel = freq < freq1 ? 1 : 2;
    }
    else if (chantype == PHY_CHNL_BW_80)
    {
        // pssel = 0, 1, 2 or 3
        int _offs = freq1 - freq;
        if (_offs > 0)
            pssel = _offs > 10 ? 0 : 1;
        else
            pssel = -_offs > 10 ? 3 : 2;
    }
    else
    {
        pssel = 0;
    }

    if (sx == 0)
    {
        riu_psselect_setf(pssel);

        //  3us for TX RAMPUP <=> TXRFRAMPUP = 360
        if (chantype == PHY_CHNL_BW_20)
        {
            mdm_txstartdelay_setf(0x00b4);
            riu_rwnxtxstartdelay_set(0x000000b4);
            // TBE for 60MHz
            mdm_tbectrl0_set(0x0c0f0700);
            mdm_waithtstf_setf(15);
            riu_rwnxagcaci20marg0_set(0);
            riu_rwnxagcaci20marg1_set(0);
            riu_rwnxagcaci20marg2_set(0);
        }
        else
        {
            mdm_txstartdelay_setf(0x0168);
            riu_rwnxtxstartdelay_set(0x00000168);
            // TBE for 120MHz
            mdm_tbectrl0_set(0x0c0f0702);
            if (chantype == PHY_CHNL_BW_40)
                mdm_waithtstf_setf(15);
            else
                mdm_waithtstf_setf(31);
            riu_rwnxagcaci20marg0_set(RIU_RWNXAGCACI20MARG0_RESET);
            riu_rwnxagcaci20marg1_set(RIU_RWNXAGCACI20MARG1_RESET);
            riu_rwnxagcaci20marg2_set(RIU_RWNXAGCACI20MARG2_RESET);
        }
    }
    else
    {
        if (FCU_PRESENT())
            fcu_fcpsselect_setf(pssel);
#if NX_RADAR_DETECT
        if (chantype == PHY_CHNL_BW_80 ||
            chantype == PHY_CHNL_BW_40)
            fcu_fccrossupthrqdbm_setf(0x3b7); // Cross-up   -73dBm
        else
            fcu_fccrossupthrqdbm_setf(0x3bc); // Cross-up   -68dBm
#endif
    }

    // freqdep_table entry
    for (i = 0; i < sizeof(freqdep_table) / sizeof(freqdep_table[0]) &&
                freqdep_table[i].RF < freq1; i++)
        ;
    if (i)
        i--;

    if (freqdep_table[i].RF >= freq1)
        dbg(D_ERR D_PHY "%s: No match found for freq %d (using %d)\n",
            __func__, freq1, freqdep_table[i].RF);



    // Sx0 (0x7c 1)
    spi_write(0x7c, 0x01 + 10 * sx); // select page SX0 (1) or SX1 (11)
    spi_write(0x78, 0x00); // XXX
    spi_write(0x79, 0x00); // XXX

    set_lo_rf_reg(&freqdep_table[i], LOGENT_RIDX);

    /**
     * 6.3.7.1
     * Calculation of the PLL division control word
     * ipart / fpart
     */
    ipart = ((freq1 * (use_pll_doubler ? 1 : 2)) << 7) / 40;
    fpart = ipart & ((1 << 7) - 1);
    // "least significant bit in the fractional part must always be set to 1"
    fpart = (fpart << (22 - 7)) | 1;
    ipart = (ipart >> 7) - 64;

    spi_write(0x7c, 0x00); // select page 0
    spi_write(0x07 + 0x50 * sx, ipart & 0xff);
    spi_write(0x06 + 0x50 * sx, (((fpart >> 16) & 0x3f) << 2) | ((ipart >> 8) & 0x3));
    spi_write(0x05 + 0x50 * sx, (fpart >>  8) & 0xff);
    spi_write(0x04 + 0x50 * sx, fpart & 0xff);

    if (rc_calibrate(chantype, sx))
        dbg(D_ERR D_PHY "%s:Calibration for freq %d failed\n", __func__, freq1);

    // RX
    if (!sx)
        spi_write(0x78, 0x55); // parallel select for RX0-3
#if 0
    spi_write(0x7c, 0x00); // select page 0
    spi_write(0x0a, 0x01); // newced Enable DC offset DAC's
    spi_write(0x09, 0x7c); // newced Configure DC offset for internal loop operation
    spi_write(0x09, 0x7d); // newced Activate DC offset
#endif
    spi_write(0x7c, 0x02 + 8 * sx); // select page RX0 (2) or RX4 (10)
    set_lo_rf_reg(&freqdep_table[i], RXRFMIXT_RIDX);
    set_lo_rf_reg(&freqdep_table[i], RXLNAT_RIDX);

    // TX
    if (!sx)
    {
        spi_write(0x78, 0xaa); // parallel select for TX0-3
        spi_write(0x7c, 0x03); // select page TX0 (3)
        set_lo_rf_reg(&freqdep_table[i], TXRFMIXT_RIDX);
        set_lo_rf_reg(&freqdep_table[i], TXPAT_RIDX);
    }
}

static void phy_hw_set_channel(uint16_t freq, uint16_t freq1, uint8_t chantype, uint8_t index)
{
    __phy_hw_set_channel(freq, freq1, chantype, index);

    // reqs for RC - RC handles 0x78 - 0x79 should be 0
    spi_write(0x7c, 0x00); // select page 0
    // remove prescaler for normal operation
    // shouldn't be needed since last op is a spi_write
    rc_sw_ctrl_set(0x01000000);
}

/**
 *
 * Datasheet:
 * - Hibernate to Standby
 *       Run procedure Calibration clock setup, described in chapter 6.3.3.1.
 *       Run procedure Bandgap enable, described in chapter 6.3.4.1.
 * - Standby to Idle
 *       Run procedure LDO enable, described in chapter 6.3.6.1.
 * - Idle to PLL ON
 *       Run procedure synthesizer initial programming, described in chapter 6.3.7.2
 *       Run procedure Rx/TX and LO enable, described in chapter 6.3.8.1
 *       Run step 1 to 3 in procedure Rx enable , 6.3.9.1, to initiate Rx chains.
 *       Run step 1 to 3 in procedure Tx enable , 6.3.10.1 to initiate Tx chains.
 *
 *
 * 0x78 0x79 : selenbypass TX:0xaa,N/A RX:0x55,0x01
 * 0x7c      : page select
 */
static void elma_init(void)
{
    int i;

    spi_write(0x7c, 0x00); // select page 0
    rev = __spi_read(0x7f);
    dbg(D_INF D_PHY "%s: Chip ID 0x%02x - rev 0x%02x, FCU %ssent\n", __func__,
        __spi_read(0x7e), rev, FCU_PRESENT() ? "pre" : "ab");

    /*
     * 6.3.7.1) "Note for chip revision A0:
     * Do not use the doubler functionality as this * degrades performance." */
    use_pll_doubler = rev != REV_A0;

    /**
     * TODO RW-WLAN-nX-RC-ELMA-DD:
     * Depending of the different version of RF chip [1] (A0 and B0 chip
     * version), the radio controller FSM can operate with 2 different modes,
     * one with the parallel mode which can be let always active (RF B0 chip
     * version) and one where parallel mode must toggle active/inactive between
     * gain setting sequence and TX/RX switching sequence (RF A0 chip version)
     */

    /**
     * HIBERNATE -> STANDBY
     */
    // Calibration clock setup 6.3.3.1
    spi_write(0x7d, 0x01); // Force reference clock on

    // Bandgap Enable { 6.3.4.1
    for (i = 0; i < 2; i++) {
        spi_write(0x7c, 0x01 + 10 * i); // select page SX0 (1) or SX1 (11)
        spi_write(0x42, 0x60); // enable SXi
    }

    spi_write(0x7c, 0x02); // select page RX0 (2)
    spi_write(0x78, 0x55); // select RX3 RX2 RX1 RX0
    spi_write(0x79, 0x01); //        and RX4 for parallel select
    spi_write(0x43, 0x60); // enable RX0-4

    spi_write(0x7c, 0x03); // select page TX0 (3)
    spi_write(0x78, 0xaa); // select TX3 TX2 TX1 TX0
    spi_write(0x79, 0x00); //        for parallel select
    spi_write(0x42, 0x60);
    // } Bandgap Enable

    /**
     * STANDBY -> IDLE
     */
    // LDO Enable { 6.3.6.1
    spi_write(0x78, 0x00); // XXX

    for (i = 0; i < 2; i++) {
        spi_write(0x7c, 0x01 + 10 * i); // select page SX0 (1) or SX1 (11)
        spi_write(0x42, 0x69); // enable SXi LO ldo
        if (rev != REV_A0)
            spi_write(0x5e, 0x29); // ??? B0 LODIV LDO control
        spi_write(0x5d, 0x29); // enable SXi CP ldo
    }

    spi_write(0x7c, 0x02); // select page RX0 (2)
    spi_write(0x78, 0x55); // select RX3 RX2 RX1 RX0
    spi_write(0x79, 0x01); //        and RX4 for parallel select
    spi_write(0x43, 0x69); // enable RX LDO’s

    spi_write(0x7c, 0x03); // select page TX0 (3)
    spi_write(0x78, 0xaa); // parallel select for TX0-3
    spi_write(0x79, 0x00); // clear parallel select for RX4
    spi_write(0x42, 0x69); // enable TX LDO’s
    // } LDO Enable

    /**
     * IDLE -> PLL-ON
     */
    // PLL enable { 6.3.7.2
    spi_write(0x78, 0x00); // clear parallel select
    for (i = 0; i < 2; i++) {
        spi_write(0x7c, 0x01 + 10 * i); // select page SX0 (1) or SX1 (11)

        if (use_pll_doubler) {
            // "Enable frequency doubling of main PLL reference clock (wlsx_refclk)"
            spi_write(0x5c, __spi_read(0x5c) | (1 << 4));
        }
        spi_write(0x54, 0x1f);
        // PLL initialization { 6.3.1.3
        spi_write(0x49, 0x52); // Disable LPF offset
        spi_write(0x56, 0x27); // LO PTAT current adj

        spi_write(0x44, 0x00); // custom XO tuning to highest frequency
    }

    // } PLL initialization
    // do we really need this: ?
    // (the doc states following section relies on LDO lock)
    phy_hw_set_channel(5500, 5500, PHY_CHNL_BW_20, 0);
    phy_hw_set_channel(5500, 5500, PHY_CHNL_BW_20, 1);
    // }
    // } PLL enable

    // RX/TX power and LO distribution drivers enable { 6.3.8.1
    // RX {
    spi_write(0x78, 0x55); // parallel select for RX0-3
    spi_write(0x79, 0x01); // parallel select for RX4
    spi_write(0x7c, 0x02); // select page RX0 (2)
    spi_write(0x44, 0xff); // Configuration, enable current sources RXs
    spi_write(0x45, 0x5F);
    spi_write(0x4a, 0x16);

    // perform analog initialization of Rx according to section 6.3.1.1 and 6.3.1.4)
    // Rx init 6.3.1.4
    spi_write(0x40, 0x01);
    spi_write(0x41, 0x7e);
    spi_write(0x42, 0x33);
    spi_write(0x46, 0x04);
    spi_write(0x47, 0x34); // not in table
    spi_write(0x48, 0x8b);
    spi_write(0x38, 0x25);
    spi_write(0x39, 0x2c);
    spi_write(0x3a, 0x2e);
    spi_write(0x3b, 0x10);
    spi_write(0x3c, 0x08);
    spi_write(0x3d, 0x08);
    spi_write(0x3e, 0x08);
    spi_write(0x3f, 0x08);

    spi_write(0x7c, 0x00); // select page (0)
    spi_write(0x0c, 0x08); // Enable LO buffer RXs
    // } RX

    // TX {
    spi_write(0x7c, 0x03); // select page TX0 (3)
    spi_write(0x78, 0xaa); // parallel select for TX0-3
    spi_write(0x79, 0x00);
    spi_write(0x40, 0x9c); // custom
    spi_write(0x46, 0xf3); // Configuration, enable current sources TX0-TX3
    spi_write(0x47, 0x0f); // Configuration, enable current sources TX0-TX3
    spi_write(0x4b, 0x16); // Configuration, enable current sources TX0-TX3
    // perform analog initialization of Tx according to section 6.3.1.1 and 6.3.1.5)
    spi_write(0x45, 0x0b); // custom
    spi_write(0x41, 0x30); // newced rfpagain
    // Tx init 6.3.1.5
    // analog initialization of Rx
    spi_write(0x44, 0x22);
    spi_write(0x43, 0x07);

    spi_write(0x7c, 0x00); // select page 0
    spi_write(0x16, 0x08); // Enable LO buffer TX0-TX3
    // } TX
    // } RX/TX power and LO distribution drivers enable ???

    // no TX chain enable .. what about RC ?
    // custom
    spi_write(0x78, 0xaa); // parallel select for TX0-3
    spi_write(0x22, 0x3f);
    spi_write(0x26, 0x0f);
    spi_write(0x21, 0x11);

    spi_write(0x78, 0x55); // parallel select for RX0-3
    spi_write(0x79, 0x01); // parallel select for RX4
    // 6.3.9 Rx chain enable
    spi_write(0x09, 0x70);
    spi_write(0x0c, 0x0b);
    spi_write(0x0d, 0x49);

    //  Enable ALL TX, RX
    // TX
    spi_write(0x78, 0xaa); // parallel select for TX0-3
    spi_write(0x79, 0x00);
    spi_write(0x10, 0x06);
    // RX
    spi_write(0x78, 0x55); // parallel select for RX0-3
    spi_write(0x79, 0x01); // parallel select for RX4
    spi_write(0x08, 0x01);

    spi_write(0x78, 0x00); // clear parallel select
    spi_write(0x79, 0x00); // clear parallel select

    /**
     * Prepare for RC {
     */

    // Fast-Write @ddresses {
    spi_write(0x7c, 0x0f); // select page SPI (15)
    // fastadr1..4   <-- RX gain control
    spi_write(0x40, 0x0d);
    if (FCU_PRESENT())
    {
        spi_write(0x41, 0x3d);
        spi_write(0x43, 0x1d);
    }
    else
    {
        spi_write(0x41, 0x1d);
        spi_write(0x43, 0x3d);
    }
    spi_write(0x42, 0x2d);

    // fastadrtx1..4 <-- TX gain control
    if (rev != REV_A0)
    {
        spi_write(0x48, 0x17);
        spi_write(0x49, 0x27);
        spi_write(0x4a, 0x37);
        spi_write(0x4b, 0x47);
    }
    // } Fast-Write @ddresses

    // Set RSSI order on SPI MOSI
    spi_write(0x45, 0x39);

    spi_write(0x7c, 0x00); // select page (0)
    // enable RSSI for RX0-3 chains
    spi_write(0x0c, 0x2b);
    spi_write(0x1c, 0x2b);
    spi_write(0x2c, 0x2b);
    spi_write(0x3c, 0x2b);

    // remove prescaler for normal operation
    // shouldn't be needed since last op is a spi_write
    rc_sw_ctrl_set(0x01000000);
    // } Prepare for RC
}

static void phy_hw_init(void)
{
    nxmac_abgn_mode_setf(MODE_802_11AC_5);
    mdm_rxmodes_set(MDM_RXSTBCEN_BIT | MDM_DUPMRCEN_BIT | MDM_RXALLOWVHT_BIT |
                    MDM_RXALLOWMM_BIT | MDM_RXALLOWLEGACY_BIT);
    mdm_propmodecfg_setf(1);
    mdm_lsigparitycfg_setf(0);
    mdm_rxndbpsmax_setf(800);
    mdm_delaynormalgi_setf(17);

    elma_adc_spi_prescaler_setf(6);
    elma_adc_reset_setf(1);
    elma_adc_reset_setf(0);
    elma_dac_reset_setf(1);
    elma_dac_reset_setf(0);
    elma_dac_12b_cntl_set(0x05020114);

    elma_diagportconf1_set(0x07001400);
    elma_phydiag_conf1_set(0x00001000);

    /**
     * RIU INIT
     */
    // TX FE delay
    riu_rwnxtxfedelay_set(0x00030001);

    // I/Q swap for DAC & ADC
    riu_rwnxiqctrl_set(0x7FC00000);

    // Active antenna set to 1
    riu_rwnxstaticconfig_set(0x00000001);

    // Digital gain
    riu_rwnxfectrl0_set(0x00111111);


    riu_rwnxagcccastate0_set(0x00d006f8);

    // Set OFDM mode only
    riu_rwnxagccntl_set(0x0001050f);
    // Set LNA gains [3:0]
    riu_rwnxagclnagainref0_set(0x0e087a69);
    // Set LNA gains [6:4]
    riu_rwnxagclnagainref1_set(0x00221c13);

    //  LNA Thr [3:0], [5:4]
    riu_rwnxagclnathrset0ref0_set(0x2621150c);
    riu_rwnxagclnathrset0ref1_set(0x00002e29);
    //  LNA NF [3:0], [6:4]
    riu_rwnxagclnanfref0_set(0x0c080d14);
    riu_rwnxagclnanfref1_set(0x00030305);
    //  LNA ICP1 [3:0], [6:4]
    riu_rwnxagclnaicp1ref0_set(0x69747d0e);
    riu_rwnxagclnaicp1ref1_set(0x005b6063);
    // Disable cross detection
    riu_rwnxagccross_set(0x0027c288);

    riu_rwnxagcccatimeout_set(8000000); // 100ms
    riu_irqmacccatimeouten_setf(1);

#if NX_RADAR_DETECT
    riu_radardeten_setf(1);
    if (FCU_PRESENT())
    {
        riu_irqmacfcradardeten_setf(1);

        riu_activeant_setf(3); // enable 2 antennas in RIU and select path
        riu_combpathsel_setf(1); // Reduce to 1 antenna in AGC

        fcu_fcsharedlnasel_setf(1); // Select antenna 1
        fcu_rwnxfcagccntl_set(FCU_FCRADARDETEN_BIT | FCU_FCENABLE_BIT);
        fcu_fcapbclkfreq_setf(80);

        fcu_fcdcradartype_setf(3);
        fcu_fccrossdnthrqdbm_setf(0x3b0); // Cross-down -80dBm
        fcu_fccrossupthrqdbm_setf(0x3b7); // Cross-up   -73dBm

        fcu_fcevt1tgtadd_setf(0x7);
        fcu_fcevt2tgtadd_setf(0xc);
        fcu_fcevt3tgtadd_setf(0x10);
        fcu_fcadcpowsupthrdbm_setf(0xba);
    }
    riu_irqmacradardeten_setf(1);
#endif

    /**
     * RC init
     */
    rc_hw_tx_gain_data_spi_set(0x3e3e3e3e);
    rc_extpa_seq_on_delay_set(0xa0a0a0a0);
    rc_extpa_seq_off_delay_set(0x05050505);
    rc_paoffdata_setf(1);
    rc_ext_lna_lut_set(0x06543777);
    rc_rf_lna_lut_set(0x00123134);

    elma_init();

    if (rev == REV_A0)
    {
        rc_hw_spi_ctrl_set(RC_RFRSSISYNCSEL_BIT | RC_PARALLELSELDIS_BIT);
#if NX_RADAR_DETECT
        elma_rfchannel_conf_set(0x04321001);
#else
        elma_rfchannel_conf_set(1);
#endif
    }
    else
    {
        rc_hw_tx_gain_addr_spi_set(0x47372717);
        rc_hw_spi_ctrl_set(RC_RFRSSISYNCSEL_BIT | RC_TXFASTWREN_BIT);
        elma_rfchannel_conf_set(ELMA_RFCHANNEL_CONF_RESET | 1);
    }
}

struct phy_env_tag phy_env[2];

void phy_init(const struct phy_cfg_tag *config)
{
    int i;

    phy_hw_init();

    for (i = 0; i < 2; i++) {
        phy_env[i].chan.band         = PHY_BAND_5G;
        phy_env[i].chan.type         = PHY_CHNL_BW_OTHER;
        phy_env[i].chan.prim20_freq  =
        phy_env[i].chan.center1_freq =
        phy_env[i].chan.center2_freq = PHY_UNUSED;
    }
}

void phy_mdm_isr(void)
{
    dbg(D_ERR D_PHY "%s: TODO\n", __func__);
}

/**
 * FIXME: This is for debug purpose only and does not call rd_event_ind
 */
void phy_rc_isr(void)
{
    uint32_t irq_status = riu_rwnxmacintstatmasked_get();

    riu_rwnxmacintack_clear(irq_status);

    #if NX_RADAR_DETECT
    if (FCU_PRESENT())
    {
        if (irq_status & RIU_IRQMACFCRADARDETMASKED_BIT)
        {
            PROF_RADAR_IRQ_SET();
            rd_event_ind(PHY_SEC);
            PROF_RADAR_IRQ_CLR();
        }
    }
    if (irq_status & RIU_IRQMACRADARDETMASKED_BIT)
    {
        PROF_RADAR_IRQ_SET();
        rd_event_ind(PHY_PRIM);
        PROF_RADAR_IRQ_CLR();
    }
    #endif

    ASSERT_REC(!(irq_status & RIU_IRQMACCCATIMEOUTMASKED_BIT));
}

void phy_get_version(uint32_t *version_1, uint32_t *version_2)
{
    *version_1 = mdm_nxversion_get();
    // TODO Add version reading for other PHY elements than modem.
    *version_2 = 0;
}

void phy_set_channel(const struct mac_chan_op *chan, uint8_t index)
{
    struct phy_env_tag *_phy_env;

    if (index > 1)
    {
        dbg(D_ERR D_PHY "%s: radio %d does not exist\n", __func__, index);
    }

    if (chan->band == PHY_BAND_2G4)
    {
        dbg(D_ERR D_PHY "%s: 2.4GHz is unsupported\n", __func__);
        return;
    }

    _phy_env = &phy_env[index];

    if (_phy_env->chan.band == chan->band &&
        _phy_env->chan.type == chan->type &&
        _phy_env->chan.flags == chan->flags &&
        _phy_env->chan.prim20_freq == chan->prim20_freq &&
        _phy_env->chan.center1_freq == chan->center1_freq &&
        _phy_env->chan.center2_freq == chan->center2_freq)
    {
        dbg(D_INF D_PHY "%s: Setting same channel, do nothing\n");
        return;
    }

    phy_hw_set_channel(chan->prim20_freq, chan->center1_freq, chan->type, index);
    _phy_env->chan = *chan;

#if NX_RADAR_DETECT
    /* HW only handles same BW for both synths
     * To cope with that practically and make synth1 bw follow synth0 bw,
     * copy synth0 settings in synth1's - if a user switches synth1, i.e.
     * index == 1, assume they know what they're doing */
    if (FCU_PRESENT() && index == 0 && phy_env[1].chan.type != chan->type)
    {
        dbg(D_INF D_PHY "%s: Align synth1 bw on synth0 bw\n", __func__);
        phy_hw_set_channel(chan->prim20_freq, chan->center1_freq, chan->type, 1);
        phy_env[1] = phy_env[0];
    }
#endif
}

void phy_get_channel(struct phy_channel_info *info, uint8_t index)
{
    struct phy_env_tag *_phy_env;

    if (index > 1)
    {
        dbg(D_ERR D_PHY "%s: radio %d does not exist\n", __func__, index);
    }
    _phy_env = &phy_env[index];
    // Map the band, channel type, primary channel index on info1
    info->info1 = _phy_env->chan.band | (_phy_env->chan.type << 8) | (_phy_env->chan.prim20_freq << 16);
    // Map center freq on info2
    info->info2 = _phy_env->chan.center1_freq | (_phy_env->chan.center2_freq << 16);
}

void phy_stop(void)
{
    if (nxmac_current_state_getf() != HW_IDLE)
        dbg(D_ERR "%s MAC state != IDLE\n", __func__);
}

uint32_t phy_get_channel_switch_dur(void)
{
    return 1500;
}

#if NX_RADAR_DETECT
bool phy_has_radar_pulse(int rd_idx)
{
    if (rd_idx == PHY_PRIM)
        return (riu_radfifoempty_getf() == 0);
    else
        return (fcu_fcradfifoempty_getf() == 0);
}

bool phy_get_radar_pulse(int rd_idx, struct phy_radar_pulse *pulse)
{
    if (rd_idx == PHY_PRIM)
    {
        // Check if FIFO is empty
        if (riu_radfifoempty_getf())
            return (false);

        pulse->pulse = REG_PL_RD(__RIU_RADARFIFO_ADDR);
    }
    else
    {
        if (fcu_fcradfifoempty_getf())
            return (false);

        pulse->pulse = REG_PL_RD(__FCU_RADARFIFO_ADDR);
    }

    return (true);
}
#endif

#if NX_DEBUG_DUMP
// same as trident
void phy_get_diag_state(struct dbg_debug_info_tag *dbg_info)
{
    int i;
    uint8_t mictor1_sel = elma_mictor_1_sel_getf();
    uint8_t diag = elma_phydiag_conf_lsb_1_getf();

    // Get diagnostic port state for the PHY
    elma_mictor_1_sel_setf(0x14);
    for (i = 0; i < DBG_DIAGS_PHY_MAX; i++)
    {
        // Go through the different banks and copies the diag values
        elma_phydiag_conf_lsb_1_setf(i);
        dbg_info->diags_phy[i] = elma_diag1_stat_get() & 0xFFFF;
    }
    elma_phydiag_conf_lsb_1_setf(diag);
    elma_mictor_1_sel_setf(mictor1_sel);
}
#endif

bool phy_uf_supported(void)
{
    return false;
}

void phy_uf_enable(bool enable)
{
}

bool phy_vht_supported(void)
{
    return true;
}

bool phy_he_supported(void)
{
    return false;
}

bool phy_ldpc_tx_supported(void)
{
    return false;
}

bool phy_ldpc_rx_supported(void)
{
    return false;
}

bool phy_bfmee_supported(void)
{
    return false;
}

bool phy_bfmer_supported(void)
{
    return false;
}

bool phy_mu_mimo_rx_supported(void)
{
    return false;
}

bool phy_mu_mimo_tx_supported(void)
{
    return false;
}

#if RW_MUMIMO_RX_EN
void phy_set_group_id_info(uint32_t membership_addr, uint32_t userpos_addr)
{
}
#endif

void phy_set_aid(uint16_t aid)
{
}

uint8_t phy_get_bw(void)
{
    return 2;
}

uint8_t phy_get_nss(void)
{
    return 0;
}

uint8_t phy_get_ntx(void)
{
    return (0);
}

uint8_t phy_get_nrx(void)
{
    return (0);
}

uint8_t phy_get_bfr_mem_size(void)
{
    return (0);
}

void phy_get_rf_gain_idx(int8_t *power, uint8_t *idx)
{
    /* TO BE COMPLETED. For now set idx to power for debug */
    *idx = (uint8_t)*power;
}

void phy_get_rf_gain_capab(int8_t *max, int8_t *min)
{
    /* TO BE UPDATED with correct value */
    *max = 25; // dBm
    *min = 0;  // dBm
}

uint8_t phy_switch_antenna_paths(void)
{
    return 0;
}

