/**
 ****************************************************************************************
 *
 * @file phy_karst.c
 *
 * Copyright (C) RivieraWaves 2014-2019
 *
 * When setting a channel, several procedures can be advantageously left-out depending on
 * whether the band, bw, frequency remain unchanged (calibrations, old modem clk mmc
 * toggles ..), e.g. for multi-channel. Although the changes are trivial they might get
 * in the way of properly measuring the RF behavior.
 ****************************************************************************************
 */
#include "rwnx_config.h"
#include "phy.h"
#include "dbg.h"

#include "hal_machw.h"
#include "reg_mac_core.h"

#if NX_MDM_VER < 30
#include "reg_mdm_stat.h"
#include "reg_mdmdsss_cfg.h"
#endif
#include "reg_mdm_cfg.h"

#include "reg_riu.h"
#include "reg_rc.h"
#include "reg_karst_if.h"
#include "reg_macbypass.h"

#ifdef CFG_VIRTEX6
#include "reg_fpgaa.h"
#endif /* CFG_VIRTEX6 */

#include "crm.h"

#include "rd.h"
#include "uf.h"

#if NX_MDM_VER < 20
#error "Needs to be compiled with modem version 20 minimum"
#endif

#if NX_UF_EN
#define RXV_REG_MIN_VERS    2
#endif

/// Structure containing the parameters of the Karst PHY configuration
struct phy_karst_cfg_tag
{
    /// TX IQ mismatch compensation in 2.4GHz
    uint32_t tx_iq_comp_2_4G[2];
    /// RX IQ mismatch compensation in 2.4GHz
    uint32_t rx_iq_comp_2_4G[2];
    /// TX IQ mismatch compensation in 5GHz
    uint32_t tx_iq_comp_5G[2];
    /// RX IQ mismatch compensation in 5GHz
    uint32_t rx_iq_comp_5G[2];
    /// RF path used by default (0 or 1)
    uint8_t path_used;
};

/// PHY driver context.
struct phy_env_tag
{
    /// Karst configuration parameters
    struct phy_karst_cfg_tag cfg;
    /// Currently configured channel
    struct mac_chan_op chan;
};

struct phy_env_tag phy_env;

#if NX_MDM_VER < 30
#define REG_DFLT_SW_CTRL    ((1 << RC_PRESCALER_LSB) & (~RC_START_DONE_BIT))
#else
#define REG_DFLT_SW_CTRL    ((1 << RC_SW_PRESCALER_LSB) & (~RC_START_DONE_BIT))
#endif

#define __RIU_RADARFIFO_ADDR 0x60C04000
#define __LDPC_RAM_ADDR      0x60C09000

// To keep correct TX quality limit pwr to 10 dBm
#define PHY_KARST_MAX_PWR    ((int8_t)10) // dBm
// And stay above 1 dBm
#define PHY_KARST_MIN_PWR    ((int8_t)-10)  // dBm

static uint16_t rev;

static int spi_read(uint16_t reg, uint16_t *val)
{
    int loops = 1 << 20;

    rc_sw_ctrl_pack(2, 1, 1, reg, 0xffff);
    while (rc_start_done_getf() && --loops)
        ;
    if (!loops)
    {
        dbg(D_ERR D_PHY "%s @0x%04x failed\n", __func__, reg);
        return -1;
    }
    *val = rc_data_getf();

    return 0;
}

static uint16_t __spi_read(uint16_t reg)
{
    uint16_t val = 0;
    return spi_read(reg, &val) ?: val;
}

static uint16_t __spi_read_(uint16_t reg)
{
    uint16_t val = 0;
    return spi_read(reg, &val) ?: val;
}

static int spi_write(uint16_t reg, uint16_t val)
{
    int loops = 1 << 20;

    rc_sw_ctrl_pack(2, 1, 0, reg, val);
    while (rc_start_done_getf() && --loops)
        ;
    if (!loops)
    {
        dbg(D_ERR D_PHY "%s @0x%04x failed\n", __func__, reg);
        return -1;
    }
    return 0;
}

#define spi_poll(r,v,v2) _spi_poll(r,v,v2,__LINE__)
static uint32_t _spi_poll(uint16_t reg, uint16_t val, uint16_t val2, int l)
{
    uint32_t spi_timeout_us = 500;
    uint32_t start = nxmac_monotonic_counter_2_lo_get();

    while ((__spi_read_(reg) & val) != val2)
    {
        if ((nxmac_monotonic_counter_2_lo_get() - start) > spi_timeout_us)
        {
            dbg(D_ERR D_PHY "%s:%d @0x%04x(0x%04x) 0x%04x 0x%04x failed\n",
                __func__, l, reg, __spi_read(reg), val, val2);
            ASSERT_WARN(0);
            return -1;
        }
    }

    return 0;
}

static int reg_poll(uint32_t reg, uint32_t val)
{
    int loops = 1 << 20;

    while ((REG_PL_RD(reg) & val) != val && --loops)
        ;
    if (!loops)
    {
        dbg(D_ERR D_PHY "%s @0x%08x 0x%08x failed\n", __func__, reg, val);
        return -1;
    }
    return !loops;
}

static inline void udelay(uint32_t us)
{
    uint32_t e = nxmac_monotonic_counter_2_lo_get() + us;

    do {
        volatile int n = 1 << 5; // relax
        while (n--)
            ;
    } while ((int32_t)(nxmac_monotonic_counter_2_lo_get() - e) < 0);
}


#if defined(CFG_VIRTEX6)
#define KARST_ADC_SPI_PRESCALER (0xF << KARST_ADC_SPI_PRESCALER_LSB)
#elif defined(CFG_VIRTEX7)
#define KARST_ADC_SPI_PRESCALER (0x1F << KARST_ADC_SPI_PRESCALER_LSB)
#else
#error("Unsupported platform")
#endif

#if defined(CFG_VIRTEX6)
#define IODELAY_ADC0 0x1A
#define IODELAY_ADC1 0x1A
#elif defined (CFG_VIRTEX7)
#define IODELAY_ADC0 0x00
#define IODELAY_ADC1 0x00
#endif /* CFG_VIRTEX6 */

static void adc_wr(uint32_t sel, uint32_t addr, uint32_t data)
{
    karst_adc_spi_addr_set(KARST_ADC_SPI_WE_BIT | ((uint32_t)sel << KARST_ADC_SPI_SEL_LSB) |
                                                  ((uint32_t)addr << KARST_ADC_SPI_ADDR_LSB));
    karst_adc_spi_data_set(data);
    karst_adc_spi_cntl_set(KARST_ADC_SPI_PRESCALER | KARST_ADC_SPI_START_BIT);
    reg_poll(KARST_ADC_SPI_CNTL_ADDR, KARST_ADC_SPI_DONE_BIT);
    karst_adc_spi_cntl_set(KARST_ADC_SPI_PRESCALER);
}

struct mode {
    uint16_t LO_5G9         :1; // 0  Enable LO for 5.9GHz RF
    uint16_t RXRF_5G9       :1; // 1  Enable RXRF for 5.9GHz RF
    uint16_t TXRF_5G9       :1; // 2  Enable TXRF for 5.9GHz RF
    uint16_t TXPA_5G9       :1; // 3  Enable internal PA for 5.9GHz RF
    uint16_t LO_2G4         :1; // 4  Enable LO for 2.4GHz RF
    uint16_t RXRF_2G4       :1; // 5  Enable RXRF for 2.4GHz RF
    uint16_t TXRF_2G4       :1; // 6  Enable TXRF for 2.4GHz RF
    uint16_t TXPA_2G4       :1; // 7  Enable internal PA for 2.4GHz RF
    uint16_t SYNTH          :1; // 8  Enable Synthesizer
    uint16_t RXBB           :1; // 9  Enable analog RX Base Band
    uint16_t TXBB           :1; // 10 Enable analog TX Base Band
    uint16_t WBPLL          :1; // 11 start Turn on and start Wide band PLL
    uint16_t unused         :1; // 12 Not used
    uint16_t BB_bw_mode     :2; // 13-14 Base band filter band width mode:
                                //       b00=10MHz, b01=20MHz, b10=40MHz
    uint16_t PLL_alt_channel:1; // 15 Use Synthesizer LUT setting 0 or 1
};

struct rv {
    uint16_t reg;
    uint16_t val;
};

static const struct rv rv_defaults[] = {
    // WF11AC Register Defaults
    // Created from WF11AC Register Definition, version 0.7, 2014-10-28
    {0x001, 0x0038},
    {0x023, 0x0000},
    {0x051, 0x0000},
    {0x052, 0x3671},
    {0x053, 0x02C6},
    {0x054, 0x3F80},
    {0x055, 0x4418},
    {0x058, 0x8384},
    {0x05B, 0x71D4},
    {0x05C, 0xCCC0},
    {0x05D, 0x0055},
    {0x05E, 0x0020},
    {0x05F, 0x3F02},
    {0x081, 0x005F},
    {0x083, 0x0411},
    {0x084, 0x0060},
    {0x088, 0x999A},
    {0x089, 0x7599},
    {0x099, 0x7600},
    {0x0A0, 0x00FF},
    {0x0A2, 0x0300},
    {0x0B0, 0x000F},
    {0x0B1, 0x6277},
    {0x0B8, 0x000F},
    {0x0B9, 0x6177},
    {0x0C0, 0x0003},
    {0x0D0, 0x0007},
    {0x0D1, 0x0007},
    {0x0D2, 0xB333},
    {0x0D3, 0x6000},
    {0x0D8, 0x0007},
    {0x0D9, 0x0007},
    {0x0DA, 0xB333},
    {0x0DB, 0x6000},
    {0x120, 0x3FEE},
    {0x121, 0x3935},
    {0x122, 0x2D77},
    {0x123, 0x23F9},
    {0x124, 0x203B},
    {0x125, 0x1CBC},
    {0x126, 0x197D},
    {0x127, 0x16BF},
    {0x128, 0x1400},
    {0x129, 0x1200},
    {0x12A, 0x1000},
    {0x12B, 0x0E41},
    {0x12C, 0x0CC3},
    {0x12D, 0x0B44},
    {0x12E, 0x0A05},
    {0x12F, 0x0906},
    {0x130, 0x0707},
    {0x131, 0x0589},
    {0x132, 0x048B},
    {0x133, 0x000C},
    {0x138, 0x0201},
    {0x139, 0x120A},
    {0x13A, 0x221A},
    {0x13B, 0x2423},
    {0x140, 0x021F},
    {0x141, 0x0403},
    {0x142, 0x2005},
    {0x143, 0x2221},
    {0x144, 0x2423},
    {0x145, 0x4025},
    {0x146, 0x4241},
    {0x147, 0x4443},
    {0x148, 0x6160},
    {0x149, 0x6362},
    {0x14A, 0x6564},
    {0x14B, 0x8281},
    {0x14C, 0x8483},
    {0x14D, 0xA185},
    {0x14E, 0xA3A2},
    {0x14F, 0xC0A4},
    {0x150, 0xC2C1},
    {0x151, 0xC4C3},
    {0x152, 0xE0C5},
    {0x153, 0xE2E1},
    {0x154, 0xE4E3},
    {0x155, 0xE6E5},
    {0x156, 0xE8E7},
    {0x157, 0xEAE9},
    {0x158, 0xECEB},
    {0x159, 0xEEED},
    {0x15A, 0xF0EF},
    {0x15B, 0xF2F1},
    {0x15C, 0xF4F3},
    {0x15D, 0xF6F5},
    {0x15E, 0xF8F7},
    {0x15F, 0xF8F7},
    {0x160, 0x0100},
    {0x161, 0x0302},
    {0x162, 0x0504},
    {0x163, 0x2120},
    {0x164, 0x2322},
    {0x165, 0x2524},
    {0x166, 0x4140},
    {0x167, 0x4342},
    {0x168, 0x4544},
    {0x169, 0x6160},
    {0x16A, 0x6362},
    {0x16B, 0x6564},
    {0x16C, 0x8180},
    {0x16D, 0x8382},
    {0x16E, 0x8584},
    {0x16F, 0x8986},
    {0x1B0, 0x0068},
    {0x1B1, 0x0069},
    {0x1B2, 0x0088},
    {0x1B3, 0x0089},
    {0x1B4, 0x00a7},
    {0x1B5, 0x00a8},
    {0x1B6, 0x00a9},
    {0x1B7, 0x00c7},
    {0x1B8, 0x00c8},
    {0x1B9, 0x00c9},
    {0x1BA, 0x00e7},
    {0x1BB, 0x00e8},
    {0x1BC, 0x00e9},
    {0x1BD, 0x0107},
    {0x1BE, 0x0108},
    {0x1BF, 0x0109},
    {0x1C0, 0x0007},
    {0x1C1, 0x0007},
    {0x1C2, 0x0007},
    {0x1C3, 0x0007},
    {0x1C4, 0x0007},
    {0x1C5, 0x0007},
    {0x1C6, 0x0007},
    {0x1C7, 0x0008},
    {0x1C8, 0x0009},
    {0x1C9, 0x0027},
    {0x1CA, 0x0028},
    {0x1CB, 0x0029},
    {0x1CC, 0x0047},
    {0x1CD, 0x0048},
    {0x1CE, 0x0049},
    {0x1CF, 0x0067},
    {0x1D1, 0x0101},
    {0x1D2, 0x0B03},
    {0x1D3, 0x0D05},
    {0x1D4, 0x0D0D},
    {0x1D5, 0xC101},
    {0x1D6, 0xCB03},
    {0x1D7, 0xCD05},
    {0x1D8, 0xCD0D},
    {0x1D9, 0x8110},
    {0x1DA, 0x8B30},
    {0x1DB, 0x8D50},
    {0x1DC, 0x8DD0},
    {0x1DD, 0x8000},
    {0x1DE, 0x8FF0},
    {0x1DF, 0x2F0F},
    {0x1E0, 0x2030},
    {0x1E1, 0x2030},
    {0x200, 0x02B0},
    {0x212, 0x6868},
    {0x216, 0x0012},
    {0x220, 0x2D40},
    {0x221, 0x1720},
    {0x222, 0x0B10},
    {0x223, 0x0608},
    {0x224, 0x0304},
    {0x225, 0x0102},
    {0x226, 0x0101},
    {0x227, 0x0101},
    {0x228, 0x2E17},
    {0x229, 0x10C8},
    {0x22A, 0x0443},
    {0x22B, 0x0421},
    {0x251, 0x0000},
    {0x258, 0x8384},
    {0x25B, 0x71D4},
    {0x25C, 0xCCC0},
    {0x25D, 0x0055},
    {0x25E, 0x0020},
    {0x25F, 0x3F02},
    {0x2A0, 0x00FF},
    {0x2A2, 0x0300},
    {0x2B0, 0x000F},
    {0x2B1, 0x6277},
    {0x2B8, 0x000F},
    {0x2B9, 0x6177},
    {0x2BB, 0x0800},
    {0x2C0, 0x0003},
    {0x2D0, 0x0007},
    {0x2D1, 0x0007},
    {0x2D2, 0xB333},
    {0x2D3, 0x6000},
    {0x2D8, 0x0007},
    {0x2D9, 0x0007},
    {0x2DA, 0xB333},
    {0x2DB, 0x6000},
    {0x320, 0x3FEE},
    {0x321, 0x3935},
    {0x322, 0x2D77},
    {0x323, 0x23F9},
    {0x324, 0x203B},
    {0x325, 0x1CBC},
    {0x326, 0x197D},
    {0x327, 0x16BF},
    {0x328, 0x1400},
    {0x329, 0x1200},
    {0x32A, 0x1000},
    {0x32B, 0x0E41},
    {0x32C, 0x0CC3},
    {0x32D, 0x0B44},
    {0x32E, 0x0A05},
    {0x32F, 0x0906},
    {0x330, 0x0707},
    {0x331, 0x0589},
    {0x332, 0x048B},
    {0x333, 0x000C},
    {0x338, 0x0201},
    {0x339, 0x120A},
    {0x33A, 0x221A},
    {0x33B, 0x2423},
    {0x340, 0x021F},
    {0x341, 0x0403},
    {0x342, 0x2005},
    {0x343, 0x2221},
    {0x344, 0x2423},
    {0x345, 0x4025},
    {0x346, 0x4241},
    {0x347, 0x4443},
    {0x348, 0x6160},
    {0x349, 0x6362},
    {0x34A, 0x6564},
    {0x34B, 0x8281},
    {0x34C, 0x8483},
    {0x34D, 0xA185},
    {0x34E, 0xA3A2},
    {0x34F, 0xC0A4},
    {0x350, 0xC2C1},
    {0x351, 0xC4C3},
    {0x352, 0xE0C5},
    {0x353, 0xE2E1},
    {0x354, 0xE4E3},
    {0x355, 0xE6E5},
    {0x356, 0xE8E7},
    {0x357, 0xEAE9},
    {0x358, 0xECEB},
    {0x359, 0xEEED},
    {0x35A, 0xF0EF},
    {0x35B, 0xF2F1},
    {0x35C, 0xF4F3},
    {0x35D, 0xF6F5},
    {0x35E, 0xF8F7},
    {0x35F, 0xF8F7},
    {0x360, 0x0100},
    {0x361, 0x0302},
    {0x362, 0x0504},
    {0x363, 0x2120},
    {0x364, 0x2322},
    {0x365, 0x2524},
    {0x366, 0x4140},
    {0x367, 0x4342},
    {0x368, 0x4544},
    {0x369, 0x6160},
    {0x36A, 0x6362},
    {0x36B, 0x6564},
    {0x36C, 0x8180},
    {0x36D, 0x8382},
    {0x36E, 0x8584},
    {0x36F, 0x8986},
    {0x3B0, 0x0068},
    {0x3B1, 0x0069},
    {0x3B2, 0x0088},
    {0x3B3, 0x0089},
    {0x3B4, 0x00a7},
    {0x3B5, 0x00a8},
    {0x3B6, 0x00a9},
    {0x3B7, 0x00c7},
    {0x3B8, 0x00c8},
    {0x3B9, 0x00c9},
    {0x3BA, 0x00e7},
    {0x3BB, 0x00e8},
    {0x3BC, 0x00e9},
    {0x3BD, 0x0107},
    {0x3BE, 0x0108},
    {0x3BF, 0x0109},
    {0x3C0, 0x0007},
    {0x3C1, 0x0007},
    {0x3C2, 0x0007},
    {0x3C3, 0x0007},
    {0x3C4, 0x0007},
    {0x3C5, 0x0007},
    {0x3C6, 0x0007},
    {0x3C7, 0x0008},
    {0x3C8, 0x0009},
    {0x3C9, 0x0027},
    {0x3CA, 0x0028},
    {0x3CB, 0x0029},
    {0x3CC, 0x0047},
    {0x3CD, 0x0048},
    {0x3CE, 0x0049},
    {0x3CF, 0x0067},
    {0x3D1, 0x2101},
    {0x3D2, 0x2B03},
    {0x3D3, 0x2D05},
    {0x3D4, 0x2D0D},
    {0x3D9, 0x0110},
    {0x3DA, 0x0B30},
    {0x3DB, 0x0D50},
    {0x3DC, 0x0DD0},
    {0x3DE, 0x0FF0},
    {0x3DF, 0x2F0F}
};

static void cal_wbpll(uint8_t band, int idx)
{
    uint16_t reg = 0x0B3;
    uint16_t reg_offs = idx ? 0x200 : 0;
    uint16_t val0 = band == PHY_BAND_5G ? 0 : 0x0C00;

    reg = band == PHY_BAND_5G ? 0x0B3 : 0x0BB;

    spi_write(0x051 | reg_offs, 0x0000); // Always start cal from zero XXX .xls says start from 6
    spi_write(reg | reg_offs, 0x110F + val0);  // div16 enable & select free running mode (disable WBPLL)
    spi_write(reg | reg_offs, 0x130F + val0);  // Start WBPLL
    spi_write(0x007, 0x0040);                  // set freq counter ref (gives 10MHz resolution)
    spi_write(0x009, 0x11D1 | (idx ? 4 : 0)); // Set WBPLL target frequency & Start Calibration
    spi_poll(0x01B, CO_BIT(14), CO_BIT(14)); // Interrupt output (WBPLL tuning done=bit14) XXX Datasheet says bit15
    spi_write(reg | reg_offs, 0x000F + val0);  // Enable WBPLL (disable free running mode)
}

static void cal_synth(/*int chan 0 or 1*/)
{
    spi_write(0x083, 0x1411); // Set Level Detector
    spi_write(0x053, 0x02C6); // Set threshold, delay time & coarse tune mode
    spi_write(0x056, 0x0005); // Start Coarse cap search
    spi_poll(0x01A, CO_BIT(8), CO_BIT(8)); // Interrupt output Synth1 (check for bit8 is cal done)
    spi_write(0x056, 0x0003); // Start Coarse I search
    spi_write(0x053, 0x02C7); // Set threshold, delay time & fine tune mode
    spi_poll(0x01A, CO_BIT(8), CO_BIT(8)); // Interrupt output Synth1 (check for bit8 is cal done)
    spi_write(0x056, 0x0005); // Start fine cap search
    spi_poll(0x01A, CO_BIT(8), CO_BIT(8)); // Interrupt output Synth1 (check for bit8 is cal done)
    spi_write(0x083, 0x0411); // deSet Level Detector
    spi_poll(0x01A, CO_BIT(1), 0); // Interrupt output Synth1 (check bit1 Synth unlocked should be 0)
}

static void cal_r_rc(void)
{
    uint16_t spi_val;

    spi_write(0x200, 0x02B2); // 1 Enable Aux ADC 0x200 0x02B2
    spi_write(0x215, 0x8688); // 2 Enable R & RC cal
    spi_write(0x216, 0x002D); // 3 Set target for RC (0x0012 for Nominal bandwidth)
    spi_write(0x217, 0xF041); // 4 Set ATEST MUX routing to do R calibration by RSBUS
    spi_write(0x219, 0x2040); // 5 Idem
    spi_write(0x21A, 0x3000); // 6 Idem
    /* RC cal after R cal not in // != datasheet */
    spi_write(0x200, 0x02B3); // 8 Trigger Aux ADC - 12bit conversion
    spi_poll(0x01A, CO_BIT(14), CO_BIT(14));    // 9 Wait for Aux ADC
    /* We could start RC cal before polling for Aux conversion but then (ATM) we
     * could miss bit 15 while polling 0x01A for RC tune since reg 1A bits are
     * read-cleared - should be at no cost since we are performing multiple spi
     * accesses before polling for RC tune */
    spi_write(0x006, 0x0A11); // 7 Start RC cal - Start tuning of TRX1 RX IF filter
    spi_val = __spi_read(0x201);                // 10 Read Aux ADC
    // 11 Limit to pos, 12 Limit to below 31, 13 Left shift by 4 steps
    spi_val = (spi_val & CO_BIT(14)) ? 0 : (spi_val << 4) & 0xF800;
    dbg(D_ERR D_PHY "%s: R cal value 0x%02x\n", __func__, spi_val >> 11);

    spi_write(0x0D3, spi_val); // 14 Write R-calibration value to TX1 5G9
    spi_write(0x0B1, spi_val | (__spi_read(0x0B1) & 0x0700) | 0x0077); // 15 Write R-calibration value to RX1 5G9
    spi_write(0x0DB, spi_val); // 16 Write R-calibration value to TX1 2G4
    spi_write(0x0B9, spi_val | (__spi_read(0x0B9) & 0x0700) | 0x0077); // 17 Write R-calibration value to RX1 2G4
    spi_write(0x2D3, spi_val); // 14 Write R-calibration value to TX1 5G9
    spi_write(0x2B1, spi_val | (__spi_read(0x2B1) & 0x0700) | 0x0077); // 15 Write R-calibration value to RX1 5G9
    spi_write(0x2DB, spi_val); // 16 Write R-calibration value to TX1 2G4
    spi_write(0x2B9, spi_val | (__spi_read(0x2B9) & 0x0700) | 0x0077); // 17 Write R-calibration value to RX1 2G4

    spi_write(0x219, 0x0000); // 18 Clear ATEST MUX
    spi_write(0x21A, 0x0000); // 19 Idem
    spi_write(0x217, 0x0000); // 20 Idem
    spi_poll(0x01A, CO_BIT(15), CO_BIT(15)); // 21 Wait for RC tune

    spi_val = __spi_read(0x0A1); // 22 Read result from TRX1 RX IF filter
    spi_write(0x0C1, spi_val);   // 22 Copy to TRX1 TX IF filter if same target BW
    spi_write(0x2A1, spi_val);
    spi_write(0x2C1, spi_val);
    spi_write(0x215, 0x8088);    // 23 Disable R & RC calibration blocks
    spi_write(0x200, 0x02B0);    // 24 Disable AuxADC
    dbg(D_ERR D_PHY "%s: 0x0A1 = 0x%04x\n", __func__, spi_val);
}

static inline void karst_sw_control_mode(void)
{
    spi_write(0x001, 0x0038);
}

static inline void karst_hw_control_mode(void)
{
    spi_write(0x001, 0x00BB);
}

static void karst_set_channel(uint8_t band, uint16_t freq1, uint8_t chantype)
{
    uint64_t div, int_frac;
    uint8_t pathmux_val;

    #ifdef CFG_VIRTEX7
    if (crm_fe_160m())
    {
        // set DCO delay adc0
        adc_wr(1, 0x17, 0x8f);
        // set DCO delay adc1
        adc_wr(2, 0x17, 0x8f);
        adc_wr(3, 0xff, 0x1);

        // DCO invert
        adc_wr(3, 0x16, 0x00);
        adc_wr(3, 0xff, 0x1);
    }
    else
    {
        // set DCO delay adc0
        adc_wr(1, 0x17, 0x80);
        // set DCO delay adc1
        adc_wr(2, 0x17, 0x9f);
        adc_wr(3, 0xff, 0x1);

        // DCO invert
        adc_wr(3, 0x16, 0x80);
        adc_wr(3, 0xff, 0x1);
    }
    #endif

    /**
     * Device Control Guide:
     *      5   GHz band Ndiv = RFfrequency*2/3/40
     *      2.4 GHz band Ndiv = RFfrequency*2*4/5/40 (not 2/4/5/40 ??)
     * matches datasheet PLL block diagram but conflicts with .xls Registers Defs: 5880 / (5/4) / 40
     */
    div = band == PHY_BAND_5G ? 60 : 25;
    int_frac = (((uint64_t)freq1 << 24) + (div - 1)) / div;

    // Set always PATHMUX to 0 when doing calibration
    pathmux_val = karst_fb_pathmux_sel_getf();
    karst_fb_pathmux_sel_setf(0);

    // for calibration
    karst_rf_force_set( (karst_rf_force_get() & 0xfcfffcff) | 0x03000300);

    cal_wbpll(band, 0);
    cal_wbpll(band, 1);

    spi_write(0x089, int_frac >> 16);
    spi_write(0x088, int_frac & 0xFFFF);

    if (phy_env.chan.band != band || phy_env.chan.type != chantype)
    {
        struct phy_karst_cfg_tag *cfg = &phy_env.cfg;
        int bw_mode;

        if (band == PHY_BAND_5G)
        {
            karst_iqcomp_adc0_set(cfg->rx_iq_comp_5G[0]);
            karst_iqcomp_adc1_set(cfg->rx_iq_comp_5G[1]);
            karst_iqcomp_dac0_set(cfg->tx_iq_comp_5G[0]);
            karst_iqcomp_dac1_set(cfg->tx_iq_comp_5G[1]);
        }
        else
        {
            karst_iqcomp_adc0_set(cfg->rx_iq_comp_2_4G[0]);
            karst_iqcomp_adc1_set(cfg->rx_iq_comp_2_4G[1]);
            karst_iqcomp_dac0_set(cfg->tx_iq_comp_2_4G[0]);
            karst_iqcomp_dac1_set(cfg->tx_iq_comp_2_4G[1]);
        }

        if (chantype == PHY_CHNL_BW_20)
            bw_mode = 0;
        else if (chantype == PHY_CHNL_BW_40)
            bw_mode = 1;
        else
            bw_mode = 2;

        spi_write(0x01D1, 0x0100 | ( 1 << (band == PHY_BAND_5G ? 0 : 4)) | (bw_mode << 13));
        spi_write(0x01D2, 0x0B00 | ( 3 << (band == PHY_BAND_5G ? 0 : 4)) | (bw_mode << 13));
        spi_write(0x01D3, 0x0D00 | ( 5 << (band == PHY_BAND_5G ? 0 : 4)) | (bw_mode << 13));
        spi_write(0x01D4, 0x0D00 | (13 << (band == PHY_BAND_5G ? 0 : 4)) | (bw_mode << 13));
        spi_write(0x03D1, 0x0100 | ( 1 << (band == PHY_BAND_5G ? 0 : 4)) | (bw_mode << 13));
        spi_write(0x03D2, 0x0B00 | ( 3 << (band == PHY_BAND_5G ? 0 : 4)) | (bw_mode << 13));
        spi_write(0x03D3, 0x0D00 | ( 5 << (band == PHY_BAND_5G ? 0 : 4)) | (bw_mode << 13));
        spi_write(0x03D4, 0x0D00 | (13 << (band == PHY_BAND_5G ? 0 : 4)) | (bw_mode << 13));
    }

    cal_synth();

    // Force trx switch
    // Note: DCOC calibration should be done with switch in TX position but DC
    //       is not compensated in 5G, works better with RX position,
    //       still under investigation !
    karst_rf_force_set( (karst_rf_force_get()&0xfcfffcff) | 0x03000000);

    // RX DCOC
    // device control guide, script and .xls are all different
    spi_write(0x050, 0x0002); // requires spi_write(0x001, 0x0038);
    spi_poll(0x01A, CO_BIT(0), 0);
    spi_write(0x062, 0x0100);
    spi_write(0x058, 0x8385);
    spi_poll(0x01A, CO_BIT(6) | CO_BIT(0), CO_BIT(6));
    spi_write(0x062, 0x0000);

    spi_write(0x250, 0x0002); // requires spi_write(0x001, 0x0038);
    spi_poll(0x01B, CO_BIT(0), 0);
    spi_write(0x262, 0x0100);
    spi_write(0x258, 0x8385);
    spi_poll(0x01B, CO_BIT(6) | CO_BIT(0), CO_BIT(6));
    spi_write(0x262, 0x0000);

    spi_write(0x050, 0x0001); // requires spi_write(0x001, 0x0038);
    spi_write(0x250, 0x0001); // requires spi_write(0x001, 0x0038);

    // Release Force TRX switch
    karst_rf_force_set(karst_rf_force_get() & 0xfcfffcff);

    // Set previous value of PATHMUX
    karst_fb_pathmux_sel_setf(pathmux_val);
}

static void riu_pssel_set(uint16_t freq, uint16_t freq1, uint8_t chantype)
{
    unsigned int pssel;

    /**
     * PSSEL - use pssel of RW-WSDM-RIU-HW-REG.xls
     */
    if (chantype == PHY_CHNL_BW_40)
    {
        // pssel = 1 or 2
        pssel = freq < freq1 ? 1 : 2;
    }
    else if (chantype == PHY_CHNL_BW_80)
    {
        // pssel = 0, 1, 2 or 3
        int _offs = freq1 - freq;
        if (_offs > 0)
            pssel = _offs > 10 ? 0 : 1;
        else
            pssel = -_offs > 10 ? 3 : 2;
    }
    else
    {
        pssel = 0;
    }

    riu_psselect_setf(pssel);
}

static void mdm_primary_set(uint16_t freq, uint16_t freq1, uint8_t chantype)
{
    #if NX_MDM_VER >= 30
    unsigned int primary;

    if (chantype == PHY_CHNL_BW_40)
    {
        primary = freq < freq1 ? 0 : 1;
    }
    else if (chantype == PHY_CHNL_BW_80)
    {
        primary = 0; // TBD
    }
    else
    {
        primary = 0;
    }

    mdm_primaryind_set(primary);
    #endif
}

static void mdm_set_channel(uint8_t band, uint16_t freq, uint16_t freq1, uint8_t chantype,
                            uint8_t flags)
{
    // Reset the MDM/RIU before configuration
    crm_mdm_reset();

    /*
     *************************************************************************************
     * Band-dependent (2.4G, 5G) settings
     *************************************************************************************
     */
    // By default we consider we won't be on a DFS channel
    #if NX_RADAR_DETECT
    riu_radardeten_setf(0); // Disable radar detection
    crm_radartimclkforce_setf(0); // Disable radar timer clock
    riu_irqmacradardeten_setf(0); // disable radar detection interrupt
    #endif
    #if NX_MAC_HE
    nxmac_disable_tbru_26_resp_setf(0); // Allow RU26
    #endif
    if(band == PHY_BAND_5G)
    {
        rc_rf_sel_2g4_setf(0);
        nxmac_abgn_mode_setf(MODE_802_11AC_5);
        mdm_rxdsssen_setf(0);
        riu_ofdmonly_setf(1);   // AGC detection OFDM only
        #if NX_RADAR_DETECT || NX_MAC_HE
        if (flags & CHAN_RADAR)
        {
            #if NX_RADAR_DETECT
            riu_radardeten_setf(1); // Enable radar detection
            crm_radartimclkforce_setf(1); // Enable radar timer clock
            riu_irqmacradardeten_setf(1); // Enable radar detection interrupt
            #endif
            #if NX_MAC_HE
            nxmac_disable_tbru_26_resp_setf(1); // No RU26 when tuned to a DFS channel
            #endif
        }
        #endif
        riu_rwnxagcevt3_set(0);     /* write 0 to disable DSSS detection correlator */
    }
    else
    {
        rc_rf_sel_2g4_setf(1);
        nxmac_abgn_mode_setf(MODE_802_11N_2_4);
        mdm_rxdsssen_setf(1);
        riu_ofdmonly_setf(0);   // AGC detection OFDM and DSSS
        riu_rwnxagcevt3_set(RIU_RWNXAGCEVT3_RESET);/* write default to enable DSSS detection correlator */
    }

    /*
     *************************************************************************************
     * Frequency-dependent settings
     *************************************************************************************
     */
    #if NX_MDM_VER >= 30
    // For symbol clock error compensation in HE TB
    mdm_invcarrierfreq_setf((1 << 26) / freq1);
    #endif

    /*
     *************************************************************************************
     * Bandwidth-dependent settings
     *************************************************************************************
     */
    // Set PSSEL
    riu_pssel_set(freq, freq1, chantype);
    // Set Primary Channel in MDM
    mdm_primary_set(freq, freq1, chantype);
    // Configure maximum BW
    mdm_txcbwmax_setf(chantype);
    nxmac_max_supported_bw_setf(chantype);
    // Put default values
    mdm_fdoctrl0_set(MDM_FDOCTRL0_RESET);
    #if NX_MDM_VER < 30
    mdm_tbectrl2_set(0x00007F05);
    #else
    mdm_conf_bw_setf(chantype);
    mdm_rxcbwmax_setf(chantype);
    // To be uncommented when NX_MDM_VER=30 supports 80 MHz
    //mdm_tbectrl2_set(MDM_TBECTRL2_RESET);
    #endif
    #if NX_MDM_VER >= 22
    mdm_smoothforcectrl_set(0);
    #else
    mdm_smoothctrl_set(0x01880C06);
    #endif

    //  3us for TX RAMPUP <=> TXRFRAMPUP = 360
    if (chantype == PHY_CHNL_BW_20)
    {
        #if NX_MDM_VER < 30
        mdm_txstartdelay_setf(180);
        mdm_txctrl1_pack(0, 0, 28, 48);
        // TBE for 60MHz
        mdm_tbe_count_adjust_20_setf(0);
        mdm_txctrl3_pack(720, 1080);
        mdm_dcestimctrl_pack(0, 0, 0, 15, 15);
        // For FPGA, divide value by 2 due to timing constraints
        mdm_waithtstf_setf(7);
        #else
        mdm_txstartdelay_setf(384);
        mdm_txctrl1_pack(64, 96);
        // TBE for 120MHz
        mdm_tbe_count_adjust_20_setf(4);
        mdm_dcestimctrl_pack(0, 2, 10, 15);
        // For FPGA, divide value by 2 due to timing constraints
        mdm_waithtstf_setf(7);
        mdm_tdsyncoff20_setf(25);
        #endif

        #if NX_MDM_VER < 30
        mdm_tddchtstfmargin_setf(6);
        #else
        mdm_tddchtstfmargin_setf(1);
        #endif

        // No ACI margin in BW=20MHz due to latency on HTSIG decoding
        riu_rwnxagcaci20marg0_set(0);
        riu_rwnxagcaci20marg1_set(0);
        riu_rwnxagcaci20marg2_set(0);

        #if NX_MDM_VER >= 30
        // Increase DC convergence due to Karst RF performance
        riu_dccenteredholdtime50ns_setf(15);
        #endif
    }
    else
    {

        #if NX_MDM_VER < 30
        // TBE for 120MHz
        mdm_tbe_count_adjust_20_setf(2);
        mdm_txstartdelay_setf(360);
        mdm_txctrl3_pack(1440, 2160);
        #endif

        if (chantype == PHY_CHNL_BW_40)
        {
            #if NX_MDM_VER < 30
            mdm_txctrl1_pack(0, 39, 82, 96);
            mdm_rxtdctrl0_pack(18, 64, 252, 13);
            mdm_dcestimctrl_pack(0, 0, 8, 30, 31);
            // For FPGA, divide value by 2 due to timing constraints
            mdm_waithtstf_setf(15);
            #else
            mdm_txstartdelay_setf(384);
            mdm_txctrl1_pack(64, 96);
            mdm_dcestimctrl_pack(0, 3, 16, 31);
            // For FPGA, divide value by 2 due to timing constraints
            mdm_waithtstf_setf(15);
            mdm_tbe_count_adjust_20_setf(4);
            mdm_tdsyncoff20_setf(24);
            #endif

            #if NX_MDM_VER < 30
            mdm_tddchtstfmargin_setf(6);
            #else
            mdm_tddchtstfmargin_setf(1);
            #endif
        }
        else // chantype == PHY_CHNL_BW_80
        {
            #if NX_MDM_VER < 30
            mdm_txctrl1_pack(22, 60, 105, 120);
            mdm_rxtdctrl0_pack(18, 64, 247, 23);
            mdm_dcestimctrl_pack(0, 0, 38, 43, 63);
            // For FPGA, divide value by 2 due to timing constraints
            mdm_waithtstf_setf(31);
            #endif

            mdm_tddchtstfmargin_setf(14);

            #if NX_MDM_VER >= 22 && NX_MDM_VER < 30
            mdm_cfgvhtsts2smoothforce_setf(1);
            mdm_cfgvhtsts2smooth_setf(2);
            #else
            mdm_smoothctrl_set(0x018E0C06);
            #endif
            mdm_tbectrl2_set(0x00007F03);
        }

        // Set back default ACI margin
        riu_rwnxagcaci20marg0_set(RIU_RWNXAGCACI20MARG0_RESET);
        riu_rwnxagcaci20marg1_set(RIU_RWNXAGCACI20MARG1_RESET);
        riu_rwnxagcaci20marg2_set(RIU_RWNXAGCACI20MARG2_RESET);

        #if NX_MDM_VER >= 30
        // Set back default DC parameters for BW > 20MHz
        riu_dccenteredholdtime50ns_setf(RIU_DCCENTEREDHOLDTIME50NS_RST);
        #endif

    }

    #if NX_MDM_VER >= 21
    /* Reset RX IQ compensation if available */
    if (riu_iqcomp_getf())
    {
        riu_iqestiterclr_setf(1);
    }
    #endif /* NX_MDM_VER >= 21 */
}

static void phy_hw_set_channel(uint8_t band, uint16_t freq, uint16_t freq1,
                               uint8_t chantype, uint8_t flags, uint8_t index)
{
    dbg(D_INF D_PHY "%s: band=%d freq=%d freq1=%d chantype=%d sx=%d\n",__func__,band,freq,freq1,chantype,index);

    // Put the RF in manual (i.e. SW) control mode
    karst_sw_control_mode();

    /*
     *************************************************************************************
     * Clock configuration
     *************************************************************************************
     */
    crm_clk_set(chantype);

    /*
     *************************************************************************************
     * MODEM/RIU configuration
     *************************************************************************************
     */
    mdm_set_channel(band, freq, freq1, chantype, flags);

    /*
     *************************************************************************************
     * RF/RF board configuration
     *************************************************************************************
     */
    karst_set_channel(band, freq1, chantype);

    // Put back the RF in automatic (i.e. HW) control mode
    karst_hw_control_mode();
}

static void karst_init(const struct phy_karst_cfg_tag *cfg)
{
    unsigned int i;

    /*
     *************************************************************************************
     * DAC configuration
     *************************************************************************************
     */
    /* AD9780: DAC reset */
    karst_dac_spi_cntl_set(KARST_DAC_RESET_BIT);
    udelay(1000);
    karst_dac_spi_cntl_set(0);

    /*  DAC clock buffer off */
    karst_dac_spi_addr_set(0x80030003);
    karst_dac_spi_data_set(0x000000b4);
    karst_dac_spi_cntl_set(0x201);
    reg_poll(KARST_DAC_SPI_CNTL_ADDR, KARST_DAC_SPI_DONE_BIT);
    karst_dac_spi_cntl_set(0);

    /* Prepare init for autopower */
    karst_dac_spi_autopower_set(0x03000384);  // poweron (all on) /poweroff (all on except input clock buffer, output clock buffer)
    karst_dac_spi_addr_set(0x80030003);  // write enable
    karst_dac_spi_cntl_set(0x202);       // prescaler 2 / autopower enabled

    /*
     *************************************************************************************
     * ADC configuration
     *************************************************************************************
     */
    /* Reset ADC */
    adc_wr(3, 0x0, 0x3c);
    udelay(10000);

    /* AD9630: fix  fullscale voltage to 2.087 Vpp */
    adc_wr(0x03, 0x18, 0x0f);
    adc_wr(0x03, 0xFF, 0x01);

    /* FPGA B: fix ADC sampling window */
    karst_iodelay_adc0_set(IODELAY_ADC0);
    karst_iodelay_adc1_set(IODELAY_ADC1);
    karst_iodelay_cntl_set(0x00);
    karst_iodelay_start_setf(0x01);
    reg_poll(KARST_IODELAY_CNTL_ADDR, KARST_IODELAY_DONE_BIT);
    karst_iodelay_start_setf(0x00);

    /*
     *************************************************************************************
     * RF configuration
     *************************************************************************************
     */
    karst_rf_force_set(0x00010000);
    karst_rf_force_set(0x00010001);

    if (spi_write(0x001, 0x0004)) // sw_reset - 1ms
        return;

    udelay(1000);

    rev = __spi_read(0x0000);
    dbg(D_ERR D_PHY "%s: Chip ID 0x%04x\n", __func__, rev);
    ASSERT_ERR(rev != 0);

    for (i = 0; i < CO_ARRAY_SIZE(rv_defaults); i++) {
        if (spi_write(rv_defaults[i].reg, rv_defaults[i].val))
        {
            dbg(D_ERR D_PHY "%s: Could not load regs defaults (step %d)\n", __func__, i);
            return;
        }
    }

    spi_write(0x212, 0x6868); // XO trim cap values (manual calibration of 40MHz)
    spi_write(0x0C0, 0x0303); // Enable TX Filter workaround - from script

    // 2.4. Turn on Regulators internal clocks
    spi_write(0x215, 0x8088); // Auxiliary blocks
    spi_write(0x080, 0x0F0F); // Synthesizer ctrl0
    // XXX Can we toggle it when switching channel ? (PHY_BAND_5G vs PHY_BAND_2G4)
    spi_write(0x0BA, 0x7000); // 2.4G
    spi_write(0x0B2, 0x7000); // 5G
    spi_write(0x0A2, 0xA300); // Baseband RX
    spi_write(0x0C2, 0x8000); // Baseband TX
    spi_write(0x2BA, 0x7000); // 2.4G
    spi_write(0x2B2, 0x7000); // 5G
    spi_write(0x2A2, 0xA300); // Baseband RX
    spi_write(0x2C2, 0x8000); // Baseband TX
    udelay(1000);

    spi_write(0x211, 0x0008); // Enable clocks to PLL
    udelay(1000);
    cal_r_rc();
    spi_write(0x050, 0x0001); // TRX mode = 1 = LO 5G9 ch0 mode
    spi_write(0x250, 0x0001); // TRX mode = 1 = LO 5G9 ch0 mode
    spi_write(0x0A0, 0x003F); // Disable 5-bit ADCs in TRX1
    spi_write(0x211, 0x0008); // Disable clocks for 5-bit ADCs

    spi_write(0x063, PHY_KARST_MAX_PWR); // By default fix the TX gain to the maximum
    spi_write(0x263, PHY_KARST_MAX_PWR); // By default fix the TX gain to the maximum

    // Mode ctrl with Parallel GPIO with TRX2 Mode slaved to TRX1 Mode
    spi_write(0x001, 0x00BB);
    rc_sw_ctrl_set(REG_DFLT_SW_CTRL);

    #if NX_MDM_VER >= 30
    // Enable automatic TX gain
    rc_auto_tx_gain_en_setf(1);
    // Set the TX gain RF register address - Note that the current version of the RC
    // supports automatic gain control only in 1x1 configuration because only one TX gain
    // register is written (so for one path only) when the power is changed
    rc_hw_tx_gain_spi_addr_set((cfg->path_used << 9) | 0x063);
    #endif

    karst_diagport_conf1_set(0x09000700);
    karst_diagport_conf2_set(0x0);

    karst_fb_pathmux_set(cfg->path_used);
}

static void mdm_init(const struct phy_karst_cfg_tag *cfg)
{
    // Check if we are compiled for this version of the PHY
    ASSERT_ERR((mdm_major_version_getf() + 2) * 10 + mdm_minor_version_getf()
                                                                 == NX_MDM_VER);

    // Reset the MDM/RIU before configuration
    crm_mdm_reset();

    /*
     *************************************************************************************
     * MODEM configuration
     *************************************************************************************
     */
    // Supported features
    mdm_rxmode_set(MDM_RXMMEN_BIT | MDM_RXDSSSEN_BIT);
    mdm_rxnssmax_setf(mdm_nss_getf() - 1);
    mdm_rxndpnstsmax_setf(mdm_nsts_getf() - 1);
    mdm_rxldpcen_setf(mdm_ldpcdec_getf());
    mdm_rxvhten_setf(phy_vht_supported());
    mdm_rxstbcen_setf(1);
    #if NX_MDM_VER < 30
    mdm_rxgfen_setf(1);
    mdm_rxmumimoen_setf(mdm_mumimorx_getf());
    mdm_rxmumimoapeplenen_setf(mdm_mumimorx_getf());
    #else
    mdm_rxdcmen_setf(mdm_he_getf());
    mdm_rxheen_setf(mdm_he_getf());
    mdm_rxvhtmumimoen_setf(mdm_vht_getf() & mdm_mumimorx_getf());
    mdm_rxhemumimoen_setf(mdm_he_getf() & mdm_mumimorx_getf());
    #endif

    // Set DSSS precomp
    mdm_precomp_setf(45);

    #if NX_MDM_VER == 20
    #if RW_NX_LDPC_DEC
    // Set LDPC table selection for FPGA limitation
    mdm_ldpcdectablesel_setf(2);
    #endif
    #endif

    // Allow GF/SGI/STBC (bit14 reset) - TEMPORARY!!!!
    mdm_rxframeviolationmask_setf(0xFFFFBFFF);

    mdm_txmode_set(MDM_TXSTBCEN_BIT | MDM_TXGFEN_BIT  |
                   MDM_TXMMEN_BIT   | MDM_TXDSSSEN_BIT);
    mdm_txnssmax_setf(mdm_nss_getf() - 1);
    mdm_ntxmax_setf(mdm_ntx_getf() - 1);
    mdm_txcbwmax_setf(mdm_chbw_getf());
    mdm_txldpcen_setf(mdm_ldpcenc_getf());
    mdm_txvhten_setf(phy_vht_supported());
    #if NX_MDM_VER >= 30
    mdm_txheen_setf(phy_he_supported());
    mdm_tdfocpeslopeen_setf(1);
    #endif
    mdm_txmumimoen_setf(mdm_mumimotx_getf());

    #if NX_MDM_VER < 30
    /* AGC reset mode
     don't turn off RF if rxreq de-asserted for few cycles after a RXERR */
    mdm_rxtdctrl1_set(mdm_rxtdctrl1_get()|1);
    #endif

    /* Enable automatic smoothing filter selection from SNR, then disable force */
    #if NX_MDM_VER < 22
    mdm_cfgsmoothforce_setf(0);
    #endif

    #if NX_MDM_VER < 30
    if (mdm_nss_getf() == 1)
    {
        /* limit NDBPSMAX to 1x1 80 MCS7 LGI(292.5Mb/s) / SGI (325.0Mb/s) */
        mdm_rxctrl1_set(0x04920492);
    }
    else
    {
        #if defined(CFG_VIRTEX6)
        /* limit NDBPSMAX to 2x2 80 MCS4 LGI(351Mb/s) / SGI (390.0Mb/s) */
        mdm_rxctrl1_set(0x057C057C);
        #elif defined(CFG_VIRTEX7)
        /* No limitation on VIRTEX7 platform */
        mdm_rxctrl1_set(0x0C300C30);
        #endif
    }
    #else
    mdm_txtdsfoctrl_set(0x10000000);
    mdm_txtdcfoctrl_set(0x10000000);
    mdm_rxctrl5_set(0x00670780);
    mdm_tdfoctrl0_set(0x00340100);
    #endif

    /*
     *************************************************************************************
     * RIU configuration
     *************************************************************************************
     */
    /* Enable RC clock */
    crm_rcclkforce_setf(1);

    /* RC delays */
    rc_hw_tx_delay_set(0xa0);

    #if NX_MDM_VER >= 21
    /* Enable RX IQ compensation if available */
    if (riu_iqcomp_getf())
    {
        riu_rxiqphaseesten_setf(1);
        riu_rxiqgainesten_setf(1);
        riu_rxiqphasecompen_setf(1);
        riu_rxiqgaincompen_setf(1);
        riu_iqestiterclr_setf(1);
    }
    #endif /* NX_MDM_VER >= 21 */

    /* limit RIU to 1 or 2 antenna active depending on modem capabilities */
    if (mdm_nss_getf() == 2)
    {
        riu_activeant_setf(3);
        riu_combpathsel_setf(3);
    }
    else
    {
        riu_activeant_setf(1);
        /* limit AGC with a single antenna (path0) */
        riu_combpathsel_setf(1);
    }

    // Tx Digital gain
    #if NX_MDM_VER < 30
    riu_rwnxfectrl0_set(0x001A1A1A);
    if (mdm_ntx_getf() == 2)
        riu_rwnxfectrl1_set(0x001A1A1A);
    #else
    riu_rifsdeten_setf(0);
    riu_rwnxfectrl0_set(0x00404040);
    if (mdm_ntx_getf() == 2)
        riu_rwnxfectrl1_set(0x00404040);

    /* disable the riu dccentered compensation */
    riu_dccenteredtype_setf(0);
    #endif

    riu_crossupthrqdbm_setf(0x200); /* write 0x200 to disable AGC cross-up */
    riu_crossdnthrqdbm_setf(0x200); /* write 0x200 to disable AGC cross-down */

    riu_rwnxagcccatimeout_set(8000000); // 100ms
    riu_irqmacccatimeouten_setf(1);

    /* Enable HW antenna selection */
    riu_rxpathselfromreg_setf(0);

    /*
     *************************************************************************************
     * MACBYPASS configuration
     *************************************************************************************
     */
    #if NX_MDM_VER >= 30
    macbyp_clken_set(1); // enable clock
    #endif
    macbyp_trigger_set(0x00000012);
    macbyp_ctrl_set(0x00000100);

    #if NX_UF_EN
    if (macbyp_version_get() >= RXV_REG_MIN_VERS)
    {
        macbyp_clken_set(1); // enable clock
        macbyp_int3_gen_setf(0x07);// configure interrupt generation
        macbyp_mode_setf(1); //set mode RX
    }
    #endif
}

static void phy_hw_init(const struct phy_karst_cfg_tag *cfg)
{
    /*
     *************************************************************************************
     * MODEM/RIU configuration
     *************************************************************************************
     */
    mdm_init(cfg);

    /*
     *************************************************************************************
     * RF/RF board configuration
     *************************************************************************************
     */
    karst_init(cfg);

    /*
     *************************************************************************************
     * Set a default channel
     *************************************************************************************
     */
    phy_hw_set_channel(PHY_BAND_5G, 5180, 5180, PHY_CHNL_BW_20, 0, 0);
}

void phy_init(const struct phy_cfg_tag *config)
{
    const struct phy_karst_cfg_tag *cfg = (const struct phy_karst_cfg_tag *)&config->parameters;

    phy_hw_init(cfg);

    phy_env.cfg               = *cfg;
    phy_env.chan.band         = PHY_BAND_5G;
    phy_env.chan.type         = PHY_CHNL_BW_OTHER;
    phy_env.chan.prim20_freq  =
    phy_env.chan.center1_freq =
    phy_env.chan.center2_freq = PHY_UNUSED;
}

void phy_mdm_isr(void)
{
    #if NX_UF_EN
    if (macbyp_int3_state_getf())
    {
        //Indicate that an unsupported HT frame has been capture
        uf_event_ind();

        //interrupt 3 ack
        macbyp_int3_ack_clearf(1);
    }
    #endif
}

/**
 * FIXME: This is for debug purpose only and does not call rd_event_ind
 */
void phy_rc_isr(void)
{
    uint32_t irq_status = riu_rwnxmacintstatmasked_get();

    riu_rwnxmacintack_clear(irq_status);

    #if NX_RADAR_DETECT
    if (irq_status & RIU_IRQMACRADARDETMASKED_BIT)
    {
        PROF_RADAR_IRQ_SET();
        rd_event_ind(PHY_PRIM);
        PROF_RADAR_IRQ_CLR();
    }
    #endif

    if (irq_status & RIU_IRQMACCCATIMEOUTMASKED_BIT)
        crm_mdm_reset();

    ASSERT_REC(!(irq_status & RIU_IRQMACCCATIMEOUTMASKED_BIT));
}

void phy_get_version(uint32_t *version_1, uint32_t *version_2)
{
    *version_1 = mdm_hdmconfig_get();
    // TODO Add version reading for other PHY elements than modem.
    *version_2 = mdm_hdmversion_get();
}

void phy_set_channel(const struct mac_chan_op *chan, uint8_t index)
{
    if (index > 0)
    {
        dbg(D_ERR D_PHY "%s: radio %d does not exist\n", __func__, index);
        return;
    }

    if (phy_env.chan.band == chan->band &&
        phy_env.chan.type == chan->type &&
        phy_env.chan.flags == chan->flags &&
        phy_env.chan.prim20_freq == chan->prim20_freq &&
        phy_env.chan.center1_freq == chan->center1_freq &&
        phy_env.chan.center2_freq == chan->center2_freq)
    {
        dbg(D_INF D_PHY "%s: Setting same channel, do nothing\n", __func__);
        return;
    }

    phy_hw_set_channel(chan->band, chan->prim20_freq, chan->center1_freq, chan->type,
                       chan->flags, index);
    phy_env.chan = *chan;
}

void phy_get_channel(struct phy_channel_info *info, uint8_t index)
{
    if (index > 0)
    {
        dbg(D_ERR D_PHY "%s: radio %d does not exist\n", __func__, index);
    }
    // Map the band, channel type, primary channel index on info1
    info->info1 = phy_env.chan.band | (phy_env.chan.type << 8) | (phy_env.chan.prim20_freq << 16);
    // Map center freq on info2
    info->info2 = phy_env.chan.center1_freq | (phy_env.chan.center2_freq << 16);
}

void phy_stop(void)
{
    if (nxmac_current_state_getf() != HW_IDLE)
        dbg(D_ERR "%s MAC state != IDLE\n", __func__);
}

uint32_t phy_get_channel_switch_dur(void)
{
    return 1000;
}

#if NX_RADAR_DETECT
bool phy_has_radar_pulse(int rd_idx)
{

    ASSERT_ERR(rd_idx == PHY_PRIM);

    return (riu_radfifoempty_getf() == 0);
}

bool phy_get_radar_pulse(int rd_idx, struct phy_radar_pulse *pulse)
{
    ASSERT_ERR(rd_idx == PHY_PRIM);

    // Check if FIFO is empty
    if (riu_radfifoempty_getf())
        return (false);

    pulse->pulse = REG_PL_RD(__RIU_RADARFIFO_ADDR);

    return (true);
}
#endif

bool phy_vht_supported(void)
{
    #if NX_MDM_VER > 20
    return ((mdm_vht_getf() != 0) || (mdm_chbw_getf() > PHY_CHNL_BW_40));
    #else
    return true;
    #endif
}

bool phy_he_supported(void)
{
    #if NX_MDM_VER < 30
    return false;
    #else
    return (mdm_he_getf() != 0);
    #endif
}

bool phy_uf_supported(void)
{
    #if NX_UF_EN
    return true ? macbyp_version_get() >= RXV_REG_MIN_VERS : false;
    #else
    return false;
    #endif
}

void phy_uf_enable(bool enable)
{
    #if NX_UF_EN
    // enable/disable macbypass interrupt line 3
    macbyp_interrupt3_en_setf(enable);
    #endif
}

bool phy_ldpc_tx_supported(void)
{
    return (mdm_ldpcenc_getf() != 0);
}

bool phy_ldpc_rx_supported(void)
{
    return (mdm_ldpcdec_getf() != 0);
}

bool phy_bfmee_supported(void)
{
    return (mdm_bfmee_getf() != 0);
}

bool phy_bfmer_supported(void)
{
    return (mdm_bfmer_getf() != 0);
}

bool phy_mu_mimo_rx_supported(void)
{
    return (mdm_mumimorx_getf() != 0);
}

bool phy_mu_mimo_tx_supported(void)
{
    return (mdm_mumimotx_getf() != 0);
}

#if RW_MUMIMO_RX_EN
void phy_set_group_id_info(uint32_t membership_addr, uint32_t userpos_addr)
{
    int i;

    // Set membership status
    for(i=0; i<MDM_MUMIMO_GROUPID_TAB_COUNT; i++)
    {
        mdm_mumimo_groupid_tab_set(i, co_read32p(membership_addr + 4 * i));
    }

    // Set user position
    for(i=0; i<MDM_MUMIMO_USERPOSITION_TAB_COUNT; i++)
    {
        mdm_mumimo_userposition_tab_set(i, co_read32p(userpos_addr + 4 * i));
    }
}
#endif

void phy_set_aid(uint16_t aid)
{
    #if NX_MDM_VER >= 30
    mdm_hestaid_setf(0, aid);
    mdm_hestaid_setf(1, 0);
    mdm_hestaid_setf(2, 2047);
    #endif
}


uint8_t phy_get_bw(void)
{
    return (mdm_chbw_getf());
}

uint8_t phy_get_nss(void)
{
    return (mdm_nss_getf() - 1);
}

uint8_t phy_get_ntx(void)
{
    return (mdm_ntx_getf() - 1);
}

uint8_t phy_get_nrx(void)
{
    return (mdm_nrx_getf() - 1);
}

#if RW_BFMER_EN
uint8_t phy_get_bfr_mem_size(void)
{
    return (mdm_bfmer_mem_size_getf());
}
#endif

void phy_get_rf_gain_idx(int8_t *power, uint8_t *idx)
{
    if (*power > PHY_KARST_MAX_PWR)
    {
        *power = PHY_KARST_MAX_PWR;
    }
    else if (*power < PHY_KARST_MIN_PWR)
    {
        *power = PHY_KARST_MIN_PWR;
    }

    /* idx is simply the power level in dBm */
    *idx = (uint8_t)*power;
}

void phy_get_rf_gain_capab(int8_t *max, int8_t *min)
{
    *max = PHY_KARST_MAX_PWR; // dBm
    *min = PHY_KARST_MIN_PWR; // dBm
}

#if NX_DEBUG_DUMP
void phy_get_diag_state(struct dbg_debug_info_tag *dbg_info)
{
#if NX_MDM_VER < 20
    int i;
    uint8_t mictor1_sel = karst_mictor_1_sel_getf();
    uint8_t diag = karst_phydiag_conf_lsb_1_getf();

    // Get diagnostic port state for the PHY
    karst_mictor_1_sel_setf(0x14);
    for (i = 0; i < DBG_DIAGS_PHY_MAX; i++)
    {
        // Go through the different banks and copies the diag values
        karst_phydiag_conf_lsb_1_setf(i);
        dbg_info->diags_phy[i] = karst_diag1_stat_get() & 0xFFFF;
    }
    karst_phydiag_conf_lsb_1_setf(diag);
    karst_mictor_1_sel_setf(mictor1_sel);
#else
#endif
}
#endif

uint8_t phy_switch_antenna_paths(void)
{
    PROF_ANT_DIV_SWITCH_SET();
    uint8_t value = karst_fb_pathmux_sel_getf();
    if (value == 0)
        value = 1;
    else
        value = 0;
    karst_fb_pathmux_sel_setf(value);
    PROF_ANT_DIV_SWITCH_CLR();

    return value;
}

