/**
 *
 *
 *
 *
 *
 *
 *
 *
 *
 * XXX Hack for MXD new RC.xls/MXD_IF.xls with RF and ADC/DAC configuration done via MXD PC tool
 ****************************************************************************************
 *
 * @file phy_mxd.c
 *
 * @brief File containing the nX MXD-based physical layer driver API.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */


/**
 ****************************************************************************************
 * @addtogroup PHY
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
// for configuration dependent directives
#include "rwnx_config.h"

// main inclusion directive
#include "phy.h"
// for ke_pause
#include "ke_timer.h"
// for dbg
#include "dbg.h"
#include "rd.h"

//maxim2829 register access and defines
#include "phy_mxd.h"

//other registers
#include "reg_mdm_stat.h"
#include "reg_mdm_cfg.h"

#include "reg_rc.h"
#include "reg_agc.h"
#include "reg_mxd_if.h"

#define SYSCTRL_R0_ADDR     0x609000F0
#define REG_DFLT_SW_CTRL    ((1 << RC_PRESCALER_LSB) & (~RC_START_DONE_BIT))

__INLINE uint32_t sysctrl_r0_get(void)
{
    return REG_PL_RD(SYSCTRL_R0_ADDR);
}
__INLINE void sysctrl_r0_set(uint32_t value)
{
    REG_PL_WR(SYSCTRL_R0_ADDR, value);
}
__INLINE uint32_t mxd_dbg(void)
{
    return sysctrl_r0_get() >> 28;
}
__INLINE bool mxd_on(void)
{
    return sysctrl_r0_get() & 1;
}

static uint16_t _rw, _oreg, _oval;

static inline void mdelay(uint32_t ms)
{
    uint32_t e = nxmac_monotonic_counter_2_lo_get() + ms * 1;

    do {
        volatile int n = 1 << 10; // relax
        while (n--)
            ;
    } while ((int32_t)(nxmac_monotonic_counter_2_lo_get() - e) < 0);
}
static int spi_read(uint16_t reg, uint16_t *val)
{
    int loops = 1 << 20;

    rc_spi_ctrl_pack(1, 1, 8, reg);
    while (rc_start_done_getf() && --loops)
        ;
    if (!loops)
    {
        dbg(D_ERR D_PHY "%s @0x%04x failed\n", __func__, reg);
    _rw = 1;
        return -1;
    }
    *val = rc_data_getf();

    if (mxd_dbg() > 7 && (!_rw || _oreg != reg || _oval != *val))
    dbg(D_ERR D_PHY "%s @ 0x%04x 0x%04x\n", __func__, reg, *val);

#if 0
    if (reg == 0xa || reg == 0xb)
        dbg(D_ERR D_PHY "%s @ 0x%04x 0x%04x\n", __func__, reg, *val);
#endif
    _oreg = reg; _oval = *val;

    _rw = 1;
    return 0;
}

static uint16_t __spi_read(uint16_t reg)
{
    uint16_t val = 0;
    return spi_read(reg, &val) ?: val;
}

static uint16_t __spi_read_(uint16_t reg)
{
    uint16_t val = 0;
    return spi_read(reg, &val) ?: val;
}

static int spi_write(uint16_t reg, uint16_t val)
{
    int loops = 1 << 20;
    _rw = 0;

#if 0
    if (reg == 0xa || reg == 0xb)
    dbg(D_ERR D_PHY "%s  @ 0x%04x <- 0x%04x\n", __func__, reg, val);
#endif
    if (mxd_dbg() > 7)
        dbg(D_ERR D_PHY "%s  @ 0x%04x <- 0x%04x\n", __func__, reg, val);

    rc_spi_data_set(val);
    rc_spi_ctrl_pack(1, 0, 8, reg);
    while (rc_start_done_getf() && --loops)
        ;
    if (!loops)
    {
        dbg(D_ERR D_PHY "%s @0x%04x failed\n", __func__, reg);
        return -1;
    }
#if 0
    if (__spi_read(reg) != val)
        dbg(D_ERR D_PHY "%s  0x%04x = 0x%04x != 0x%04x\n", __func__, reg, __spi_read(reg), val);
#endif
    return 0;
}

#define RW_MXD 1

#if RW_MXD
#include "mxd/mxd_acrux2_sdk.c"
#endif /* RW_MXD */


/*
 * STRUCTURES
 ****************************************************************************************
 */

/// Structure containing the parameters of the Maxscend PHY configuration
struct phy_mxd_cfg_tag
{
    /// MDM type(nxm)(upper nibble) and MDM2RF path mapping(lower nibble)
    uint8_t path_mapping;
    /// TX DC offset compensation
    uint32_t tx_dc_off_comp;
};

/*
 * GLOBALS
 ****************************************************************************************
 */
/// Global PHY driver environment.
struct phy_env_tag phy_env;


/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @brief Perform a write access on radio registers through the radio controller.
 * The SPI command is sent only to the radios on the used RC paths (rc_path_sel).
 * To limit the RC register access, the whole RC register value is set in one access
 * except the START_DONE bit which will trigger the SPI transfer and also signal its end
 * when cleared by HW.
 *
 * @param addr   Address of the register to write in.
 * @param value  Value to write.
 *
 * @warning The function does not check the consistency of the parameters provided.
 ****************************************************************************************
 */
static void phy_rc_rf_reg_write(uint8_t addr, uint16_t value)
{
#if 0
    //extract chip select values from Rc used paths
    uint8_t ch0sel = phy_env.rc_path_sel & 0x01;
    uint8_t ch1sel =(phy_env.rc_path_sel & 0x02)>>1;
    uint8_t ch2sel =(phy_env.rc_path_sel & 0x04)>>2;

    //copy all SPI transfer related info into the Rc register, start_done=0
    rc_maxim_spi_pack(0, ch2sel, ch1sel, ch0sel, RF_SPI_PRESCALER, addr, value);

    //trigger spi transfer to radio(s)
    rc_start_done_setf(1);

    //wait until transfer is complete
    while (rc_start_done_getf());
#endif
}

/**
 ****************************************************************************************
 * @brief Perform a write access on 12b ADC registers through the ADC controller.
 *
 * @param addr   Address of the register to write on. 5b
 * @param value  Value to write. 11b
 *
 * @warning The function does not check the consistency of the parameters provided.
 ****************************************************************************************
 */
static void phy_mxd_adc_reg_write(uint8_t addr, uint16_t value, uint8_t spisel)
{
    //value to write
    uint16_t wdata = (((uint16_t)addr)<<11)|value;

#if 0
    //select targeted ADCs - same bit encoding as targeted radios
    mxd_adc_spi_sel_setf(spisel);

    //copy address and value to write into the SPI control register
    mxd_adc_spi_wdata_setf(wdata);

    //trigger spi transfer to ADC(s)-self clearing
    mxd_adc_spi_start_setf(1);

    //wait until transfer is complete
    while(mxd_adc_spi_busy_getf());
#else
    //copy address and value to write into the SPI control register
    mxd_adc_spi_data_setf(wdata);

    //trigger spi transfer to ADC(s)-self clearing
    mxd_adc_spi_start_setf(1);

    //wait until transfer is complete
    while(mxd_adc_spi_busy_getf());
#endif
}

/**
 ****************************************************************************************
 * @brief MXD board IO Expander init.
 * Given it is the components which routes important power down and lock detect signals,
 * it should be initialized first before the components are interacted with.
 ****************************************************************************************
 */
static void phy_iox_init(void)
{
    //enable it
    mxd_ioexp_en_setf(1);
    //poll until ready
    while(!mxd_ioexp_rdy_getf());

    dbg(D_INF D_PHY " 2.IO Expander enabled and ready\n");
}

/**
 ****************************************************************************************
 * @brief MXD board 8b ADCs initialization for RSSI/Temperature/Power read back.
 ****************************************************************************************
 */
//todo init or cfg in case we change ADCs during run for rssi/power?
static void phy_adc8b_init(void)
{
    //do nothing, use reg force when we'll need fcns to gather data
}

/**
 ****************************************************************************************
 * @brief MXD board 12b ADCs calibration.
 ****************************************************************************************
 */
static void phy_adc12b_init(void)
{
    //disable RC auto power mode
    mxd_adc_autopower_setf(0);

    //reset ADCs,
    mxd_adc_reset_setf(1);
    mxd_adc_reset_setf(0);
    // prescaler=6: spi_freq=80/(presc+1) ; max 20MHz
#if 0
    mxd_adc_spi_prescaler_setf(4);
#else
    mxd_prescaler_setf(4);
#endif

    //switch off all ADCs
    phy_mxd_adc_reg_write(0x00, 0x00D, 7);

    //switch on used ADC
    phy_mxd_adc_reg_write(0x00, 0x000, phy_env.rf_path_sel);

    //OTHER REG SETTINGS
    phy_mxd_adc_reg_write(0x04, 0x000, phy_env.rf_path_sel);
    //2's complement
    phy_mxd_adc_reg_write(0x0A, 0x000, phy_env.rf_path_sel);
    phy_mxd_adc_reg_write(0x0B, 0x000, phy_env.rf_path_sel);
    //fine gain 6dB
    phy_mxd_adc_reg_write(0x0C, 0x000, phy_env.rf_path_sel);
    //Override, byte wise, MSB first, coarse gain 3.5dB,12 bits serial,ddr clocking, 2 wire interface
    phy_mxd_adc_reg_write(0x0D, 0x421, phy_env.rf_path_sel); //401 without coarse gain
    //termination, drive clk output
    phy_mxd_adc_reg_write(0x10, 0x000, phy_env.rf_path_sel);
    //bit/byte wise, termination of data outputs
    phy_mxd_adc_reg_write(0x11, 0x000, phy_env.rf_path_sel);

#if 0
    //DESERIALIZER----------------------------
    //Path A
    if (phy_is_bit_set(phy_env.rf_path_sel, RF_PATH_A))
    {
        //deserializer reset
        mxd_deser_cha_rst_n_setf(0);
        //lift reset
        mxd_deser_cha_rst_n_setf(1);

        //deserializer calibration
        mxd_deser_cha_calib_start_setf(1);
        //poll until done
        while(!mxd_deser_cha_calib_done_getf());

        //todo does the start have to be put back to 0 by SW or auto-clear at done?
        mxd_deser_cha_calib_start_setf(0);
    }
    //Path B
    if (phy_is_bit_set(phy_env.rf_path_sel, RF_PATH_B))
    {
        //deserializer reset
        mxd_deser_chb_rst_n_setf(0);
        //lift reset
        mxd_deser_chb_rst_n_setf(1);

        //deserializer calibration
        mxd_deser_chb_calib_start_setf(1);
        //poll until done
        while(!mxd_deser_chb_calib_done_getf());

        //todo does the start have to be put back to 0 by SW or auto-clear at done?
        mxd_deser_chb_calib_start_setf(0);
    }
    //Path C
    if (phy_is_bit_set(phy_env.rf_path_sel, RF_PATH_C))
    {
        //deserializer reset
        mxd_deser_chc_rst_n_setf(0);
        //lift reset
        mxd_deser_chc_rst_n_setf(1);

        //deserializer calibration
        mxd_deser_chc_calib_start_setf(1);
        //poll until done
        while(!mxd_deser_chc_calib_done_getf());

        //todo does the start have to be put back to 0 by SW or auto-clear at done?
        mxd_deser_chc_calib_start_setf(0);
    }
#else
    //DESERIALIZER----------------------------
    //deserializer reset
    mxd_deser_rst_n_setf(0);
    //lift reset
    mxd_deser_rst_n_setf(1);

    //deserializer calibration
    mxd_deser_calib_start_setf(1);
    //poll until done
#if 1 // XXX
    while(!mxd_deser_calib_done_getf());
#endif

    //todo does the start have to be put back to 0 by SW or auto-clear at done?
    mxd_deser_calib_start_setf(0);
#endif

    dbg(D_INF D_PHY " 5.12b ADCs deser calib Ok, RC autopower on\n");
}

/**
 ****************************************************************************************
 * @brief Adjust TX digital gains wrt DC offset compensation to avoid DACs saturation
 ****************************************************************************************
 */
static void adjust_txdiggains(uint32_t dc_cmp)
{
    uint8_t i_abscmp, j_abscmp, max_abscmps;

    i_abscmp = (dc_cmp & MDM_TXIDCOFFSET0_MASK) >> MDM_TXIDCOFFSET0_LSB;
    j_abscmp = (dc_cmp & MDM_TXQDCOFFSET0_MASK) >> MDM_TXQDCOFFSET0_LSB;
    max_abscmps = i_abscmp > j_abscmp ? i_abscmp : j_abscmp;

    // 12bits range
    mdm_fectrl0_pack(MDM_TX80DIGGAINLIN0_RST,
                     MDM_TX40DIGGAINLIN0_RST * (2047 - max_abscmps) / 2047,
                     MDM_TX20DIGGAINLIN0_RST * (2047 - max_abscmps) / 2047);
}

/**
 ****************************************************************************************
 * @brief Modem initialization function.
 * This function is called at reset time together with radio init, and
 * prepares the modem for rx or tx.
 ****************************************************************************************
 */
static void phy_mdm_init(uint32_t tx_dc_off_comp)
{
    //turn on the 3 blocks: TX, AGC-CCA, RX - they are at 1 by default in regs ...
    mdm_rxtxpwrctrl_pack(1,1,1);

    mdm_txctrl0_set(0x00000168);
    mdm_dcoffset0_set(tx_dc_off_comp);
    adjust_txdiggains(tx_dc_off_comp);

    mdm_rxctrl0_set(0x00160005);
    mdm_tbectrl0_set(0x0C0F0702);

    mdm_waithtstf_setf(15);
    mdm_delaynormalgi_setf(17);

    // CPE mode
    mdm_cpemode_setf(0);

    // Enable Radar detection IRQ
    #if NX_RADAR_DETECT
    mdm_irqctrl_set(MDM_IRQRADARDETEN_BIT | MDM_IRQCCATIMEOUTEN_BIT);
    #else
    mdm_irqctrl_set(MDM_IRQCCATIMEOUTEN_BIT);
    #endif

    dbg(D_INF D_PHY " 9.Static MDM settings done");

    //set number of RX paths - //todo can be extrapolated from NVDS info for MDM cfg?
//    mdm_nrxpath_setf(phy_env.nb_rx_paths);
}

/**
 ****************************************************************************************
 * @brief AGC initialization function.
 ****************************************************************************************
 */
static void phy_agc_init(void)
{
#if 0
    #if NX_RADAR_DETECT
    agc_rwnxagcevt0_set(0x05044854);
    agc_rwnxagcevt1_set(0x3d4490bb);
    agc_rwnxagcevt2_set(0x3955900c);
    #else
    agc_rwnxagcevt2_set(0x3955b004);
    #endif
    // AGCCROSS (disable crossing detection)
    agc_rwnxagccross_set(0x002803f0);
    // AGCRAMP (reduce ramp-down detection level)
    agc_rwnxagcramp_set(0x07200710);

    // RWNXAGCCCA1 (CCA{FALL,RISE}THRDBM)
    // for when RWNXAGCCCACTRL[CCAENERGYEN] is on
    agc_rwnxagccca1_set((agc_rwnxagccca1_get() & ~0x000ff0ff) | 0x000bf0c3);

    // RWNXAGCCCACTRL
    agc_rwnxagcccactrl_set((agc_rwnxagcccactrl_get() & ~0x00000fff) | 0x00000377);

    // Configure CCA timeout
    agc_rwnxagcccatimeout_set(8000000); // 100ms
#endif
}

/**
 ****************************************************************************************
 * @brief Set RF band 2.4GHz/5GHz in RC and RF chips.
 * @param band        Band value - 0=2.4GHz, 1=5GHz
 * @param freq        Channel frequency in MHz
 ****************************************************************************************
 */
static void phy_set_rf_band(uint8_t band, uint16_t freq)
{
    uint8_t sub_band_5g = 0;

    if (band  == PHY_BAND_2G4)
    {
        //set RF band related regs
        rf_bandsel_set(band, 2, phy_env.mimo_en);
    }
    //5GHz band is special
    else
    {
        if (freq < 5500)
        {
            sub_band_5g = RF_5G_LOW;
        }
        else
        {
            sub_band_5g = RF_5G_HIGH;
        }
#if 0
        rc_pll_shadow_pack(phy_env.mimo_en,
                           RF_FIXED1, RF_FIXED1, RF_DONT_CARE,  RF_DONT_CARE,
                           RF_VCO_SPI_BANDSW_DIS,  0/*RF_VCO_BANDSW_EN*/, sub_band_5g,
                           RF_DEFAULT1,  RF_FIXED0, RF_DEFAULT0,  RF_DEFAULT1,  RF_DEFAULT0,
                           band);
#endif
    }
}

/**
 ****************************************************************************************
 * @brief Set band 2.4GHz/5GHz in RC and RF chips.
 * @param band        Band value - 0=2.4GHz, 1=5GHz
 * @param freq        Channel frequency in MHz
 ****************************************************************************************
 */
static void phy_set_band(uint8_t band, uint16_t freq)
{
#if 0
    //RC band setting
    rc_band_setf(band);
#endif
    //RF band registers setting
    phy_set_rf_band(band, freq);

    //MDM and AGC band related settings
    if (band  == PHY_BAND_2G4)
    {
        //mdm settings
        mdm_rxallowdsss_setf(1);
        //agc cca
        agc_ofdmonly_setf(0);
    }
    else
    {
        //mdm settings
        mdm_rxallowdsss_setf(0);
        //agc cca
        agc_ofdmonly_setf(1);
    }
}

/**
 ****************************************************************************************
 * @brief Function used to regroup the waiting , polling and Lock Detect values
 * @return Boolean value of lock state: true=locked, false=lock failed
 ****************************************************************************************
 */
static bool phy_locked(void)
{
#if 0
    //waste some time
    for (int i=0; i<1000; i++)
    {
        rc_cntl_stat_get();
    }
    //trigger LD refresh (triggers a read through IO_EXP SPI)
    rc_refresh_ld_setf(1);
    while(rc_refresh_ld_getf());

    //check used radios for lock state OK
    if ((((phy_is_bit_set(phy_env.rc_path_sel, RC_PATH_0)) && (rc_ch0_ld_getf() == 0))) ||
        (((phy_is_bit_set(phy_env.rc_path_sel, RC_PATH_1)) && (rc_ch1_ld_getf() == 0))) ||
        (((phy_is_bit_set(phy_env.rc_path_sel, RC_PATH_2)) && (rc_ch2_ld_getf() == 0))))
    {
        dbg(D_ERR D_PHY " RF not locked\n");
        return false;
    }
    dbg(D_INF D_PHY " RF locked\n");
    return true;
#else
    return true;
#endif
}

/**
 ****************************************************************************************
 * @brief MAXIM Radio(s) initialization function.
 * This function is called at FW initialization or after phy_stop().
 * It will set RC register values and also RF register value using RC SPI control. Most
 * of the register values set in the radio(s) are static for the duration of SW run.
 ****************************************************************************************
 */
static void phy_rf_init(void)
{
#if 0
#if 0
    //RF regs init------------------------------------------------------------------------
    //enable blocks in stand by - vref=1
    rf_standby_set(0, 1, phy_env.mimo_en);

    //default dividers for synthesizer are for chnl 5 in 2.4GHz band
    rf_intdiv_set(BAND_2G_DIV[phy_env.chan.center1_freq].int_div);
    rf_fracdiv_set(BAND_2G_DIV[phy_env.chan.center1_freq].frac_div);

    //band select and pll - RF and RC shadow - use RF for 2.4, RC for 5
    phy_set_rf_band(phy_env.chan.band, phy_env.chan.center1_freq);

#else
    //Temporary replace of RF init for simple sequence according to .sh
    //phy_rc_rf_reg_write(0x00, 0x1140); //-reset value
    //phy_rc_rf_reg_write(0x01, 0x00CA); //-reset value
    // Non present in datasheet - c.f. ppl here that adjusted this
    // when trying to fix TXOP issue
    // bit6 in 0x3FFF loops tx back to rx
    phy_rc_rf_reg_write(0x02, 0x3FBF);   //Vref=1 compared to rst value 0x1007
    phy_rc_rf_reg_write(0x03, 0x30A2); //-reset value
    phy_rc_rf_reg_write(0x04, 0x1ddd); //-reset value
    //phy_rc_rf_reg_write(0x05, 0x1824); //-reset value
    //phy_rc_rf_reg_write(0x06, 0x1c00); //-reset value
    //phy_rc_rf_reg_write(0x07, 0x002a); //-reset value
    //phy_rc_rf_reg_write(0x08, 0x0025); //-reset value
    //phy_rc_rf_reg_write(0x09, 0x0200); //-reset value
    //phy_rc_rf_reg_write(0x0A, 0x03c0); //-reset value
    //phy_rc_rf_reg_write(0x0b, 0x007f); //-reset value
    //phy_rc_rf_reg_write(0x0C, 0x0000); //-reset value
#endif

    //RC----------------------------------------------------------------------------------
    //Path 0
    if (phy_is_bit_set(phy_env.rc_path_sel, RC_PATH_0))
    {
        rc_ch0_rx_onoff_delay_pack(  1, 1);
        rc_ch0_tx_onoff_delay_pack(  1, 1);   //old on delay 0x41
        rc_ch0_pa_onoff_delay_pack(  1, 0x10);//old on delay 0x41
        rc_ch0_shdn_onoff_delay_pack(1, 1);

        rc_ch0_en_setf(1);
    }
    //Path 1
    if (phy_is_bit_set(phy_env.rc_path_sel, RC_PATH_1))
    {
        rc_ch1_rx_onoff_delay_pack(  1, 1);
        rc_ch1_tx_onoff_delay_pack(  1, 0x41);
        rc_ch1_pa_onoff_delay_pack(  1, 0x41);
        rc_ch1_shdn_onoff_delay_pack(1, 1);

        rc_ch1_en_setf(1);
    }
    //Path 2
    if (phy_is_bit_set(phy_env.rc_path_sel, RC_PATH_2))
    {
        rc_ch2_rx_onoff_delay_pack(  1, 1);
        rc_ch2_tx_onoff_delay_pack(  1, 0x41);
        rc_ch2_pa_onoff_delay_pack(  1, 0x41);
        rc_ch2_shdn_onoff_delay_pack(1, 1);

        rc_ch2_en_setf(1);
    }

    //timeout for PLL lock detection
    rc_ld_timeout_set(0x10);

    //todo enable when interrupts are handled
    //enable interrupts for lock fail for the radios that are used!
    //rc_int_en_set(phy_env.rc_path_sel);

    //Enable RC - RADIO SHUTDOWN ->STANDBY------------------------------------------------
    rc_band_setf(PHY_BAND_2G4);

    rc_rc_en_setf(1);
#else
    rc_rf_rstn_setf(1);
#endif
    dbg(D_INF D_PHY " 7.RC static regs set and RC enabled\n");

    if (!phy_locked())
    {
        //todo do what???
    }
}

/**
 ****************************************************************************************
 * @brief MXD board components init
 ****************************************************************************************
 */
static void phy_mxd_init(void)
{
    //IO Expander first - special component
    phy_iox_init();

#if 0
    //RF Reset
    rc_spi_reset_setf(1);
    rc_spi_reset_setf(0);
#else
//      mxd_io_override_data_set(1 << 31);
    mxd_io_override_en_set(1 << 31);

    mxd_rf_maxscend_sel_setf(1); // crashes the platform with bad ampl. of 26MHz clock
    // nothing spi specific - try this
    rc_rf_rstn_setf(0);
    rc_rf_rstn_setf(1);
#endif

    dbg(D_INF D_PHY " 3.RF register reset done\n");

    //init diagnostic ports - change through dbg interface if needed
    mxd_diagportconf1_set(0x00001314);
    mxd_diagportconf2_set(0x00000017);

    dbg(D_INF D_PHY " 4.FPGA B static settings done\n");

#if 0
    //front end delay
    rc_fe_rx_del_set(0x12c);
#endif

    phy_adc12b_init();
    phy_adc8b_init();
}

static inline bool p40m_ok(uint8_t chantype)
{
    return true;
}

static int force2040_toggle(uint8_t chantype)
{
    uint32_t rxmodes, f2040_val, f2040_msk;

    f2040_msk = MDM_FORCE40_BIT | MDM_FORCE20_BIT;
    if (chantype == PHY_CHNL_BW_20)
        f2040_val = MDM_FORCE20_BIT;
    else if (chantype == PHY_CHNL_BW_40)
        f2040_val = MDM_FORCE40_BIT;
    else
        f2040_val = 0;

    rxmodes = (mdm_rxmodes_get() & ~f2040_msk) | f2040_val;

    mdm_rxmodes_set(rxmodes);
    while ((mxd_clkrst_stat_get() & 0xFF) != (MXD_MMC_1_LOCK_BIT | MXD_MMC_0_LOCK_BIT |
                                              MXD_IODELCTRL_READY_BIT))
        ;

    do
    {
        mdm_rxmodes_set(rxmodes ^ f2040_msk);
        while ((mxd_clkrst_stat_get() & 0xFF) != (MXD_MMC_1_LOCK_BIT | MXD_MMC_0_LOCK_BIT |
                                                  MXD_IODELCTRL_READY_BIT))
            ;

        mdm_rxmodes_set(rxmodes);
        while ((mxd_clkrst_stat_get() & 0xFF) != (MXD_MMC_1_LOCK_BIT | MXD_MMC_0_LOCK_BIT |
                                                  MXD_IODELCTRL_READY_BIT))
            ;
    } while (!p40m_ok(chantype));

    return 0;
}

/**
 ****************************************************************************************
 * @brief Bandwidth change function.
 * This function is called during set channel procedure if the bandwidth configuration
 * has changed
 *
 * @param[in] bw  Bandwidth type
 ****************************************************************************************
 */
static void phy_change_bw(uint8_t bw)
{
    switch(bw)
    {
        case PHY_CHNL_BW_20:
            mdm_txstartdelay_setf(0x000000B4);
            mdm_tbectrl0_set(0x0C0F0700);
            agc_rwnxagcaci20marg0_set(0);
            agc_rwnxagcaci20marg1_set(0);
            agc_rwnxagcaci20marg2_set(0);
            break;
        case PHY_CHNL_BW_40:
            mdm_txstartdelay_setf(0x00000168);
            mdm_tbectrl0_set(0x0C0F0702);
            agc_rwnxagcaci20marg0_set(AGC_RWNXAGCACI20MARG0_RESET);
            agc_rwnxagcaci20marg1_set(AGC_RWNXAGCACI20MARG1_RESET);
            agc_rwnxagcaci20marg2_set(AGC_RWNXAGCACI20MARG2_RESET);
            break;
        case PHY_CHNL_BW_80:
        case PHY_CHNL_BW_160:
        case PHY_CHNL_BW_80P80:
            dbg(D_ERR D_PHY "Unsupported channel width\n");
            break;
    }

    force2040_toggle(bw);
}

void phy_init(const struct phy_cfg_tag *config)
{
    const struct phy_mxd_cfg_tag *cfg = (const struct phy_mxd_cfg_tag *)&config->parameters;

    //PHY ENVIRONMENT---------------------------------------------------------------------
    //Extract which RC and RF paths are used, MIMO or not
    phy_get_config(0);

    //Band = 2.4GHz by default
    phy_env.chan.band         = PHY_BAND_5G;
    phy_env.chan.type         = PHY_CHNL_BW_OTHER;
    phy_env.chan.prim20_freq  =
    phy_env.chan.center1_freq =
    phy_env.chan.center2_freq = PHY_UNUSED;
    dbg(D_INF D_PHY " 1.PHY INIT RC_PATHs=0x%X , RF_PATHS=0x%X\n", phy_env.rc_path_sel, phy_env.rf_path_sel);

    //PHY blocks initialization-----------------------------------------------------------
    //MXD components
    phy_mxd_init();
    dbg(D_ERR D_PHY "%s: JMA Chip ID 0x%04x\n", __func__, __spi_read(0x0002));
#if RW_MXD
    if (mxd_on()) {
        //acrux2ComInit();
        MxdRfInit();
        MxdRfSetWifiMode(1);
        MxdRfModeCfg(_RF_MODE_WIFI_TX);
        // MxdRfSetFreqChWifi(2412);
       spi_write(0x408, 0x1f);
    }
#endif
    rc_rx_onoff_delay_pack(0x5, 0x50);
    rc_tx_onoff_delay_pack(0x5, 0x50);
    rc_pa_onoff_delay_pack(0x5, 0x50);
    rc_tx_switch_onoff_delay_pack(0x5, 0x5);

    //Radios alone through RC
    phy_rf_init();
    //MODEM - contains AGC?
    phy_mdm_init(cfg->tx_dc_off_comp);
    //AGC - separate or in MDM?
    phy_agc_init();
}

void phy_reset(void)
{
}

void phy_mdm_isr(void)
{
    uint32_t irq_status = mdm_irqstat_get();

    // Acknowledge the pending interrupts
    mdm_irqack_set(irq_status);

    #if NX_RADAR_DETECT
    // Check interrupt status
    if (irq_status & MDM_IRQRADARDET_BIT)
    {
        PROF_RADAR_IRQ_SET();
        // Check if FIFO is empty
        if (!mdm_radfifoempty_getf())
        {
            rd_event_ind(PHY_PRIM);
        }
        PROF_RADAR_IRQ_CLR();
    }
    #endif

    ASSERT_REC(!(irq_status & MDM_IRQCCATIMEOUT_BIT));
}

void phy_rc_isr(void)
{
    dbg(D_ERR D_PHY "%s: TODO\n", __func__);
}

void phy_get_version(uint32_t *version_1, uint32_t *version_2)
{
    *version_1 = mdm_nxversion_get();
    // TODO Add version reading for other PHY elements than modem.
    *version_2 = 0;
}


void phy_set_channel(const struct mac_chan_op *chan, uint8_t index)
{
#if 0
    /*todo 11ac: when 2 radio transceivers on one path, careful how many writes go to both,
            and when the two need separate SPi commands*/
    uint8_t psel = 0;
    uint32_t tmp;
    uint16_t div_int, div_frac;

    if (chan->band != PHY_BAND_2G4)
    {
        dbg(D_ERR D_PHY "%s: 5GHz is unsupported\n", __func__);
        return;
    }

    //if same channel is set just skip
    if (phy_env.chan.band == chan->band &&
        phy_env.chan.type == chan->type &&
        phy_env.chan.flags == chan->flags &&
        phy_env.chan.prim20_freq == chan->prim20_freq &&
        phy_env.chan.center1_freq == chan->center1_freq &&
        phy_env.chan.center2_freq == chan->center2_freq)
    {
        dbg(D_INF D_PHY " Setting same channel, do nothing");
        return;
    }

    //todo add check for band 5+type 80, 160 and 80+80

    /* Every time a channel is changed, RC is disabled causing the entire PHY shut down.
     * Once all settings are made, RC is enable and SW polls status of wake up and lock
     * to know when ready to use
     */
    //Disable RC to turn everything off
#if 0
    rc_rc_en_setf(0);
#else
    rc_rf_rstn_setf(0);
#endif
    //todo poll shutdown status bits for radios of interest

    if (phy_env.chan.type != chan->type)
    {
        phy_change_bw(chan->type);
    }

    //BAND--------------------------------------------------------------------------------
    //set band info where needed(RC, Radios, ...)
    phy_set_band(chan->band, chan->center1_freq);

#if 0
    //TYPE--------------------------------------------------------------------------------
    //todo what values for LPF in RF?Now it's at default
    switch(chan->type)
    {
        case PHY_CHNL_BW_20:
            rf_lpf_set(2,1,1,0);
            psel = 0;
            break;
        case PHY_CHNL_BW_40:
            rf_lpf_set(2,3,3,0);
            psel = (chan->center1_freq + 10 - chan->prim20_freq) / 20;
            break;
        case PHY_CHNL_BW_80:
            psel = (chan->prim20_freq - (chan->center1_freq - 30)) / 20;
        case PHY_CHNL_BW_160:
        case PHY_CHNL_BW_80P80:
            dbg(D_ERR D_PHY "Unsupported channel width\n");
            break;
    }

    //CHANNEL-----------------------------------------------------------------------------
    //set divider values for synthesizer in radio(s)
    if (chan->band == PHY_BAND_2G4)
        tmp = (((uint32_t)chan->center1_freq)<<16) / 15;
    else
        tmp = (((uint32_t)chan->center1_freq)<<16) / 25;

    // extract integral and fractional part
    div_int   = (tmp >> 16) & 0xFF;
    div_frac  = tmp & 0xFFFF;

    rf_intdiv_set(((div_frac & 3) << 12) | div_int);
    rf_fracdiv_set((div_frac >> 2) & 0x3FFF);
#else
    if (mxd_on()) {
        MxdRfSetFreqChWifi(chan->center1_freq);
    }
#endif

    // Set the PSEL value in modem
    mdm_psselect20_setf(psel & 0x01);
    mdm_psselect40_setf((psel & 0x02) >> 1);


    //Enable RC and thus trigger the configuration start
#if 0
    rc_rc_en_setf(1);
#else
    rc_rf_rstn_setf(1);
#endif

    //Lock--------------------------------------------------------------------------------
    if (!phy_locked())
    {
        //failure measures - reset, shut down...
    }

#if 0
    rc_ch0_tx_onoff_delay_set(0x00010010);
    rc_ch0_pa_onoff_delay_set(0x000A00E0);
#endif

    //keep currently set values
    phy_env.chan = *chan;

    #if NX_RADAR_DETECT
    if (chan->band == PHY_BAND_5G)
    {
        agc_radardeten_setf(1);
    }
    else
    {
        agc_radardeten_setf(0);
    }
    #endif

    phy_adc12b_init();
    phy_adc8b_init();
# else
    if (mxd_on()) {
        nxmac_abgn_mode_setf(MODE_802_11N_2_4);
        MxdRfSetFreqChWifi(chan->center1_freq);
    }

#endif
}

void phy_get_channel(struct phy_channel_info *info, uint8_t index)
{
    // Map the band, channel type, primary channel index on info1
    info->info1 = phy_env.chan.band | (phy_env.chan.type << 8) | (phy_env.chan.prim20_freq << 16);
    // Map center freq on info2
    info->info2 = phy_env.chan.center1_freq | (phy_env.chan.center2_freq << 16);

}

void phy_stop(void)
{
    //RC disable only - everything will be shut down
#if 0
    rc_rc_en_setf(0);
#else
    rc_rf_rstn_setf(0);
#endif

    //todo stop modem and MXD components too?
    //todo anything else?
}

uint32_t phy_get_channel_switch_dur(void)
{
    return 1500;
}

bool phy_has_radar_pulse(int rd_idx)
{
    return (mdm_radfifoempty_getf() == 0);
}

bool phy_get_radar_pulse(int rd_idx, struct phy_radar_pulse *pulse)
{
    // Check if FIFO is empty
    if (mdm_radfifoempty_getf())
        return (false);

    // FIFO is not empty, get an element
    pulse->pulse = REG_PL_RD(0x60C04000);

    return (true);
}

#if NX_DEBUG_DUMP
void phy_get_diag_state(struct dbg_debug_info_tag *dbg_info)
{
    int i;
    uint8_t mictor1_sel = mxd_mictor_1_sel_getf();
    uint8_t diag = mxd_phydiag_conf_lsb_1_getf();

    // Get diagnostic port state for the PHY
    mxd_mictor_1_sel_setf(0x14);
    for (i = 0; i < DBG_DIAGS_PHY_MAX; i++)
    {
        // Go through the different banks and copies the diag values
        mxd_phydiag_conf_lsb_1_setf(i);
        dbg_info->diags_phy[i] = mxd_diag1_stat_get() & 0xFFFF;
    }
    mxd_phydiag_conf_lsb_1_setf(diag);
    mxd_mictor_1_sel_setf(mictor1_sel);
}
#endif

void phy_ld_fail_irq(void)
{
#if 0
    //even if just one radio fails to lock, all should be reset and brought back together
    //because in MIMO they are synchronised

    //rc_en_setf(0); //before ACK handling?
    //we can use SPI RESET for radios now

    uint32_t int_stat = rc_int_stat_get();

    //todo: delete the following part if all radios are reset and reinitialized because
    //all 3 INTs can be ACKed then at once
    //check which radio did not lock - out of those that should have!
    if (phy_is_bit_set(int_stat, RC_PATH_0) && phy_is_bit_set(phy_env.rc_path_sel, RC_PATH_0))
    {
        //do calib again? shutdown? spi_reset?

        //ack INT to reactivate it
        rc_ch0_ld_timeout_ack_setf(1);
    }
    else if (phy_is_bit_set(int_stat, RC_PATH_1) && (phy_is_bit_set(phy_env.rc_path_sel, RC_PATH_1)))
    {
        //do calib again? shutdown? spi_reset?

        //ack INT to reactivate it
        rc_ch1_ld_timeout_ack_setf(1);
    }
    else if (phy_is_bit_set(int_stat, RC_PATH_2) && (phy_is_bit_set(phy_env.rc_path_sel, RC_PATH_2)))
    {
        //do calib again? shutdown? spi_reset?

        //ack INT to reactivate it
        rc_ch2_ld_timeout_ack_setf(1);
    }
    //ack all at once?
#else
    dbg(D_ERR "%s:%d:\n", __func__, __LINE__);
#endif
}

bool phy_uf_supported(void)
{
    return false;
}

void phy_uf_enable(bool enable)
{
}

bool phy_vht_supported(void)
{
    return false;
}

bool phy_he_supported(void)
{
    return false;
}

bool phy_ldpc_tx_supported(void)
{
    return false;
}

bool phy_ldpc_rx_supported(void)
{
    return false;
}

bool phy_bfmee_supported(void)
{
    return false;
}

bool phy_bfmer_supported(void)
{
    return false;
}

bool phy_mu_mimo_rx_supported(void)
{
    return false;
}

bool phy_mu_mimo_tx_supported(void)
{
    return false;
}

#if RW_MUMIMO_RX_EN
void phy_set_group_id_info(uint32_t membership_addr, uint32_t userpos_addr)
{
}
#endif

void phy_set_aid(uint16_t aid)
{
}

uint8_t phy_get_bw(void)
{
    return 0;
}

uint8_t phy_get_nss(void)
{
    return 0;
}

uint8_t phy_get_ntx(void)
{
    return (0);
}

uint8_t phy_get_nrx(void)
{
    return (0);
}

uint8_t phy_get_bfr_mem_size(void)
{
    return (0);
}

uint8_t phy_switch_antenna_paths(void)
{
    return 0;
}

/// @}
