/**
 ****************************************************************************************
 *
 * @file nx_time.c
 *
 * @brief Provide time related function for the reference platform.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @addtogroup TIME
 * @{
 ****************************************************************************************
 */

#include "rwnx_config.h"

#if NX_FULLY_HOSTED

#include "nx_time.h"
#include "reg_mac_core.h"

/// Number of seconds elapsed since EPOCH when firmware has been initialized
static uint32_t epoch_sec;
/// Number of microseconds (modulo 1 sec) elapsed since EPOCH when firmware has been initialized
static uint32_t epoch_usec;

void time_init(uint32_t sec, uint32_t usec)
{
    epoch_sec = sec;
    epoch_usec = usec;
}

/**
 ****************************************************************************************
 * @brief Read two part monotic counter 2
 *
 * Read msb twice to detect if counter wrap while reading it.
 *
 * @param[out] msb Updated with msb part of the counter (16bits only)
 * @param[out] lsb Updated with lsb part of the counter (32bits)
 ****************************************************************************************
 */
__INLINE void read_time(uint32_t *msb, uint32_t *lsb)
{
    uint32_t _msb, _msb_r, _lsb;
    _msb = nxmac_monotonic_counter_2_hi_get() & 0xffff;
    _lsb = nxmac_monotonic_counter_2_lo_get();
    _msb_r = nxmac_monotonic_counter_2_hi_get() & 0xffff;

    if (_msb_r != _msb)
    {
        _lsb = nxmac_monotonic_counter_2_lo_get();
    }
    *msb = _msb_r;
    *lsb = _lsb;
}

int get_time(enum time_origin_t origin, uint32_t *sec, uint32_t *usec)
{
    uint32_t msb, lsb, _sec, _usec, tmp, fact;

    if (origin > SINCE_EPOCH)
        return -1;

    read_time(&msb, &lsb);

    /* Replace uint64_t / 1000000 (uint48_t in practice)
       by 3 ~fmul (mul + shift).
       Yes this is more complicated but also 20 times faster on cpu without div
       instruction.
       48bits time value is divided in 4 parts 0xAABBCCCCDDDD
       First two parts (A ,B) are only 8bits to allow more bits in the factor
       (merging this two parts with a 16 bits factor doesn't provide enough precision).
       For the third part (C) a factor on 16bits is enough so part is also 16bits.
       Fourth part (D) is always < 1000000 so no need to divide it.
       factor = 0x8637bd = 1 / 1000000 in Q43 = ((1<<43) / 1000000) (truncated on 24bits)
     */
    fact = 0x8637bd;
    tmp = (msb >> 8) * fact;
    _sec = (tmp >> 3);
    tmp = (msb & 0xff) * fact;
    _sec += (tmp >> 11);
    fact >>= 8;
    tmp = (lsb >> 16) * fact;
    _sec += (tmp >> 19);
    _usec = lsb - (_sec * 1000000);

    if (origin == SINCE_EPOCH)
    {
        _sec += epoch_sec;
        _usec += epoch_usec;

    }

    /* Previous computation ensure that loop won't run more than 2 times */
    while (_usec > 1000000)
    {
        _usec -= 1000000;
        _sec ++;
    }

    *sec = _sec;
    *usec = _usec;

    return 0;
}

int get_time_us(enum time_origin_t origin, uint64_t *usec)
{
    uint32_t msb, lsb;
    uint64_t val;

    if (origin > SINCE_EPOCH)
        return -1;

    read_time(&msb, &lsb);
    val = ((uint64_t)msb << 32) + (uint64_t)lsb;

    if (origin == SINCE_EPOCH)
    {
        val += (uint64_t)epoch_usec;
        val += (uint64_t)(epoch_sec * 1000000);
    }

    *usec = val;
    return 0;
}

#endif /* NX_FULLY_HOSTED */
/**
 * @}
 */
