/**
 ****************************************************************************************
 *
 * @file nx_time.h
 *
 * @brief Provide time related function specific to the platform.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

#ifndef _PLATFORM_H_
#define _PLATFORM_H_
/**
 ****************************************************************************************
 * @defgroup TIME TIME
 * @ingroup PLATFORM_DRIVERS
 * @{
 ****************************************************************************************
 */

/**
 * Time origin
 */
enum time_origin_t {
    /** Since boot time */
    SINCE_BOOT,
    /** Since Epoch : 1970-01-01 00:00:00 +0000 (UTC) */
    SINCE_EPOCH,
};

/**
 ****************************************************************************************
 * @brief Initialize time.
 *
 * @param[in] sec  Number of seconds since Epoch when the system started.
 * @param[in] usec Number of microseconds since Epoch when the system started
 *                 (excluding the seconds in sec).
 ****************************************************************************************
 */
void time_init(uint32_t sec, uint32_t usec);

/**
 ****************************************************************************************
 * @brief Get current time.
 *
 * return the current time, from the selected origin, in a sec/usec split.
 *
 * @param[in]  origin Select the time origin (Since boot or since Epoch)
 * @param[out] sec    Udapted with the number of seconds since the selected origin.
 * @param[out] usec   Updated with the number of microseconds since the selected origin.
 *                   (excluding the seconds in sec)
 *
 * @return 0 on success and != 0 otherwise.
 ****************************************************************************************
 */
int get_time(enum time_origin_t origin, uint32_t *sec, uint32_t *usec);

/**
 ****************************************************************************************
 * @brief Get current time.
 *
 * return the current time, from the selected origin, in usec.
 *
 * @param[in]  origin Select the time origin (Since boot or since Epoch)
 * @param[out] usec   Updated with the number of microseconds since the selected origin.
 *
 * @return 0 on success and != 0 otherwise.
 ****************************************************************************************
 */
int get_time_us(enum time_origin_t origin, uint64_t *usec);

/**
 * @}
 */
#endif /* _PLATFORM_H_ */
