#!/usr/bin/env perl

use strict;
use Getopt::Long;

my %memory;
my %memory_ref;
my $debug = 0;
my $no_mod = 0;
my $link_info_ref;
my $max_mod_length = 0;

sub help {
    print "usage: $0 [-h] [-d]* [-f] [-r <linkinfo-ref.xxx.txt>] <linkinfo.xxx.txt>
options:
-h, --help           : show this help and quit
-f, --file           : Show result per file (and not per module)
-d, --debug          : Enable denug info. Set it several time for more debug
-r, --ref <linkinfo> : Show memory usage difference with this reference.
                       Difference in shown in bytes for the module/file

Note:
The meaning of the percent value differs between module/file line and the the TOTAL line
- For module/file it represents the percentage of the memory used by module
  relative to the total amount of memory used for the section.
- For the 'TOTAL' line it represents the percentage of memory used by the sectin relative
  to the total amount available for this memory.

For example:

|----------------------------------------------------------------|
|                   Code/Data (1048576 Bytes)                    |
|----------------------------------------------------------------|
|                |     Total     |     Code      |     Data      |
|     Module     |---------------|---------------|---------------|
|                | bytes  |  %   | bytes  |  %   | bytes  |  %   |
|----------------------------------------------------------------|
| WPA_SUPPLICANT | 124918 | 27.1 | 103318 | 27.6 |  21600 | 25.0 |
...
|----------------------------------------------------------------|
|          TOTAL | 460132 | 43.9 | 373848 | 35.7 |  86284 |  8.2 |
|----------------------------------------------------------------|

The code used by wpa_supplicant (103318) represent 27.6 % of all
the memory used by code (373848).
The total amount of memory used for data (86284) represent 8.2 %
of the memory available for Code/Data (1048576)
";

    exit;
}

GetOptions ("help" => \&help,
            "debug" => sub { $debug++ },
            "file" => \$no_mod,
            "ref=s" => \$link_info_ref,
    );

my $link_info = shift;

&help() unless (defined($link_info));

if ($link_info_ref) {
    &parse_link_info($link_info_ref, \%memory_ref);
}

&parse_link_info($link_info, \%memory);

if ($link_info_ref) {
    &dump_memory_usage_ref(\%memory, \%memory_ref);
} else {
    &dump_memory_usage(\%memory);
}

##############################################################################
## gcc parsing (and helpers)
##############################################################################
sub gcc_parse_linkinfo
{
    my ($file, $memhash) = @_;

    open INFO, "$file" or die "Cannot open $file";
    my $line;
    while ($line = <INFO>)
    {
        if ($line =~ /Memory Configuration/)
        {
            &gcc_parse_memory_configuration($memhash);
        }
        elsif($line =~ /^([\w.]+) +(0x\w+) +(0x\w+)( +load address (0x\w+))?/)
        {
            &gcc_parse_section($memhash, $1, hex($2), hex($3), hex($5));
        }
    }

    close INFO;
}

sub gcc_parse_memory_configuration
{
    my ($memhash) = @_;
    my $line;
    my $start = 0;

    while ($line = <INFO>)
    {
        if ($line =~ /^$/)
        {
            last if $start;
        }
        elsif ($line =~ /(\w+)\s+(0x\w+)\s+(0x\w+)/)
        {
            $start = 1;
            $memhash->{$1} = { address => hex($2),
                               size    => hex($3),
                               used    => 0,
                               mod     => {},
                               name    => $1,
                               last_address => hex($2)};
        }
    }

    my ($code, $code_size, $data, $data_size,
        $shared, $shared_size) = &extract_mem_info($memhash);

    # Data and code stored in the same memory. Define sections hash to distinguish them.
    if (defined($code) && !defined($data)) {
        $code->{sections}->{"Code"} = { used => 0,
                                        mod => {}};
        $code->{sections}->{"Data"} = { used => 0,
                                        mod => {}};
    }
}

sub gcc_mod_update_size
{
    my ($mem, $section, $mod, $size) = @_;

    $mem->{used} += $size;
    $mem->{mod}->{$mod} += $size;
    if (defined($section)) {
        $section->{used} += $size;
        $section->{mod}->{$mod} += $size
    }

    $max_mod_length = length($mod) if (length($mod) > $max_mod_length);
}

sub gcc_parse_section
{
    my ($memhash, $name, $addr, $size, $addr_load) = @_;
    my $skip = 0;
    my $mem;
    my $line;
    my $mod;
    my $check_size = 0;
    my $section;

    if ($name =~ /(\.(debug|comment|line|stab|fake|dummy)|LARAM|MACHW|LA|HW_MIB)/) {
        $skip = 1;
         debug(2, "Skip Section $1");
    } else {
        foreach my $s (keys %$memhash)
        {
            if ($addr >= $memhash->{$s}->{address} &&
                $addr <= $memhash->{$s}->{address} + $memhash->{$s}->{size}) {
                $mem = $memhash->{$s};
                &debug(1, "\n\nSection $name is in [$s]");
                last;
            }
        }

        if (!defined($mem)) {
            warn("Cannot find memory for section $name\n");
            $skip = 1;
        }

        # memory used for both code and data, select between code and data based on section name
        if (defined($mem->{sections})) {
            my $s = "Code";
            if ($name =~ /(data|bss)/) {
                $s = "Data";
            }
            &debug(1, "$mem->{name} used for code/data, section $name is $s");
            if (defined($mem->{sections}->{$s})) {
                $section = $mem->{sections}->{$s};
            } else {
                warn("Section $s not initialize for $mem->{name}\n");
            }
        }
    }

    my $check_addr = $addr;
    while ($line = <INFO>)
    {

        last if ($line =~ /^$/);
        next if ($skip > 0);

        if ($line =~ /^\s*([^ ]*)\s+(0x\w+)\s+(0x\w+)\s+(\/.*)$/)
        {
            my $o_size = hex($3);
            my $o_addr = hex($2);

            if ($o_addr < $mem->{last_address})
            {
                # ignore linker optimization ?
                &debug(1, "linker optimization ?");
                next;
            }

            if ($o_addr != $check_addr) {
                # addr of object doesn't match addr of previous object + size of previous object
                if ($o_addr == $mem->{last_address}) {
                    # Sometimes, at least with riscv compilation, object are shown in the linkinfo
                    # with some size but not included in the final firmware (maybe some linker optimation ?)
                    my $prev_o_size = $check_addr - $mem->{last_address};
                    &gcc_mod_update_size($mem, $section, $mod, -$prev_o_size);
                    $check_size -= $prev_o_size;
                    &debug(1, "Address did not increased, remove previous object");
                } else {
                    warn sprintf("expected address 0x%08x got 0x%08x\n",
                                 $check_addr, $o_addr);
                }
            }

            $mod = &file2mod($4);
            $mem->{last_address} = $o_addr;
            &gcc_mod_update_size($mem, $section, $mod, $o_size);
            $check_size += $o_size;
            $check_addr = $mem->{last_address} + $o_size;
            &debug(2, "Add $o_size for mod $mod in [$mem->{name}]");

        } elsif ($line =~ /^\s*\*fill\*\s+(0x\w+)\s+(0x\w+)/) {
            my $o_size = hex($2);
            my $o_addr = hex($1);

            if ($o_addr != $check_addr) {
                if ($o_addr == $mem->{last_address}) {
                    my $prev_o_size = $check_addr - $mem->{last_address};
                    &gcc_mod_update_size($mem, $section, $mod, -$prev_o_size);
                    $check_size -= $prev_o_size;
                    &debug(1, "Address did not increased, remove previous object");
                } else {
                    warn sprintf("expected address 0x%08x got 0x%08x\n",
                                 $check_addr, $o_addr);
                }
            }

            if (defined($mod)) {
                &gcc_mod_update_size($mem, $section, $mod, $o_size);
            }

            $mem->{last_address} = $o_addr;
            $check_size += $o_size;
            $check_addr = $mem->{last_address} + $o_size;
            &debug(2, "Add (fill) $o_size for mod $mod in [$mem->{name}]");
        }
    }

    if (!$skip && !defined($mod)) {
        # section without object associated
        $mod = &file2mod($name);
        &gcc_mod_update_size($mem, $section, $mod, $size);
        &debug(2, "Add (no symbol) $size for mod $mod in [$mem->{name}]");
    }

    if (!$skip && $check_size != $size) {
        warn "Section $name: memory found ($check_size) doesn't match total memory ($size)\n";
    }
}

##############################################################################
## ceva Parsing (and helpers)
##############################################################################
sub ceva_parse_linkinfo
{
    my ($file, $memhash, $char_size) = @_;
    my $line;
    my %symbols;
    my %segments;
    my %linker_symbols;

    open INFO, "$file" or die "Cannot open $file";

    while ($line = <INFO>)
    {
        if ($line =~ /SYMBOL TABLE:/)
        {
            &ceva_parse_symbol_table(\%symbols, \%linker_symbols);
        }
        elsif($line =~ /(Internal|External).*memory map table:/)
        {
            my $id = 0;
            $id = 1 if ($1 eq "External");
            &ceva_parse_segment_info($memhash, $id, \%segments, $char_size);
        }
        elsif($line =~ /classes:/)
        {
            &ceva_parse_classes($memhash, \%linker_symbols, $char_size);
        }
        elsif($line =~ /Command line:/)
        {
            &ceva_parse_cmdline($memhash, $line, $char_size);
        }
    }
    close INFO;

    &ceva_dispatch_symbols(\%symbols, \%segments, $char_size);

    # sanity check
    foreach my $m (keys(%$memhash)) {
        if ($memhash->{$m}->{used} != $memhash->{$m}->{length}) {
            print "Mem $m: found=$memhash->{$m}->{used} size=$memhash->{$m}->{length}\n";

            my $section = $memhash->{$m}->{sec};
            foreach my $s (keys(%{$segments{$section}})) {
                my $size = $segments{$section}->{$s}->{size};
                my $used = $segments{$section}->{$s}->{used};
                if ($used != $size) {
                    my $diff = $size - $used;
                    &debug(1, "  - $segments{$section}->{$s}->{name}: missing $diff/$size");
                }
            }
        }
    }
}

# Get size for internal code and data memory
sub ceva_parse_cmdline
{
    my ($memhash, $cmd_line, $char_size) = @_;

    if ($cmd_line =~ /internalCode(\d+)/) {
        $memhash->{"code"}->{size} = $1 * 1024 * $char_size;
    }

    if ($cmd_line =~ /internalData(\d+)/) {
        $memhash->{"data"}->{size} = $1 * 1024 * $char_size;
    }

    &debug(1, "Internal memory: code=$memhash->{code}->{size} data=$memhash->{data}->{size}");
}

# Get size of other memory defined
sub ceva_parse_classes
{
    my ($memhash, $linker_symbols, $char_size) = @_;

    while (<INFO>) {
        last if ($_ =~ /^\s*$/);

        if ($_ =~ /(\w+)\s+\[[CD]:(\w+),\s+(\w+)/) {
            my ($class, $addr, $size) = ($1, $2, $3);
            $class =~ tr/a-z/A-Z/;
            $memhash->{$class}->{size} = hex($3) * $char_size;
            &debug(1, "Add memory: $class $memhash->{$class}->{size}");
        } elsif ($_ =~ /\s(\w+)\sat.*symbol/ ) {
            # get symbol defined in linker script to ignore them
            $linker_symbols->{$1} = 1;
            &debug(2, "Ignore linker symbol [$1]");
        }
    }
}

# For each symbol store:
# - its section (code/data internal/external)
# - its address
# - the file in which it is defined
sub ceva_parse_symbol_table
{
    my ($symbols, $linker_symbols) = @_;

    while (<INFO>) {
        last if ($_ =~ /^\s*$/);

        if ($_ =~ /(.*):\s*(\w+):(\w+), in file ([^ ]+)/) {
            my $symbol = $1;
            my $section = $2;
            my $addr = hex($3);
            my $file = $4;

            $file =~ s/\r$//;

            # Ignore linker symbols
            next if (defined $linker_symbols->{$symbol});

            $symbols->{$section} = {} unless defined($symbols->{$section});
            $symbols->{$section}->{$addr} = $file;
        }
    }
}

# Get segment informations:
# - Size
# - Associated memory (note: merge code and code_ext memory)
sub ceva_parse_segment_info
{
    my ($memhash, $id, $segments, $char_size) = @_;
    my $new_line=0;

    while(<INFO>) {
        if ($_ =~ /(.*)@((\w)(\d))\s+at (\w+)-(\w+), size: (\w+)(, mapped in class: (\w+))?/){
            my $seg_name = $1;
            my $sec = $3 . "0" . "$4";
            $sec =~ tr/a-z/A-Z/;
            my $start = hex($5);
            my $end = hex($6);
            my $size = hex($7) * $char_size; # store size in bytes
            my $mem;

            warn("unexpected segment number $4 ($id)") unless ($4 == $id);

            $segments->{$sec} = {} unless defined($segments->{$sec});
            $segments->{$sec}->{$start} = {start => $start,
                                           end => $end,
                                           size => $size,
                                           name => $seg_name,
                                           used => 0,
                                           processed => 0};

            if (defined $9) {
                $mem = $9;
                # merge code and code_ext in same memory
                $mem = "code" if ($mem eq "code_ext");
                $mem = "data" if ($mem eq "data_ext");
            } elsif ($sec =~ "C0[01]") {
                $mem = "code";
            } elsif ($sec =~ "D0[01]") {
                $mem = "data";
            } else {
                warn("No class for External data: $_\n");
                $mem = "extra";
            }

            if (!defined($memhash->{$mem}) ||
                !defined($memhash->{$mem}->{address})) {
                $memhash->{$mem}->{address} = hex($start);
                $memhash->{$mem}->{length}  = $size;
                $memhash->{$mem}->{used}    = 0;
                $memhash->{$mem}->{name}    = $mem;
                $memhash->{$mem}->{mod}     = {};
            } else {
                $memhash->{$mem}->{length} += $size;
            }
            $segments->{$sec}->{$start}->{mem} = $memhash->{$mem};
            $memhash->{$mem}->{sec} = $sec;
            &debug(2, "[$sec] Create segment $seg_name for memory [$mem]");
            $new_line = 0;
        } elsif ($_ =~ /^\s*$/) {
            $new_line++;
            last if ($new_line == 3);
        }
    }
}

# get the segment that contains the address
sub ceva_get_segment
{
    my ($addr, $section, $segments, $r) = @_;
    if (!defined($segments->{$section})) {
        die("No segment for $section\n");
        return;
    }

    foreach my $s (keys(%{$segments->{$section}})) {
        if ($addr >= $s && $addr <= $segments->{$section}->{$s}->{end}){
            $segments->{$section}->{$s}->{processed} = 1;
            return $segments->{$section}->{$s};
        }
    }

    die("Cannot find segment for $section\@$addr ($r) \n");
}

# Factorize size update
sub ceva_mod_update_size
{
    my ($mem, $seg, $mod, $size) = @_;

    $mem->{used} += $size;
    $mem->{mod}->{$mod} += $size;
    $seg->{used} += $size;

    $max_mod_length = length($mod) if (length($mod) > $max_mod_length);
}

# Dispatch each symbol:
# - find its corresponding segment
# - determine its size
# - determine its module
# - update memory used by its module it the segment's memory
#
# Also process segment without symbols: add the whole segment
# to its corresponding module for the associated memory
sub ceva_dispatch_symbols
{
    my ($symbols, $segments, $char_size) = @_;
    my $mem;
    my $mem_ext = 0;

    foreach my $section (sort(keys %$symbols)) {
        my $mod;
        my $addr_start;
        my $size;
        my $mem;
        my $first = 1;
        my $seg;
        my $seg_end;
        my $file;

        foreach my $addr (sort({$a <=> $b} keys %{$symbols->{$section}})) {
            if ($first == 1) {
                $seg = &ceva_get_segment($addr, $section, $segments);
                $seg_end = $seg->{end};
                $mem = $seg->{mem};
                $first = 0;
                &debug(2, "\nFirst $section segment ($seg->{name}) " .
                       sprintf("%x - %x", $addr, $seg_end) . " => [$mem->{name}]");
            }

            if (defined($addr_start)) {
                if ($addr > $seg_end) {
                    $size = ($seg_end + 1 - $addr_start) * $char_size;
                } else  {
                    $size = ($addr - $addr_start) * $char_size;
                }

                &ceva_mod_update_size($mem, $seg, $mod, $size);
                &debug(2, "Add $size for mod $mod ($file) in [$mem->{name}]");

                if ($addr > $seg_end) {
                    # move to next segment
                    $seg = &ceva_get_segment($addr, $section, $segments);
                    $seg_end = $seg->{end};
                    $mem = $seg->{mem};
                    &debug(2, "\nNext $section segment ($seg->{name}) " .
                           sprintf("%x - %x", $addr, $seg_end) . " => [$mem->{name}]");
                }
            }
            $file = $symbols->{$section}->{$addr};
            chomp($file);
            $mod = &file2mod($file);
            $addr_start = $addr;
        }

        # special case for last symbol of the section
        $size = ($seg_end + 1 - $addr_start) * $char_size;
        &debug(2, "Add (last symbol) $size for mod $mod ($file) in [$mem->{name}]");
        &ceva_mod_update_size($mem, $seg, $mod, $size);
    }

    # check segment that do not contains symbol (like inttbl/stack)
    foreach my $s (keys(%{$segments})) {
        foreach my $ss (keys(%{$segments->{$s}})) {
            my $seg = $segments->{$s}->{$ss};
            if (${seg}->{processed} == 0) {
                my $mem = $seg->{mem};
                my $mod = &file2mod($seg->{name});
                &ceva_mod_update_size($mem, $seg, $mod, $seg->{size});
                &debug(2, "Add (no symbol) $seg->{size} for mod $mod in [$mem->{name}]");
            }
        }
    }
}

##############################################################################
## Common Subroutines
##############################################################################
sub parse_link_info()
{
    my ($file, $memhash) = @_;

    if ($file =~ /(aps|riscv32|arm)/) {
        &gcc_parse_linkinfo($file, $memhash);
    } elsif ($file =~ /tl4/ ) {
        &ceva_parse_linkinfo($file, $memhash, 2);
    } elsif ($file =~ /ceva-x-cc/ ) {
        &ceva_parse_linkinfo($file, $memhash, 1);
    } else {
        print "Cannot find ARCH from filename [$file]\n";
        exit;
    }
}

sub strip_file_name
{
    my ($file) = @_;
    if ($file =~ /.*(\/|\\)build(\/|\\)[^\/\\]+(\/|\\)[^\/\\]+(\/|\\)(.*)/) {
        return $5;
    } elsif ($file =~ /.*(\/|\\)([^\/\\]+)$/) {
        return $2;
    } else {
        return $file;
    }
}

sub file2mod
{
    my $file = $_[0];

    return &strip_file_name($file) if ($no_mod);

    # remove path prior to build directory to avoid incorrect classification
    $file = $3 if ($file =~ /.*(\/|\\)build(\/|\\)(.*)/);

    if ($file =~ /CEVA|Cortus|\.asm$|riscv32-unknown-elf/) {
        return "STDLIB";
    } elsif ( $file =~ /(\/|\\)ipc(_fhost)?(\/|\\)/) {
        return "IPC";
    } elsif ( $file =~ /(\/|\\)(macif|app)(\/|\\).*ipc/) {
        return "IPC";
    } elsif ( $file =~ /(\/|\\)(refip|phy|sysctrl|dma|crm|time)(\/|\\)/) {
        return "REFIP";
    } elsif ($file =~ /(\/|\\)(txu?|td)(\/|\\)/) {
        return "TX";
    } elsif ($file =~ /(\/|\\)(rxu?|uf)(\/|\\)/) {
        return "RX";
    } elsif ($file =~ /(\/|\\)scanu?(\/|\\)/) {
        return "SCAN";
    } elsif ($file =~ /(\/|\\)(dbg|la)(\/|\\)/) {
        return "DBG";
    } elsif ($file =~ /(\/|\\)ke(\/|\\)/) {
        return "KE";
    } elsif ($file =~ /((\/|\\)(rwnx(\/|\\)|arch_main|boot|intc|build_version)|inttbl|crtn)/) {
        return "MAIN/BOOT";
    } elsif ($file =~ /(\/|\\)common(\/|\\)/) {
        return "COMMON";
    } elsif ($file =~ /(\/|\\)(mac|mm|llc)(\/|\\)/) {
        return "MAC";
    } elsif ($file =~ /(\/|\\)bam(\/|\\)/) {
        return "BAM";
    } elsif ($file =~ /(\/|\\)hal(\/|\\)/) {
        return "HAL";
    } elsif ($file =~ /(\/|\\)(me|sm|sta)(\/|\\)/) {
        return "STA";
    }  elsif ($file =~ /(\/|\\)(apm)(\/|\\)/) {
        return "AP";
    } elsif ($file =~ /(\/|\\)ps(\/|\\)/) {
        return "PS";
    } elsif ($file =~ /(\/|\\)vif(\/|\\)/) {
        return "VIF";
    } elsif ($file =~ /(\/|\\)mfp(\/|\\)/) {
        return "MFP";
    } elsif ($file =~ /(\/|\\)p2p(\/|\\)/) {
        return "P2P";
    } elsif ($file =~ /(\/|\\)rd(\/|\\)/) {
        return "RADAR";
    } elsif ($file =~ /(\/|\\)chan(\/|\\)/) {
        return "CHAN";
    } elsif ($file =~ /(\/|\\)tdls(\/|\\)/) {
        return "TDLS";
    } elsif ($file =~ /(\/|\\)rc(\/|\\)/) {
        return "RC";
    } elsif ($file =~ /(\/|\\)bfr(\/|\\)/) {
        return "BFR";
    } elsif ($file =~ /(\/|\\)tpc(\/|\\)/) {
        return "TPC";
    } elsif ($file =~ /(\/|\\)mesh(\/|\\)/) {
        return "MESH";
    } elsif ($file =~ /(\/|\\)hsu(\/|\\)/) {
        return "HSU";
    } elsif ($file =~ /stack/i) {
        return "STACK";
    } elsif ($file =~ /heap/i) {
        return "HEAP";
    } elsif ($file =~ /(\/|\\)lwip-STABLE-2_0_2_RELEASE_VER|net_al(\/|\\)/i) {
        return "LWIP";
    } elsif ($file =~ /(\/|\\)(FreeRTOS|rhino)(\/|\\)/i) {
        return "RTOS";
    } elsif ($file =~ /(\/|\\)(rtos|app|macif|rtos_al|rtos_port)(\/|\\)/i) {
        return "RTOS_PORT";
    } elsif ($file =~ /(\/|\\)fhost(\/|\\)/i) {
        return "FHOST";
    } elsif ($file =~ /(\/|\\)(wpa_supplicant|src(\/|\\)(ap|common|crypto|drivers|
                                                         eap_common|eapol_auth|eapol_supp|
                                                         eap_peer|eap_server|fst|l2_packet|
                                                         p2p|pae|radius|rsn_supp|tls|
                                                         utils|wps))(\/|\\)/xi) {
        return "WPA_SUPPLICANT";
    } elsif ($file =~ /(\/|\\)mbedtls(\/|\\)/i) {
        return "MBEDTLS";
    } else {
        return &strip_file_name("/build/".$file);
    }
}

sub debug
{
    my $level = shift;

    if ($debug >= $level) {
        print "($.) " if ($.);
        print "@_\n";
    }
}

sub extract_mem_info
{
    my ($memhash) = @_;
    my ($code, $data, $shared);
    my ($code_size, $data_size, $shared_size) = (0, 0, 0);

    foreach my $m (sort { "\L$a" cmp "\L$b" } keys(%$memhash)) {
        if ($m =~ /(code|program|cpu)/i) {
            $code = $memhash->{$m};
            $code_size = $memhash->{$m}->{size};
        } elsif ($m =~ /data/i) {
            $data = $memhash->{$m};
            $data_size = $memhash->{$m}->{size};
        } elsif ($m =~ /(shram|shared)/i) {
            $shared = $memhash->{$m};
            $shared_size = $memhash->{$m}->{size};
        } else {
            &debug(1, "ignore memory [$m]");
        }
    }

    return ($code, $code_size, $data, $data_size, $shared, $shared_size);
}

sub dump_memory_usage
{
    my ($memhash) = @_;

    my ($code, $code_size, $data, $data_size,
        $shared, $shared_size) = &extract_mem_info($memhash);

    if (defined($code) && !defined($data)) {
        &dump_one_memory($code, "Code/Data", $code_size);
    } else {
        &dump_one_memory($code, "Code", $code_size) if defined($code);
        &dump_one_memory($data, "Data", $data_size) if defined($data);
    }
    &dump_one_memory($shared, "Shared Memory", $shared_size) if defined($shared);
}

# Add value in the array reference passed as first parameter
# Array is then printed by the format in used
#
sub push_val
{
    my ($res, $val, $tot, $ref_val) = @_;

    push @{$res}, $val;
    if ($val && $tot) {
        push @{$res}, ($val / $tot) * 100;
    } else {
        push @{$res}, 0;
    }

    if (defined($ref_val)) {
        my $d = $ref_val - $val;
        if ($d > 0) {
            push @{$res}, "-", $d;
        } elsif ($d == 0) {
            push @{$res}, " ", " ";
        } else {
            push @{$res}, "+", -$d;
        }
    }
}

sub dump_one_memory
{
    my ($mem, $name, $size) = @_;

    # Generat header and data formats (based on $max_mod_length and the sections names if any)
    my $col_mod_length = $max_mod_length + 2;
    my $col_bytes_length = 10; # Min value for value < 100000 is 7.
    my $col_percent_length = 8; # Should be at least 5
    my $col_section_length = $col_bytes_length + $col_percent_length + 1;
    my @sect = ("Total");
    push @sect, (sort(keys %{$mem->{sections}})) if (defined($mem->{sections}));

    my $max_section_length = 0;
    foreach my $s (@sect) {
        $max_section_length = length($s) if (length($s) > $max_section_length);
    }

    if ($max_section_length > $col_section_length) {
        my $delta = $max_section_length - $col_section_length;
        $col_bytes_length += $delta / 2;
        $delta -= ($delta / 2);
        $col_percent_length += $delta;
        $col_section_length = $max_section_length;
    }

    my $line_break = "|"."-" x (($col_section_length + 1) * @sect + $col_mod_length) . "|\n";

    my $format_header = "format HEADER =\n";
    my $t;
    #1st line
    $format_header .= "|@". "|"x(length($line_break) - 4)."@\n";
    $format_header .= "\$mm,\"|\"\n";
    $format_header .= $line_break;
    #2nd line
    $format_header .= "|". " "x$col_mod_length."|";
    $t = "@"."|"x($col_section_length - 1)."@";
    $format_header .= $t x @sect . "\n";
    $format_header .= join(",\"|\",", @sect) . ",\"|\"\n";
    #3 line
    $format_header .= "|@"."|"x($col_mod_length - 1)."@";
    $t = "-"x($col_section_length)."|";
    $format_header .= $t x @sect . "\n";
    $format_header .= "\"Module\",\"|\"\n";
    #3rd line
    $format_header .= "|". " "x$col_mod_length."|";
    $t = "@"."|"x($col_bytes_length - 1)."@@"."|"x($col_percent_length - 1)."@";
    $format_header .= $t x @sect . "\n";
    $t = "\"bytes\",\"|\",\"%\",\"|\",";
    $format_header .= $t x @sect . "\n";
    $format_header .= ".\n";

    my $format_mod = "format MODULE = \n";
    $format_mod .= "| @" . "<"x($col_mod_length - 3) . " |";
    $t = "@" . "#"x($col_bytes_length - 2) . " |@" . "#"x($col_percent_length - 4) . ".# |";
    $format_mod .= $t x @sect . "\n";
    $format_mod .= "\$mm";
    for ($t = 0 ; $t < @sect ; $t++) {
        $format_mod .= ",\$val[" . (2*$t) . "],\$val[".(2*$t + 1)."]"
    }
    $format_mod .= "\n.\n";

    # Define variables used in format before loding them
    my ($mm, @val);

    eval $format_header;
    $~ = "HEADER";
    $mm="$name";
    if ($size) {
        $mm .= " ($size Bytes)";
    }
    print $line_break;
    write;
    print $line_break;

    eval $format_mod;
    $~ = "MODULE";

    my $mods = $mem->{mod};
    foreach my $m (sort {$mods->{$b} <=> $mods->{$a}} keys %{$mods}) {
        @val = ();
        &push_val(\@val, $mods->{$m}, $mem->{used});

        foreach my $s (sort keys %{$mem->{sections}}) {
            my $v = $mem->{sections}->{$s}->{mod}->{$m} || 0;
            &push_val(\@val, $v, $mem->{sections}->{$s}->{used});
        }
        #for some reason using $m in the format doesn't work...
        $mm = $m;
        write;
    }

    $mm = "TOTAL";
    @val = ();
    &push_val(\@val, $mem->{used}, $size);
    foreach my $s (sort keys %{$mem->{sections}}) {
        &push_val(\@val, $mem->{sections}->{$s}->{used}, $size);
    }
    print $line_break;
    write;
    print $line_break;

    print "\n";
}

sub dump_memory_usage_ref
{
    my ($memhash, $memhash_ref) = @_;

    my ($code, $code_size, $data, $data_size,
        $shared, $shared_size) = &extract_mem_info($memhash);

    my ($code_ref, $code_ref_size, $data_ref, $data_ref_size,
        $shared_ref, $shared_ref_size) = &extract_mem_info($memhash_ref);

    if (defined($code) && !defined($data)) {
        &dump_one_memory_ref($code, "Code/Data", $code_size, $code_ref, $code_ref_size);
    } else {
        &dump_one_memory_ref($code, "Code", $code_size, $code_ref, $code_ref_size) if (defined($code));
        &dump_one_memory_ref($data, "Data", $data_size, $data_ref, $data_ref_size) if defined($data);
    }
    &dump_one_memory_ref($shared, "Shared Memory", $shared_size, $shared_ref, $shared_ref_size) if defined($shared);
}

sub dump_one_memory_ref
{
    my ($mem, $name, $size, $mem_ref, $size_ref) = @_;

    # Generat header and data formats (based on $max_mod_length and the sections names if any)
    my $col_mod_length = $max_mod_length + 2;
    my $col_bytes_length = 10;
    my $col_percent_length = 8;
    my $col_delta = $col_bytes_length + 2;
    my $col_section_length = $col_bytes_length + $col_percent_length + $col_delta + 2;
    my @sect = ("Total");
    push @sect, (sort(keys %{$mem->{sections}})) if (defined($mem->{sections}));

    my $max_section_length = 0;
    foreach my $s (@sect) {
        $max_section_length = length($s) if (length($s) > $max_section_length);
    }

    if ($max_section_length > $col_section_length) {
        my $delta = $max_section_length - $col_section_length;
        $col_bytes_length += $delta / 3;
        $delta -= ($delta / 3);
        $col_percent_length += ($delta / 2);
        $delta -= ($delta / 2);
        $col_delta += $delta;
        $col_section_length = $max_section_length;
    }

    my $line_break = "|"."-" x (($col_section_length + 1) * @sect + $col_mod_length) . "|\n";

    my $format_header = "format HEADER =\n";
    my $t;
    #Title line
    $format_header .= "|@". "|"x(length($line_break) - 4)."@\n";
    $format_header .= "\$mm,\"|\"\n";
    $format_header .= $line_break;
    #Section name line
    $format_header .= "|". " "x$col_mod_length."|";
    $t = "@"."|"x($col_section_length - 1)."@";
    $format_header .= $t x @sect . "\n";
    $format_header .= join(",\"|\",", @sect) . ",\"|\"\n";
    #Module name line
    $format_header .= "|@"."|"x($col_mod_length - 1)."@";
    $t = "-"x($col_section_length)."|";
    $format_header .= $t x @sect . "\n";
    $format_header .= "\"Module\",\"|\"\n";
    #'bytes', '%', 'delta' line
    $format_header .= "|". " "x$col_mod_length."|";
    $t = "@"."|"x($col_bytes_length - 1)."@@"."|"x($col_percent_length - 1)."@@"."|"x($col_delta - 1)."@";
    $format_header .= $t x @sect . "\n";
    $t = "\"bytes\",\"|\",\"%\",\"|\",\"delta\",\"|\",";
    $format_header .= $t x @sect . "\n";
    $format_header .= ".\n";

    my $format_mod = "format MODULE = \n";
    $format_mod .= "| @" . "<"x($col_mod_length - 3) . " |";
    $t = "@" . "#"x($col_bytes_length - 2) . " |@" . "#"x($col_percent_length - 4) . ".# |@ @" . "#"x($col_delta - 4) . " |";
    $format_mod .= $t x @sect . "\n";
    $format_mod .= "\$mm";
    for ($t = 0 ; $t < @sect ; $t++) {
        $format_mod .= ",\$val[" . (4*$t) . "],\$val[" . (4*$t + 1) . "],\$val[" . (4*$t + 2) . "],\$val[" . (4*$t + 3) . "]";
    }
    $format_mod .= "\n.\n";

    # Define variables used in format before loding them
    my ($mm, @val);

    eval $format_header;
    $~ = "HEADER";
    $mm="$name";
    if ($size) {
        $mm .= " ($size Bytes)";
    }
    print $line_break;
    write;
    print $line_break;

    eval $format_mod;
    $~ = "MODULE";

    my $mods = $mem->{mod};
    my $mods_ref = $mem_ref->{mod};
    foreach my $m (sort {$mods->{$b} <=> $mods->{$a}} keys %{$mods}) {
        @val = ();
        &push_val(\@val, $mods->{$m}, $mem->{used}, $mods_ref->{$m} || 0);
        delete $mods_ref->{$m};

        foreach my $s (sort keys %{$mem->{sections}}) {
            my $v = $mem->{sections}->{$s}->{mod}->{$m} || 0;
            my $r = $mem_ref->{sections}->{$s}->{mod}->{$m} || 0;
            &push_val(\@val, $v, $mem->{sections}->{$s}->{used}, $r);
        }
        #for some reason using $m in the format doesn't work...
        $mm = $m;
        write;
    }

    # process module only present in reference file
    foreach my $m (sort {$mods_ref->{$b} <=> $mods_ref->{$a}} keys %{$mods_ref}) {
        @val = ();
        &push_val(\@val, 0, 0, $mods_ref->{$m});

        # Assume same sections between mem and mem_ref
        foreach my $s (sort keys %{$mem->{sections}}) {
            my $r = $mem_ref->{sections}->{$s}->{mod}->{$m} || 0;
            &push_val(\@val, 0, 0, $r);
        }

        $mm = $m;
        write;
    }

    $mm = "TOTAL";
    @val=();
    &push_val(\@val, $mem->{used}, $size, $mem_ref->{used});

    foreach my $s (sort keys %{$mem->{sections}}) {
        &push_val(\@val, $mem->{sections}->{$s}->{used}, $size,
                  $mem_ref->{sections}->{$s}->{used} || 0);
    }
    print $line_break;
    write;
    print $line_break;

    print "\n";
}
