#!/usr/bin/env python
#########################################################################################
#
# @file macsw_build.py
#
# @brief Script allowing to compile SW target (MACSW, DRV)
#
# Version History
#    v0.1.0 - 2016/01/12 - Initial Draft (Simple MAC SW compiling).
#    v0.1.1 - 2016/01/13 - Add command option and use colorama module.
#    v0.2.0 - 2016/01/14 - Parse aps-gcc linkinfo files (SharedRAM + Code).
#                          Add option for building only desired targets.
#                          Add quiet mode and run resume
#                          Delete logs older than 15 days.
#    v0.2.1 - 2016/01/15 - Add support of ARM7 TDMI CPU
#    v1.0.0 - 2016/XX/XX - Initial Release.
#
# Copyright (C) RivieraWaves 2015-2019
#
# $Rev$
#
#########################################################################################

#########################################################################################
##@addtogroup Jenkins
# @ingroup ROOT
# @brief Jenkins run script
#
# @{
#########################################################################################

#########################################################################################
#         Imports ---
#########################################################################################

import getopt
import re
import sys
import os
import subprocess
import platform
import exceptions
import traceback
import csv
import time
import shutil
from colorama import init, Fore, Back, Style

#########################################################################################
#         Script Definitions ---
#########################################################################################

VERSION_NB   = "0.2.1"
PRODUCT_NAME = "MAC Compilation Tool"
SCRIPT_NAME  = PRODUCT_NAME + " v" + VERSION_NB
LINE_LEN     = 100
# Age after which a log folder is considered as obsolete and can be automatically removed
# Given in days
LOG_MAX_DAY  = 15


HEADER_STR   = """
                                  /------------------------------\\
>--------------------------------<   """+SCRIPT_NAME+"""  >--------------------------------<
                                  \\------------------------------/
               """

USAGE_STR = """
    Product : """ + PRODUCT_NAME + """
    Platform: Win32/64, Linux32/64, MacOSX
    Version : """ + VERSION_NB + """

>--------------------------------------------------------------------------------------------------<

    +----------------------+
    | RUNTIME OPTIONS      |
    +----------------------+
    -h           : Display all the tool options

    -t TARGETS   : List of targets to be built, separated by a slash ONLY
                   Example: TARGET_1,TARGET_2

    -j           : Indicate that the tool is launched by Jenkins. Generate a last_result.xml file
                   in the logs folder

    -p           : Print commands that are used for cleaning, compilation,...

    -q           : Activate the quiet mode

    -m           : Extract memory information for each traget for which these information are
                   available (currently only aps-gcc)

    -s           : Stop on first error

>--------------------------------------------------------------------------------------------------<
    (c) Copyright - RivieraWaves 2015-2019.  All rights reserved.
    """

#########################################################################################
#         Definitions ---
#########################################################################################

# Status Codes ---
PASSED = 0
FAILED = PASSED + 1

#########################################################################################
#         Global Variables ---
#########################################################################################

# Target products supported by the tools and their associated print name
supp_product = {
           "lmac"    : 'SoftMAC',
           "fmac"    : 'FullMAC',
           "fhost"   : 'FullHost',
}

# Target CPUs supported by the tools and their associated print name
supp_cpu = {
           "aps3"   : 'Cortus APS3',
           "arm"    : 'ARM7 TDMI',
           "tl4xx"  : 'CEVA TeakLite 4',
}

# Target PHYs supported by the tools and their associated print name
supp_phy = {
           "trident"   : 'Trident',
           "karst"     : 'Karst/Cactus',
}

# Supported status codes and associated print strings
supp_status = {
           PASSED   : 'PASSED',
           FAILED   : '        FAILED',
}

# Paths Dictionary
dict_path = {}

dict_path["base_dir"]        = os.getcwd()
dict_path["csv_dir"]         = os.path.join(dict_path["base_dir"], "csv")
dict_path["logs_dir"]        = os.path.join(dict_path["base_dir"], "logs")
dict_path["curr_logs_dir"]   = None
dict_path["fw_build_csv"]    = os.path.join(dict_path["csv_dir"], "fw_config.csv")
dict_path["drv_build_csv"]   = os.path.join(dict_path["csv_dir"], "drv_config.csv")
dict_path["macsw_dir"]       = os.path.join(dict_path["base_dir"], "../../..")
dict_path["macsw_conf_dir"]  = os.path.join(dict_path["macsw_dir"], "config")
dict_path["macsw_build_dir"] = os.path.join(dict_path["macsw_dir"], "build")
dict_path["result_xml"]      = "result.xml"
dict_path["run_log"]         = "log.txt"

# Environment Dictionary
env = {}

env["result_xml"]  = None
env["log"]         = None
env["jenkins"]     = False
env["print_cmd"]   = False
env["target_list"] = None
env["quiet"]       = False
env["get_mem"]     = False
env["stop_on_err"] = False

#########################################################################################
#         Print Definitions ---
#########################################################################################

# Color Codes ---
passed_cc = Fore.GREEN + Style.BRIGHT
failed_cc = Fore.RED + Style.BRIGHT
title_cc  = Fore.YELLOW + Style.BRIGHT

# Alignment Codes ---
LEFT_ALIGN = 0
RIGHT_ALIGN = LEFT_ALIGN + 1

#########################################################################################
#         Print Functions ---
#########################################################################################

def print_log_line(line):
    env["log"].write(line)
    env["log"].write("\n")

def print_console_line(line, force = False):
    if ((env["quiet"] == False) or (force == True)):
        print line

def print_header(log = False):
    if (log):
        print_log_line(HEADER_STR)
    else:
        print_console_line(HEADER_STR, force = True)

def print_usage():
    print_console_line(USAGE_STR, force = True)

def print_section(color_code = '', force = False):
    line = "+" + (LINE_LEN - 2) * "-" + "+"

    print_console_line(color_code + line, force = force)
    print_log_line(line)

def print_line(line, split = False, color_code = '', align = LEFT_ALIGN, force = False):
    # Remaining line length ("| " and " |" removed)
    rem_len = LINE_LEN - 4
    # List of sublines to be printed
    lines = []

    while (True):
        # Verify that line length is valid
        if (len(line) > rem_len):
            # Split line
            subline = line[0:rem_len]
            line = line[rem_len:]
            # Keep subline
            lines.append(subline)

            if (not split):
                break
        else:
            lines.append(line)
            break

    for subline in lines:
        # White space needed
        wh_spaces = (rem_len - len(subline)) * " "

        if (align == LEFT_ALIGN):
            packed_line = "| " + subline + wh_spaces + " |"
        elif (align == RIGHT_ALIGN):
            packed_line = "| " + wh_spaces + subline + " |"

        print_console_line(color_code + packed_line, force = force)
        print_log_line(packed_line)

def print_empty_line(force = False):
    print_console_line("", force = force)
    print_log_line("")

def print_title(title, force = False):
    print_empty_line(force = force)
    print_section(color_code = title_cc, force = force)
    print_line(title, color_code = title_cc, force = force)
    print_section(color_code = title_cc, force = force)

def print_exception():
    print_line("")
    print_line("An unhandled exception occured, here's the traceback!")
    traceback.print_exc()

def print_cmd(cmd):
    if (env["print_cmd"] == True):
        print_line(cmd)

def print_error(error):
    print_line("ERROR: %s" % (error), color_code = failed_cc)

def print_status(name, status):
    nb_wh_spaces = (LINE_LEN / 2) - len(name)
    wh_spaces = nb_wh_spaces * " "
    ext_name = name + wh_spaces

    if (status == PASSED):
        print_line(ext_name + supp_status[status], color_code = passed_cc, force = True)
    elif (status == FAILED):
        print_line(ext_name + supp_status[status], color_code = failed_cc, force = True)

#########################################################################################
#         Jenkins Oriented Functions ---
#########################################################################################

#########################################################################################
## @brief Replace forbidden XML character with corresponding "predefined entities"
#########################################################################################
def result_xml_escape_characters(line):
    line=line.replace("&","&amp;")
    line=line.replace(">","&gt;")
    line=line.replace("<","&lt;")
    line=line.replace("'","&apos;")
    line=line.replace("\"","&quot;")
    return line

#########################################################################################
## @brief Print test case result in XML format
#########################################################################################
def update_result_xml(dict_target, classname, passed):
    if (env["result_xml"] != None):
        env["result_xml"].write("<testcase classname=\"" + classname + "\" name=\"" + dict_target["name"] + "\">\n")

        if (not passed) :
            env["result_xml"].write("<failure type=\"RunError\">\n")

            # Path to the generated log file
            if (classname == "fw_build"):
                log_path = os.path.join(dict_path["curr_logs_dir"], "log_fw_build_" + dict_target["name"] + ".txt")

                # Verify that log file exists
                if os.path.isfile(log_path):
                    # Open log file
                    log_file = open(log_path, 'rb')

                    for line in log_file.readlines():
                        env["result_xml"].write(result_xml_escape_characters(line))

                    # Close log file
                    log_file.close()

            env["result_xml"].write("</failure>\n")
        env["result_xml"].write("</testcase>\n")

#########################################################################################
## @brief
#########################################################################################
def result_xml_open():
    if (env["jenkins"]):
        # Create and initialize the XML result file
        dict_path["result_xml"] = os.path.join(dict_path["curr_logs_dir"], dict_path["result_xml"])
        env["result_xml"] = open(dict_path["result_xml"], 'w')
        env["result_xml"].write('<testsuite>\n')

#########################################################################################
## @brief
#########################################################################################
def result_xml_close():
    if (env["result_xml"]):
        env["result_xml"].write('</testsuite>\n')
        env["result_xml"].close()

#########################################################################################
## @brief
#########################################################################################
def result_xml_copy():
    if (env["result_xml"] != None):
        shutil.copy(dict_path["result_xml"], os.path.join(dict_path["logs_dir"], "last_result.xml"))

#########################################################################################
#         Memory Information Functions ---
#########################################################################################

#########################################################################################
## @brief
#
# Sample of SHAREDRAM Section:
#
# SHAREDRAM       0x60000000     0xa9a4
#                 0x60000000                _sshram = .
#  *ipc_shared.o(COMMON)
#  COMMON         0x60000000      0xe44 C:\Workspace\WLAN\macsw\build\lmac_aps3_trident\aps-gcc\driver\ipc\ipc_shared.o
#                 0x60000000                ipc_shared_env
#  *hal_desc.o(COMMON)
#  COMMON         0x60000e44     0x62a0 C:\Workspace\WLAN\macsw\build\lmac_aps3_trident\aps-gcc\hal\hal_desc.o
#                 0x60000e44                tx_hw_desc2
#                 0x60002444                tx_hw_desc1
#                 0x60003a44                rx_dma_hdrdesc
#                 0x600040a4                tx_hw_desc3
#                 0x60004364                tx_hw_desc0
#                 0x60004624                debug_info
#                 0x60004c80                rx_payload_desc
#  *txl_buffer_shared.o(COMMON)
#
#########################################################################################
def extract_aps_sram_info(linkinfo_path):
    sram_info = None

    # Check that provided file exists
    if (os.path.isfile(linkinfo_path)):
        # Content will be [total_size, []]
        sram_sect_info = None

        # Open link info file
        linkinfo = open(linkinfo_path, 'rb')

        # Go through the link info file
        for line in linkinfo:
            # Checked if SHAREDRAM section has already been found
            if (sram_info == None):
                # Look for start of SHAREDRAM section, total SRAM size is computed on this line
                sram_pattern = re.compile("^SHAREDRAM\s+")
                result = sram_pattern.match(line)

                if (result != None):
                    sram_info = []
            else:
                if (line in ["\n", "\r\n"]):
                    # All SHAREDRAM information have been extracted, store current info...
                    if (sram_sect_info != None):
                        sram_info.append(sram_sect_info)
                    # .. and stop reading file
                    break

                # Look for SHAREDRAM subsections
                common_pattern = re.compile("^\s\*(.*)\.o\(COMMON\)")
                result = common_pattern.match(line)

                if (result != None):
                    # Check if we were reading a subsection
                    if (sram_sect_info != None):
                        sram_info.append(sram_sect_info)

                    # Store file name
                    sram_sect_info = [result.group(1)]
                    # And go to next line
                    continue

                # Get total subsize
                subsize_pattern = re.compile("^\s.*0x.*(0x[0-9A-Fa-f]+)\s")
                result = subsize_pattern.match(line)

                if (result != None):
                    if (sram_sect_info != None):
                        sram_sect_info.append(str(int(result.group(1), 16)))
                        # Create list for subsection
                        sram_sect_info.append([])
                        # Go to next line
                        continue

                # Get start address and structure name
                addr_name_pattern = re.compile("^\s.*(0x[0-9A-Fa-f]+)\s+([a-z0-9_]+)")
                result = addr_name_pattern.match(line)

                if (result != None):
                    # Verify that subsection has been started and that full size field has been read
                    if ((sram_sect_info != None) and (len(sram_sect_info) == 3)):
                        sram_sect_info[-1].append([result.group(2), result.group(1)])

        # Close the file...
        linkinfo.close()
    else:
        print_error("Link Info File not found")

    return (sram_info)

#########################################################################################
## @brief
#
# Sample of CODE Section:
#
# CODE            0x80001c54     0xa470
#                 0x80005800                . = (_edata + 0x80000000)
#  *fill*         0x80001c54     0x3bac 00
# [...]
#  .text.phy_locked
#                 0x80005b10       0x8a C:\Workspace\WLAN\macsw\build\lmac_aps3_trident\aps-gcc\driver\phy\phy_trident.o
#  *fill*         0x80005b9a        0x2 00
#  .text.phy_adc12b_init
#                 0x80005b9c      0x1a2 C:\Workspace\WLAN\macsw\build\lmac_aps3_trident\aps-gcc\driver\phy\phy_trident.o
#  *fill*         0x80005d3e        0x2 00
# [...]
#  .text.la_init  0x80007298       0x5c C:\Workspace\WLAN\macsw\build\lmac_aps3_trident\aps-gcc\driver\la\la.o
#                 0x80007298                la_init
#  .text.la_stop  0x800072f4       0x14 C:\Workspace\WLAN\macsw\build\lmac_aps3_trident\aps-gcc\driver\la\la.o
#                 0x800072f4                la_stop
#########################################################################################
def extract_aps_code_info(linkinfo_path):
    code_info = None

    # Check that provided file exists
    if (os.path.isfile(linkinfo_path)):
        # Content will be [func_name, file_name, func_size]
        func_info = None

        # Open link info file
        linkinfo = open(linkinfo_path, 'rb')

        # Go through the link info file
        for line in linkinfo:
            # Checked if CODE section has already been found
            if (code_info == None):
                # Look for start of CODE section, total SRAM size is computed on this line
                code_pattern = re.compile("^CODE.*")
                result = code_pattern.match(line)

                if (result != None):
                    # Initialize code_info
                    code_info = []
            else:
                # Detect end of CODE section (does not start with a space character)
                eos_patter = re.compile("^\s.*")
                result = eos_patter.match(line)

                if (result == None):
                    # Store parsing file
                    break

                # Look for function subsections
                # Important Note - Following function name length, function size can be on same line, start by
                # checking this case (see function header for example)
                func_pattern_full = re.compile("^\s\.text\.([a-z0-9_]+)\s+0x[a-f0-9]+\s+(0x[a-f0-9]+)\s(.+)\.o")
                result = func_pattern_full.match(line)

                if (result != None):
                    # Store all parameters
                    func_info = [result.group(1),
                                 result.group(3).split(os.sep)[-1],
                                 str(int(result.group(2), 16))]
                    code_info.append(func_info)
                    func_info = None
                    # Go to next line
                    continue

                # Case where function name and function information are on two lines - Get function name
                func_pattern = re.compile("^\s\.text\.([a-z0-9_]+)")
                result = func_pattern.match(line)

                if (result != None):
                    func_info = [result.group(1)]
                    # Go to next line
                    continue

                if (func_info != None):
                    # Get function size and file name
                    func_info_pattern = re.compile("^\s+0x[0-9a-f]+\s+(0x[0-9a-f]+)\s(.+)\.o")
                    result = func_info_pattern.match(line)

                    if (result != None):
                        # Split .o file address  with / or \ character
                        func_info.append(result.group(2).split(os.sep)[-1])
                        func_info.append(str(int(result.group(1), 16)))
                        code_info.append(func_info)
                        func_info = None

        # Close the file...
        linkinfo.close()
    else:
        print_error("Link Info File not found")

    return (code_info)

#########################################################################################
## @brief
# @param[in] sram_info is a list whose elements are:
#               -> Elem 0: File Name - String
#               -> Elem 1: SRAM Section Size - String
#               -> Elem 2: List whose elements are:
#                    -> Elem 0: SRAM Section Element Name
#                    -> Elem 1: SRAM Section Element Start Address
#########################################################################################
def export_sram_info(sram_info, dict_target):
    if (sram_info != None):
        # Create and open CSV file
        sram_csv = open(os.path.join(dict_path["curr_logs_dir"], dict_target["name"] + "_sram_info.csv"), 'wb')
        sram_csv_writer = csv.writer(sram_csv, delimiter = ',')

        # Format SRAM Information before printing them
        format_sram_info = []

        for elem in sram_info:
            last_address = None
            start_elem_addr = None

            # Go to next element if section is empty
            if (len(elem) == 1):
                continue

            for subelem in elem[2]:
                # Store start address of first element
                if (start_elem_addr == None):
                    start_elem_addr = subelem[1]

                # Create a new element whose format will be [Section Element Name, File Name, Section Element Size]
                # Current element Size cannot be computed right now
                new_elem = [subelem[0], elem[0]]

                start_address = subelem[1]
                # We have start address of new element, we can deduce size of previous element
                if (last_address != None):
                    prev_elem_size = int(start_address, 16) - int(last_address, 16)
                    # Insert computed size in previous element
                    format_sram_info[-1].append(str(prev_elem_size))

                # Store start address of current element
                last_address = start_address

                format_sram_info.append(new_elem)

            # We can now deduce size of last element
            last_elem_size = int(start_elem_addr, 16) + int(elem[1]) - int(last_address, 16)
            # Insert computed size in previous element
            format_sram_info[-1].append(str(last_elem_size))

        # Copy all read information
        for sram in format_sram_info:
            sram_csv_writer.writerow(sram)

        # Close CSV File
        sram_csv.close()

#########################################################################################
## @brief
# @param[in] code_info is a list of function elements.
#            Each function element is also a list whose elements are:
#                -> Elem 0: Function Name - String
#                -> Elem 1: File Name - String
#                -> Elem 2: Function Size - String
#########################################################################################
def export_code_info(code_info, dict_target):
    if (code_info != None):
        # Create and open CSV file
        func_csv = open(os.path.join(dict_path["curr_logs_dir"], dict_target["name"] + "_code_info.csv"), 'wb')
        func_csv_writer = csv.writer(func_csv, delimiter = ',')

        # Copy all read information
        for func in code_info:
            func_csv_writer.writerow(func)

        # Close CSV File
        func_csv.close()

#########################################################################################
## @brief
#########################################################################################
def read_aps_link_info(dict_target):
    # Retrieve path of the generated linkinfo.aps-gcc.txt file
    build_folder = dict_target["product"] + "_" + dict_target["cpu"] + "_" + dict_target["phy"]
    linkinfo_path = os.path.join(dict_path["macsw_build_dir"], build_folder, "linkinfo.aps-gcc.txt")

    # Read the file
    sram_info = extract_aps_sram_info(linkinfo_path)
    code_info = extract_aps_code_info(linkinfo_path)

    # Export the obtained information to CSV files
    export_sram_info(sram_info, dict_target)
    export_code_info(code_info, dict_target)

#########################################################################################
## @brief Call the script in charge of generating the Memory information file for Jenkins
#########################################################################################
def extract_meminfo(dict_target):
    if (platform.system() == "Linux"):
        build_folder = dict_target["product"] + "_" + dict_target["cpu"] + "_" + dict_target["phy"]

        os.chdir(dict_path["base_dir"])

        # Build the mem_info command
        mem_info_cmd = [
                        os.path.join(dict_path["base_dir"], "build_mem_info.sh"),
                        dict_path["macsw_dir"],
                        dict_target["name"],
                        build_folder,
                        dict_path["curr_logs_dir"],
                       ]

        build_mem_info = subprocess.Popen(' '.join(mem_info_cmd), stdout = subprocess.PIPE,
                                                                  stderr = subprocess.PIPE,
                                                                  shell = True)
        build_mem_info.wait()

#########################################################################################
#         FW Compilation Functions ---
#########################################################################################

#########################################################################################
## @brief
#########################################################################################
def print_compile_resume(results):
    print_title("FW Compilation")

    for result in results:
        print_status(result[0], result[1])

    print_section()

#########################################################################################
## @brief Perform target clean
#########################################################################################
def clean_fw(dict_target):
    # Go to the MACSW configuration folder
    os.chdir(dict_path["macsw_conf_dir"])
    # Path to the generated log file
    log_path = os.path.join(dict_path["curr_logs_dir"], "log_fw_clean_" + dict_target["name"] + ".txt")

    clean_cmd = ["python",                 # Run python
                 "../tools/scons.py .",    # Path to scons script
                 "-c",                     # Clean option
                ]
    clean_cmd = ' '.join(clean_cmd)
    print_cmd(clean_cmd)

    out = open(log_path, 'w')
    clean = subprocess.Popen(clean_cmd, stdout=out, stderr=out, shell=True)
    clean.wait()
    out.close()

    # Remove created log
    os.remove(log_path)

    # Return to base directory
    os.chdir(dict_path["base_dir"])

#########################################################################################
## @brief Perform target clean
#########################################################################################
def build_fw(dict_target):
    # Go to the MACSW configuration folder
    os.chdir(dict_path["macsw_conf_dir"])
    # Path to the generated log file
    log_path = os.path.join(dict_path["curr_logs_dir"], "log_fw_build_" + dict_target["name"] + ".txt")

    build_cmd = ["python",                     # Run python
                 "../tools/scons.py",          # Path to scons script
                 ". -j 4",                     # Compilation options
                 dict_target["options"],       # Target compilation options
                ]
    build_cmd = ' '.join(build_cmd)
    print_cmd(build_cmd)

    out = open(log_path, 'w')
    build = subprocess.Popen(build_cmd, stdout=out, stderr=out, shell=True)
    retcode = build.wait()
    out.close()

    # Remove created log if build is succesfull
    if (retcode == 0):
        os.remove(log_path)

    # Return to base directory
    os.chdir(dict_path["base_dir"])

    env["last_build_cmd"]=build_cmd
    env["last_log"]=log_path

    return (retcode == 0)

#########################################################################################
## @brief Perform target clean
#########################################################################################
def handle_fw_target(dict_target):
    print_title("Build %s FW Target" % (dict_target["name"]))

    # Print build details
    print_line("-> Product: %s" % (supp_product[dict_target["product"]]))
    print_line("-> CPU: %s" % (supp_cpu[dict_target["cpu"]]))
    print_line("-> PHY: %s" % (supp_phy[dict_target["phy"]]))
    print_line("-> Options: %s" % (dict_target["options"]), split = True)
    print_line("")

    # Clean FW ---
    clean_fw(dict_target)

    # Build FW ---
    build_passed = build_fw(dict_target)

    if (build_passed == True):
        status = PASSED
    else:
        status = FAILED

    print_section()
    print_status(dict_target["name"], status)
    print_section()

    # Print result ---
    update_result_xml(dict_target, "fw_build", build_passed)

    if ((build_passed == True) and (env["get_mem"] == True)):
        if (dict_target["cpu"] == "aps3"):
            # Extract memory information
            extract_meminfo(dict_target)

            # Read Link Info file
            read_aps_link_info(dict_target)

    return (status)

#########################################################################################
## @brief
#########################################################################################
def extract_target_info(row, list_fw_flags):
    dict_target = None

    # Ignore commented and empty lines
    commentpattern = re.compile("^(#.*|)$")
    comment        = commentpattern.match(row[0])

    if (comment == None):
        # Dictionary containing build target information
        dict_target = {}

        # Get the name of the project to build
        dict_target["name"]    = row[0]
        # Get the product to build
        dict_target["product"] = row[1]
        # Get the CPU to use
        dict_target["cpu"]     = row[2]
        # Get the PHY to use
        dict_target["phy"]     = row[3]

        # Compilation Options
        dict_target["options"] = ""
        for flag, column in list_fw_flags:
            if (row[column] != ""):
                dict_target["options"] += (flag + "=" + row[column] + " ")
            else:
                dict_target["options"] += (flag + "=off" + " ")

    return (dict_target)

#########################################################################################
## @brief
#########################################################################################
def fw_compilation():
    results = []

    try:
        # List of Compilation Flags [flag, column idx]
        list_fw_flags = None

        # Read the targets file
        fw_build_csv = open(dict_path["fw_build_csv"], 'rb')
        fw_config_reader = csv.reader(fw_build_csv, delimiter=',')

        # If quiet mode is active, start resume section
        if (env["quiet"] == True):
            print_title("FW Compilation", force = True)

        for row in fw_config_reader:
            # First row contains the compilation flags that have to be used
            if (list_fw_flags == None):
                # Initialize the list
                list_fw_flags = []
                column = 0
                for elmt in row:
                    if (elmt != ""):
                        list_fw_flags.append([elmt, column])
                    column = column + 1
                # Go to the next row
                continue

            dict_target = extract_target_info(row, list_fw_flags)

            # Check if instructions have been read on this row
            if (dict_target == None):
                continue

            # Check if tool is authorized to compile the target
            if ((env["target_list"] != None) and
                not (dict_target["name"] in env["target_list"])):
                continue

            # Compile target
            build_status = handle_fw_target(dict_target)

            # Store status
            results.append([dict_target["name"], build_status])

            if build_status == FAILED and env["stop_on_err"]:
                print "Error while executing: \n " + env["last_build_cmd"]
                print "Check log: " + env["last_log"]
                break

        # If quiet mode is active, resume section is already printed, so close it
        if (env["quiet"] == True):
            print_section(force = True)
        else:
            # Else print the resume
            print_compile_resume(results)

        # Close opened file
        fw_build_csv.close()
    except:
        print_exception()
        pass

#########################################################################################
#         Linux Driver Compilation Functions ---
#########################################################################################

#########################################################################################
#         Log Functions ---
#########################################################################################

#########################################################################################
## @brief
#########################################################################################
def log_create():
    # Create logs folder if does not exist
    if (not os.path.exists(dict_path["logs_dir"])):
        os.makedirs(dict_path["logs_dir"])

    # Create folder that will contain logs for current run
    year, mth, day, h, m, s, x, y, z = time.gmtime()
    today = ('%4d-%02d-%02d_%02dh%02dm%02ds') % (year, mth, day, h, m, s)
    dict_path["curr_logs_dir"] = os.path.join(dict_path["logs_dir"], today)
    os.makedirs(dict_path["curr_logs_dir"])

    # Create log file
    dict_path["run_log"] = os.path.join(dict_path["curr_logs_dir"], dict_path["run_log"])
    env["log"] = open(dict_path["run_log"], 'w')
    print_header(log = True)

#########################################################################################
## @brief
#########################################################################################
def log_close():
    env["log"].close()

#########################################################################################
## @brief
#########################################################################################
def delete_old_logs():
    # Go to logs directory, if exists
    if (os.path.isdir(dict_path["logs_dir"])):
        # Get list of existing directories
        listdir = os.listdir(dict_path["logs_dir"])

        # Time value are given in seconds
        now_time   = time.time()
        limit_time = now_time - LOG_MAX_DAY * 24 * 3600

        for f in listdir:
            fullpath = os.path.join(dict_path["logs_dir"], f)
            log_time = os.stat(fullpath).st_ctime

            # Remove folder if too old
            if os.path.isdir(fullpath) and (log_time < limit_time):
                try:
                    shutil.rmtree(fullpath)
                except:
                    print_exception()
                    pass

#########################################################################################
#         Scripts Functions ---
#########################################################################################

#########################################################################################
## @brief
#########################################################################################
def check_options():
    try:
        opts, args = getopt.getopt(sys.argv[1:], "hjpt:qms", ["help"])
    except getopt.GetoptError:
        # Print help information and exit:
        print_usage()
        return (False)

    # Go through the provided options
    for o, a in opts:
        if (o == "-h") | (o == "--help"):
            print_usage()
            return (False)

        elif (o == "-j"):
            env["jenkins"] = True

        elif (o == "-p"):
            env["print_cmd"] = True

        elif (o == "-t"):
            env["target_list"] = a.split(",")
            # Check if there is at list one target to compie
            if (len(env["target_list"]) == 0):
                # At least one target must be provided
                print_usage()
                return (False)

        elif (o == "-q"):
            env["quiet"] = True

        elif (o == "-m"):
            env["get_mem"] = True

        elif (o == "-s"):
            env["stop_on_err"] = True

    return (True)

#########################################################################################
## @brief Main function of the application, get user runtime arguments, creates
#         the environment (devices, connections...) and runs the list of TCs.
#########################################################################################
def main():
    print_header()

    # Init Colorama Module - Allow to reset color allocation after each print
    init(autoreset = True)

    #===============================================================================
    # Check provided options ---
    #===============================================================================
    if (not check_options()):
        sys.exit(0)

    #================================================================================
    # Configure Log Folder and Log File ---
    #================================================================================
    log_create()

    #================================================================================
    # Delete logs older than 1 month ---
    #================================================================================
    delete_old_logs()

    #================================================================================
    # Open result.xml file for Jenkins ---
    #================================================================================
    result_xml_open()

    #===============================================================================
    # FW Compilation ---
    #===============================================================================
    fw_compilation()

    #================================================================================
    # Close all remaining opened files ---
    #================================================================================
    result_xml_close()
    log_close()

    #================================================================================
    # Copy result.xml file for Jenkins ---
    #================================================================================
    result_xml_copy()

    # And leave the script and python
    sys.exit(0)

#########################################################################################
# Execution of main(): in Python files (which are modules), if a module is not imported
# and it launched directly, the main section of the file will run. When a module is
# imported , this section is ignored.
#########################################################################################
if __name__ == "__main__":
    main()

# @} module
