########################################################################################
#
#
# Copyright (C) RivieraWaves 2009-2019
#
# Description:
# Tool to read out register information from excel sheet and generate source header files
#
########################################################################################

usage_doc = """
Synopsis:
    reg_xls2h.py xlsname
          to test the validity of a register definition

    reg_xls2h.py [--verbose] [--simu] [--short] [--sheet=N] xlsname prefix regaddr outfile
          to generate the header file of a HW block

where:
    --verbose to turn on verbose mode
    --simu means that the file must be generated for simulation environment
    --short means that the file must generate short macros (field name
            not prefixed with register name)
    --sheet=N means that the XLS file provided is in the MODEM register definition
            format and that the sheet N must be parsed to extract the registers
    "xlsname" designates the excel file
    "prefix" designates the prefix to use for all the generated registers
    "regaddr" designates the base address of the registers
    "outfile" designates the file to generate the output to
    ""
"""


import os, sys, getopt
from common.legalexception import *
from reg_xls2h_xls import *
from reg_xls2h_h import *

class reg_xls2h:
    def __init__(self, xlsname, sheetname, access, prefix, verbose=False):
        self.verbose = verbose
        if self.verbose:
            print "  - Excel File Name : " + xlsname

        # check that the input file exists
        if not os.path.isfile(xlsname):
            raise LegalException("Error: excel file %s does not exist" % xlsname, 1)

        # create the entity to read the excel sheet
        self.xls = xls(xlsname, sheetname, access, prefix, self.verbose)

    def read_xls(self):
        self.xls.extract_all()

    def gen_header(self, prefix, regaddr, outfile, forsimu, longname):
        # create a header instance
        self.header = header(prefix, regaddr, outfile)
        # generate the file
        self.header.generate(self.xls, forsimu, longname, True)
        
    def gen_reg_grp(self, prefix, base_grp_addr):
        #list of register details in order
        reg_grp = [] 
        
        block = self.xls.blocks[0]
        
        nb_regs = block.num_regs
        
        # loop on all the registers
        for reg in block.registers:
            # if the regname is a noregister do not put it
            if reg.name.upper().find("NOREGISTER") != -1:
                continue
            
            reg_dict = {}
            reg_dict['name']   = reg.name.upper()
            reg_dict['addr']   = base_grp_addr+reg.addr
            reg_dict['width']  = reg.width
            reg_dict['desc']   = reg.desc
            reg_dict['fields'] = [] #list of fields in order
            
            if len(reg.fields) != 0:
                for field in reg.fields:
                    field_dict = {}
                    
                    field_dict['name']   = field.name
                    field_dict['offset'] = field.low_bitpos
                    field_dict['mask']   = field.mask
                    field_dict['size']   = field.width
                    field_dict['perm']   = reg.sw.lower()
                    field_dict['rst']    = field.reset
                    field_dict['desc']   = field.desc
                    
                    reg_dict['fields'].append(field_dict)
                    
            #else:
                #print 'REG WITH NO FIELDS', reg.name
            reg_grp.append(reg_dict)
        return reg_grp
        
def usage():
    sys.stdout.write(usage_doc)


def main():
    # parse the command line
    try:
        opts, args = getopt.getopt(sys.argv[1:], "", ["simu", "short", "verbose", "sheet="])
    except getopt.GetoptError:
        # print help information and exit:
        usage()
        sys.exit(2)

    verbose = False
    sheetname = "Registers"
    forsimu = False
    longname = True

    if ("--simu","") in opts:
        forsimu = True
    if ("--short","") in opts:
        longname = False
    if ("--verbose","") in opts:
        verbose = True

    for opt in opts:
        if opt[0] =="--sheet":
            sheetname = opt[1]

    if len(args) < 1:
        usage()
        sys.exit("Wrong number of arguments")

    # create the main instance
    my_reg_xls2h = reg_xls2h(args[0], sheetname, verbose)

    # read the excel sheet content
    my_reg_xls2h.read_xls()

    # check the number of parameters is correct
    if len(args) == 4:
        # generate the file only if requested
        my_reg_xls2h.gen_header(args[1], args[2], args[3], forsimu, longname)
    elif len(args) != 1:
        # print help information and exit:
        usage()
        sys.exit("ERROR: unexpected number of parameters")


if __name__ == "__main__":
    try:
        main()
    except LegalException, e:
        print e
    except:
        raise
