"""SCons.Tool.gcc

Tool-specific initialization for APS3-GCC

There normally shouldn't be any need to import this module directly.
It will usually be imported through the generic SCons.Tool.Tool()
selection method.

"""

#
# Copyright (c) 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008 The SCons Foundation
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
# KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
# WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

__revision__ = ""

import os
import os.path
import string
import sys

import SCons.Action
import SCons.Builder
import SCons.Defaults
import SCons.Tool
import SCons.Util

def exists(env):
    return find(env)

# This is what we search for to find aps3-gcc:
key_program = 'aps3-gcc'

def find(env):
    return env.WhereIs(key_program) or SCons.Util.WhereIs(key_program)

def generate(env):

    # Mostly it is the same as gcc and friends...
    gnu_tools = ['gcc', 'g++', 'gnulink', 'ar', 'gas']
    for tool in gnu_tools:
        SCons.Tool.Tool(tool)(env)

    # find where APS3 binaries are hidden
    aps3_gcc_path = find(env)
    if aps3_gcc_path:
        dir = os.path.dirname(aps3_gcc_path)
        env.PrependENVPath('PATH', dir )
    
    # Include mini Cortus C library for link
    if(os.path.exists(os.path.join(dir, '../Clib/Release'))):
        env.Append(LIBPATH = [os.path.join(dir, '../Clib/Release')])
    else:
        print "\n Warning : Cortus mini library (Clib) not found"
        print "\t=> only standard C library will be used for link"
        print "\t=> to use it, Clib folder has to be copied into toolchain base directory\n"

    # retrieve the LD_LIBRARY_PATH
    LD_LIBRARY_PATH= os.getenv('LD_LIBRARY_PATH')
    if LD_LIBRARY_PATH != 'none':
      env['ENV']['LD_LIBRARY_PATH']=LD_LIBRARY_PATH    
        
    # retrieve the parameters of the compilation
    env['APS3OPTIM'] = '${_concat("-O", OPTIM, "", __env__)}'

    #... but a few things differ:
    #  in case we use linux, we need to preload the libstdc++ in order
    #  to avoid conflicting system libraries
    if 'linux' in sys.platform:
      cortus_home= `os.environ.get("CORTUS_HOME")`
      env['CC'] = 'LD_PRELOAD=' + cortus_home + '/lib/libstdc++.so.6 aps3-gcc'
    else:
      env['CC'] = 'aps3-gcc'
      
    env['AS'] = 'aps3-as'
    env['LINK'] = 'aps3-gcc'

    
    # add some flags common to C and C++ and to assembly
    env['CCCOM']     = '$CC -c $APS3OPTIM $_CCCOMCOM $LIBINC $CCFLAGS $TRACEFLAGS -o$TARGET $SOURCES'
    env['ASCOM']     = '$AS $ASFLAGS -o $TARGET $SOURCES'

    # add the info file output in the link flags
    env['LINKINFO'] = '${_concat("-Wl,-Map,", LINKINFOFILE, "", __env__)}'
    # add the linker script at the end of the link command line
    env['LINKMAP'] = '${_concat("-Wl,-T,", LINKMAPFILE, "", __env__)}'
    env.Append(_LIBFLAGS = " $LINKINFO $LINKMAP")

    # Some setting from the platform also have to be overridden:
    env['OBJSUFFIX'] = '.o'
    env['LIBPREFIX'] = 'lib'
    env['LIBSUFFIX'] = '.a'
    env['PROGSUFFIX'] = '.elf'

    
    env['LINKFLAGS'] = SCons.Util.CLVar('$LINKFLAGS -shared')
    env['LINKCOM']   = '$LINK $_LIBDIRFLAGS $APS3OPTIM $LINKFLAGS $_LIBFLAGS -o$TARGET $SOURCES'
