"""SCons.Tool.gcc

Tool-specific initialization for Risc-V

There normally shouldn't be any need to import this module directly.
It will usually be imported through the generic SCons.Tool.Tool()
selection method.

"""

#
# Copyright (c) 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008 The SCons Foundation
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
# KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
# WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

__revision__ = ""

import os
import os.path

import SCons.Tool

def exists(env):
    return find(env)

def find(env):
    try:
        riscv_path = os.environ['RISCV']
    except KeyError:
        print "No environment variable RISCV defined"
        raise
    return riscv_path

def generate(env):

    # Load GNU tools default configuration
    gnu_tools = ['gcc', 'gnulink', 'ar', 'gas']
    for tool in gnu_tools:
        SCons.Tool.Tool(tool)(env)

    # retrieve the riscv install directory, and add it to bin
    riscv_path = find(env)
    env.PrependENVPath('PATH', os.path.normpath(os.path.join(riscv_path, './bin')))

    #=====================================================================================
    # Tools to launch
    #=====================================================================================
    env['CC'] = "riscv32-unknown-elf-gcc"
    env['AS'] = "riscv32-unknown-elf-gcc"
    env['LINK'] = "riscv32-unknown-elf-gcc"
    env['AR'] = "riscv32-unknown-elf-ar"

    #=====================================================================================
    # File extensions
    #=====================================================================================
    env['OBJSUFFIX'] = '.o'
    env['PROGSUFFIX'] = '.elf'
    env['LIBPREFIX'] = 'lib'
    env['LIBSUFFIX'] = '.ld'

    #=====================================================================================
    # Tool options
    #=====================================================================================
    env['MACH'] = '-march=rv32imc -mabi=ilp32'

    #=====================================================================================
    # Tools command line formation
    #=====================================================================================
    env['CCCOM'] = '$CC -c $MACH ' \
                 + '-O$OPTIM $LIBINC $CCFLAGS $TRACEFLAGS $_CCCOMCOM -o$TARGET $SOURCES'

    env['ASCOM'] = '$AS -c $MACH $ASFLAGS $_CCCOMCOM -o$TARGET $SOURCES'

    env['LINKCOM'] = '$LINK $_LIBDIRFLAGS $MACH $LINKFLAGS ' \
                   + '-Wl,-Map,"$LINKINFOFILE" -Wl,-T,"$LINKMAPFILE" -o$TARGET $SOURCES'

    # Call command with potential very long command line using env['TEMPFILE'].
    # This is set to TempFileMunge which will:
    # - Check if command line is longer than env['MAXLINELENGTH']
    # - If not then it simply calls the command
    # - If yes, it dumps all the parameters inside in a temporary file (env['TEMPFILENAME'])
    # and then call the command using env['TEMPFILEOPT'] as parameter (set to @${TEMPFILENAME})
    # In practice it will call 'riscv32-unknown-elf-gcc @</path/to/tmpfile>'
    env['LINKCOM'] = "${TEMPFILE('" + env['LINKCOM'] + "')}"
    env['ARCOM'] = "${TEMPFILE('" + env['ARCOM'] + "')}"
