#!/usr/bin/env perl

use strict;
use Getopt::Long;
use File::Basename;
use Cwd qw(abs_path);

my %ctxts;
my %vifs;
my $outfile;
my $out_fh;
my $outdir = ".";
my $plot;
my $plot_exportdir;
my $plot_start;
my $plot_end;
my $plot_dur;
my $suffix = ".txt";
my $no_connless_ctxt = 0;
my $first_connless_ctxt = 3;

my %ctxt_print = (
    "status" => { 0 => { idx => 2,
                         name => "NOT_SCHEDULED",
                         color => "rgbcolor \"#000000\""},
                  1 => { idx => 3,
                         name => "NOT_PROG",
                         color => "rgbcolor \"#dd0000\""},
                  2 => { idx => 4,
                         name => "GOT_IDLE",
                         color => "rgbcolor \"#F59B26\""},
                  3 => { idx => 5,
                         name => "WAITING_NOA_CFM",
                         color => "rgbcolor \"#cc0066\""},
                  4 => { idx => 6,
                         name => "WAITING_END",
                         color => "rgbcolor \"#009900\""},
                  5 => { idx => 7,
                         name => "PRESENT",
                         color => "rgbcolor \"#0000ee\""},
    },
);


my %vif_print = (
    "tbtt" => { idx => 2,
                color => "rgbcolor \"#006600\""},
    "tbtt_skip" => { idx => 3,
                     color => "rgbcolor \"#f70404\""},
    "tbtt_miss" => { idx => 4,
                     color => "rgbcolor \"#0404f7\""},
    "traffic" => { idx => 5,
                   color => "rgbcolor \"#000000\"" },
    "noa_0" => { idx => 6,
                 color => "rgbcolor \"#f97e10\"" },
    "noa_1" => { idx => 7,
                 color => "rgbcolor \"#297ed0\"" },
    "noa_2" => { idx => 8,
                 color => "rgbcolor \"#99ff10\"" },
    "noa_3" => { idx => 9,
                 color => "rgbcolor \"#107ef9\"" },
    "beacon" => { idx => 10,
                  color => "rgbcolor \"#FFCD00\""}
    );


sub help {
    return "
usage: $0 --out[=<dir>] [--plot[=<dir>]] <trace_file>

Trace file must be a decoded fw trace generated with STATUS and/or TBTT filters enabled on CHAN component.
For P2P interface, enabling NOA filter on P2P component will provide NOA information in the graph.

e.g. fw_trace_level CHAN+=STATUS/TBTT P2P+=NOA


--out[=<dir>] : specify output dir for parsed and data files
                if not set, output in current directory
                if set without dirname, directory <trace_file>-chan is created
--plot[=<dir>] : Run gnuplot. If specified generate png file in <dir> instead of out

--start_plot=<start time>   : Timestamp at which the graph starts
                              default: first timestamp in the trace.
--end_plot=<start end>      : Timestamp at which the graph ends.
                              default: last timestamp in the trace.
--dur_plot=<duration in s>  : Maximum duration of one graph. If time between start and end timestamp
                              is greater that the limit set, several graphs are genertated.
                              default: timestamp end - timestamp start.
--no_connless               : Do not show connection less channel-contexts (i.e. SCAN and ROC) in
                              graph.
";
}

# read parameters
GetOptions ("plot:s" => \$plot,
            "out:s" => \$outdir,
            "start_plot:f" => \$plot_start,
            "end_plot:f" => \$plot_end,
            "dur_plot:f" => \$plot_dur,
            "no_connless" => \$no_connless_ctxt,
            "help" => sub { print &help ; exit 0 ;});

my $file=shift||"-";
my $filename = basename($file);

if ( $outdir eq "" ) {
    if ($file eq "-") {
        $outdir=".";
    } else {
        $outdir = dirname($file);
    }

    if ($filename =~ /(.*)\.([^.]+)/) {
        $outfile = "$1.chan.$2";
        $outdir = "$outdir/$1-parsed";
        $suffix = "\\.$2";
    } else {
        $outfile = "$file.chan.txt";
        $outdir = "$outdir/$filename-parsed";
    }
} elsif (defined $outdir) {

    if ($file eq "-") {
        $outfile="chan.txt";
    } elsif ($filename =~ /(.*)\.([^.]+)/) {
        $outfile = "$1.chan.$2";
        $suffix = "\\.$2";
    } else {
        $outfile = "$filename.chan.txt";
    }
}

my $cmdfile = $outfile;
$cmdfile =~ s/\.txt/.cmd/;
my $pngfile = $outfile;
$pngfile =~ s/$suffix/.png/;

$outdir = abs_path($outdir);
if ($plot eq "" ) {
    $plot_exportdir = $outdir;
} elsif (defined $plot) {
    $plot_exportdir = abs_path($plot);
}

print "logfile = $file\noutdir  = $outdir\noutfile = $outfile\ncmdfile = $cmdfile\npngfile = $pngfile\nplotdir = $plot_exportdir\n";

if (defined $outdir) {
    if ( ! -d $outdir) {
        mkdir $outdir or die "Cannot create $outdir: $!";
    }

    #open $out_fh, ">", "$outdir/$outfile" or die "Cannot create $outdir/$outfile: $!"
}

open LOG, "$file" or die "cannot open $file: $!";

my $line;
my $timestamp;
my $timestamp_start;
my $ctxt;
my $vif;
my $max_slot=0;
my @pending;

while($line = <LOG>)
{
    my $skip = 0;
    my $flush = 0;

    if ($line =~ /^\[ *(\d+\.\d{3})(_| )(\d{3})\]/) {
        $timestamp = $1.$3;
        $timestamp_start = $timestamp unless $timestamp_start;
    }

    # for gnuplot, we need to print in order
    while(@pending && $pending[0]->{ts} < $timestamp) {
        my $pend = shift @pending;
        if (defined $pend->{ctxt}) {
            &print_ctxt_data($pend->{ctxt}, $pend->{ts}, $pend->{data});
        } elsif (defined $pend->{vif}) {
            &print_vif_data($pend->{vif}, $pend->{ts}, $pend->{data});
        }
    }

    if ($line =~ /\{CTXT-(\d+)} switch to channel freq=(\d+)MHz bw=(\d+)MHz/) {
        my ($id, $freq, $bw) = ($1, $2, $3);
        $ctxt = &get_ctxt($id, $freq, $bw);
    } elsif ($line =~ /\{CTXT-(\d+)} Update status (\d+)/) {
        my ($id, $status) = ($1, $2);
        $ctxt = &get_ctxt($id);
        my $data = {};
        my $idx;

        if (defined $ctxt->{status}) {
            $idx = $ctxt_print{status}->{$ctxt->{status}}->{idx};
            $data->{$idx} = 0;
        }
        $ctxt->{status} = $status;
        $idx = $ctxt_print{status}->{$ctxt->{status}}->{idx};
        $ctxt->{map} |= (1 << $idx);
        $data->{$idx} = 1;

        &print_ctxt_data($ctxt, $timestamp, $data);
    } elsif ($line =~ /\{VIF-(\d+)}\{CTXT-(\d+)} TBTT start \(?skip_(cnt|it)=(\d+)/) {
        my ($vid, $cid, $skip) = ($1, $2, $4);
        $ctxt = &get_ctxt($cid);
        $vif = &get_vif($vid, $ctxt);
        my $data = {};

        $data->{$vif_print{tbtt}->{idx}} = 0.8 ;
        if ($skip > 0)
        {
            $data->{$vif_print{tbtt_skip}->{idx}} = 0.8
        }

        &print_vif_data($vif, $timestamp, $data);
    } elsif ($line =~ /\{VIF-(\d+)}\{CTXT-(\d+)} BCN TimeOut.*beacon_lost = (\d+)/) {
        my ($vid, $cid, $lost) = ($1, $2, $3);
        $ctxt = &get_ctxt($cid);

        # this trace may be incorrect in standalone test, so don't use it to assign vif to ctxt
        if (defined $vifs{$vid}) {
            $vif = &get_vif($vid, $ctxt);
            my $data = {};
            $data->{$vif_print{tbtt}->{idx}} = 0;
            $data->{$vif_print{tbtt_skip}->{idx}} = 0;
            if ($lost > 0) {
                $data->{$vif_print{tbtt_miss}->{idx}} = $lost * 0.1;
            }
            &print_vif_data($vif, $timestamp, $data);
        }
    } elsif ($line =~ /\{CTXT-(\d+)} update P2P absence=(\d)/ ) {
        my ($id, $abs) = ($1, $2);
        $ctxt = &get_ctxt($id);
        &print_ctxt_data($ctxt, $timestamp, {$ctxt_print{noa}->{idx} => (0.1 * $abs)});
    } elsif ($line =~ /\{CTXT-(\d+)} (un)?link (to|from) \{VIF-(\d+)}/) {
        my ($vid, $cid) = ($4, $1);
        $ctxt = &get_ctxt($cid);
        $vif = &get_vif($vid, $ctxt);
    } elsif ($line =~ /\{VIF-(\d+)\} NOA-(\d+) timer expire status=(\d+)/ ) {
        my ($vid, $nid, $status) = ($1, $2, $3);
        $vif = &get_vif($vid);

        next if (! defined $vif->{c});
        die ("Only Support up to noa id 3") if ($nid >= 4);

        $status = ($status % 2) * ($nid + 1) * 0.1 ;
        my $data = {};
        $nid="noa_$nid";
        $data->{$vif_print{$nid}->{idx}} = $status;

        $vif->{c}->{v}->{$vid}->{$nid} = 1;
        &print_vif_data($vif, $timestamp, $data);
    } elsif ($line =~ /\{VIF-(\d+)} stop NOA-(\d+)/) {
        my ($vid, $nid) = ($1, $2);
        $vif = &get_vif($vid);

        next if (! defined $vif->{c});
        die ("Only Support up to noa id 3") if ($nid >= 4);

        $nid="noa_$nid";
        &print_vif_data($vif, $timestamp, {$vif_print{$nid}->{idx} => 0});
    } elsif ($line =~ /\{VIF-(\d+)\} (mm_tbtt_compute set tbtt_timer|beacon transmitted)/) {
        my ($vid, $nid, $status) = ($1, $2, $3);
        $vif = &get_vif($vid);
        next if (! defined $vif->{c});
        &print_vif_data($vif, $timestamp, {$vif_print{beacon}->{idx} => 1});
        $vif->{c}->{v}->{$vid}->{beacon} = 1;
    }
}
close LOG;

$plot_start = $timestamp_start if ((! defined $plot_start) || ($plot_start < $timestamp_start));
$plot_end = $timestamp if ((! defined $plot_end) || ($plot_end > $timestamp));

&gnuplot_cmd;

if ( defined $plot && -e "$outdir/$cmdfile" ) {
    if ((! defined $plot_dur) || ($plot_dur >= ($plot_end - $plot_start))) {
        system("gnuplot $outdir/$cmdfile");
    } else {
        my ($start, $end, $idx) = ($plot_start, $plot_start + $plot_dur, 1);
        while ($start < $plot_end) {
            &gnuplot_cmd_update("$outdir/$cmdfile", $start, $end, $idx);
            $start = $end;
            $end += $plot_dur;
            $end = $plot_end if ($end > $plot_end);
            $idx++;
            system("gnuplot $outdir/$cmdfile");
        }
    }
}


################################################################################################

sub get_ctxt
{
    my ($id, $freq, $bw) = @_;

    if (!defined($ctxts{$id})) {
        $ctxts{$id} = {};
        $ctxts{$id}->{name} = "Chan Context $id";
        $ctxts{$id}->{name} = "SCAN Chan context" if ($id == $first_connless_ctxt);
        $ctxts{$id}->{name} = "ROC Chan context" if ($id > $first_connless_ctxt);
        $ctxts{$id}->{id} = $id;
        $ctxts{$id}->{freq} = $freq if $freq ;
        $ctxts{$id}->{bw} = $bw if $bw;
        $ctxts{$id}->{map} = 0;
        $ctxts{$id}->{filename} = "$outdir/ctxt-$id.data.txt";
        $ctxts{$id}->{v} = {};
        open $ctxts{$id}->{fh}, ">", $ctxts{$id}->{filename} or die "cannot open file $ctxts{$id}->{filename}";
        &reset_ctxt_data($ctxts{$id}, $timestamp_start);
    } else {
        if (! defined($ctxts{$id}->{freq}) && $freq) {
            $ctxts{$id}->{freq} = $freq;
        }
        if (! defined($ctxts{$id}->{bw}) && $bw) {
            $ctxts{$id}->{bw} = $bw;
        }
    }

    return $ctxts{$id};
}

sub link_vif
{
    my ($v, $c) = @_;
    my $v_id = $v->{id};
    my $c_id = $c->{id};
    my $filename = "$outdir/ctxt-$c_id-vif-$v_id.data.txt";

    $v->{c} = $c;
    $v->{filename} = $filename;

    return if (defined $c->{v}->{$v_id});

    $c->{v}->{$v_id} = {};
    my $ref = $c->{v}->{$v_id};

    $ref->{filename} = $filename;
    open $ref->{fh}, ">", $ref->{filename} or die "cannot open file $ref->{filename}";

    &reset_vif_data($v, $timestamp);
}

sub unlink_vif
{
    my ($id) = @_;
    my $v = $vifs{$id};

    &reset_vif_data($v, $timestamp);
    delete $v->{c};
    delete $v->{filename};
}

sub get_vif
{
    my ($id, $c) = @_;

    if (!defined($vifs{$id})) {
        $vifs{$id} = {};
        $vifs{$id}->{id} = $id;
    }

    if (defined $c) {
        if (!defined($vifs{$id}->{c})) {
            &link_vif($vifs{$id}, $c);
        } elsif ($vifs{$id}->{c} != $c) {
            &unlink_vif($id);
            &link_vif($vifs{$id}, $c);
        }
    }

    return $vifs{$id};
}

sub reset_ctxt_data()
{
    my ($c, $ts) = @_;
    my $data = {};
    my $idx;

     for my $i (values %ctxt_print) {
        if (ref($i) eq "HASH") {
            if (defined $i->{idx}) {
                $data->{$i->{idx}} = 0;
            }
            else {
                for my $j (values %{$i}) {
                    $data->{$j->{idx}} = 0 if (defined $j->{idx});
                }
            }
        }
    }

    &print_ctxt_data($c, $ts, $data);
}

sub reset_vif_data()
{
    my ($v, $ts) = @_;
    my $data = {};
    for my $j (values %vif_print) {
        if ((ref($j) eq "HASH") && (defined $j->{idx})) {
            $data->{$j->{idx}} = 0;
        }
    }

    &print_vif_data($v, $ts, $data);
}

sub print_data
{
    my ($fh, $ts, $ref) = @_;
    my $prev_idx=1;
    my $res = "$ts ";

    foreach my $idx (sort {$a <=> $b} keys(%$ref)) {
        $res .= "- "x($idx - $prev_idx - 1);
        $res .= "$ref->{$idx} ";
        $prev_idx = $idx;
    }

    print $fh "$res\n";
}

sub print_ctxt_data
{
    my ($c, $ts, $ref) = @_;
    my $fh = $c->{fh};

    &print_data($fh, $ts, $ref);
}

sub print_vif_data
{
    my ($v, $ts, $ref) = @_;
    my $v_id = $v->{id};
    my $c = $v->{c};
    my $fh = $c->{v}->{$v_id}->{fh};

    die("No file open for vif $v_id on ctxt $c->{id}") if (!defined $fh);
    &print_data($fh, $ts, $ref);
}

sub close_data
{
    my $nb_graph = 0;

    foreach my $v (values %vifs) {
        &reset_vif_data($v, $timestamp);
    }

    foreach my $c (values %ctxts) {
        &reset_ctxt_data($c, $timestamp);
        close $c->{fh};
        if ( $c->{id} >= $first_connless_ctxt) {
            if ($no_connless_ctxt || ($c->{id} == 255))
            {
                delete $ctxts{$c->{id}};
            }
            else
            {
                $nb_graph++;
            }
        } else {

            foreach my $cv (values %{$c->{v}}) {
                close $cv->{fh};
                $nb_graph++;
            }
        }
    }

    return $nb_graph;
}

sub v_print_idx()
{
    my ($t) = @_;
    return $vif_print{$t}->{idx}
}

sub plot_ctxt
{
    my ($c, $last) = @_;
    my $d = $c->{filename};
    my $cmd;
    my $y2_range;

    my $common_cmd;
    my $sep="";
    foreach my $s (sort(keys(%{$ctxt_print{status}}))) {
        my $color = $ctxt_print{status}->{$s}->{color};
        my $idx = $ctxt_print{status}->{$s}->{idx};
        my $name = $ctxt_print{status}->{$s}->{name};
        next unless ((1 << $idx) & $c->{map});
        $common_cmd .= "$sep\\
\"$d\" using 1:$idx axes x1y1 title \"$name\" with fillsteps fc $color fs transparent solid 0.6";
        $sep=",";
    }

    if ($max_slot) {
        $common_cmd .= "$sep\\
\"$d\" using 1:$ctxt_print{slot}->{idx} axes x1y2 title \"slot\" with impulses fc $ctxt_print{slot}->{color} lw 9, \\
\"$d\" using 1:$ctxt_print{op}->{idx} axes x1y1 title \"ctxt op\" with impulses fc $ctxt_print{op}->{color} lw 9";
        $sep=",";

        $max_slot = 100 if ($max_slot > 100);
        $y2_range ="
set y2range[0 : $max_slot]
set y2tics 50
set grid y2tics
";
    }

    $cmd="
########## Context $c->{id}
";

    my @vif_list = sort {$a <=> $b} keys(%{$c->{v}});
    my $nb_vif = @vif_list;

    if ($nb_vif) {
        my $first_vif = 1;

        foreach my $v_id (@vif_list) {
            my $v = $vifs{$v_id};
            my ($bmargin, $tmargin) = (0,0);

            $nb_vif--;
            if ($first_vif) {
                $tmargin = 0.5;
                $first_vif = 0;
            }

            if (!$nb_vif) {
                $bmargin = 0.5;
                if ($last) {
                    $bmargin = 2;
                    $cmd .= &xtics();
                }
            }

            my $dv = $c->{v}->{$v_id}->{filename};

            $cmd.="
set tmargin $tmargin
set bmargin $bmargin
$y2_range
set yrange[0 : 1]
unset ytics

set title \"{CTXT->$c->{id}} {VIF-$v->{id}}\" offset 0,-2.2
plot $common_cmd";

            $cmd .= "$sep\\
\"$dv\" using 1:".&v_print_idx("tbtt")." axes x1y1 title \"tbtt\" with steps fc $vif_print{tbtt}->{color} lw 2";

            $cmd .= ",\\
\"$dv\" using 1:".&v_print_idx("tbtt_skip")." axes x1y1 title \"tbtt skipped\" with fillsteps fc $vif_print{tbtt_skip}->{color} fs transparent pattern 4";

            $cmd .= ",\\
\"$dv\" using 1:".&v_print_idx("tbtt_miss")." axes x1y1 title \"tbtt miss\" with impulses fc $vif_print{tbtt_miss}->{color} lw 5";

            for (my $i = 0 ; $i < 4 ; $i++)
            {
                my $noa_id="noa_$i";
                if (defined $c->{v}->{$v_id}->{$noa_id})
                {
                    $cmd .= ",\\
\"$dv\" using 1:".&v_print_idx($noa_id)." axes x1y1 title \"noa\" with fillsteps fc $vif_print{$noa_id}->{color} fs transparent solid 0.4";
                }
            }

            if (defined $c->{v}->{$v_id}->{beacon})
            {
                $cmd .= ",\\
\"$dv\" using 1:".&v_print_idx("beacon")." axes x1y1 title \"Beacon\" with impulses fc $vif_print{beacon}->{color} lw 3";
            }

            $cmd .= "
unset grid
unset y2tics
";
        }
    } else {
        my ($bmargin, $tmargin) = (0.5,0.5);
        if ($last) {
            $bmargin = 2;
            $cmd .= &xtics();
        }

        $cmd.="
set tmargin $tmargin
set bmargin $bmargin
$y2_range
set yrange[0 : 1]
unset ytics

set title \"$c->{name}\" offset 0,-2.2
plot $common_cmd

unset grid
unset y2tics
";
    }

    return $cmd;
}

my $graph_dur;
sub xtics()
{
    my $tic = $graph_dur / 100;

    if ($tic < 0.001) {
        $tic = "0.001 format \"%.3f\"";
    } elsif ( $tic < 0.005) {
        $tic = "0.005 format \"%.3f\"";
    } elsif ( $tic < 0.01) {
        $tic = "0.01 format \"%.3f\"";
    } elsif ( $tic < 0.05) {
        $tic = "0.05 format \"%.3f\"";
    } elsif ( $tic < 0.1) {
        $tic = "0.1 format \"%.3f\"";
    }  elsif ( $tic < 0.5) {
        $tic = "0.5 format \"%.3f\"";
    } else {
        $tic = "1";
    }

    return "
set xtics $tic nomirror
set xlabel \"time (in s)\"
"
}

sub gnuplot_cmd
{
    my $nb_graph = &close_data;

    if ($nb_graph == 0) {
        print "Nothing to plot\n";
        return;
    }

    open CMD, ">$outdir/$cmdfile" or die "Cannot create $cmdfile";

    $graph_dur = ($plot_end - $plot_start);
    if (defined $plot_dur && ($plot_dur <= $graph_dur)) {
        $graph_dur = $plot_dur;
    }

    my $png_width = int(8 * $graph_dur * 1000);
    $png_width = 8000 if ($png_width < 8000);
    $png_width = 30000 if ($png_width > 30000);

    my $png_height = 1000;
    $png_height = $nb_graph * 500 if ($nb_graph > 2);

    print CMD "
set term pngcairo size $png_width, 1000
set output \"$plot_exportdir/$pngfile\"

set multiplot layout $nb_graph,1
unset xtics
# max range [$plot_start : $plot_end]
set xrange [$plot_start : $plot_end]

set tmargin 0
set bmargin 0
set lmargin 0
set rmargin 0

set border lc 0 lw 2

";

    my $nb_ctxt = keys(%ctxts);
    foreach my $idx (sort {$a <=> $b} keys(%ctxts)) {
        my $c = $ctxts{$idx};
        $nb_ctxt--;
        print CMD &plot_ctxt($c, ($nb_ctxt==0));
    }

    print CMD "
unset multiplot
";

    close CMD;
}

sub gnuplot_cmd_update
{
    my ($file, $start, $end, $idx) = @_;

    print("update [$start - $end] ($idx)\n");

    {
        # inplace editing (without backup)
        local ($^I, @ARGV) = ('', ($file));

        while (my $line = <>) {
            if ( $line =~ /set xrange.*/ ) {
                $line = "set xrange [$start: $end]\n";
            } elsif ( $line =~ /set output "(.*)"/ ) {
                my $outfile = $1;
                $idx = sprintf("%03d", $idx);
                $outfile =~ s/\.(\d+\.)?png/.$idx.png/;
                $line = "set output \"$outfile\"\n";
            }
            print $line;
        }
    }
}
