#!/usr/bin/env perl

use strict;
use Getopt::Long;
use File::Basename;
use FindBin;              # define $FindBin::Bin that contains the path in which this script is located
use lib $FindBin::Bin;    # Add script directory to find module task
use IO::Handle; # for sysopen
use Fcntl;      # for O_RDWR ...

my %frame_type = (
    0 => { 0 =>  "Association request",
           1 =>  "Association response",
           2 =>  "Reassociation request",
           3 =>  "Reassociation response",
           4 =>  "Probe request",
           5 =>  "Probe response",
           6 =>  "Timing Advertisement",
           7 =>  "Reserved (mgmt)",
           8 =>  "Beacon",
           9 =>  "ATIM",
           10 => "Disassociation",
           11 => "Authentication",
           12 => "Deauthentication",
           13 => "Action",
           14 => "Action No Ack",
    },
    1 => { 4 =>  "Beamforming Report Poll",
           5 =>  "VHT NDP Announcement",
           6 =>  "Control Frame Extension",
           7 =>  "Control Wrapper",
           8 =>  "Block Ack Request",
           9 =>  "Block Ack",
           10 => "PS-Poll",
           11 => "RTS",
           12 => "CTS",
           13 => "ACK",
           14 => "CF-End",
           15 => "CF-End + CF-Ack",
    },
    2 => { 0 =>  "Data",
           1 =>  "Data + CF-Ack",
           2 =>  "Data + CF-Poll",
           3 =>  "Data + CF-Ack + CF-Poll",
           4 =>  "Null (no data)",
           5 =>  "CF-Ack (no data)",
           6 =>  "CF-Poll (no data)",
           7 =>  "CF-Ack + CF-Poll (no data)",
           8 =>  "QoS Data",
           9 =>  "QoS Data + CF-Ack",
           10 => "QoS Data + CF-Poll",
           11 => "QoS Data + CF-Ack + CF-Poll",
           12 => "QoS Null (no data)",
           13 => "Reserved (data)",
           14 => "QoS CF-Poll (no data)",
           15 => "QoS CF-Ack + CF-Poll (no data)",
    },
    );

my %frame_flags = (
    0 => "TODS",
    1 => "FROMDS",
    2 => "MORE_FRAG",
    3 => "RETRY",
    4 => "PWR",
    5 => "MORE_DATA",
    6 => "PROT",
    7 => "ORDER"
    );

my $open_flags = O_RDONLY;
my $target;
my $out;
my $out_fh;
my $out_mode = ">";
my $stdout=0;
my $dictionary;
my $nonblock = 0;
my $help = "
usage: fw_decode_trace.pl [--target=<fullmac|softmac|fhost>] [--nonblock] [--out=[outfile]] [trace_file]

  --target=<fullmac|softmac|fhost> : Specify which dictionary (fmacfw.pm,lmacfw.pm, fhostfw.pm) will be loaded
                                     If omitted script try to determine which fw is in use (assuming that
                                     decoder is called on the host platform), otherwise default to softmac
  --fullmac : same as --target=fullmac
  --softmac : same as --target=softmac
  --fhost   : same as --target=fhost

  --nonblock : Open trace file with O_NONBLOCK option. (only used when trace_file is provided)

  --out=[outfile]            : Specify the output file. If set without parameter the outfile is generated
                               based on input file as follow:
                               infile=<filename>.<suffix> -> outfile=<filename>.decoded.<suffix>
                               infile=<filename>          -> outfile=<filename>.decoded.txt
                               infile=STDIN               -> outfile=fw_trace.decoded.txt
                               If omitted, trace is displayed on STDOUT

  --append                   : Append data in output file instead of overwriting the output file.
                               (only used if --out option is set)

  --stdout                   : To force displaying output on stdout even if --out option is passed

  [trace_file]               : specify the trace file to decode. If omitted read trace from STDIN

  NOTE: Dictionary is loaded as a module so it should be available in perl libs path.
  The script add the directory in which it is contained in the include directory.
  (So put dictionary next to this script, in a directory included in PERL5LIB environment variable or
   use perl -I <path/to/dictionary> -- $0 ...)
";

$SIG{INT} = sub { die "interrupted" };

# read parameters
GetOptions(
    "target=s" => \$target,
    "softmac"  => sub { $target = "lmac" },
    "fullmac"  => sub { $target = "fmac" },
    "fhost"    => sub { $target = "fhost" },
    "nonblock" => sub { $open_flags |= O_NONBLOCK},
    "out:s"    => \$out,
    "stdout"   => \$stdout,
    "append"   => sub { $out_mode = ">>"},
    "help"     => sub { print $help ; exit 0; }
);

my $file = shift;
my %warnings;

if (!defined($target)) {
    if (!system("lsmod | grep -q rwnx_fdrv")) {
        $target = "fmac";
    } elsif (!system("lsmod | grep -q rwnx_fhost")) {
        $target = "fhost";
    } else {
        $target = "lmac";
    }
}

if ($target =~ /^(fmac|umac|full)/i) {
    $dictionary = "fmacfw";
} elsif ($target =~ /^(lmac|smac|soft)/i) {
    $dictionary = "lmacfw";
} elsif ($target =~ /fhost/i) {
    $dictionary = "fhostfw";
}

our %fw_dict;
our %tasks;
our %rtos_tasks;
my %task_ids;
eval "use $dictionary";
if ($@) {
    if ($@ =~ /Can't locate/) {
        die("Cannot find fw dictionary ($dictionary.pm).
Ensure that path to $dictionary is present in PERL5LIB variable, or use
\"perl -I <path/to/dictionary> -- $0 ...\"" );
    } else {
        die("Error while loading $dictionary.pm. Please check dictionary content");
    }
}

&init_task_id($dictionary);

if (defined $out && $out eq "") {
    if ($file eq "-") {
        $out = "fw_trace.decoded.txt";
    } elsif ($file =~ /(.*)\.([^.]+)/) {
        $out = "$1.decoded.$2";
    } else {
        $out = $file . ".decoded.txt";
    }
} elsif (! defined $out) {
    $stdout = 1;
}

my $trace_fd;
if (!defined($file)) {
    open($trace_fd, "<-") or die "Can't open STDIN: $!";
} else {
    sysopen($trace_fd, $file, $open_flags) or die "Can't open $file: $!";
}

if (defined $out) {
    my $out_dir = dirname($out);
    if (!-d $out_dir) {
        mkdir $out_dir or die "Cannot create $out_dir: $!";
    }
    open $out_fh, "$out_mode", "$out" or die "Cannot create $out: $!";
}

# Enable autoflush when decoding potential "live" input
if (!defined($file) || ($file =~ /^\/(dev|sys)\//)) {
    $| = 1;
}

my $trace_ts;
while (my $line = <$trace_fd>) {
    # split timestamp for easy reading
    if ($line =~ /ts= *(\d+)/) {
        $trace_ts = $1;
        my $ts = sprintf "[%15s]", &split_ts_us($trace_ts);
        $line =~ s/ts= *(\d+)/$ts/;
    }
    if ($line =~ /(.+)(ID= *(\d+)((,\s*\d+)*))/) {
        my ($prefix, $pattern, $id, $params) = ($1, $2, $3, $4);
        $params =~ s/^,\s*//;
        my $decoded = &decode_line($id, length($prefix), split(/, */, $params));

        if ($decoded ne "") {
            $line =~ s/$pattern/$decoded/;
        }
    }

    &out_print("$line");
}

close $trace_fd;


sub out_print {

    if (defined $out_fh) {
        print $out_fh @_;
    }

    if ($stdout) {
        print @_;
    }
}

sub split_ts_us {
    my ($ts) = @_;
    my ($us, $ms, $len, $tmp);

    $len = length($ts);
    $us  = substr($ts, -3);
    $tmp = length($us);
    $len -= $tmp;
    $us = "0" x (3 - $tmp) . $us;

    $ts = substr($ts, 0, $len);
    $ms = substr($ts, -3);
    $tmp = length($ms);
    $len -= $tmp;
    $ms = "0" x (3 - $tmp) . $ms;

    $ts = substr($ts, 0, $len);
    $ts = 0 unless ($ts);

    return "$ts.${ms}_$us";
}

sub warn_once {
    my ($id, $type, $str) = @_;
    if (!defined($warnings{$id}{$type})) {
        warn("($.) " . $str);
        $warnings{$id}{$type} = 1;
    }
}

my $last_word;

# $size param is in bytes
sub get_param {
    my ($size, $ref, $array_ref, $id) = @_;
    my (@word, $nb_word, $tmp);

    if (($size == 1) && (defined $last_word)) {
        $$ref = $last_word >> 8;
        undef $last_word;
        return 0;
    } else {
        $nb_word = int(($size + 1) / 2);
        while ($nb_word) {
            $tmp = shift(@$array_ref);
            if (!defined $tmp) {
                warn_once($id, 'not_enough', "not enough param for: $fw_dict{$id}\n");
                return -1;
            }
            push @word, $tmp;
            $nb_word--;
        }
    }

    if ($size == 1) {
        $$ref      = $word[0] & 0xff;
        $last_word = $word[0];
    } elsif ($size & 0x1) {
        warn("unsupported size $size\n");
        return -1;
    } else {
        $tmp = 0;
        while ($size) {
            $tmp = ($tmp << 16) + shift @word;
            $size -= 2;
        }
        $$ref = $tmp;
    }

    return 0;
}

sub decode_line {
    my $res = "";
    my $id  = shift @_;
    my $prefix_len = shift @_;

    undef $last_word;

    if (!defined $fw_dict{$id}) {
        my ($file, $line);
        $file = ($id >> 16) & 0xff;
        $line = $id & 0xffff;
        if (defined $fw_dict{$file}) {
            $res = "$fw_dict{$file}";
        } else {
            $res = "file $file";
        }
        $res .= " line $line";
        return $res;
    } else {
        $res = $fw_dict{$id};
    }

    while ($res =~ /(%.*)/) {
        my $match = ($1);

        if ($match =~ /^(%\d*(l+)?(d|x|X|u|c))/) {
            my ($format, $long, $type) = ($1, $2, $3);
            my $val;
            my $size = 2;
            if ($long) {
                $size *= 2**length($long)
            }

            return $res if (&get_param($size, \$val, \@_, $id));

            # val are printed as unsigned, need to convert back to signed for %d
            if ($type eq "d")
            {
                if ($size == 2) {
                    $val = unpack('s', pack('S', $val));
                } elsif ($size == 4) {
                    $val = unpack('l', pack('L', $val));
                } else {
                    $val = unpack('q', pack('Q', $val));
                }
            }

            $match = sprintf("$format", $val);
            $res =~ s/$format/$match/;
        } elsif ($match =~ /^(%s)/) {
            my ($format) = ($1);
            my ($len, $offset, $char);

            return $res if &get_param(2, \$len, \@_, $id);
            $offset = ($len & 0xff);
            &get_param(1, \$char, \@_, $id) if ($offset);    # offset one char if string ptr was not aligned
            $len = ($len >> 8) - $offset;

            $match = "";
            while ($len) {
                return $res if &get_param(1, \$char, \@_, $id);
                $len--;
                if ($char == 0) {
                    while ($len) {
                        &get_param(1, \$char, \@_, $id);
                        $len--;
                    }
                } else {
                    $match = "$match" . sprintf("%c", $char);
                }
            }
            undef $last_word;

            $res =~ s/$format/$match/;
        } elsif ($match =~ /^(%pM)/) {
            my ($format, $mac, $i) = ($1, 0);

            $match = "";
            for ($i = 0 ; $i < 3 ; $i++) {
                return $res if &get_param(2, \$mac, \@_, $id);
                $match = $match . ":" if ($match);
                $match = $match . sprintf("%02X:%02X", ($mac & 0xff), ($mac >> 8));
            }

            $res =~ s/$format/$match/;
        } elsif ($match =~ /^(%p[iI]4)/) {
            my ($format, $ip, $str_fmt, $i) = ($1, 0);

            $match = "";
            return $res if &get_param(4, \$ip, \@_, $id);

            if ($format =~ /I4/ ) {
                $str_fmt = "%d.%d.%d.%d"
            } else {
                $str_fmt = "%03d.%03d.%03d.%03d"
            }

            $match = sprintf($str_fmt, ($ip & 0xff), ($ip >> 8) & 0xff,
                             ($ip >> 16) & 0xff, ($ip >> 24) & 0xff);

            $res =~ s/$format/$match/;
        } elsif ($match =~ /^(%p(i6|I6c?))/) {
            my ($format, $i) = ($1, 0);
            my @ip;
            my ($max_zero, $max_zero_start, $cur_zero, $cur_zero_start) = (0, 0, 0, 0);

            $match = "";

            for ($i = 0; $i < 8; $i++)
            {
                my $hextet;
                return $res if &get_param(2, \$hextet, \@_, $id);
                push @ip, $hextet;
                if ($hextet == 0) {
                    $cur_zero_start = $i unless ($cur_zero > 0);
                    $cur_zero ++;
                    if (($i == 7) && ($cur_zero > $max_zero)) {
                        $max_zero = $cur_zero;
                        $max_zero_start = $cur_zero_start;
                    }
                } elsif ($cur_zero > 0) {
                    if ($cur_zero > $max_zero) {
                        $max_zero = $cur_zero;
                        $max_zero_start = $cur_zero_start;
                    }
                    $cur_zero = 0;
                }
            }

            if ($format =~ /I6c/ && ($max_zero > 1)) {
                for ($i=0; $i < $max_zero_start; $i++) {
                    $match.=sprintf("%x:", shift(@ip))
                }
                for ($i; $i < ($max_zero_start + $max_zero); $i++) {
                    shift(@ip);
                }
                $match.=":";
                for ($i; $i < 8; $i++) {
                    $match.=sprintf("%x:", shift(@ip))
                }
                $match=substr($match, 0, -1);
            } else {
                if ($format =~ /I6/) {
                    for ($i = 0; $i < 8; $i++) {
                        $match.=sprintf("%x:", shift(@ip));
                    }
                    $match=substr($match, 0, -1);
                } else {
                    for ($i = 0; $i < 8; $i++) {
                        $match.=sprintf("%04x", shift(@ip))
                    }
                }
            }

            $res =~ s/$format/$match/;
        } elsif ($match =~ /^(%pB(F)?(\d+)?([bhw])?([dxXc])?)/) {
            my ($format, $words_per_line, $word_size, $repr, $i) = ($1, 8, "b", "x", 0);
            my ($len, $byte, $word, $word_size_num, $tmp, $print_format, $nb_byte, $post, $mac_hdr);

            $mac_hdr        = $2;
            $words_per_line = $3 if ($3);
            $word_size      = $4 if ($4);
            $repr           = $5 if ($5);

            return $res if &get_param(2, \$len, \@_, $id);

            # This assume that %pB is the only format is this trace
            $nb_byte = @_ * 2;

            if ($len & 0x200) {
                # unaligned pointer, must skip first byte;
                &get_param(1, \$byte, \@_, $id);
                $nb_byte--;
            }
            if ($len & 0x400) {
                $post = " (incomplete)"
            }
            $len = ($len & 0x1FF);
            if ($nb_byte < $len) {
                my $miss = $len - $nb_byte;
                $post = sprintf" (missing %d byte%c)", $miss, ($miss > 1) ? 115 : 0;
                $len = $nb_byte
            }

            if ($word_size eq "b") {
                $word_size = 1;
            } elsif ($word_size eq "h") {
                $word_size = 2;
            } elsif ($word_size eq "w") {
                $word_size = 4;
            }

            if ($repr =~ /[xX]/) {
                $print_format = $print_format . sprintf("%%0%d%s", $word_size * 2, $repr);
            } else {
                $print_format = '%' . $repr;
            }

            $match = "";
            $word = 0;
            $tmp = $res;
            $tmp =~ /^(.+)?$format/;
            my $new_line = "\n" . ' ' x (length($1) + $prefix_len);

            $len = &parse_mac_header(\@_, $id, $len, $new_line, \$match) if ($mac_hdr);

            for ($i = 1 ; $i <= $len ; $i++) {
                &get_param(1, \$byte, \@_, $id);
                $word = $word | $byte << (8 * (($i - 1) % $word_size));

                if (($i % $word_size) == 0) {
                    $match = "$match" . sprintf($print_format, $word);
                    if ((($i % ($words_per_line * $word_size)) == 0) && ($i != $len)) {
                        $match .= $new_line;
                    } else {
                        $match .= " ";
                    }
                    $word = 0;
                }
            }

            if (($len % $word_size) > 0) {
                $match .= $new_line if ($i == $len);
                $match .= sprintf($print_format, $word);
            }
            if ($post) {
                $match .= $post
            }

            $res =~ s/$format/$match/;
        } elsif ($match =~ /^(%p)/) {
            my ($format, $ptr) = ($1, 0);
            return $res if &get_param(4, \$ptr, \@_, $id);
            $match = sprintf("\@0x%08x", $ptr);

            $res =~ s/$format/$match/;
        } elsif ($match =~ /^(%F)/) {
            my ($format, $file_id) = ($1, 0);
            return $res if &get_param(2, \$file_id, \@_, $id);

            $file_id = ($file_id & 0xff);
            if (defined $fw_dict{$file_id}) {
                $match = "$fw_dict{$file_id}";
            } else {
                $match = "File_$file_id";
            }

            $res =~ s/$format/$match/;
        } elsif ($match =~ /^(%k([MTS]))/) {
            my ($format, $type, $val) = ($1, $2, 0);
            my ($msg, $tsk, $idx, $state);

            return $res if &get_param(2, \$val, \@_, $id);

            if ($type eq 'M') {
                $msg = $val & 0x3ff;
                $tsk = $val >> 10;
                if (defined($task_ids{$tsk})) {
                    $tsk = $task_ids{$tsk};
                    if (defined $tasks{$tsk}->{messages}[$msg]) {
                        $match = $tasks{$tsk}->{messages}[$msg];
                    } else {
                        warn_once($id, 'ke', "unknown MSG $msg of KE tsk $tsk\n");
                        $match = "$tsk:MSG_$msg";
                    }
                } else {
                    warn_once($id, 'ke', "unknown KE tsk $tsk\n");
                    $match = "TSK_$tsk:MSG_$msg";
                }
            } elsif ($type eq 'T' || $type eq 'S') {
                $tsk = $val & 0xff;
                $idx = $val >> 8;
                if (defined($task_ids{$tsk})) {
                    $tsk   = $task_ids{$tsk};
                    $match = "$tsk($idx)";
                } else {
                    warn_once($id, 'ke', "unknown KE tsk $tsk\n");
                    $match = "TSK_$tsk($idx)";
                }
                if ($type eq 'S') {
                    return $res if &get_param(2, \$state, \@_, $id);
                    if (defined $tasks{$tsk}->{states}[$state]) {
                        $state = $tasks{$tsk}->{states}[$state];
                    } else {
                        warn_once($id, 'ke', "unknown state $state KE tsk $tsk\n");
                    }
                    $match .= ":$state";
                }
            }

            $res =~ s/$format/$match/;
        } elsif ($match =~ /^(%r([T]))/) {
            my ($format, $type, $tsk) = ($1, $2, 0);

            return $res if &get_param(2, \$tsk, \@_, $id);
            if (defined $rtos_tasks{$tsk}) {
                $match = $rtos_tasks{$tsk};
            } else {
                $match = "RTOS_TSK_$tsk";
            }

            $res =~ s/$format/$match/;
        } elsif ($match =~ /^(%t)/) {
            my ($format, $timestamp, $delta) = ($1, 0, 0);

            return $res if &get_param(4, \$timestamp, \@_, $id);
            $delta = ($timestamp - $trace_ts);
            if ($delta < -2147483648) {
                $delta &= 0xffffffff;
            }
            $match="@".&split_ts_us($timestamp);

            if (abs($delta) < 1000) {
                $match.=sprintf(" (%+d us)", $delta)
            } elsif ( abs($delta) < 1000000) {
                $match.=sprintf(" (%+.2f ms)", ($delta / 1000))
            } else {
                $match.=sprintf(" (%+.2f s)", ($delta / 1000000))
            }

            $res =~ s/$format/$match/;
        } elsif ($match =~ /^(%fc)/) {
            my ($format, $fc) = ($1, 0);
            return $res if &get_param(2, \$fc, \@_, $id);

            $match = &parse_fctl($fc);
            $res =~ s/$format/$match/;
        } elsif ($match =~ /^(%%)/ ) {
            my ($format) = ($1);
            $res =~ s/$format/_put_percent_sign_/;
        } else {
            warn_once($id, "unsupported", "Unsupported format [$match] for ($fw_dict{$id})\n");
            $res =~ s/$match//;
        }
    }

    $res =~ s/_put_percent_sign_/%/g;
    $res .= " " . join(",", @_);
    return $res;
}

sub init_task_id {
    my ($dic) = @_;
    my $id;

    if ($dic =~ /lmac/) {
        $id = "id_lmac";
    } else {
        $id = "id_fmac";
    }

    foreach my $t (keys %tasks) {
        $task_ids{ $tasks{$t}->{$id} } = $t;
    }

    my $type = 0;
    my $subtype = 2;
}


sub parse_mac_header()
{
    my ($data_ref, $id, $nb_byte, $new_line, $res_ref) = @_;
    my ($byte, $byte2);
    my ($fc, $dur, $seq, @addr, $qos, $ht, $sec);

    goto DONE unless ($nb_byte >= 2);
    &get_param(1, \$byte, $data_ref, $id);
    &get_param(1, \$byte2, $data_ref, $id);
    $fc = &parse_fctl(($byte2 << 8) + $byte);
    $nb_byte -= 2;

    goto DONE unless ($nb_byte >= 2);
    &get_param(1, \$byte, $data_ref, $id);
    &get_param(1, \$byte2, $data_ref, $id);
    $dur = "Dur=".(($byte2 << 8) + $byte)."us";
    $nb_byte -= 2;

    for (my $i = 0; $i < 3; $i++) {
        goto DONE unless ($nb_byte >= 6);
        my $tmp = '';
        for (my $j = 0; $j < 6; $j++) {
            $tmp .= ":" if ($j);
            &get_param(1, \$byte, $data_ref, $id);
            $tmp .= sprintf("%02x", $byte)
        }
        $addr[$i] = "A".($i+1)."=$tmp";
        $nb_byte -= 6;
    }

    goto DONE unless ($nb_byte >= 2);
    &get_param(1, \$byte, $data_ref, $id);
    &get_param(1, \$byte2, $data_ref, $id);
    $seq = "Seq=". ((($byte2 << 8) + $byte) >> 4) . " frag=" . ($byte & 0xf);
    $nb_byte -= 2;

    if ($fc =~ /TODS\|FROMDS/) {
        goto DONE unless ($nb_byte >= 6);
        my $tmp = '';
        for (my $j = 0; $j < 6; $j++) {
            $tmp .= ":" if ($j);
            &get_param(1, \$byte, $data_ref, $id);
            $tmp .= sprintf("%02x", $byte)
        }
        $addr[3] = "A4=$tmp";
        $nb_byte -= 6;
    }

    if ($fc =~ /QoS/) {
        goto DONE unless ($nb_byte >= 2);
        &get_param(1, \$byte, $data_ref, $id);
        &get_param(1, \$byte2, $data_ref, $id);
        $qos="Qos=" . sprintf("%02x%02x", $byte2, $byte);
        $nb_byte -= 2;
    }

    if ($fc =~ /Qos.*ORDER/) {
        goto DONE unless ($nb_byte >= 4);
        $ht="HT ctrl=";
        for (my $i=0; $i<4; $i++) {
            &get_param(1, \$byte, $data_ref, $id);
            $ht.= sprintf("%02x", $byte);
        }
        $nb_byte -= 4;
    }

    if ($fc =~ /PROT/) {
        goto DONE unless ($nb_byte >= 4);
        my ($b1, $b2, $b3, $b4, $pn, $key);
        &get_param(1, \$b1, $data_ref, $id);
        &get_param(1, \$b2, $data_ref, $id);
        &get_param(1, \$b3, $data_ref, $id);
        &get_param(1, \$b4, $data_ref, $id);
        $nb_byte -= 4;
        $key = ($b4 >> 6);
        if ($b4 & (1<<5)) {
            goto DONE unless ($nb_byte >= 4);
            my ($b5, $b6, $b7, $b8);
            &get_param(1, \$b5, $data_ref, $id);
            &get_param(1, \$b6, $data_ref, $id);
            &get_param(1, \$b7, $data_ref, $id);
            &get_param(1, \$b8, $data_ref, $id);
            $nb_byte -= 4;
            if ($b2 == (($b1 | 0x20) & 0x7f)) {
                $sec="TKIP:";
                $pn = sprintf("%02x%02x%02x%02x%02x%02x", $b8, $b7, $b6, $b5, $b1, $b3);
            } else {
                $sec="G/CCMP:";
                $pn = sprintf("%02x%02x%02x%02x%02x%02x", $b8, $b7, $b6, $b5, $b2, $b1);
            }
        } else {
            $sec="WEP:";
            $pn = sprintf("%02x%02x%02x". $b1, $b2, $b3)
        }

        $sec .= "PN=$pn Key_id=$key";
    }

  DONE:

    $$res_ref = "$fc $dur $seq $qos $sec" . $new_line;
    $$res_ref .= "$addr[0] $addr[1] $addr[2] $addr[3]";
    $$res_ref .= $new_line if ($nb_byte);
    return $nb_byte;
}

sub parse_fctl()
{
    my ($fctl) = @_;
    my ($type, $subtype) = (($fctl >> 2) & 0x3 , ($fctl >> 4) & 0xf);
    my $flags = ($fctl >> 8);
    my $res;

    if (defined $frame_type{$type}->{$subtype}) {
        $res = $frame_type{$type}->{$subtype};
    } else {
        $res = "FC: type=$type subtype=$subtype"
    }

    my $sep = " [";
    foreach my $f (sort keys %frame_flags)
    {
        if ($flags & (1 << $f))
        {
            $res .= "$sep$frame_flags{$f}";
            $sep = "|";
        }
    }
    $res .= "]" unless ($sep eq " [");

    return $res;
}


