#!/usr/bin/env perl

use strict;
use Getopt::Long;
use File::Basename;
use FindBin;              # define $FindBin::Bin that contains the path in which this script is located
use lib $FindBin::Bin;    # Add script directory to find module task

my $target;
my $dictionary;
my $out_type="compo";
my $help = "
usage: fw_trace_compo_id.pl [--target=<fullmac|softmac|fhost>] <--compo> | <--string|--num|--desc> <compo_id>[:trace_level]

  Convert a trace component/filter id from string to numerical value and vice-versa using dictionary

  --target=<fullmac|softmac|fhost> : Specify which dictionary (fmacfw.pm, lmacfw.pm or fhostfw.pm) will be loaded
                                     If omitted script try to determine which fw is in use (assuming that
                                     decoder is called on the host platform), otherwise default to softmac
  --fullmac : same as --target=fullmac
  --softmac : same as --target=softmac
  --fhost : same as --target=fhost

  --string : convert id into string
  --num    : convert id into numerical value
  --desc   : convert id into string and show description for each level
  --compo  : List component name (default)

  The provided componenent id <compo_id> can be a number (decimal or hexadecimal) or a string
  The provided <trace_level> can be a number (decimal or hexadecimal) or set of string separated by '/'
  Invalid component/level name or id are silently ignored.

  examples:
  fw_trace_compo_id.pl --compo
  => ALL
     KERNEL
     RTOS

  fw_trace_compo_id.pl --desc KERNEL
  => ERR       : Error
     MSG_API   : Messages to API layer
     MSG_LOCAL : Messages from API layer or internal task
     STATE     : State changes

  fw_trace_compo_id.pl --desc KERNEL:MSG_API/MSG_LOCAL
  => MSG_API   : Messages to API layer
     MSG_LOCAL : Messages from API layer or internal task

  fw_trace_compo_id.pl --string 0
  => KERNEL

  fw_trace_compo_id.pl --string 0:8
  => KERNEL:STATE

  fw_trace_compo_id.pl --num 0:0xffffffff
  => 0:0x0000000f

  fw_trace_compo_id.pl --string 0:MSG_API/MSG_LOCAL
  => KERNEL:MSG_API/MSG_LOCAL

  fw_trace_compo_id.pl --num 0:MSG_API/ERR
  => 0:0x00000003

  NOTE: Dictionnay is loaded as a module so it should be available in perl libs path.
  The script add the directory in which it is contained in the include directory.
  (So put dictionary next to this script, in a directory included in PERL5LIB environment variable or
   use perl -I <path/to/dictionnnary> -- $0 ...)
";

# read parameters
GetOptions(
    "target=s" => \$target,
    "softmac"  => sub { $target = "lmac" },
    "fullmac"  => sub { $target = "fmac" },
    "fhost"    => sub { $target = "fhost" },
    "num"      => sub { $out_type = "num"},
    "string"   => sub { $out_type = "string"},
    "desc"     => sub { $out_type = "desc"},
    "compo"    => sub { $out_type = "compo"},
    "help"     => sub { print $help ; exit 0; }
);

if (!defined($target)) {
    if (!system("lsmod | grep -q rwnx_fdrv")) {
        $target = "fmac";
    } elsif (!system("lsmod | grep -q rwnx_fhost")) {
        $target = "fhost";
    } else {
        $target = "lmac";
    }
}

if ($target =~ /^(fmac|umac|full)/i) {
    $dictionary = "fmacfw";
} elsif ($target =~ /^(lmac|smac|soft)/i) {
    $dictionary = "lmacfw";
} elsif ($target =~ /fhost/i) {
    $dictionary = "fhostfw";
}

eval "use $dictionary";
if ($@) {
    die(
        "Cannot find fw dictionary ($dictionary.pm).
Ensure that path to $dictionary is present in PERL5LIB variable, or use
\"perl -I <path/to/dictionary> -- $0 ...\" "
    );
}
our %trace_compo;

# Simply list the components
if ($out_type eq "compo") {
    for my $c (sort (keys %trace_compo)) {
        print "$c\n";
    }
    exit;
}

# Else read parameter
my $param = shift;

if ($param =~ /(\w+)(:(.*))?/)
{
    my $compo = $1;
    my $level = $3;

    if ( $out_type eq "desc" ) {
        $compo = &to_compo_name($compo);
        exit unless ($compo);
        $level = 0xffffffff unless (defined $level);
        my @desc;
        my $max=0;
        my $idx=0;

        for my $l (&to_level_name($compo, $level)) {
            my $desc = $trace_compo{$compo}->{filters}->{$l}->{desc};

            push @desc, {name=>"$l", desc=>"$desc"};
            $max = length($l) unless ($max > length($l));
        }

        for my $l (@desc) {
            print "$l->{name}"." "x($max-length($l->{name}))." : $l->{desc}\n";
        }

    } elsif ($out_type eq "string") {
        print &to_compo_name($compo);
        if (defined($level)) {
            my $sep = ":";
            for my $l (&to_level_name($compo, $level)) {
                print "$sep$l";
                $sep = "/";
            }
        }
        print "\n";
    } elsif ( $out_type eq "num") {
        print &to_compo_id($compo);
        if (defined($level)) {
            $level = &to_level_id($compo, $level);
            print sprintf(":0x%08x", $level);
        }
        print "\n";
    }
}

sub to_int {
    my ($val) = @_;
    my $int;

    if ( $val =~ /^0x[a-fA-F0-9]+$/) {
        return hex($val);
    } elsif ( $val =~ /^-?\d+$/) {
        return int($val);
    }

    return undef
}

sub to_compo_id {
    my ($id) = @_;
    my $int = to_int($id);
    return $int if (defined $int);
    return $trace_compo{$id}->{id} if (defined($trace_compo{$id}));
    return ""
}

sub to_compo_name {
    my ($id) = @_;
    my $int = to_int($id);

    if (defined $int) {
        for my $c (keys %trace_compo)
        {
            if ($trace_compo{$c}->{id} == $int) {
                return $c;
            }
        }
    } elsif (defined($trace_compo{$id})) {
        return $id;
    }

    return "";
}

sub to_level_name {
    my ($compo, $level) = @_;
    my @res = ();

    $compo = &to_compo_name($compo);
    return @res unless ($compo);

    my $int = to_int($level);

    if (defined $int) {
        my @lvl = ();
        foreach my $f (keys %{$trace_compo{$compo}->{filters}}) {
            $lvl[$trace_compo{$compo}->{filters}->{$f}->{bit}] = $f;
        }

        for (my $i = 0 ; $i < 32 ; $i++) {
            next unless ($int & (1 << $i));
            push @res, $lvl[$i] if (defined $lvl[$i]);
        }
    } else {
        my @lvl = split(/\//, $level);
        foreach my $f (@lvl) {
            push @res, $f if (defined($trace_compo{$compo}->{filters}->{$f}));
        }

    }

    return @res;
}

sub to_level_id {
    my ($compo, $level) = @_;
    my $res = 0;

    $compo = &to_compo_name($compo);
    return $res unless ($compo);

    my $int = to_int($level);
    if (defined $int) {
        my @lvl = ();
        foreach my $f (keys %{$trace_compo{$compo}->{filters}}) {
            $lvl[$trace_compo{$compo}->{filters}->{$f}->{bit}] = $f;
        }

        for (my $i = 0 ; $i < 32 ; $i++) {
            next unless ($int & (1 << $i));
            $res += (1 << $i) if (defined $lvl[$i]);
        }
    } else {
        my @lvl = split(/\//, $level);
        foreach my $f (@lvl) {
            if (defined($trace_compo{$compo}->{filters}->{$f})) {
                $res += (1 << $trace_compo{$compo}->{filters}->{$f}->{bit})
            }
        }
    }

    return $res;
}
