#!/usr/bin/env perl

use strict;
use Getopt::Long;
use FindBin;              # define $FindBin::Bin that contains the path in which this script is located
use lib $FindBin::Bin;    # Add script directory to find module task

#  lmac set tsk:5:0
# [ 8604.184934] lmac Dispatching msg:1E from tsk:5 to tsk:0
# [ 8604.185002] lmac Dispatching msg:31 from tsk:5 to tsk:0
# [ 8604.185090] lmac Dispatching msg:1F from tsk:0 to tsk:5
# [ 8604.187479] lmac Dispatching msg:1C00 from tsk:7 to tsk:3

my %task_ids;
my $fw;
my %tasks_used;
my %events;
my $graph;
my $time_ref;
my $time_loop_count = 0;
my $inplace;
my $out;
my $png;
our %tasks;

sub help {
    print "usage: task.pl [--soft|--full] [--out=<outfile>|--inplace] [--graph [--png]] <trace file>

Replace ID with task and message name in firmware traces

--soft|--full|--fhost: Specify which id to use (softmac or fullmac), and also define dictionnary
                       to use if task.pm is not available.
                       fullmac if not specified

--out=<file> : Specify the file to write trace with id replaced
--inplace    : Modify trace in-place
               If neither --inplace nor --out is specified, output trace on stdout

--graph      : Generate graph from traces (need mscgen).
               If epstopdf is available graph are generated in pdf format and in png format otherwise.
               Graph name is: <outfile>.pdf if --out option is used and <trace_file>.pdf otherwise

--png        : Force graph generation in png format even if epstopdf is available
";
    exit;
}

# read parameters
GetOptions(
    "softmac" => sub { $fw = "lmac" },
    "fullmac" => sub { $fw = "fmac" },
    "fhost"   => sub { $fw = "fhost" },
    "out=s"   => \$out,
    "graph"   => \$graph,
    "inplace" => \$inplace,
    "png"     => \$png,
    "help"    => \&help
);

# load module that contains task ID dictionnary
my $module;
foreach my $dir (@INC) {
    if (-e "$dir/task.pm") {
        $module = "task";
        last;
    } elsif (-e "$dir/${fw}fw.pm") {
        $module = "${fw}fw";
    }
}

if (!$module) {
    die("Cannot find task.pm nor ${fw}fw.pm");
} else {
    eval "use $module";
    if ($@) {
        die("Error while loading ${module}.pm");
    } elsif (!%tasks) {
        die("module ${module}.pm doesn't contains %tasks hash");
    }
}

# get trace file
my $file = shift;

&help if (!defined $file);

# Enable autoflush when decoding potential "live" input
if ($file eq "-") {
    $| = 1;
}

if (!defined $fw) {
    if (system("lsmod | grep rwnx_fdrv")) {
        $fw = "fmac";
    } elsif (system("lsmod | grep rwnx_drv")) {
        $fw = "lmac";
    } else {
        $fw = "fmac";
        print "Parsing using default fw type: $fw\n";
    }
}

&init_task_id();

if ($inplace) {

    # inplace editing (without backup)
    local ($^I, @ARGV) = ('', ($file));

    while (my $line = <>) {
        print &process_line($line);
    }
} else {

    my $fh;

    open FILE, "$file" or die "cannot open $file : $!\n";

    if ($out) {
        open $fh, ">", "$out" or die "cannot create $out: $!\n";
    } else {
        open $fh, ">-" or die "cannot open stdout: $!\n";
    }

    while (my $line = <FILE>) {
        print $fh &process_line($line);
    }

    close FILE;
    close $fh;
}

&generate_graph if ($graph);

sub process_line {
    my ($line) = @_;

    if ($line =~ /set ((tsk:)?([\w_()]+):(\w+))/) {
        my ($pat, $task_id, $state_idx) = ($1, $3, $4);
        my ($task_name, $task_idx, $state_name);

        if ($task_id =~ /([\w_]+)\((\d+)\)/) {

            # trace already processed
            $task_name  = $1;
            $task_idx   = $2;
            $state_name = $state_idx;
        } else {
            ($task_name, $task_idx) = &tsk_id(hex($task_id));

            if (!defined $tasks{$task_name}) {
                print("unknown task $task_name\n");
                $state_name = $state_idx;
            } elsif (!$tasks{$task_name}->{states}[$state_idx]) {
                print("unknown state $state_idx for task $task_name\n");
                $state_name = $state_idx;
            } else {
                $state_name = $tasks{$task_name}->{states}[$state_idx];
            }

            $line =~ s/$pat/tsk:$task_name($task_idx):$state_name/;
        }

        &set_state(&get_time($line), $task_name . "_" . $task_idx, $state_name) if $graph;

    } elsif ($line =~ /(.*) ((msg:)?([\w_:]+) from (tsk:)?([\w_()]+) to (tsk:)?([\w_()]+))/) {
        my ($action, $pat, $msg, $src, $dst) = ($1, $2, $4, $6, $8);
        my ($msg_name, $src_name, $src_idx, $dest_name, $dest_idx);

        if ($src =~ /([\w_]+)\((\d+)\)/) {

            # trace already processed
            $src_name = $1;
            $src_idx  = $2;
            ($dest_name, $dest_idx) = ($dst =~ /([\w_]+)\((\d+)\)/);
            $msg_name = $msg;
        } else {

            my ($task_name, $msg_id) = &msg_id(hex($msg));
            ($src_name,  $src_idx)  = &tsk_id(hex($src));
            ($dest_name, $dest_idx) = &tsk_id(hex($dst));

            if (!defined $tasks{$src_name}) {
                print("unknown task $src_name\n");
            }
            if (!defined $tasks{$dest_name}) {
                print("unknown task $dest_name\n");
            }

            if (!$tasks{$task_name} || !$tasks{$task_name}->{messages}[$msg_id]) {
                print("unknown message for task ($task_name) / message ($msg_id)\n");
                $msg_name = $task_name . "#" . $msg_id;
            } else {
                $msg_name = $tasks{$task_name}->{messages}[$msg_id];
            }

            $line =~ s/$pat/msg:$msg_name from tsk:$src_name($src_idx) to tsk:$dest_name($dest_idx)/;
        }

        if ($graph) {
            if ($action =~ /dispatching/i) {
                &send_msg(&get_time($line), $src_name . "_" . $src_idx, $dest_name . "_" . $dest_idx, $msg_name);
            } elsif ($action =~ /no handler/i) {
                &no_handler($src_name . "_" . $src_idx, $dest_name . "_" . $dest_idx, $msg_name);
            } else {
                warn("$.: unknown action [$action]");
            }
        }
    }

    return $line;
}

sub tsk_id {
    my $tsk_id  = $_[0];
    my $tsk_idx = $tsk_id >> 8;
    $tsk_id = $tsk_id & 0xff;

    if (defined($task_ids{$tsk_id})) {
        $tsk_id = $task_ids{$tsk_id};
    }

    return ($tsk_id, $tsk_idx);
}

sub msg_id {
    my $tmp    = $_[0];
    my $msg_id = $tmp & 0x3ff;
    my $tsk_id = $tmp >> 10;

    if (defined $task_ids{$tsk_id}) {
        $tsk_id = $task_ids{$tsk_id};
    }

    return ($tsk_id, $msg_id);
}

sub init_task_id {
    my $id;

    if ($fw eq "lmac") {
        $id = "id_lmac";
    } else {
        $id = "id_fmac";
    }

    foreach my $t (keys %tasks) {
        $task_ids{ $tasks{$t}->{$id} } = $t;
    }
}

sub set_state {
    my ($timetamp, $task, $state) = @_;
    my $evt;

    $tasks_used{$task}++;
    $evt = "$task rbox $task [label=\"$state\"]";

    $events{$timetamp} = $evt;
}

sub send_msg {
    my ($timetamp, $src, $dst, $msg) = @_;
    my $evt;

    # TODO: find a better way to handle NONE task
    # if ($src =~  /none/i) {
    # 	$evt = "$dst x- $dst [label=\"$msg\"]";
    # 	$tasks_used{$dst}++;
    # } else
    {
        $evt = "$src => $dst [label=\"$msg\"]";
        $tasks_used{$src}++;
        $tasks_used{$dst}++;
    }

    $events{$timetamp} = $evt;
}

sub no_handler {
    my ($src, $dst, $msg) = @_;
    my $evt;

    $evt = "$src => $dst [label=\"$msg\"]";

    # Find latest matching evt
    foreach my $time (sort { $b <=> $a } (keys %events)) {
        if ($events{$time} eq "$evt") {

            #$events{$time} =~ s/=>/-x/;
            my $tmp = $events{$time};
            $tmp =~ s/=>/-x/;
            $events{$time} = "$dst rbox $dst [label=\"No handler\", textbgcolour=\"#ff7f7f\"], $tmp";
            return;
        }
    }

    warn("Cannot find match for ($evt)");
}

sub get_time {
    my ($line) = @_;
    my $time;

    if ($line =~ /\[\s*(\d+\.\d+)\].*lmac/) {
        $time = $1;
        $time_ref = $time unless (defined $time_ref);
    } elsif ($line =~ /\[\s*(\d+\.\d+(_| )\d+)\]/) {
        $time = $1;
        $time =~ s/$2//g;
        if (defined $time_ref && ($time < $time_ref)) {
            $time_loop_count++;
        }
        $time_ref = $time unless (defined $time_ref);
        $time += $time_loop_count * 65536;    # to always have timestamp increase
    } elsif (!defined($time_ref)) {

        # no timestamp in trace, use line number
        $time = $.;
    } else {
        warn("Cannot find timestamp: $line");
    }

    return $time;
}

sub generate_graph {
    my ($previous, $delta, $fh, $line, $ref, $time_fmt);

    return unless (keys(%tasks_used));

    my $msc;
    my $loop_count = 0;

    if (defined $out) {
        $msc = $out;
    } else {
        $msc = $file;
    }
    $msc =~ s/\.[^.]*$//;
    $msc .= ".msc";

    open $fh, ">", "$msc" or die "Cannot create $msc: $!";

    print $fh "msc { \n";
    print $fh "\twidth=1900;\n";

    if (!defined $time_ref) {
        print $fh "\tt [label=\"Line #\"]";
        $time_fmt = "%d";
    } elsif ($time_loop_count) {
        print $fh "\tt [label=\"time\"]";
        $time_fmt = "%.6f";
    } else {
        print $fh "\tt [label=\"time (from $time_ref)\"]";
        $time_fmt = "%.6f";
    }
    foreach my $t (sort (keys %tasks_used)) {
        my ($name, $id) = ($t =~ /(.*)_(\d+)$/);
        print $fh ",\n\t$t [label=\"$name($id)\"]";
    }
    print $fh ";\n\n";

    $delta = 0;
    foreach my $time (sort { $a <=> $b } (keys %events)) {
        $delta = ($time - $previous) if defined $previous;

        if (($delta == 0) || (!defined $time_ref)) {
            $delta = "";
        } elsif ($delta < 0.001) {
            $delta = sprintf "%dus", int($delta * 1000000);
        } elsif ($delta < 1) {
            $delta = sprintf "%3.3fms", $delta * 1000;
        } else {
            $delta = sprintf "%.3fs", $delta;
        }

        if ($delta) {

            # if (length($delta) < 9) {
            # 	$delta = " "x(9-length($delta)).$delta;
            # }
            $delta = "(+$delta)";
        }

        my $time_show;
        if (!defined $time_ref) {
            $time_show = $time;
        } elsif ($time_loop_count) {
            if ((int($previous) & 0xffff0000) != (int($time) & 0xffff0000)) {
                $delta = "";

                # we have gap in timeline
                print $fh "\t...;\n";
            }
            $time_show = $time - (int($time) & 0xffff0000);
        } else {
            $time_show = $time - $time_ref;
        }

        print $fh sprintf("\tt rbox t [label=\"$time_fmt %s\", linecolour=\"white\"],\n", $time_show, $delta);
        print $fh "\t" . $events{$time} . ";\n\n";

        $previous = $time;
    }

    print $fh "}\n";
    close $fh;

    my $type  = "-Teps";
    my $graph = $msc;
    $graph =~ s/.msc$/.eps/;

    if ($png || system("which epstopdf > /dev/null")) {
        $png = 1;
        $graph =~ s/.eps$/.png/;
        $type = "-Tpng";
    }

    if (system("which mscgen > /dev/null")) {

        # cannot find mscgen in path
        print "Cannot find mscgen in PATH. Use following command to generate graph\n";
        print "mscgen $type -o $graph $msc\n";
        if (!$png) {
            print "epstopdf $graph\n";
        }
        return;
    }

    if (system("mscgen $type -o $graph $msc")) {
        print "Error while running: mscgen $type -o $graph $msc\n";
    } else {
        if (!$png) {
            system("epstopdf $graph");
            unlink $graph;
            $graph =~ s/.eps$/.pdf/;
        }

        #unlink $msc;
        print "graph generated in $graph\n";
    }
}
