#!/usr/bin/env perl

use strict;
use Getopt::Long;
use File::Find;
use FindBin;    # define $FindBin::Bin that contains the path in which this script is located
use Cwd qw(abs_path);

my @directories_to_search;
my @files;
my $rtos_file;
my $outdir;
my %task;
my %task_rtos;
my $no_package;

sub help {
    print "usage: task_update.pl [--out=<output directory>] [--help] [macsw directory]

Parse macsw repository for xx_task.h files and extract messages/states id into a perl package task.pm

--out=<dir> : specificy in which directory task.pm is generated.
              If not specified generate task.pm in directory that contains this script

--no-package : Do not add perl package header.

macsw : Top macsw directory to search.
        If not specified and \$RWNX_TOP_DIR is defined use \$RWNX_TOP_DIR/macsw
        otherwise use ../../ from the location of this script
";
    exit;
}

if (
    !GetOptions(
        "out=s"      => \$outdir,
        "help"       => sub { print &help; exit 0; },
        "no-package" => \$no_package
    )
  )
{
    print &help;
    exit 1;
}

my $macsw = shift;

if (defined $macsw) {
    if (!-d $macsw) {
        die "[$macsw] not a valid directory\n";
    }
    $macsw = abs_path($macsw);
} else {
    if (   (defined $ENV{RWNX_TOP_DIR})
        && (-d $ENV{RWNX_TOP_DIR} . "/macsw"))
    {
        $macsw = abs_path($ENV{RWNX_TOP_DIR} . "/macsw");
    } else {
        $macsw = abs_path($FindBin::Bin . "/../..");
    }
}

foreach my $s (("ip", "modules")) {
    push @directories_to_search, $macsw . "/" . $s;
}

print "** Parsing directory $macsw --\n";
&find(\&wanted, @directories_to_search);

# Look for all file named _task.h
sub wanted {
    if ($_ =~ /(\.git|\.svn)/) {
        $File::Find::prune = 1;
    } elsif ($_ eq "ke_task.h") {
        unshift @files, $File::Find::name;
    } elsif ($_ =~ /(.*)_task.h/) {
        push @files, $File::Find::name;
    } elsif ($_ =~ /rtos_al.h/) {
        $rtos_file = $File::Find::name;
    }
}

# ke_task.h is the first element of the list
&parse_task_list(shift @files);

# Then parse all oterh _task.h files
foreach my $f (@files) {
    &parse_task($f);
}

&parse_rtos_task_list($rtos_file);

# generate output file
&dump();

##################################################################################################
#
##################################################################################################
# get list of task and their id
sub parse_task_list {
    my ($file) = @_;
    my $line;
    my $id_lmac = 0;
    my $id_fmac = 0;
    my $case;
    my $ifdef = 0;
    my @items;
    my $fh;

    open $fh, "<", "$file" or die "cannot open $_ : $!";

    while ($line = <$fh>) {
        if ($line =~ /enum/ && $id_fmac == 0) {

            @items = &read_enum($line, $fh);
            while (my $item = shift @items) {
                if ($item =~ /#/) {
                    if ($item =~ /#if(n)?(def)?.*UMAC_PRESENT/) {
                        if ($1 eq "n") {
                            $case = "lmac";
                        } else {
                            $case = "fmac";
                        }
                        $ifdef++;
                    } elsif ($item =~ /#endif/) {
                        $ifdef--;
                        if ($ifdef == 0) {
                            $case = "";
                        }
                    } elsif ($item =~ /#else/) {
                        if ($ifdef == 1) {
                            if ($case eq "fmac") {
                                $case = "lmac";
                            } elsif ($case eq "lmac") {
                                $case = "fmac";
                            } else {
                                die "uncoherent case \n";
                            }
                        }
                    } elsif ($item =~ /#if(n)?(def)?/) {
                        $ifdef++;
                        warn("  - ifdef inside UMAC, will probably not work as expected\n");
                    }
                    next;
                }

                my ($task, $val) = split /=/, $item;

                if ($task =~ /TASK_(.*)/) {
                    $task = $1;
                } else {
                    warn("  - unexpected item [$task]\n");
                    next;
                }

                if ($task eq "LAST_EMB") {
                    next;
                }

                if (!defined $val) {
                    if ($case eq "lmac" || $case eq "") {
                        $id_lmac++;
                    }
                    if ($case eq "fmac" || $case eq "") {
                        $id_fmac++;
                    }
                } elsif ($val =~ /(\(.*\))?(-?\d+)/) {
                    $val = ($2 & 0xff);
                    if ($case eq "lmac" || $case eq "") {
                        $id_lmac = $val;
                    }
                    if ($case eq "fmac" || $case eq "") {
                        $id_fmac = $val;
                    }
                } elsif ($val =~ /TASK_(.+)/) {
                    my $target = $1;

                    if (!defined $task{$target}) {
                        die "task $target no defined ($file line ~ $.)\n";
                    }
                    if ($case eq "lmac" || $case eq "") {
                        $id_lmac = $task{$target}->{id_lmac};
                    }
                    if ($case eq "fmac" || $case eq "") {
                        $id_fmac = $task{$target}->{id_fmac};
                    }
                }

                if ($case eq "lmac" || $case eq "") {
                    $task{$task}->{id_lmac} = $id_lmac;
                }
                if ($case eq "fmac" || $case eq "") {
                    $task{$task}->{id_fmac} = $id_fmac;
                }
            }
        }
    }

    close $fh;
}

# Parse task header to get a list of states and messages
sub parse_task {
    my ($file) = @_;
    my $name;
    my $line;
    my $fh;
    my %found = (
        messages => 0,
        states   => 0
    );

    ($name) = ($file =~ /.*\/(.*)_task.h/);
    $name =~ tr/a-z/A-Z/;

    if (!defined $task{$name}) {
        warn("  - Skip unknown task [$name]\n");
        return;
    }

    $task{$name}->{states}   = [];
    $task{$name}->{messages} = [];

    #print "processing $file [$name]\n";
    open $fh, "<", "$file" or die "cannot open $_ : $!";

    while ($line = <$fh>) {
        if ($line =~ /enum(.*)/) {
            my $enum_name = $1;
            my @items     = &read_enum($line, $fh);
            my $id        = 0;
            my $type;
            my $f = 0;

            if ($enum_name =~ /state_tag/) {
                $type = "states";
                $f    = 2;
            } elsif ($enum_name =~ /msg_tag/) {
                $type = "messages";
                $f    = 2;
            } else {
                foreach my $t (@items) {
                    if ($t =~ /_(REQ|CFM|IND)$/) {
                        $type = "messages";
                        $f    = 1;
                        last;
                    } elsif ($t =~ /_STATE_MAX/) {
                        $type = "states";
                        $f    = 1;
                        last;
                    }
                }
            }

            if (!$f || ($found{$type} > $f)) {
                next;
            } elsif ($found{$type} == $f) {
                warn("  - [$file] found two enum for $type\n");
                next;
            } elsif ($found{$type}) {

                #previous enum was not the correct one
                $task{$name}->{$type} = [];
            }
            $found{$type} = $f;

            while (my $item = shift @items) {
                if ($item =~ /#/) {
                    warn("  - unparsed [$item] in $file ($.) for state_tag\n");
                    next;
                }

                my ($state, $val) = split /=/, $item;
                my $push = 1;
                if (!defined $val
                    || &check_value($val, $task{$name}->{$type}, \$id, $file))
                {
                    push @{ $task{$name}->{$type} }, $state;
                    $id++;
                }
            }
        }

        last if ($found{state} == 2 && $found{msg} == 2);
    }

    close $fh;
}

sub parse_rtos_task_list {
    my ($file) = @_;
    my $line;
    my $id = 0;
    my $case;
    my $ifdef = 0;
    my @items;
    my $fh;

    return unless($file);

    open $fh, "<", "$file" or die "cannot open $_ : $!";

    while ($line = <$fh>) {
        if ($line =~ /enum/) {
            @items = &read_enum($line, $fh);
            while (my $item = shift @items) {

                my ($task , $val) = split /=/, $item;

                if ($task =~ /(.*)_TASK/) {
                    $task = $1;
                } else {
                    warn("  - unexpected item [$task] in $file\n");
                    next;
                }

                if ($task eq "MAX") {
                    next;
                }

                if (!defined $val) {
                    $id++;
                } elsif ($val =~ /(\(.*\))?(-?\d+)/) {
                    $id = ($2 & 0xff);
                }

                $task_rtos{$id} = $task;
            }
        }
    }

    close $fh;
}


# if enum value is defined with an value, try to convert value into integer
sub check_value {
    my ($val, $ref_array, $ref_id, $file) = @_;
    my ($enum, $num, $new_id);

    if ($val =~ /KE_FIRST_MSG/) {
        return 1;
    }

    if ($val =~ /([^0-9].*)(\+(\d+))?/) {
        $enum = $1;
        $num  = $3;
    } elsif ($val =~ /(\d+)(\+([^0-9].*))?/) {
        $num  = $1;
        $enum = $3;
    }

    # if literate assume it is a already defined enum
    if (defined $enum) {
        my ($id, $found) = (0, 0);
        foreach my $v (@$ref_array) {
            if ($v eq "$enum") {
                $found = 1;
                last;
            }
            $id++;
        }
        if (!$found) {
            die("check (val=$val) $file line $.\n");
        }
        $enum = $id;
    } else {
        $enum = 0;
    }

    $num = 0 unless (defined $num);

    $new_id = $enum + $num;

    if ($new_id < ($$ref_id - 1)) {

        # only allow enum with same value as previous enum
        die("check (current_id=$$ref_id new_id=$new_id) $file line $.\n");
    } elsif ($new_id == ($$ref_id - 1)) {
        warn("  - Skip dupplicate for $val in $file ($.)\n");
        return 0;
    } else {
        while ($new_id > $$ref_id) {
            push @$ref_array, "undef";
            $$ref_id++;
        }
    }

    return 1;
}

sub dump {

    my $fh;
    my $out;

    if (!defined $outdir) {
        $out = $FindBin::Bin . "/task.pm";
    } else {
        $out = $outdir . "/task.pm";
    }

    print "** Generating $out\n";
    open $fh, ">", "$out" or die "Cannot create $out: $!";

    if (!$no_package) {
        print $fh "package task;

use strict;
use Exporter;
use vars qw(\@ISA \@EXPORT);
\@ISA = qw(Exporter);
";
    }

    print $fh "
push \@EXPORT, qw(%tasks);
our %tasks = (\n";

    foreach my $t (sort (keys(%task))) {
        my $align = " " x (length($t) + 10);
        my $first;

        print $fh "  \"$t\" => {\n";

        if (defined $task{$t}->{id_lmac}) {
            print $fh $align . "id_lmac => $task{$t}->{id_lmac},\n";
        }

        if (defined $task{$t}->{id_fmac}) {
            print $fh $align . "id_fmac => $task{$t}->{id_fmac},\n";
        }

        if (defined $task{$t}->{states}) {
            $first = 1;
            print $fh $align . "states => [";
            foreach my $s (@{ $task{$t}->{states} }) {
                print $fh ", " unless $first;
                print $fh "\"$s\"";
                $first = 0;
            }
            print $fh "],\n";
        }
        if (defined $task{$t}->{messages}) {
            $first = 1;
            print $fh $align . "messages => [";
            foreach my $s (@{ $task{$t}->{messages} }) {
                print $fh ", " unless $first;
                print $fh "\"$s\"";
                $first = 0;
            }
            print $fh "],\n";
        }

        print $fh "  },\n";
    }
    print $fh ");\n";

    print $fh "
push \@EXPORT, qw(%rtos_tasks);
our %rtos_tasks = (\n";
    foreach my $t (sort {$a <=> $b} (keys(%task_rtos))) {
        print $fh "  $t => \"$task_rtos{$t}\",\n"
    }
    print $fh ");\n";

    close $fh;
}

sub read_enum {
    my ($line, $fh) = @_;
    my $enum;
    my $comment = 0;
    my $started = 0;
    my $first   = 1;

    while (1) {
        my $new_comment = 0;
        chomp($line = <$fh>) unless ($first == 1);
        $first = 0;

        last if (!defined $line);
        # remove all space (for easier regex)
        $line =~ s/\s*//g;

        if ($line =~ /(.*)\/\*.*\*\/(.*)/) {
            $line = $1.$2;
        }

        if ($line =~ /(.*)\/\*/) {
            $line = $1 if (!$comment);
            $comment++;
            $new_comment = 1 if ($comment == 1);
        }

        if ($line =~ /\*\/(.*)/) {
            $comment=0;
            $line = $1 if (!$comment);
        }

        if ($line =~ /^([^\/]*)\/\//) {
            $line = $1;
        }

        if ($comment && !$new_comment) {
            next;
        }

        if ($line =~ /{(.*)/) {
            $enum .= $1;
            $started = 1;
        } elsif ($started) {
            if ($line =~ /(.*)}/) {
                $enum .= $1;
                last;
            } else {
                $enum .= $line;
                $enum .= "," if ($line =~ /^#/);
            }
        }
    }

    return (split /,/, $enum);
}
